<?php

namespace Drupal\Tests\api_status\Unit;

use Drupal\api_status\ApiStatusService;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Tests\UnitTestCase;
use Psr\Log\LoggerInterface;

/**
 * Tests the ApiStatusService.
 *
 * @group api_status
 * @coversDefaultClass \Drupal\api_status\ApiStatusService
 */
class ApiStatusServiceTest extends UnitTestCase {

  /**
   * The mocked state service.
   *
   * @var \Drupal\Core\State\StateInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $state;

  /**
   * The mocked date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $dateFormatter;

  /**
   * The mocked logger.
   *
   * @var \Psr\Log\LoggerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * The service under test.
   *
   * @var \Drupal\api_status\ApiStatusService
   */
  protected $service;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->state = $this->createMock(StateInterface::class);
    $this->dateFormatter = $this->createMock(DateFormatterInterface::class);
    $this->logger = $this->createMock(LoggerInterface::class);

    $this->service = new ApiStatusService(
      $this->state,
      $this->dateFormatter,
      $this->logger
    );
  }

  /**
   * Tests logging a successful API call.
   *
   * @covers ::log
   */
  public function testLogSuccess() {
    $this->state->expects($this->exactly(2))
      ->method('set');

    $this->state->expects($this->once())
      ->method('get')
      ->with('api_status.tracked', [])
      ->willReturn([]);

    $result = $this->service->log('test_api', 'success', '/api/test');

    $this->assertTrue($result);
  }

  /**
   * Tests logging a failed API call.
   *
   * @covers ::log
   */
  public function testLogFailed() {
    $this->state->expects($this->atLeastOnce())
      ->method('set');

    $this->state->expects($this->once())
      ->method('get')
      ->with('api_status.tracked', [])
      ->willReturn([]);

    $result = $this->service->log('test_api', 'failed');

    $this->assertTrue($result);
  }

  /**
   * Tests logging with invalid status.
   *
   * @covers ::log
   */
  public function testLogInvalidStatus() {
    $result = $this->service->log('test_api', 'invalid');

    $this->assertFalse($result);
  }

  /**
   * Tests getting API status.
   *
   * @covers ::getStatus
   */
  public function testGetStatus() {
    $this->state->expects($this->exactly(3))
      ->method('get')
      ->willReturnCallback(function ($key) {
        $values = [
          'api_status.success.test_api' => 1000,
          'api_status.failed.test_api' => 900,
          'api_status.endpoint.test_api' => '/api/test',
        ];
        return $values[$key] ?? NULL;
      });

    $status = $this->service->getStatus('test_api');

    $this->assertEquals(1000, $status['last_success']);
    $this->assertEquals(900, $status['last_failed']);
    $this->assertEquals('success', $status['current_status']);
    $this->assertEquals('/api/test', $status['endpoint']);
  }

  /**
   * Tests getting status when failed is more recent.
   *
   * @covers ::getStatus
   */
  public function testGetStatusFailed() {
    $this->state->expects($this->exactly(3))
      ->method('get')
      ->willReturnCallback(function ($key) {
        $values = [
          'api_status.success.test_api' => 900,
          'api_status.failed.test_api' => 1000,
          'api_status.endpoint.test_api' => '/api/test',
        ];
        return $values[$key] ?? NULL;
      });

    $status = $this->service->getStatus('test_api');

    $this->assertEquals('failed', $status['current_status']);
  }

  /**
   * Tests getting status with unknown state.
   *
   * @covers ::getStatus
   */
  public function testGetStatusUnknown() {
    $this->state->expects($this->exactly(3))
      ->method('get')
      ->willReturn(NULL);

    $status = $this->service->getStatus('test_api');

    $this->assertEquals('unknown', $status['current_status']);
  }

  /**
   * Tests clearing API status.
   *
   * @covers ::clear
   */
  public function testClear() {
    $this->state->expects($this->exactly(3))
      ->method('delete');

    $this->state->expects($this->once())
      ->method('get')
      ->with('api_status.tracked', [])
      ->willReturn(['test_api', 'other_api']);

    $this->state->expects($this->once())
      ->method('set')
      ->with('api_status.tracked', ['other_api']);

    $this->service->clear('test_api');
  }

  /**
   * Tests clearing all API data.
   *
   * @covers ::clearAll
   */
  public function testClearAll() {
    $this->state->expects($this->once())
      ->method('get')
      ->with('api_status.tracked', [])
      ->willReturn(['api1', 'api2']);

    // 3 deletes per API + 1 for tracked list.
    $this->state->expects($this->exactly(7))
      ->method('delete');

    $this->service->clearAll();
  }

}

