<?php

namespace Drupal\api_token_entity\Form;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\api_token_entity\ApiTokenManager;

/**
 * Form handler for the Character Event entity edit forms.
 */
class ApiTokenEntityForm extends ContentEntityForm {

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line because Drupal's core inherited PHPDoc has array parameters with no value type specified.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);

    /** @var \Drupal\api_token_entity\Entity\ApiToken $entity */
    $entity = $this->entity;

    if ($entity->isNew()) {
      $form['value']['widget'][0]['value']['#default_value'] = ApiTokenManager::generateSecureApiTokenValue();
      $form['value']['widget'][0]['value']['#attributes']['readonly'] = 'readonly';
      $form['expiration_date']['widget'][0]['value']['#default_value'] = new DrupalDateTime('+1 year');
    }
    else {
      $form['name']['widget'][0]['value']['#disabled'] = TRUE;
      $form['value']['#access'] = FALSE;

      $form['actions']['rotate'] = [
        '#type' => 'submit',
        '#value' => $this->t('Rotate Token'),
        '#submit' => [[$this, 'rotateToken']],
        '#weight' => 10,
        '#button_type' => 'secondary',
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line because Drupal's core inherited PHPDoc has array parameters with no value type specified.
   */
  public function save(array $form, FormStateInterface $form_state) {
    $result = parent::save($form, $form_state);
    $form_state->setRedirect('entity.api_token_entity_api_token.collection');
    return $result;
  }

  /**
   * Submit handler for rotating the token.
   *
   * @param mixed[] $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function rotateToken(array &$form, FormStateInterface $form_state): void {
    /** @var \Drupal\Core\Entity\EntityFormInterface $entityForm */
    $entityForm = $form_state->getFormObject();
    /** @var \Drupal\api_token_entity\Entity\ApiToken $entity */
    $entity = $entityForm->getEntity();
    $token = ApiTokenManager::generateSecureApiTokenValue();
    $entity
      ->set('value', md5($token))
      ->save();

    \Drupal::messenger()
      ->addMessage('Token rotated successfully! Copy the token value now as it cannot be recovered.')
      ->addMessage("Token value: $token");
    $form_state->setRebuild();
  }

}
