<?php

namespace Drupal\api_token_entity\Access;

use Drupal\api_token_entity\ApiTokenManager;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Routing\Access\AccessInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Route;

/**
 * Provides an access checker for API token type entities.
 */
class ApiTokenTypeAccessCheck implements AccessInterface {

  /**
   * Constructs a ApiTokenTypeAccessCheck object.
   */
  public function __construct(
    protected LoggerInterface $logger,
    protected ApiTokenManager $apiTokenManager,
  ) {}

  /**
   * Grants access when matching required API token type.
   */
  public function access(Request $request, Route $route): AccessResult {
    $api_token_type = $route->getRequirement('_api_token_type');
    if (empty($api_token_type)) {
      $this->logger->error('Missing API token type');
      return AccessResult::forbidden();
    }

    $auth_header = $request->headers->get('authorization');
    if (!($auth_header && strpos($auth_header, 'ApiKey ') === 0)) {
      $this->logger->error('Missing API token value');
      return AccessResult::forbidden();
    }
    $api_token_value = substr($auth_header, 7);

    $consumer = $this->apiTokenManager->checkApiToken($api_token_value, $api_token_type);
    if (empty($consumer)) {
      $this->logger->error('Invalid API token');
      return AccessResult::forbidden();
    }

    $this->logger->debug("API token accepted for '$consumer'");
    return AccessResult::allowed();
  }

}
