<?php

namespace Drupal\api_token_entity\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\ContentEntityDeleteForm;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityViewBuilder;
use Drupal\Core\Entity\Form\DeleteMultipleForm;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\Entity\Sql\SqlContentEntityStorage;
use Drupal\Core\Entity\Sql\SqlContentEntityStorageSchema;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\api_token_entity\Form\ApiTokenEntityForm;
use Drupal\api_token_entity\ApiTokenEntityAccessControlHandler;
use Drupal\api_token_entity\ApiTokenInterface;
use Drupal\api_token_entity\ApiTokenListBuilder;
use Drupal\views\EntityViewsData;

/**
 * Defines the API token entity class.
 */
#[ContentEntityType(
  id: 'api_token_entity_api_token',
  label: new TranslatableMarkup('API token'),
  label_collection: new TranslatableMarkup('API tokens'),
  label_singular: new TranslatableMarkup('API token'),
  label_plural: new TranslatableMarkup('API tokens'),
  entity_keys: [
    'id' => 'id',
    'label' => 'name',
  ],
  handlers: [
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
    'storage' => SqlContentEntityStorage::class,
    'storage_schema' => SqlContentEntityStorageSchema::class,
    'view_builder' => EntityViewBuilder::class,
    'access' => ApiTokenEntityAccessControlHandler::class,
    'views_data' => EntityViewsData::class,
    'form' => [
      'default' => ApiTokenEntityForm::class,
      'delete' => ContentEntityDeleteForm::class,
      'edit' => ApiTokenEntityForm::class,
      'delete-multiple-confirm' => DeleteMultipleForm::class,
    ],
    'list_builder' => ApiTokenListBuilder::class,
  ],
  links: [
    'delete-form' => '/admin/config/services/api-token-entity/api-token/{api_token_entity_api_token}/delete',
    'delete-multiple-form' => '/admin/config/services/api-token-entity/api-token/delete',
    'add-form' => '/admin/config/services/api-token-entity/api-token/add',
    'edit-form' => '/admin/config/services/api-token-entity/api-token/{api_token_entity_api_token}/edit',
    'collection' => '/admin/config/services/api-token-entity/api-token/list',
  ],
  admin_permission: 'administer api_token_entity entities',
  collection_permission: 'administer api_token_entity entities',
  base_table: 'api_token_entity_api_token',
  data_table: 'api_token_entity_api_token_field_data',
  label_count: [
    'singular' => '@count API token item',
    'plural' => '@count API token items',
  ],
  list_cache_tags: [
    'api_token_entity_api_token',
  ],
)]
class ApiToken extends ContentEntityBase implements ApiTokenInterface {

  /**
   * {@inheritDoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['type'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Type'))
      ->setDescription(t('The API token type.'))
      ->setRequired(TRUE)
      ->setSetting('target_type', 'api_token_entity_api_token_type')
      ->setSetting('handler', 'default')
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'entity_reference_label',
        'weight' => 1,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => 1,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Consumer ID'))
      ->setDescription(t('The API token consumer id.'))
      ->setRequired(TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'type' => 'string',
        'weight' => 2,
      ])
      ->setDisplayOptions('form', [
        'type' => 'sintrg_textfield',
        'weight' => 2,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->addConstraint('UniqueField')
      ->addConstraint('MachineName');

    $fields['value'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Token Value'))
      ->setDescription(t("The API token value, automatically generated. <b>Once set, it cannot be retrieved; copy it before it's gone</b>."))
      ->setRequired(TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 3,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 3,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['expiration_date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Expiration date'))
      ->setDescription(t("The API token expiration date."))
      ->setRequired(TRUE)
      ->setSettings([
        'datetime_type' => 'date',
      ])
      ->setDisplayOptions('form', [
        'type' => 'datetime_default',
        'weight' => 4,
      ])
      ->setDisplayOptions('view', [
        'type' => 'datetime_default',
        'weight' => 4,
        'settings' => [
          'format_type' => 'medium',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    return $fields;
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line because Drupal's core inherited PHPDoc has array parameters with no value type specified.
   */
  public function preSave($storage) {
    parent::preSave($storage);

    $current_value = $this->get('value')->value;

    // If the value looks like a plain token (not an MD5 hash), hash it.
    // MD5 hashes are always 32 characters and hexadecimal.
    if (!empty($current_value) && (strlen($current_value) !== 32 || !ctype_xdigit($current_value))) {
      $this->set('value', md5($current_value));
    }
  }

}
