<?php

namespace Drupal\api_token_entity\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\api_token_entity\ApiTokenManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form to check API tokens.
 */
final class ApiTokenCheckerForm extends FormBase {

  /**
   * Constructs an ApiTokenCheckerForm object.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ApiTokenManager $apiTokenManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('api_token_entity.api_token.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'api_token_checker_form';
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line because Drupal's core inherited PHPDoc has array parameters with no value type specified.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $options_api_token_type = [];
    foreach ($this->entityTypeManager->getStorage('api_token_entity_api_token_type')->loadMultiple() as $token_type) {
      $label = (string) $token_type->label();
      $options_api_token_type[$label] = $label;
    }
    $form['type'] = [
      '#type' => 'select',
      '#title' => $this->t('Token type'),
      '#description' => $this->t('Select the API token type.'),
      '#required' => TRUE,
      '#options' => $options_api_token_type,
      '#empty_option' => $this->t('- Select a API token type -'),
    ];

    $form['value'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token value'),
      '#description' => $this->t('Enter the API token to validate.'),
      '#required' => TRUE,
      '#maxlength' => 255,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Validate Token'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line because Drupal's core inherited PHPDoc has array parameters with no value type specified.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $token_value = $form_state->getValue('value');
    $token_type_name = $form_state->getValue('type');

    if ($token_name = $this->apiTokenManager->checkApiToken($token_value, $token_type_name)) {
      $this->messenger()->addMessage($this->t("Token valid, it belongs to @token_full_name.", ['@token_full_name' => "$token_type_name:$token_name"]));
    }
    else {
      $this->messenger()->addWarning($this->t('Token is not valid.'));
    }
  }

}
