<?php

namespace Drupal\Tests\api_token_entity\Functional;

use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests the API token type entities.
 *
 * @group api_token_entity
 */
#[RunTestsInSeparateProcesses]
class ApiTokenTypeEntityTest extends ApiTokenEntityTestBase {

  /**
   * Tests the API token type collection page.
   */
  public function testCollectionPage(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/services/api-token-entity/api-token-type/list');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('API token types');
  }

  /**
   * Tests creating an API token type entity.
   */
  public function testCreateApiTokenType(): void {
    $this->drupalLogin($this->adminUser);

    // Navigate to the add form.
    $this->drupalGet('/admin/config/services/api-token-entity/api-token-type/add');
    $this->assertSession()->statusCodeEquals(200);

    // Fill in the form.
    $edit = [
      'name[0][value]' => 'test_token_type',
    ];
    $this->submitForm($edit, 'Save');

    // Verify the entity was created by checking the page content.
    $this->assertSession()->statusCodeEquals(200);
    $this->drupalGet('/admin/config/services/api-token-entity/api-token-type/list');
    $this->assertSession()->pageTextContains('test_token_type');
  }

  /**
   * Tests editing an API token type entity.
   */
  public function testEditApiTokenType(): void {
    $this->drupalLogin($this->adminUser);

    // Create an entity programmatically.
    $entity = $this->entityTypeManager
      ->getStorage('api_token_entity_api_token_type')
      ->create(['name' => 'original_name']);
    $entity->save();

    // Navigate to the edit form.
    $this->drupalGet('/admin/config/services/api-token-entity/api-token-type/' . $entity->id() . '/edit');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->fieldValueEquals('name[0][value]', 'original_name');

    // Update the entity.
    $edit = [
      'name[0][value]' => 'updated_name',
    ];
    $this->submitForm($edit, 'Save');

    // Verify the update by navigating back to edit form.
    $this->drupalGet('/admin/config/services/api-token-entity/api-token-type/' . $entity->id() . '/edit');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->fieldValueEquals('name[0][value]', 'updated_name');
  }

  /**
   * Tests deleting an API token type entity.
   */
  public function testDeleteApiTokenType(): void {
    $this->drupalLogin($this->adminUser);

    // Create an entity programmatically.
    $entity = $this->entityTypeManager
      ->getStorage('api_token_entity_api_token_type')
      ->create(['name' => 'delete_test']);
    $entity->save();
    $entity_id = $entity->id();

    // Navigate to the delete form.
    $this->drupalGet('/admin/config/services/api-token-entity/api-token-type/' . $entity_id . '/delete');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Are you sure you want to delete');

    // Confirm deletion.
    $this->submitForm([], 'Delete');

    // Verify the entity was deleted by trying to access the edit form.
    $this->drupalGet('/admin/config/services/api-token-entity/api-token-type/' . $entity_id . '/edit');
    $this->assertSession()->statusCodeEquals(404);
  }

  /**
   * Tests the unique field constraint on the name field.
   */
  public function testUniqueNameConstraint(): void {
    $this->drupalLogin($this->adminUser);

    // Create the first entity.
    $entity1 = $this->entityTypeManager
      ->getStorage('api_token_entity_api_token_type')
      ->create(['name' => 'unique_name']);
    $entity1->save();

    // Try to create a second entity with the same name.
    $this->drupalGet('/admin/config/services/api-token-entity/api-token-type/add');
    $edit = [
      'name[0][value]' => 'unique_name',
    ];
    $this->submitForm($edit, 'Save');

    // Verify validation error appears.
    $this->assertSession()->pageTextContains('already exists');
  }

  /**
   * Tests access control for anonymous users.
   */
  public function testAnonymousAccess(): void {
    // Try to access the collection page as anonymous user.
    $this->drupalGet('/admin/config/services/api-token-entity/api-token-type/list');
    $this->assertSession()->statusCodeEquals(403);

    // Try to access the add form as anonymous user.
    $this->drupalGet('/admin/config/services/api-token-entity/api-token-type/add');
    $this->assertSession()->statusCodeEquals(403);
  }

}
