<?php

namespace Drupal\api_token_entity;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a class to build a list of Token entities.
 */
final class ApiTokenListBuilder extends EntityListBuilder {

  /**
   * Constructs a new EntityListBuilder object.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    EntityStorageInterface $storage,
    protected FormBuilderInterface $formBuilder,
  ) {
    parent::__construct($entity_type, $storage);
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('form_builder'),
    );
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line because Drupal's core inherited PHPDoc has array parameters with no value type specified.
   */
  public function buildHeader(): array {
    $header = [];
    $header['name'] = [
      'data' => $this->t('Name'),
      'field' => 'name',
      'specifier' => 'name',
    ];
    $header['type'] = [
      'data' => $this->t('Type'),
      'field' => 'type',
      'specifier' => 'type',
    ];
    $header['value'] = $this->t('Value');
    $header['expiration_date'] = [
      'data' => $this->t('Expiration date'),
      'field' => 'expiration_date',
      'specifier' => 'expiration_date',
      'sort' => 'asc',
    ];
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line because Drupal's core inherited PHPDoc has array parameters with no value type specified.
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\api_token_entity\Entity\ApiToken $entity */
    $row = [];
    $expiration = strtotime($entity->get('expiration_date')->getString());
    $now = new DrupalDateTime('now')->getTimestamp();
    $label = $entity->label();
    if ($expiration !== FALSE && $expiration < $now) {
      $label = '⛔ ' . $label;
    }
    $row['name'] = $label;
    $row['type'] = $entity->get('type')->entity->label();
    $row['value'] = '••••••••••••••••';
    $row['expiration_date'] = $entity->get('expiration_date')->getString();
    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line because Drupal's core inherited PHPDoc has array parameters with no value type specified.
   */
  protected function getEntityIds() {
    $header = $this->buildHeader();
    // @phpstan-ignore-next-line false positive.
    return $this->getStorage()->getQuery()
      ->accessCheck(FALSE)
      ->tableSort($header)
      ->execute();
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line because Drupal's core inherited PHPDoc has array parameters with no value type specified.
   */
  public function render(): array {
    $build = parent::render();

    // Add a create button above the table.
    $build['create_button'] = [
      '#type' => 'link',
      '#title' => $this->t('Create a new @entityLabel', ['@entityLabel' => $this->entityType->getLabel()]),
      '#url' => Url::fromRoute('entity.api_token_entity_api_token.add_form'),
      '#attributes' => [
        'class' => ['button', 'button--action', 'button--primary'],
      ],
      '#weight' => -10,
    ];

    // Add token validation form.
    $build['validate_form'] = [
      '#type' => 'details',
      '#title' => $this->t('Validate Token'),
      '#open' => FALSE,
      '#weight' => -5,
    ];

    $build['validate_form']['form'] = $this->formBuilder->getForm('Drupal\api_token_entity\Form\ApiTokenCheckerForm');

    return $build;
  }

}
