<?php

namespace Drupal\Tests\api_token_entity\Functional;

use Drupal\api_token_entity\ApiTokenManager;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Tests\BrowserTestBase;
use Drupal\user\UserInterface;
use Symfony\Component\HttpKernel\HttpKernelInterface;

/**
 * Abstract class for the API token entity tests.
 */
abstract class ApiTokenEntityTestBase extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'claro';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['api_token_entity', 'api_token_entity_test'];

  /**
   * The Drupal entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * A user with permission to administer API token entities.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $adminUser;

  /**
   * The HTTP kernel.
   *
   * @var \Symfony\Component\HttpKernel\HttpKernelInterface
   */
  protected HttpKernelInterface $httpKernel;

  /**
   * Stored API keys.
   *
   * @var array<string, array<string, string>>
   */
  protected array $apiKeys;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->container->get('entity_type.manager');

    /** @var \Drupal\user\UserInterface $newUser */
    $newUser = $this->drupalCreateUser(['administer api_token_entity entities']);
    $this->adminUser = $newUser;
    $this->httpKernel = $this->container->get('http_kernel');

    // Rebuild routes to pick up test module routes.
    $this->container->get('router.builder')->rebuild();

    [$tokenValueRecipe, $tokenValueRestaurant] = $this->createApiTokensMockup();
    $this->apiKeys = [
      'recipes' => [
        'alice' => $tokenValueRecipe,
      ],
      'restaurants' => [
        'bob' => $tokenValueRestaurant,
      ],
    ];
  }

  /**
   * Create test mockups.
   *
   * @return string[]
   *   The test API token for recipes and restaurants.
   */
  private function createApiTokensMockup(): array {
    $tokenTypeRecipes = $this->entityTypeManager
      ->getStorage('api_token_entity_api_token_type')
      ->create(['name' => 'recipes']);
    $tokenTypeRecipes->save();
    $tokenValueRecipe = ApiTokenManager::generateSecureApiTokenValue();
    $token = $this->entityTypeManager
      ->getStorage('api_token_entity_api_token')
      ->create([
        'type' => $tokenTypeRecipes->id(),
        'name' => 'alice',
        'value' => $tokenValueRecipe,
        'expiration_date' => date('Y-m-d', strtotime('+1 year')),
      ]);
    $token->save();

    $tokenTypeRestaurants = $this->entityTypeManager
      ->getStorage('api_token_entity_api_token_type')
      ->create(['name' => 'restaurants']);
    $tokenTypeRestaurants->save();
    $tokenValueRestaurant = ApiTokenManager::generateSecureApiTokenValue();
    $token = $this->entityTypeManager
      ->getStorage('api_token_entity_api_token')
      ->create([
        'type' => $tokenTypeRestaurants->id(),
        'name' => 'bob',
        'value' => $tokenValueRestaurant,
        'expiration_date' => date('Y-m-d', strtotime('+1 year')),
      ]);
    $token->save();

    return [$tokenValueRecipe, $tokenValueRestaurant];
  }

}
