<?php

namespace Drupal\api_toolkit\Plugin\Validation\Constraint;

use Symfony\Component\Validator\Constraint;

/**
 * Validates whether an entity field has unique values.
 *
 * @Annotation
 * @Target({"PROPERTY", "METHOD", "ANNOTATION"})
 *
 * @Constraint(
 *     id = "entity_unique",
 *     label = @Translation("Checks if an entity field has unique values", context = "Validation"),
 * )
 */
#[\Attribute(\Attribute::TARGET_PROPERTY | \Attribute::TARGET_METHOD | \Attribute::IS_REPEATABLE)]
class EntityUnique extends Constraint {

  public const NOT_UNIQUE_ERROR = '15a3f795-1371-4758-93cb-1078503dea1b';

  /**
   * {@inheritdoc}
   */
  protected static $errorNames = [
    self::NOT_UNIQUE_ERROR => 'NOT_UNIQUE_ERROR',
  ];

  /**
   * The entity type ID.
   *
   * @var string|null
   */
  public ?string $entityTypeId = NULL;

  /**
   * The entity bundle.
   *
   * @var string|null
   */
  public ?string $bundle = NULL;

  /**
   * The field name to check.
   *
   * @var string|null
   */
  public ?string $fieldName = NULL;

  /**
   * The error message in case an entity with the same value already exists.
   *
   * @var string
   */
  public string $message = 'An entity with %fieldLabel %value already exists.';

  /**
   * {@inheritdoc}
   */
  public function getRequiredOptions(): array {
    return ['entityTypeId', 'fieldName'];
  }

  /**
   * {@inheritdoc}
   */
  public function __construct(?array $options = NULL, ?string $entityTypeId = NULL, ?string $bundle = NULL, ?string $fieldName = NULL, ?string $message = NULL, ?array $groups = NULL, mixed $payload = NULL) {
    $options['entityTypeId'] ??= $entityTypeId ?? $this->entityTypeId;
    $options['bundle'] ??= $bundle ?? $this->bundle;
    $options['fieldName'] ??= $fieldName ?? $this->fieldName;
    $options['message'] ??= $message ?? $this->message;

    parent::__construct($options, $groups, $payload);
  }

}
