<?php

/**
 * Copyright 2018 Google Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

namespace Drupal\apigee_edge_teams\Entity\Storage;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Cache\MemoryCache\MemoryCacheInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\apigee_edge\Entity\AppInterface;
use Drupal\apigee_edge\Entity\Controller\AppControllerInterface;
use Drupal\apigee_edge\Entity\Controller\EdgeEntityControllerInterface;
use Drupal\apigee_edge\Entity\Controller\OrganizationControllerInterface;
use Drupal\apigee_edge\Entity\Storage\AppStorage;
use Drupal\apigee_edge_teams\Entity\Controller\TeamAppApigeeXEntityControllerProxy;
use Drupal\apigee_edge_teams\Entity\Controller\TeamAppControllerFactoryInterface;
use Drupal\apigee_edge_teams\Entity\Controller\TeamAppEdgeEntityControllerProxy;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Entity storage class for Team app entities.
 *
 * @phpstan-ignore-next-line
 */
class TeamAppStorage extends AppStorage implements TeamAppStorageInterface {

  /**
   * The team app controller factory.
   *
   * @var \Drupal\apigee_edge_teams\Entity\Controller\TeamAppControllerFactoryInterface
   */
  private $teamAppControllerFactory;

  /**
   * The organization controller service.
   *
   * @var \Drupal\apigee_edge\Entity\Controller\OrganizationControllerInterface
   */
  private $orgController;

  /**
   * AppStorage constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   The cache backend to be used.
   * @param \Drupal\Core\Cache\MemoryCache\MemoryCacheInterface $memory_cache
   *   The memory cache.
   * @param \Drupal\Component\Datetime\TimeInterface $system_time
   *   The system time.
   * @param \Drupal\apigee_edge_teams\Entity\Controller\TeamAppControllerFactoryInterface $team_app_controller_factory
   *   The team app controller factory service.
   * @param \Drupal\apigee_edge\Entity\Controller\AppControllerInterface $app_controller
   *   The app controller service.
   * @param \Drupal\apigee_edge\Entity\Controller\OrganizationControllerInterface $org_controller
   *   The organization controller service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config
   *   Configuration factory.
   */
  public function __construct(EntityTypeInterface $entity_type, CacheBackendInterface $cache_backend, MemoryCacheInterface $memory_cache, TimeInterface $system_time, TeamAppControllerFactoryInterface $team_app_controller_factory, AppControllerInterface $app_controller, OrganizationControllerInterface $org_controller, protected ?ConfigFactoryInterface $config = NULL) {
    parent::__construct($entity_type, $cache_backend, $memory_cache, $system_time, $app_controller);
    $this->teamAppControllerFactory = $team_app_controller_factory;
    $this->orgController = $org_controller;
    if ($config === NULL) {
      @trigger_error('Calling ' . __METHOD__ . ' without the $config is deprecated in apigee_edge:4.0.3 and it will be required in apigee_edge:5.0.0. See https://github.com/apigee/apigee-edge-drupal/pull/1155.', E_USER_DEPRECATED);
      $config = \Drupal::configFactory();
    }

    $config = $config->get('apigee_edge_teams.team_app_settings');
    $this->cacheExpiration = $config->get('cache_expiration');
    $this->cacheInsertChunkSize = $config->get('cache_insert_chunk_size') ?? static::DEFAULT_PERSISTENT_CACHE_INSERT_CHUNK_SIZE;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('cache.apigee_edge_entity'),
      $container->get('entity.memory_cache'),
      $container->get('datetime.time'),
      $container->get('apigee_edge_teams.controller.team_app_controller_factory'),
      $container->get('apigee_edge.controller.app'),
      $container->get('apigee_edge.controller.organization'),
      $container->get('config.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function entityController(): EdgeEntityControllerInterface {
    if ($this->orgController->isOrganizationApigeeX()) {
      $teamAppEntityControllerProxy = new TeamAppApigeeXEntityControllerProxy($this->teamAppControllerFactory, $this->appController);
    }
    else {
      $teamAppEntityControllerProxy = new TeamAppEdgeEntityControllerProxy($this->teamAppControllerFactory, $this->appController);
    }
    return $teamAppEntityControllerProxy;
  }

  /**
   * {@inheritdoc}
   */
  protected function getCacheTagsByOwner(AppInterface $app): array {
    // Add team's name to ensure when the owner of the app (team)
    // gets deleted then _all_ its cached team app data gets purged along with
    // it.
    return ["team:{$app->getAppOwner()}"];
  }

}
