<?php

declare(strict_types=1);

namespace Drupal\apns_php\Service;

use Drupal\apns_php\ApnsPhpMessagingApiInterface;
use Drupal\apns_php\Enum\LoggingLevel;
use Drupal\apns_php\Exception\ApnsPhpInvalidArgumentException;
use Pushok\Notification;
use Pushok\Payload;
use Pushok\Payload\Alert;

/**
 * Provides a Drupal API for sending APNs push notifications.
 */
class ApnsPhpMessagingApi extends ApnsPhpMessagingService implements ApnsPhpMessagingApiInterface {

  /**
   * {@inheritdoc}
   */
  public function sendMessageSingleDevice(
    string $device_token,
    ?string $notification_title,
    ?string $notification_body,
    ?string $notification_url = NULL,
    ?int $badge_count = NULL,
    ?bool $silent = FALSE,
  ): array {
    return $this->sendMessageMultipleDevices([$device_token], $notification_title, $notification_body, $notification_url, $badge_count, $silent);
  }

  /**
   * {@inheritdoc}
   */
  public function sendMessageMultipleDevices(
    array $device_tokens,
    ?string $notification_title,
    ?string $notification_body,
    ?string $notification_url = NULL,
    ?int $badge_count = NULL,
    ?bool $silent = FALSE,
  ): array {
    $payload = $this->createPayload($notification_title, $notification_body, $notification_url, $badge_count, $silent);
    $notifications = [];
    foreach ($device_tokens as $device_token) {
      $notifications[] = new Notification($payload, $device_token);
    }
    $this->client->addNotifications($notifications);
    $responses = $this->client->push();
    $this->logResponses($responses);
    return $responses;
  }

  /**
   * Logs the response depending on the module config logging level.
   *
   * Always logs failures. Will log success too in debug mode.
   *
   * @param array $responses
   *   The responses to log to Drupal.
   */
  public function logResponses(array $responses): void {
    /** @var \Pushok\ApnsResponseInterface[] $responses */
    foreach ($responses as $response) {
      // The device token.
      $device_token = $response->getDeviceToken();
      // A canonical UUID that is the unique ID for the notification.
      // E.g. 123e4567-e89b-12d3-a456-4266554400a0.
      $apns_id = $response->getApnsId();
      // Status code. E.g. 200 (Success),
      // 410 (The device token is no longer active for the topic.)
      $status_code = $response->getStatusCode();

      // We have an error, log it.
      if ($status_code !== 200) {
        // E.g. The device token is no longer active for the topic.
        $reason_phrase = $response->getReasonPhrase();
        // E.g. Unregistered.
        $error_reason = $response->getErrorReason();
        // E.g. The device token is inactive for the specified topic.
        $error_description = $response->getErrorDescription();
        $timestamp_401 = $response->get410Timestamp();

        if ($this->loggingLevel !== LoggingLevel::None) {
          $this->logger->warning(
          "Push notification to device token $device_token failed with status code $status_code.\nReason phrase: $reason_phrase\nError reason: $error_reason\nError description: $error_description\n401 timestamp: $timestamp_401"
          );
        }
      }
      elseif ($this->loggingLevel === LoggingLevel::Debug) {
        $this->loggerDebug->info("Push notification to device token $device_token returned status code $status_code. APNS ID: $apns_id");
      }
    }
  }

  /**
   * Creates a notification for use in sending messages.
   *
   * All messages need a notification.
   *
   * @param string|null $title
   *   The title of the notification.
   * @param string|null $body
   *   The body of the notification.
   * @param string|null $url
   *   The optional URL for the notification.
   * @param int|null $badge_count
   *   The optional badge count.
   * @param bool|null $silent
   *   TRUE if the notification should be silent (not play a sound).
   *
   * @return \Pushok\Payload
   *   The notification.
   */
  private function createPayload(?string $title, ?string $body, ?string $url = NULL, ?int $badge_count = NULL, $silent = NULL): Payload {
    $alert = Alert::create();
    if (is_string($title) && $title !== '') {
      $alert->setTitle($title);
    }
    if (is_string($body) && $body !== '') {
      $alert->setBody($body);
    }
    // @todo Figure out how to pass the URL.
    $payload = Payload::create()->setAlert($alert);
    if (is_int($badge_count)) {
      $payload->setBadge($badge_count);
    }
    if ($silent === TRUE) {
      $payload->setSound('');
    }
    else {
      $payload->setSound('default');
    }
    return $payload;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function validateToken(string $token): bool {
    if (trim($token) === '') {
      throw new ApnsPhpInvalidArgumentException('Token cannot be an empty string!');
    }
    /** @var non-empty-string $token */
    // @todo Implement token validation.
    return TRUE;
  }

}
