<?php

declare(strict_types=1);

namespace Drupal\apns_php\Service;

use Drupal\apns_php\Enum\ApnsPhpSettings;
use Drupal\apns_php\Enum\LoggingLevel;
use Drupal\apns_php\Exception\ApnsPhpCertificateNotFoundException;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Session\AccountInterface;
use Psr\Log\LoggerInterface;
use Pushok\AuthProvider\Token as AuthProviderToken;
use Pushok\AuthProviderInterface;
use Pushok\Client;

/**
 * Service for sending push notifications to Apple devices using APNs.
 */
class ApnsPhpMessagingService {

  /**
   * The pushok auth provider (only token is used).
   *
   * @var \Pushok\AuthProviderInterface
   */
  protected AuthProviderInterface $authProvider;

  /**
   * The pushok client for sending push notifications.
   *
   * @var \Pushok\Client
   */
  protected Client $client;

  /**
   * This module's configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * The amount of logging information to include.
   *
   * @var \Drupal\apns_php\Enum\LoggingLevel
   */
  protected LoggingLevel $loggingLevel;

  public function __construct(
    protected LoggerInterface $logger,
    protected LoggerInterface $loggerDebug,
    protected AccountInterface $account,
    protected ConfigFactory $configFactory,
  ) {
    $config = $this->configFactory->get('apns_php.settings');
    $key_id = $config->get(ApnsPhpSettings::CertificateKeyId->value);
    if (!is_string($key_id) || $key_id === '') {
      throw new ApnsPhpCertificateNotFoundException(
        "Failed to get Apple Developer Key ID!",
      );
    }
    $team_id = $config->get(ApnsPhpSettings::AppleDeveloperTeamId->value);
    if (!is_string($team_id) || $team_id === '') {
      throw new ApnsPhpCertificateNotFoundException(
        "Failed to get Apple Developer Team ID!",
      );
    }
    $app_bundle_id = $config->get(ApnsPhpSettings::AppBundleId->value);
    if (!is_string($app_bundle_id) || $app_bundle_id === '') {
      throw new ApnsPhpCertificateNotFoundException(
        "Failed to get app bundle ID!",
      );
    }
    $certificate_path = $config->get(ApnsPhpSettings::CertificatePath->value);
    if ($certificate_path === NULL) {
      throw new ApnsPhpCertificateNotFoundException(
        "Failed to get APNs certificate path!",
      );
    }
    $certificate_secret = $config->get(ApnsPhpSettings::CertificateSecret->value);
    $is_production = (bool) $config->get(ApnsPhpSettings::UseProduction->value);

    $options = [
      'key_id' => $key_id,
      'team_id' => $team_id,
      'app_bundle_id' => $app_bundle_id,
      'private_key_path' => $certificate_path,
      'private_key_secret' => $certificate_secret ?? NULL,
    ];

    $this->authProvider = AuthProviderToken::create($options);

    $this->client = new Client($this->authProvider, $is_production);

    $this->loggingLevel = LoggingLevel::tryFrom((string) $config->get(ApnsPhpSettings::LoggingLevel->value)) ?? LoggingLevel::Standard;

    $this->config = $config;
  }

}
