<?php

declare(strict_types=1);

namespace Drupal\apns_php\Form;

use Drupal\apns_php\Enum\ApnsPhpSettings;
use Drupal\apns_php\Enum\LoggingLevel;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configures the APNs PHP module.
 */
class ApnsPhpConfigurationForm extends ConfigFormBase {

  const FORM_ID = 'apns_php.settings';

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getFormId(): string {
    return self::FORM_ID;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  protected function getEditableConfigNames(): array {
    return [self::FORM_ID];
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config(self::FORM_ID);

    $form[self::FORM_ID] = [
      '#type' => 'details',
      '#title' => $this->t('Configure APNs'),
      '#open' => TRUE,
    ];

    $form[self::FORM_ID][ApnsPhpSettings::AppBundleId->value] = [
      '#type' => 'textfield',
      '#title' => $this->t('App Bundle ID'),
      '#description' => $this->t(
        'The app bundle id (for example, com.example.app).'),
      '#default_value' => $config->get(ApnsPhpSettings::AppBundleId->value),
      '#required' => TRUE,
    ];

    $form[self::FORM_ID][ApnsPhpSettings::AppleDeveloperTeamId->value] = [
      '#type' => 'textfield',
      '#title' => $this->t('Apple Developer Team ID'),
      '#description' => $this->t(
        'Your Apple Developer team ID from App Store Connect.'),
      '#default_value' => $config->get(ApnsPhpSettings::AppleDeveloperTeamId->value),
      '#required' => TRUE,
    ];

    $form[self::FORM_ID][ApnsPhpSettings::CertificateKeyId->value] = [
      '#type' => 'textfield',
      '#title' => $this->t('Apple Developer Key ID'),
      '#description' => $this->t(
        'The key ID for the certificate (found on App Store Connect).'),
      '#default_value' => $config->get(ApnsPhpSettings::CertificateKeyId->value),
      '#required' => TRUE,
    ];

    $form[self::FORM_ID][ApnsPhpSettings::CertificatePath->value] = [
      '#type' => 'textfield',
      '#title' => $this->t('Certificate Path'),
      '#description' => $this->t(
        'The path to the certificate file. SECURITY WARNING: This file MUST be outside the Drupal webroot. The path may be absolute (e.g., %abs), relative to the Drupal directory (e.g., %rel), or defined using a stream wrapper (e.g., %str).', [
          '%abs' => '/etc/foobar.json',
          '%rel' => '../apns/foobar.json',
          '%str' => 'private://apns/foobar.json',
        ]),
      '#default_value' => $config->get(ApnsPhpSettings::CertificatePath->value),
      '#required' => TRUE,
    ];

    $form[self::FORM_ID][ApnsPhpSettings::LoggingLevel->value] = [
      '#type' => 'select',
      '#title' => $this->t('Logging Level'),
      '#description' => $this->t('Choose the level of detail to be logged to Drupal.'),
      '#options' => $this->getLoggingOptions(),
      '#default_value' => $config->get(ApnsPhpSettings::LoggingLevel->value),
      '#required' => TRUE,
    ];

    $form[self::FORM_ID][ApnsPhpSettings::LogTokenValidationResult->value] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log the result when validating a token'),
      '#description' => $this->t('Logs the validation result to the database when a token is validated.'),
      '#default_value' => $config->get(ApnsPhpSettings::LogTokenValidationResult->value),
      '#required' => FALSE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config(self::FORM_ID);
    $config
      ->set(ApnsPhpSettings::CertificatePath->value, $form_state->getValue(ApnsPhpSettings::CertificatePath->value))
      ->set(ApnsPhpSettings::CertificateSecret->value, $form_state->getValue(ApnsPhpSettings::CertificatePath->value))
      ->set(ApnsPhpSettings::AppBundleId->value, $form_state->getValue(ApnsPhpSettings::AppBundleId->value))
      ->set(ApnsPhpSettings::CertificateKeyId->value, $form_state->getValue(ApnsPhpSettings::CertificateKeyId->value))
      ->set(ApnsPhpSettings::AppleDeveloperTeamId->value, $form_state->getValue(ApnsPhpSettings::AppleDeveloperTeamId->value))
      ->set(ApnsPhpSettings::LoggingLevel->value, $form_state->getValue(ApnsPhpSettings::LoggingLevel->value))
      ->set(ApnsPhpSettings::LogTokenValidationResult->value, $form_state->getValue(ApnsPhpSettings::LogTokenValidationResult->value))
      ->set(ApnsPhpSettings::UseProduction->value, $form_state->getValue(ApnsPhpSettings::UseProduction->value))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $settings = $form_state->getValues();
    $file_path = $settings[ApnsPhpSettings::CertificatePath->value];

    // Does the file exist?
    if (!is_file($file_path)) {
      $form_state->setErrorByName('file_location', $this->t('There is no file at the specified location.'));
      return;
    }

    // Is the file readable?
    if ((!is_readable($file_path))) {
      $form_state->setErrorByName('file_location', $this->t('The file at the specified location is not readable.'));
      return;
    }

    // Is it obviously insecure?
    if (str_starts_with('public://', $file_path)) {
      $form_state->setErrorByName('path_invalid', $this->t('You cannot use a public file. That is insecure.'));
    }

  }

  /**
   * Get the logging level options.
   *
   * @return array
   *   List of options, keyed by logging level.
   */
  protected function getLoggingOptions(): array {
    return [
      LoggingLevel::Debug->value => LoggingLevel::Debug->value,
      LoggingLevel::Standard->value => LoggingLevel::Standard->value,
      LoggingLevel::None->value => LoggingLevel::None->value,
    ];
  }

}
