<?php

namespace Drupal\assets_for_text;

use Drupal\Core\Asset\LibraryDiscoveryInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Extension\ProfileExtensionList;
use Drupal\Core\Extension\ThemeExtensionList;

/**
 * Methods to get available libraries and load our keyword-library pairings.
 */
class Libraries {

  /**
   * Creates a Libraries instance.
   *
   * @param \Drupal\Core\Extension\ModuleExtensionList $module_extension_list
   *   The module extension list service.
   * @param \Drupal\Core\Extension\ThemeExtensionList $theme_extension_list
   *   The theme extension list service.
   * @param \Drupal\Core\Extension\ProfileExtensionList $profile_extension_list
   *   The profile extension list service.
   * @param \Drupal\Core\Asset\LibraryDiscovery $library_discovery
   *   The library discovery service.
   */
  public function __construct(
    protected ModuleExtensionList $module_extension_list,
    protected ThemeExtensionList $theme_extension_list,
    protected ProfileExtensionList $profile_extension_list,
    protected LibraryDiscoveryInterface $library_discovery,
  ) {
    // No need to set properties courtesy PHP 8 constructor property promotion.
  }

  /**
   * Return a list of installed modules, themes, and profiles.
   *
   * This excludes theme engines and other possible types of extensions.
   */
  public function listInstalledExtensions(): array {
    $extensions = [];
    $extensions += array_keys($this->module_extension_list->getAllInstalledInfo());
    $extensions += array_keys($this->theme_extension_list->getAllInstalledInfo());
    $extensions += array_keys($this->profile_extension_list->getAllInstalledInfo());
    return $extensions;
  }

  /**
   * Get all libraries, grouped by extension, with info.
   */
  public function getAllLibrariesByExtensionWithInfo(): array {
    // Start our list with the crucial not-actually-an-extension 'core'.
    $extensions = ['core'];
    $extensions += $this->listInstalledExtensions();
    foreach ($extensions as $extension) {
      // \Drupal::service('library.discovery')
      $libraries[$extension] = $this->library_discovery->getLibrariesByExtension($extension);
    }
    return $libraries;
  }

  /**
   * Get all libraries by name only.
   */
  public function getAllLibraryNames(): array {
    $libs = $this->getAllLibrariesByExtensionWithInfo();
    foreach ($libs as $ext => $ext_libs) {
      foreach ($ext_libs as $lib_name => $lib) {
        unset($lib);
        $lib_id = $ext . '/' . $lib_name;
        $lib_names[$lib_id] = $lib_id;
      }
    }
    return $lib_names;
  }

}
