<?php

namespace Drupal\assets_for_text\Plugin\Filter;

use DOMXPath;
use Drupal\Component\Utility\Html;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\filter\Attribute\Filter;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;
use Drupal\filter\Plugin\FilterInterface;

/**
 * Provides filter to add configure asset libraries (JS, CSS) per keyword.
 */
#[Filter(
  id: "assets_for_text_filter",
  title: new TranslatableMarkup("Assets For Text Filter"),
  type: FilterInterface::TYPE_TRANSFORM_REVERSIBLE,
  description: new TranslatableMarkup("Provides filter to add configure asset libraries (JS, CSS) per keyword."),
  weight: "242",
  status: "TRUE",
)]
class AssetsForTextFilter extends FilterBase {

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode) {
    // We do not want to process the text at all, only attach libraries.
    $result = new FilterProcessResult($text);
    $libraries = [];

    $dom = Html::load($text);
    $keys_libraries = \Drupal::service('assets_for_text.key_library_config')->listKeyLibraryPairs();
    foreach ($keys_libraries as $id => $key_library) {
      $key = $key_library['key'];
      $library = $key_library['library'];
      if (stripos($key, '#') === 0) {
        $key = substr($key, 1);
        $match = (bool) $dom->getElementById($key);
      }
      elseif (stripos($key, '.') === 0) {
        $key = substr($key, 1);
        $xpath = new DOMXPath($dom);
        $dom_nodes = $xpath->query('//*[contains(@class, "' . $key . '")]');
        $match = (bool) count($dom_nodes);
      }
      else {
        // Key was invalid.
        $match = FALSE;
        \Drupal::messenger()->addWarning(t("Invalid key for Assets for Text filter: :key", [':key' => $key]));
        \Drupal::logger('assets_for_text')->warning("Invalid key: :key", [':key' => $key]);
      }
      if ($match) {
        $libraries[] = $library;
      }
    }

    if ($libraries) {
      // We started with a plain string so there should never be an occasion
      // of needing to first check `$result->getAttachments()`
      $result->setAttachments([
        'library' => $libraries,
      ]);
    }

    return $result;
  }

}
