<?php

declare(strict_types=1);

namespace Drupal\Tests\atd_test\Kernel;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\KernelTests\KernelTestBase;
use Drupal\language\Entity\ConfigurableLanguage;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the ATD translation discovery functionality.
 */
#[Group('atd')]
class AtdTranslationDiscoveryKernelTest extends KernelTestBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['language', 'locale', 'user'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig(['language', 'locale']);
    $this->installSchema('locale', ['locales_source', 'locales_target', 'locales_location', 'locale_file']);

    // Add a fake test language, which is used by the atd_test module to
    // provide a custom translation for a string.
    ConfigurableLanguage::createFromLangcode('netl')->save();
  }

  /**
   * Tests the discovery of custom string translations based on enabled modules.
   *
   * @param string[] $modules
   *   An array of modules that should be installed.
   * @param string $expected_translation
   *   The expected result of translating the test string.
   */
  #[DataProvider('discoveryOfCustomStringTranslationsProvider')]
  public function testDiscoveryOfCustomStringTranslations(array $modules, string $expected_translation): void {
    // Install all provided modules.
    $this->container->get('module_installer')->install($modules);

    $this->processAllBatchOperations();

    // Translate the test string to the non-existing test language.
    $translated = $this->t('This is a test string.', [], ['langcode' => 'netl']);

    // Assert that the translation matches the expected result.
    $this->assertEquals($expected_translation, $translated);
  }

  /**
   * Provides data to test translating custom strings provided by .po file.
   *
   * @return \Generator
   *   Yields test cases, consisting of:
   *   - An array of modules that should be installed.
   *   - The expected translation of the test string.
   */
  public static function discoveryOfCustomStringTranslationsProvider(): \Generator {
    yield 'no modules enabled should not translate the string' => [
      [],
      'This is a test string.',
    ];

    yield 'only atd module enabled should not translate the string' => [
      ['atd'],
      'This is a test string.',
    ];

    yield 'only atd_test module enabled should not translate the string' => [
      ['atd_test'],
      'This is a test string.',
    ];

    yield 'both atd and atd_test modules enabled should translate the string' => [
      ['atd', 'atd_test'],
      'Dit is een testzin.',
    ];

    yield 'both atd_test and atd modules enabled should translate the string' => [
      ['atd_test', 'atd'],
      'Dit is een testzin.',
    ];
  }

  /**
   * Processes any pending batch operations.
   */
  private function processAllBatchOperations(): void {
    // Process any pending batch operations.
    $batch =& batch_get();
    if (!empty($batch)) {
      // Mark this batch as non-progressive to bypass the progress bar and
      // redirect.
      $batch['progressive'] = FALSE;
      batch_process();
    }
  }

}
