<?php

namespace Drupal\audio_embed_media_core\Form;

use Drupal\audio_embed_field\Plugin\Field\FieldWidget\AudioTextfield;
use Drupal\audio_embed_field\ProviderManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\media_library\Form\AddFormBase;
use Drupal\media_library\MediaLibraryUiBuilder;
use Drupal\media_library\OpenerResolverInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form to create audio embed media items from within the media library.
 */
class AudioEmbedFieldMediaLibraryForm extends AddFormBase {

  /**
   * Constructs an AddFormBase object.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    MediaLibraryUiBuilder $libraryUiBuilder,
    OpenerResolverInterface $openerResolver,
    protected ProviderManagerInterface $providerManager,
  ) {
    parent::__construct($entityTypeManager, $libraryUiBuilder, $openerResolver);
    $this->providerManager = $providerManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('media_library.ui_builder'),
      $container->get('media_library.opener_resolver'),
      $container->get('audio_embed_field.provider_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return $this->getBaseFormId() . '_media_remote';
  }

  /**
   * {@inheritdoc}
   */
  protected function buildInputElement(array $form, FormStateInterface $formState) {
    // Add a container to group the input elements for styling purposes.
    $form['container'] = [
      '#type' => 'container',
    ];

    $form['container']['url'] = [
      '#type' => 'url',
      '#title' => $this->t('URL'),
      '#required' => TRUE,
      '#description' => $this->t('Enter the URL of the audio file you want to embed.'),
      '#attributes' => [
        'placeholder' => 'https://',
      ],
    ];

    $form['container']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add'),
      '#button_type' => 'primary',
      '#validate' => ['::validateUrl'],
      '#submit' => ['::addButtonSubmit'],
      '#ajax' => [
        'callback' => '::updateFormCallback',
        'wrapper' => 'media-library-wrapper',
        'url' => Url::fromRoute('media_library.ui'),
        'options' => [
          'query' => $this->getMediaLibraryState($formState)->all() + [
            FormBuilderInterface::AJAX_FORM_REQUEST => TRUE,
          ],
        ],
      ],
    ];

    return $form;
  }

  /**
   * Submit handler for the add button.
   *
   * @param array $form
   *   The form render array.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The form state.
   */
  public function addButtonSubmit(array $form, FormStateInterface $formState) {
    $this->processInputValues([$formState->getValue('url')], $form, $formState);
  }

  /**
   * Validates the Media Remote URL.
   *
   * @param array $form
   *   The complete form.
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *   The current form state.
   */
  public function validateUrl(array &$form, FormStateInterface $formState) {
    $url = $formState->getValue('url');
    $mediaType = $this->getMediaType($formState);
    $mediaStorage = $this->entityTypeManager->getStorage('media');
    $sourceFieldName = $this->getSourceFieldName($mediaType);
    $media = $this->createMediaFromValue($mediaType, $mediaStorage, $sourceFieldName, $url);
    $options = $mediaType->getSource()->getSourceFieldDefinition($mediaType)->getSetting('allowed_providers') ?? [];
    $enabledProviders = $this->providerManager->loadDefinitionsFromOptionList($options);

    if (!$this->providerManager->filterApplicableDefinitions($enabledProviders, $url)) {
      $formState->setErrorByName('url', AudioTextfield::getProviderErrorMessage());
    }

    $violations = $media->validate();

    if ($violations->count() > 0) {
      /** @var \Symfony\Component\Validator\ConstraintViolation $violation */
      foreach ($violations as $violation) {
        $formState->setErrorByName('url', $violation->getMessage());
      }
    }
  }

}
