<?php

namespace Drupal\audio_embed_field;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\State\StateInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Log\LoggerInterface;

/**
 * Service for managing SoundCloud OAuth 2.1 authentication.
 */
class SoundCloudOAuthService {

  /**
   * The HTTP client.
   */
  protected ClientInterface $httpClient;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The state service.
   */
  protected StateInterface $state;

  /**
   * The logger service.
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a SoundCloudOAuthService object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(ClientInterface $http_client, ConfigFactoryInterface $config_factory, StateInterface $state, LoggerInterface $logger) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->state = $state;
    $this->logger = $logger;
  }

  /**
   * Gets a valid access token, refreshing if necessary.
   *
   * @return string|null
   *   The access token, or NULL if unable to obtain one.
   */
  public function getAccessToken(): ?string {
    $token_data = $this->state->get('audio_embed_field.soundcloud_token');

    // Check if we have a valid token.
    if ($token_data && isset($token_data['access_token'], $token_data['expires_at'])) {
      // Add a 5-minute buffer before expiration.
      if ($token_data['expires_at'] > (time() + 300)) {
        return $token_data['access_token'];
      }
    }

    // Need to fetch a new token.
    return $this->fetchNewToken();
  }

  /**
   * Fetches a new access token from SoundCloud using OAuth 2.1.
   *
   * @return string|null
   *   The access token, or NULL on failure.
   */
  protected function fetchNewToken(): ?string {
    $config = $this->configFactory->get('audio_embed_field.settings');
    $client_id = $config->get('soundcloud_id');
    $client_secret = $config->get('soundcloud_secret');

    if (empty($client_id) || empty($client_secret)) {
      $this->logger->error('SoundCloud OAuth credentials not configured.');
      return NULL;
    }

    try {
      $response = $this->httpClient->request('POST', 'https://api.soundcloud.com/oauth2/token', [
        'form_params' => [
          'grant_type' => 'client_credentials',
          'client_id' => $client_id,
          'client_secret' => $client_secret,
        ],
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);

      if (isset($data['access_token'], $data['expires_in'])) {
        $token_data = [
          'access_token' => $data['access_token'],
          'expires_at' => time() + $data['expires_in'],
        ];

        $this->state->set('audio_embed_field.soundcloud_token', $token_data);
        return $data['access_token'];
      }

      $this->logger->error('Invalid response from SoundCloud OAuth endpoint.');
      return NULL;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to fetch SoundCloud access token: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Clears the stored access token.
   */
  public function clearToken(): void {
    $this->state->delete('audio_embed_field.soundcloud_token');
  }

}
