<?php

namespace Drupal\audio_embed_field\Plugin\audio_embed_field\Provider;

use Drupal\Component\Utility\DeprecationHelper;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Utility\Error;
use Drupal\audio_embed_field\ProviderPluginBase;
use Drupal\audio_embed_field\SoundCloudOAuthService;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A SoundCloud provider plugin.
 *
 * @AudioEmbedProvider(
 *   id = "soundcloud",
 *   title = @Translation("SoundCloud")
 * )
 */
class SoundCloud extends ProviderPluginBase {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Logger service.
   */
  protected LoggerInterface $logger;

  /**
   * The SoundCloud OAuth service.
   *
   * @var \Drupal\audio_embed_field\SoundCloudOAuthService
   */
  protected SoundCloudOAuthService $oauthService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->configFactory = $container->get('config.factory');
    $instance->logger = $container->get('logger.channel.audio_embed_field');
    $instance->oauthService = $container->get('audio_embed_field.soundcloud_oauth');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function renderEmbedCode($width, $height, $autoplay) {
    if ($autoplay == 0) {
      $autoplay = 'false';
    }
    if ($autoplay == 1) {
      $autoplay = 'true';
    }
    $embed_code = [
      '#type' => 'audio_embed_iframe',
      '#provider' => 'soundcloud',
      '#url' => sprintf('https://w.soundcloud.com/player/?url=https%%3A//api.soundcloud.com/tracks/%s', $this->getAudioId()),
      '#query' => [
        'auto_play' => $autoplay,
        'visual' => 'true',
        'show_user' => 'false',
        'show_reposts' => 'false',
        'hide_related' => 'true',
        'show_comments' => 'false',
      ],
      '#attributes' => [
        'width' => $width,
        'height' => $height,
        'frameborder' => '0',
      ],
    ];

    return $embed_code;
  }

  /**
   * {@inheritdoc}
   */
  public function getRemoteThumbnailUrl() {
    $access_token = $this->oauthService->getAccessToken();
    if (empty($access_token)) {
      return NULL;
    }

    try {
      $client = new Client();
      $res = $client->request('GET', 'https://api.soundcloud.com/resolve', [
        'query' => [
          'url' => $this->getInput(),
        ],
        'headers' => [
          'Authorization' => 'OAuth ' . $access_token,
        ],
      ]);
      return json_decode($res->getBody())->artwork_url;
    }
    catch (ClientException $e) {
      if (version_compare(\Drupal::VERSION, '10.1', '>=')) {
        Error::logException($this->logger, $e);
      }
      else {
        // @phpstan-ignore-next-line
        DeprecationHelper::backwardsCompatibleCall(\Drupal::VERSION, '10.1.0', fn() => Error::logException(\Drupal::logger('audio_embed_field'), $e), fn() => watchdog_exception('audio_embed_field', $e));
      }
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getIdFromInput($input) {
    // Generate cache key from input URL.
    $cache_key = 'soundcloud_id:' . md5($input);
    $cache = \Drupal::cache('default');

    // Check if we have a cached ID.
    if ($cached = $cache->get($cache_key)) {
      return $cached->data;
    }

    /** @var \Drupal\audio_embed_field\SoundCloudOAuthService $oauth_service */
    $oauth_service = \Drupal::service('audio_embed_field.soundcloud_oauth');
    $access_token = $oauth_service->getAccessToken();

    if (empty($access_token)) {
      return NULL;
    }

    try {
      $client = new Client();
      $res = $client->request('GET', 'https://api.soundcloud.com/resolve', [
        'query' => [
          'url' => $input,
        ],
        'headers' => [
          'Authorization' => 'OAuth ' . $access_token,
        ],
      ]);
      $id = json_decode($res->getBody())->id;

      // Cache the ID for 30 days.
      $cache->set($cache_key, $id, time() + (86400 * 30));

      return $id;
    }
    catch (ClientException $e) {
      if (version_compare(\Drupal::VERSION, '10.1', '>=')) {
        Error::logException(\Drupal::logger('audio_embed_field'), $e);
      }
      else {
        // @phpstan-ignore-next-line
        DeprecationHelper::backwardsCompatibleCall(\Drupal::VERSION, '10.1.0', fn() => Error::logException(\Drupal::logger('audio_embed_field'), $e), fn() => watchdog_exception('audio_embed_field', $e));
      }
      return NULL;
    }

  }

}
