# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Module Overview

**audit** is a Drupal 10 module for auditing and analyzing Drupal sites. It provides:
- Plugin-based architecture (`AuditAnalyzer`) for extensible analyzers
- Real-time analysis on page view (no caching)
- Score storage in State API for dashboard display
- Centralized configuration form for all active submodules

## Commands

```bash
# Enable the module and submodules
drush en audit audit_status audit_modules audit_fields

# Clear cache after changes
drush cr

# Export config
drush cex

# Run phpcs for coding standards
phpcs --standard=Drupal,DrupalPractice web/modules/contrib/audit

# Run phpstan
phpstan analyse web/modules/contrib/audit
```

## Architecture

```
audit/
├── audit.info.yml
├── audit.module                          # hook_help
├── audit.services.yml
├── audit.routing.yml
├── audit.permissions.yml
├── audit.links.menu.yml
├── audit.links.task.yml                  # Primary tabs
├── config/
│   ├── install/audit.settings.yml
│   └── schema/audit.schema.yml
├── src/
│   ├── AuditAnalyzerInterface.php
│   ├── AuditAnalyzerBase.php
│   ├── AuditAnalyzerPluginManager.php
│   ├── Attribute/AuditAnalyzer.php
│   ├── Controller/AuditResultsController.php
│   ├── Form/AuditSettingsForm.php
│   └── Service/
│       ├── AuditRunner.php               # Executes analyzers
│       ├── AuditScoreStorage.php         # Stores scores in State API
│       └── AuditComponentBuilder.php     # UI component builder
└── modules/                              # Submodules
    ├── audit_status/
    ├── audit_modules/
    ├── audit_fields/
    ├── audit_twig/
    ├── audit_views/
    ├── audit_i18n/
    ├── audit_seo/
    ├── audit_performance/
    ├── audit_database/
    ├── audit_images/
    ├── audit_updates/
    └── audit_blocks/
```

## How It Works

### Data Flow (No File Storage)

1. **User visits detail page** (`/admin/reports/audit/{analyzer_id}`)
2. **Controller** calls `AuditRunner::runAnalyzer()`
3. **Analyzer** executes `analyze()` method in real-time
4. **Results** are returned as structured array (not saved to files)
5. **Score only** is saved to State API via `AuditScoreStorage`
6. **UI** renders results immediately using `buildDetailedResults()`

### The `_files` Structure

The `_files` key in analyzer results is **NOT for file storage**. It's an internal structure to organize results by section:

```php
return [
  '_files' => [
    'section1' => ['summary' => [...], 'results' => [...]],
    'section2' => ['summary' => [...], 'results' => [...]],
  ],
  'score' => ['factors' => [...]],
];
```

This allows `getAuditChecks()` to map sections to their data via `file_key`.

### What Gets Stored

Only **scores** are persisted (in State API):
- Individual analyzer scores with factors
- Project score (weighted average of all analyzers)
- Timestamps for last run

Results are **regenerated on each page view**.

## Plugin System: AuditAnalyzer

### Creating a New Analyzer

1. Create submodule in `modules/audit_example/`
2. Add `.info.yml` with dependency on `audit:audit`
3. Create plugin in `src/Plugin/AuditAnalyzer/ExampleAnalyzer.php`

```php
<?php

declare(strict_types=1);

namespace Drupal\audit_example\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;

#[AuditAnalyzer(
  id: 'example',
  label: new TranslatableMarkup('Example Analyzer'),
  description: new TranslatableMarkup('Description of what this analyzes.'),
  menu_title: new TranslatableMarkup('Example'),
  weight: 3,
)]
class ExampleAnalyzer extends AuditAnalyzerBase {

  protected const SCORE_WEIGHTS = [
    'factor1' => 60,
    'factor2' => 40,
  ];

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    // Gather data and detect issues.
    $results = [];
    $results[] = $this->createResultItem(
      'warning',           // 'error', 'warning', or 'notice'
      'EXAMPLE_CODE',      // Unique code
      'Example message',   // Human-readable message
      ['key' => 'value']   // Additional details
    );

    // Return structure with sections.
    return [
      '_files' => [
        'section1' => $this->createResult($results, $errors, $warnings, $notices),
        'section2' => $this->createResult([], 0, 0, 0),
      ],
      'score' => [
        'factors' => [
          'factor1' => [
            'score' => 85,
            'weight' => self::SCORE_WEIGHTS['factor1'],
            'label' => 'Factor 1',
            'description' => 'Description of factor 1',
          ],
          'factor2' => [
            'score' => 100,
            'weight' => self::SCORE_WEIGHTS['factor2'],
            'label' => 'Factor 2',
            'description' => 'All good',
          ],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      'section1' => [
        'label' => $this->t('Section 1'),
        'description' => $this->t('Description.'),
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['factor1'],
        'file_key' => 'section1',
        'score_factor_key' => 'factor1',
      ],
      'section2' => [
        'label' => $this->t('Section 2 (Informational)'),
        'description' => $this->t('Informational only.'),
        'affects_score' => FALSE,
        'weight' => 0,
        'file_key' => 'section2',
        'score_factor_key' => NULL,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    $files = $data['_files'] ?? [];

    return match ($check_id) {
      'section1' => $this->buildSection1Content($files),
      'section2' => $this->buildSection2Content($files),
      default => [],
    };
  }

  protected function buildSection1Content(array $files): array {
    $sectionData = $files['section1'] ?? [];

    // Use buildIssueListFromResults for faceted items.
    return $this->ui->buildIssueListFromResults(
      $sectionData['results'] ?? [],
      'Success message when no issues.',
      function (array $item, $ui): array {
        $details = $item['details'] ?? [];
        return [
          'severity' => $ui->normalizeSeverity($item['severity'] ?? 'warning'),
          'code' => $item['code'] ?? 'CODE',
          'file' => $details['identifier'] ?? '',  // Shown in header
          'label' => $item['message'] ?? '',       // Shown on expand
          'description' => ['#markup' => '<p>Fix instructions...</p>'],
          'tags' => ['performance', 'cache'],
        ];
      }
    );
  }

  protected function buildSection2Content(array $files): array {
    // Informational sections use tables.
    $headers = [
      $this->ui->header('Column 1'),
      $this->ui->header('Column 2'),
    ];
    $rows = [
      $this->ui->row([
        $this->ui->cell('Value 1'),
        $this->ui->cell('Value 2'),
      ]),
    ];
    return $this->ui->table($headers, $rows);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $config): array {
    return [
      'my_option' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable option'),
        '#default_value' => $config['my_option'] ?? FALSE,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function checkRequirements(): array {
    $warnings = [];
    if (!file_exists('/path/to/required/tool')) {
      $warnings[] = (string) $this->t('Tool not installed');
    }
    return $warnings;
  }

}
```

## Services

| Service | Purpose |
|---------|---------|
| `plugin.manager.audit_analyzer` | Discovers and manages analyzer plugins |
| `audit.runner` | Executes analyzers and saves scores |
| `audit.score_storage` | Stores scores in State API |
| `audit.component_builder` | UI component builder for tables, messages, etc. |
| `logger.channel.audit` | Logging channel |

## Routes and Permissions

| Route | Path | Permission |
|-------|------|------------|
| `audit.reports` | `/admin/reports/audit` | `view audit results` |
| `audit.settings` | `/admin/reports/audit/settings` | `administer audit configuration` |
| `audit.reports.detail` | `/admin/reports/audit/{analyzer_id}` | `view audit results` |
| `audit.run_all` | `/admin/reports/audit/run-all` | `view audit results` |

## UI Components (AuditComponentBuilder)

The `$this->ui` property in analyzers provides these methods:

### For Faceted Issue Lists
- `buildIssueListFromResults($results, $successMessage, $callback)` - Renders faceted items with filters

### For Tables
- `table($headers, $rows, $options)` - Render a table
- `header($label, $align)` - Create header cell
- `row($cells, $severity)` - Create table row
- `cell($content, $options)` - Create table cell
- `itemName($label, $machineName)` - Label with machine name below

### For Messages
- `message($text, $type)` - Status message (success, warning, error, info)

### For Scores
- `score($value, $label, $size)` - Score circle

## Development Guidelines

- All code, comments, and user-facing text must be in English
- Follow Drupal coding standards (Drupal, DrupalPractice)
- Use `declare(strict_types=1);` in all PHP files
- Use PHP 8 Attributes for plugins (not Annotations)
- Implement `ContainerFactoryPluginInterface` for dependency injection
- Sections with `affects_score: TRUE` should use `buildIssueListFromResults()` for faceted display
- Sections with `affects_score: FALSE` (informational) should use tables
- Always include `tags` in faceted items for filtering
- Use `checkRequirements()` to validate external tools (phpstan, phpcs, rector)
