/**
 * @file
 * Audit faceted filters for issue lists.
 *
 * This filter system supports both built-in filters and custom filters
 * defined by submodules. Custom filters are passed via the data-custom-filters
 * attribute on the container element as a JSON object.
 *
 * Custom filter format:
 * {
 *   "filter_key": {
 *     "label": "Filter Label",
 *     "attribute": "data-attribute-name",
 *     "styled": true,  // optional, enables color styling
 *     "styles": {      // optional, custom styles for values
 *       "value1": { "label": "Display Label", "bg": "#ffffff", "text": "#212529" }
 *     }
 *   }
 * }
 */

(function (Drupal) {
  'use strict';

  /**
   * Default style for unstyled filter values (white background like Issue Type).
   */
  const DEFAULT_STYLE = { bg: '#ffffff', text: '#212529' };

  /**
   * Centralized style configuration for built-in filter values.
   * Defines labels, background colors, and text colors for each value.
   * Text colors are dark for readability.
   */
  const FILTER_STYLES = {
    // Severity levels
    severity: {
      error: { label: Drupal.t('Error'), bg: '#ffcdd2', text: '#4a1515' },
      warning: { label: Drupal.t('Warning'), bg: '#fff3e0', text: '#4a2800' },
      notice: { label: Drupal.t('Notice'), bg: '#e3f2fd', text: '#0d3c61' },
      info: { label: Drupal.t('Info'), bg: '#f5f5f5', text: '#424242' }
    },
    // Category tags
    tags: {
      security: { label: Drupal.t('Security'), bg: '#ffcdd2', text: '#4a1515' },
      performance: { label: Drupal.t('Performance'), bg: '#ffe0b2', text: '#4a2800' },
      cache: { label: Drupal.t('Cache'), bg: '#bbdefb', text: '#0d3c61' },
      maintainability: { label: Drupal.t('Maintainability'), bg: '#e1bee7', text: '#3d1a4a' },
      maintenance: { label: Drupal.t('Maintenance'), bg: '#b0bec5', text: '#263238' },
      complexity: { label: Drupal.t('Complexity'), bg: '#d1c4e9', text: '#4a148c' },
      configuration: { label: Drupal.t('Configuration'), bg: '#b2dfdb', text: '#004d40' },
      cleanup: { label: Drupal.t('Cleanup'), bg: '#b3e5fc', text: '#01579b' },
      best_practice: { label: Drupal.t('Best Practice'), bg: '#c8e6c9', text: '#1a3d1c' },
      deprecation: { label: Drupal.t('Deprecation'), bg: '#e0e0e0', text: '#2a2a2a' },
      debug: { label: Drupal.t('Debug'), bg: '#ffebee', text: '#4a1515' },
      privacy: { label: Drupal.t('Privacy'), bg: '#f8bbd0', text: '#4a0d2a' },
      reliability: { label: Drupal.t('Reliability'), bg: '#fff8e1', text: '#4a3800' },
      integrity: { label: Drupal.t('Integrity'), bg: '#f3e5f5', text: '#4a148c' },
      production: { label: Drupal.t('Production'), bg: '#ffebee', text: '#b71c1c' },
      missing: { label: Drupal.t('Missing'), bg: '#fff3e0', text: '#e65100' },
      development: { label: Drupal.t('Development'), bg: '#fce4ec', text: '#880e4f' },
      security_risk: { label: Drupal.t('Security Risk'), bg: '#ffcdd2', text: '#c62828' },
      required: { label: Drupal.t('Required'), bg: '#ffcdd2', text: '#b71c1c' },
      recommended: { label: Drupal.t('Recommended'), bg: '#fff3e0', text: '#e65100' },
      contrib: { label: Drupal.t('Contrib'), bg: '#e3f2fd', text: '#0d47a1' },
      custom: { label: Drupal.t('Custom'), bg: '#e8f5e9', text: '#1b5e20' }
    },
    // File types
    'file-type': {
      twig: { label: 'Twig', bg: '#e8f5e9', text: '#1a3d1c' },
      php: { label: 'PHP', bg: '#e3f2fd', text: '#0d3c61' },
      yml: { label: 'YAML', bg: '#fce4ec', text: '#4a1535' },
      yaml: { label: 'YAML', bg: '#fce4ec', text: '#4a1535' },
      js: { label: 'JavaScript', bg: '#fff3e0', text: '#4a2800' },
      css: { label: 'CSS', bg: '#e0f7fa', text: '#004d57' },
      html: { label: 'HTML', bg: '#fff8e1', text: '#4a3800' },
      module: { label: 'Module', bg: '#ede7f6', text: '#2a1a4a' },
      theme: { label: 'Theme', bg: '#e8f5e9', text: '#1a3d1c' },
      inc: { label: 'Include', bg: '#e3f2fd', text: '#0d3c61' }
    },
    // Code snippet presence - white background, black text (like Issue Type)
    'has-code': {
      '1': { label: Drupal.t('With code'), bg: '#ffffff', text: '#212529' },
      '0': { label: Drupal.t('Without code'), bg: '#ffffff', text: '#212529' }
    }
  };

  /**
   * Base filter configuration - defines the built-in filter groups.
   * Order matters: filters are rendered in the order defined here.
   * Custom filters from submodules are inserted before the 'code' filter.
   */
  const BASE_FILTER_CONFIG = {
    severity: {
      label: Drupal.t('Severity'),
      attribute: 'data-severity',
      order: ['error', 'warning', 'notice', 'info'],
      styled: true
    },
    tags: {
      label: Drupal.t('Category'),
      attribute: 'data-tags',
      isMultiValue: true,
      styled: true
    },
    'file-type': {
      label: Drupal.t('File Type'),
      attribute: 'data-file-type',
      styled: true
    },
    'has-code': {
      label: Drupal.t('Code Snippet'),
      attribute: 'data-has-code',
      styled: true
    },
    code: {
      label: Drupal.t('Issue Type'),
      attribute: 'data-label',
      fullWidth: true,
      useRawLabel: true
    }
  };

  /**
   * Get style configuration for a filter value.
   *
   * @param {string} filterKey - The filter key (e.g., 'severity', 'tags').
   * @param {string} value - The filter value.
   * @param {Object} customStyles - Optional custom styles from filter config.
   * @returns {Object|null} - Style object with label, bg, and text properties.
   */
  function getFilterStyle(filterKey, value, customStyles = null) {
    // Check custom styles first (from submodule config).
    if (customStyles && customStyles[value]) {
      return customStyles[value];
    }
    // Check built-in styles.
    if (FILTER_STYLES[filterKey] && FILTER_STYLES[filterKey][value]) {
      return FILTER_STYLES[filterKey][value];
    }
    return null;
  }

  /**
   * Build filter configuration by merging base config with custom filters.
   *
   * @param {HTMLElement} container - The container element.
   * @returns {Object} - Merged filter configuration.
   */
  function buildFilterConfig(container) {
    // Start with base config without the 'code' filter.
    const config = {};
    const baseKeys = Object.keys(BASE_FILTER_CONFIG);

    // Add all base filters except 'code' first.
    baseKeys.forEach(key => {
      if (key !== 'code') {
        config[key] = { ...BASE_FILTER_CONFIG[key] };
      }
    });

    // Parse custom filters from data attribute.
    const customFiltersJson = container.dataset.customFilters;
    if (customFiltersJson) {
      try {
        const customFilters = JSON.parse(customFiltersJson);
        // Add custom filters (they appear before Issue Type).
        Object.entries(customFilters).forEach(([key, filterConfig]) => {
          config[key] = {
            label: filterConfig.label || key,
            attribute: filterConfig.attribute || `data-${key}`,
            styled: filterConfig.styled !== false, // Default to styled with white bg.
            isMultiValue: filterConfig.isMultiValue || false,
            order: filterConfig.order || null,
            customStyles: filterConfig.styles || null
          };
        });
      } catch (e) {
        console.error('Failed to parse custom filters:', e);
      }
    }

    // Add 'code' (Issue Type) filter last.
    if (BASE_FILTER_CONFIG.code) {
      config.code = { ...BASE_FILTER_CONFIG.code };
    }

    return config;
  }

  /**
   * AuditFilter class - manages filtering for a single issue list.
   */
  class AuditFilter {
    constructor(container) {
      this.container = container;
      this.filtersContainer = container.querySelector('.audit-filters__groups');
      this.itemsContainer = container.querySelector('.audit-issue-list__items');
      this.resetButton = container.querySelector('.audit-filters__reset');
      this.countDisplay = container.querySelector('.audit-filters__count');
      this.noResultsMessage = container.querySelector('.audit-issue-list__no-results');

      if (!this.filtersContainer || !this.itemsContainer) {
        return;
      }

      this.items = Array.from(this.itemsContainer.querySelectorAll('.audit-issue'));
      this.activeFilters = {};
      // Build filter config with any custom filters from the container.
      this.filterConfig = buildFilterConfig(container);

      this.init();
    }

    /**
     * Initialize the filter system.
     */
    init() {
      this.buildFilters();
      this.bindEvents();
      this.updateDisplay();
    }

    /**
     * Extract unique values for a filter dimension.
     */
    extractValues(filterKey, config) {
      const values = new Map();

      this.items.forEach(item => {
        let value = item.getAttribute(config.attribute);

        if (!value) return;

        if (config.isMultiValue) {
          // Split comma-separated values
          value.split(',').forEach(v => {
            v = v.trim();
            if (v) {
              values.set(v, (values.get(v) || 0) + 1);
            }
          });
        } else {
          values.set(value, (values.get(value) || 0) + 1);
        }
      });

      return values;
    }

    /**
     * Build the filter UI.
     */
    buildFilters() {
      this.filtersContainer.innerHTML = '';

      Object.entries(this.filterConfig).forEach(([filterKey, config]) => {
        const values = this.extractValues(filterKey, config);

        // Don't show empty filter groups.
        if (values.size === 0) return;

        // Sort values.
        let sortedValues;
        if (config.order) {
          sortedValues = config.order.filter(v => values.has(v));
        } else {
          sortedValues = Array.from(values.keys()).sort((a, b) => {
            // Sort by count descending, then alphabetically.
            const countDiff = values.get(b) - values.get(a);
            if (countDiff !== 0) return countDiff;
            return a.localeCompare(b);
          });
        }

        if (sortedValues.length === 0) return;

        const group = document.createElement('div');
        group.className = 'audit-filters__group';
        if (config.fullWidth) {
          group.classList.add('audit-filters__group--full-width');
        }
        group.dataset.filterKey = filterKey;

        const groupHeader = document.createElement('div');
        groupHeader.className = 'audit-filters__group-header';
        groupHeader.textContent = config.label;
        group.appendChild(groupHeader);

        const optionsList = document.createElement('div');
        optionsList.className = 'audit-filters__options';

        sortedValues.forEach(value => {
          const count = values.get(value);
          // Get style, checking custom styles first.
          let style = getFilterStyle(filterKey, value, config.customStyles);
          // For styled filters without explicit style, use default white bg.
          if (config.styled && !style && !config.useRawLabel) {
            style = DEFAULT_STYLE;
          }
          let label;

          if (config.useRawLabel) {
            // Use the value directly without formatting.
            label = value;
          } else if (style && style.label) {
            // Use label from style configuration.
            label = style.label;
          } else {
            label = this.formatLabel(value);
          }

          // Apply lowercase if configured.
          if (config.lowercase) {
            label = label.toLowerCase();
          }

          const option = document.createElement('label');
          option.className = 'audit-filters__option';
          if (config.fullWidth) {
            option.classList.add('audit-filters__option--full-text');
          }

          // Apply styled colors if available.
          if (config.styled && style) {
            option.classList.add('audit-filters__option--styled');
            option.style.setProperty('--filter-bg', style.bg);
            option.style.setProperty('--filter-text', style.text);
            option.style.backgroundColor = style.bg;
            option.style.borderColor = style.bg;
          }

          const checkbox = document.createElement('input');
          checkbox.type = 'checkbox';
          checkbox.className = 'audit-filters__checkbox';
          checkbox.dataset.filterKey = filterKey;
          checkbox.dataset.filterValue = value;

          const labelText = document.createElement('span');
          labelText.className = 'audit-filters__label';
          labelText.textContent = label;

          // Apply text color if styled.
          if (config.styled && style) {
            labelText.style.color = style.text;
          }

          const countSpan = document.createElement('span');
          countSpan.className = 'audit-filters__option-count';
          countSpan.textContent = `(${count})`;
          countSpan.dataset.originalCount = count;

          // Apply count color if styled.
          if (config.styled && style) {
            countSpan.style.color = style.text;
            countSpan.style.opacity = '0.7';
          }

          option.appendChild(checkbox);
          option.appendChild(labelText);
          option.appendChild(countSpan);
          optionsList.appendChild(option);
        });

        group.appendChild(optionsList);
        this.filtersContainer.appendChild(group);
      });
    }

    /**
     * Format a machine name as a readable label.
     */
    formatLabel(value) {
      return value
        .replace(/_/g, ' ')
        .replace(/([A-Z])/g, ' $1')
        .replace(/^\s+/, '')
        .split(' ')
        .map(word => word.charAt(0).toUpperCase() + word.slice(1).toLowerCase())
        .join(' ');
    }

    /**
     * Bind event listeners.
     */
    bindEvents() {
      this.filtersContainer.addEventListener('change', (e) => {
        if (e.target.classList.contains('audit-filters__checkbox')) {
          this.handleFilterChange(e.target);
        }
      });

      if (this.resetButton) {
        this.resetButton.addEventListener('click', (e) => {
          e.stopPropagation(); // Prevent closing the details element
          this.resetFilters();
        });
      }
    }

    /**
     * Handle filter checkbox change.
     */
    handleFilterChange(checkbox) {
      const filterKey = checkbox.dataset.filterKey;
      const filterValue = checkbox.dataset.filterValue;
      const option = checkbox.closest('.audit-filters__option');

      if (!this.activeFilters[filterKey]) {
        this.activeFilters[filterKey] = new Set();
      }

      if (checkbox.checked) {
        this.activeFilters[filterKey].add(filterValue);
        option.classList.add('audit-filters__option--active');
      } else {
        this.activeFilters[filterKey].delete(filterValue);
        option.classList.remove('audit-filters__option--active');
        if (this.activeFilters[filterKey].size === 0) {
          delete this.activeFilters[filterKey];
        }
      }

      this.applyFilters();
      this.updateCounts();
      this.updateDisplay();
    }

    /**
     * Apply active filters to show/hide items.
     */
    applyFilters() {
      let visibleCount = 0;

      this.items.forEach(item => {
        const isVisible = this.itemMatchesFilters(item);
        item.style.display = isVisible ? '' : 'none';
        if (isVisible) visibleCount++;
      });

      // Show/hide no results message
      if (this.noResultsMessage) {
        this.noResultsMessage.style.display = visibleCount === 0 ? '' : 'none';
      }

      return visibleCount;
    }

    /**
     * Check if an item matches all active filters.
     */
    itemMatchesFilters(item) {
      for (const [filterKey, activeValues] of Object.entries(this.activeFilters)) {
        if (activeValues.size === 0) continue;

        const config = this.filterConfig[filterKey];
        if (!config) continue;

        const itemValue = item.getAttribute(config.attribute);

        if (!itemValue) return false;

        if (config.isMultiValue) {
          // For multi-value attributes, check if any of the item's values match.
          const itemValues = itemValue.split(',').map(v => v.trim());
          const hasMatch = itemValues.some(v => activeValues.has(v));
          if (!hasMatch) return false;
        } else {
          // For single-value attributes, check if the value matches any active filter.
          if (!activeValues.has(itemValue)) return false;
        }
      }

      return true;
    }

    /**
     * Update counts for all filter options based on currently visible items.
     */
    updateCounts() {
      const visibleItems = this.items.filter(item => item.style.display !== 'none');

      Object.entries(this.filterConfig).forEach(([filterKey, config]) => {
        const counts = new Map();

        visibleItems.forEach(item => {
          let value = item.getAttribute(config.attribute);
          if (!value) return;

          if (config.isMultiValue) {
            value.split(',').forEach(v => {
              v = v.trim();
              if (v) counts.set(v, (counts.get(v) || 0) + 1);
            });
          } else {
            counts.set(value, (counts.get(value) || 0) + 1);
          }
        });

        // Update count displays for this filter group.
        const group = this.filtersContainer.querySelector(`[data-filter-key="${filterKey}"]`);
        if (!group) return;

        group.querySelectorAll('.audit-filters__option').forEach(option => {
          const checkbox = option.querySelector('.audit-filters__checkbox');
          const countSpan = option.querySelector('.audit-filters__option-count');
          const value = checkbox.dataset.filterValue;
          const count = counts.get(value) || 0;
          const originalCount = parseInt(countSpan.dataset.originalCount, 10);
          const isActive = this.activeFilters[filterKey] && this.activeFilters[filterKey].has(value);

          // If this filter is active, show original count.
          if (isActive) {
            countSpan.textContent = `(${originalCount})`;
            option.classList.add('audit-filters__option--active');
          } else {
            countSpan.textContent = `(${count})`;
            option.classList.remove('audit-filters__option--active');
          }

          // Disable options with 0 count (but not if they're checked).
          if (count === 0 && !checkbox.checked) {
            option.classList.add('audit-filters__option--disabled');
          } else {
            option.classList.remove('audit-filters__option--disabled');
          }
        });
      });
    }

    /**
     * Update the display (count, reset button visibility).
     */
    updateDisplay() {
      const visibleCount = this.items.filter(item => item.style.display !== 'none').length;
      const totalCount = this.items.length;
      const hasActiveFilters = Object.keys(this.activeFilters).length > 0;

      if (this.countDisplay) {
        if (hasActiveFilters) {
          this.countDisplay.textContent = Drupal.t('@visible of @total', {
            '@visible': visibleCount,
            '@total': totalCount
          });
        } else {
          this.countDisplay.textContent = Drupal.t('@total items', {
            '@total': totalCount
          });
        }
      }

      if (this.resetButton) {
        this.resetButton.style.display = hasActiveFilters ? '' : 'none';
      }
    }

    /**
     * Reset all filters.
     */
    resetFilters() {
      this.activeFilters = {};

      this.filtersContainer.querySelectorAll('.audit-filters__checkbox').forEach(checkbox => {
        checkbox.checked = false;
      });

      this.filtersContainer.querySelectorAll('.audit-filters__option').forEach(option => {
        option.classList.remove('audit-filters__option--active');
      });

      this.items.forEach(item => {
        item.style.display = '';
      });

      if (this.noResultsMessage) {
        this.noResultsMessage.style.display = 'none';
      }

      this.buildFilters(); // Rebuild to reset counts
      this.updateDisplay();
    }
  }

  /**
   * Drupal behavior for audit filters.
   */
  Drupal.behaviors.auditFilters = {
    attach: function (context) {
      const lists = context.querySelectorAll('[data-audit-filterable]');
      lists.forEach(list => {
        if (!list.dataset.auditFilterInitialized) {
          new AuditFilter(list);
          list.dataset.auditFilterInitialized = 'true';
        }
      });
    }
  };

})(Drupal);
