<?php

declare(strict_types=1);

namespace Drupal\audit_blocks\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\block\BlockInterface;
use Drupal\block_content\BlockContentTypeInterface;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\Context\ContextRepositoryInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Analyzes block types, block instances, and their cache configuration.
 */
#[AuditAnalyzer(
  id: 'blocks',
  label: new TranslatableMarkup('Blocks Analyzer'),
  description: new TranslatableMarkup('Analyzes block types, block placements, and cache configuration.'),
  menu_title: new TranslatableMarkup('Blocks'),
  output_directory: 'blocks',
  weight: 2,
)]
class BlocksAnalyzer extends AuditAnalyzerBase {

  /**
   * Score weights for different factors.
   */
  protected const SCORE_WEIGHTS = [
    'cache_issues' => 60,
    'disabled_blocks' => 40,
  ];

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The context repository.
   */
  protected ContextRepositoryInterface $contextRepository;

  /**
   * The renderer service.
   */
  protected RendererInterface $renderer;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->moduleHandler = $container->get('module_handler');
    $instance->contextRepository = $container->get('context.repository');
    $instance->renderer = $container->get('renderer');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $config): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function checkRequirements(): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    // Gather all data once.
    $blockTypes = $this->getBlockTypesData();
    $blockPlacements = $this->getBlockPlacementsData();

    // Build separate files for each section.
    $files = $this->buildSectionFiles($blockTypes, $blockPlacements);

    // Calculate scores.
    $scores = $this->calculateScores($blockPlacements);

    return [
      '_files' => $files,
      'score' => $scores,
    ];
  }

  /**
   * Gets block types data.
   *
   * @return array
   *   Block types data.
   */
  protected function getBlockTypesData(): array {
    $blockTypes = [];

    if (!$this->moduleHandler->moduleExists('block_content')) {
      return [
        'enabled' => FALSE,
        'types' => [],
      ];
    }

    try {
      $types = $this->entityTypeManager
        ->getStorage('block_content_type')
        ->loadMultiple();

      foreach ($types as $type_id => $type) {
        if (!$type instanceof BlockContentTypeInterface) {
          continue;
        }

        $block_count = $this->entityTypeManager
          ->getStorage('block_content')
          ->getQuery()
          ->accessCheck(FALSE)
          ->condition('type', $type_id)
          ->count()
          ->execute();

        $blockTypes[] = [
          'id' => $type_id,
          'label' => $type->label(),
          'description' => $type->getDescription(),
          'block_count' => (int) $block_count,
        ];
      }
    }
    catch (\Exception $e) {
      // Log error but continue.
    }

    return [
      'enabled' => TRUE,
      'types' => $blockTypes,
    ];
  }

  /**
   * Gets block placements data.
   *
   * @return array
   *   Block placements data with stats.
   */
  protected function getBlockPlacementsData(): array {
    $blocks = [];
    $stats = [
      'total' => 0,
      'enabled' => 0,
      'disabled' => 0,
      'no_cache' => 0,
      'permanent_cache' => 0,
      'time_based_cache' => 0,
      'by_region' => [],
      'by_theme' => [],
      'by_plugin_type' => [],
    ];

    try {
      $blockEntities = $this->entityTypeManager
        ->getStorage('block')
        ->loadMultiple();

      foreach ($blockEntities as $block_id => $block) {
        if (!$block instanceof BlockInterface) {
          continue;
        }

        $stats['total']++;

        $plugin_id = $block->getPluginId();
        $region = $block->getRegion();
        $theme = $block->getTheme();
        $status = $block->status();
        $weight = $block->getWeight();
        $label = (string) $block->label();
        $plugin_type = $this->getPluginType($plugin_id);

        // Track stats.
        $stats['by_region'][$region] = ($stats['by_region'][$region] ?? 0) + 1;
        $stats['by_theme'][$theme] = ($stats['by_theme'][$theme] ?? 0) + 1;
        $stats['by_plugin_type'][$plugin_type] = ($stats['by_plugin_type'][$plugin_type] ?? 0) + 1;

        if ($status) {
          $stats['enabled']++;
        }
        else {
          $stats['disabled']++;
        }

        // Get cache configuration.
        $cache_info = $this->getBlockCacheInfo($block);

        // Track cache types.
        if ($cache_info['max_age'] === 0) {
          $stats['no_cache']++;
        }
        elseif ($cache_info['max_age'] === Cache::PERMANENT) {
          $stats['permanent_cache']++;
        }
        else {
          $stats['time_based_cache']++;
        }

        $blocks[] = [
          'id' => $block_id,
          'label' => $label,
          'plugin_id' => $plugin_id,
          'plugin_type' => $plugin_type,
          'region' => $region,
          'theme' => $theme,
          'status' => $status,
          'weight' => $weight,
          'cache_max_age' => $cache_info['max_age'],
          'cache_max_age_formatted' => $cache_info['max_age_formatted'],
          'cache_tags' => $cache_info['tags'],
          'cache_contexts' => $cache_info['contexts'],
          'has_cache_issue' => $status && $cache_info['max_age'] === 0,
        ];
      }
    }
    catch (\Exception $e) {
      // Log error but continue.
    }

    return [
      'blocks' => $blocks,
      'stats' => $stats,
    ];
  }

  /**
   * Builds separate files for each section.
   *
   * @param array $blockTypes
   *   Block types data.
   * @param array $blockPlacements
   *   Block placements data.
   *
   * @return array
   *   Array of file data keyed by file name.
   */
  protected function buildSectionFiles(array $blockTypes, array $blockPlacements): array {
    $files = [];
    $blocks = $blockPlacements['blocks'] ?? [];
    $stats = $blockPlacements['stats'] ?? [];

    // Cache issues section.
    $files['cache_issues'] = $this->buildCacheIssuesFile($blocks, $stats);

    // Disabled blocks section.
    $files['disabled_blocks'] = $this->buildDisabledBlocksFile($blocks, $stats);

    // Block types section (informational).
    $files['block_types'] = $this->buildBlockTypesFile($blockTypes);

    // Block inventory section (informational).
    $files['block_inventory'] = $this->buildBlockInventoryFile($blocks, $stats);

    return $files;
  }

  /**
   * Builds the cache issues file.
   */
  protected function buildCacheIssuesFile(array $blocks, array $stats): array {
    $results = [];
    $errors = 0;
    $warnings = 0;

    foreach ($blocks as $block) {
      if (!$block['has_cache_issue']) {
        continue;
      }

      $results[] = $this->createResultItem(
        'warning',
        'BLOCK_NO_CACHE',
        (string) $this->t('Block "@label" has no cache configured (max-age: 0).', [
          '@label' => $block['label'],
        ]),
        [
          'id' => $block['id'],
          'label' => $block['label'],
          'plugin_id' => $block['plugin_id'],
          'plugin_type' => $block['plugin_type'],
          'region' => $block['region'],
          'theme' => $block['theme'],
          'cache_contexts' => $block['cache_contexts'],
          'cache_tags' => $block['cache_tags'],
        ]
      );
      $warnings++;
    }

    $output = $this->createResult($results, $errors, $warnings, 0);
    $output['stats'] = [
      'total_enabled' => $stats['enabled'] ?? 0,
      'no_cache_count' => $stats['no_cache'] ?? 0,
      'percentage' => ($stats['enabled'] ?? 0) > 0
        ? round((($stats['no_cache'] ?? 0) / $stats['enabled']) * 100, 1)
        : 0,
    ];

    return $output;
  }

  /**
   * Builds the disabled blocks file.
   */
  protected function buildDisabledBlocksFile(array $blocks, array $stats): array {
    $results = [];
    $warnings = 0;

    foreach ($blocks as $block) {
      if ($block['status']) {
        continue;
      }

      $results[] = $this->createResultItem(
        'warning',
        'BLOCK_DISABLED',
        (string) $this->t('Block "@label" is disabled.', [
          '@label' => $block['label'],
        ]),
        [
          'id' => $block['id'],
          'label' => $block['label'],
          'plugin_id' => $block['plugin_id'],
          'plugin_type' => $block['plugin_type'],
          'region' => $block['region'],
          'theme' => $block['theme'],
        ]
      );
      $warnings++;
    }

    $output = $this->createResult($results, 0, $warnings, 0);
    $output['stats'] = [
      'total' => $stats['total'] ?? 0,
      'disabled_count' => $stats['disabled'] ?? 0,
      'percentage' => ($stats['total'] ?? 0) > 0
        ? round((($stats['disabled'] ?? 0) / $stats['total']) * 100, 1)
        : 0,
    ];

    return $output;
  }

  /**
   * Builds the block types file.
   */
  protected function buildBlockTypesFile(array $blockTypesData): array {
    $results = [];
    $notices = 0;

    if (!$blockTypesData['enabled']) {
      $results[] = $this->createResultItem(
        'info',
        'NO_BLOCK_CONTENT_MODULE',
        (string) $this->t('Block Content module is not enabled.'),
        []
      );
      $notices++;

      $output = $this->createResult($results, 0, 0, $notices);
      $output['block_content_enabled'] = FALSE;
      $output['types'] = [];
      return $output;
    }

    $types = $blockTypesData['types'] ?? [];

    foreach ($types as $type) {
      $results[] = $this->createResultItem(
        'info',
        'BLOCK_TYPE',
        (string) $this->t('Block type: @label', ['@label' => $type['label']]),
        $type
      );
      $notices++;
    }

    $output = $this->createResult($results, 0, 0, $notices);
    $output['block_content_enabled'] = TRUE;
    $output['types'] = $types;
    $output['total_types'] = count($types);

    return $output;
  }

  /**
   * Builds the block inventory file.
   */
  protected function buildBlockInventoryFile(array $blocks, array $stats): array {
    $output = $this->createResult([], 0, 0, 0);
    $output['blocks'] = $blocks;
    $output['stats'] = $stats;
    $output['totals'] = [
      'total' => $stats['total'] ?? 0,
      'enabled' => $stats['enabled'] ?? 0,
      'disabled' => $stats['disabled'] ?? 0,
      'no_cache' => $stats['no_cache'] ?? 0,
      'permanent_cache' => $stats['permanent_cache'] ?? 0,
      'time_based_cache' => $stats['time_based_cache'] ?? 0,
    ];

    return $output;
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      'cache_issues' => [
        'label' => $this->t('Cache Configuration'),
        'description' => $this->t('Enabled blocks with no cache configured (max-age: 0) impact performance.'),
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['cache_issues'],
        'file_key' => 'cache_issues',
        'score_factor_key' => 'cache_issues',
      ],
      'disabled_blocks' => [
        'label' => $this->t('Disabled Blocks'),
        'description' => $this->t('Disabled blocks that may need cleanup or removal.'),
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['disabled_blocks'],
        'file_key' => 'disabled_blocks',
        'score_factor_key' => 'disabled_blocks',
      ],
      'block_types' => [
        'label' => $this->t('Custom Block Types'),
        'description' => $this->t('Inventory of custom block types (Block Content module). Informational only.'),
        'affects_score' => FALSE,
        'weight' => 0,
        'file_key' => 'block_types',
        'score_factor_key' => NULL,
      ],
      'block_inventory' => [
        'label' => $this->t('Block Inventory'),
        'description' => $this->t('Complete inventory of all block placements by theme. Informational only.'),
        'affects_score' => FALSE,
        'weight' => 0,
        'file_key' => 'block_inventory',
        'score_factor_key' => NULL,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    $files = $data['_files'] ?? [];

    return match ($check_id) {
      'cache_issues' => $this->buildCacheIssuesContent($files),
      'disabled_blocks' => $this->buildDisabledBlocksContent($files),
      'block_types' => $this->buildBlockTypesContent($files),
      'block_inventory' => $this->buildBlockInventoryContent($files),
      default => [],
    };
  }

  /**
   * Builds the cache issues section content.
   */
  protected function buildCacheIssuesContent(array $files): array {
    $cacheData = $files['cache_issues'] ?? [];
    $stats = $cacheData['stats'] ?? [];

    $fix_html = '<p><strong>Problem:</strong> Blocks with max-age: 0 are rebuilt on every request, causing performance issues.</p>'
      . '<p><strong>Possible causes and solutions:</strong></p>'
      . '<ul>'
      . '<li><strong>User-specific content:</strong> If the block shows personalized content, consider using cache contexts like <code>user</code> or <code>session</code> instead of disabling cache entirely.</li>'
      . '<li><strong>Dynamic content:</strong> Use appropriate cache tags to invalidate when content changes, rather than disabling cache.</li>'
      . '<li><strong>Views blocks:</strong> Configure Views cache settings in the view display configuration.</li>'
      . '<li><strong>Custom blocks:</strong> Implement proper cache metadata in the block plugin\'s <code>getCacheMaxAge()</code>, <code>getCacheContexts()</code>, and <code>getCacheTags()</code> methods.</li>'
      . '</ul>';

    return $this->ui->buildIssueListFromResults(
      $cacheData['results'] ?? [],
      (string) $this->t('All enabled blocks have cache configured. Total enabled: @total.', [
        '@total' => $stats['total_enabled'] ?? 0,
      ]),
      function (array $item, $ui) use ($fix_html): array {
        $details = $item['details'] ?? [];
        $blockId = $details['id'] ?? '';
        $label = $details['label'] ?? '';
        $pluginType = $details['plugin_type'] ?? '';
        $region = $details['region'] ?? '';
        $theme = $details['theme'] ?? '';
        $contexts = $details['cache_contexts'] ?? [];
        $tags = $details['cache_tags'] ?? [];

        $contextsStr = !empty($contexts) ? implode(', ', $contexts) : 'none';
        $tagsStr = !empty($tags) ? implode(', ', array_slice($tags, 0, 5)) : 'none';
        if (count($tags) > 5) {
          $tagsStr .= '... (+' . (count($tags) - 5) . ')';
        }

        $description = '<p><strong>Details:</strong></p>'
          . '<ul>'
          . '<li><strong>Theme:</strong> ' . $theme . '</li>'
          . '<li><strong>Region:</strong> ' . $region . '</li>'
          . '<li><strong>Plugin type:</strong> ' . $pluginType . '</li>'
          . '<li><strong>Cache contexts:</strong> ' . $contextsStr . '</li>'
          . '<li><strong>Cache tags:</strong> ' . $tagsStr . '</li>'
          . '</ul>'
          . $fix_html;

        return [
          'severity' => 'warning',
          'code' => $item['code'] ?? 'BLOCK_NO_CACHE',
          'file' => $blockId,
          'label' => $item['message'] ?? '',
          'description' => ['#markup' => $description],
          'tags' => ['performance', 'cache'],
        ];
      }
    );
  }

  /**
   * Builds the disabled blocks section content.
   */
  protected function buildDisabledBlocksContent(array $files): array {
    $disabledData = $files['disabled_blocks'] ?? [];
    $stats = $disabledData['stats'] ?? [];

    $fix_html = '<p><strong>Problem:</strong> Disabled blocks are still loaded in memory and processed by Drupal, consuming resources unnecessarily.</p>'
      . '<p><strong>Recommendation:</strong> Review and remove disabled blocks that are no longer needed.</p>'
      . '<p><strong>Solutions:</strong></p>'
      . '<ul>'
      . '<li><strong>Delete (recommended):</strong> Remove the block from the block layout if it is no longer needed. This frees memory and reduces processing overhead.</li>'
      . '<li><strong>Enable:</strong> If the block should be visible, enable it and configure its visibility settings properly.</li>'
      . '</ul>'
      . '<p><strong>Note:</strong> Keeping disabled blocks "just in case" is technical debt. Block configurations can be recreated or exported to configuration if needed later.</p>';

    return $this->ui->buildIssueListFromResults(
      $disabledData['results'] ?? [],
      (string) $this->t('No disabled blocks found. Total blocks: @total.', [
        '@total' => $stats['total'] ?? 0,
      ]),
      function (array $item, $ui) use ($fix_html): array {
        $details = $item['details'] ?? [];
        $blockId = $details['id'] ?? '';
        $pluginType = $details['plugin_type'] ?? '';
        $region = $details['region'] ?? '';
        $theme = $details['theme'] ?? '';

        $description = '<p><strong>Details:</strong></p>'
          . '<ul>'
          . '<li><strong>Theme:</strong> ' . $theme . '</li>'
          . '<li><strong>Region:</strong> ' . $region . '</li>'
          . '<li><strong>Plugin type:</strong> ' . $pluginType . '</li>'
          . '</ul>'
          . $fix_html;

        return [
          'severity' => 'warning',
          'code' => $item['code'] ?? 'BLOCK_DISABLED',
          'file' => $blockId,
          'label' => $item['message'] ?? '',
          'description' => ['#markup' => $description],
          'tags' => ['cleanup', 'maintenance'],
        ];
      }
    );
  }

  /**
   * Builds the block types section content (informational table).
   */
  protected function buildBlockTypesContent(array $files): array {
    $typesData = $files['block_types'] ?? [];

    if (!($typesData['block_content_enabled'] ?? FALSE)) {
      return [
        '#type' => 'container',
        'info' => [
          '#type' => 'html_tag',
          '#tag' => 'p',
          '#attributes' => ['class' => ['audit-info-text']],
          '#value' => '<em>' . $this->t('This section is informational only and does not affect the audit score.') . '</em>',
        ],
        'message' => $this->ui->message(
          (string) $this->t('Block Content module is not enabled. Enable it to create custom block types with fields.'),
          'info'
        ),
      ];
    }

    $types = $typesData['types'] ?? [];

    if (empty($types)) {
      return [
        '#type' => 'container',
        'info' => [
          '#type' => 'html_tag',
          '#tag' => 'p',
          '#attributes' => ['class' => ['audit-info-text']],
          '#value' => '<em>' . $this->t('This section is informational only and does not affect the audit score.') . '</em>',
        ],
        'message' => $this->ui->message(
          (string) $this->t('No custom block types defined.'),
          'info'
        ),
      ];
    }

    $headers = [
      $this->ui->header((string) $this->t('Label')),
      $this->ui->header((string) $this->t('Machine Name')),
      $this->ui->header((string) $this->t('Description')),
      $this->ui->header((string) $this->t('Blocks'), 'center'),
    ];

    $rows = [];
    foreach ($types as $type) {
      $rows[] = $this->ui->row([
        $this->ui->cell($type['label'] ?? ''),
        $this->ui->cell('<code>' . htmlspecialchars($type['id'] ?? '') . '</code>'),
        $this->ui->cell($type['description'] ?? '-'),
        $this->ui->cell((string) ($type['block_count'] ?? 0), ['align' => 'center']),
      ]);
    }

    return [
      '#type' => 'container',
      'info' => [
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#attributes' => ['class' => ['audit-info-text']],
        '#value' => '<em>' . $this->t('This section is informational only and does not affect the audit score.') . '</em>',
      ],
      'summary' => [
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#value' => $this->t('@count custom block type(s) defined.', ['@count' => count($types)]),
      ],
      'table' => $this->ui->table($headers, $rows, [
        'empty' => (string) $this->t('No block types found.'),
      ]),
    ];
  }

  /**
   * Builds the block inventory section content (informational table).
   */
  protected function buildBlockInventoryContent(array $files): array {
    $inventoryData = $files['block_inventory'] ?? [];
    $blocks = $inventoryData['blocks'] ?? [];
    $stats = $inventoryData['stats'] ?? [];
    $totals = $inventoryData['totals'] ?? [];

    if (empty($blocks)) {
      return [
        '#type' => 'container',
        'info' => [
          '#type' => 'html_tag',
          '#tag' => 'p',
          '#attributes' => ['class' => ['audit-info-text']],
          '#value' => '<em>' . $this->t('This section is informational only and does not affect the audit score.') . '</em>',
        ],
        'message' => $this->ui->message(
          (string) $this->t('No blocks configured in block layout.'),
          'info'
        ),
      ];
    }

    $summaryText = $this->t('Total: @total blocks (Enabled: @enabled, Disabled: @disabled). Cache: @permanent permanent, @timed time-based, @nocache no cache.', [
      '@total' => $totals['total'] ?? 0,
      '@enabled' => $totals['enabled'] ?? 0,
      '@disabled' => $totals['disabled'] ?? 0,
      '@permanent' => $totals['permanent_cache'] ?? 0,
      '@timed' => $totals['time_based_cache'] ?? 0,
      '@nocache' => $totals['no_cache'] ?? 0,
    ]);

    // Group blocks by theme.
    $blocksByTheme = [];
    foreach ($blocks as $block) {
      $theme = $block['theme'] ?? 'unknown';
      $blocksByTheme[$theme][] = $block;
    }

    $content = [
      '#type' => 'container',
      'info' => [
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#attributes' => ['class' => ['audit-info-text']],
        '#value' => '<em>' . $this->t('This section is informational only and does not affect the audit score.') . '</em>',
      ],
      'summary' => [
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#value' => $summaryText,
      ],
    ];

    // Build a table per theme.
    foreach ($blocksByTheme as $theme => $themeBlocks) {
      $headers = [
        $this->ui->header((string) $this->t('Block')),
        $this->ui->header((string) $this->t('Region')),
        $this->ui->header((string) $this->t('Type')),
        $this->ui->header((string) $this->t('Status'), 'center'),
        $this->ui->header((string) $this->t('Cache'), 'center'),
      ];

      $rows = [];
      foreach ($themeBlocks as $block) {
        $status = $block['status'] ?? FALSE;
        $statusIcon = $status ? $this->ui->icon('check') : $this->ui->icon('cross');
        $statusClass = $status ? 'success' : 'warning';

        $maxAge = $block['cache_max_age'] ?? -1;
        $cacheFormatted = $block['cache_max_age_formatted'] ?? 'Unknown';
        $cacheClass = 'success';
        if ($maxAge === 0) {
          $cacheClass = 'warning';
        }
        elseif ($maxAge < 0) {
          $cacheClass = 'notice';
        }

        $rowSeverity = NULL;
        if (!$status) {
          $rowSeverity = 'warning';
        }
        elseif ($maxAge === 0) {
          $rowSeverity = 'warning';
        }

        $rows[] = $this->ui->row([
          $this->ui->itemName((string) ($block['label'] ?? ''), (string) ($block['id'] ?? '')),
          $this->ui->cell((string) ($block['region'] ?? '')),
          $this->ui->cell((string) ($block['plugin_type'] ?? '')),
          $this->ui->cell($statusIcon, ['align' => 'center', 'status' => $statusClass]),
          $this->ui->cell($cacheFormatted, ['align' => 'center', 'status' => $cacheClass]),
        ], $rowSeverity);
      }

      $content['theme_' . $theme] = $this->ui->table($headers, $rows, [
        'caption' => (string) $this->t('Theme: @theme (@count blocks)', [
          '@theme' => $theme,
          '@count' => count($themeBlocks),
        ]),
        'empty' => (string) $this->t('No blocks found.'),
      ]);
    }

    return $content;
  }

  // =========================================================================
  // Helper methods.
  // =========================================================================

  /**
   * Gets cache information for a block.
   *
   * @param \Drupal\block\BlockInterface $block
   *   The block entity.
   *
   * @return array
   *   Cache information with max_age, tags, and contexts.
   */
  protected function getBlockCacheInfo(BlockInterface $block): array {
    $cache_info = [
      'max_age' => Cache::PERMANENT,
      'max_age_formatted' => 'Permanent',
      'tags' => [],
      'contexts' => [],
    ];

    try {
      // Start with the block entity's cache metadata.
      $cache_metadata = CacheableMetadata::createFromObject($block);

      // Get the block plugin.
      $plugin = $block->getPlugin();

      // Add cache metadata from the plugin itself.
      $plugin_cache = new CacheableMetadata();
      $plugin_cache->setCacheContexts($plugin->getCacheContexts());
      $plugin_cache->setCacheTags($plugin->getCacheTags());
      $plugin_cache->setCacheMaxAge($plugin->getCacheMaxAge());
      $cache_metadata = $cache_metadata->merge($plugin_cache);

      // Try to build the block to get full cache metadata including content.
      try {
        $view_builder = $this->entityTypeManager->getViewBuilder('block');
        $build = $view_builder->view($block);

        if (isset($build['#cache'])) {
          $render_cache = new CacheableMetadata();
          if (isset($build['#cache']['contexts'])) {
            $render_cache->setCacheContexts($build['#cache']['contexts']);
          }
          if (isset($build['#cache']['tags'])) {
            $render_cache->setCacheTags($build['#cache']['tags']);
          }
          if (isset($build['#cache']['max-age'])) {
            $render_cache->setCacheMaxAge($build['#cache']['max-age']);
          }
          $cache_metadata = $cache_metadata->merge($render_cache);
        }

        if (isset($build['content']) && is_array($build['content'])) {
          $content_cache = $this->extractCacheMetadataFromRenderArray($build['content']);
          $cache_metadata = $cache_metadata->merge($content_cache);
        }
      }
      catch (\Exception $e) {
        // View builder may fail for some blocks, that's okay.
      }

      $cache_info['contexts'] = $cache_metadata->getCacheContexts();
      $cache_info['tags'] = $cache_metadata->getCacheTags();
      $cache_info['max_age'] = $cache_metadata->getCacheMaxAge();
      $cache_info['max_age_formatted'] = $this->formatMaxAge($cache_metadata->getCacheMaxAge());
    }
    catch (\Exception $e) {
      $cache_info['max_age'] = -1;
      $cache_info['max_age_formatted'] = 'Unknown';
    }

    return $cache_info;
  }

  /**
   * Extracts cache metadata from a render array recursively.
   *
   * @param array $build
   *   The render array.
   *
   * @return \Drupal\Core\Cache\CacheableMetadata
   *   The extracted cache metadata.
   */
  protected function extractCacheMetadataFromRenderArray(array $build): CacheableMetadata {
    $cache_metadata = new CacheableMetadata();

    if (isset($build['#cache'])) {
      if (isset($build['#cache']['contexts'])) {
        $cache_metadata->addCacheContexts($build['#cache']['contexts']);
      }
      if (isset($build['#cache']['tags'])) {
        $cache_metadata->addCacheTags($build['#cache']['tags']);
      }
      if (isset($build['#cache']['max-age'])) {
        $cache_metadata->mergeCacheMaxAge($build['#cache']['max-age']);
      }
    }

    foreach ($build as $key => $value) {
      if (is_array($value) && !str_starts_with($key, '#')) {
        $child_cache = $this->extractCacheMetadataFromRenderArray($value);
        $cache_metadata = $cache_metadata->merge($child_cache);
      }
    }

    return $cache_metadata;
  }

  /**
   * Formats max-age value for display.
   *
   * @param int $max_age
   *   The max-age value.
   *
   * @return string
   *   Formatted max-age string.
   */
  protected function formatMaxAge(int $max_age): string {
    if ($max_age === Cache::PERMANENT) {
      return (string) $this->t('Permanent');
    }

    if ($max_age === 0) {
      return (string) $this->t('No cache (0)');
    }

    if ($max_age < 0) {
      return (string) $this->t('Unknown');
    }

    if ($max_age < 60) {
      return (string) $this->t('@sec seconds', ['@sec' => $max_age]);
    }

    if ($max_age < 3600) {
      $minutes = floor($max_age / 60);
      return (string) $this->t('@min minutes', ['@min' => $minutes]);
    }

    if ($max_age < 86400) {
      $hours = floor($max_age / 3600);
      return (string) $this->t('@hours hours', ['@hours' => $hours]);
    }

    $days = floor($max_age / 86400);
    return (string) $this->t('@days days', ['@days' => $days]);
  }

  /**
   * Gets the plugin type from plugin ID.
   *
   * @param string $plugin_id
   *   The plugin ID.
   *
   * @return string
   *   The plugin type.
   */
  protected function getPluginType(string $plugin_id): string {
    if (str_starts_with($plugin_id, 'block_content:')) {
      return 'block_content';
    }

    if (str_starts_with($plugin_id, 'views_block:')) {
      return 'views_block';
    }

    if (str_starts_with($plugin_id, 'system_')) {
      return 'system';
    }

    if (str_starts_with($plugin_id, 'field_block:')) {
      return 'field_block';
    }

    if (str_starts_with($plugin_id, 'extra_field_block:')) {
      return 'extra_field_block';
    }

    if (str_starts_with($plugin_id, 'inline_block:')) {
      return 'inline_block';
    }

    return 'plugin';
  }

  // =========================================================================
  // Score calculation methods.
  // =========================================================================

  /**
   * Calculates scores for all factors.
   *
   * @param array $blockPlacements
   *   Block placements data.
   *
   * @return array
   *   The scores.
   */
  protected function calculateScores(array $blockPlacements): array {
    $stats = $blockPlacements['stats'] ?? [];
    $total = $stats['total'] ?? 0;
    $enabled = $stats['enabled'] ?? 0;
    $disabled = $stats['disabled'] ?? 0;
    $no_cache = $stats['no_cache'] ?? 0;

    // Cache issues score.
    $cacheScore = $this->calculateCacheScore($enabled, $no_cache);
    $cacheDescription = $no_cache > 0
      ? (string) $this->t('@count enabled blocks have no cache configured', ['@count' => $no_cache])
      : (string) $this->t('All enabled blocks have cache configured');

    // Disabled blocks score.
    $disabledScore = $this->calculateDisabledScore($total, $disabled);
    $disabledDescription = $disabled > 0
      ? (string) $this->t('@count blocks are disabled and may need cleanup', ['@count' => $disabled])
      : (string) $this->t('No disabled blocks found');

    return [
      'factors' => [
        'cache_issues' => [
          'score' => $cacheScore,
          'weight' => self::SCORE_WEIGHTS['cache_issues'],
          'label' => (string) $this->t('Cache Configuration'),
          'description' => $cacheDescription,
        ],
        'disabled_blocks' => [
          'score' => $disabledScore,
          'weight' => self::SCORE_WEIGHTS['disabled_blocks'],
          'label' => (string) $this->t('Block Cleanup'),
          'description' => $disabledDescription,
        ],
      ],
    ];
  }

  /**
   * Calculates cache configuration score.
   */
  protected function calculateCacheScore(int $enabled, int $noCache): int {
    if ($enabled === 0) {
      return 100;
    }

    $percentage = ($noCache / $enabled) * 100;
    return max(0, 100 - (int) ($percentage * 1.5));
  }

  /**
   * Calculates disabled blocks score.
   *
   * Each disabled block reduces the score. Disabled blocks consume memory
   * and should be removed if not needed.
   */
  protected function calculateDisabledScore(int $total, int $disabled): int {
    if ($total === 0 || $disabled === 0) {
      return 100;
    }

    // Each disabled block reduces score by 10 points.
    // Max penalty is 60 points (6+ disabled blocks = 40 score minimum).
    return max(40, 100 - ($disabled * 10));
  }

}
