<?php

declare(strict_types=1);

namespace Drupal\audit_complexity\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Process\Process;

/**
 * Analyzes code complexity metrics for custom modules and themes.
 *
 * This analyzer uses phploc to measure:
 * - Lines of Code (LOC): Production vs test code volume
 * - Cyclomatic Complexity (CCN): Code branching and decision paths
 * - Maintainability metrics: Classes, methods, functions structure
 *
 * Additionally performs custom analysis for Drupal-specific anti-patterns:
 * - Service locators (\Drupal::service() calls)
 * - Deep arrays (3+ levels of nesting)
 * - Hardcoded entity IDs
 * - Direct database queries without DI
 *
 * @see https://dri.es/measuring-drupal-core-code-complexity
 * @see https://dbuytaert.github.io/drupal-core-metrics/
 */
#[AuditAnalyzer(
  id: 'complexity',
  label: new TranslatableMarkup('Code Complexity'),
  description: new TranslatableMarkup('Analyzes code complexity metrics (LOC, cyclomatic complexity, maintainability) and Drupal anti-patterns for custom modules and themes.'),
  menu_title: new TranslatableMarkup('Complexity'),
  output_directory: 'complexity',
  weight: 2,
)]
class ComplexityAnalyzer extends AuditAnalyzerBase {

  /**
   * Score weights for different factors.
   *
   * Weights reflect relative importance for code maintainability:
   * - Cyclomatic complexity: High CCN = hard to test/maintain
   * - Anti-patterns: Drupal-specific technical debt
   * - Code structure: Balance between classes/functions
   */
  protected const SCORE_WEIGHTS = [
    'cyclomatic_complexity' => 40,
    'anti_patterns' => 35,
    'code_structure' => 25,
  ];

  /**
   * Default thresholds for complexity metrics.
   */
  protected const DEFAULT_CCN_WARNING = 10;
  protected const DEFAULT_CCN_ERROR = 20;
  protected const DEFAULT_MI_WARNING = 65;
  protected const DEFAULT_MI_ERROR = 40;
  protected const DEFAULT_DEEP_ARRAY_THRESHOLD = 4;
  protected const DEFAULT_TIMEOUT = 120;
  protected const DEFAULT_MAX_HOTSPOTS = 20;

  /**
   * Required packages for complexity analysis.
   */
  protected const REQUIRED_PACKAGES = [
    'phploc/phploc' => 'PHP Lines Of Code - measures LOC, complexity, and structure metrics',
  ];

  /**
   * Drupal service locator patterns to detect.
   */
  protected const SERVICE_LOCATOR_PATTERNS = [
    '\\Drupal::service(' => 'Direct service container access',
    '\\Drupal::entityTypeManager(' => 'Direct entity type manager access',
    '\\Drupal::database(' => 'Direct database access',
    '\\Drupal::config(' => 'Direct config access',
    '\\Drupal::state(' => 'Direct state access',
    '\\Drupal::cache(' => 'Direct cache access',
    '\\Drupal::moduleHandler(' => 'Direct module handler access',
    '\\Drupal::currentUser(' => 'Direct current user access',
    '\\Drupal::languageManager(' => 'Direct language manager access',
    '\\Drupal::messenger(' => 'Direct messenger access',
  ];

  /**
   * The config factory service.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->configFactory = $container->get('config.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function checkRequirements(): array {
    $warnings = [];

    // Check for phploc binary.
    $phploc_path = $this->findBinary('phploc');
    if (!$phploc_path) {
      $warnings[] = (string) $this->t('phploc is not installed. Run: composer require --dev phploc/phploc');
      return $warnings;
    }

    // Verify phploc works.
    try {
      $process = new Process([$phploc_path, '--version']);
      $process->setTimeout(10);
      $process->run();

      if (!$process->isSuccessful()) {
        $warnings[] = (string) $this->t('phploc is installed but not working properly. Error: @error', [
          '@error' => $process->getErrorOutput(),
        ]);
      }
    }
    catch (\Exception $e) {
      $warnings[] = (string) $this->t('Could not verify phploc installation: @error', [
        '@error' => $e->getMessage(),
      ]);
    }

    return $warnings;
  }

  /**
   * Gets detailed installation instructions.
   *
   * @return array
   *   Array with installation steps.
   */
  protected function getInstallationInstructions(): array {
    return [
      'title' => (string) $this->t('How to install phploc for code complexity analysis'),
      'steps' => [
        [
          'command' => 'composer require --dev phploc/phploc',
          'description' => (string) $this->t('Install phploc package. This provides metrics for lines of code, cyclomatic complexity, and code structure.'),
        ],
        [
          'command' => './vendor/bin/phploc --version',
          'description' => (string) $this->t('Verify installation. You should see the phploc version number.'),
        ],
      ],
      'notes' => [
        (string) $this->t('phploc is a lightweight tool that quickly scans PHP code for various size and structure metrics.'),
        (string) $this->t('For more detailed metrics including Maintainability Index, consider also installing phpmetrics: <code>composer require --dev phpmetrics/phpmetrics</code>'),
        (string) $this->t('The analysis focuses on custom modules and themes in modules/custom/ and themes/custom/ directories.'),
      ],
      'documentation' => 'https://github.com/sebastianbergmann/phploc',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    $config = $this->configFactory->get('audit_complexity.settings');
    $audit_config = $this->configFactory->get('audit.settings');

    $include_tests = (bool) ($config->get('include_tests') ?? FALSE);
    $timeout = (int) ($config->get('timeout') ?? self::DEFAULT_TIMEOUT);

    // Get paths from global audit configuration.
    $paths = $this->getScanDirectories($audit_config);
    $exclude_patterns = $this->getExcludePatterns($audit_config, $include_tests);

    if (empty($paths)) {
      return $this->buildEmptyResults();
    }

    // Run phploc analysis.
    $phploc_results = $this->runPhploc($paths, $exclude_patterns, $timeout);

    // Run anti-pattern analysis.
    $antipattern_results = $this->analyzeAntiPatterns($paths, $exclude_patterns, $config);

    // Analyze complexity hotspots (functions with high CCN).
    $hotspots = $this->analyzeComplexityHotspots($paths, $exclude_patterns, $config);

    // Analyze Drupal API usage.
    $api_usage = $this->analyzeApiUsage($paths, $exclude_patterns);

    // Calculate scores.
    $scores = $this->calculateScores($phploc_results, $antipattern_results, $hotspots, $config);

    return [
      '_files' => [
        'overview' => $this->createResult([], 0, 0, 0),
        'complexity_hotspots' => $this->createResult(
          $hotspots['results'] ?? [],
          $hotspots['errors'] ?? 0,
          $hotspots['warnings'] ?? 0,
          $hotspots['notices'] ?? 0
        ),
        'anti_patterns' => $this->createResult(
          $antipattern_results['results'] ?? [],
          $antipattern_results['errors'] ?? 0,
          $antipattern_results['warnings'] ?? 0,
          $antipattern_results['notices'] ?? 0
        ),
        'code_metrics' => $this->createResult([], 0, 0, 0),
        'api_usage' => $this->createResult([], 0, 0, 0),
      ],
      'score' => $scores,
      'phploc' => $phploc_results,
      'antipatterns' => $antipattern_results,
      'hotspots' => $hotspots,
      'api_usage' => $api_usage,
      'paths_analyzed' => $paths,
      'exclude_patterns' => $exclude_patterns,
      'config' => [
        'include_tests' => $include_tests,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      'complexity_hotspots' => [
        'label' => $this->t('Complexity Hotspots'),
        'description' => $this->t('Functions and methods with high cyclomatic complexity that are difficult to test and maintain.'),
        'file_types' => ['php'],
        'affects_score' => TRUE,
        'file_key' => 'complexity_hotspots',
        'score_factor_key' => 'cyclomatic_complexity',
        'weight' => self::SCORE_WEIGHTS['cyclomatic_complexity'],
      ],
      'anti_patterns' => [
        'label' => $this->t('Drupal Anti-Patterns'),
        'description' => $this->t('Code patterns that violate Drupal best practices: service locators, deep arrays, hardcoded IDs, direct queries.'),
        'file_types' => ['php', 'module'],
        'affects_score' => TRUE,
        'file_key' => 'anti_patterns',
        'score_factor_key' => 'anti_patterns',
        'weight' => self::SCORE_WEIGHTS['anti_patterns'],
      ],
      'overview' => [
        'label' => $this->t('Metrics Overview'),
        'description' => $this->t('Summary of code complexity metrics from phploc analysis.'),
        'file_types' => [],
        'affects_score' => FALSE,
        'file_key' => 'overview',
      ],
      'code_metrics' => [
        'label' => $this->t('Detailed Code Metrics'),
        'description' => $this->t('Detailed breakdown of lines of code, classes, methods, and functions.'),
        'file_types' => ['php'],
        'affects_score' => FALSE,
        'file_key' => 'code_metrics',
      ],
      'api_usage' => [
        'label' => $this->t('Drupal API Usage'),
        'description' => $this->t('Analysis of Drupal Core APIs used in custom code: hooks, preprocess functions, event subscribers, and global functions.'),
        'file_types' => ['php', 'module', 'theme', 'install'],
        'affects_score' => FALSE,
        'file_key' => 'api_usage',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    return match ($check_id) {
      'overview' => $this->buildOverviewContent($data),
      'complexity_hotspots' => $this->buildHotspotsContent($data),
      'anti_patterns' => $this->buildAntiPatternsContent($data),
      'code_metrics' => $this->buildCodeMetricsContent($data),
      'api_usage' => $this->buildApiUsageContent($data),
      default => [],
    };
  }

  /**
   * Builds content for the overview section.
   */
  protected function buildOverviewContent(array $data): array {
    $phploc = $data['phploc'] ?? [];

    if (empty($phploc) || !empty($phploc['error'])) {
      $error_msg = $phploc['error'] ?? (string) $this->t('No analysis data available.');
      return [
        'message' => $this->ui->message($error_msg, 'warning'),
        'instructions' => $this->buildInstallationInstructions(),
      ];
    }

    $metrics = $phploc['metrics'] ?? [];
    $paths = $data['paths_analyzed'] ?? [];

    // Build summary cards.
    $content = [];

    // Paths analyzed.
    $content['paths'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-info-box']],
      'text' => [
        '#markup' => '<p><strong>' . $this->t('Analyzed paths:') . '</strong> ' .
          implode(', ', array_map(fn($p) => '<code>' . basename($p) . '</code>', $paths)) . '</p>',
      ],
    ];

    // Key metrics summary table.
    $headers = [
      $this->ui->header((string) $this->t('Metric'), 'left', '40%'),
      $this->ui->header((string) $this->t('Value'), 'right'),
      $this->ui->header((string) $this->t('Assessment'), 'left'),
    ];

    $rows = [];

    // Lines of Code.
    $loc = $metrics['loc'] ?? 0;
    $rows[] = $this->ui->row([
      $this->ui->itemName((string) $this->t('Lines of Code'), 'LOC'),
      $this->ui->cell(number_format($loc)),
      $this->ui->cell($this->assessLoc($loc)),
    ]);

    // Logical Lines of Code.
    $lloc = $metrics['lloc'] ?? 0;
    $rows[] = $this->ui->row([
      $this->ui->itemName((string) $this->t('Logical Lines'), 'LLOC'),
      $this->ui->cell(number_format($lloc)),
      $this->ui->cell((string) $this->t('Executable statements')),
    ]);

    // Average CCN.
    $avg_ccn = $metrics['ccn_avg'] ?? $metrics['ccnByLloc'] ?? 0;
    $ccn_assessment = $this->assessCcn($avg_ccn);
    $rows[] = $this->ui->row([
      $this->ui->itemName((string) $this->t('Avg. Cyclomatic Complexity'), 'CCN'),
      $this->ui->cell(number_format($avg_ccn, 2)),
      $this->ui->cell($ccn_assessment['label']),
    ], $ccn_assessment['severity']);

    // Classes.
    $classes = $metrics['classes'] ?? 0;
    $rows[] = $this->ui->row([
      $this->ui->itemName((string) $this->t('Classes'), 'classes'),
      $this->ui->cell(number_format($classes)),
      $this->ui->cell((string) $this->t('Total class definitions')),
    ]);

    // Methods.
    $methods = $metrics['methods'] ?? 0;
    $rows[] = $this->ui->row([
      $this->ui->itemName((string) $this->t('Methods'), 'methods'),
      $this->ui->cell(number_format($methods)),
      $this->ui->cell((string) $this->t('Class methods')),
    ]);

    // Functions.
    $functions = $metrics['functions'] ?? 0;
    $rows[] = $this->ui->row([
      $this->ui->itemName((string) $this->t('Functions'), 'functions'),
      $this->ui->cell(number_format($functions)),
      $this->ui->cell((string) $this->t('Standalone functions')),
    ]);

    $content['table'] = $this->ui->table($headers, $rows);

    return $content;
  }

  /**
   * Builds content for complexity hotspots section.
   */
  protected function buildHotspotsContent(array $data): array {
    $hotspots = $data['hotspots'] ?? [];
    $results = $hotspots['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('No complexity hotspots detected. All functions are within acceptable complexity thresholds.'),
          'success'
        ),
      ];
    }

    return $this->ui->buildIssueListFromResults(
      $results,
      (string) $this->t('All functions are within acceptable complexity thresholds.'),
      function (array $item, $ui): array {
        $details = $item['details'] ?? [];
        $ccn = $details['ccn'] ?? 0;

        $description = '<p>' . $this->t('Cyclomatic complexity: <strong>@ccn</strong>', ['@ccn' => $ccn]) . '</p>';

        // Build code snippet using absolute path.
        $snippet_details = [
          'file' => $details['absolute_file'] ?? $details['file'] ?? '',
          'line' => $details['line'] ?? NULL,
        ];
        $code_snippet = $ui->buildIssueCodeSnippet($snippet_details, $item['severity'] ?? 'warning');

        return [
          'severity' => $item['severity'] ?? 'warning',
          'code' => $item['code'] ?? 'HIGH_CCN',
          'label' => $details['function'] ?? $item['message'] ?? '',
          'file' => $details['file'] ?? '',
          'line' => $details['line'] ?? NULL,
          'description' => ['#markup' => $description],
          'code_snippet' => $code_snippet,
          'tags' => ['complexity', 'refactor'],
        ];
      }
    );
  }

  /**
   * Builds content for anti-patterns section.
   */
  protected function buildAntiPatternsContent(array $data): array {
    $antipatterns = $data['antipatterns'] ?? [];
    $results = $antipatterns['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('No Drupal anti-patterns detected. Code follows best practices.'),
          'success'
        ),
      ];
    }

    return $this->ui->buildIssueListFromResults(
      $results,
      (string) $this->t('No anti-patterns detected.'),
      function (array $item, $ui): array {
        $details = $item['details'] ?? [];
        $pattern_type = $details['type'] ?? 'unknown';

        $tags = ['anti-pattern'];
        if ($pattern_type === 'service_locator') {
          $tags[] = 'dependency-injection';
        }
        elseif ($pattern_type === 'deep_array') {
          $tags[] = 'structure';
        }
        elseif ($pattern_type === 'hardcoded_id') {
          $tags[] = 'configuration';
        }
        elseif ($pattern_type === 'direct_query') {
          $tags[] = 'database';
        }

        $description = '';
        if (!empty($details['pattern'])) {
          $description .= '<p>' . $this->t('Pattern: <code>@pattern</code>', ['@pattern' => $details['pattern']]) . '</p>';
        }
        if (!empty($details['suggestion'])) {
          $description .= '<p><strong>' . $this->t('Suggestion:') . '</strong> ' . $details['suggestion'] . '</p>';
        }

        // Build code snippet using absolute path.
        $snippet_details = [
          'file' => $details['absolute_file'] ?? $details['file'] ?? '',
          'line' => $details['line'] ?? NULL,
        ];
        $code_snippet = $ui->buildIssueCodeSnippet($snippet_details, $item['severity'] ?? 'warning');

        return [
          'severity' => $item['severity'] ?? 'warning',
          'code' => $item['code'] ?? 'ANTI_PATTERN',
          'label' => $item['message'] ?? '',
          'file' => $details['file'] ?? '',
          'line' => $details['line'] ?? NULL,
          'description' => $description ? ['#markup' => $description] : NULL,
          'code_snippet' => $code_snippet,
          'tags' => $tags,
          'custom_data' => [
            'type' => $pattern_type,
          ],
        ];
      }
    );
  }

  /**
   * Builds content for detailed code metrics section.
   */
  protected function buildCodeMetricsContent(array $data): array {
    $phploc = $data['phploc'] ?? [];
    $metrics = $phploc['metrics'] ?? [];

    if (empty($metrics)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('No detailed metrics available. Run the analysis first.'),
          'info'
        ),
      ];
    }

    $content = [];

    // Size metrics.
    $size_headers = [
      $this->ui->header((string) $this->t('Metric')),
      $this->ui->header((string) $this->t('Value'), 'right'),
    ];

    $size_rows = [
      $this->ui->row([
        (string) $this->t('Lines of Code (LOC)'),
        number_format($metrics['loc'] ?? 0),
      ]),
      $this->ui->row([
        (string) $this->t('Comment Lines of Code (CLOC)'),
        number_format($metrics['cloc'] ?? 0),
      ]),
      $this->ui->row([
        (string) $this->t('Non-Comment Lines of Code (NCLOC)'),
        number_format($metrics['ncloc'] ?? 0),
      ]),
      $this->ui->row([
        (string) $this->t('Logical Lines of Code (LLOC)'),
        number_format($metrics['lloc'] ?? 0),
      ]),
    ];

    $content['size'] = $this->ui->section(
      (string) $this->t('Size Metrics'),
      $this->ui->table($size_headers, $size_rows),
      ['open' => FALSE, 'count' => 4]
    );

    // Structure metrics.
    $structure_rows = [
      $this->ui->row([
        (string) $this->t('Files'),
        number_format($metrics['files'] ?? 0),
      ]),
      $this->ui->row([
        (string) $this->t('Directories'),
        number_format($metrics['directories'] ?? 0),
      ]),
      $this->ui->row([
        (string) $this->t('Namespaces'),
        number_format($metrics['namespaces'] ?? 0),
      ]),
      $this->ui->row([
        (string) $this->t('Interfaces'),
        number_format($metrics['interfaces'] ?? 0),
      ]),
      $this->ui->row([
        (string) $this->t('Traits'),
        number_format($metrics['traits'] ?? 0),
      ]),
      $this->ui->row([
        (string) $this->t('Classes'),
        number_format($metrics['classes'] ?? 0),
      ]),
      $this->ui->row([
        (string) $this->t('Abstract Classes'),
        number_format($metrics['abstractClasses'] ?? 0),
      ]),
      $this->ui->row([
        (string) $this->t('Concrete Classes'),
        number_format($metrics['concreteClasses'] ?? 0),
      ]),
    ];

    $content['structure'] = $this->ui->section(
      (string) $this->t('Structure Metrics'),
      $this->ui->table($size_headers, $structure_rows),
      ['open' => FALSE, 'count' => 8]
    );

    // Complexity metrics.
    $complexity_rows = [
      $this->ui->row([
        (string) $this->t('Average Complexity per LLOC'),
        number_format($metrics['ccnByLloc'] ?? 0, 2),
      ]),
      $this->ui->row([
        (string) $this->t('Average Complexity per Class'),
        number_format($metrics['ccnByNom'] ?? 0, 2),
      ]),
      $this->ui->row([
        (string) $this->t('Methods'),
        number_format($metrics['methods'] ?? 0),
      ]),
      $this->ui->row([
        (string) $this->t('Non-Static Methods'),
        number_format($metrics['nonStaticMethods'] ?? 0),
      ]),
      $this->ui->row([
        (string) $this->t('Static Methods'),
        number_format($metrics['staticMethods'] ?? 0),
      ]),
      $this->ui->row([
        (string) $this->t('Public Methods'),
        number_format($metrics['publicMethods'] ?? 0),
      ]),
      $this->ui->row([
        (string) $this->t('Protected Methods'),
        number_format($metrics['protectedMethods'] ?? 0),
      ]),
      $this->ui->row([
        (string) $this->t('Private Methods'),
        number_format($metrics['privateMethods'] ?? 0),
      ]),
    ];

    $content['complexity'] = $this->ui->section(
      (string) $this->t('Complexity & Methods'),
      $this->ui->table($size_headers, $complexity_rows),
      ['open' => FALSE, 'count' => 8]
    );

    return $content;
  }

  /**
   * Builds content for Drupal API Usage section.
   *
   * @param array $data
   *   Analysis data.
   *
   * @return array
   *   Render array.
   */
  protected function buildApiUsageContent(array $data): array {
    $api_usage = $data['api_usage'] ?? [];

    if (empty($api_usage)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('No API usage data available. Run the analysis first.'),
          'info'
        ),
      ];
    }

    $content = [];
    $summary = $api_usage['summary'] ?? [];

    // Summary info box.
    $summary_text = $this->t('Found @hooks hooks, @preprocess preprocess functions, @events event subscribers, and @funcs global function usages.', [
      '@hooks' => $summary['total_hooks'] ?? 0,
      '@preprocess' => $summary['total_preprocess'] ?? 0,
      '@events' => $summary['total_event_subscribers'] ?? 0,
      '@funcs' => $summary['total_global_functions'] ?? 0,
    ]);

    $content['summary'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-info-box']],
      'text' => ['#markup' => '<p>' . $summary_text . '</p>'],
    ];

    // Hooks section.
    $hooks = $api_usage['hooks'] ?? [];
    if (!empty($hooks)) {
      $content['hooks'] = $this->buildApiUsageTable(
        (string) $this->t('Hooks Implemented (@count)', ['@count' => count($hooks)]),
        $hooks,
        'hook'
      );
    }

    // Preprocess section.
    $preprocess = $api_usage['preprocess'] ?? [];
    if (!empty($preprocess)) {
      $content['preprocess'] = $this->buildApiUsageTable(
        (string) $this->t('Preprocess Functions (@count)', ['@count' => count($preprocess)]),
        $preprocess,
        'preprocess'
      );
    }

    // Event subscribers section.
    $event_subscribers = $api_usage['event_subscribers'] ?? [];
    if (!empty($event_subscribers)) {
      $content['events'] = $this->buildEventSubscribersTable($event_subscribers);
    }

    // Global functions section.
    $global_functions = $api_usage['global_functions'] ?? [];
    if (!empty($global_functions)) {
      $content['global_functions'] = $this->buildGlobalFunctionsTable($global_functions);
    }

    if (empty($hooks) && empty($preprocess) && empty($event_subscribers) && empty($global_functions)) {
      $content['empty'] = $this->ui->message(
        (string) $this->t('No Drupal API usage detected in the analyzed code.'),
        'info'
      );
    }

    return $content;
  }

  /**
   * Builds a table for hooks or preprocess functions.
   *
   * @param string $title
   *   Section title.
   * @param array $items
   *   Items to display.
   * @param string $type
   *   Type of items ('hook' or 'preprocess').
   *
   * @return array
   *   Render array.
   */
  protected function buildApiUsageTable(string $title, array $items, string $type): array {
    $headers = [
      $this->ui->header((string) $this->t('Name'), 'left', '30%'),
      $this->ui->header((string) $this->t('Description'), 'left', '25%'),
      $this->ui->header((string) $this->t('Uses'), 'center', '10%'),
      $this->ui->header((string) $this->t('Locations'), 'left', '35%'),
    ];

    $rows = [];
    foreach ($items as $item) {
      $name = $item[$type] ?? $item['hook'] ?? '';
      $locations = $item['locations'] ?? [];
      $count = count($locations);

      // Build locations as list items.
      $locations_html = '<ul class="audit-locations-list">';
      foreach ($locations as $loc) {
        $locations_html .= '<li>' . $loc['file'] . ':' . $loc['line'] . '</li>';
      }
      $locations_html .= '</ul>';

      $rows[] = $this->ui->row([
        $this->ui->cell($name),
        $this->ui->cell($item['description'] ?? ''),
        $this->ui->cell((string) $count, ['align' => 'center']),
        $this->ui->cell($locations_html),
      ]);
    }

    return $this->ui->section(
      $title,
      $this->ui->table($headers, $rows),
      ['open' => TRUE, 'count' => count($items)]
    );
  }

  /**
   * Builds the event subscribers table.
   *
   * @param array $subscribers
   *   Event subscribers data.
   *
   * @return array
   *   Render array.
   */
  protected function buildEventSubscribersTable(array $subscribers): array {
    $headers = [
      $this->ui->header((string) $this->t('Class'), 'left', '30%'),
      $this->ui->header((string) $this->t('Events'), 'left', '40%'),
      $this->ui->header((string) $this->t('Location'), 'left', '30%'),
    ];

    $rows = [];
    foreach ($subscribers as $subscriber) {
      $events_list = $subscriber['events'] ?? [];
      $events_str = !empty($events_list) ? implode(', ', $events_list) : '-';

      $rows[] = $this->ui->row([
        $this->ui->cell($subscriber['class'] ?? ''),
        $this->ui->cell($events_str),
        $this->ui->cell(($subscriber['file'] ?? '') . ':' . ($subscriber['line'] ?? '')),
      ]);
    }

    return $this->ui->section(
      (string) $this->t('Event Subscribers (@count)', ['@count' => count($subscribers)]),
      $this->ui->table($headers, $rows),
      ['open' => TRUE, 'count' => count($subscribers)]
    );
  }

  /**
   * Builds the global functions table.
   *
   * @param array $functions
   *   Global functions data.
   *
   * @return array
   *   Render array.
   */
  protected function buildGlobalFunctionsTable(array $functions): array {
    $headers = [
      $this->ui->header((string) $this->t('Function'), 'left', '20%'),
      $this->ui->header((string) $this->t('Description'), 'left', '25%'),
      $this->ui->header((string) $this->t('Uses'), 'center', '10%'),
      $this->ui->header((string) $this->t('Locations'), 'left', '45%'),
    ];

    $rows = [];
    foreach ($functions as $func) {
      $func_name = $func['function'] ?? '';
      $is_deprecated = $func['is_deprecated'] ?? FALSE;
      $locations = $func['locations'] ?? [];
      $count = count($locations);

      // Build locations as list items (limit to first 10).
      $locations_html = '<ul class="audit-locations-list">';
      $shown = 0;
      foreach ($locations as $loc) {
        if ($shown >= 10) {
          $locations_html .= '<li><em>... +' . ($count - 10) . ' more</em></li>';
          break;
        }
        $locations_html .= '<li>' . $loc['file'] . ':' . $loc['line'] . '</li>';
        $shown++;
      }
      $locations_html .= '</ul>';

      // Add deprecated indicator to function name.
      $display_name = $func_name . '()';
      if ($is_deprecated) {
        $display_name .= ' [DEPRECATED]';
      }

      $severity = $is_deprecated ? 'warning' : NULL;

      $rows[] = $this->ui->row([
        $this->ui->cell($display_name),
        $this->ui->cell($func['description'] ?? ''),
        $this->ui->cell((string) $count, ['align' => 'center']),
        $this->ui->cell($locations_html),
      ], $severity);
    }

    return $this->ui->section(
      (string) $this->t('Global Functions (@count unique, @uses usages)', [
        '@count' => count($functions),
        '@uses' => array_sum(array_map(fn($f) => count($f['locations'] ?? []), $functions)),
      ]),
      $this->ui->table($headers, $rows),
      ['open' => TRUE, 'count' => count($functions)]
    );
  }

  /**
   * Builds installation instructions render array.
   */
  protected function buildInstallationInstructions(): array {
    $instructions = $this->getInstallationInstructions();

    $build = [
      '#type' => 'details',
      '#title' => $instructions['title'],
      '#open' => TRUE,
    ];

    $steps_html = '<ol>';
    foreach ($instructions['steps'] as $step) {
      $steps_html .= '<li>';
      $steps_html .= '<code>' . $step['command'] . '</code>';
      $steps_html .= '<br><small>' . $step['description'] . '</small>';
      $steps_html .= '</li>';
    }
    $steps_html .= '</ol>';

    $build['steps'] = ['#markup' => $steps_html];

    if (!empty($instructions['notes'])) {
      $notes_html = '<ul>';
      foreach ($instructions['notes'] as $note) {
        $notes_html .= '<li>' . $note . '</li>';
      }
      $notes_html .= '</ul>';
      $build['notes'] = [
        '#prefix' => '<p><strong>' . $this->t('Notes:') . '</strong></p>',
        '#markup' => $notes_html,
      ];
    }

    if (!empty($instructions['documentation'])) {
      $build['docs'] = [
        '#markup' => '<p>' . $this->t('Documentation: <a href="@url" target="_blank">@url</a>', [
          '@url' => $instructions['documentation'],
        ]) . '</p>',
      ];
    }

    return $build;
  }

  /**
   * Gets scan directories from global audit configuration.
   *
   * @param \Drupal\Core\Config\ImmutableConfig $audit_config
   *   The audit configuration.
   *
   * @return array
   *   Array of absolute paths to scan.
   */
  protected function getScanDirectories($audit_config): array {
    $scan_dirs_raw = $audit_config->get('scan_directories') ?? "web/modules/custom\nweb/themes/custom";
    $scan_dirs = array_filter(array_map('trim', explode("\n", $scan_dirs_raw)));

    $paths = [];
    foreach ($scan_dirs as $dir) {
      // Try relative to DRUPAL_ROOT first.
      $full_path = DRUPAL_ROOT . '/' . ltrim($dir, '/');
      if (is_dir($full_path)) {
        $paths[] = $full_path;
        continue;
      }

      // Try as absolute path.
      if (is_dir($dir)) {
        $paths[] = $dir;
        continue;
      }

      // Try relative to project root (one level up from DRUPAL_ROOT).
      $project_path = dirname(DRUPAL_ROOT) . '/' . ltrim($dir, '/');
      if (is_dir($project_path)) {
        $paths[] = $project_path;
        continue;
      }

      // Try without web/ prefix (common in some setups).
      if (str_starts_with($dir, 'web/')) {
        $without_web = DRUPAL_ROOT . '/' . substr($dir, 4);
        if (is_dir($without_web)) {
          $paths[] = $without_web;
        }
      }
    }

    return array_unique($paths);
  }

  /**
   * Gets exclude patterns from global audit configuration.
   *
   * @param \Drupal\Core\Config\ImmutableConfig $audit_config
   *   The audit configuration.
   * @param bool $include_tests
   *   Whether to include test files.
   *
   * @return array
   *   Array of exclude patterns.
   */
  protected function getExcludePatterns($audit_config, bool $include_tests = FALSE): array {
    $default_patterns = "node_modules/\nvendor/";
    if (!$include_tests) {
      $default_patterns = "*Test.php\n*TestBase.php\ntests/\nTests/\n" . $default_patterns;
    }

    $exclude_raw = $audit_config->get('exclude_patterns') ?? $default_patterns;
    return array_filter(array_map('trim', explode("\n", $exclude_raw)));
  }

  /**
   * Runs phploc on the specified paths.
   *
   * @param array $paths
   *   Paths to analyze.
   * @param array $exclude_patterns
   *   Patterns to exclude from analysis.
   * @param int $timeout
   *   Execution timeout in seconds.
   *
   * @return array
   *   Analysis results.
   */
  protected function runPhploc(array $paths, array $exclude_patterns, int $timeout): array {
    $phploc_path = $this->findBinary('phploc');
    if (!$phploc_path) {
      return ['error' => (string) $this->t('phploc is not installed. Run: composer require --dev phploc/phploc')];
    }

    // Build command.
    $command = [$phploc_path];

    // Add exclude patterns.
    foreach ($exclude_patterns as $pattern) {
      // Directory patterns (ending with /).
      if (str_ends_with($pattern, '/')) {
        $command[] = '--exclude=' . rtrim($pattern, '/');
      }
    }

    // Output format.
    $command[] = '--log-json';
    $command[] = 'php://stdout';

    // Add paths.
    foreach ($paths as $path) {
      $command[] = $path;
    }

    try {
      $process = new Process($command);
      $process->setTimeout($timeout);
      $process->run();

      $output = $process->getOutput();

      // Try to parse JSON output.
      $json_data = json_decode($output, TRUE);
      if (json_last_error() === JSON_ERROR_NONE && !empty($json_data)) {
        return [
          'metrics' => $json_data,
          'raw_output' => $output,
        ];
      }

      // Fallback: parse text output if JSON not available.
      return $this->parsePhplocTextOutput($output);
    }
    catch (\Exception $e) {
      return ['error' => (string) $this->t('Error running phploc: @error', ['@error' => $e->getMessage()])];
    }
  }

  /**
   * Parses phploc text output when JSON is not available.
   *
   * @param string $output
   *   The text output from phploc.
   *
   * @return array
   *   Parsed metrics.
   */
  protected function parsePhplocTextOutput(string $output): array {
    $metrics = [];

    // Common patterns in phploc text output.
    $patterns = [
      'loc' => '/Lines of Code \(LOC\)\s+([\d,]+)/i',
      'cloc' => '/Comment Lines of Code \(CLOC\)\s+([\d,]+)/i',
      'ncloc' => '/Non-Comment Lines of Code \(NCLOC\)\s+([\d,]+)/i',
      'lloc' => '/Logical Lines of Code \(LLOC\)\s+([\d,]+)/i',
      'classes' => '/Classes\s+([\d,]+)/i',
      'methods' => '/Methods\s+([\d,]+)/i',
      'functions' => '/Functions\s+([\d,]+)/i',
      'files' => '/(?:Directories|Files)\s+([\d,]+)/i',
      'ccnByLloc' => '/Average Complexity per LLOC\s+([\d.]+)/i',
    ];

    foreach ($patterns as $key => $pattern) {
      if (preg_match($pattern, $output, $matches)) {
        $value = str_replace(',', '', $matches[1]);
        $metrics[$key] = is_numeric($value) ? (float) $value : 0;
      }
    }

    return [
      'metrics' => $metrics,
      'raw_output' => $output,
    ];
  }

  /**
   * Analyzes Drupal-specific anti-patterns in the code.
   *
   * @param array $paths
   *   Paths to analyze.
   * @param array $exclude_patterns
   *   Patterns to exclude from analysis.
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *   Configuration object.
   *
   * @return array
   *   Analysis results.
   */
  protected function analyzeAntiPatterns(array $paths, array $exclude_patterns, $config): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    $detect_service_locators = (bool) ($config->get('detect_service_locators') ?? TRUE);
    $detect_deep_arrays = (bool) ($config->get('detect_deep_arrays') ?? TRUE);
    $detect_hardcoded_ids = (bool) ($config->get('detect_hardcoded_ids') ?? TRUE);
    $detect_direct_queries = (bool) ($config->get('detect_direct_queries') ?? TRUE);
    $deep_threshold = (int) ($config->get('deep_array_threshold') ?? self::DEFAULT_DEEP_ARRAY_THRESHOLD);
    $allow_procedural = (bool) ($config->get('allow_service_locators_in_procedural') ?? TRUE);

    foreach ($paths as $base_path) {
      $files = $this->findPhpFiles($base_path, $exclude_patterns);

      foreach ($files as $file) {
        $content = @file_get_contents($file);
        if ($content === FALSE) {
          continue;
        }

        $relative_path = str_replace(DRUPAL_ROOT . '/', '', $file);

        // Check if this is a procedural file (.module, .install, .theme).
        $is_procedural = (bool) preg_match('/\.(module|install|theme)$/', $file);

        // Detect service locators with line numbers.
        // Skip procedural files when allow_service_locators_in_procedural is enabled.
        if ($detect_service_locators && !($allow_procedural && $is_procedural)) {
          foreach (self::SERVICE_LOCATOR_PATTERNS as $pattern => $description) {
            $occurrences = $this->findPatternOccurrences($content, preg_quote($pattern, '/'));
            foreach ($occurrences as $occurrence) {
              $results[] = $this->createResultItem(
                'warning',
                'SERVICE_LOCATOR',
                (string) $this->t('@pattern at line @line', [
                  '@pattern' => $pattern,
                  '@line' => $occurrence['line'],
                ]),
                [
                  'file' => $relative_path,
                  'absolute_file' => $file,
                  'line' => $occurrence['line'],
                  'type' => 'service_locator',
                  'pattern' => $pattern,
                  'suggestion' => (string) $this->t('Use dependency injection instead. Inject the service in the constructor.'),
                ]
              );
              $warnings++;
            }
          }
        }

        // Detect direct database queries with line numbers.
        if ($detect_direct_queries) {
          $query_patterns = [
            'db_query(' => (string) $this->t('Use dependency injection with Connection service'),
            'db_select(' => (string) $this->t('Use dependency injection with Connection service'),
            'db_insert(' => (string) $this->t('Use dependency injection with Connection service'),
            'db_update(' => (string) $this->t('Use dependency injection with Connection service'),
            'db_delete(' => (string) $this->t('Use dependency injection with Connection service'),
          ];

          foreach ($query_patterns as $pattern => $suggestion) {
            $occurrences = $this->findPatternOccurrences($content, preg_quote($pattern, '/'));
            foreach ($occurrences as $occurrence) {
              $results[] = $this->createResultItem(
                'warning',
                'DIRECT_QUERY',
                (string) $this->t('@pattern at line @line', [
                  '@pattern' => $pattern,
                  '@line' => $occurrence['line'],
                ]),
                [
                  'file' => $relative_path,
                  'absolute_file' => $file,
                  'line' => $occurrence['line'],
                  'type' => 'direct_query',
                  'pattern' => $pattern,
                  'suggestion' => $suggestion,
                ]
              );
              $warnings++;
            }
          }
        }

        // Detect hardcoded entity IDs with line numbers.
        if ($detect_hardcoded_ids) {
          $occurrences = $this->findPatternOccurrences($content, '->load\s*\(\s*(\d+)\s*\)');
          foreach ($occurrences as $occurrence) {
            // Extract the ID from the match.
            if (preg_match('/->load\s*\(\s*(\d+)\s*\)/', $occurrence['match'], $id_match)) {
              $id = (int) $id_match[1];
              if ($id > 0) {
                $results[] = $this->createResultItem(
                  'notice',
                  'HARDCODED_ID',
                  (string) $this->t('Hardcoded entity ID @id at line @line', [
                    '@id' => $id,
                    '@line' => $occurrence['line'],
                  ]),
                  [
                    'file' => $relative_path,
                    'absolute_file' => $file,
                    'line' => $occurrence['line'],
                    'type' => 'hardcoded_id',
                    'pattern' => "->load($id)",
                    'suggestion' => (string) $this->t('Use configuration or machine names instead of hardcoded IDs.'),
                  ]
                );
                $notices++;
              }
            }
          }
        }

        // Detect deep arrays.
        if ($detect_deep_arrays) {
          $deep_array_info = $this->findDeepArrayWithLine($content, $deep_threshold);
          if ($deep_array_info !== NULL) {
            $results[] = $this->createResultItem(
              'notice',
              'DEEP_ARRAY',
              (string) $this->t('Deep array nesting (@depth levels) at line @line', [
                '@depth' => $deep_array_info['depth'],
                '@line' => $deep_array_info['line'],
              ]),
              [
                'file' => $relative_path,
                'absolute_file' => $file,
                'line' => $deep_array_info['line'],
                'type' => 'deep_array',
                'depth' => $deep_array_info['depth'],
                'threshold' => $deep_threshold,
                'suggestion' => (string) $this->t('Consider restructuring to reduce nesting depth.'),
              ]
            );
            $notices++;
          }
        }
      }
    }

    return [
      'results' => $results,
      'errors' => $errors,
      'warnings' => $warnings,
      'notices' => $notices,
      'total' => count($results),
    ];
  }

  /**
   * Finds all occurrences of a pattern in content with line numbers.
   *
   * @param string $content
   *   The content to search.
   * @param string $pattern
   *   The regex pattern (without delimiters).
   *
   * @return array
   *   Array of occurrences with 'line' and 'match' keys.
   */
  protected function findPatternOccurrences(string $content, string $pattern): array {
    $occurrences = [];

    if (preg_match_all('/' . $pattern . '/', $content, $matches, PREG_OFFSET_CAPTURE)) {
      foreach ($matches[0] as $match) {
        $position = $match[1];
        $line = substr_count(substr($content, 0, $position), "\n") + 1;
        $occurrences[] = [
          'line' => $line,
          'match' => $match[0],
          'position' => $position,
        ];
      }
    }

    return $occurrences;
  }

  /**
   * Finds deep array nesting with line number.
   *
   * @param string $content
   *   PHP file content.
   * @param int $threshold
   *   Depth threshold.
   *
   * @return array|null
   *   Array with 'depth' and 'line' or NULL if not found.
   */
  protected function findDeepArrayWithLine(string $content, int $threshold): ?array {
    $max_depth = 0;
    $max_depth_position = 0;
    $current_depth = 0;

    $len = strlen($content);
    $in_string = FALSE;
    $string_char = '';

    for ($i = 0; $i < $len; $i++) {
      $char = $content[$i];
      $prev = $i > 0 ? $content[$i - 1] : '';

      // Track strings.
      if (!$in_string && ($char === '"' || $char === "'")) {
        $in_string = TRUE;
        $string_char = $char;
      }
      elseif ($in_string && $char === $string_char && $prev !== '\\') {
        $in_string = FALSE;
      }

      if (!$in_string) {
        if ($char === '[') {
          $current_depth++;
          if ($current_depth > $max_depth) {
            $max_depth = $current_depth;
            $max_depth_position = $i;
          }
        }
        elseif ($char === ']') {
          $current_depth = max(0, $current_depth - 1);
        }
      }
    }

    if ($max_depth >= $threshold) {
      $line = substr_count(substr($content, 0, $max_depth_position), "\n") + 1;
      return [
        'depth' => $max_depth,
        'line' => $line,
      ];
    }

    return NULL;
  }

  /**
   * Analyzes complexity hotspots using simple pattern matching.
   *
   * @param array $paths
   *   Paths to analyze.
   * @param array $exclude_patterns
   *   Patterns to exclude from analysis.
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *   Configuration object.
   *
   * @return array
   *   Analysis results with hotspots.
   */
  protected function analyzeComplexityHotspots(array $paths, array $exclude_patterns, $config): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    $ccn_warning = (int) ($config->get('ccn_warning_threshold') ?? self::DEFAULT_CCN_WARNING);
    $ccn_error = (int) ($config->get('ccn_error_threshold') ?? self::DEFAULT_CCN_ERROR);
    $max_hotspots = (int) ($config->get('max_hotspots_display') ?? self::DEFAULT_MAX_HOTSPOTS);

    $hotspots = [];

    foreach ($paths as $base_path) {
      $files = $this->findPhpFiles($base_path, $exclude_patterns);

      foreach ($files as $file) {
        $content = @file_get_contents($file);
        if ($content === FALSE) {
          continue;
        }

        $relative_path = str_replace(DRUPAL_ROOT . '/', '', $file);

        // Extract functions and methods with their complexity.
        $functions = $this->extractFunctionsWithComplexity($content, $relative_path, $file);

        foreach ($functions as $func) {
          if ($func['ccn'] >= $ccn_warning) {
            $hotspots[] = $func;
          }
        }
      }
    }

    // Sort by complexity (highest first).
    usort($hotspots, fn($a, $b) => $b['ccn'] - $a['ccn']);

    // Limit to max hotspots.
    $hotspots = array_slice($hotspots, 0, $max_hotspots);

    // Convert to result items.
    foreach ($hotspots as $hotspot) {
      $severity = 'warning';
      if ($hotspot['ccn'] >= $ccn_error) {
        $severity = 'error';
        $errors++;
      }
      else {
        $warnings++;
      }

      $results[] = $this->createResultItem(
        $severity,
        'HIGH_CCN',
        (string) $this->t('@function has CCN of @ccn', [
          '@function' => $hotspot['function'],
          '@ccn' => $hotspot['ccn'],
        ]),
        [
          'function' => $hotspot['function'],
          'ccn' => $hotspot['ccn'],
          'file' => $hotspot['file'],
          'absolute_file' => $hotspot['absolute_file'] ?? NULL,
          'line' => $hotspot['line'] ?? NULL,
        ]
      );
    }

    return [
      'results' => $results,
      'errors' => $errors,
      'warnings' => $warnings,
      'notices' => $notices,
      'total_hotspots' => count($hotspots),
    ];
  }

  /**
   * Extracts functions/methods and calculates basic CCN.
   *
   * @param string $content
   *   PHP file content.
   * @param string $file_path
   *   Relative file path.
   * @param string|null $absolute_path
   *   Absolute file path for code snippets.
   *
   * @return array
   *   Array of functions with their complexity.
   */
  protected function extractFunctionsWithComplexity(string $content, string $file_path, ?string $absolute_path = NULL): array {
    $functions = [];

    // Find function/method definitions.
    $pattern = '/(?:public|protected|private|static|\s)*\s*function\s+(\w+)\s*\([^)]*\)\s*(?::\s*[?\w\\\\]+)?\s*\{/';

    if (preg_match_all($pattern, $content, $matches, PREG_OFFSET_CAPTURE)) {
      foreach ($matches[1] as $index => $match) {
        $func_name = $match[0];
        $start_pos = $matches[0][$index][1];

        // Find the line number.
        $line = substr_count(substr($content, 0, $start_pos), "\n") + 1;

        // Extract function body (simplified - find matching braces).
        $body = $this->extractFunctionBody($content, $start_pos);

        // Calculate CCN.
        $ccn = $this->calculateCcn($body);

        $functions[] = [
          'function' => $func_name,
          'file' => $file_path,
          'absolute_file' => $absolute_path,
          'line' => $line,
          'ccn' => $ccn,
        ];
      }
    }

    return $functions;
  }

  /**
   * Extracts function body starting from a position.
   *
   * @param string $content
   *   Full file content.
   * @param int $start_pos
   *   Position of function definition.
   *
   * @return string
   *   Function body content.
   */
  protected function extractFunctionBody(string $content, int $start_pos): string {
    $brace_pos = strpos($content, '{', $start_pos);
    if ($brace_pos === FALSE) {
      return '';
    }

    $depth = 0;
    $body_start = $brace_pos + 1;
    $length = strlen($content);
    $in_string = FALSE;
    $string_char = '';

    for ($i = $brace_pos; $i < $length; $i++) {
      $char = $content[$i];
      $prev = $i > 0 ? $content[$i - 1] : '';

      // Track strings to avoid counting braces inside them.
      if (!$in_string && ($char === '"' || $char === "'")) {
        $in_string = TRUE;
        $string_char = $char;
      }
      elseif ($in_string && $char === $string_char && $prev !== '\\') {
        $in_string = FALSE;
      }

      if (!$in_string) {
        if ($char === '{') {
          $depth++;
        }
        elseif ($char === '}') {
          $depth--;
          if ($depth === 0) {
            return substr($content, $body_start, $i - $body_start);
          }
        }
      }
    }

    return substr($content, $body_start);
  }

  /**
   * Calculates cyclomatic complexity for a code block.
   *
   * CCN = 1 + number of decision points.
   * Decision points: if, elseif, else, switch, case, for, foreach, while,
   * do, catch, &&, ||, ?:, throw.
   *
   * @param string $code
   *   The code block to analyze.
   *
   * @return int
   *   The cyclomatic complexity number.
   */
  protected function calculateCcn(string $code): int {
    $ccn = 1;

    // Decision keywords (word boundaries to avoid partial matches).
    $patterns = [
      '/\bif\s*\(/',
      '/\belseif\s*\(/',
      '/\belse\s*\{/',
      '/\bswitch\s*\(/',
      '/\bcase\s+/',
      '/\bfor\s*\(/',
      '/\bforeach\s*\(/',
      '/\bwhile\s*\(/',
      '/\bdo\s*\{/',
      '/\bcatch\s*\(/',
      '/\bthrow\s+/',
      '/\?\?/',  // Null coalescing.
      '/\?(?!=)/',  // Ternary (not ?=).
      '/&&/',
      '/\|\|/',
    ];

    foreach ($patterns as $pattern) {
      $ccn += preg_match_all($pattern, $code);
    }

    return $ccn;
  }

  /**
   * Analyzes Drupal API usage in custom code.
   *
   * Detects:
   * - Hooks implemented (function patterns like modulename_hook_name)
   * - Preprocess functions (modulename_preprocess_*)
   * - Event subscribers (classes implementing EventSubscriberInterface)
   * - Global Drupal functions (t(), drupal_*(), etc.)
   *
   * @param array $paths
   *   Paths to analyze.
   * @param array $exclude_patterns
   *   Patterns to exclude.
   *
   * @return array
   *   API usage analysis results.
   */
  protected function analyzeApiUsage(array $paths, array $exclude_patterns): array {
    $hooks = [];
    $preprocess = [];
    $event_subscribers = [];
    $global_functions = [];

    // Common Drupal global functions to track.
    $drupal_functions = [
      't' => 'Translation function',
      'drupal_flush_all_caches' => 'Cache flush',
      'drupal_set_message' => 'Deprecated message function',
      'watchdog' => 'Deprecated logging',
      'format_date' => 'Deprecated date formatting',
      'check_plain' => 'Deprecated sanitization',
      'drupal_render' => 'Render function',
      'drupal_get_path' => 'Deprecated path function',
      'drupal_static' => 'Static variable storage',
      'drupal_static_reset' => 'Static reset',
      'module_load_include' => 'Module include',
      'db_query' => 'Deprecated direct database query',
      'db_select' => 'Deprecated database select',
      'db_insert' => 'Deprecated database insert',
      'db_update' => 'Deprecated database update',
      'db_delete' => 'Deprecated database delete',
      'entity_load' => 'Deprecated entity load',
      'node_load' => 'Deprecated node load',
      'user_load' => 'Deprecated user load',
      'taxonomy_term_load' => 'Deprecated term load',
      'file_save_data' => 'File save',
      'file_prepare_directory' => 'Deprecated directory preparation',
      'variable_get' => 'Deprecated D7 variable',
      'variable_set' => 'Deprecated D7 variable',
      'cache_get' => 'Deprecated cache get',
      'cache_set' => 'Deprecated cache set',
      'cache_clear_all' => 'Deprecated cache clear',
    ];

    foreach ($paths as $base_path) {
      $files = $this->findPhpFiles($base_path, $exclude_patterns);

      foreach ($files as $file) {
        $content = @file_get_contents($file);
        if ($content === FALSE) {
          continue;
        }

        $relative_path = str_replace(DRUPAL_ROOT . '/', '', $file);
        $extension = pathinfo($file, PATHINFO_EXTENSION);

        // Detect hooks in .module, .install, .theme files.
        if (in_array($extension, ['module', 'install', 'theme'], TRUE)) {
          $this->detectHooks($content, $relative_path, $hooks, $preprocess);
        }

        // Detect event subscribers in PHP files.
        if ($extension === 'php') {
          $this->detectEventSubscribers($content, $relative_path, $event_subscribers);
        }

        // Detect global function usage in all PHP files.
        $this->detectGlobalFunctions($content, $relative_path, $drupal_functions, $global_functions);
      }
    }

    // Sort by usage count (descending).
    uasort($hooks, fn($a, $b) => count($b['locations']) - count($a['locations']));
    uasort($preprocess, fn($a, $b) => count($b['locations']) - count($a['locations']));
    uasort($global_functions, fn($a, $b) => count($b['locations']) - count($a['locations']));

    return [
      'hooks' => $hooks,
      'preprocess' => $preprocess,
      'event_subscribers' => $event_subscribers,
      'global_functions' => $global_functions,
      'summary' => [
        'total_hooks' => count($hooks),
        'total_preprocess' => count($preprocess),
        'total_event_subscribers' => count($event_subscribers),
        'total_global_functions' => count($global_functions),
      ],
    ];
  }

  /**
   * Detects hook implementations in module/install/theme files.
   *
   * @param string $content
   *   File content.
   * @param string $relative_path
   *   Relative file path.
   * @param array $hooks
   *   Reference to hooks array.
   * @param array $preprocess
   *   Reference to preprocess array.
   */
  protected function detectHooks(string $content, string $relative_path, array &$hooks, array &$preprocess): void {
    // Get module/theme name from filename.
    $filename = basename($relative_path);
    $module_name = preg_replace('/\.(module|install|theme)$/', '', $filename);

    // Pattern to find function definitions.
    $pattern = '/^\s*function\s+(' . preg_quote($module_name, '/') . '_\w+)\s*\(/m';

    if (preg_match_all($pattern, $content, $matches, PREG_OFFSET_CAPTURE)) {
      foreach ($matches[1] as $match) {
        $function_name = $match[0];
        $position = $match[1];
        $line = substr_count(substr($content, 0, $position), "\n") + 1;

        // Check if it's a preprocess function.
        if (preg_match('/^' . preg_quote($module_name, '/') . '_preprocess_(.+)$/', $function_name, $preprocess_match)) {
          $hook_name = 'preprocess_' . $preprocess_match[1];
          if (!isset($preprocess[$hook_name])) {
            $preprocess[$hook_name] = [
              'hook' => $hook_name,
              'description' => 'Preprocess for ' . $preprocess_match[1],
              'locations' => [],
            ];
          }
          $preprocess[$hook_name]['locations'][] = [
            'file' => $relative_path,
            'line' => $line,
            'function' => $function_name,
          ];
        }
        // Check if it's a hook implementation.
        elseif (preg_match('/^' . preg_quote($module_name, '/') . '_(.+)$/', $function_name, $hook_match)) {
          $potential_hook = $hook_match[1];
          // Skip internal functions (starting with underscore after module name).
          if (!str_starts_with($potential_hook, '_')) {
            $hook_name = 'hook_' . $potential_hook;
            if (!isset($hooks[$hook_name])) {
              $hooks[$hook_name] = [
                'hook' => $hook_name,
                'description' => $this->getHookDescription($potential_hook),
                'locations' => [],
              ];
            }
            $hooks[$hook_name]['locations'][] = [
              'file' => $relative_path,
              'line' => $line,
              'function' => $function_name,
            ];
          }
        }
      }
    }
  }

  /**
   * Gets a description for common hooks.
   *
   * @param string $hook_name
   *   The hook name (without hook_ prefix).
   *
   * @return string
   *   Description of the hook.
   */
  protected function getHookDescription(string $hook_name): string {
    $descriptions = [
      'form_alter' => 'Alter any form',
      'form_FORM_ID_alter' => 'Alter specific form',
      'entity_presave' => 'Before entity save',
      'entity_insert' => 'After entity insert',
      'entity_update' => 'After entity update',
      'entity_delete' => 'After entity delete',
      'entity_view' => 'Alter entity view',
      'node_presave' => 'Before node save',
      'node_insert' => 'After node insert',
      'node_update' => 'After node update',
      'node_delete' => 'After node delete',
      'user_login' => 'User login event',
      'user_logout' => 'User logout event',
      'cron' => 'Cron task',
      'theme' => 'Theme hook definitions',
      'install' => 'Module install',
      'uninstall' => 'Module uninstall',
      'requirements' => 'Requirements check',
      'schema' => 'Database schema',
      'update_N' => 'Update hook',
      'help' => 'Help page',
      'permission' => 'Permission definitions',
      'menu_links_discovered_alter' => 'Alter menu links',
      'library_info_alter' => 'Alter libraries',
      'page_attachments' => 'Page attachments',
      'page_attachments_alter' => 'Alter page attachments',
      'theme_suggestions_alter' => 'Alter theme suggestions',
      'views_data' => 'Views data definitions',
      'views_data_alter' => 'Alter views data',
      'query_alter' => 'Alter database query',
      'entity_type_alter' => 'Alter entity types',
      'field_widget_form_alter' => 'Alter field widgets',
      'tokens' => 'Token definitions',
      'token_info' => 'Token info',
    ];

    // Check for pattern matches.
    foreach ($descriptions as $pattern => $desc) {
      if ($hook_name === $pattern || preg_match('/^' . str_replace('_N', '_\d+', preg_quote($pattern, '/')) . '$/', $hook_name)) {
        return $desc;
      }
    }

    // Check for common patterns.
    if (str_starts_with($hook_name, 'form_') && str_ends_with($hook_name, '_alter')) {
      return 'Form alter';
    }
    if (str_ends_with($hook_name, '_alter')) {
      return 'Alter hook';
    }
    if (str_starts_with($hook_name, 'preprocess_')) {
      return 'Preprocess';
    }

    return 'Hook implementation';
  }

  /**
   * Detects event subscriber implementations.
   *
   * @param string $content
   *   File content.
   * @param string $relative_path
   *   Relative file path.
   * @param array $event_subscribers
   *   Reference to event subscribers array.
   */
  protected function detectEventSubscribers(string $content, string $relative_path, array &$event_subscribers): void {
    // Check if class implements EventSubscriberInterface.
    if (!str_contains($content, 'EventSubscriberInterface')) {
      return;
    }

    // Find class name.
    if (preg_match('/class\s+(\w+)\s+.*implements\s+[^{]*EventSubscriberInterface/s', $content, $class_match)) {
      $class_name = $class_match[1];
      $class_line = substr_count(substr($content, 0, strpos($content, $class_match[0])), "\n") + 1;

      // Find getSubscribedEvents method.
      $events = [];
      if (preg_match('/function\s+getSubscribedEvents\s*\(\s*\)[^{]*\{([^}]+)\}/s', $content, $method_match)) {
        // Extract event names from the method.
        if (preg_match_all('/[\'"]([A-Za-z_:\\\\]+)[\'"]/', $method_match[1], $event_matches)) {
          $events = $event_matches[1];
        }
      }

      $event_subscribers[$class_name] = [
        'class' => $class_name,
        'file' => $relative_path,
        'line' => $class_line,
        'events' => $events,
      ];
    }
  }

  /**
   * Detects usage of global Drupal functions.
   *
   * @param string $content
   *   File content.
   * @param string $relative_path
   *   Relative file path.
   * @param array $drupal_functions
   *   List of functions to detect.
   * @param array $global_functions
   *   Reference to results array.
   */
  protected function detectGlobalFunctions(string $content, string $relative_path, array $drupal_functions, array &$global_functions): void {
    foreach ($drupal_functions as $func_name => $description) {
      // Pattern for function calls (not definitions or string literals).
      $pattern = '/\b' . preg_quote($func_name, '/') . '\s*\(/';

      if (preg_match_all($pattern, $content, $matches, PREG_OFFSET_CAPTURE)) {
        foreach ($matches[0] as $match) {
          $position = $match[1];

          // Skip if this is inside a string literal.
          if ($this->isInsideString($content, $position)) {
            continue;
          }

          // Get the line content before the match.
          $line_start = strrpos(substr($content, 0, $position), "\n");
          if ($line_start === FALSE) {
            $line_start = 0;
          }
          $line_content = substr($content, $line_start, $position - $line_start);

          // Skip if it's part of a comment.
          if (str_contains($line_content, '//') || preg_match('/^\s*\*/', $line_content)) {
            continue;
          }

          $line = substr_count(substr($content, 0, $position), "\n") + 1;

          if (!isset($global_functions[$func_name])) {
            $global_functions[$func_name] = [
              'function' => $func_name,
              'description' => $description,
              'is_deprecated' => str_contains(strtolower($description), 'deprecated'),
              'locations' => [],
            ];
          }
          $global_functions[$func_name]['locations'][] = [
            'file' => $relative_path,
            'line' => $line,
          ];
        }
      }
    }
  }

  /**
   * Checks if a position in the content is inside a string literal.
   *
   * @param string $content
   *   The file content.
   * @param int $position
   *   The position to check.
   *
   * @return bool
   *   TRUE if the position is inside a string literal.
   */
  protected function isInsideString(string $content, int $position): bool {
    $in_single_quote = FALSE;
    $in_double_quote = FALSE;

    for ($i = 0; $i < $position && $i < strlen($content); $i++) {
      $char = $content[$i];
      $prev = $i > 0 ? $content[$i - 1] : '';

      // Skip escaped characters.
      if ($prev === '\\') {
        continue;
      }

      if ($char === "'" && !$in_double_quote) {
        $in_single_quote = !$in_single_quote;
      }
      elseif ($char === '"' && !$in_single_quote) {
        $in_double_quote = !$in_double_quote;
      }
    }

    return $in_single_quote || $in_double_quote;
  }

  /**
   * Finds all PHP files in a directory recursively.
   *
   * @param string $directory
   *   Base directory to search.
   *
   * @return array
   *   Array of file paths.
   */
  protected function findPhpFiles(string $directory, array $exclude_patterns = []): array {
    $files = [];

    if (!is_dir($directory)) {
      return $files;
    }

    $iterator = new \RecursiveIteratorIterator(
      new \RecursiveDirectoryIterator($directory, \RecursiveDirectoryIterator::SKIP_DOTS)
    );

    foreach ($iterator as $file) {
      if ($file->isFile()) {
        $extension = $file->getExtension();
        if (in_array($extension, ['php', 'module', 'inc', 'install', 'theme'], TRUE)) {
          $filepath = $file->getPathname();

          // Check if file matches any exclude pattern.
          if ($this->shouldExcludeFile($filepath, $exclude_patterns)) {
            continue;
          }

          $files[] = $filepath;
        }
      }
    }

    return $files;
  }

  /**
   * Checks if a file should be excluded based on patterns.
   *
   * @param string $filepath
   *   The file path to check.
   * @param array $exclude_patterns
   *   Array of exclude patterns.
   *
   * @return bool
   *   TRUE if the file should be excluded.
   */
  protected function shouldExcludeFile(string $filepath, array $exclude_patterns): bool {
    foreach ($exclude_patterns as $pattern) {
      // Directory patterns (ending with /).
      if (str_ends_with($pattern, '/')) {
        $dir_pattern = '/' . rtrim($pattern, '/') . '/';
        if (str_contains($filepath, $dir_pattern)) {
          return TRUE;
        }
      }
      // File patterns (like *Test.php).
      elseif (str_starts_with($pattern, '*')) {
        $suffix = substr($pattern, 1);
        if (str_ends_with($filepath, $suffix)) {
          return TRUE;
        }
      }
      // Exact match.
      elseif (str_contains($filepath, '/' . $pattern)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Calculates scores for all factors.
   *
   * @param array $phploc_results
   *   Phploc analysis results.
   * @param array $antipattern_results
   *   Anti-pattern analysis results.
   * @param array $hotspots
   *   Complexity hotspots.
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *   Configuration object.
   *
   * @return array
   *   Score data with overall and factors.
   */
  protected function calculateScores(array $phploc_results, array $antipattern_results, array $hotspots, $config): array {
    $factors = [];

    // Cyclomatic complexity score.
    $hotspot_errors = $hotspots['errors'] ?? 0;
    $hotspot_warnings = $hotspots['warnings'] ?? 0;
    $ccn_score = 100;
    $ccn_score -= $hotspot_errors * 15;  // -15 per error.
    $ccn_score -= $hotspot_warnings * 5;  // -5 per warning.
    $ccn_score = max(0, $ccn_score);

    $factors['cyclomatic_complexity'] = [
      'score' => $ccn_score,
      'weight' => self::SCORE_WEIGHTS['cyclomatic_complexity'],
      'label' => (string) $this->t('Cyclomatic Complexity'),
      'description' => $hotspot_errors + $hotspot_warnings === 0
        ? (string) $this->t('All functions within acceptable complexity')
        : (string) $this->t('@count function(s) with high complexity', [
            '@count' => $hotspot_errors + $hotspot_warnings,
          ]),
    ];

    // Anti-patterns score.
    $antipattern_warnings = $antipattern_results['warnings'] ?? 0;
    $antipattern_notices = $antipattern_results['notices'] ?? 0;
    $antipattern_score = 100;
    $antipattern_score -= $antipattern_warnings * 3;  // -3 per warning.
    $antipattern_score -= $antipattern_notices * 1;   // -1 per notice.
    $antipattern_score = max(0, $antipattern_score);

    $antipattern_total = $antipattern_warnings + $antipattern_notices;
    $factors['anti_patterns'] = [
      'score' => $antipattern_score,
      'weight' => self::SCORE_WEIGHTS['anti_patterns'],
      'label' => (string) $this->t('Drupal Best Practices'),
      'description' => $antipattern_total === 0
        ? (string) $this->t('No anti-patterns detected')
        : (string) $this->t('@count anti-pattern(s) found', ['@count' => $antipattern_total]),
    ];

    // Code structure score (based on phploc metrics).
    $metrics = $phploc_results['metrics'] ?? [];
    $structure_score = 100;

    // Penalize if too few classes (procedural code).
    $classes = $metrics['classes'] ?? 0;
    $functions = $metrics['functions'] ?? 0;
    if ($classes === 0 && $functions > 10) {
      $structure_score -= 20;  // Too much procedural code.
    }

    // Penalize very low comment ratio.
    $loc = $metrics['loc'] ?? 0;
    $cloc = $metrics['cloc'] ?? 0;
    if ($loc > 0) {
      $comment_ratio = $cloc / $loc;
      if ($comment_ratio < 0.05) {
        $structure_score -= 10;  // Less than 5% comments.
      }
    }

    $structure_score = max(0, $structure_score);

    $factors['code_structure'] = [
      'score' => $structure_score,
      'weight' => self::SCORE_WEIGHTS['code_structure'],
      'label' => (string) $this->t('Code Structure'),
      'description' => $structure_score === 100
        ? (string) $this->t('Good code organization')
        : (string) $this->t('Structure could be improved'),
    ];

    return [
      'factors' => $factors,
    ];
  }

  /**
   * Builds empty results when no paths to analyze.
   *
   * @return array
   *   Empty results structure.
   */
  protected function buildEmptyResults(): array {
    $message = (string) $this->t('No custom modules or themes found to analyze.');

    return [
      '_files' => [
        'overview' => $this->createResult([], 0, 0, 0),
        'complexity_hotspots' => $this->createResult([], 0, 0, 0),
        'anti_patterns' => $this->createResult([], 0, 0, 0),
        'code_metrics' => $this->createResult([], 0, 0, 0),
      ],
      'score' => [
        'factors' => [
          'cyclomatic_complexity' => [
            'score' => 100,
            'weight' => self::SCORE_WEIGHTS['cyclomatic_complexity'],
            'label' => (string) $this->t('Cyclomatic Complexity'),
            'description' => $message,
          ],
          'anti_patterns' => [
            'score' => 100,
            'weight' => self::SCORE_WEIGHTS['anti_patterns'],
            'label' => (string) $this->t('Drupal Best Practices'),
            'description' => $message,
          ],
          'code_structure' => [
            'score' => 100,
            'weight' => self::SCORE_WEIGHTS['code_structure'],
            'label' => (string) $this->t('Code Structure'),
            'description' => $message,
          ],
        ],
      ],
      'phploc' => ['error' => $message],
      'antipatterns' => [],
      'hotspots' => [],
      'paths_analyzed' => [],
    ];
  }

  /**
   * Assesses LOC value.
   *
   * @param int|float $loc
   *   Lines of code.
   *
   * @return string
   *   Assessment text.
   */
  protected function assessLoc(int|float $loc): string {
    if ($loc < 1000) {
      return (string) $this->t('Small codebase');
    }
    elseif ($loc < 10000) {
      return (string) $this->t('Medium codebase');
    }
    elseif ($loc < 50000) {
      return (string) $this->t('Large codebase');
    }
    else {
      return (string) $this->t('Very large codebase');
    }
  }

  /**
   * Assesses CCN value.
   *
   * @param float $ccn
   *   Average cyclomatic complexity.
   *
   * @return array
   *   Assessment with label and severity.
   */
  protected function assessCcn(float $ccn): array {
    if ($ccn <= 4) {
      return [
        'label' => (string) $this->t('Excellent - Low complexity'),
        'severity' => NULL,
      ];
    }
    elseif ($ccn <= 7) {
      return [
        'label' => (string) $this->t('Good - Moderate complexity'),
        'severity' => NULL,
      ];
    }
    elseif ($ccn <= 10) {
      return [
        'label' => (string) $this->t('Acceptable - Higher complexity'),
        'severity' => 'warning',
      ];
    }
    else {
      return [
        'label' => (string) $this->t('High - Consider refactoring'),
        'severity' => 'error',
      ];
    }
  }

  /**
   * Finds a binary in vendor/bin directories.
   *
   * @param string $name
   *   Binary name.
   *
   * @return string|null
   *   Path to binary or NULL if not found.
   */
  protected function findBinary(string $name): ?string {
    $paths = [
      DRUPAL_ROOT . '/../vendor/bin/' . $name,
      DRUPAL_ROOT . '/vendor/bin/' . $name,
      dirname(DRUPAL_ROOT) . '/vendor/bin/' . $name,
    ];

    foreach ($paths as $path) {
      if (file_exists($path) && is_executable($path)) {
        return $path;
      }
      // Also check without executable permission (Windows).
      if (file_exists($path)) {
        return $path;
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $config): array {
    $form = [];

    // Analysis scope.
    $form['scope'] = [
      '#type' => 'details',
      '#title' => $this->t('Analysis Scope'),
      '#open' => TRUE,
      '#tree' => FALSE,
    ];

    $form['scope']['paths_info'] = [
      '#type' => 'item',
      '#markup' => $this->t('Paths to analyze and exclude patterns are configured in the <a href=":url">main audit settings</a> (scan_directories and exclude_patterns).', [
        ':url' => '/admin/reports/audit/settings',
      ]),
    ];

    $form['scope']['include_tests'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include test files'),
      '#description' => $this->t('Include test files in analysis. Not recommended - tests have different complexity patterns.'),
      '#default_value' => $config['include_tests'] ?? FALSE,
    ];

    // Complexity thresholds.
    $form['thresholds'] = [
      '#type' => 'details',
      '#title' => $this->t('Complexity Thresholds'),
      '#open' => FALSE,
      '#tree' => FALSE,
    ];

    $form['thresholds']['ccn_warning_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('CCN warning threshold'),
      '#description' => $this->t('Functions with cyclomatic complexity above this value will generate warnings. <strong>Recommended: 10</strong>'),
      '#default_value' => $config['ccn_warning_threshold'] ?? self::DEFAULT_CCN_WARNING,
      '#min' => 1,
      '#max' => 50,
    ];

    $form['thresholds']['ccn_error_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('CCN error threshold'),
      '#description' => $this->t('Functions with cyclomatic complexity above this value will generate errors. <strong>Recommended: 20</strong>'),
      '#default_value' => $config['ccn_error_threshold'] ?? self::DEFAULT_CCN_ERROR,
      '#min' => 1,
      '#max' => 100,
    ];

    $form['thresholds']['deep_array_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Deep array threshold'),
      '#description' => $this->t('Arrays nested deeper than this level will be flagged. Recommended: 4 (Drupal render arrays commonly use 3-4 levels).'),
      '#default_value' => $config['deep_array_threshold'] ?? self::DEFAULT_DEEP_ARRAY_THRESHOLD,
      '#min' => 2,
      '#max' => 10,
    ];

    // Anti-pattern detection.
    $form['antipatterns'] = [
      '#type' => 'details',
      '#title' => $this->t('Anti-Pattern Detection'),
      '#open' => FALSE,
      '#tree' => FALSE,
    ];

    $form['antipatterns']['detect_service_locators'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Detect service locators'),
      '#description' => $this->t('Flag usage of \\Drupal::service() and similar static calls. Use dependency injection instead.'),
      '#default_value' => $config['detect_service_locators'] ?? TRUE,
    ];

    $form['antipatterns']['allow_service_locators_in_procedural'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow service locators in procedural files'),
      '#description' => $this->t('When enabled, \\Drupal:: calls in .module, .install, and .theme files will not be flagged. These files often cannot use dependency injection.'),
      '#default_value' => $config['allow_service_locators_in_procedural'] ?? TRUE,
    ];

    $form['antipatterns']['detect_deep_arrays'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Detect deep array nesting'),
      '#description' => $this->t('Flag arrays with excessive nesting levels.'),
      '#default_value' => $config['detect_deep_arrays'] ?? TRUE,
    ];

    $form['antipatterns']['detect_hardcoded_ids'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Detect hardcoded entity IDs'),
      '#description' => $this->t('Flag hardcoded numeric IDs in entity load calls.'),
      '#default_value' => $config['detect_hardcoded_ids'] ?? TRUE,
    ];

    $form['antipatterns']['detect_direct_queries'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Detect direct database queries'),
      '#description' => $this->t('Flag usage of deprecated db_query() and similar functions.'),
      '#default_value' => $config['detect_direct_queries'] ?? TRUE,
    ];

    // Display settings.
    $form['display'] = [
      '#type' => 'details',
      '#title' => $this->t('Display Settings'),
      '#open' => FALSE,
      '#tree' => FALSE,
    ];

    $form['display']['max_hotspots_display'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum hotspots to display'),
      '#description' => $this->t('Limit the number of complexity hotspots shown. <strong>Recommended: 20</strong>'),
      '#default_value' => $config['max_hotspots_display'] ?? self::DEFAULT_MAX_HOTSPOTS,
      '#min' => 5,
      '#max' => 100,
    ];

    $form['display']['timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Execution timeout (seconds)'),
      '#description' => $this->t('Maximum time for phploc to complete. <strong>Recommended: 120</strong>'),
      '#default_value' => $config['timeout'] ?? self::DEFAULT_TIMEOUT,
      '#min' => 30,
      '#max' => 600,
    ];

    return $form;
  }

}
