<?php

declare(strict_types=1);

namespace Drupal\audit_database\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Analyzes database tables for size, primary keys, and potential issues.
 */
#[AuditAnalyzer(
  id: 'database',
  label: new TranslatableMarkup('Database Audit'),
  description: new TranslatableMarkup('Analyzes database size, table sizes, primary keys, and cache bin configuration.'),
  menu_title: new TranslatableMarkup('Database'),
  output_directory: 'database',
  weight: 3,
)]
class DatabaseAnalyzer extends AuditAnalyzerBase {

  protected const DEFAULT_TABLE_THRESHOLD = 100;
  protected const DEFAULT_DATABASE_THRESHOLD = 1000;

  /**
   * Score weights for different factors.
   *
   * Primary keys and cache backend have the highest weight as they are
   * critical for performance and data integrity.
   * Fragmentation affects disk usage and query performance.
   * Database and table sizes are important but more configurable.
   * Logging affects write performance.
   * Cache configuration is important when external cache is available.
   */
  protected const SCORE_WEIGHTS = [
    'database_size' => 12,
    'table_sizes' => 12,
    'primary_keys' => 18,
    'cache_backend' => 18,
    'cache_config' => 13,
    'logging' => 12,
    'fragmentation' => 15,
  ];

  /**
   * Cache bins that MUST remain in MySQL database.
   *
   * cache_form stores form state tokens and CSRF data. Moving it to Redis
   * causes "form expired" errors and breaks multi-step wizards.
   */
  protected const CACHE_BINS_DATABASE_ONLY = [
    'cache_form',
  ];

  /**
   * Cache bins that should use ChainedFastBackend (APCu + persistent).
   *
   * These bins have high-read, low-write patterns and are accessed on every
   * request. Direct Redis access adds unnecessary network latency.
   */
  protected const CACHE_BINS_CHAINEDFAST = [
    'cache_bootstrap',
    'cache_discovery',
    'cache_config',
  ];

  /**
   * Cache bins that benefit from Redis/Memcache.
   *
   * High volume bins where external cache provides clear performance benefits.
   */
  protected const CACHE_BINS_EXTERNAL_RECOMMENDED = [
    'cache_render',
    'cache_page',
    'cache_entity',
    'cache_dynamic_page_cache',
    'cache_menu',
    'cache_toolbar',
    'cache_data',
    'cache_default',
  ];

  protected Connection $database;
  protected ModuleHandlerInterface $moduleHandler;
  protected ConfigFactoryInterface $configFactory;
  protected LoggerInterface $logger;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->database = $container->get('database');
    $instance->moduleHandler = $container->get('module_handler');
    $instance->configFactory = $container->get('config.factory');
    $instance->logger = $container->get('logger.factory')->get('audit_database');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $config): array {
    return [
      'table_size_threshold' => [
        '#type' => 'number',
        '#title' => $this->t('Table size threshold (MB)'),
        '#description' => $this->t('Tables exceeding this size will be flagged as warnings. Recommended: 50-100 MB (small sites), 100-500 MB (medium), 500-2000 MB (large).'),
        '#default_value' => $config['table_size_threshold'] ?? self::DEFAULT_TABLE_THRESHOLD,
        '#min' => 10,
        '#max' => 10000,
        '#required' => TRUE,
      ],
      'database_size_threshold' => [
        '#type' => 'number',
        '#title' => $this->t('Total database size threshold (MB)'),
        '#description' => $this->t('Total database size exceeding this value will be flagged. Recommended: 500-1000 MB (small), 1000-5000 MB (medium), 5000-50000 MB (large).'),
        '#default_value' => $config['database_size_threshold'] ?? self::DEFAULT_DATABASE_THRESHOLD,
        '#min' => 100,
        '#max' => 100000,
        '#required' => TRUE,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    $config = $this->configFactory->get('audit_database.settings');
    $tableThreshold = $config->get('table_size_threshold') ?? self::DEFAULT_TABLE_THRESHOLD;
    $databaseThreshold = $config->get('database_size_threshold') ?? self::DEFAULT_DATABASE_THRESHOLD;

    // Gather all data once for efficiency.
    $cacheStatus = $this->getCacheStatus();
    $loggingStatus = $this->getLoggingStatus();
    $tables = $this->getTableInfo();
    $tableData = $this->buildTableData($tables, $tableThreshold, $cacheStatus);
    $cacheAnalysis = $this->analyzeCacheBins($tableData, $cacheStatus);
    $metrics = $this->calculateMetrics($tableData, $databaseThreshold, $cacheAnalysis, $loggingStatus);

    // Build separate files for each section.
    $files = $this->buildSectionFiles($tableData, $metrics, $tableThreshold, $databaseThreshold, $cacheStatus, $cacheAnalysis, $loggingStatus);

    // Calculate scores using collected data.
    $scores = $this->calculateScores($metrics, $databaseThreshold, $cacheStatus, $cacheAnalysis, $loggingStatus);

    return [
      '_files' => $files,
      'score' => $scores,
      // Keep config at root level for reference.
      'config' => [
        'table_size_threshold' => $tableThreshold,
        'database_size_threshold' => $databaseThreshold,
        'has_redis' => $cacheStatus['has_redis'],
        'has_memcache' => $cacheStatus['has_memcache'],
        'has_syslog' => $loggingStatus['has_syslog'],
      ],
    ];
  }

  /**
   * Builds separate files for each section.
   *
   * @param array $tableData
   *   Processed table data.
   * @param array $metrics
   *   Calculated metrics.
   * @param int $tableThreshold
   *   Table size threshold in MB.
   * @param int $databaseThreshold
   *   Database size threshold in MB.
   * @param array $cacheStatus
   *   Cache system status.
   * @param array $cacheAnalysis
   *   Cache bin analysis.
   * @param array $loggingStatus
   *   Logging system status.
   *
   * @return array
   *   Array of file data keyed by file name.
   */
  protected function buildSectionFiles(
    array $tableData,
    array $metrics,
    int $tableThreshold,
    int $databaseThreshold,
    array $cacheStatus,
    array $cacheAnalysis,
    array $loggingStatus,
  ): array {
    $files = [];

    // Database size section.
    $files['database_size'] = $this->buildDatabaseSizeFile($metrics, $databaseThreshold);

    // Table sizes section (only tables with issues).
    $files['table_sizes'] = $this->buildTableSizesFile($tableData, $tableThreshold);

    // Primary keys section (only tables without PK).
    $files['primary_keys'] = $this->buildPrimaryKeysFile($tableData);

    // Cache backend section.
    $files['cache_backend'] = $this->buildCacheBackendFile($cacheStatus, $cacheAnalysis, $metrics);

    // Cache configuration section.
    $files['cache_config'] = $this->buildCacheConfigFile($tableData, $cacheStatus, $cacheAnalysis);

    // Logging section.
    $files['logging'] = $this->buildLoggingFile($loggingStatus, $tableData);

    // Fragmentation section (only fragmented tables).
    $files['fragmentation'] = $this->buildFragmentationFile($tableData);

    // Table inventory section (informational - all tables).
    $files['table_inventory'] = $this->buildTableInventoryFile($tableData, $tableThreshold, $databaseThreshold);

    return $files;
  }

  /**
   * Builds the database size file.
   */
  protected function buildDatabaseSizeFile(array $metrics, int $databaseThreshold): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $percentage = $databaseThreshold > 0 ? ($metrics['total_mb'] / $databaseThreshold) * 100 : 0;

    if ($percentage >= 90) {
      $results[] = $this->createResultItem(
        'error',
        'DATABASE_SIZE_CRITICAL',
        (string) $this->t('Database size (@size MB) exceeds 90% of the configured threshold (@threshold MB).', [
          '@size' => $metrics['total_mb'],
          '@threshold' => $databaseThreshold,
        ]),
        [
          'size_mb' => $metrics['total_mb'],
          'threshold_mb' => $databaseThreshold,
          'percentage' => round($percentage, 1),
        ]
      );
      $errors++;
    }
    elseif ($percentage >= 50) {
      $results[] = $this->createResultItem(
        'warning',
        'DATABASE_SIZE_WARNING',
        (string) $this->t('Database size (@size MB) exceeds 50% of the configured threshold (@threshold MB).', [
          '@size' => $metrics['total_mb'],
          '@threshold' => $databaseThreshold,
        ]),
        [
          'size_mb' => $metrics['total_mb'],
          'threshold_mb' => $databaseThreshold,
          'percentage' => round($percentage, 1),
        ]
      );
      $warnings++;
    }

    $output = $this->createResult($results, $errors, $warnings, 0);
    $output['metrics'] = [
      'total_size_bytes' => $metrics['total_size'],
      'total_size_mb' => $metrics['total_mb'],
      'threshold_mb' => $databaseThreshold,
      'percentage' => round($percentage, 1),
      'total_tables' => $metrics['total_tables'],
    ];

    return $output;
  }

  /**
   * Builds the table sizes file.
   */
  protected function buildTableSizesFile(array $tableData, int $tableThreshold): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $oversizedTables = [];

    foreach ($tableData as $table) {
      if ($table['percentage_of_threshold'] >= 100) {
        $results[] = $this->createResultItem(
          'error',
          'TABLE_SIZE_CRITICAL',
          (string) $this->t('Table "@table" (@size) exceeds the size threshold.', [
            '@table' => $table['name'],
            '@size' => $table['size_display'],
          ]),
          [
            'table' => $table['name'],
            'size_mb' => $table['size_mb'],
            'rows' => $table['rows'],
            'percentage' => $table['percentage_of_threshold'],
          ]
        );
        $errors++;
        $oversizedTables[] = $table;
      }
      elseif ($table['percentage_of_threshold'] >= 80) {
        $results[] = $this->createResultItem(
          'warning',
          'TABLE_SIZE_WARNING',
          (string) $this->t('Table "@table" (@size) exceeds 80% of the size threshold.', [
            '@table' => $table['name'],
            '@size' => $table['size_display'],
          ]),
          [
            'table' => $table['name'],
            'size_mb' => $table['size_mb'],
            'rows' => $table['rows'],
            'percentage' => $table['percentage_of_threshold'],
          ]
        );
        $warnings++;
        $oversizedTables[] = $table;
      }
    }

    $output = $this->createResult($results, $errors, $warnings, 0);
    $output['tables'] = $oversizedTables;
    $output['threshold_mb'] = $tableThreshold;

    return $output;
  }

  /**
   * Builds the primary keys file.
   */
  protected function buildPrimaryKeysFile(array $tableData): array {
    $results = [];
    $tablesWithoutPk = [];

    foreach ($tableData as $table) {
      if (!$table['has_primary_key']) {
        $results[] = $this->createResultItem(
          'error',
          'MISSING_PRIMARY_KEY',
          (string) $this->t('Table "@table" does not have a primary key.', [
            '@table' => $table['name'],
          ]),
          [
            'table' => $table['name'],
            'rows' => $table['rows'],
            'size_display' => $table['size_display'],
            'fix' => (string) $this->t('Add a primary key to improve query performance and data integrity. Example: ALTER TABLE @table ADD PRIMARY KEY (id);', [
              '@table' => $table['name'],
            ]),
          ]
        );
        $tablesWithoutPk[] = $table;
      }
    }

    $output = $this->createResult($results, count($results), 0, 0);
    $output['tables'] = $tablesWithoutPk;

    return $output;
  }

  /**
   * Builds the cache backend file.
   */
  protected function buildCacheBackendFile(array $cacheStatus, array $cacheAnalysis, array $metrics): array {
    $results = [];
    $errors = 0;
    $warnings = 0;

    if (!$cacheStatus['has_external_cache']) {
      if ($cacheAnalysis['external_recommended_bins_in_db'] > 0) {
        $cacheSizeMb = round($metrics['total_cache_size'] / 1024 / 1024, 2);
        $results[] = $this->createResultItem(
          'warning',
          'NO_EXTERNAL_CACHE',
          (string) $this->t('No Redis/Memcache detected. @count high-volume cache bins (@size MB) are in MySQL, causing performance bottlenecks.', [
            '@count' => $cacheAnalysis['external_recommended_bins_in_db'],
            '@size' => $cacheSizeMb,
          ]),
          [
            'recommendation' => (string) $this->t('Install and configure redis or memcache module for better cache performance.'),
            'bins_count' => $cacheAnalysis['external_recommended_bins_in_db'],
            'cache_size_mb' => $cacheSizeMb,
            'fix' => '<p><strong>Problem:</strong> All cache data is stored in MySQL, which adds database load and increases query times.</p>'
              . '<p><strong>Solution:</strong> Install Redis or Memcache:</p>'
              . '<ol><li>Install Redis server: <code>apt install redis-server</code></li>'
              . '<li>Install Drupal module: <code>composer require drupal/redis</code></li>'
              . '<li>Enable module: <code>drush en redis</code></li>'
              . '<li>Configure in settings.php</li></ol>',
          ]
        );
        $warnings++;
      }
    }

    $output = $this->createResult($results, $errors, $warnings, 0);
    $output['status'] = [
      'has_redis' => $cacheStatus['has_redis'],
      'has_memcache' => $cacheStatus['has_memcache'],
      'has_external_cache' => $cacheStatus['has_external_cache'],
      'total_cache_tables' => $cacheAnalysis['total_cache_tables'],
      'total_cache_size_mb' => round($cacheAnalysis['total_cache_size'] / 1024 / 1024, 2),
    ];

    return $output;
  }

  /**
   * Builds the cache configuration file.
   */
  protected function buildCacheConfigFile(array $tableData, array $cacheStatus, array $cacheAnalysis): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;
    $cacheTables = [];

    // Check for misconfigured bins when external cache is available.
    if ($cacheStatus['has_external_cache'] && !empty($cacheAnalysis['bins_misconfigured'])) {
      $results[] = $this->createResultItem(
        'warning',
        'CACHE_BINS_MISCONFIGURED',
        (string) $this->t('Redis/Memcache is installed but @count cache bins still have data in MySQL.', [
          '@count' => count($cacheAnalysis['bins_misconfigured']),
        ]),
        [
          'bins' => $cacheAnalysis['bins_misconfigured'],
          'fix' => '<p><strong>Problem:</strong> Cache bins are not properly routed to the external cache backend.</p>'
            . '<p><strong>Solution:</strong> Configure cache bins in settings.php:</p>'
            . '<pre>$settings[\'cache\'][\'default\'] = \'cache.backend.redis\';</pre>',
        ]
      );
      $warnings++;
    }

    // Check ChainedFast recommendation.
    if ($cacheAnalysis['chainedfast_bins_in_db'] > 0 && $cacheStatus['has_external_cache']) {
      $results[] = $this->createResultItem(
        'notice',
        'CHAINEDFAST_RECOMMENDATION',
        (string) $this->t('cache_bootstrap, cache_discovery, and cache_config should use ChainedFastBackend (APCu + Redis) for optimal performance.'),
        [
          'reason' => (string) $this->t('These bins are accessed on every request. APCu provides microsecond access vs milliseconds for Redis.'),
          'fix' => '<p><strong>Recommendation:</strong> Use ChainedFastBackend for high-frequency bins:</p>'
            . '<pre>$settings[\'cache\'][\'bins\'][\'bootstrap\'] = \'cache.backend.chainedfast\';<br>'
            . '$settings[\'cache\'][\'bins\'][\'discovery\'] = \'cache.backend.chainedfast\';<br>'
            . '$settings[\'cache\'][\'bins\'][\'config\'] = \'cache.backend.chainedfast\';</pre>',
        ]
      );
      $notices++;
    }

    // Build cache tables list with their configuration status.
    foreach ($tableData as $table) {
      if ($table['is_cache_table']) {
        $binStatus = 'ok';
        if (in_array($table['name'], $cacheAnalysis['bins_misconfigured'], TRUE)) {
          $binStatus = 'misconfigured';
        }
        elseif (in_array($table['name'], $cacheAnalysis['bins_correctly_configured'], TRUE)) {
          $binStatus = 'correct';
        }

        $cacheTables[] = [
          'name' => $table['name'],
          'rows' => $table['rows'],
          'size_display' => $table['size_display'],
          'cache_bin_type' => $table['cache_bin_type'],
          'problem_reason' => $table['problem_reason'],
          'config_status' => $binStatus,
        ];
      }
    }

    $output = $this->createResult($results, $errors, $warnings, $notices);
    $output['cache_tables'] = $cacheTables;
    $output['analysis'] = $cacheAnalysis;

    return $output;
  }

  /**
   * Builds the logging file.
   */
  protected function buildLoggingFile(array $loggingStatus, array $tableData): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $watchdogInfo = NULL;

    // Find watchdog table info.
    foreach ($tableData as $table) {
      if ($table['name'] === 'watchdog') {
        $watchdogInfo = $table;
        break;
      }
    }

    if ($loggingStatus['logs_to_database']) {
      $message = (string) $this->t('Database logging (dblog) is enabled without syslog.');
      if ($watchdogInfo) {
        $message .= ' ' . (string) $this->t('Watchdog table has @rows rows (@size).', [
          '@rows' => number_format($watchdogInfo['rows']),
          '@size' => $watchdogInfo['size_display'],
        ]);
      }

      $results[] = $this->createResultItem(
        'warning',
        'DBLOG_WITHOUT_SYSLOG',
        $message,
        [
          'watchdog_rows' => $watchdogInfo['rows'] ?? 0,
          'watchdog_size' => $watchdogInfo['size_display'] ?? 'N/A',
          'fix' => '<p><strong>Problem:</strong> Every log entry causes a database write operation, impacting performance especially on high-traffic sites.</p>'
            . '<p><strong>Solution:</strong></p>'
            . '<ol><li>Enable syslog module: <code>drush en syslog</code></li>'
            . '<li>Configure syslog at /admin/config/development/logging</li>'
            . '<li>Disable dblog: <code>drush pm-uninstall dblog</code></li></ol>'
            . '<p><strong>Note:</strong> Use external log management tools (ELK, Graylog, etc.) for log viewing instead of the database-based UI.</p>',
        ]
      );
      $warnings++;
    }

    $output = $this->createResult($results, $errors, $warnings, 0);
    $output['status'] = [
      'has_syslog' => $loggingStatus['has_syslog'],
      'has_dblog' => $loggingStatus['has_dblog'],
      'logs_to_database' => $loggingStatus['logs_to_database'],
      'watchdog_rows' => $watchdogInfo['rows'] ?? 0,
      'watchdog_size' => $watchdogInfo['size_display'] ?? 'N/A',
    ];

    return $output;
  }

  /**
   * Builds the fragmentation file.
   */
  protected function buildFragmentationFile(array $tableData): array {
    $results = [];
    $fragmentedTables = [];

    foreach ($tableData as $table) {
      if (!empty($table['is_fragmented'])) {
        $results[] = $this->createResultItem(
          'warning',
          'TABLE_FRAGMENTED',
          (string) $this->t('Table "@table" is @percent% fragmented (@free free space).', [
            '@table' => $table['name'],
            '@percent' => $table['fragmentation'],
            '@free' => $this->formatSize($table['data_free']),
          ]),
          [
            'table' => $table['name'],
            'fragmentation' => $table['fragmentation'],
            'data_free' => $table['data_free'],
            'size_display' => $table['size_display'],
            'fix' => (string) $this->t('Run: OPTIMIZE TABLE @table;', ['@table' => $table['name']]),
          ]
        );
        $fragmentedTables[] = $table;
      }
    }

    $output = $this->createResult($results, 0, count($results), 0);
    $output['tables'] = $fragmentedTables;
    $output['total_free_space'] = array_sum(array_column($fragmentedTables, 'data_free'));

    return $output;
  }

  /**
   * Builds the table inventory file (informational).
   */
  protected function buildTableInventoryFile(array $tableData, int $tableThreshold, int $databaseThreshold): array {
    // Calculate totals.
    $totalSize = 0;
    $totalRows = 0;
    foreach ($tableData as $table) {
      $totalSize += $table['size_bytes'];
      $totalRows += $table['rows'];
    }

    $output = $this->createResult([], 0, 0, 0);
    $output['tables'] = $tableData;
    $output['totals'] = [
      'total_tables' => count($tableData),
      'total_size_bytes' => $totalSize,
      'total_size_mb' => round($totalSize / 1024 / 1024, 2),
      'total_size_display' => $this->formatSize($totalSize),
      'total_rows' => $totalRows,
    ];
    $output['thresholds'] = [
      'table_size_mb' => $tableThreshold,
      'database_size_mb' => $databaseThreshold,
    ];

    return $output;
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      'database_size' => [
        'label' => $this->t('Database Size'),
        'description' => $this->t('Total database size compared to the configured threshold.'),
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['database_size'],
        'file_key' => 'database_size',
        'score_factor_key' => 'database_size',
      ],
      'table_sizes' => [
        'label' => $this->t('Table Sizes'),
        'description' => $this->t('Individual tables that exceed the configured size threshold.'),
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['table_sizes'],
        'file_key' => 'table_sizes',
        'score_factor_key' => 'table_sizes',
      ],
      'primary_keys' => [
        'label' => $this->t('Primary Keys'),
        'description' => $this->t('Tables missing primary keys cause performance issues and data integrity problems.'),
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['primary_keys'],
        'file_key' => 'primary_keys',
        'score_factor_key' => 'primary_keys',
      ],
      'cache_backend' => [
        'label' => $this->t('Cache Backend'),
        'description' => $this->t('Availability of external cache systems (Redis/Memcache) for better performance.'),
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['cache_backend'],
        'file_key' => 'cache_backend',
        'score_factor_key' => 'cache_backend',
      ],
      'cache_config' => [
        'label' => $this->t('Cache Configuration'),
        'description' => $this->t('Cache bins correctly routed to appropriate backends per best practices.'),
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['cache_config'],
        'file_key' => 'cache_config',
        'score_factor_key' => 'cache_config',
      ],
      'logging' => [
        'label' => $this->t('Logging Configuration'),
        'description' => $this->t('Using syslog instead of database logging reduces write operations.'),
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['logging'],
        'file_key' => 'logging',
        'score_factor_key' => 'logging',
      ],
      'fragmentation' => [
        'label' => $this->t('Table Fragmentation'),
        'description' => $this->t('Tables with significant fragmentation that need optimization.'),
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['fragmentation'],
        'file_key' => 'fragmentation',
        'score_factor_key' => 'fragmentation',
      ],
      'table_inventory' => [
        'label' => $this->t('Table Inventory'),
        'description' => $this->t('Complete list of all database tables with their sizes and row counts. Informational only, does not affect the audit score.'),
        'affects_score' => FALSE,
        'weight' => 0,
        'file_key' => 'table_inventory',
        'score_factor_key' => NULL,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    $files = $data['_files'] ?? [];

    return match ($check_id) {
      'database_size' => $this->buildDatabaseSizeContent($files, $data),
      'table_sizes' => $this->buildTableSizesContent($files),
      'primary_keys' => $this->buildPrimaryKeysContent($files),
      'cache_backend' => $this->buildCacheBackendContent($files),
      'cache_config' => $this->buildCacheConfigContent($files),
      'logging' => $this->buildLoggingContent($files),
      'fragmentation' => $this->buildFragmentationContent($files),
      'table_inventory' => $this->buildTableInventoryContent($files),
      default => [],
    };
  }

  /**
   * Builds the database size section content.
   */
  protected function buildDatabaseSizeContent(array $files, array $data): array {
    $sizeData = $files['database_size'] ?? [];
    $metrics = $sizeData['metrics'] ?? [];
    $config = $data['config'] ?? [];

    $totalMb = $metrics['total_size_mb'] ?? 0;
    $threshold = $metrics['threshold_mb'] ?? $config['database_size_threshold'] ?? 1000;
    $percentage = $metrics['percentage'] ?? 0;

    $fix_html = '<p><strong>Problem:</strong> The database size is approaching or exceeding the configured threshold. Large databases can cause performance issues, slower backups, and increased hosting costs.</p>'
      . '<p><strong>Solutions:</strong></p>'
      . '<ul>'
      . '<li>Externalize cache tables to Redis or Memcache instead of MySQL.</li>'
      . '<li>Clean up old content revisions using node_revision_delete module.</li>'
      . '<li>Remove unpublished content, orphaned paragraphs, and unused media.</li>'
      . '<li>Enable syslog and reduce watchdog table retention.</li>'
      . '<li>Verify cron is processing queue tables correctly.</li>'
      . '<li>Consider increasing the threshold if this database size is expected for your project.</li>'
      . '</ul>';

    return $this->ui->buildIssueListFromResults(
      $sizeData['results'] ?? [],
      (string) $this->t('Database size (@size MB) is healthy and within the configured threshold of @threshold MB (@percent%).', [
        '@size' => $totalMb,
        '@threshold' => $threshold,
        '@percent' => round($percentage, 1),
      ]),
      function (array $item, $ui) use ($fix_html): array {
        return [
          'severity' => $ui->normalizeSeverity($item['severity'] ?? 'warning'),
          'code' => $item['code'] ?? 'DATABASE_SIZE',
          'label' => $item['message'] ?? '',
          'description' => ['#markup' => $fix_html],
          'tags' => ['performance', 'storage'],
        ];
      }
    );
  }

  /**
   * Builds the table sizes section content.
   */
  protected function buildTableSizesContent(array $files): array {
    $sizeData = $files['table_sizes'] ?? [];
    $threshold = $sizeData['threshold_mb'] ?? 100;

    $fix_html = '<p><strong>Problem:</strong> Large tables can slow down queries, backups, and overall site performance.</p>'
      . '<p><strong>Possible causes and solutions:</strong></p>'
      . '<ul>'
      . '<li><strong>Cache tables:</strong> Externalize cache to Redis or Memcache instead of MySQL.</li>'
      . '<li><strong>Content tables:</strong> Clean up old revisions (use node_revision_delete module), remove unpublished content that is no longer needed, and check for orphaned paragraphs or media.</li>'
      . '<li><strong>Log tables (watchdog):</strong> Enable syslog module and optionally uninstall dblog, or reduce log retention settings.</li>'
      . '<li><strong>Queue tables:</strong> Verify cron is running correctly and processing all queues. If queues are accumulating, there may be a cron issue or the site has more queue items than cron can process.</li>'
      . '<li><strong>Key/value tables:</strong> May indicate contrib or custom modules storing excessive data. Investigate which modules and keys are using the most space and consider patching or replacing problematic modules.</li>'
      . '</ul>';

    return $this->ui->buildIssueListFromResults(
      $sizeData['results'] ?? [],
      (string) $this->t('All tables are within the configured size threshold of @threshold MB.', [
        '@threshold' => $threshold,
      ]),
      function (array $item, $ui) use ($fix_html): array {
        $details = $item['details'] ?? [];
        $tableName = $details['table'] ?? '';
        $sizeMb = $details['size_mb'] ?? 0;
        $percentage = $details['percentage'] ?? 0;
        $rows = $details['rows'] ?? 0;

        $description = '<p><strong>Details:</strong> ' . number_format($rows) . ' rows, ' . $sizeMb . ' MB (' . $percentage . '% of threshold)</p>'
          . $fix_html;

        return [
          'severity' => $ui->normalizeSeverity($item['severity'] ?? 'warning'),
          'code' => $item['code'] ?? 'TABLE_SIZE',
          'file' => $tableName,
          'label' => $item['message'] ?? '',
          'description' => ['#markup' => $description],
          'tags' => ['performance', 'storage'],
        ];
      }
    );
  }

  /**
   * Builds the primary keys section content.
   */
  protected function buildPrimaryKeysContent(array $files): array {
    $pkData = $files['primary_keys'] ?? [];

    $fix_html = '<p><strong>Problem:</strong> Tables without primary keys have poor query performance and can cause data integrity issues.</p>'
      . '<p><strong>Solution:</strong> Add a primary key to the table. Example:</p>'
      . '<pre>ALTER TABLE table_name ADD PRIMARY KEY (id);</pre>'
      . '<p><strong>Note:</strong> If this is a contrib or core table, report the issue to the module maintainer.</p>';

    return $this->ui->buildIssueListFromResults(
      $pkData['results'] ?? [],
      (string) $this->t('All tables have primary keys. This is optimal for query performance and data integrity.'),
      function (array $item, $ui) use ($fix_html): array {
        $details = $item['details'] ?? [];
        $tableName = $details['table'] ?? '';
        $rows = $details['rows'] ?? 0;
        $sizeDisplay = $details['size_display'] ?? '';

        $description = '<p><strong>Details:</strong> ' . number_format($rows) . ' rows, ' . $sizeDisplay . '</p>'
          . $fix_html;

        return [
          'severity' => 'error',
          'code' => $item['code'] ?? 'MISSING_PRIMARY_KEY',
          'file' => $tableName,
          'label' => $item['message'] ?? '',
          'description' => ['#markup' => $description],
          'tags' => ['performance', 'integrity'],
        ];
      }
    );
  }

  /**
   * Builds the cache backend section content.
   */
  protected function buildCacheBackendContent(array $files): array {
    $cacheData = $files['cache_backend'] ?? [];
    $status = $cacheData['status'] ?? [];

    $hasExternalCache = $status['has_external_cache'] ?? FALSE;
    $totalCacheTables = $status['total_cache_tables'] ?? 0;
    $totalCacheSizeMb = $status['total_cache_size_mb'] ?? 0;

    $successMessage = $hasExternalCache
      ? (string) $this->t('External cache backend (Redis or Memcache) is installed. @tables cache tables (@size MB) remain in database.', [
          '@tables' => $totalCacheTables,
          '@size' => $totalCacheSizeMb,
        ])
      : (string) $this->t('No external cache issues detected.');

    return $this->ui->buildIssueListFromResults(
      $cacheData['results'] ?? [],
      $successMessage,
      fn(array $item, $ui): array => [
        'severity' => $ui->normalizeSeverity($item['severity'] ?? 'warning'),
        'code' => $item['code'] ?? 'CACHE_BACKEND',
        'label' => $item['message'] ?? '',
        'description' => isset($item['details']['fix']) ? ['#markup' => $item['details']['fix']] : NULL,
        'tags' => ['performance', 'cache'],
      ]
    );
  }

  /**
   * Builds the cache configuration section content.
   */
  protected function buildCacheConfigContent(array $files): array {
    $configData = $files['cache_config'] ?? [];
    $cacheTables = $configData['cache_tables'] ?? [];
    $analysis = $configData['analysis'] ?? [];

    $totalCacheTables = count($cacheTables);
    $correctlyConfigured = count($analysis['bins_correctly_configured'] ?? []);

    $successMessage = $totalCacheTables === 0
      ? (string) $this->t('No cache tables found in database. Cache may be fully externalized.')
      : (string) $this->t('All @count cache bins in database are correctly configured.', [
          '@count' => $totalCacheTables,
        ]);

    return $this->ui->buildIssueListFromResults(
      $configData['results'] ?? [],
      $successMessage,
      fn(array $item, $ui): array => [
        'severity' => $ui->normalizeSeverity($item['severity'] ?? 'warning'),
        'code' => $item['code'] ?? 'CACHE_CONFIG',
        'label' => $item['message'] ?? '',
        'description' => isset($item['details']['fix']) ? ['#markup' => $item['details']['fix']] : NULL,
        'tags' => ['performance', 'cache', 'configuration'],
      ]
    );
  }

  /**
   * Builds the logging section content.
   */
  protected function buildLoggingContent(array $files): array {
    $loggingData = $files['logging'] ?? [];
    $status = $loggingData['status'] ?? [];

    $hasSyslog = $status['has_syslog'] ?? FALSE;
    $hasDblog = $status['has_dblog'] ?? FALSE;

    if ($hasSyslog) {
      $successMessage = (string) $this->t('Syslog is enabled. Log entries are written to system logs instead of the database.');
    }
    elseif (!$hasDblog) {
      $successMessage = (string) $this->t('No database logging module is enabled.');
    }
    else {
      $successMessage = (string) $this->t('Logging configuration is optimal.');
    }

    return $this->ui->buildIssueListFromResults(
      $loggingData['results'] ?? [],
      $successMessage,
      fn(array $item, $ui): array => [
        'severity' => $ui->normalizeSeverity($item['severity'] ?? 'warning'),
        'code' => $item['code'] ?? 'LOGGING',
        'label' => $item['message'] ?? '',
        'description' => isset($item['details']['fix']) ? ['#markup' => $item['details']['fix']] : NULL,
        'tags' => ['performance', 'logging'],
      ]
    );
  }

  /**
   * Builds the fragmentation section content.
   */
  protected function buildFragmentationContent(array $files): array {
    $fragData = $files['fragmentation'] ?? [];
    $totalFreeSpace = $fragData['total_free_space'] ?? 0;

    $fix_html = '<p><strong>Problem:</strong> Fragmented tables have wasted space and can slow down queries. This happens when rows are deleted or updated frequently.</p>'
      . '<p><strong>Solution:</strong> Run the following command to optimize the table:</p>'
      . '<pre>OPTIMIZE TABLE table_name;</pre>'
      . '<p><strong>Note:</strong> For large tables, this operation may take time and lock the table. Consider running during low-traffic periods.</p>';

    return $this->ui->buildIssueListFromResults(
      $fragData['results'] ?? [],
      (string) $this->t('No tables with significant fragmentation detected. Tables are well optimized.'),
      function (array $item, $ui) use ($fix_html): array {
        $details = $item['details'] ?? [];
        $tableName = $details['table'] ?? '';
        $fragmentation = $details['fragmentation'] ?? 0;
        $dataFree = $details['data_free'] ?? 0;
        $sizeDisplay = $details['size_display'] ?? '';

        $description = '<p><strong>Details:</strong> ' . $sizeDisplay . ', ' . $fragmentation . '% fragmented, ' . number_format($dataFree / 1024 / 1024, 2) . ' MB reclaimable</p>'
          . $fix_html;

        return [
          'severity' => $ui->normalizeSeverity($item['severity'] ?? 'warning'),
          'code' => $item['code'] ?? 'TABLE_FRAGMENTED',
          'file' => $tableName,
          'label' => $item['message'] ?? '',
          'description' => ['#markup' => $description],
          'tags' => ['performance', 'storage', 'optimization'],
        ];
      }
    );
  }

  /**
   * Builds the table inventory section content (informational).
   */
  protected function buildTableInventoryContent(array $files): array {
    $inventoryData = $files['table_inventory'] ?? [];
    $tables = $inventoryData['tables'] ?? [];
    $totals = $inventoryData['totals'] ?? [];
    $thresholds = $inventoryData['thresholds'] ?? [];

    if (empty($tables)) {
      return ['#markup' => '<p>' . $this->t('No tables found in database.') . '</p>'];
    }

    $summaryText = $this->t('Database contains @count tables totaling @size (@rows rows). Threshold: @threshold MB per table, @db_threshold MB total.', [
      '@count' => $totals['total_tables'] ?? 0,
      '@size' => $totals['total_size_display'] ?? 'N/A',
      '@rows' => number_format($totals['total_rows'] ?? 0),
      '@threshold' => $thresholds['table_size_mb'] ?? 100,
      '@db_threshold' => $thresholds['database_size_mb'] ?? 1000,
    ]);

    $headers = [
      $this->ui->header((string) $this->t('Table Name')),
      $this->ui->header((string) $this->t('Rows'), 'right'),
      $this->ui->header((string) $this->t('PK'), 'center'),
      $this->ui->header((string) $this->t('Size'), 'right'),
      $this->ui->header((string) $this->t('% of Limit'), 'right'),
      $this->ui->header((string) $this->t('Frag.'), 'right'),
    ];

    $rows = [];
    foreach ($tables as $table) {
      $nameContent = $table['name'];
      if (!empty($table['is_problematic']) && !empty($table['problem_reason'])) {
        $nameContent = $this->ui->itemName($table['name'], $table['problem_reason']);
      }

      $percentage = $table['percentage_of_threshold'] ?? 0;
      $percentStatus = 'success';
      if ($percentage >= 100) {
        $percentStatus = 'error';
      }
      elseif ($percentage >= 80) {
        $percentStatus = 'warning';
      }

      $pkIcon = $table['has_primary_key'] ? $this->ui->icon('check') : $this->ui->icon('cross');

      $fragmentation = $table['fragmentation'] ?? 0;
      $fragStatus = 'success';
      if (!empty($table['is_fragmented'])) {
        $fragStatus = 'warning';
      }
      $fragDisplay = $fragmentation > 0 ? $fragmentation . '%' : '-';

      $rowSeverity = NULL;
      if (($table['status'] ?? '') === 'error') {
        $rowSeverity = 'error';
      }
      elseif (($table['status'] ?? '') === 'warning') {
        $rowSeverity = 'warning';
      }

      $rows[] = $this->ui->row([
        $this->ui->cell($nameContent),
        $this->ui->cell(number_format($table['rows']), ['align' => 'right']),
        $this->ui->cell($pkIcon, ['align' => 'center']),
        $this->ui->cell($table['size_display'], ['align' => 'right']),
        $this->ui->cell($percentage . '%', ['align' => 'right', 'status' => $percentStatus]),
        $this->ui->cell($fragDisplay, ['align' => 'right', 'status' => $fragStatus]),
      ], $rowSeverity);
    }

    return [
      '#type' => 'container',
      'info' => [
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#attributes' => ['class' => ['audit-info-text']],
        '#value' => '<em>' . $this->t('This section is informational only and does not affect the audit score.') . '</em>',
      ],
      'summary' => [
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#value' => $summaryText,
      ],
      'table' => $this->ui->table($headers, $rows, [
        'empty' => (string) $this->t('No tables found.'),
      ]),
    ];
  }

  // =========================================================================
  // Data gathering methods (unchanged - single MySQL query).
  // =========================================================================

  /**
   * Gets cache system status.
   */
  protected function getCacheStatus(): array {
    $hasRedis = $this->moduleHandler->moduleExists('redis');
    $hasMemcache = $this->moduleHandler->moduleExists('memcache');

    return [
      'has_redis' => $hasRedis,
      'has_memcache' => $hasMemcache,
      'has_external_cache' => $hasRedis || $hasMemcache,
    ];
  }

  /**
   * Gets logging system status.
   */
  protected function getLoggingStatus(): array {
    $hasSyslog = $this->moduleHandler->moduleExists('syslog');
    $hasDblog = $this->moduleHandler->moduleExists('dblog');

    return [
      'has_syslog' => $hasSyslog,
      'has_dblog' => $hasDblog,
      'logs_to_database' => $hasDblog && !$hasSyslog,
    ];
  }

  /**
   * Gets table information from database (single query).
   */
  protected function getTableInfo(): array {
    $tables = [];

    try {
      $dbName = $this->database->getConnectionOptions()['database'] ?? '';

      $query = "SELECT
        t.table_name AS `name`,
        COALESCE(t.data_length, 0) + COALESCE(t.index_length, 0) AS `size`,
        COALESCE(t.table_rows, 0) AS `row_count`,
        COALESCE(t.data_free, 0) AS `data_free`,
        CASE WHEN pk.constraint_name IS NOT NULL THEN 1 ELSE 0 END AS `has_primary_key`
        FROM information_schema.tables t
        LEFT JOIN information_schema.table_constraints pk
          ON t.table_schema = pk.table_schema
          AND t.table_name = pk.table_name
          AND pk.constraint_type = 'PRIMARY KEY'
        WHERE t.table_schema = :database
          AND t.table_type = 'BASE TABLE'
        ORDER BY `size` DESC";

      $result = $this->database->query($query, [':database' => $dbName]);

      foreach ($result as $row) {
        $size = (int) ($row->size ?? 0);
        $data_free = (int) ($row->data_free ?? 0);

        // Calculate fragmentation percentage.
        $total_size = $size + $data_free;
        $fragmentation = $total_size > 0 ? ($data_free / $total_size) * 100 : 0;

        $tables[] = [
          'name' => $row->name,
          'size' => $size,
          'rows' => (int) ($row->row_count ?? 0),
          'has_primary_key' => (bool) $row->has_primary_key,
          'data_free' => $data_free,
          'fragmentation' => round($fragmentation, 1),
        ];
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to query table information: @message', [
        '@message' => $e->getMessage(),
      ]);
    }

    return $tables;
  }

  /**
   * Builds enriched table data from raw table info.
   */
  protected function buildTableData(array $tables, int $tableThreshold, array $cacheStatus): array {
    $tableData = [];

    foreach ($tables as $table) {
      $sizeMb = $table['size'] / 1024 / 1024;
      $isCacheTable = str_starts_with($table['name'], 'cache_');
      $percentageOfThreshold = $tableThreshold > 0 ? ($sizeMb / $tableThreshold) * 100 : 0;
      $fragmentation = $table['fragmentation'] ?? 0;

      $problemInfo = $this->getTableProblemInfo($table['name'], $isCacheTable, $cacheStatus);

      // Check for fragmentation issues (>5% is considered fragmented).
      $is_fragmented = $fragmentation > 5 && $table['size'] > 1048576;

      $status = $this->determineTableStatus(
        $table['has_primary_key'],
        $percentageOfThreshold,
        $problemInfo['severity'] ?? 'ok',
        $is_fragmented
      );

      // Update problem info if fragmented.
      if ($is_fragmented && !$problemInfo['is_problematic']) {
        $problemInfo['is_problematic'] = TRUE;
        $problemInfo['reason'] = (string) $this->t('Table is @percent% fragmented - run OPTIMIZE TABLE', [
          '@percent' => $fragmentation,
        ]);
        $problemInfo['severity'] = 'warning';
      }

      $tableData[] = [
        'name' => $table['name'],
        'rows' => $table['rows'],
        'has_primary_key' => $table['has_primary_key'],
        'size_bytes' => $table['size'],
        'size_mb' => round($sizeMb, 2),
        'size_display' => $this->formatSize($table['size']),
        'percentage_of_threshold' => round($percentageOfThreshold, 1),
        'status' => $status,
        'is_problematic' => $problemInfo['is_problematic'],
        'is_cache_table' => $isCacheTable,
        'cache_bin_type' => $problemInfo['cache_bin_type'] ?? NULL,
        'problem_reason' => $problemInfo['reason'],
        'data_free' => $table['data_free'] ?? 0,
        'fragmentation' => $fragmentation,
        'is_fragmented' => $is_fragmented,
      ];
    }

    return $tableData;
  }

  /**
   * Gets problem information for a table.
   */
  protected function getTableProblemInfo(string $tableName, bool $isCacheTable, array $cacheStatus): array {
    if ($isCacheTable) {
      return $this->getCacheBinProblemInfo($tableName, $cacheStatus);
    }

    if ($tableName === 'watchdog') {
      return $this->getWatchdogProblemInfo();
    }

    return ['is_problematic' => FALSE, 'reason' => '', 'severity' => 'ok'];
  }

  /**
   * Gets problem information for the watchdog table.
   */
  protected function getWatchdogProblemInfo(): array {
    $hasSyslog = $this->moduleHandler->moduleExists('syslog');

    if ($hasSyslog) {
      return [
        'is_problematic' => FALSE,
        'reason' => (string) $this->t('Syslog module is enabled'),
        'severity' => 'ok',
      ];
    }

    return [
      'is_problematic' => TRUE,
      'reason' => (string) $this->t('Constant write operations - consider enabling syslog module'),
      'severity' => 'warning',
    ];
  }

  /**
   * Gets problem information for a cache bin.
   */
  protected function getCacheBinProblemInfo(string $tableName, array $cacheStatus): array {
    $hasExternalCache = $cacheStatus['has_external_cache'];

    if (in_array($tableName, self::CACHE_BINS_DATABASE_ONLY, TRUE)) {
      return [
        'is_problematic' => FALSE,
        'reason' => (string) $this->t('Correctly in database (form state, CSRF tokens)'),
        'cache_bin_type' => 'database_only',
        'severity' => 'ok',
      ];
    }

    if (in_array($tableName, self::CACHE_BINS_CHAINEDFAST, TRUE)) {
      if ($hasExternalCache) {
        return [
          'is_problematic' => TRUE,
          'reason' => (string) $this->t('Should use ChainedFastBackend (APCu + Redis), not direct Redis'),
          'cache_bin_type' => 'chainedfast',
          'severity' => 'warning',
        ];
      }
      return [
        'is_problematic' => TRUE,
        'reason' => (string) $this->t('High-read bin - consider ChainedFastBackend with APCu'),
        'cache_bin_type' => 'chainedfast',
        'severity' => 'info',
      ];
    }

    if (in_array($tableName, self::CACHE_BINS_EXTERNAL_RECOMMENDED, TRUE)) {
      if (!$hasExternalCache) {
        return [
          'is_problematic' => TRUE,
          'reason' => (string) $this->t('High-volume bin - Redis/Memcache recommended'),
          'cache_bin_type' => 'external_recommended',
          'severity' => 'warning',
        ];
      }
      return [
        'is_problematic' => TRUE,
        'reason' => (string) $this->t('Should be in Redis/Memcache, not database'),
        'cache_bin_type' => 'external_recommended',
        'severity' => 'warning',
      ];
    }

    if (!$hasExternalCache) {
      return [
        'is_problematic' => TRUE,
        'reason' => (string) $this->t('Cache in database - consider Redis/Memcache'),
        'cache_bin_type' => 'other',
        'severity' => 'info',
      ];
    }

    return [
      'is_problematic' => TRUE,
      'reason' => (string) $this->t('Cache table still in database with Redis enabled'),
      'cache_bin_type' => 'other',
      'severity' => 'info',
    ];
  }

  /**
   * Analyzes cache bins from table data.
   */
  protected function analyzeCacheBins(array $tableData, array $cacheStatus): array {
    $analysis = [
      'total_cache_tables' => 0,
      'total_cache_size' => 0,
      'bins_in_database' => [],
      'bins_correctly_configured' => [],
      'bins_misconfigured' => [],
      'chainedfast_bins_in_db' => 0,
      'external_recommended_bins_in_db' => 0,
    ];

    foreach ($tableData as $table) {
      if (!$table['is_cache_table']) {
        continue;
      }

      $analysis['total_cache_tables']++;
      $analysis['total_cache_size'] += $table['size_bytes'];
      $analysis['bins_in_database'][] = $table['name'];

      $binType = $table['cache_bin_type'] ?? 'other';

      if ($binType === 'database_only') {
        $analysis['bins_correctly_configured'][] = $table['name'];
      }
      elseif ($binType === 'chainedfast' && $table['rows'] > 0) {
        $analysis['chainedfast_bins_in_db']++;
        if ($cacheStatus['has_external_cache']) {
          $analysis['bins_misconfigured'][] = $table['name'];
        }
      }
      elseif ($binType === 'external_recommended' && $table['rows'] > 0) {
        $analysis['external_recommended_bins_in_db']++;
        $analysis['bins_misconfigured'][] = $table['name'];
      }
    }

    return $analysis;
  }

  /**
   * Determines table status based on various factors.
   */
  protected function determineTableStatus(
    bool $hasPrimaryKey,
    float $percentageOfThreshold,
    string $problemSeverity,
    bool $isFragmented = FALSE,
  ): string {
    if (!$hasPrimaryKey) {
      return 'error';
    }

    if ($percentageOfThreshold >= 100) {
      return 'error';
    }

    if ($percentageOfThreshold >= 80 || $problemSeverity === 'warning' || $isFragmented) {
      return 'warning';
    }

    return 'ok';
  }

  /**
   * Formats bytes to human-readable size.
   */
  protected function formatSize(int $bytes): string {
    $mb = $bytes / 1024 / 1024;
    if ($mb >= 1) {
      return round($mb, 2) . ' MB';
    }

    $kb = $bytes / 1024;
    if ($kb >= 1) {
      return round($kb, 2) . ' KB';
    }

    return $bytes . ' B';
  }

  /**
   * Calculates metrics from table data.
   */
  protected function calculateMetrics(array $tableData, int $databaseThreshold, array $cacheAnalysis, array $loggingStatus): array {
    $metrics = [
      'total_size' => 0,
      'total_mb' => 0,
      'tables_without_pk' => 0,
      'cache_tables_with_data' => $cacheAnalysis['total_cache_tables'],
      'total_cache_size' => $cacheAnalysis['total_cache_size'],
      'oversized_tables' => 0,
      'tables_over_50' => 0,
      'tables_over_90' => 0,
      'fragmented_tables' => 0,
      'total_tables' => count($tableData),
      'errors' => 0,
      'warnings' => 0,
      'misconfigured_cache_bins' => count($cacheAnalysis['bins_misconfigured']),
    ];

    foreach ($tableData as $table) {
      $metrics['total_size'] += $table['size_bytes'];

      if (!$table['has_primary_key']) {
        $metrics['tables_without_pk']++;
        $metrics['errors']++;
      }

      if ($table['percentage_of_threshold'] >= 100) {
        $metrics['tables_over_90']++;
        if ($table['has_primary_key']) {
          $metrics['oversized_tables']++;
          $metrics['errors']++;
        }
      }
      elseif ($table['percentage_of_threshold'] >= 80) {
        $metrics['tables_over_50']++;
        $metrics['warnings']++;
      }

      if (!empty($table['is_fragmented'])) {
        $metrics['fragmented_tables']++;
        $metrics['warnings']++;
      }
    }

    $metrics['warnings'] += $metrics['misconfigured_cache_bins'];

    if ($loggingStatus['logs_to_database']) {
      $metrics['warnings']++;
    }

    $metrics['total_mb'] = round($metrics['total_size'] / 1024 / 1024, 2);
    $dbPercentage = $databaseThreshold > 0 ? ($metrics['total_mb'] / $databaseThreshold) * 100 : 0;

    if ($dbPercentage >= 90) {
      $metrics['errors']++;
    }
    elseif ($dbPercentage >= 50) {
      $metrics['warnings']++;
    }

    return $metrics;
  }

  // =========================================================================
  // Score calculation methods (unchanged logic).
  // =========================================================================

  /**
   * Calculates scores for all factors.
   */
  protected function calculateScores(array $metrics, int $databaseThreshold, array $cacheStatus, array $cacheAnalysis, array $loggingStatus): array {
    $fragmentedTables = $metrics['fragmented_tables'] ?? 0;

    $scores = [
      'database_size' => $this->calculateDatabaseSizeScore($metrics['total_mb'], $databaseThreshold),
      'table_sizes' => $this->calculateTableSizesScore($metrics['tables_over_90'], $metrics['tables_over_50']),
      'primary_keys' => $this->calculatePrimaryKeysScore($metrics['tables_without_pk']),
      'cache_backend' => $this->calculateCacheBackendScore($cacheStatus, $cacheAnalysis),
      'cache_config' => $this->calculateCacheConfigScore($cacheStatus, $cacheAnalysis),
      'logging' => $this->calculateLoggingScore($loggingStatus),
      'fragmentation' => $this->calculateFragmentationScore($fragmentedTables),
    ];

    $fragDescription = $fragmentedTables === 0
      ? (string) $this->t('No fragmented tables detected')
      : (string) $this->t('@count table(s) need OPTIMIZE TABLE', ['@count' => $fragmentedTables]);

    return [
      'factors' => [
        'database_size' => [
          'score' => $scores['database_size'],
          'weight' => self::SCORE_WEIGHTS['database_size'],
          'label' => (string) $this->t('Database Size'),
          'description' => (string) $this->t('Total database size vs configured threshold'),
        ],
        'table_sizes' => [
          'score' => $scores['table_sizes'],
          'weight' => self::SCORE_WEIGHTS['table_sizes'],
          'label' => (string) $this->t('Table Sizes'),
          'description' => (string) $this->t('Individual table sizes vs configured threshold'),
        ],
        'primary_keys' => [
          'score' => $scores['primary_keys'],
          'weight' => self::SCORE_WEIGHTS['primary_keys'],
          'label' => (string) $this->t('Primary Keys'),
          'description' => (string) $this->t('All tables should have primary keys'),
        ],
        'cache_backend' => [
          'score' => $scores['cache_backend'],
          'weight' => self::SCORE_WEIGHTS['cache_backend'],
          'label' => (string) $this->t('Cache Backend'),
          'description' => (string) $this->t('External cache (Redis/Memcache) availability'),
        ],
        'cache_config' => [
          'score' => $scores['cache_config'],
          'weight' => self::SCORE_WEIGHTS['cache_config'],
          'label' => (string) $this->t('Cache Configuration'),
          'description' => (string) $this->t('Cache bins correctly configured per best practices'),
        ],
        'logging' => [
          'score' => $scores['logging'],
          'weight' => self::SCORE_WEIGHTS['logging'],
          'label' => (string) $this->t('Logging'),
          'description' => (string) $this->t('Using syslog instead of database logging'),
        ],
        'fragmentation' => [
          'score' => $scores['fragmentation'],
          'weight' => self::SCORE_WEIGHTS['fragmentation'],
          'label' => (string) $this->t('Table Fragmentation'),
          'description' => $fragDescription,
        ],
      ],
    ];
  }

  /**
   * Calculates database size score.
   */
  protected function calculateDatabaseSizeScore(float $totalMb, int $threshold): int {
    $percentage = $threshold > 0 ? ($totalMb / $threshold) * 100 : 0;

    return match (TRUE) {
      $percentage >= 90 => 0,
      $percentage >= 80 => 25,
      $percentage >= 50 => 50,
      $percentage >= 25 => 75,
      default => 100,
    };
  }

  /**
   * Calculates table sizes score.
   */
  protected function calculateTableSizesScore(int $tablesOver90, int $tablesOver50): int {
    if ($tablesOver90 > 0) {
      return max(0, 100 - ($tablesOver90 * 25) - ($tablesOver50 * 10));
    }

    if ($tablesOver50 > 0) {
      return max(0, 100 - ($tablesOver50 * 15));
    }

    return 100;
  }

  /**
   * Calculates primary keys score.
   */
  protected function calculatePrimaryKeysScore(int $tablesWithoutPk): int {
    if ($tablesWithoutPk === 0) {
      return 100;
    }

    return max(0, 100 - ($tablesWithoutPk * 30));
  }

  /**
   * Calculates cache backend score.
   */
  protected function calculateCacheBackendScore(array $cacheStatus, array $cacheAnalysis): int {
    if ($cacheStatus['has_external_cache']) {
      return 100;
    }

    if ($cacheAnalysis['total_cache_tables'] === 0) {
      return 75;
    }

    $externalRecommended = $cacheAnalysis['external_recommended_bins_in_db'];
    return max(0, 75 - ($externalRecommended * 10));
  }

  /**
   * Calculates cache configuration score.
   */
  protected function calculateCacheConfigScore(array $cacheStatus, array $cacheAnalysis): int {
    if (!$cacheStatus['has_external_cache']) {
      return $cacheAnalysis['total_cache_tables'] === 0 ? 75 : 50;
    }

    $misconfigured = count($cacheAnalysis['bins_misconfigured']);
    if ($misconfigured === 0) {
      return 100;
    }

    return max(0, 100 - ($misconfigured * 15));
  }

  /**
   * Calculates logging score.
   */
  protected function calculateLoggingScore(array $loggingStatus): int {
    if ($loggingStatus['has_syslog']) {
      return 100;
    }

    if (!$loggingStatus['has_dblog']) {
      return 100;
    }

    return 50;
  }

  /**
   * Calculates fragmentation score.
   */
  protected function calculateFragmentationScore(int $fragmentedTables): int {
    if ($fragmentedTables === 0) {
      return 100;
    }

    return max(0, 100 - ($fragmentedTables * 15));
  }

}
