<?php

declare(strict_types=1);

namespace Drupal\audit_fields\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\Display\EntityFormDisplayInterface;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\field\FieldConfigInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Analyzes entity fields and view modes.
 */
#[AuditAnalyzer(
  id: 'fields',
  label: new TranslatableMarkup('Fields Audit'),
  description: new TranslatableMarkup('Analyzes entity types, bundles, fields, and display configurations.'),
  menu_title: new TranslatableMarkup('Fields'),
  output_directory: 'fields',
  weight: 2,
)]
class FieldsAnalyzer extends AuditAnalyzerBase {

  /**
   * Score weights for different factors.
   */
  protected const SCORE_WEIGHTS = [
    'unused_fields' => 40,
    'field_visibility' => 20,
    'field_count' => 20,
    'view_modes' => 10,
    'form_modes' => 10,
  ];

  /**
   * Entity types to exclude from analysis.
   */
  protected const EXCLUDED_ENTITY_TYPES = [
    'content_moderation_state',
    'crop',
    'file',
    'menu_link_content',
    'paragraphs_library_item',
    'path_alias',
    'redirect',
    'search_api_task',
    'shortcut',
    'simplenews_subscriber',
    'simplenews_subscriber_history',
    'smart_date_override',
    'smart_date_rule',
    'webform_submission',
  ];

  /**
   * Cached module configuration.
   */
  protected ?ImmutableConfig $moduleConfig = NULL;

  protected EntityTypeManagerInterface $entityTypeManager;
  protected EntityFieldManagerInterface $entityFieldManager;
  protected EntityTypeBundleInfoInterface $entityTypeBundleInfo;
  protected EntityDisplayRepositoryInterface $entityDisplayRepository;
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->entityTypeBundleInfo = $container->get('entity_type.bundle.info');
    $instance->entityDisplayRepository = $container->get('entity_display.repository');
    $instance->configFactory = $container->get('config.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $config): array {
    return [
      'max_fields_per_bundle' => [
        '#type' => 'number',
        '#title' => $this->t('Maximum fields per bundle'),
        '#description' => $this->t('Maximum recommended fields per bundle. Guidelines: Simple sites: 15-20 | Medium sites: 25-35 | Complex sites: 40-50'),
        '#default_value' => $config['max_fields_per_bundle'] ?? 30,
        '#min' => 5,
        '#max' => 100,
        '#required' => TRUE,
      ],
      'max_view_modes' => [
        '#type' => 'number',
        '#title' => $this->t('Maximum view modes per bundle'),
        '#description' => $this->t('Maximum active view modes per bundle. Guidelines: Simple sites: 2-3 | Medium sites: 4-6 | Component-based: 8-12'),
        '#default_value' => $config['max_view_modes'] ?? 5,
        '#min' => 1,
        '#max' => 20,
        '#required' => TRUE,
      ],
      'max_form_modes' => [
        '#type' => 'number',
        '#title' => $this->t('Maximum form modes per bundle'),
        '#description' => $this->t('Maximum active form modes per bundle. Most sites only need 1-2.'),
        '#default_value' => $config['max_form_modes'] ?? 2,
        '#min' => 1,
        '#max' => 10,
        '#required' => TRUE,
      ],
    ];
  }

  /**
   * Gets the module configuration.
   */
  protected function getConfig(): ImmutableConfig {
    if ($this->moduleConfig === NULL) {
      $this->moduleConfig = $this->configFactory->get('audit_fields.settings');
    }
    return $this->moduleConfig;
  }

  /**
   * Gets configured limits and thresholds.
   *
   * Severity calculation:
   * - OK: count <= limit
   * - Warning: count > limit AND count <= error_threshold
   * - Error: count > error_threshold (limit * 1.2)
   */
  protected function getLimitsAndThresholds(): array {
    $max_fields = (int) ($this->getConfig()->get('max_fields_per_bundle') ?? 30);
    $max_view_modes = (int) ($this->getConfig()->get('max_view_modes') ?? 5);
    $max_form_modes = (int) ($this->getConfig()->get('max_form_modes') ?? 2);

    return [
      'limits' => [
        'fields' => $max_fields,
        'view_modes' => $max_view_modes,
        'form_modes' => $max_form_modes,
      ],
      // Error thresholds: limit * 1.2 (20% over limit triggers error).
      'error_thresholds' => [
        'fields' => (int) floor($max_fields * 1.2),
        'view_modes' => (int) floor($max_view_modes * 1.2),
        'form_modes' => (int) floor($max_form_modes * 1.2),
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    // Gather all data once.
    $inventory = $this->buildCompleteInventory();
    $config = $this->getLimitsAndThresholds();

    // Build separate files for each section.
    $files = $this->buildSectionFiles($inventory, $config);

    // Calculate scores.
    $scores = $this->calculateScores($inventory, $config);

    // Calculate stats for reference.
    $stats = $this->calculateStats($inventory, $config);

    return [
      '_files' => $files,
      'score' => $scores,
      'stats' => $stats,
      'config' => $config,
    ];
  }

  /**
   * Builds separate files for each section.
   */
  protected function buildSectionFiles(array $inventory, array $config): array {
    $files = [];
    $limits = $config['limits'];
    $errorThresholds = $config['error_thresholds'];

    // Initialize collections.
    $unusedFields = [];
    $visibilityIssues = [];
    $fieldCountIssues = [];
    $viewModeIssues = [];
    $formModeIssues = [];

    foreach ($inventory['entity_types'] as $entity_type_id => $entity_type_data) {
      foreach ($entity_type_data['bundles'] as $bundle_id => $bundle_data) {
        $bundleLabel = $entity_type_data['label'] . ':' . $bundle_data['label'];
        $fieldCount = $bundle_data['stats']['field_count'];
        $viewModeCount = count($bundle_data['view_modes'] ?? []);
        $formModeCount = count($bundle_data['form_modes'] ?? []);

        // Field count issues.
        // Error: count > error_threshold (limit * 1.2)
        // Warning: count > limit but <= error_threshold
        if ($fieldCount > $errorThresholds['fields']) {
          $fieldCountIssues[] = $this->createResultItem(
            'error',
            'FIELDS_CRITICAL',
            (string) $this->t('Bundle "@bundle" has @count fields (limit: @max, critical: @critical).', [
              '@bundle' => $bundleLabel,
              '@count' => $fieldCount,
              '@max' => $limits['fields'],
              '@critical' => $errorThresholds['fields'],
            ]),
            [
              'entity_type' => $entity_type_id,
              'entity_type_label' => $entity_type_data['label'],
              'bundle' => $bundle_id,
              'bundle_label' => $bundle_data['label'],
              'field_count' => $fieldCount,
              'limit' => $limits['fields'],
              'error_threshold' => $errorThresholds['fields'],
              'severity_level' => 'error',
            ]
          );
        }
        elseif ($fieldCount > $limits['fields']) {
          $fieldCountIssues[] = $this->createResultItem(
            'warning',
            'FIELDS_OVER_LIMIT',
            (string) $this->t('Bundle "@bundle" has @count fields (limit: @max).', [
              '@bundle' => $bundleLabel,
              '@count' => $fieldCount,
              '@max' => $limits['fields'],
            ]),
            [
              'entity_type' => $entity_type_id,
              'entity_type_label' => $entity_type_data['label'],
              'bundle' => $bundle_id,
              'bundle_label' => $bundle_data['label'],
              'field_count' => $fieldCount,
              'limit' => $limits['fields'],
              'error_threshold' => $errorThresholds['fields'],
              'severity_level' => 'warning',
            ]
          );
        }

        // View mode issues.
        // Error: count > error_threshold (limit * 1.2)
        // Warning: count > limit but <= error_threshold
        if ($viewModeCount > $errorThresholds['view_modes']) {
          $viewModeIssues[] = $this->createResultItem(
            'error',
            'VIEW_MODES_CRITICAL',
            (string) $this->t('Bundle "@bundle" has @count view modes (limit: @max, critical: @critical).', [
              '@bundle' => $bundleLabel,
              '@count' => $viewModeCount,
              '@max' => $limits['view_modes'],
              '@critical' => $errorThresholds['view_modes'],
            ]),
            [
              'entity_type' => $entity_type_id,
              'entity_type_label' => $entity_type_data['label'],
              'bundle' => $bundle_id,
              'bundle_label' => $bundle_data['label'],
              'view_mode_count' => $viewModeCount,
              'limit' => $limits['view_modes'],
              'error_threshold' => $errorThresholds['view_modes'],
              'view_modes' => array_keys($bundle_data['view_modes'] ?? []),
              'severity_level' => 'error',
            ]
          );
        }
        elseif ($viewModeCount > $limits['view_modes']) {
          $viewModeIssues[] = $this->createResultItem(
            'warning',
            'VIEW_MODES_OVER_LIMIT',
            (string) $this->t('Bundle "@bundle" has @count view modes (limit: @max).', [
              '@bundle' => $bundleLabel,
              '@count' => $viewModeCount,
              '@max' => $limits['view_modes'],
            ]),
            [
              'entity_type' => $entity_type_id,
              'entity_type_label' => $entity_type_data['label'],
              'bundle' => $bundle_id,
              'bundle_label' => $bundle_data['label'],
              'view_mode_count' => $viewModeCount,
              'limit' => $limits['view_modes'],
              'error_threshold' => $errorThresholds['view_modes'],
              'view_modes' => array_keys($bundle_data['view_modes'] ?? []),
              'severity_level' => 'warning',
            ]
          );
        }

        // Form mode issues.
        // Error: count > error_threshold (limit * 1.2)
        // Warning: count > limit but <= error_threshold
        if ($formModeCount > $errorThresholds['form_modes']) {
          $formModeIssues[] = $this->createResultItem(
            'error',
            'FORM_MODES_CRITICAL',
            (string) $this->t('Bundle "@bundle" has @count form modes (limit: @max, critical: @critical).', [
              '@bundle' => $bundleLabel,
              '@count' => $formModeCount,
              '@max' => $limits['form_modes'],
              '@critical' => $errorThresholds['form_modes'],
            ]),
            [
              'entity_type' => $entity_type_id,
              'entity_type_label' => $entity_type_data['label'],
              'bundle' => $bundle_id,
              'bundle_label' => $bundle_data['label'],
              'form_mode_count' => $formModeCount,
              'limit' => $limits['form_modes'],
              'error_threshold' => $errorThresholds['form_modes'],
              'form_modes' => array_keys($bundle_data['form_modes'] ?? []),
              'severity_level' => 'error',
            ]
          );
        }
        elseif ($formModeCount > $limits['form_modes']) {
          $formModeIssues[] = $this->createResultItem(
            'warning',
            'FORM_MODES_OVER_LIMIT',
            (string) $this->t('Bundle "@bundle" has @count form modes (limit: @max).', [
              '@bundle' => $bundleLabel,
              '@count' => $formModeCount,
              '@max' => $limits['form_modes'],
            ]),
            [
              'entity_type' => $entity_type_id,
              'entity_type_label' => $entity_type_data['label'],
              'bundle' => $bundle_id,
              'bundle_label' => $bundle_data['label'],
              'form_mode_count' => $formModeCount,
              'limit' => $limits['form_modes'],
              'error_threshold' => $errorThresholds['form_modes'],
              'form_modes' => array_keys($bundle_data['form_modes'] ?? []),
              'severity_level' => 'warning',
            ]
          );
        }

        // Field visibility issues.
        foreach ($bundle_data['fields'] as $field_name => $field_info) {
          $hiddenInView = $field_info['hidden_in_all_view_modes'] ?? FALSE;
          $hiddenInForm = $field_info['hidden_in_all_form_modes'] ?? FALSE;

          if ($hiddenInView && $hiddenInForm) {
            $unusedFields[] = $this->createResultItem(
              'warning',
              'FIELD_UNUSED',
              (string) $this->t('Field "@field" in @bundle is not used in any display or form.', [
                '@field' => $field_info['label'],
                '@bundle' => $bundleLabel,
              ]),
              [
                'entity_type' => $entity_type_id,
                'entity_type_label' => $entity_type_data['label'],
                'bundle' => $bundle_id,
                'bundle_label' => $bundle_data['label'],
                'field_name' => $field_name,
                'field_label' => $field_info['label'],
                'field_type' => $field_info['type'],
              ]
            );
          }
          elseif ($hiddenInView) {
            $visibilityIssues[] = $this->createResultItem(
              'notice',
              'FIELD_NOT_DISPLAYED',
              (string) $this->t('Field "@field" in @bundle is not displayed in any view mode.', [
                '@field' => $field_info['label'],
                '@bundle' => $bundleLabel,
              ]),
              [
                'entity_type' => $entity_type_id,
                'entity_type_label' => $entity_type_data['label'],
                'bundle' => $bundle_id,
                'bundle_label' => $bundle_data['label'],
                'field_name' => $field_name,
                'field_label' => $field_info['label'],
                'field_type' => $field_info['type'],
                'issue_type' => 'not_displayed',
              ]
            );
          }
          elseif ($hiddenInForm) {
            $visibilityIssues[] = $this->createResultItem(
              'notice',
              'FIELD_NOT_IN_FORM',
              (string) $this->t('Field "@field" in @bundle is not in any form mode.', [
                '@field' => $field_info['label'],
                '@bundle' => $bundleLabel,
              ]),
              [
                'entity_type' => $entity_type_id,
                'entity_type_label' => $entity_type_data['label'],
                'bundle' => $bundle_id,
                'bundle_label' => $bundle_data['label'],
                'field_name' => $field_name,
                'field_label' => $field_info['label'],
                'field_type' => $field_info['type'],
                'issue_type' => 'not_in_form',
              ]
            );
          }
        }
      }
    }

    // Build files.
    $unusedWarnings = count($unusedFields);
    $files['unused_fields'] = $this->createResult($unusedFields, 0, $unusedWarnings, 0);
    $files['unused_fields']['count'] = $unusedWarnings;

    $visibilityNotices = count($visibilityIssues);
    $files['field_visibility'] = $this->createResult($visibilityIssues, 0, 0, $visibilityNotices);
    $files['field_visibility']['count'] = $visibilityNotices;

    $fieldCountErrors = count(array_filter($fieldCountIssues, fn($i) => $i['severity'] === 'error'));
    $fieldCountWarnings = count(array_filter($fieldCountIssues, fn($i) => $i['severity'] === 'warning'));
    $files['field_count'] = $this->createResult($fieldCountIssues, $fieldCountErrors, $fieldCountWarnings, 0);
    $files['field_count']['limits'] = $limits;
    $files['field_count']['error_thresholds'] = $errorThresholds;

    $viewModeErrors = count(array_filter($viewModeIssues, fn($i) => $i['severity'] === 'error'));
    $viewModeWarnings = count(array_filter($viewModeIssues, fn($i) => $i['severity'] === 'warning'));
    $files['view_modes'] = $this->createResult($viewModeIssues, $viewModeErrors, $viewModeWarnings, 0);
    $files['view_modes']['limits'] = $limits;
    $files['view_modes']['error_thresholds'] = $errorThresholds;

    $formModeErrors = count(array_filter($formModeIssues, fn($i) => $i['severity'] === 'error'));
    $formModeWarnings = count(array_filter($formModeIssues, fn($i) => $i['severity'] === 'warning'));
    $files['form_modes'] = $this->createResult($formModeIssues, $formModeErrors, $formModeWarnings, 0);
    $files['form_modes']['limits'] = $limits;
    $files['form_modes']['error_thresholds'] = $errorThresholds;

    // Inventory file (informational).
    $files['inventory'] = $this->createResult([], 0, 0, 0);
    $files['inventory']['entity_types'] = $inventory['entity_types'];
    $files['inventory']['totals'] = $inventory['totals'];

    return $files;
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      'unused_fields' => [
        'label' => $this->t('Unused Fields'),
        'description' => $this->t('Fields not used in any display mode or form mode. These consume database space and add complexity.'),
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['unused_fields'],
        'file_key' => 'unused_fields',
        'score_factor_key' => 'unused_fields',
      ],
      'field_visibility' => [
        'label' => $this->t('Field Visibility'),
        'description' => $this->t('Fields with partial visibility (only in forms or only in displays).'),
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['field_visibility'],
        'file_key' => 'field_visibility',
        'score_factor_key' => 'field_visibility',
      ],
      'field_count' => [
        'label' => $this->t('Field Count'),
        'description' => $this->t('Bundles with excessive fields increase complexity and maintenance effort.'),
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['field_count'],
        'file_key' => 'field_count',
        'score_factor_key' => 'field_count',
      ],
      'view_modes' => [
        'label' => $this->t('View Modes'),
        'description' => $this->t('Bundles with excessive view modes increase maintenance overhead.'),
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['view_modes'],
        'file_key' => 'view_modes',
        'score_factor_key' => 'view_modes',
      ],
      'form_modes' => [
        'label' => $this->t('Form Modes'),
        'description' => $this->t('Bundles with excessive form modes. Most sites only need 1-2.'),
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['form_modes'],
        'file_key' => 'form_modes',
        'score_factor_key' => 'form_modes',
      ],
      'inventory' => [
        'label' => $this->t('Field Inventory'),
        'description' => $this->t('Complete inventory of entity types, bundles, fields, and display modes. Informational only.'),
        'affects_score' => FALSE,
        'weight' => 0,
        'file_key' => 'inventory',
        'score_factor_key' => NULL,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    $files = $data['_files'] ?? [];

    return match ($check_id) {
      'unused_fields' => $this->buildUnusedFieldsContent($files),
      'field_visibility' => $this->buildFieldVisibilityContent($files),
      'field_count' => $this->buildFieldCountContent($files),
      'view_modes' => $this->buildViewModesContent($files),
      'form_modes' => $this->buildFormModesContent($files),
      'inventory' => $this->buildInventoryContent($files),
      default => [],
    };
  }

  /**
   * Returns custom filter configuration for fields audit.
   */
  protected function getFieldsCustomFilters(): array {
    return [
      'entity-type' => [
        'label' => (string) $this->t('Entity Type'),
        'attribute' => 'data-entity-type',
        'styled' => TRUE,
      ],
      'bundle' => [
        'label' => (string) $this->t('Bundle'),
        'attribute' => 'data-bundle',
        'styled' => TRUE,
      ],
      'field-type' => [
        'label' => (string) $this->t('Field Type'),
        'attribute' => 'data-field-type',
        'styled' => TRUE,
      ],
    ];
  }

  /**
   * Builds the unused fields section content.
   */
  protected function buildUnusedFieldsContent(array $files): array {
    $unusedData = $files['unused_fields'] ?? [];

    $fix_html = '<p><strong>Problem:</strong> This field is not configured in any view mode or form mode. It consumes database space and adds complexity without providing value.</p>'
      . '<p><strong>Solutions:</strong></p>'
      . '<ul>'
      . '<li><strong>Delete field:</strong> If the field contains no data or the data is no longer needed, delete the field configuration.</li>'
      . '<li><strong>Configure display:</strong> If the field should be visible, configure it in the appropriate view modes.</li>'
      . '<li><strong>Configure form:</strong> If the field should be editable, configure it in the appropriate form modes.</li>'
      . '<li><strong>Programmatic use:</strong> If the field is used programmatically (in code, Views, etc.), document this and consider adding it to at least one display for reference.</li>'
      . '</ul>';

    return $this->ui->buildIssueListFromResults(
      $unusedData['results'] ?? [],
      (string) $this->t('All fields are used in at least one display mode or form mode.'),
      function (array $item, $ui) use ($fix_html): array {
        $details = $item['details'] ?? [];
        $fieldName = $details['field_name'] ?? '';
        $fieldLabel = $details['field_label'] ?? '';
        $fieldType = $details['field_type'] ?? '';
        $entityType = $details['entity_type'] ?? '';
        $entityTypeLabel = $details['entity_type_label'] ?? '';
        $bundle = $details['bundle'] ?? '';
        $bundleLabel = $details['bundle_label'] ?? '';

        $description = '<p><strong>Details:</strong></p>'
          . '<ul>'
          . '<li><strong>Entity type:</strong> ' . $entityTypeLabel . ' (' . $entityType . ')</li>'
          . '<li><strong>Bundle:</strong> ' . $bundleLabel . ' (' . $bundle . ')</li>'
          . '<li><strong>Field:</strong> ' . $fieldLabel . ' (' . $fieldName . ')</li>'
          . '<li><strong>Field type:</strong> ' . $fieldType . '</li>'
          . '</ul>'
          . $fix_html;

        return [
          'severity' => 'warning',
          'code' => 'UNUSED',
          'file' => $fieldName,
          'label' => (string) $this->t('Unused field'),
          'description' => ['#markup' => $description],
          'tags' => ['cleanup', 'maintenance'],
          'custom_data' => [
            'entity-type' => $entityType,
            'bundle' => $bundle,
            'field-type' => $fieldType,
          ],
        ];
      },
      $this->getFieldsCustomFilters()
    );
  }

  /**
   * Builds the field visibility section content.
   */
  protected function buildFieldVisibilityContent(array $files): array {
    $visibilityData = $files['field_visibility'] ?? [];

    $fix_html = '<p><strong>Note:</strong> This field has partial visibility, which may be intentional.</p>'
      . '<p><strong>Common scenarios:</strong></p>'
      . '<ul>'
      . '<li><strong>Only in form:</strong> Field is editable but displayed via custom code or Views.</li>'
      . '<li><strong>Only in display:</strong> Field is computed, auto-populated, or edited via other means (API, migrations).</li>'
      . '</ul>'
      . '<p><strong>Recommendation:</strong> Review if this configuration is intentional. If the field should be visible/editable, configure it appropriately.</p>';

    return $this->ui->buildIssueListFromResults(
      $visibilityData['results'] ?? [],
      (string) $this->t('All fields are visible in both display modes and form modes.'),
      function (array $item, $ui) use ($fix_html): array {
        $details = $item['details'] ?? [];
        $fieldName = $details['field_name'] ?? '';
        $fieldLabel = $details['field_label'] ?? '';
        $fieldType = $details['field_type'] ?? '';
        $entityType = $details['entity_type'] ?? '';
        $entityTypeLabel = $details['entity_type_label'] ?? '';
        $bundle = $details['bundle'] ?? '';
        $bundleLabel = $details['bundle_label'] ?? '';
        $issueType = $details['issue_type'] ?? 'unknown';

        $issueTypeLabel = $issueType === 'not_displayed'
          ? (string) $this->t('Not in any display mode')
          : (string) $this->t('Not in any form mode');

        // Simplified label for Issue Type filter.
        $simpleLabel = $issueType === 'not_displayed'
          ? (string) $this->t('Not displayed')
          : (string) $this->t('Not in form');

        $description = '<p><strong>Details:</strong></p>'
          . '<ul>'
          . '<li><strong>Entity type:</strong> ' . $entityTypeLabel . ' (' . $entityType . ')</li>'
          . '<li><strong>Bundle:</strong> ' . $bundleLabel . ' (' . $bundle . ')</li>'
          . '<li><strong>Field:</strong> ' . $fieldLabel . ' (' . $fieldName . ')</li>'
          . '<li><strong>Field type:</strong> ' . $fieldType . '</li>'
          . '<li><strong>Issue:</strong> ' . $issueTypeLabel . '</li>'
          . '</ul>'
          . $fix_html;

        return [
          'severity' => 'notice',
          'code' => 'VISIBILITY',
          'file' => $fieldName,
          'label' => $simpleLabel,
          'description' => ['#markup' => $description],
          'tags' => ['configuration'],
          'custom_data' => [
            'entity-type' => $entityType,
            'bundle' => $bundle,
            'field-type' => $fieldType,
          ],
        ];
      },
      $this->getFieldsCustomFilters()
    );
  }

  /**
   * Returns custom filter configuration for bundle-level audit sections.
   */
  protected function getBundleCustomFilters(): array {
    return [
      'entity-type' => [
        'label' => (string) $this->t('Entity Type'),
        'attribute' => 'data-entity-type',
        'styled' => TRUE,
      ],
      'bundle' => [
        'label' => (string) $this->t('Bundle'),
        'attribute' => 'data-bundle',
        'styled' => TRUE,
      ],
    ];
  }

  /**
   * Builds the field count section content.
   */
  protected function buildFieldCountContent(array $files): array {
    $fieldCountData = $files['field_count'] ?? [];
    $limits = $fieldCountData['limits'] ?? [];

    $fix_html = '<p><strong>Problem:</strong> Bundles with too many fields become difficult to maintain, slow down entity operations, and increase page load times.</p>'
      . '<p><strong>Solutions:</strong></p>'
      . '<ul>'
      . '<li><strong>Remove unused fields:</strong> Delete fields that are no longer needed.</li>'
      . '<li><strong>Use Paragraphs:</strong> Group related fields into Paragraph types for better organization.</li>'
      . '<li><strong>Split content type:</strong> Consider creating separate content types for different use cases.</li>'
      . '<li><strong>Use references:</strong> Move complex data to separate entities and reference them.</li>'
      . '</ul>';

    return $this->ui->buildIssueListFromResults(
      $fieldCountData['results'] ?? [],
      (string) $this->t('All bundles have reasonable field counts (limit: @max).', [
        '@max' => $limits['fields'] ?? 30,
      ]),
      function (array $item, $ui) use ($fix_html): array {
        $details = $item['details'] ?? [];
        $entityType = $details['entity_type'] ?? '';
        $entityTypeLabel = $details['entity_type_label'] ?? '';
        $bundle = $details['bundle'] ?? '';
        $bundleLabel = $details['bundle_label'] ?? '';
        $fieldCount = $details['field_count'] ?? 0;
        $limit = $details['limit'] ?? 30;
        $errorThreshold = $details['error_threshold'] ?? (int) floor($limit * 1.2);
        $severityLevel = $details['severity_level'] ?? 'warning';

        // Simplified label for Issue Type filter.
        $simpleLabel = $severityLevel === 'error'
          ? (string) $this->t('Critical')
          : (string) $this->t('Over limit');

        $statusText = $severityLevel === 'error'
          ? (string) $this->t('Critical (exceeds @threshold)', ['@threshold' => $errorThreshold])
          : (string) $this->t('Over limit');

        $description = '<p><strong>Details:</strong></p>'
          . '<ul>'
          . '<li><strong>Entity type:</strong> ' . $entityTypeLabel . ' (' . $entityType . ')</li>'
          . '<li><strong>Bundle:</strong> ' . $bundleLabel . ' (' . $bundle . ')</li>'
          . '<li><strong>Field count:</strong> ' . $fieldCount . '</li>'
          . '<li><strong>Limit:</strong> ' . $limit . '</li>'
          . '<li><strong>Critical threshold:</strong> ' . $errorThreshold . '</li>'
          . '<li><strong>Status:</strong> ' . $statusText . '</li>'
          . '</ul>'
          . $fix_html;

        return [
          'severity' => $item['severity'] ?? 'warning',
          'code' => 'FIELD_COUNT',
          'file' => $entityType . ':' . $bundle,
          'label' => $simpleLabel,
          'description' => ['#markup' => $description],
          'tags' => ['complexity', 'performance'],
          'custom_data' => [
            'entity-type' => $entityType,
            'bundle' => $bundle,
          ],
        ];
      },
      $this->getBundleCustomFilters()
    );
  }

  /**
   * Builds the view modes section content.
   */
  protected function buildViewModesContent(array $files): array {
    $viewModesData = $files['view_modes'] ?? [];
    $limits = $viewModesData['limits'] ?? [];

    $fix_html = '<p><strong>Problem:</strong> Too many view modes increase maintenance overhead. Each view mode needs field configuration.</p>'
      . '<p><strong>Solutions:</strong></p>'
      . '<ul>'
      . '<li><strong>Consolidate view modes:</strong> Merge similar view modes that have minor differences.</li>'
      . '<li><strong>Use CSS:</strong> Use CSS variations instead of separate view modes for styling differences.</li>'
      . '<li><strong>Review usage:</strong> Check if all view modes are actually used in the site.</li>'
      . '<li><strong>Disable unused:</strong> Disable view modes that are not actively used.</li>'
      . '</ul>';

    return $this->ui->buildIssueListFromResults(
      $viewModesData['results'] ?? [],
      (string) $this->t('All bundles have reasonable view mode counts (limit: @max).', [
        '@max' => $limits['view_modes'] ?? 5,
      ]),
      function (array $item, $ui) use ($fix_html): array {
        $details = $item['details'] ?? [];
        $entityType = $details['entity_type'] ?? '';
        $entityTypeLabel = $details['entity_type_label'] ?? '';
        $bundle = $details['bundle'] ?? '';
        $bundleLabel = $details['bundle_label'] ?? '';
        $viewModeCount = $details['view_mode_count'] ?? 0;
        $limit = $details['limit'] ?? 5;
        $errorThreshold = $details['error_threshold'] ?? (int) floor($limit * 1.2);
        $viewModes = $details['view_modes'] ?? [];
        $severityLevel = $details['severity_level'] ?? 'warning';

        // Simplified label for Issue Type filter.
        $simpleLabel = $severityLevel === 'error'
          ? (string) $this->t('Critical')
          : (string) $this->t('Over limit');

        $description = '<p><strong>Details:</strong></p>'
          . '<ul>'
          . '<li><strong>Entity type:</strong> ' . $entityTypeLabel . ' (' . $entityType . ')</li>'
          . '<li><strong>Bundle:</strong> ' . $bundleLabel . ' (' . $bundle . ')</li>'
          . '<li><strong>View mode count:</strong> ' . $viewModeCount . '</li>'
          . '<li><strong>Limit:</strong> ' . $limit . '</li>'
          . '<li><strong>Critical threshold:</strong> ' . $errorThreshold . '</li>'
          . '<li><strong>View modes:</strong> ' . implode(', ', $viewModes) . '</li>'
          . '</ul>'
          . $fix_html;

        return [
          'severity' => $item['severity'] ?? 'warning',
          'code' => 'VIEW_MODES',
          'file' => $entityType . ':' . $bundle,
          'label' => $simpleLabel,
          'description' => ['#markup' => $description],
          'tags' => ['complexity', 'maintenance'],
          'custom_data' => [
            'entity-type' => $entityType,
            'bundle' => $bundle,
          ],
        ];
      },
      $this->getBundleCustomFilters()
    );
  }

  /**
   * Builds the form modes section content.
   */
  protected function buildFormModesContent(array $files): array {
    $formModesData = $files['form_modes'] ?? [];
    $limits = $formModesData['limits'] ?? [];

    $fix_html = '<p><strong>Problem:</strong> Most sites only need 1-2 form modes. Excessive form modes add complexity.</p>'
      . '<p><strong>Solutions:</strong></p>'
      . '<ul>'
      . '<li><strong>Review necessity:</strong> Check if each form mode is actually needed.</li>'
      . '<li><strong>Use field permissions:</strong> Use field-level permissions instead of separate form modes for access control.</li>'
      . '<li><strong>Consolidate:</strong> Merge form modes with minor differences.</li>'
      . '</ul>';

    return $this->ui->buildIssueListFromResults(
      $formModesData['results'] ?? [],
      (string) $this->t('All bundles have reasonable form mode counts (limit: @max).', [
        '@max' => $limits['form_modes'] ?? 2,
      ]),
      function (array $item, $ui) use ($fix_html): array {
        $details = $item['details'] ?? [];
        $entityType = $details['entity_type'] ?? '';
        $entityTypeLabel = $details['entity_type_label'] ?? '';
        $bundle = $details['bundle'] ?? '';
        $bundleLabel = $details['bundle_label'] ?? '';
        $formModeCount = $details['form_mode_count'] ?? 0;
        $limit = $details['limit'] ?? 2;
        $errorThreshold = $details['error_threshold'] ?? (int) floor($limit * 1.2);
        $formModes = $details['form_modes'] ?? [];
        $severityLevel = $details['severity_level'] ?? 'warning';

        // Simplified label for Issue Type filter.
        $simpleLabel = $severityLevel === 'error'
          ? (string) $this->t('Critical')
          : (string) $this->t('Over limit');

        $description = '<p><strong>Details:</strong></p>'
          . '<ul>'
          . '<li><strong>Entity type:</strong> ' . $entityTypeLabel . ' (' . $entityType . ')</li>'
          . '<li><strong>Bundle:</strong> ' . $bundleLabel . ' (' . $bundle . ')</li>'
          . '<li><strong>Form mode count:</strong> ' . $formModeCount . '</li>'
          . '<li><strong>Limit:</strong> ' . $limit . '</li>'
          . '<li><strong>Critical threshold:</strong> ' . $errorThreshold . '</li>'
          . '<li><strong>Form modes:</strong> ' . implode(', ', $formModes) . '</li>'
          . '</ul>'
          . $fix_html;

        return [
          'severity' => $item['severity'] ?? 'warning',
          'code' => 'FORM_MODES',
          'file' => $entityType . ':' . $bundle,
          'label' => $simpleLabel,
          'description' => ['#markup' => $description],
          'tags' => ['complexity', 'maintenance'],
          'custom_data' => [
            'entity-type' => $entityType,
            'bundle' => $bundle,
          ],
        ];
      },
      $this->getBundleCustomFilters()
    );
  }

  /**
   * Builds the inventory section content (informational table).
   */
  protected function buildInventoryContent(array $files): array {
    $inventoryData = $files['inventory'] ?? [];
    $entityTypes = $inventoryData['entity_types'] ?? [];
    $totals = $inventoryData['totals'] ?? [];

    if (empty($entityTypes)) {
      return [
        '#type' => 'container',
        'info' => [
          '#type' => 'html_tag',
          '#tag' => 'p',
          '#attributes' => ['class' => ['audit-info-text']],
          '#value' => '<em>' . $this->t('This section is informational only and does not affect the audit score.') . '</em>',
        ],
        'message' => $this->ui->message(
          (string) $this->t('No content entity types found.'),
          'info'
        ),
      ];
    }

    $summaryText = $this->t('@types entity types, @bundles bundles, @fields fields, @entities total entities.', [
      '@types' => $totals['entity_types'] ?? 0,
      '@bundles' => $totals['bundles'] ?? 0,
      '@fields' => $totals['fields'] ?? 0,
      '@entities' => number_format($totals['entities'] ?? 0),
    ]);

    $content = [
      '#type' => 'container',
      'info' => [
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#attributes' => ['class' => ['audit-info-text']],
        '#value' => '<em>' . $this->t('This section is informational only and does not affect the audit score.') . '</em>',
      ],
      'summary' => [
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#value' => $summaryText,
      ],
    ];

    // Build overview table.
    $headers = [
      $this->ui->header((string) $this->t('Entity Type')),
      $this->ui->header((string) $this->t('Bundles'), 'center'),
      $this->ui->header((string) $this->t('Fields'), 'center'),
      $this->ui->header((string) $this->t('View Modes'), 'center'),
      $this->ui->header((string) $this->t('Entities'), 'right'),
    ];

    $rows = [];
    foreach ($entityTypes as $entity_type_id => $entity_type_data) {
      $bundleCount = count($entity_type_data['bundles']);
      $fieldCount = 0;
      foreach ($entity_type_data['bundles'] as $bundle_data) {
        $fieldCount += $bundle_data['stats']['field_count'];
      }
      $viewModeCount = count($entity_type_data['view_modes']);
      $entityCount = $entity_type_data['entity_count'];

      $rows[] = $this->ui->row([
        $this->ui->itemName((string) $entity_type_data['label'], $entity_type_id),
        $this->ui->cell((string) $bundleCount, ['align' => 'center']),
        $this->ui->cell((string) $fieldCount, ['align' => 'center']),
        $this->ui->cell((string) $viewModeCount, ['align' => 'center']),
        $this->ui->cell(number_format($entityCount), ['align' => 'right']),
      ]);
    }

    $content['table'] = $this->ui->table($headers, $rows, [
      'empty' => (string) $this->t('No entity types found.'),
    ]);

    return $content;
  }

  // =========================================================================
  // Data gathering methods.
  // =========================================================================

  /**
   * Builds the complete inventory of entity types, bundles, fields, and displays.
   */
  protected function buildCompleteInventory(): array {
    $inventory = [
      'entity_types' => [],
      'totals' => [
        'entity_types' => 0,
        'bundles' => 0,
        'fields' => 0,
        'entities' => 0,
      ],
    ];

    $content_entity_types = $this->getContentEntityTypes();

    foreach ($content_entity_types as $entity_type_id => $entity_type) {
      $entity_count = $this->getEntityCount($entity_type_id);
      $bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);

      $all_view_modes = $this->entityDisplayRepository->getViewModeOptions($entity_type_id);
      $all_form_modes = $this->entityDisplayRepository->getFormModeOptions($entity_type_id);

      $entity_type_data = [
        'label' => (string) $entity_type->getLabel(),
        'entity_count' => $entity_count,
        'bundles' => [],
        'view_modes' => [],
        'form_modes' => [],
      ];

      foreach ($bundles as $bundle_id => $bundle_info) {
        $bundle_count = $this->getBundleCount($entity_type_id, $bundle_id, $entity_type);

        $active_view_modes = $this->getActiveViewModes($entity_type_id, $bundle_id, $all_view_modes);
        $active_form_modes = $this->getActiveFormModes($entity_type_id, $bundle_id, $all_form_modes);

        $bundle_data = $this->analyzeBundleFields($entity_type_id, $bundle_id, $bundle_info, $active_view_modes, $active_form_modes);
        $bundle_data['entity_count'] = $bundle_count;
        $bundle_data['view_modes'] = $active_view_modes;
        $bundle_data['form_modes'] = $active_form_modes;

        $entity_type_data['bundles'][$bundle_id] = $bundle_data;
        $inventory['totals']['bundles']++;
        $inventory['totals']['fields'] += $bundle_data['stats']['field_count'];

        foreach ($active_view_modes as $mode_id => $mode_label) {
          $entity_type_data['view_modes'][$mode_id] = $mode_label;
        }
        foreach ($active_form_modes as $mode_id => $mode_label) {
          $entity_type_data['form_modes'][$mode_id] = $mode_label;
        }
      }

      $inventory['entity_types'][$entity_type_id] = $entity_type_data;
      $inventory['totals']['entity_types']++;
      $inventory['totals']['entities'] += $entity_count;
    }

    return $inventory;
  }

  /**
   * Gets active view modes for a bundle.
   */
  protected function getActiveViewModes(string $entity_type_id, string $bundle_id, array $all_view_modes): array {
    $active_modes = [];

    foreach ($all_view_modes as $mode_id => $mode_label) {
      $display = $this->loadViewDisplay($entity_type_id, $bundle_id, $mode_id);
      if ($display && $display->status()) {
        $active_modes[$mode_id] = (string) $mode_label;
      }
    }

    if (empty($active_modes) && isset($all_view_modes['default'])) {
      $active_modes['default'] = (string) $all_view_modes['default'];
    }

    return $active_modes;
  }

  /**
   * Gets active form modes for a bundle.
   */
  protected function getActiveFormModes(string $entity_type_id, string $bundle_id, array $all_form_modes): array {
    $active_modes = [];

    foreach ($all_form_modes as $mode_id => $mode_label) {
      $display = $this->loadFormDisplay($entity_type_id, $bundle_id, $mode_id);
      if ($display && $display->status()) {
        $active_modes[$mode_id] = (string) $mode_label;
      }
    }

    if (empty($active_modes) && isset($all_form_modes['default'])) {
      $active_modes['default'] = (string) $all_form_modes['default'];
    }

    return $active_modes;
  }

  /**
   * Analyzes fields for a specific bundle.
   */
  protected function analyzeBundleFields(string $entity_type_id, string $bundle_id, array $bundle_info, array $active_view_modes, array $active_form_modes): array {
    $fields = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle_id);
    $view_modes = array_keys($active_view_modes);
    $form_modes = array_keys($active_form_modes);

    $field_data = [];
    $hidden_in_all_view_modes = 0;
    $hidden_in_all_form_modes = 0;

    foreach ($fields as $field_name => $field_definition) {
      if (!$field_definition instanceof FieldConfigInterface) {
        continue;
      }

      $field_info = [
        'label' => (string) $field_definition->getLabel(),
        'type' => $field_definition->getType(),
        'required' => $field_definition->isRequired(),
        'view_modes' => [],
        'form_modes' => [],
      ];

      $visible_in_view_mode = FALSE;
      foreach ($view_modes as $view_mode) {
        $display = $this->loadViewDisplay($entity_type_id, $bundle_id, $view_mode);
        $component = $display ? $display->getComponent($field_name) : NULL;

        $field_info['view_modes'][$view_mode] = [
          'visible' => $component !== NULL,
          'type' => $component['type'] ?? NULL,
          'weight' => $component['weight'] ?? NULL,
        ];

        if ($component !== NULL) {
          $visible_in_view_mode = TRUE;
        }
      }

      $visible_in_form_mode = FALSE;
      foreach ($form_modes as $form_mode) {
        $display = $this->loadFormDisplay($entity_type_id, $bundle_id, $form_mode);
        $component = $display ? $display->getComponent($field_name) : NULL;

        $field_info['form_modes'][$form_mode] = [
          'visible' => $component !== NULL,
          'type' => $component['type'] ?? NULL,
          'weight' => $component['weight'] ?? NULL,
        ];

        if ($component !== NULL) {
          $visible_in_form_mode = TRUE;
        }
      }

      $field_info['hidden_in_all_view_modes'] = !$visible_in_view_mode;
      $field_info['hidden_in_all_form_modes'] = !$visible_in_form_mode;

      if (!$visible_in_view_mode) {
        $hidden_in_all_view_modes++;
      }
      if (!$visible_in_form_mode) {
        $hidden_in_all_form_modes++;
      }

      $field_data[$field_name] = $field_info;
    }

    return [
      'label' => (string) ($bundle_info['label'] ?? $bundle_id),
      'fields' => $field_data,
      'stats' => [
        'field_count' => count($field_data),
        'hidden_in_all_view_modes' => $hidden_in_all_view_modes,
        'hidden_in_all_form_modes' => $hidden_in_all_form_modes,
      ],
    ];
  }

  /**
   * Loads a view display configuration.
   */
  protected function loadViewDisplay(string $entity_type_id, string $bundle_id, string $view_mode): ?EntityViewDisplayInterface {
    $display = $this->entityTypeManager
      ->getStorage('entity_view_display')
      ->load($entity_type_id . '.' . $bundle_id . '.' . $view_mode);
    return $display instanceof EntityViewDisplayInterface ? $display : NULL;
  }

  /**
   * Loads a form display configuration.
   */
  protected function loadFormDisplay(string $entity_type_id, string $bundle_id, string $form_mode): ?EntityFormDisplayInterface {
    $display = $this->entityTypeManager
      ->getStorage('entity_form_display')
      ->load($entity_type_id . '.' . $bundle_id . '.' . $form_mode);
    return $display instanceof EntityFormDisplayInterface ? $display : NULL;
  }

  /**
   * Gets total entity count for an entity type.
   */
  protected function getEntityCount(string $entity_type_id): int {
    try {
      $storage = $this->entityTypeManager->getStorage($entity_type_id);
      return (int) $storage->getQuery()->accessCheck(FALSE)->count()->execute();
    }
    catch (\Exception $e) {
      return 0;
    }
  }

  /**
   * Gets entity count for a specific bundle.
   */
  protected function getBundleCount(string $entity_type_id, string $bundle_id, EntityTypeInterface $entity_type): int {
    try {
      $bundle_key = $entity_type->getKey('bundle');
      if (!$bundle_key) {
        return $this->getEntityCount($entity_type_id);
      }

      $storage = $this->entityTypeManager->getStorage($entity_type_id);
      return (int) $storage->getQuery()
        ->accessCheck(FALSE)
        ->condition($bundle_key, $bundle_id)
        ->count()
        ->execute();
    }
    catch (\Exception $e) {
      return 0;
    }
  }

  /**
   * Gets all content entity types.
   */
  protected function getContentEntityTypes(): array {
    $entity_types = [];

    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
      if ($entity_type->getGroup() === 'content' && !in_array($entity_type_id, self::EXCLUDED_ENTITY_TYPES, TRUE)) {
        $entity_types[$entity_type_id] = $entity_type;
      }
    }

    return $entity_types;
  }

  // =========================================================================
  // Statistics and score calculation.
  // =========================================================================

  /**
   * Calculates statistics from inventory.
   *
   * Uses the new severity calculation:
   * - Error (critical): count > error_threshold (limit * 1.2)
   * - Warning: count > limit AND count <= error_threshold
   * - OK: count <= limit
   */
  protected function calculateStats(array $inventory, array $config): array {
    $limits = $config['limits'];
    $errorThresholds = $config['error_thresholds'];

    $stats = [
      'total_entity_types' => $inventory['totals']['entity_types'],
      'total_bundles' => $inventory['totals']['bundles'],
      'total_fields' => $inventory['totals']['fields'],
      'total_entities' => $inventory['totals']['entities'],
      // Field count stats.
      'bundles_field_critical' => 0,
      'bundles_field_warning' => 0,
      // View mode stats.
      'bundles_view_mode_critical' => 0,
      'bundles_view_mode_warning' => 0,
      // Form mode stats.
      'bundles_form_mode_critical' => 0,
      'bundles_form_mode_warning' => 0,
      // Field visibility stats.
      'fields_unused' => 0,
      'fields_not_displayed' => 0,
      'fields_not_in_form' => 0,
    ];

    foreach ($inventory['entity_types'] as $entity_type_data) {
      foreach ($entity_type_data['bundles'] as $bundle_data) {
        $field_count = $bundle_data['stats']['field_count'];
        $view_mode_count = count($bundle_data['view_modes'] ?? []);
        $form_mode_count = count($bundle_data['form_modes'] ?? []);

        // Field count: error if > error_threshold, warning if > limit.
        if ($field_count > $errorThresholds['fields']) {
          $stats['bundles_field_critical']++;
        }
        elseif ($field_count > $limits['fields']) {
          $stats['bundles_field_warning']++;
        }

        // View modes: error if > error_threshold, warning if > limit.
        if ($view_mode_count > $errorThresholds['view_modes']) {
          $stats['bundles_view_mode_critical']++;
        }
        elseif ($view_mode_count > $limits['view_modes']) {
          $stats['bundles_view_mode_warning']++;
        }

        // Form modes: error if > error_threshold, warning if > limit.
        if ($form_mode_count > $errorThresholds['form_modes']) {
          $stats['bundles_form_mode_critical']++;
        }
        elseif ($form_mode_count > $limits['form_modes']) {
          $stats['bundles_form_mode_warning']++;
        }

        foreach ($bundle_data['fields'] as $field_info) {
          $hidden_in_view = $field_info['hidden_in_all_view_modes'] ?? FALSE;
          $hidden_in_form = $field_info['hidden_in_all_form_modes'] ?? FALSE;

          if ($hidden_in_view && $hidden_in_form) {
            $stats['fields_unused']++;
          }
          elseif ($hidden_in_view) {
            $stats['fields_not_displayed']++;
          }
          elseif ($hidden_in_form) {
            $stats['fields_not_in_form']++;
          }
        }
      }
    }

    return $stats;
  }

  /**
   * Calculates scores for all factors.
   *
   * Score penalties:
   * - Critical (error): -25 points per bundle
   * - Warning: -10 points per bundle
   */
  protected function calculateScores(array $inventory, array $config): array {
    $stats = $this->calculateStats($inventory, $config);
    $factors = [];

    // Unused fields score - penalizes heavily: -10 points per field.
    $fields_unused = $stats['fields_unused'];
    $unused_score = $fields_unused === 0 ? 100 : max(0, 100 - ($fields_unused * 10));
    $factors['unused_fields'] = [
      'score' => $unused_score,
      'weight' => self::SCORE_WEIGHTS['unused_fields'],
      'label' => (string) $this->t('Unused Fields'),
      'description' => $fields_unused === 0
        ? (string) $this->t('All fields are used in displays or forms')
        : (string) $this->t('@count fields possibly unused', ['@count' => $fields_unused]),
    ];

    // Field visibility score - penalizes lightly: -2 points per field.
    $notice_fields = $stats['fields_not_displayed'] + $stats['fields_not_in_form'];
    $visibility_score = $notice_fields === 0 ? 100 : max(70, 100 - ($notice_fields * 2));
    $factors['field_visibility'] = [
      'score' => $visibility_score,
      'weight' => self::SCORE_WEIGHTS['field_visibility'],
      'label' => (string) $this->t('Field Visibility'),
      'description' => $notice_fields === 0
        ? (string) $this->t('All fields visible in displays and forms')
        : (string) $this->t('@count fields with partial visibility', ['@count' => $notice_fields]),
    ];

    // Field count score: critical -25, warning -10.
    $bundles_field_critical = $stats['bundles_field_critical'];
    $bundles_field_warning = $stats['bundles_field_warning'];
    $field_penalty = ($bundles_field_critical * 25) + ($bundles_field_warning * 10);
    $field_score = max(0, 100 - $field_penalty);
    $factors['field_count'] = [
      'score' => $field_score,
      'weight' => self::SCORE_WEIGHTS['field_count'],
      'label' => (string) $this->t('Field Count'),
      'description' => ($bundles_field_critical + $bundles_field_warning) === 0
        ? (string) $this->t('All bundles have reasonable field counts')
        : (string) $this->t('@count bundles have field count issues', ['@count' => $bundles_field_critical + $bundles_field_warning]),
    ];

    // View modes score: critical -25, warning -10.
    $bundles_view_critical = $stats['bundles_view_mode_critical'];
    $bundles_view_warning = $stats['bundles_view_mode_warning'];
    $view_mode_penalty = ($bundles_view_critical * 25) + ($bundles_view_warning * 10);
    $view_modes_score = max(0, 100 - $view_mode_penalty);
    $factors['view_modes'] = [
      'score' => $view_modes_score,
      'weight' => self::SCORE_WEIGHTS['view_modes'],
      'label' => (string) $this->t('View Modes'),
      'description' => ($bundles_view_critical + $bundles_view_warning) === 0
        ? (string) $this->t('All bundles have reasonable view mode counts')
        : (string) $this->t('@count bundles have view mode issues', ['@count' => $bundles_view_critical + $bundles_view_warning]),
    ];

    // Form modes score: critical -25, warning -10.
    $bundles_form_critical = $stats['bundles_form_mode_critical'];
    $bundles_form_warning = $stats['bundles_form_mode_warning'];
    $form_mode_penalty = ($bundles_form_critical * 25) + ($bundles_form_warning * 10);
    $form_modes_score = max(0, 100 - $form_mode_penalty);
    $factors['form_modes'] = [
      'score' => $form_modes_score,
      'weight' => self::SCORE_WEIGHTS['form_modes'],
      'label' => (string) $this->t('Form Modes'),
      'description' => ($bundles_form_critical + $bundles_form_warning) === 0
        ? (string) $this->t('All bundles have reasonable form mode counts')
        : (string) $this->t('@count bundles have form mode issues', ['@count' => $bundles_form_critical + $bundles_form_warning]),
    ];

    return [
      'factors' => $factors,
    ];
  }

}
