<?php

declare(strict_types=1);

namespace Drupal\audit_i18n\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Analyzes multilingual configuration.
 */
#[AuditAnalyzer(
  id: 'i18n',
  label: new TranslatableMarkup('Internationalization Audit'),
  description: new TranslatableMarkup('Analyzes language configuration and translation status.'),
  menu_title: new TranslatableMarkup('i18n'),
  output_directory: 'i18n',
  weight: 2,
)]
class I18nAnalyzer extends AuditAnalyzerBase {

  /**
   * Score weights for different factors.
   */
  protected const SCORE_WEIGHTS = [
    'modules_installed' => 40,
    'translation_coverage' => 40,
    'hreflang_seo' => 20,
  ];

  /**
   * Required modules for multilingual sites (core modules).
   */
  protected const REQUIRED_MODULES = [
    'language' => [
      'label' => 'Language',
      'description' => 'Allows adding and managing languages on the site. This is the foundation module required for any multilingual setup.',
    ],
    'locale' => [
      'label' => 'Interface Translation',
      'description' => 'Translates the built-in user interface, including strings from modules and themes. Essential for displaying the admin and frontend UI in multiple languages.',
    ],
    'content_translation' => [
      'label' => 'Content Translation',
      'description' => 'Allows content entities (nodes, taxonomy terms, blocks, etc.) to be translated into different languages. Required if you need multilingual content.',
    ],
    'config_translation' => [
      'label' => 'Configuration Translation',
      'description' => 'Allows configuration (site name, views, menus, field labels, etc.) to be translated. Required for a fully localized experience.',
    ],
  ];

  /**
   * Recommended modules for multilingual sites (contributed modules).
   */
  protected const RECOMMENDED_MODULES = [
    'drush_language' => [
      'label' => 'Drush Language',
      'description' => 'Provides Drush commands to export and import interface translations between environments (local, development, production). Allows storing translations as code in Git and deploying them consistently across environments.',
    ],
  ];

  /**
   * SEO modules for multilingual sites (hreflang implementation).
   *
   * The hreflang attribute is critical for multilingual SEO as it tells
   * search engines which language version of a page to show users.
   */
  protected const HREFLANG_MODULES = [
    'hreflang' => [
      'label' => 'Hreflang',
      'description' => 'Adds hreflang link elements to pages to improve multilingual SEO. This is the standalone solution that works out of the box.',
      'project_url' => 'https://www.drupal.org/project/hreflang',
    ],
    'metatag_hreflang' => [
      'label' => 'Metatag: Hreflang',
      'description' => 'Provides hreflang meta tags through the Metatag module. Offers more granular control per content type but requires Metatag to be installed.',
      'requires' => 'metatag',
    ],
  ];

  /**
   * Default minimum translation percentage.
   */
  protected const DEFAULT_MIN_TRANSLATION_PERCENTAGE = 80;

  /**
   * Cached module configuration.
   */
  protected ?ImmutableConfig $moduleConfig = NULL;

  protected LanguageManagerInterface $languageManager;
  protected ModuleHandlerInterface $moduleHandler;
  protected Connection $database;
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->languageManager = $container->get('language_manager');
    $instance->moduleHandler = $container->get('module_handler');
    $instance->database = $container->get('database');
    $instance->configFactory = $container->get('config.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $config): array {
    return [
      'min_translation_percentage' => [
        '#type' => 'number',
        '#title' => $this->t('Minimum translation percentage'),
        '#description' => $this->t('Minimum required translation coverage for non-default languages. Recommended: 80% minimum for production, 100% ideal. Languages below this threshold will be flagged. Full translation coverage is easily achievable today using AI-powered automated translation tools.'),
        '#default_value' => $config['min_translation_percentage'] ?? self::DEFAULT_MIN_TRANSLATION_PERCENTAGE,
        '#min' => 0,
        '#max' => 100,
        '#step' => 1,
        '#field_suffix' => '%',
        '#required' => TRUE,
      ],
      'check_multilingual_seo' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Analyze multilingual SEO configuration'),
        '#description' => $this->t('When enabled, analyzes SEO-related configurations for multilingual sites. Disable this if SEO is not a priority for your site.'),
        '#default_value' => $config['check_multilingual_seo'] ?? TRUE,
      ],
    ];
  }

  /**
   * Gets the module configuration.
   */
  protected function getConfig(): ImmutableConfig {
    if ($this->moduleConfig === NULL) {
      $this->moduleConfig = $this->configFactory->get('audit_i18n.settings');
    }
    return $this->moduleConfig;
  }

  /**
   * Gets the configured minimum translation percentage.
   */
  protected function getMinTranslationPercentage(): int {
    return (int) ($this->getConfig()->get('min_translation_percentage') ?? self::DEFAULT_MIN_TRANSLATION_PERCENTAGE);
  }

  /**
   * Checks if multilingual SEO analysis is enabled.
   */
  protected function isMultilingualSeoCheckEnabled(): bool {
    return (bool) ($this->getConfig()->get('check_multilingual_seo') ?? TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    $languages = $this->languageManager->getLanguages();
    $language_count = count($languages);

    // Monolingual site.
    if ($language_count === 1) {
      $results[] = $this->createResultItem(
        'notice',
        'MONOLINGUAL_SITE',
        (string) $this->t('Site is configured with a single language. Multilingual analysis skipped.'),
        ['language_count' => 1]
      );
      $notices++;

      $output = $this->createResult($results, 0, 0, $notices);
      $output['_filename'] = 'translations';
      $output['score'] = $this->calculateMonolingualScores();
      $output['stats'] = ['languages' => 1, 'is_multilingual' => FALSE];
      return $output;
    }

    // Build language data.
    $language_data = $this->buildLanguageData($languages);

    // Check for recommended multilingual modules.
    $module_status = $this->checkModuleStatus($results, $warnings, $notices);

    // Check multilingual SEO configuration (hreflang, etc.).
    $seo_check_enabled = $this->isMultilingualSeoCheckEnabled();
    $hreflang_status = [];
    if ($seo_check_enabled) {
      $hreflang_status = $this->checkHreflangStatus($results, $errors, $warnings, $notices);
    }

    // Calculate translation statistics.
    $translation_stats = [];
    if ($this->moduleHandler->moduleExists('locale')) {
      $translation_stats = $this->getTranslationStats($languages, $results, $errors, $warnings, $notices);
    }

    $stats = [
      'languages' => $language_count,
      'is_multilingual' => TRUE,
      'modules' => $module_status,
      'modules_installed' => $module_status['installed_count'],
      'modules_total' => $module_status['total_count'],
      'required_installed' => $module_status['required_installed'],
      'required_total' => $module_status['required_total'],
      'recommended_installed' => $module_status['recommended_installed'],
      'recommended_total' => $module_status['recommended_total'],
      'hreflang' => $hreflang_status,
      'seo_check_enabled' => $seo_check_enabled,
      'translation_stats' => $translation_stats,
    ];

    $scores = $this->calculateScores($stats, $translation_stats, $hreflang_status);

    $output = $this->createResult($results, $errors, $warnings, $notices);
    $output['_filename'] = 'translations';
    $output['score'] = $scores;
    $output['stats'] = $stats;
    $output['languages'] = $language_data;

    return $output;
  }

  /**
   * Builds language data array.
   *
   * @param \Drupal\Core\Language\LanguageInterface[] $languages
   *   Array of language objects.
   *
   * @return array
   *   Language data keyed by langcode.
   */
  protected function buildLanguageData(array $languages): array {
    $language_data = [];

    foreach ($languages as $langcode => $language) {
      $language_data[$langcode] = [
        'name' => $language->getName(),
        'default' => $language->isDefault(),
        'direction' => $language->getDirection(),
      ];
    }

    return $language_data;
  }

  /**
   * Checks the status of required and recommended multilingual modules.
   *
   * @param array $results
   *   Results array (passed by reference).
   * @param int $warnings
   *   Warning counter (passed by reference).
   * @param int $notices
   *   Notice counter (passed by reference).
   *
   * @return array
   *   Module status data.
   */
  protected function checkModuleStatus(array &$results, int &$warnings, int &$notices): array {
    $required_count = count(self::REQUIRED_MODULES);
    $recommended_count = count(self::RECOMMENDED_MODULES);

    $module_status = [
      'modules' => [],
      'installed_count' => 0,
      'total_count' => $required_count + $recommended_count,
      'required_installed' => 0,
      'required_total' => $required_count,
      'recommended_installed' => 0,
      'recommended_total' => $recommended_count,
    ];

    // Check required modules (warnings if missing).
    foreach (self::REQUIRED_MODULES as $module => $info) {
      $is_installed = $this->moduleHandler->moduleExists($module);

      $module_status['modules'][$module] = [
        'label' => $info['label'],
        'description' => $info['description'],
        'installed' => $is_installed,
        'required' => TRUE,
      ];

      if ($is_installed) {
        $module_status['installed_count']++;
        $module_status['required_installed']++;
      }
      else {
        $results[] = $this->createResultItem(
          'warning',
          'MISSING_REQUIRED_MODULE',
          (string) $this->t('Required multilingual module not installed: @module', [
            '@module' => $info['label'],
          ]),
          [
            'module' => $module,
            'label' => $info['label'],
            'required' => TRUE,
          ]
        );
        $warnings++;
      }
    }

    // Check recommended modules (notices if missing).
    foreach (self::RECOMMENDED_MODULES as $module => $info) {
      $is_installed = $this->moduleHandler->moduleExists($module);

      $module_status['modules'][$module] = [
        'label' => $info['label'],
        'description' => $info['description'],
        'installed' => $is_installed,
        'required' => FALSE,
      ];

      if ($is_installed) {
        $module_status['installed_count']++;
        $module_status['recommended_installed']++;
      }
      else {
        $results[] = $this->createResultItem(
          'notice',
          'MISSING_RECOMMENDED_MODULE',
          (string) $this->t('Recommended multilingual module not installed: @module', [
            '@module' => $info['label'],
          ]),
          [
            'module' => $module,
            'label' => $info['label'],
            'required' => FALSE,
          ]
        );
        $notices++;
      }
    }

    return $module_status;
  }

  /**
   * Checks the hreflang implementation status.
   *
   * Hreflang is critical for multilingual SEO. This method checks if either
   * the standalone hreflang module or the metatag_hreflang submodule is active.
   *
   * Logic:
   * - Neither hreflang nor metatag installed: ERROR (critical SEO issue)
   * - Metatag installed but metatag_hreflang not enabled: WARNING
   * - metatag_hreflang enabled: NOTICE (valid configuration)
   * - hreflang module enabled: OK (no message needed, or notice)
   *
   * @param array $results
   *   Results array (passed by reference).
   * @param int $errors
   *   Error counter (passed by reference).
   * @param int $warnings
   *   Warning counter (passed by reference).
   * @param int $notices
   *   Notice counter (passed by reference).
   *
   * @return array
   *   Hreflang status data.
   */
  protected function checkHreflangStatus(array &$results, int &$errors, int &$warnings, int &$notices): array {
    $has_hreflang = $this->moduleHandler->moduleExists('hreflang');
    $has_metatag = $this->moduleHandler->moduleExists('metatag');
    $has_metatag_hreflang = $this->moduleHandler->moduleExists('metatag_hreflang');

    $status = [
      'hreflang_installed' => $has_hreflang,
      'metatag_installed' => $has_metatag,
      'metatag_hreflang_installed' => $has_metatag_hreflang,
      'has_hreflang_support' => $has_hreflang || $has_metatag_hreflang,
      'implementation' => 'none',
    ];

    // Determine the implementation type and create appropriate result.
    if ($has_hreflang) {
      // Best case: standalone hreflang module is active.
      $status['implementation'] = 'hreflang';
      $results[] = $this->createResultItem(
        'notice',
        'HREFLANG_OK',
        (string) $this->t('Hreflang module is installed. Multilingual SEO is properly configured.'),
        [
          'module' => 'hreflang',
          'implementation' => 'standalone',
        ]
      );
      $notices++;
    }
    elseif ($has_metatag_hreflang) {
      // Good: metatag_hreflang submodule is active.
      $status['implementation'] = 'metatag_hreflang';
      $results[] = $this->createResultItem(
        'notice',
        'HREFLANG_METATAG',
        (string) $this->t('Metatag: Hreflang submodule is installed. Ensure hreflang tags are configured for each content type.'),
        [
          'module' => 'metatag_hreflang',
          'implementation' => 'metatag',
        ]
      );
      $notices++;
    }
    elseif ($has_metatag) {
      // Warning: Metatag is installed but hreflang submodule is not enabled.
      $status['implementation'] = 'metatag_only';
      $results[] = $this->createResultItem(
        'warning',
        'HREFLANG_METATAG_MISSING',
        (string) $this->t('Metatag module is installed but Metatag: Hreflang submodule is not enabled. Enable it to add hreflang tags for better multilingual SEO.'),
        [
          'metatag_installed' => TRUE,
          'metatag_hreflang_installed' => FALSE,
          'recommendation' => 'Enable metatag_hreflang submodule',
        ]
      );
      $warnings++;
    }
    else {
      // Error: No hreflang implementation at all.
      $status['implementation'] = 'none';
      $results[] = $this->createResultItem(
        'error',
        'HREFLANG_MISSING',
        (string) $this->t('No hreflang implementation found. This is critical for multilingual SEO. Install either the Hreflang module or enable Metatag with Metatag: Hreflang submodule.'),
        [
          'hreflang_installed' => FALSE,
          'metatag_installed' => FALSE,
          'recommendation' => 'Install hreflang module (https://www.drupal.org/project/hreflang) or metatag with metatag_hreflang',
        ]
      );
      $errors++;
    }

    return $status;
  }

  /**
   * Gets translation statistics for all non-default languages.
   *
   * @param \Drupal\Core\Language\LanguageInterface[] $languages
   *   Array of language objects.
   * @param array $results
   *   Results array (passed by reference).
   * @param int $errors
   *   Error counter (passed by reference).
   * @param int $warnings
   *   Warning counter (passed by reference).
   * @param int $notices
   *   Notice counter (passed by reference).
   *
   * @return array
   *   Translation statistics per language.
   */
  protected function getTranslationStats(array $languages, array &$results, int &$errors, int &$warnings, int &$notices): array {
    $stats = [];

    // Get total translatable strings count once.
    $total_strings = $this->getTotalTranslatableStrings();

    if ($total_strings === 0) {
      return $stats;
    }

    foreach ($languages as $langcode => $language) {
      if ($language->isDefault()) {
        continue;
      }

      $translated = $this->getTranslatedStringsCount($langcode);
      $pending = $total_strings - $translated;
      $percentage = round(($translated / $total_strings) * 100, 1);

      $stats[$langcode] = [
        'total' => $total_strings,
        'translated' => $translated,
        'pending' => $pending,
        'percentage' => $percentage,
      ];

      $this->addTranslationResult($language, $percentage, $translated, $pending, $total_strings, $results, $warnings, $notices, $errors);
    }

    return $stats;
  }

  /**
   * Gets total count of translatable strings.
   */
  protected function getTotalTranslatableStrings(): int {
    try {
      return (int) $this->database->select('locales_source', 'ls')
        ->countQuery()
        ->execute()
        ->fetchField();
    }
    catch (\Exception $e) {
      return 0;
    }
  }

  /**
   * Gets count of translated strings for a language.
   */
  protected function getTranslatedStringsCount(string $langcode): int {
    try {
      return (int) $this->database->select('locales_target', 'lt')
        ->condition('lt.language', $langcode)
        ->countQuery()
        ->execute()
        ->fetchField();
    }
    catch (\Exception $e) {
      return 0;
    }
  }

  /**
   * Adds a translation result item.
   */
  protected function addTranslationResult(
    LanguageInterface $language,
    float $percentage,
    int $translated,
    int $pending,
    int $total,
    array &$results,
    int &$warnings,
    int &$notices,
    int &$errors,
  ): void {
    $min_percentage = $this->getMinTranslationPercentage();
    $warning_threshold = $min_percentage - 10;

    if ($percentage >= $min_percentage) {
      // At or above minimum: OK (notice).
      $severity = 'notice';
      $notices++;
    }
    elseif ($percentage >= $warning_threshold) {
      // Below minimum but within 10%: warning.
      $severity = 'warning';
      $warnings++;
    }
    else {
      // More than 10% below minimum: error.
      $severity = 'error';
      $errors++;
    }

    $results[] = $this->createResultItem(
      $severity,
      'TRANSLATION_COVERAGE',
      (string) $this->t('@language: @percentage% translated (@translated/@total strings, @pending pending)', [
        '@language' => $language->getName(),
        '@percentage' => $percentage,
        '@translated' => number_format($translated),
        '@total' => number_format($total),
        '@pending' => number_format($pending),
      ]),
      [
        'langcode' => $language->getId(),
        'language' => $language->getName(),
        'total_strings' => $total,
        'translated_strings' => $translated,
        'pending_strings' => $pending,
        'percentage' => $percentage,
      ]
    );
  }

  /**
   * Calculates scores for monolingual sites.
   */
  protected function calculateMonolingualScores(): array {
    $factors = [
      'modules_installed' => [
        'score' => 100,
        'weight' => self::SCORE_WEIGHTS['modules_installed'],
        'label' => (string) $this->t('Multilingual Modules'),
        'description' => (string) $this->t('Not applicable for monolingual sites'),
      ],
      'translation_coverage' => [
        'score' => 100,
        'weight' => self::SCORE_WEIGHTS['translation_coverage'],
        'label' => (string) $this->t('Translation Coverage'),
        'description' => (string) $this->t('Not applicable for monolingual sites'),
      ],
    ];

    // Include SEO factor only if check is enabled.
    if ($this->isMultilingualSeoCheckEnabled()) {
      $factors['hreflang_seo'] = [
        'score' => 100,
        'weight' => self::SCORE_WEIGHTS['hreflang_seo'],
        'label' => (string) $this->t('Hreflang SEO'),
        'description' => (string) $this->t('Not applicable for monolingual sites'),
      ];
    }

    return [
      'factors' => $factors,
    ];
  }

  /**
   * Calculates scores for all factors.
   *
   * @param array $stats
   *   General statistics.
   * @param array $translation_stats
   *   Translation statistics per language.
   * @param array $hreflang_status
   *   Hreflang implementation status.
   *
   * @return array
   *   Score data with overall score and individual factors.
   */
  protected function calculateScores(array $stats, array $translation_stats, array $hreflang_status = []): array {
    $factors = [];
    $seo_check_enabled = $this->isMultilingualSeoCheckEnabled();

    // Modules installed score.
    // Only required modules affect the score (25 points penalty each).
    // Recommended modules do not penalize the score.
    $missing_required = $stats['required_total'] - $stats['required_installed'];
    $modules_penalty = $missing_required * 25;
    $modules_score = max(0, 100 - $modules_penalty);

    $factors['modules_installed'] = [
      'score' => $modules_score,
      'weight' => self::SCORE_WEIGHTS['modules_installed'],
      'label' => (string) $this->t('Multilingual Modules'),
      'description' => $this->getModulesDescription($stats),
    ];

    // Translation coverage score.
    // Languages at or above the minimum configured percentage count as 100%.
    // Languages below count with their actual percentage.
    $min_percentage = $this->getMinTranslationPercentage();
    $avg_translation = 0;
    $effective_scores = [];

    if (!empty($translation_stats)) {
      foreach ($translation_stats as $lang_stats) {
        $percentage = $lang_stats['percentage'];
        // If at or above minimum, count as 100%.
        $effective_scores[] = $percentage >= $min_percentage ? 100 : $percentage;
      }
      $avg_translation = array_sum($effective_scores) / count($effective_scores);
    }

    $translation_score = (int) round($avg_translation);
    $actual_avg = !empty($translation_stats)
      ? array_sum(array_column($translation_stats, 'percentage')) / count($translation_stats)
      : 0;

    $factors['translation_coverage'] = [
      'score' => $translation_score,
      'weight' => self::SCORE_WEIGHTS['translation_coverage'],
      'label' => (string) $this->t('Translation Coverage'),
      'description' => (string) $this->t('Average @percent% translation coverage (minimum required: @min%)', [
        '@percent' => round($actual_avg, 1),
        '@min' => $min_percentage,
      ]),
    ];

    // Multilingual SEO score (hreflang implementation).
    // Score based on implementation status:
    // - hreflang or metatag_hreflang installed: 100 points
    // - metatag installed but no hreflang submodule: 50 points (warning)
    // - nothing installed: 0 points (error)
    if ($seo_check_enabled) {
      $implementation = $hreflang_status['implementation'] ?? 'none';

      switch ($implementation) {
        case 'hreflang':
          $hreflang_score = 100;
          $hreflang_description = (string) $this->t('Hreflang module installed - SEO properly configured');
          break;

        case 'metatag_hreflang':
          $hreflang_score = 100;
          $hreflang_description = (string) $this->t('Metatag: Hreflang installed - SEO properly configured');
          break;

        case 'metatag_only':
          $hreflang_score = 50;
          $hreflang_description = (string) $this->t('Metatag installed but hreflang submodule missing');
          break;

        default:
          $hreflang_score = 0;
          $hreflang_description = (string) $this->t('No hreflang implementation - critical SEO issue');
      }

      $factors['hreflang_seo'] = [
        'score' => $hreflang_score,
        'weight' => self::SCORE_WEIGHTS['hreflang_seo'],
        'label' => (string) $this->t('Hreflang SEO'),
        'description' => $hreflang_description,
      ];
    }

    return [
      'factors' => $factors,
    ];
  }

  /**
   * Gets the modules description for the score factor.
   */
  protected function getModulesDescription(array $stats): string {
    $missing_required = $stats['required_total'] - $stats['required_installed'];

    if ($missing_required > 0) {
      return (string) $this->t('@count of @total required modules missing', [
        '@count' => $missing_required,
        '@total' => $stats['required_total'],
      ]);
    }

    return (string) $this->t('All @count required modules installed', [
      '@count' => $stats['required_total'],
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    $checks = [
      'languages' => [
        'label' => $this->t('Configured Languages'),
        'description' => $this->t('Lists all configured languages on the site.'),
        'file_key' => 'translations',
        'affects_score' => FALSE,
        'score_factor_key' => NULL,
      ],
      'modules' => [
        'label' => $this->t('Multilingual Modules'),
        'description' => $this->t('Checks for required and recommended multilingual modules. Required modules (Language, Interface Translation, Content Translation, Configuration Translation) are essential for proper multilingual support.'),
        'file_key' => 'translations',
        'affects_score' => TRUE,
        'score_factor_key' => 'modules_installed',
      ],
      'translations' => [
        'label' => $this->t('Translation Coverage'),
        'description' => $this->t('Analyzes translation coverage for each language, based on interface translation status.'),
        'file_key' => 'translations',
        'affects_score' => TRUE,
        'score_factor_key' => 'translation_coverage',
      ],
    ];

    // Add hreflang check only if SEO check is enabled.
    if ($this->isMultilingualSeoCheckEnabled()) {
      $checks['hreflang'] = [
        'label' => $this->t('Hreflang SEO'),
        'description' => $this->t('Checks for hreflang implementation which is critical for multilingual SEO. Ensures search engines show the correct language version to users.'),
        'file_key' => 'translations',
        'affects_score' => TRUE,
        'score_factor_key' => 'hreflang_seo',
      ];
    }

    return $checks;
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    $stats = $data['stats'] ?? [];

    // Handle monolingual sites.
    if (!($stats['is_multilingual'] ?? FALSE)) {
      return [
        'monolingual' => $this->ui->message(
          (string) $this->t('This is a monolingual site. No multilingual analysis available.'),
          'info'
        ),
      ];
    }

    return match ($check_id) {
      'languages' => $this->buildLanguagesContent($data),
      'modules' => $this->buildModulesContent($stats),
      'hreflang' => $this->buildHreflangContent($stats['hreflang'] ?? []),
      'translations' => $this->buildTranslationsContent($data, $stats),
      default => [],
    };
  }

  /**
   * Builds the languages content (without section wrapper).
   */
  protected function buildLanguagesContent(array $data): array {
    if (empty($data['languages'])) {
      return ['empty' => $this->ui->message((string) $this->t('No languages configured.'), 'info')];
    }

    $headers = [
      (string) $this->t('Language'),
      (string) $this->t('Code'),
      (string) $this->t('Status'),
    ];

    $rows = [];
    foreach ($data['languages'] as $langcode => $lang_data) {
      $status = $lang_data['default']
        ? $this->ui->icon('check') . ' ' . (string) $this->t('Default')
        : '';

      $rows[] = [
        $lang_data['name'],
        $langcode,
        $status,
      ];
    }

    $table = $this->ui->table($headers, $rows, [
      'empty' => (string) $this->t('No languages configured.'),
    ]);

    return ['table' => $table];
  }

  /**
   * Builds the modules content (without section wrapper).
   */
  protected function buildModulesContent(array $stats): array {
    $modules = $stats['modules']['modules'] ?? [];

    if (empty($modules)) {
      return ['empty' => $this->ui->message((string) $this->t('No module information available.'), 'info')];
    }

    $headers = [
      $this->ui->header('', 'center', '40px'),
      $this->ui->header('', 'center'),
      (string) $this->t('Module'),
      (string) $this->t('Description'),
    ];

    $rows = [];
    foreach ($modules as $module => $info) {
      // Determine icon based on installed status.
      $icon = $info['installed'] ? $this->ui->icon('check') : $this->ui->icon('cross');

      // Determine badge variant based on installed status and requirement level.
      if ($info['required']) {
        $badge_variant = $info['installed'] ? 'success' : 'warning';
        $badge_label = (string) $this->t('Required');
      }
      else {
        $badge_variant = $info['installed'] ? 'success' : 'info';
        $badge_label = (string) $this->t('Recommended');
      }
      $badge = $this->ui->badge($badge_label, $badge_variant);

      // Determine row severity based on installed status and requirement level.
      $severity = NULL;
      if (!$info['installed']) {
        $severity = $info['required'] ? 'warning' : 'notice';
      }

      $rows[] = $this->ui->row([
        $this->ui->cell($icon, ['align' => 'center']),
        $this->ui->cell($badge, ['align' => 'center']),
        $this->ui->itemName($info['label'], $module),
        $info['description'],
      ], $severity);
    }

    $table = $this->ui->table($headers, $rows, [
      'empty' => (string) $this->t('No module information available.'),
    ]);

    return ['table' => $table];
  }

  /**
   * Builds the hreflang content (without section wrapper).
   *
   * @param array $hreflang_status
   *   The hreflang status data.
   *
   * @return array
   *   A render array for the hreflang content.
   */
  protected function buildHreflangContent(array $hreflang_status): array {
    $implementation = $hreflang_status['implementation'] ?? 'none';
    $content = [];

    // Determine status message and type based on implementation.
    switch ($implementation) {
      case 'hreflang':
        $message_type = 'success';
        $status_text = (string) $this->t('Hreflang module is installed and active. Your site is using the standalone Hreflang module which automatically adds hreflang link elements to all pages. This is the recommended setup for multilingual SEO.');
        break;

      case 'metatag_hreflang':
        $message_type = 'success';
        $status_text = (string) $this->t('Metatag: Hreflang submodule is installed and active. Make sure hreflang tags are properly configured in your Metatag settings for each content type.');
        break;

      case 'metatag_only':
        $message_type = 'warning';
        $status_text = (string) $this->t('Metatag is installed but Hreflang submodule is missing. Enable it with: drush en metatag_hreflang');
        break;

      default:
        $message_type = 'error';
        $status_text = (string) $this->t('No hreflang implementation found. Your multilingual site is missing hreflang tags, which is critical for SEO. Search engines may show the wrong language version to users. Install the Hreflang module (https://www.drupal.org/project/hreflang) or enable Metatag with the Metatag: Hreflang submodule.');
    }

    $content['status'] = $this->ui->message($status_text, $message_type);

    // Build module status table.
    $headers = [
      $this->ui->header('', 'center', '40px'),
      (string) $this->t('Module'),
      (string) $this->t('Description'),
    ];

    $rows = [];

    // Hreflang module row.
    $hreflang_installed = $hreflang_status['hreflang_installed'] ?? FALSE;
    $rows[] = $this->ui->row([
      $this->ui->cell(
        $hreflang_installed ? $this->ui->icon('check') : $this->ui->icon('cross'),
        ['align' => 'center']
      ),
      $this->ui->itemName('Hreflang', 'hreflang'),
      self::HREFLANG_MODULES['hreflang']['description'],
    ], $hreflang_installed ? NULL : 'warning');

    // Metatag: Hreflang row.
    $metatag_hreflang_installed = $hreflang_status['metatag_hreflang_installed'] ?? FALSE;
    $rows[] = $this->ui->row([
      $this->ui->cell(
        $metatag_hreflang_installed ? $this->ui->icon('check') : $this->ui->icon('cross'),
        ['align' => 'center']
      ),
      $this->ui->itemName('Metatag: Hreflang', 'metatag_hreflang'),
      self::HREFLANG_MODULES['metatag_hreflang']['description'],
    ], $metatag_hreflang_installed ? NULL : 'notice');

    $content['table'] = $this->ui->table($headers, $rows);

    // Add note about why hreflang is important.
    $content['note'] = $this->ui->message(
      (string) $this->t('The hreflang attribute tells search engines which language version of a page to show users based on their location and language preferences. Without it, search engines may index and display the wrong language version.'),
      'info'
    );

    return $content;
  }

  /**
   * Builds the translations content (without section wrapper).
   */
  protected function buildTranslationsContent(array $data, array $stats): array {
    if (empty($stats['translation_stats'])) {
      return [];
    }

    $min_percentage = $this->getMinTranslationPercentage();
    $warning_threshold = $min_percentage - 10;
    $content = [];

    $headers = [
      (string) $this->t('Language'),
      $this->ui->header((string) $this->t('Coverage'), 'center'),
      $this->ui->header((string) $this->t('Translated'), 'right'),
      $this->ui->header((string) $this->t('Pending'), 'right'),
      $this->ui->header((string) $this->t('Total'), 'right'),
    ];

    $rows = [];
    foreach ($stats['translation_stats'] as $langcode => $trans_stats) {
      $percentage = $trans_stats['percentage'];

      // Determine status and row severity based on thresholds.
      if ($percentage >= $min_percentage) {
        // At or above minimum: OK (green).
        $percentage_status = 'ok';
        $row_severity = NULL;
      }
      elseif ($percentage >= $warning_threshold) {
        // Below minimum but within 10%: warning (orange).
        $percentage_status = 'warning';
        $row_severity = 'warning';
      }
      else {
        // More than 10% below minimum: error (red).
        $percentage_status = 'error';
        $row_severity = 'error';
      }

      $language_name = $data['languages'][$langcode]['name'] ?? $langcode;

      // Determine pending status.
      $pending_status = $trans_stats['pending'] > 0 ? 'warning' : NULL;

      $rows[] = $this->ui->row([
        $language_name,
        $this->ui->cell($percentage . '%', ['align' => 'center', 'status' => $percentage_status]),
        $this->ui->cell(number_format($trans_stats['translated']), ['align' => 'right']),
        $this->ui->cell(number_format($trans_stats['pending']), ['align' => 'right', 'status' => $pending_status]),
        $this->ui->cell(number_format($trans_stats['total']), ['align' => 'right']),
      ], $row_severity);
    }

    $content['table'] = $this->ui->table($headers, $rows, [
      'empty' => (string) $this->t('No translation statistics available.'),
    ]);

    // Add note about minimum recommended percentage.
    $content['note'] = $this->ui->message(
      (string) $this->t('Configured minimum translation coverage: @percent%. Languages below @warning% are marked as errors.', [
        '@percent' => $min_percentage,
        '@warning' => $warning_threshold,
      ]),
      'info'
    );

    return $content;
  }

}
