<?php

declare(strict_types=1);

namespace Drupal\audit_images\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\field\FieldConfigInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Analyzes image display configuration.
 */
#[AuditAnalyzer(
  id: 'images',
  label: new TranslatableMarkup('Images Audit'),
  description: new TranslatableMarkup('Analyzes image display configuration including image styles, responsive styles, breakpoints, and field formatters.'),
  menu_title: new TranslatableMarkup('Images'),
  output_directory: 'images',
  weight: 3,
)]
class ImagesAnalyzer extends AuditAnalyzerBase {

  /**
   * Score weights for image formatters.
   *
   * If responsive_image module is not installed, score is 0.
   * Otherwise, score is 100% based on image field configuration.
   */
  protected const FORMATTER_SCORES = [
    'responsive' => 100,
    'styled' => 60,
    'no_style' => 0,
  ];

  /**
   * Essential modules for responsive images.
   *
   * Only responsive_image is checked since image and breakpoint
   * are required dependencies that will be installed automatically.
   */
  protected const ESSENTIAL_MODULES = [
    'responsive_image' => [
      'label' => 'Responsive Image',
      'description' => 'Provides responsive image display formatters for optimal performance across devices',
    ],
  ];

  /**
   * Recommended modules for image optimization.
   */
  protected const RECOMMENDED_MODULES = [
    'imageapi_optimize' => [
      'label' => 'ImageAPI Optimize',
      'description' => 'Optimizes images to reduce file size without quality loss using various optimization tools',
    ],
    'webp' => [
      'label' => 'WebP',
      'description' => 'Provides WebP image format support for significantly smaller file sizes with equal quality',
    ],
  ];

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity field manager.
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The entity type bundle info service.
   */
  protected EntityTypeBundleInfoInterface $entityTypeBundleInfo;

  /**
   * The entity display repository.
   */
  protected EntityDisplayRepositoryInterface $entityDisplayRepository;

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->entityTypeBundleInfo = $container->get('entity_type.bundle.info');
    $instance->entityDisplayRepository = $container->get('entity_display.repository');
    $instance->moduleHandler = $container->get('module_handler');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    $modules_analysis = $this->analyzeModules();
    $image_styles = $this->analyzeImageStyles();
    $responsive_styles = $this->analyzeResponsiveStyles();
    $image_fields = $this->analyzeImageFields();

    // Analyze unused styles (requires image_fields data for used styles).
    $unused_styles = $this->analyzeUnusedStyles($image_styles, $responsive_styles, $image_fields);

    // Analyze lazy loading configuration.
    $lazy_loading = $this->analyzeLazyLoading($image_fields);

    $scores = $this->calculateScores($modules_analysis, $image_fields, $unused_styles, $lazy_loading);

    // Build separate files for each section with their own results.
    $files = [
      'modules' => $this->buildModulesFile($modules_analysis),
      'image_styles' => $this->buildImageStylesFile($image_styles),
      'responsive_styles' => $this->buildResponsiveStylesFile($responsive_styles),
      'unused_styles' => $this->buildUnusedStylesFile($unused_styles),
      'lazy_loading' => $this->buildLazyLoadingFile($lazy_loading),
      'image_fields' => $this->buildImageFieldsFile($image_fields),
      'image_fields_issues' => $this->buildImageFieldsIssuesFile($image_fields),
    ];

    return [
      '_files' => $files,
      'score' => $scores,
      'stats' => [
        'modules' => $modules_analysis['stats'],
        'image_styles' => $image_styles['stats'],
        'responsive_styles' => $responsive_styles['stats'],
        'image_fields' => $image_fields['stats'],
        'unused_styles' => $unused_styles['stats'],
        'lazy_loading' => $lazy_loading['stats'],
      ],
      'data' => [
        'modules' => $modules_analysis,
        'image_styles' => $image_styles,
        'responsive_styles' => $responsive_styles,
        'image_fields' => $image_fields,
        'unused_styles' => $unused_styles,
        'lazy_loading' => $lazy_loading,
      ],
    ];
  }

  /**
   * Builds the modules file with results.
   */
  protected function buildModulesFile(array $modules_analysis): array {
    return $this->createResult(
      $modules_analysis['results'],
      $this->countBySeverity($modules_analysis['results'], 'error'),
      $this->countBySeverity($modules_analysis['results'], 'warning'),
      $this->countBySeverity($modules_analysis['results'], 'notice')
    );
  }

  /**
   * Builds the image styles file (informational, no results).
   */
  protected function buildImageStylesFile(array $image_styles): array {
    return $this->createResult([], 0, 0, 0);
  }

  /**
   * Builds the responsive styles file (informational, no results).
   */
  protected function buildResponsiveStylesFile(array $responsive_styles): array {
    return $this->createResult([], 0, 0, 0);
  }

  /**
   * Builds the unused styles file with results.
   */
  protected function buildUnusedStylesFile(array $unused_styles): array {
    return $this->createResult(
      $unused_styles['results'],
      $this->countBySeverity($unused_styles['results'], 'error'),
      $this->countBySeverity($unused_styles['results'], 'warning'),
      $this->countBySeverity($unused_styles['results'], 'notice')
    );
  }

  /**
   * Builds the lazy loading file (informational, no results).
   */
  protected function buildLazyLoadingFile(array $lazy_loading): array {
    return $this->createResult([], 0, 0, 0);
  }

  /**
   * Builds the image fields file with results.
   */
  protected function buildImageFieldsFile(array $image_fields): array {
    return $this->createResult(
      $image_fields['results'],
      $this->countBySeverity($image_fields['results'], 'error'),
      $this->countBySeverity($image_fields['results'], 'warning'),
      $this->countBySeverity($image_fields['results'], 'notice')
    );
  }

  /**
   * Builds the image fields issues file (same results for faceted view).
   */
  protected function buildImageFieldsIssuesFile(array $image_fields): array {
    // Filter only the issue-type results (errors and warnings).
    $issues = array_filter($image_fields['results'], function ($result) {
      $code = $result['code'] ?? '';
      return in_array($code, ['IMAGE_WITHOUT_STYLE', 'IMAGE_NOT_RESPONSIVE'], TRUE);
    });

    return $this->createResult(
      array_values($issues),
      $this->countBySeverity($issues, 'error'),
      $this->countBySeverity($issues, 'warning'),
      $this->countBySeverity($issues, 'notice')
    );
  }

  /**
   * Counts results by severity.
   */
  protected function countBySeverity(array $results, string $severity): int {
    return count(array_filter($results, fn($r) => ($r['severity'] ?? '') === $severity));
  }

  /**
   * Analyzes essential and recommended modules.
   *
   * @return array
   *   Analysis results with modules status.
   */
  protected function analyzeModules(): array {
    $results = [];
    $essential_installed = [];
    $essential_missing = [];
    $recommended_installed = [];
    $recommended_missing = [];

    // Check essential modules (errors if missing).
    foreach (self::ESSENTIAL_MODULES as $module => $info) {
      if ($this->moduleHandler->moduleExists($module)) {
        $essential_installed[$module] = $info;
      }
      else {
        $essential_missing[$module] = $info;
        $results[] = $this->createResultItem(
          'error',
          'MISSING_ESSENTIAL_MODULE',
          (string) $this->t('Essential module "@module" is not installed. @description.', [
            '@module' => $info['label'],
            '@description' => $info['description'],
          ]),
          [
            'module' => $module,
            'label' => $info['label'],
            'required' => TRUE,
          ]
        );
      }
    }

    // Check recommended modules (notices if missing).
    foreach (self::RECOMMENDED_MODULES as $module => $info) {
      if ($this->moduleHandler->moduleExists($module)) {
        $recommended_installed[$module] = $info;
      }
      else {
        $recommended_missing[$module] = $info;
        $results[] = $this->createResultItem(
          'notice',
          'MISSING_RECOMMENDED_MODULE',
          (string) $this->t('Recommended module "@module" is not installed. @description.', [
            '@module' => $info['label'],
            '@description' => $info['description'],
          ]),
          [
            'module' => $module,
            'label' => $info['label'],
            'required' => FALSE,
          ]
        );
      }
    }

    return [
      'results' => $results,
      'essential_installed' => $essential_installed,
      'essential_missing' => $essential_missing,
      'recommended_installed' => $recommended_installed,
      'recommended_missing' => $recommended_missing,
      'stats' => [
        'essential_installed' => count($essential_installed),
        'essential_missing' => count($essential_missing),
        'essential_total' => count(self::ESSENTIAL_MODULES),
        'recommended_installed' => count($recommended_installed),
        'recommended_missing' => count($recommended_missing),
        'recommended_total' => count(self::RECOMMENDED_MODULES),
        // Legacy fields for compatibility.
        'installed_count' => count($essential_installed),
        'missing_count' => count($essential_missing),
        'total' => count(self::ESSENTIAL_MODULES),
      ],
    ];
  }

  /**
   * Analyzes image styles.
   *
   * @return array
   *   Analysis results with image styles.
   */
  protected function analyzeImageStyles(): array {
    $styles = [];

    try {
      $storage = $this->entityTypeManager->getStorage('image_style');
      $entities = $storage->loadMultiple();

      foreach ($entities as $style) {
        $effects = [];
        foreach ($style->getEffects() as $effect) {
          $effects[] = [
            'id' => $effect->getPluginId(),
            'label' => (string) $effect->label(),
            'configuration' => $effect->getConfiguration(),
          ];
        }

        $styles[$style->id()] = [
          'id' => $style->id(),
          'label' => $style->label(),
          'effects' => $effects,
          'effects_count' => count($effects),
        ];
      }
    }
    catch (\Exception $e) {
      // Image styles not available.
    }

    return [
      'styles' => $styles,
      'stats' => [
        'total' => count($styles),
      ],
    ];
  }

  /**
   * Analyzes responsive image styles.
   *
   * @return array
   *   Analysis results with responsive styles.
   */
  protected function analyzeResponsiveStyles(): array {
    $styles = [];

    if (!$this->moduleHandler->moduleExists('responsive_image')) {
      return [
        'styles' => [],
        'stats' => [
          'total' => 0,
          'module_missing' => TRUE,
        ],
      ];
    }

    try {
      $storage = $this->entityTypeManager->getStorage('responsive_image_style');
      $entities = $storage->loadMultiple();

      foreach ($entities as $style) {
        $mappings = [];
        $image_styles_used = [];

        foreach ($style->getImageStyleMappings() as $mapping) {
          $mapping_type = $mapping['image_mapping_type'] ?? '';
          $mapping_data = $mapping['image_mapping'] ?? [];
          $breakpoint_id = $mapping['breakpoint_id'] ?? '';
          $multiplier = $mapping['multiplier'] ?? '1x';

          // Get a simplified breakpoint name.
          $breakpoint_name = $this->getSimplifiedBreakpointName($breakpoint_id, $multiplier);

          // Extract image styles based on mapping type.
          if ($mapping_type === 'image_style' && !empty($mapping_data)) {
            $image_styles_used[$mapping_data] = $mapping_data;
            $mappings[] = [
              'breakpoint' => $breakpoint_name,
              'image_style' => $mapping_data,
            ];
          }
          elseif ($mapping_type === 'sizes' && !empty($mapping_data['sizes_image_styles'])) {
            foreach ($mapping_data['sizes_image_styles'] as $img_style) {
              $image_styles_used[$img_style] = $img_style;
            }
            $mappings[] = [
              'breakpoint' => $breakpoint_name,
              'image_style' => implode(', ', $mapping_data['sizes_image_styles']),
            ];
          }
        }

        // Add fallback style to the list.
        $fallback = $style->getFallbackImageStyle();
        if (!empty($fallback)) {
          $image_styles_used[$fallback] = $fallback;
        }

        $styles[$style->id()] = [
          'id' => $style->id(),
          'label' => $style->label(),
          'breakpoint_group' => $style->getBreakpointGroup(),
          'fallback_image_style' => $fallback,
          'mappings' => $mappings,
          'image_styles_used' => array_values($image_styles_used),
          'image_styles_count' => count($image_styles_used),
        ];
      }
    }
    catch (\Exception $e) {
      // Responsive image styles not available.
    }

    return [
      'styles' => $styles,
      'stats' => [
        'total' => count($styles),
      ],
    ];
  }

  /**
   * Gets a simplified breakpoint name from the full breakpoint ID.
   *
   * @param string $breakpoint_id
   *   The full breakpoint ID (e.g., "theme_name.breakpoint_name").
   * @param string $multiplier
   *   The multiplier (e.g., "1x", "2x").
   *
   * @return string
   *   A simplified name for display.
   */
  protected function getSimplifiedBreakpointName(string $breakpoint_id, string $multiplier): string {
    // Extract the last part of the breakpoint ID (after the last dot).
    $parts = explode('.', $breakpoint_id);
    $name = end($parts);

    // Clean up common prefixes/suffixes.
    $name = str_replace(['_', '-'], ' ', $name);
    $name = ucfirst($name);

    // Add multiplier if not 1x.
    if ($multiplier !== '1x') {
      $name .= " ({$multiplier})";
    }

    return $name;
  }

  /**
   * Gets all content entity types that can have fields.
   *
   * @return array
   *   Array of entity type IDs.
   */
  protected function getContentEntityTypes(): array {
    $entity_types = [];

    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
      // Only content entities with bundles can have configurable fields.
      if ($entity_type->getGroup() === 'content' && $entity_type->get('field_ui_base_route')) {
        $entity_types[] = $entity_type_id;
      }
    }

    return $entity_types;
  }

  /**
   * Analyzes image fields across all view modes.
   *
   * @return array
   *   Analysis results with image field formatters.
   */
  protected function analyzeImageFields(): array {
    $results = [];
    $fields_data = [];
    $stats = [
      'fields_with_responsive' => 0,
      'fields_with_image_style' => 0,
      'fields_without_style' => 0,
      'fields_with_lazy_loading' => 0,
      'fields_without_lazy_loading' => 0,
    ];
    // Track used styles for unused detection.
    $used_image_styles = [];
    $used_responsive_styles = [];

    // Get all content entity types dynamically.
    $entity_types = $this->getContentEntityTypes();

    foreach ($entity_types as $entity_type_id) {
      try {
        $bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);
      }
      catch (\Exception $e) {
        continue;
      }

      // Get entity type label.
      $entity_type_definition = $this->entityTypeManager->getDefinition($entity_type_id);
      $entity_type_label = $entity_type_definition ? (string) $entity_type_definition->getLabel() : $entity_type_id;

      foreach ($bundles as $bundle_id => $bundle_info) {
        $fields = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle_id);
        $image_fields = [];

        // Find image fields.
        foreach ($fields as $field_name => $field_definition) {
          if (!$field_definition instanceof FieldConfigInterface) {
            continue;
          }

          if ($field_definition->getType() === 'image') {
            $image_fields[$field_name] = [
              'label' => (string) $field_definition->getLabel(),
            ];
          }
        }

        if (empty($image_fields)) {
          continue;
        }

        // Analyze view modes for this bundle.
        $view_modes = $this->entityDisplayRepository->getViewModeOptions($entity_type_id);

        foreach ($view_modes as $view_mode_id => $view_mode_label) {
          $display = $this->loadViewDisplay($entity_type_id, $bundle_id, $view_mode_id);
          if (!$display || !$display->status()) {
            continue;
          }

          foreach ($image_fields as $field_name => $field_info) {
            $component = $display->getComponent($field_name);

            if ($component === NULL) {
              // Field is hidden in this view mode - that's fine.
              continue;
            }

            $formatter_type = $component['type'] ?? '';
            $formatter_settings = $component['settings'] ?? [];
            $formatter_analysis = $this->analyzeFormatter($formatter_type, $formatter_settings);

            // Track used styles.
            if (!empty($formatter_analysis['image_style'])) {
              $used_image_styles[$formatter_analysis['image_style']] = TRUE;
            }
            if (!empty($formatter_analysis['responsive_style'])) {
              $used_responsive_styles[$formatter_analysis['responsive_style']] = TRUE;
            }

            // Track lazy loading.
            $has_lazy_loading = $formatter_analysis['has_lazy_loading'] ?? FALSE;
            if ($has_lazy_loading) {
              $stats['fields_with_lazy_loading']++;
            }
            else {
              $stats['fields_without_lazy_loading']++;
            }

            // Determine severity based on status.
            $severity = match ($formatter_analysis['status']) {
              'no_style' => 'error',
              'styled' => 'warning',
              'responsive' => 'ok',
              default => 'ok',
            };

            // Initialize entity type in fields_data if needed.
            if (!isset($fields_data[$entity_type_id])) {
              $fields_data[$entity_type_id] = [
                'label' => $entity_type_label,
                'bundles' => [],
                'errors' => 0,
                'warnings' => 0,
              ];
            }

            // Initialize bundle in fields_data if needed.
            if (!isset($fields_data[$entity_type_id]['bundles'][$bundle_id])) {
              $fields_data[$entity_type_id]['bundles'][$bundle_id] = [
                'label' => $bundle_info['label'] ?? $bundle_id,
                'fields' => [],
                'errors' => 0,
                'warnings' => 0,
              ];
            }

            // Store field data.
            $fields_data[$entity_type_id]['bundles'][$bundle_id]['fields'][] = [
              'field_name' => $field_name,
              'field_label' => $field_info['label'],
              'view_mode' => $view_mode_id,
              'view_mode_label' => (string) $view_mode_label,
              'formatter_type' => $formatter_type,
              'status' => $formatter_analysis['status'],
              'severity' => $severity,
              'image_style' => $formatter_analysis['image_style'],
              'responsive_style' => $formatter_analysis['responsive_style'],
              'has_lazy_loading' => $formatter_analysis['has_lazy_loading'] ?? FALSE,
              // Additional data for faceted filtering.
              'entity_type' => $entity_type_id,
              'entity_type_label' => $entity_type_label,
              'bundle' => $bundle_id,
              'bundle_label' => $bundle_info['label'] ?? $bundle_id,
            ];

            // Update counters.
            if ($severity === 'error') {
              $fields_data[$entity_type_id]['errors']++;
              $fields_data[$entity_type_id]['bundles'][$bundle_id]['errors']++;
              $stats['fields_without_style']++;

              $results[] = $this->createResultItem(
                'error',
                'IMAGE_WITHOUT_STYLE',
                (string) $this->t('Image field "@field" in @entity:@bundle (@view_mode) displays the original image without any style. This causes serious performance issues.', [
                  '@field' => $field_info['label'],
                  '@entity' => $entity_type_label,
                  '@bundle' => $bundle_info['label'] ?? $bundle_id,
                  '@view_mode' => $view_mode_label,
                ]),
                [
                  'entity_type' => $entity_type_id,
                  'entity_type_label' => $entity_type_label,
                  'bundle' => $bundle_id,
                  'bundle_label' => $bundle_info['label'] ?? $bundle_id,
                  'field_name' => $field_name,
                  'field_label' => $field_info['label'],
                  'view_mode' => $view_mode_id,
                  'view_mode_label' => (string) $view_mode_label,
                  'formatter_type' => $formatter_type,
                ]
              );
            }
            elseif ($severity === 'warning') {
              $fields_data[$entity_type_id]['warnings']++;
              $fields_data[$entity_type_id]['bundles'][$bundle_id]['warnings']++;
              $stats['fields_with_image_style']++;

              $results[] = $this->createResultItem(
                'warning',
                'IMAGE_NOT_RESPONSIVE',
                (string) $this->t('Image field "@field" in @entity:@bundle (@view_mode) uses image style "@style" but not responsive images.', [
                  '@field' => $field_info['label'],
                  '@entity' => $entity_type_label,
                  '@bundle' => $bundle_info['label'] ?? $bundle_id,
                  '@view_mode' => $view_mode_label,
                  '@style' => $formatter_analysis['image_style'] ?? '',
                ]),
                [
                  'entity_type' => $entity_type_id,
                  'entity_type_label' => $entity_type_label,
                  'bundle' => $bundle_id,
                  'bundle_label' => $bundle_info['label'] ?? $bundle_id,
                  'field_name' => $field_name,
                  'field_label' => $field_info['label'],
                  'view_mode' => $view_mode_id,
                  'view_mode_label' => (string) $view_mode_label,
                  'formatter_type' => $formatter_type,
                  'image_style' => $formatter_analysis['image_style'],
                ]
              );
            }
            else {
              $stats['fields_with_responsive']++;
            }
          }
        }
      }
    }

    return [
      'results' => $results,
      'fields' => $fields_data,
      'stats' => $stats,
      'used_image_styles' => array_keys($used_image_styles),
      'used_responsive_styles' => array_keys($used_responsive_styles),
    ];
  }

  /**
   * Analyzes a formatter configuration.
   *
   * @param string $formatter_type
   *   The formatter type.
   * @param array $settings
   *   The formatter settings.
   *
   * @return array
   *   Analysis of the formatter.
   */
  protected function analyzeFormatter(string $formatter_type, array $settings): array {
    $analysis = [
      'uses_responsive' => FALSE,
      'uses_image_style' => FALSE,
      'image_style' => NULL,
      'responsive_style' => NULL,
      'status' => 'unknown',
      'has_lazy_loading' => FALSE,
    ];

    // Check for lazy loading setting (Drupal 10+ supports this natively).
    // The setting 'image_loading' can be 'lazy' or 'eager'.
    $image_loading = $settings['image_loading']['attribute'] ?? NULL;
    $analysis['has_lazy_loading'] = ($image_loading === 'lazy');

    // Check for responsive image formatter.
    if ($formatter_type === 'responsive_image') {
      $analysis['uses_responsive'] = TRUE;
      $analysis['responsive_style'] = $settings['responsive_image_style'] ?? NULL;
      $analysis['status'] = 'responsive';
      return $analysis;
    }

    // Check for standard image formatter with style.
    if ($formatter_type === 'image' || $formatter_type === 'image_url') {
      $image_style = $settings['image_style'] ?? '';
      if (!empty($image_style)) {
        $analysis['uses_image_style'] = TRUE;
        $analysis['image_style'] = $image_style;
        $analysis['status'] = 'styled';
      }
      else {
        $analysis['status'] = 'no_style';
      }
      return $analysis;
    }

    // Other formatters (media, entity reference, etc.) - consider them OK.
    $analysis['status'] = 'other_formatter';
    return $analysis;
  }

  /**
   * Loads a view display configuration.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param string $view_mode
   *   The view mode.
   *
   * @return \Drupal\Core\Entity\Display\EntityViewDisplayInterface|null
   *   The view display or NULL.
   */
  protected function loadViewDisplay(string $entity_type_id, string $bundle_id, string $view_mode): ?EntityViewDisplayInterface {
    try {
      $display = $this->entityTypeManager
        ->getStorage('entity_view_display')
        ->load($entity_type_id . '.' . $bundle_id . '.' . $view_mode);
      return $display instanceof EntityViewDisplayInterface ? $display : NULL;
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

  /**
   * Analyzes unused image styles and responsive image styles.
   *
   * @param array $image_styles
   *   All defined image styles.
   * @param array $responsive_styles
   *   All defined responsive image styles.
   * @param array $image_fields
   *   Image field analysis data with used styles.
   *
   * @return array
   *   Unused styles analysis.
   */
  protected function analyzeUnusedStyles(array $image_styles, array $responsive_styles, array $image_fields): array {
    $results = [];
    $unused_image_styles = [];
    $unused_responsive_styles = [];

    $used_image_styles = $image_fields['used_image_styles'] ?? [];
    $used_responsive_styles = $image_fields['used_responsive_styles'] ?? [];

    // Also collect image styles used within responsive image styles.
    foreach ($responsive_styles['styles'] ?? [] as $style_data) {
      foreach ($style_data['image_styles_used'] ?? [] as $img_style) {
        $used_image_styles[] = $img_style;
      }
    }
    $used_image_styles = array_unique($used_image_styles);

    // Find unused image styles.
    foreach ($image_styles['styles'] ?? [] as $style_id => $style_data) {
      if (!in_array($style_id, $used_image_styles, TRUE)) {
        $unused_image_styles[$style_id] = $style_data;
        $results[] = $this->createResultItem(
          'notice',
          'UNUSED_IMAGE_STYLE',
          (string) $this->t('Image style "@style" is defined but not used in any image field or responsive style.', [
            '@style' => $style_data['label'] ?? $style_id,
          ]),
          [
            'style_id' => $style_id,
            'style_label' => $style_data['label'] ?? $style_id,
            'effects_count' => $style_data['effects_count'] ?? 0,
            'style_type' => 'image_style',
          ]
        );
      }
    }

    // Find unused responsive image styles.
    foreach ($responsive_styles['styles'] ?? [] as $style_id => $style_data) {
      if (!in_array($style_id, $used_responsive_styles, TRUE)) {
        $unused_responsive_styles[$style_id] = $style_data;
        $results[] = $this->createResultItem(
          'notice',
          'UNUSED_RESPONSIVE_STYLE',
          (string) $this->t('Responsive image style "@style" is defined but not used in any image field.', [
            '@style' => $style_data['label'] ?? $style_id,
          ]),
          [
            'style_id' => $style_id,
            'style_label' => $style_data['label'] ?? $style_id,
            'image_styles_count' => $style_data['image_styles_count'] ?? 0,
            'style_type' => 'responsive_style',
          ]
        );
      }
    }

    return [
      'results' => $results,
      'unused_image_styles' => $unused_image_styles,
      'unused_responsive_styles' => $unused_responsive_styles,
      'stats' => [
        'unused_image_styles' => count($unused_image_styles),
        'unused_responsive_styles' => count($unused_responsive_styles),
        'total_unused' => count($unused_image_styles) + count($unused_responsive_styles),
      ],
    ];
  }

  /**
   * Analyzes lazy loading configuration.
   *
   * @param array $image_fields
   *   Image field analysis data.
   *
   * @return array
   *   Lazy loading analysis.
   */
  protected function analyzeLazyLoading(array $image_fields): array {
    $with_lazy = $image_fields['stats']['fields_with_lazy_loading'] ?? 0;
    $without_lazy = $image_fields['stats']['fields_without_lazy_loading'] ?? 0;
    $total = $with_lazy + $without_lazy;

    $percentage = $total > 0 ? round(($with_lazy / $total) * 100, 1) : 0;

    return [
      'stats' => [
        'with_lazy_loading' => $with_lazy,
        'without_lazy_loading' => $without_lazy,
        'total' => $total,
        'percentage' => $percentage,
      ],
    ];
  }

  /**
   * Points deducted per error.
   */
  protected const POINTS_PER_ERROR = 20;

  /**
   * Points deducted per warning.
   */
  protected const POINTS_PER_WARNING = 10;

  /**
   * Points deducted per notice (unused style).
   */
  protected const POINTS_PER_UNUSED_STYLE = 5;

  /**
   * Calculates scores for image configuration.
   *
   * @param array $modules_analysis
   *   Module analysis data.
   * @param array $image_fields
   *   Image field analysis data.
   * @param array $unused_styles
   *   Unused styles analysis data.
   * @param array $lazy_loading
   *   Lazy loading analysis data.
   *
   * @return array
   *   Score data with factors.
   */
  protected function calculateScores(array $modules_analysis, array $image_fields, array $unused_styles, array $lazy_loading): array {
    $factors = [];

    // Factor 1: Essential Modules.
    $missing_modules = $modules_analysis['stats']['missing_count'] ?? 0;
    $modules_score = $missing_modules > 0 ? 0 : 100;
    $factors['modules'] = [
      'score' => $modules_score,
      'label' => (string) $this->t('Essential Modules'),
      'description' => $missing_modules > 0
        ? (string) $this->t('Responsive Image module is not installed')
        : (string) $this->t('All essential modules installed'),
    ];

    // Factor 2: Image Field Configuration.
    $fields_stats = $image_fields['stats'];
    $errors = $fields_stats['fields_without_style'] ?? 0;
    $warnings = $fields_stats['fields_with_image_style'] ?? 0;
    $ok = $fields_stats['fields_with_responsive'] ?? 0;
    $total = $errors + $warnings + $ok;

    if ($total === 0) {
      $fields_score = 100;
      $fields_description = (string) $this->t('No image fields to analyze');
    }
    else {
      $fields_score = max(0, 100 - ($errors * self::POINTS_PER_ERROR) - ($warnings * self::POINTS_PER_WARNING));

      if ($errors > 0) {
        $fields_description = (string) $this->t('@errors without style (-@error_points), @warnings not responsive (-@warning_points)', [
          '@errors' => $errors,
          '@error_points' => $errors * self::POINTS_PER_ERROR,
          '@warnings' => $warnings,
          '@warning_points' => $warnings * self::POINTS_PER_WARNING,
        ]);
      }
      elseif ($warnings > 0) {
        $fields_description = (string) $this->t('@warnings not using responsive images (-@points points)', [
          '@warnings' => $warnings,
          '@points' => $warnings * self::POINTS_PER_WARNING,
        ]);
      }
      else {
        $fields_description = (string) $this->t('All @count image fields use responsive images', [
          '@count' => $ok,
        ]);
      }
    }

    $factors['image_fields'] = [
      'score' => $fields_score,
      'label' => (string) $this->t('Image Fields'),
      'description' => $fields_description,
    ];

    // Factor 3: Unused Styles.
    $unused_stats = $unused_styles['stats'] ?? [];
    $unused_image_count = $unused_stats['unused_image_styles'] ?? 0;
    $unused_responsive_count = $unused_stats['unused_responsive_styles'] ?? 0;
    $total_unused = $unused_image_count + $unused_responsive_count;

    if ($total_unused === 0) {
      $unused_score = 100;
      $unused_description = (string) $this->t('All styles are in use');
    }
    else {
      $unused_score = max(0, 100 - ($total_unused * self::POINTS_PER_UNUSED_STYLE));
      $unused_description = (string) $this->t('@count unused styles (-@points points)', [
        '@count' => $total_unused,
        '@points' => $total_unused * self::POINTS_PER_UNUSED_STYLE,
      ]);
    }

    $factors['unused_styles'] = [
      'score' => $unused_score,
      'label' => (string) $this->t('Unused Styles'),
      'description' => $unused_description,
    ];

    // Factor 4: Lazy Loading.
    $lazy_stats = $lazy_loading['stats'] ?? [];
    $lazy_percentage = $lazy_stats['percentage'] ?? 0;
    $with_lazy = $lazy_stats['with_lazy_loading'] ?? 0;
    $without_lazy = $lazy_stats['without_lazy_loading'] ?? 0;
    $lazy_total = $lazy_stats['total'] ?? 0;

    if ($lazy_total === 0) {
      $lazy_score = 100;
      $lazy_description = (string) $this->t('No image fields to analyze');
    }
    else {
      // Score is directly the percentage of fields with lazy loading.
      $lazy_score = (int) $lazy_percentage;

      if ($lazy_percentage >= 80) {
        $lazy_description = (string) $this->t('@percent% of fields have lazy loading (@with of @total)', [
          '@percent' => $lazy_percentage,
          '@with' => $with_lazy,
          '@total' => $lazy_total,
        ]);
      }
      elseif ($lazy_percentage >= 50) {
        $lazy_description = (string) $this->t('@percent% have lazy loading, consider enabling more (@with of @total)', [
          '@percent' => $lazy_percentage,
          '@with' => $with_lazy,
          '@total' => $lazy_total,
        ]);
      }
      else {
        $lazy_description = (string) $this->t('Only @percent% have lazy loading (@with of @total)', [
          '@percent' => $lazy_percentage,
          '@with' => $with_lazy,
          '@total' => $lazy_total,
        ]);
      }
    }

    $factors['lazy_loading'] = [
      'score' => $lazy_score,
      'label' => (string) $this->t('Lazy Loading'),
      'description' => $lazy_description,
    ];

    return [
      'factors' => $factors,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      'modules' => [
        'label' => $this->t('Modules'),
        'description' => $this->t('Checks essential and recommended modules for optimal image handling.'),
        'file_key' => 'modules',
        'affects_score' => TRUE,
        'score_factor_key' => 'modules',
      ],
      'image_styles' => [
        'label' => $this->t('Image Styles'),
        'description' => $this->t('Lists all configured image styles and their effects.'),
        'file_key' => 'image_styles',
        'affects_score' => FALSE,
      ],
      'responsive_styles' => [
        'label' => $this->t('Responsive Image Styles'),
        'description' => $this->t('Lists all configured responsive image styles and their breakpoint mappings.'),
        'file_key' => 'responsive_styles',
        'affects_score' => FALSE,
      ],
      'unused_styles' => [
        'label' => $this->t('Unused Styles'),
        'description' => $this->t('Lists image styles and responsive styles that are defined but not used.'),
        'file_key' => 'unused_styles',
        'affects_score' => TRUE,
        'score_factor_key' => 'unused_styles',
      ],
      'lazy_loading' => [
        'label' => $this->t('Lazy Loading'),
        'description' => $this->t('Analyzes lazy loading configuration for image fields.'),
        'file_key' => 'lazy_loading',
        'affects_score' => TRUE,
        'score_factor_key' => 'lazy_loading',
      ],
      'image_fields' => [
        'label' => $this->t('Image Field Analysis'),
        'description' => $this->t('Analyzes image field formatters across all view modes to ensure they use responsive images.'),
        'file_key' => 'image_fields',
        'affects_score' => FALSE,
      ],
      'image_fields_issues' => [
        'label' => $this->t('Image Field Issues'),
        'description' => $this->t('Filterable list of image field configuration issues.'),
        'file_key' => 'image_fields_issues',
        'affects_score' => TRUE,
        'score_factor_key' => 'image_fields',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    return match ($check_id) {
      'modules' => $this->getModulesContent($data),
      'image_styles' => $this->getImageStylesContent($data),
      'responsive_styles' => $this->getResponsiveStylesContent($data),
      'unused_styles' => $this->getUnusedStylesContent($data),
      'lazy_loading' => $this->getLazyLoadingContent($data),
      'image_fields' => $this->getImageFieldsContent($data),
      'image_fields_issues' => $this->getImageFieldsIssuesContent($data),
      default => [],
    };
  }

  /**
   * Gets the modules content (without section wrapper).
   *
   * @param array $data
   *   The analysis data.
   *
   * @return array
   *   Render array for modules content.
   */
  protected function getModulesContent(array $data): array {
    $modules_data = $data['data']['modules'] ?? [];

    $headers = [
      $this->ui->header('', 'center', '40px'),
      $this->ui->header('', 'center'),
      (string) $this->t('Module'),
      (string) $this->t('Description'),
    ];

    $rows = [];

    // Add essential installed modules.
    foreach ($modules_data['essential_installed'] ?? [] as $module => $info) {
      $rows[] = $this->ui->row([
        $this->ui->cell($this->ui->icon('check'), ['align' => 'center']),
        $this->ui->cell($this->ui->badge((string) $this->t('Essential'), 'success'), ['align' => 'center']),
        $this->ui->itemName($info['label'], $module),
        $info['description'],
      ]);
    }

    // Add essential missing modules.
    foreach ($modules_data['essential_missing'] ?? [] as $module => $info) {
      $rows[] = $this->ui->row([
        $this->ui->cell($this->ui->icon('cross'), ['align' => 'center']),
        $this->ui->cell($this->ui->badge((string) $this->t('Essential'), 'error'), ['align' => 'center']),
        $this->ui->itemName($info['label'], $module),
        $info['description'],
      ], 'error');
    }

    // Add recommended installed modules.
    foreach ($modules_data['recommended_installed'] ?? [] as $module => $info) {
      $rows[] = $this->ui->row([
        $this->ui->cell($this->ui->icon('check'), ['align' => 'center']),
        $this->ui->cell($this->ui->badge((string) $this->t('Recommended'), 'success'), ['align' => 'center']),
        $this->ui->itemName($info['label'], $module),
        $info['description'],
      ]);
    }

    // Add recommended missing modules.
    foreach ($modules_data['recommended_missing'] ?? [] as $module => $info) {
      $rows[] = $this->ui->row([
        $this->ui->cell($this->ui->icon('cross'), ['align' => 'center']),
        $this->ui->cell($this->ui->badge((string) $this->t('Recommended'), 'info'), ['align' => 'center']),
        $this->ui->itemName($info['label'], $module),
        $info['description'],
      ], 'notice');
    }

    return $this->ui->table($headers, $rows, [
      'empty' => (string) $this->t('No modules to display.'),
    ]);
  }

  /**
   * Gets the image styles content (without section wrapper).
   *
   * @param array $data
   *   The analysis data.
   *
   * @return array
   *   Render array for image styles content.
   */
  protected function getImageStylesContent(array $data): array {
    $styles = $data['data']['image_styles']['styles'] ?? [];

    $headers = [
      (string) $this->t('Name'),
      (string) $this->t('Machine name'),
      (string) $this->t('Effects'),
      $this->ui->header((string) $this->t('Effects count'), 'center'),
    ];

    $rows = [];
    foreach ($styles as $style) {
      $effects_list = [];
      foreach ($style['effects'] as $effect) {
        $effects_list[] = $effect['label'];
      }

      $rows[] = $this->ui->row([
        $style['label'],
        $style['id'],
        implode(', ', $effects_list) ?: (string) $this->t('No effects'),
        $this->ui->cell((string) $style['effects_count'], ['align' => 'center']),
      ]);
    }

    return $this->ui->table($headers, $rows, [
      'empty' => (string) $this->t('No image styles found.'),
    ]);
  }

  /**
   * Gets the responsive image styles content (without section wrapper).
   *
   * @param array $data
   *   The analysis data.
   *
   * @return array
   *   Render array for responsive styles content.
   */
  protected function getResponsiveStylesContent(array $data): array {
    $styles = $data['data']['responsive_styles']['styles'] ?? [];

    if (!$this->moduleHandler->moduleExists('responsive_image')) {
      return $this->ui->message(
        (string) $this->t('The Responsive Image module is not installed. Install it to enable responsive image styles.'),
        'warning'
      );
    }

    $headers = [
      (string) $this->t('Name'),
      (string) $this->t('Machine name'),
      $this->ui->header((string) $this->t('Styles count'), 'center'),
      (string) $this->t('Breakpoint configuration'),
    ];

    $rows = [];
    foreach ($styles as $style) {
      // Build list items for breakpoint: image_style format.
      $mapping_items = [];
      foreach ($style['mappings'] ?? [] as $mapping) {
        $mapping_items[] = $mapping['breakpoint'] . ': ' . $mapping['image_style'];
      }

      // Add fallback as a separate item.
      if (!empty($style['fallback_image_style'])) {
        $mapping_items[] = (string) $this->t('Fallback') . ': ' . $style['fallback_image_style'];
      }

      $rows[] = $this->ui->row([
        $style['label'],
        $style['id'],
        $this->ui->cell((string) ($style['image_styles_count'] ?? 0), ['align' => 'center']),
        implode('<br>', array_map('htmlspecialchars', $mapping_items)),
      ]);
    }

    return $this->ui->table($headers, $rows, [
      'empty' => (string) $this->t('No responsive image styles found. Consider creating responsive image styles for better performance.'),
    ]);
  }

  /**
   * Gets the image fields content (without section wrapper).
   *
   * @param array $data
   *   The analysis data.
   *
   * @return array
   *   Render array for image fields content.
   */
  protected function getImageFieldsContent(array $data): array {
    $fields_data = $data['data']['image_fields']['fields'] ?? [];

    if (empty($fields_data)) {
      return $this->ui->message(
        (string) $this->t('No image fields found.'),
        'info'
      );
    }

    // Build nested content for entity types.
    $content = [];

    // Build grouped details by entity type.
    foreach ($fields_data as $entity_type_id => $entity_type_data) {
      $entity_content = [];

      // Build grouped details by bundle.
      foreach ($entity_type_data['bundles'] as $bundle_id => $bundle_data) {
        $headers = [
          (string) $this->t('Field'),
          (string) $this->t('View mode'),
          (string) $this->t('Formatter'),
          (string) $this->t('Style'),
          (string) $this->t('Status'),
        ];

        $rows = [];
        foreach ($bundle_data['fields'] as $field) {
          $severity = match ($field['severity']) {
            'error' => 'error',
            'warning' => 'warning',
            default => NULL,
          };

          $style_display = match ($field['status']) {
            'responsive' => $field['responsive_style'] ?? '-',
            'styled' => $field['image_style'] ?? '-',
            'no_style' => (string) $this->t('Original'),
            default => '-',
          };

          $status_badge = match ($field['severity']) {
            'error' => $this->ui->badge((string) $this->t('Error'), 'error'),
            'warning' => $this->ui->badge((string) $this->t('Warning'), 'warning'),
            'ok' => $this->ui->badge((string) $this->t('OK'), 'success'),
            default => '-',
          };

          $status_cell = match ($field['severity']) {
            'error' => $this->ui->cell($status_badge, ['status' => 'error']),
            'warning' => $this->ui->cell($status_badge, ['status' => 'warning']),
            'ok' => $this->ui->cell($status_badge, ['status' => 'ok']),
            default => '-',
          };

          $rows[] = $this->ui->row([
            $field['field_label'],
            $field['view_mode_label'],
            $field['formatter_type'],
            $style_display,
            $status_cell,
          ], $severity);
        }

        $bundle_severity = $this->determineSeverity($bundle_data['errors'], $bundle_data['warnings']);

        $entity_content[$bundle_id] = $this->ui->section(
          $bundle_data['label'] . $this->buildCounterSuffix($bundle_data['errors'], $bundle_data['warnings']),
          $this->ui->table($headers, $rows),
          [
            'open' => $bundle_data['errors'] > 0,
            'severity' => $bundle_severity,
          ]
        );
      }

      $entity_severity = $this->determineSeverity($entity_type_data['errors'], $entity_type_data['warnings']);

      $content[$entity_type_id] = $this->ui->section(
        $entity_type_data['label'] . $this->buildCounterSuffix($entity_type_data['errors'], $entity_type_data['warnings']),
        $entity_content,
        [
          'open' => $entity_type_data['errors'] > 0,
          'severity' => $entity_severity,
        ]
      );
    }

    return $content;
  }

  /**
   * Determines severity based on error and warning counts.
   *
   * @param int $errors
   *   Number of errors.
   * @param int $warnings
   *   Number of warnings.
   *
   * @return string|null
   *   Severity level or NULL if no issues.
   */
  protected function determineSeverity(int $errors, int $warnings): ?string {
    if ($errors > 0) {
      return 'error';
    }
    if ($warnings > 0) {
      return 'warning';
    }
    return NULL;
  }

  /**
   * Builds a counter suffix string for titles.
   *
   * @param int $errors
   *   Number of errors.
   * @param int $warnings
   *   Number of warnings.
   *
   * @return string
   *   Formatted counter suffix.
   */
  protected function buildCounterSuffix(int $errors, int $warnings): string {
    $parts = [];

    if ($errors > 0) {
      $parts[] = $this->t('@count errors', ['@count' => $errors]);
    }
    if ($warnings > 0) {
      $parts[] = $this->t('@count warnings', ['@count' => $warnings]);
    }

    if (empty($parts)) {
      return '';
    }

    return ' (' . implode(', ', $parts) . ')';
  }

  /**
   * Gets the unused styles content.
   *
   * @param array $data
   *   The analysis data.
   *
   * @return array
   *   Render array for unused styles content.
   */
  protected function getUnusedStylesContent(array $data): array {
    $unused_data = $data['data']['unused_styles'] ?? [];
    $unused_image = $unused_data['unused_image_styles'] ?? [];
    $unused_responsive = $unused_data['unused_responsive_styles'] ?? [];

    if (empty($unused_image) && empty($unused_responsive)) {
      return $this->ui->message(
        (string) $this->t('All defined image styles and responsive styles are in use.'),
        'success'
      );
    }

    $content = [];

    // Unused image styles.
    if (!empty($unused_image)) {
      $headers = [
        (string) $this->t('Name'),
        (string) $this->t('Machine name'),
        $this->ui->header((string) $this->t('Effects'), 'center'),
      ];

      $rows = [];
      foreach ($unused_image as $style_id => $style_data) {
        $rows[] = $this->ui->row([
          $style_data['label'] ?? $style_id,
          $style_id,
          $this->ui->cell((string) ($style_data['effects_count'] ?? 0), ['align' => 'center']),
        ], 'notice');
      }

      $content['image_styles'] = $this->ui->section(
        (string) $this->t('Unused Image Styles (@count)', ['@count' => count($unused_image)]),
        $this->ui->table($headers, $rows),
        ['severity' => 'notice']
      );
    }

    // Unused responsive styles.
    if (!empty($unused_responsive)) {
      $headers = [
        (string) $this->t('Name'),
        (string) $this->t('Machine name'),
        $this->ui->header((string) $this->t('Styles count'), 'center'),
      ];

      $rows = [];
      foreach ($unused_responsive as $style_id => $style_data) {
        $rows[] = $this->ui->row([
          $style_data['label'] ?? $style_id,
          $style_id,
          $this->ui->cell((string) ($style_data['image_styles_count'] ?? 0), ['align' => 'center']),
        ], 'notice');
      }

      $content['responsive_styles'] = $this->ui->section(
        (string) $this->t('Unused Responsive Styles (@count)', ['@count' => count($unused_responsive)]),
        $this->ui->table($headers, $rows),
        ['severity' => 'notice']
      );
    }

    $content['note'] = $this->ui->message(
      (string) $this->t('Unused styles can be safely removed to simplify configuration. However, check if they are used in custom code or Views before deleting.'),
      'info'
    );

    return $content;
  }

  /**
   * Gets the lazy loading content.
   *
   * @param array $data
   *   The analysis data.
   *
   * @return array
   *   Render array for lazy loading content.
   */
  protected function getLazyLoadingContent(array $data): array {
    $lazy_stats = $data['stats']['lazy_loading'] ?? [];
    $with_lazy = $lazy_stats['with_lazy_loading'] ?? 0;
    $without_lazy = $lazy_stats['without_lazy_loading'] ?? 0;
    $total = $lazy_stats['total'] ?? 0;
    $percentage = $lazy_stats['percentage'] ?? 0;

    if ($total === 0) {
      return $this->ui->message(
        (string) $this->t('No image fields to analyze for lazy loading.'),
        'info'
      );
    }

    $content = [];

    // Summary message.
    if ($percentage >= 80) {
      $message_type = 'success';
      $message = (string) $this->t('Good! @percent% of image fields have lazy loading enabled (@with of @total).', [
        '@percent' => $percentage,
        '@with' => $with_lazy,
        '@total' => $total,
      ]);
    }
    elseif ($percentage >= 50) {
      $message_type = 'warning';
      $message = (string) $this->t('@percent% of image fields have lazy loading enabled (@with of @total). Consider enabling it for more fields.', [
        '@percent' => $percentage,
        '@with' => $with_lazy,
        '@total' => $total,
      ]);
    }
    else {
      $message_type = 'warning';
      $message = (string) $this->t('Only @percent% of image fields have lazy loading enabled (@with of @total). Lazy loading improves page performance.', [
        '@percent' => $percentage,
        '@with' => $with_lazy,
        '@total' => $total,
      ]);
    }

    $content['summary'] = $this->ui->message($message, $message_type);

    // Stats table.
    $headers = [
      (string) $this->t('Configuration'),
      $this->ui->header((string) $this->t('Count'), 'center'),
      $this->ui->header((string) $this->t('Percentage'), 'center'),
    ];

    $rows = [
      $this->ui->row([
        (string) $this->t('With lazy loading'),
        $this->ui->cell((string) $with_lazy, ['align' => 'center']),
        $this->ui->cell($percentage . '%', ['align' => 'center', 'status' => $percentage >= 80 ? 'ok' : NULL]),
      ]),
      $this->ui->row([
        (string) $this->t('Without lazy loading'),
        $this->ui->cell((string) $without_lazy, ['align' => 'center']),
        $this->ui->cell(round(100 - $percentage, 1) . '%', ['align' => 'center', 'status' => $without_lazy > 0 ? 'warning' : NULL]),
      ], $without_lazy > 0 ? 'warning' : NULL),
    ];

    $content['table'] = $this->ui->table($headers, $rows);

    $content['note'] = $this->ui->message(
      (string) $this->t('Lazy loading defers loading of off-screen images until they are needed, improving initial page load time. In Drupal 10+, you can enable lazy loading in the image formatter settings.'),
      'info'
    );

    return $content;
  }

  /**
   * Gets the image fields issues content as a faceted list.
   *
   * @param array $data
   *   The analysis data.
   *
   * @return array
   *   Render array for image fields issues list.
   */
  protected function getImageFieldsIssuesContent(array $data): array {
    $results = $data['results'] ?? [];

    // Filter only image field issues.
    $image_field_results = array_filter($results, function ($result) {
      $code = $result['code'] ?? '';
      return in_array($code, ['IMAGE_WITHOUT_STYLE', 'IMAGE_NOT_RESPONSIVE'], TRUE);
    });

    if (empty($image_field_results)) {
      return $this->ui->message(
        (string) $this->t('No image field configuration issues found.'),
        'success'
      );
    }

    // Define custom filters for the faceted list.
    $custom_filters = [
      'entity-type' => [
        'label' => (string) $this->t('Entity Type'),
        'attribute' => 'data-entity-type',
      ],
      'bundle' => [
        'label' => (string) $this->t('Bundle'),
        'attribute' => 'data-bundle',
      ],
      'view-mode' => [
        'label' => (string) $this->t('View Mode'),
        'attribute' => 'data-view-mode',
      ],
      'field-name' => [
        'label' => (string) $this->t('Field'),
        'attribute' => 'data-field-name',
      ],
    ];

    // Build issues with custom data attributes for filtering.
    $issues = [];
    foreach ($image_field_results as $result) {
      $details = $result['details'] ?? [];
      $code = $result['code'] ?? '';

      // Determine tags based on issue type.
      $tags = ['performance'];
      if ($code === 'IMAGE_WITHOUT_STYLE') {
        $tags[] = 'best_practice';
      }

      $issues[] = $this->ui->issue([
        'severity' => $result['severity'] ?? 'warning',
        'code' => $code,
        'label' => $result['message'] ?? '',
        'file' => $details['entity_type_label'] . ':' . $details['bundle_label'],
        'line' => NULL,
        'description' => $this->getIssueDescription($code),
        'tags' => $tags,
        'custom_data' => [
          'entity-type' => $details['entity_type'] ?? '',
          'bundle' => $details['bundle'] ?? '',
          'view-mode' => $details['view_mode'] ?? '',
          'field-name' => $details['field_name'] ?? '',
        ],
      ]);
    }

    return $this->ui->issueList(
      $issues,
      (string) $this->t('No image field issues found.'),
      $custom_filters
    );
  }

  /**
   * Gets the description for an issue code.
   *
   * @param string $code
   *   The issue code.
   *
   * @return string
   *   The issue description.
   */
  protected function getIssueDescription(string $code): string {
    return match ($code) {
      'IMAGE_WITHOUT_STYLE' => (string) $this->t('This image field displays the original uploaded image without any resizing or optimization. This can cause serious performance issues as large images are loaded at full size. Configure an image style or responsive image style for this field.'),
      'IMAGE_NOT_RESPONSIVE' => (string) $this->t('This image field uses a fixed image style instead of a responsive image style. Responsive images serve appropriately sized images based on the viewport, improving performance on mobile devices. Consider switching to a responsive image formatter.'),
      default => '',
    };
  }

}
