<?php

declare(strict_types=1);

namespace Drupal\audit_modules\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ThemeExtensionList;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Analyzes installed modules and themes, and provides recommendations.
 */
#[AuditAnalyzer(
  id: 'modules',
  label: new TranslatableMarkup('Modules & Themes Audit'),
  description: new TranslatableMarkup('Analyzes active modules/themes, detects unused extensions, and provides recommendations based on project configuration.'),
  menu_title: new TranslatableMarkup('Modules'),
  output_directory: 'modules',
  weight: 3,
)]
class ModulesAnalyzer extends AuditAnalyzerBase {

  protected const SCORE_WEIGHTS = [
    'removable_modules' => 10,
    'removable_themes' => 5,
    'redundant_themes' => 10,
    'recommended_modules' => 30,
    'not_recommended' => 25,
    'ui_modules' => 10,
    'duplicate_modules' => 20,
  ];

  /**
   * Module recommendations indexed by module name.
   *
   * Keys: label, description, user_types[], requires_dedicated_server, priority, category.
   */
  protected const MODULE_RECOMMENDATIONS = [
    'page_cache' => [
      'label' => 'Internal Page Cache',
      'description' => 'Caches pages for anonymous users. Essential for sites with anonymous traffic.',
      'user_types' => ['anonymous', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'required',
      'category' => 'cache',
    ],
    'dynamic_page_cache' => [
      'label' => 'Internal Dynamic Page Cache',
      'description' => 'Caches pages for all users, with placeholders for dynamic content.',
      'user_types' => ['anonymous', 'registered', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'required',
      'category' => 'cache',
    ],
    'purge' => [
      'label' => 'Purge',
      'description' => 'External cache invalidation framework. Required for Varnish integration.',
      'user_types' => ['anonymous', 'mixed'],
      'requires_dedicated_server' => TRUE,
      'priority' => 'recommended',
      'category' => 'cache',
    ],
    'varnish_purge' => [
      'label' => 'Varnish Purge',
      'description' => 'Varnish cache invalidation. Requires Varnish reverse proxy.',
      'user_types' => ['anonymous', 'mixed'],
      'requires_dedicated_server' => TRUE,
      'priority' => 'recommended',
      'category' => 'cache',
    ],
    'big_pipe' => [
      'label' => 'BigPipe',
      'description' => 'Sends pages using BigPipe technique for faster perceived performance for authenticated users.',
      'user_types' => ['registered', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'required',
      'category' => 'performance',
    ],
    'big_pipe_sessionless' => [
      'label' => 'BigPipe Sessionless',
      'description' => 'BigPipe for anonymous users. Improves perceived performance for anonymous traffic.',
      'user_types' => ['anonymous', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'recommended',
      'category' => 'performance',
    ],
    'imageapi_optimize' => [
      'label' => 'ImageAPI Optimize',
      'description' => 'Optimizes images for better performance and smaller file sizes.',
      'user_types' => ['anonymous', 'registered', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'recommended',
      'category' => 'performance',
    ],
    'advagg' => [
      'label' => 'Advanced CSS/JS Aggregation',
      'description' => 'Advanced aggregation for CSS and JavaScript files.',
      'user_types' => ['anonymous', 'registered', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'recommended',
      'category' => 'performance',
    ],
    'seckit' => [
      'label' => 'Security Kit',
      'description' => 'Provides security hardening options including CSP, clickjacking protection.',
      'user_types' => ['anonymous', 'registered', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'recommended',
      'category' => 'security',
    ],
    'password_policy' => [
      'label' => 'Password Policy',
      'description' => 'Enforces password strength requirements for user accounts.',
      'user_types' => ['registered', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'recommended',
      'category' => 'security',
    ],
    'flood_control' => [
      'label' => 'Flood Control',
      'description' => 'Configurable flood control to prevent brute force attacks.',
      'user_types' => ['registered', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'recommended',
      'category' => 'security',
    ],
    'username_enumeration_prevention' => [
      'label' => 'Username Enumeration Prevention',
      'description' => 'Prevents username discovery through login/password reset forms.',
      'user_types' => ['registered', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'recommended',
      'category' => 'security',
    ],
    'responsive_image' => [
      'label' => 'Responsive Image',
      'description' => 'Provides responsive images for different screen sizes. Required for mobile-friendly SEO.',
      'user_types' => ['anonymous', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'required',
      'category' => 'seo',
    ],
    'metatag' => [
      'label' => 'Metatag',
      'description' => 'Manages meta tags for better SEO and social sharing.',
      'user_types' => ['anonymous', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'required',
      'category' => 'seo',
    ],
    'pathauto' => [
      'label' => 'Pathauto',
      'description' => 'Automatically generates URL aliases for better SEO.',
      'user_types' => ['anonymous', 'registered', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'required',
      'category' => 'seo',
    ],
    'redirect' => [
      'label' => 'Redirect',
      'description' => 'Manages URL redirects to prevent 404 errors and preserve SEO.',
      'user_types' => ['anonymous', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'recommended',
      'category' => 'seo',
    ],
    'simple_sitemap' => [
      'label' => 'Simple XML Sitemap',
      'description' => 'Generates XML sitemaps for search engine indexing.',
      'user_types' => ['anonymous', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'recommended',
      'category' => 'seo',
    ],
    'schema_metatag' => [
      'label' => 'Schema.org Metatag',
      'description' => 'Adds structured data for rich search results.',
      'user_types' => ['anonymous', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'recommended',
      'category' => 'seo',
    ],
    'session_limit' => [
      'label' => 'Session Limit',
      'description' => 'Limits concurrent user sessions for better security.',
      'user_types' => ['registered', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'recommended',
      'category' => 'security',
    ],
    'autologout' => [
      'label' => 'Automated Logout',
      'description' => 'Automatically logs out inactive users for security.',
      'user_types' => ['registered', 'mixed'],
      'requires_dedicated_server' => FALSE,
      'priority' => 'recommended',
      'category' => 'security',
    ],
  ];

  /**
   * Modules NOT recommended for production environments.
   *
   * These modules should generate warnings when detected as active.
   */
  protected const NOT_RECOMMENDED_MODULES = [
    // Development modules - NEVER in production.
    'devel' => [
      'label' => 'Devel',
      'description' => 'Debug and development module. Must be disabled in production environments.',
      'category' => 'development',
      'severity' => 'error',
    ],
    'devel_php' => [
      'label' => 'Devel PHP',
      'description' => 'Allows arbitrary PHP execution. Extremely dangerous security risk in production.',
      'category' => 'development',
      'severity' => 'error',
    ],
    'devel_debug_log' => [
      'label' => 'Devel Debug Log',
      'description' => 'Debug logging module. Should not be active in production.',
      'category' => 'development',
      'severity' => 'warning',
    ],
    'devel_generate' => [
      'label' => 'Devel Generate',
      'description' => 'Content generation for testing. Should not be active in production.',
      'category' => 'development',
      'severity' => 'warning',
    ],
    'migrate_devel' => [
      'label' => 'Migrate Devel',
      'description' => 'Migration debugging tools. Only needed during development.',
      'category' => 'development',
      'severity' => 'warning',
    ],
    'config_devel' => [
      'label' => 'Config Devel',
      'description' => 'Configuration development tools. Should not be active in production.',
      'category' => 'development',
      'severity' => 'warning',
    ],
    'kint' => [
      'label' => 'Kint',
      'description' => 'Debug output library. Must be disabled in production environments.',
      'category' => 'development',
      'severity' => 'error',
    ],
    'stage_file_proxy' => [
      'label' => 'Stage File Proxy',
      'description' => 'Downloads files from production server. Only for development/staging environments.',
      'category' => 'development',
      'severity' => 'warning',
    ],
    'reroute_email' => [
      'label' => 'Reroute Email',
      'description' => 'Redirects all emails to test addresses. Only for development/staging environments.',
      'category' => 'development',
      'severity' => 'warning',
    ],
    'styleguide' => [
      'label' => 'Style Guide',
      'description' => 'Theme development tool. Should not be active in production.',
      'category' => 'development',
      'severity' => 'warning',
    ],
    'storybook' => [
      'label' => 'Storybook',
      'description' => 'Component development tool. Should not be active in production.',
      'category' => 'development',
      'severity' => 'warning',
    ],
    // Security risk modules.
    'backup_migrate' => [
      'label' => 'Backup and Migrate',
      'description' => 'Security risk if misconfigured. Use server-level backups instead.',
      'category' => 'security_risk',
      'severity' => 'warning',
    ],
    'masquerade' => [
      'label' => 'Masquerade',
      'description' => 'User impersonation module. Security risk if permissions are misconfigured.',
      'category' => 'security_risk',
      'severity' => 'warning',
    ],
    'php' => [
      'label' => 'PHP Filter',
      'description' => 'Allows PHP in content. Extreme security risk - never use in production.',
      'category' => 'security_risk',
      'severity' => 'error',
    ],
  ];

  /**
   * Grouped requirements where one of the modules must be installed.
   */
  protected const GROUPED_REQUIREMENTS = [
    'cache_backend' => [
      'label' => 'Memcache or Redis',
      'description' => 'High-performance caching backend. One of these modules is required for optimal performance.',
      'modules' => ['memcache', 'redis'],
      'user_types' => ['anonymous', 'registered', 'mixed'],
      'requires_dedicated_server' => TRUE,
      'priority' => 'required',
      'category' => 'cache',
    ],
  ];

  protected ModuleExtensionList $moduleExtensionList;
  protected ModuleHandlerInterface $moduleHandler;
  protected ThemeExtensionList $themeExtensionList;
  protected ThemeHandlerInterface $themeHandler;
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->moduleExtensionList = $container->get('extension.list.module');
    $instance->moduleHandler = $container->get('module_handler');
    $instance->themeExtensionList = $container->get('extension.list.theme');
    $instance->themeHandler = $container->get('theme_handler');
    $instance->configFactory = $container->get('config.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    $config = $this->configFactory->get('audit_modules.settings');
    $audit_config = $this->configFactory->get('audit.settings');
    $user_type = $audit_config->get('user_type') ?? 'mixed';
    $shared_hosting = (bool) $config->get('shared_hosting');
    $ignore_ui_modules = (bool) $config->get('ignore_ui_modules');
    $ignored_categories = [
      'cache' => (bool) $config->get('ignore_cache'),
      'performance' => (bool) $config->get('ignore_performance'),
      'security' => (bool) $config->get('ignore_security'),
      'seo' => (bool) $config->get('ignore_seo'),
    ];

    $active_modules = $this->analyzeActiveModules();
    $removable_modules = $this->analyzeRemovableModules();
    $active_themes = $this->analyzeActiveThemes();
    $removable_themes = $this->analyzeRemovableThemes();
    $redundant_themes = $this->analyzeRedundantThemes();
    $recommendations = $this->analyzeRecommendations($user_type, $shared_hosting, $ignored_categories);
    $recommendations_issues = $this->buildMissingRecommendationsFile($recommendations);
    $not_recommended = $this->analyzeNotRecommendedModules();
    $ui_modules = $this->analyzeUiModules($ignore_ui_modules);
    $duplicate_modules = $this->analyzeDuplicateModules();
    $composer_patches = $this->analyzeComposerPatches();

    $scores = $this->calculateScores(
      $removable_modules,
      $removable_themes,
      $redundant_themes,
      $recommendations,
      $not_recommended,
      $ui_modules,
      $ignore_ui_modules,
      $duplicate_modules
    );

    return [
      '_files' => [
        'active' => $active_modules,
        'removable' => $removable_modules,
        'active_themes' => $active_themes,
        'removable_themes' => $removable_themes,
        'redundant_themes' => $redundant_themes,
        'recommendations' => $recommendations,
        'recommendations_issues' => $recommendations_issues,
        'not_recommended' => $not_recommended,
        'ui_modules' => $ui_modules,
        'duplicate_modules' => $duplicate_modules,
        'composer_patches' => $composer_patches,
      ],
      'score' => $scores,
      'module_counts' => $this->getModuleCounts($active_modules),
      'theme_counts' => $this->getThemeCounts($active_themes),
      'config' => [
        'user_type' => $user_type,
        'shared_hosting' => $shared_hosting,
        'ignore_ui_modules' => $ignore_ui_modules,
        'ignored_categories' => $ignored_categories,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      // Sections that affect score - use faceted lists.
      'not_recommended' => [
        'label' => $this->t('Production Readiness Issues'),
        'description' => $this->t('Development and risky modules that should not be active in production.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'not_recommended',
        'score_factor_key' => 'not_recommended',
        'weight' => self::SCORE_WEIGHTS['not_recommended'],
      ],
      'duplicate_modules' => [
        'label' => $this->t('Duplicate Module Issues'),
        'description' => $this->t('Modules installed in multiple locations causing potential conflicts.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'duplicate_modules',
        'score_factor_key' => 'duplicate_modules',
        'weight' => self::SCORE_WEIGHTS['duplicate_modules'],
      ],
      'ui_modules' => [
        'label' => $this->t('UI Module Issues'),
        'description' => $this->t('Administration UI modules that can impact performance in production.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'ui_modules',
        'score_factor_key' => 'ui_modules',
        'weight' => self::SCORE_WEIGHTS['ui_modules'],
      ],
      'recommendations_issues' => [
        'label' => $this->t('Missing Required Modules'),
        'description' => $this->t('Required modules that are not installed.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'recommendations_issues',
        'score_factor_key' => 'recommended_modules',
        'weight' => self::SCORE_WEIGHTS['recommended_modules'],
      ],
      'removable_modules' => [
        'label' => $this->t('Removable Module Issues'),
        'description' => $this->t('Modules that exist in code but are not enabled.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'removable',
        'score_factor_key' => 'removable_modules',
        'weight' => self::SCORE_WEIGHTS['removable_modules'],
      ],
      'removable_themes' => [
        'label' => $this->t('Removable Theme Issues'),
        'description' => $this->t('Themes that exist in code but are not enabled.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'removable_themes',
        'score_factor_key' => 'removable_themes',
        'weight' => self::SCORE_WEIGHTS['removable_themes'],
      ],
      'redundant_themes' => [
        'label' => $this->t('Redundant Active Themes'),
        'description' => $this->t('Themes that are active but not used as default, admin, or base theme. These consume memory and add maintenance overhead.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'redundant_themes',
        'score_factor_key' => 'redundant_themes',
        'weight' => self::SCORE_WEIGHTS['redundant_themes'],
      ],
      // Informational sections - use tables (no score impact).
      'recommendations_analysis' => [
        'label' => $this->t('Recommended Modules Analysis'),
        'description' => $this->t('Overview of recommended modules and their installation status.'),
        'file_types' => ['config'],
        'affects_score' => FALSE,
        'file_key' => 'recommendations',
      ],
      'active_modules' => [
        'label' => $this->t('Active Modules Analysis'),
        'description' => $this->t('Overview of currently active modules.'),
        'file_types' => ['config'],
        'affects_score' => FALSE,
        'file_key' => 'active',
      ],
      'active_themes' => [
        'label' => $this->t('Active Themes Analysis'),
        'description' => $this->t('Overview of currently active themes.'),
        'file_types' => ['config'],
        'affects_score' => FALSE,
        'file_key' => 'active_themes',
      ],
      'composer_patches' => [
        'label' => $this->t('Composer Patches'),
        'description' => $this->t('Patches applied via composer.json to modules and packages.'),
        'file_types' => ['config'],
        'affects_score' => FALSE,
        'file_key' => 'composer_patches',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    $files = $data['_files'] ?? [];
    $config = $data['config'] ?? [];

    return match ($check_id) {
      // Faceted lists for scoring sections.
      'not_recommended' => $this->buildNotRecommendedIssues($files),
      'duplicate_modules' => $this->buildDuplicateModulesIssues($files),
      'ui_modules' => $this->buildUiModulesIssues($files, $config),
      'recommendations_issues' => $this->buildRecommendationsIssues($files),
      'removable_modules' => $this->buildRemovableModulesIssues($files),
      'removable_themes' => $this->buildRemovableThemesIssues($files),
      'redundant_themes' => $this->buildRedundantThemesIssues($files),
      // Tables for informational sections.
      'recommendations_analysis' => $this->buildRecommendationsAnalysis($files),
      'active_modules' => $this->buildActiveModulesContent($files, $data),
      'active_themes' => $this->buildActiveThemesContent($files, $data),
      'composer_patches' => $this->buildComposerPatchesContent($files),
      default => [],
    };
  }

  /**
   * Builds faceted list for not recommended modules issues.
   */
  protected function buildNotRecommendedIssues(array $files): array {
    $results = $files['not_recommended']['results'] ?? [];

    if (empty($results)) {
      return ['message' => $this->ui->message(
        (string) $this->t('No development or risky modules are active. The site is ready for production.'),
        'success'
      )];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $severity = $item['severity'] ?? 'warning';
      $category = $details['category'] ?? 'development';

      $issues[] = $this->ui->issue([
        'severity' => $severity,
        'code' => $item['code'] ?? 'NOT_RECOMMENDED_MODULE',
        'label' => $details['label'] ?? $details['module'] ?? '',
        'file' => $details['module'] ?? '',
        'description' => $details['description'] ?? $item['message'] ?? '',
        'tags' => [$category],
        'custom_data' => [
          'category' => $category,
          'module' => $details['module'] ?? '',
        ],
      ]);
    }

    $custom_filters = [
      'category' => [
        'label' => (string) $this->t('Category'),
        'attribute' => 'data-category',
      ],
    ];

    return [
      'list' => $this->ui->issueList(
        $issues,
        (string) $this->t('No issues match the selected filters.'),
        $custom_filters
      ),
    ];
  }

  /**
   * Builds faceted list for duplicate modules issues.
   */
  protected function buildDuplicateModulesIssues(array $files): array {
    $results = $files['duplicate_modules']['results'] ?? [];

    if (empty($results)) {
      return ['message' => $this->ui->message(
        (string) $this->t('No duplicate modules detected. All modules are installed in single locations.'),
        'success'
      )];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $locations = $details['locations'] ?? [];
      $paths = array_column($locations, 'path');

      $description = (string) $this->t('Found in @count locations:', ['@count' => count($paths)]);
      $description .= '<ul>';
      foreach ($locations as $loc) {
        $description .= '<li><code>' . htmlspecialchars($loc['path'] ?? '', ENT_QUOTES, 'UTF-8') . '</code>';
        if (!empty($loc['version'])) {
          $description .= ' (v' . htmlspecialchars($loc['version'], ENT_QUOTES, 'UTF-8') . ')';
        }
        $description .= '</li>';
      }
      $description .= '</ul>';

      $issues[] = $this->ui->issue([
        'severity' => 'error',
        'code' => $item['code'] ?? 'DUPLICATE_MODULE',
        'label' => $details['module'] ?? '',
        'file' => $details['module'] ?? '',
        'description' => $description,
        'tags' => ['integrity'],
        'custom_data' => [
          'module' => $details['module'] ?? '',
          'count' => (string) count($paths),
        ],
      ]);
    }

    return [
      'list' => $this->ui->issueList(
        $issues,
        (string) $this->t('No issues match the selected filters.')
      ),
    ];
  }

  /**
   * Builds faceted list for UI modules issues.
   */
  protected function buildUiModulesIssues(array $files, array $config): array {
    $results = $files['ui_modules']['results'] ?? [];
    $is_ignored = !empty($config['ignore_ui_modules']);

    if (empty($results)) {
      return ['message' => $this->ui->message(
        (string) $this->t('No administration UI modules are active. Great for production performance!'),
        'success'
      )];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $severity = $is_ignored ? 'notice' : ($item['severity'] ?? 'warning');
      $package = $details['package'] ?? 'Other';

      $issues[] = $this->ui->issue([
        'severity' => $severity,
        'code' => $item['code'] ?? 'UI_MODULE_ACTIVE',
        'label' => $details['label'] ?? $details['module'] ?? '',
        'file' => $details['module'] ?? '',
        'description' => $details['description'] ?? $item['message'] ?? '',
        'tags' => ['performance'],
        'custom_data' => [
          'module' => $details['module'] ?? '',
          'package' => $package,
        ],
      ]);
    }

    $custom_filters = [
      'package' => [
        'label' => (string) $this->t('Package'),
        'attribute' => 'data-package',
      ],
    ];

    $content = [];

    if ($is_ignored) {
      $content['message'] = $this->ui->message(
        (string) $this->t('UI module warnings are ignored per configuration.'),
        'info'
      );
    }

    $content['list'] = $this->ui->issueList(
      $issues,
      (string) $this->t('No issues match the selected filters.'),
      $custom_filters
    );

    return $content;
  }

  /**
   * Builds faceted list for missing required modules issues.
   */
  protected function buildRecommendationsIssues(array $files): array {
    // Use the pre-filtered recommendations_issues file.
    $results = $files['recommendations_issues']['results'] ?? [];

    if (empty($results)) {
      return ['message' => $this->ui->message(
        (string) $this->t('All required modules are installed and active.'),
        'success'
      )];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $priority = $details['priority'] ?? 'recommended';
      $category = $details['category'] ?? 'other';
      $severity = $item['severity'] ?? 'warning';

      $issues[] = $this->ui->issue([
        'severity' => $severity,
        'code' => $item['code'] ?? 'RECOMMENDATION_MISSING',
        'label' => $details['label'] ?? $details['module'] ?? '',
        'file' => $details['module'] ?? '',
        'description' => $details['description'] ?? $item['message'] ?? '',
        'tags' => [$category, $priority],
        'custom_data' => [
          'category' => $category,
          'priority' => $priority,
          'module' => $details['module'] ?? '',
        ],
      ]);
    }

    $custom_filters = [
      'category' => [
        'label' => (string) $this->t('Category'),
        'attribute' => 'data-category',
      ],
      'priority' => [
        'label' => (string) $this->t('Priority'),
        'attribute' => 'data-priority',
      ],
    ];

    return [
      'list' => $this->ui->issueList(
        $issues,
        (string) $this->t('No issues match the selected filters.'),
        $custom_filters
      ),
    ];
  }

  /**
   * Builds table for recommendations analysis (informational).
   */
  protected function buildRecommendationsAnalysis(array $files): array {
    $results = $files['recommendations']['results'] ?? [];

    if (empty($results)) {
      return ['message' => $this->ui->message(
        (string) $this->t('No module recommendations configured for this site.'),
        'info'
      )];
    }

    // Group by category.
    $by_category = [];
    foreach ($results as $item) {
      $category = $item['details']['category'] ?? 'other';
      $by_category[$category][] = $item;
    }

    // Category order and labels.
    $categories = [
      'cache' => (string) $this->t('Cache'),
      'performance' => (string) $this->t('Performance'),
      'security' => (string) $this->t('Security'),
      'seo' => (string) $this->t('SEO'),
    ];

    $content = [];

    foreach ($categories as $category => $label) {
      if (empty($by_category[$category])) {
        continue;
      }

      $headers = [
        $this->ui->header((string) $this->t('Status'), 'center'),
        (string) $this->t('Module'),
        (string) $this->t('Description'),
        (string) $this->t('Priority'),
      ];

      $rows = [];
      foreach ($by_category[$category] as $item) {
        $details = $item['details'] ?? [];
        $installed = $details['installed'] ?? FALSE;
        $priority = $details['priority'] ?? 'recommended';

        $status_badge = $installed
          ? $this->ui->badge((string) $this->t('Installed'), 'success')
          : $this->ui->badge((string) $this->t('Not installed'), 'warning');

        $priority_badge = match ($priority) {
          'required' => $this->ui->badge((string) $this->t('Required'), 'error'),
          'recommended' => $this->ui->badge((string) $this->t('Recommended'), 'warning'),
          default => $this->ui->badge((string) $this->t('Optional'), 'info'),
        };

        $rows[] = $this->ui->row([
          $status_badge,
          $this->ui->itemName($details['label'] ?? '', $details['module'] ?? ''),
          $details['description'] ?? '',
          $priority_badge,
        ], $installed ? NULL : 'warning');
      }

      $content[$category] = $this->ui->section(
        $label,
        $this->ui->table($headers, $rows),
        ['open' => FALSE, 'count' => count($by_category[$category])]
      );
    }

    return $content;
  }

  /**
   * Builds faceted list for removable modules issues.
   */
  protected function buildRemovableModulesIssues(array $files): array {
    $results = $files['removable']['results'] ?? [];

    if (empty($results)) {
      return ['message' => $this->ui->message(
        (string) $this->t('No removable modules detected. All modules in the codebase are either active or are core modules.'),
        'success'
      )];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $type = $details['type'] ?? 'contrib';

      $issues[] = $this->ui->issue([
        'severity' => 'warning',
        'code' => $item['code'] ?? 'MODULE_REMOVABLE',
        'label' => $details['label'] ?? $details['name'] ?? '',
        'file' => $details['path'] ?? '',
        'description' => (string) $this->t('This module exists in the codebase but is not enabled. Consider removing it to reduce codebase size.'),
        'tags' => ['cleanup'],
        'custom_data' => [
          'module' => $details['name'] ?? '',
          'type' => $type,
          'package' => $details['package'] ?? 'Other',
        ],
      ]);
    }

    $custom_filters = [
      'type' => [
        'label' => (string) $this->t('Type'),
        'attribute' => 'data-type',
      ],
      'package' => [
        'label' => (string) $this->t('Package'),
        'attribute' => 'data-package',
      ],
    ];

    return [
      'list' => $this->ui->issueList(
        $issues,
        (string) $this->t('No issues match the selected filters.'),
        $custom_filters
      ),
    ];
  }

  /**
   * Builds faceted list for removable themes issues.
   */
  protected function buildRemovableThemesIssues(array $files): array {
    $results = $files['removable_themes']['results'] ?? [];

    if (empty($results)) {
      return ['message' => $this->ui->message(
        (string) $this->t('No removable themes detected. All themes in the codebase are either active or are core themes.'),
        'success'
      )];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $type = $details['type'] ?? 'contrib';

      $issues[] = $this->ui->issue([
        'severity' => 'notice',
        'code' => $item['code'] ?? 'THEME_REMOVABLE',
        'label' => $details['label'] ?? $details['name'] ?? '',
        'file' => $details['path'] ?? '',
        'description' => (string) $this->t('This theme exists in the codebase but is not enabled. Consider removing it to reduce codebase size.'),
        'tags' => ['cleanup'],
        'custom_data' => [
          'theme' => $details['name'] ?? '',
          'type' => $type,
        ],
      ]);
    }

    $custom_filters = [
      'type' => [
        'label' => (string) $this->t('Type'),
        'attribute' => 'data-type',
      ],
    ];

    return [
      'list' => $this->ui->issueList(
        $issues,
        (string) $this->t('No issues match the selected filters.'),
        $custom_filters
      ),
    ];
  }

  /**
   * Builds content for the active modules check.
   */
  protected function buildActiveModulesContent(array $files, array $data): array {
    $results = $files['active']['results'] ?? [];
    $module_counts = $data['module_counts'] ?? [];

    if (empty($results)) {
      return ['message' => $this->ui->message(
        (string) $this->t('No active modules found.'),
        'info'
      )];
    }

    // Group by type.
    $by_type = [];
    foreach ($results as $item) {
      $type = $item['details']['type'] ?? 'contrib';
      $by_type[$type][] = $item;
    }

    // Type order and labels.
    $types = [
      'custom' => (string) $this->t('Custom Modules'),
      'contrib' => (string) $this->t('Contributed Modules'),
      'core' => (string) $this->t('Core Modules'),
    ];

    $content = [];

    foreach ($types as $type => $label) {
      if (empty($by_type[$type])) {
        continue;
      }

      $headers = [
        (string) $this->t('Module'),
        (string) $this->t('Version'),
        (string) $this->t('Package'),
      ];

      $rows = [];
      foreach ($by_type[$type] as $item) {
        $details = $item['details'] ?? [];
        $rows[] = [
          $this->ui->itemName($details['label'] ?? '', $details['name'] ?? ''),
          $details['version'] ?? '-',
          $details['package'] ?? 'Other',
        ];
      }

      $content[$type] = $this->ui->section(
        $label . ' (' . count($by_type[$type]) . ')',
        $this->ui->table($headers, $rows),
        ['open' => FALSE, 'count' => count($by_type[$type])]
      );
    }

    return $content;
  }

  /**
   * Builds content for the active themes check.
   */
  protected function buildActiveThemesContent(array $files, array $data): array {
    $results = $files['active_themes']['results'] ?? [];

    if (empty($results)) {
      return ['message' => $this->ui->message(
        (string) $this->t('No active themes found.'),
        'info'
      )];
    }

    $headers = [
      (string) $this->t('Theme'),
      (string) $this->t('Version'),
      (string) $this->t('Type'),
      (string) $this->t('Status'),
    ];

    $rows = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $is_default = $details['is_default'] ?? FALSE;
      $is_admin = $details['is_admin'] ?? FALSE;

      $status = [];
      if ($is_default) {
        $status[] = $this->ui->badge((string) $this->t('Default'), 'success');
      }
      if ($is_admin) {
        $status[] = $this->ui->badge((string) $this->t('Admin'), 'info');
      }

      $rows[] = [
        $this->ui->itemName($details['label'] ?? '', $details['name'] ?? ''),
        $details['version'] ?? '-',
        $details['type'] ?? 'contrib',
        implode(' ', $status) ?: '-',
      ];
    }

    return ['table' => $this->ui->table($headers, $rows)];
  }

  /**
   * Builds faceted list for redundant active themes issues.
   */
  protected function buildRedundantThemesIssues(array $files): array {
    $results = $files['redundant_themes']['results'] ?? [];

    if (empty($results)) {
      return ['message' => $this->ui->message(
        (string) $this->t('No redundant themes detected. All active themes are being used.'),
        'success'
      )];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $type = $details['type'] ?? 'contrib';

      $issues[] = $this->ui->issue([
        'severity' => 'warning',
        'code' => $item['code'] ?? 'THEME_REDUNDANT',
        'label' => $details['label'] ?? $details['name'] ?? '',
        'file' => $details['path'] ?? '',
        'description' => (string) $this->t('This theme is active but not used as default, admin, or base theme. It consumes memory and adds maintenance overhead. Consider disabling it unless there is custom logic that switches themes dynamically.'),
        'tags' => ['maintenance', 'performance'],
        'custom_data' => [
          'theme' => $details['name'] ?? '',
          'type' => $type,
        ],
      ]);
    }

    $custom_filters = [
      'type' => [
        'label' => (string) $this->t('Type'),
        'attribute' => 'data-type',
      ],
    ];

    return [
      'list' => $this->ui->issueList(
        $issues,
        (string) $this->t('No issues match the selected filters.'),
        $custom_filters
      ),
    ];
  }

  /**
   * Filters results to include only actionable items for the summary.
   */
  protected function filterActionableResults(array $results): array {
    $actionable = [];
    $actionable_codes = ['RECOMMENDATION_MISSING', 'GROUPED_REQUIREMENT_MISSING'];

    foreach ($results as $result) {
      $severity = $result['severity'] ?? 'notice';

      if ($severity === 'error' || $severity === 'warning') {
        $actionable[] = $result;
        continue;
      }

      // Include notices for missing recommendations (not installed).
      if ($severity === 'notice') {
        $code = $result['code'] ?? '';
        $is_installed = $result['details']['installed'] ?? TRUE;

        if (in_array($code, $actionable_codes, TRUE) && !$is_installed) {
          $actionable[] = $result;
        }
      }
    }

    return $actionable;
  }

  /**
   * Analyzes active modules.
   */
  protected function analyzeActiveModules(): array {
    $results = [];

    foreach ($this->moduleHandler->getModuleList() as $name => $extension) {
      $info = $this->moduleExtensionList->getExtensionInfo($name);
      $path = $this->moduleExtensionList->getPath($name);

      $results[] = $this->createResultItem(
        'notice',
        'MODULE_ACTIVE',
        (string) $this->t('Module @name is active', ['@name' => $name]),
        [
          'name' => $name,
          'label' => $info['name'] ?? $name,
          'version' => $info['version'] ?? 'unknown',
          'package' => $info['package'] ?? 'Other',
          'type' => $this->getExtensionType($path, $info),
          'path' => $path,
        ]
      );
    }

    return $this->createResult($results, 0, 0, count($results));
  }

  /**
   * Analyzes modules that exist in code but are not enabled.
   */
  protected function analyzeRemovableModules(): array {
    $results = [];
    $warnings = 0;

    $all_modules = $this->moduleExtensionList->getList();
    $enabled_modules = $this->moduleHandler->getModuleList();
    $enabled_names = array_keys($enabled_modules);

    // Build a map of ALL module paths (active and inactive) to detect submodules.
    $all_module_paths = [];
    foreach ($all_modules as $name => $extension) {
      $all_module_paths[$name] = $extension->getPath();
    }

    // First pass: collect all candidate modules with their paths.
    $candidates = [];
    foreach ($all_modules as $name => $extension) {
      if (isset($enabled_modules[$name])) {
        continue;
      }

      $info = $extension->info;
      $path = $extension->getPath();

      if (!empty($info['hidden']) || str_contains($name, '_test')) {
        continue;
      }

      $type = $this->getExtensionType($path, $info);

      if ($type === 'core') {
        continue;
      }

      if ($this->isSubmoduleOfActiveModule($info['dependencies'] ?? [], $enabled_names, $all_modules)) {
        continue;
      }

      $candidates[$name] = [
        'info' => $info,
        'path' => $path,
        'type' => $type,
      ];
    }

    // Second pass: filter out submodules by comparing against ALL module paths.
    $parent_modules = $this->filterSubmodules($candidates, $all_module_paths);

    foreach ($parent_modules as $name => $data) {
      $results[] = $this->createResultItem(
        'warning',
        'MODULE_REMOVABLE',
        (string) $this->t('Module @name is not enabled and can be removed from the codebase.', ['@name' => $name]),
        [
          'name' => $name,
          'label' => $data['info']['name'] ?? $name,
          'version' => $data['info']['version'] ?? 'unknown',
          'package' => $data['info']['package'] ?? 'Other',
          'type' => $data['type'],
          'path' => $data['path'],
        ]
      );
      $warnings++;
    }

    return $this->createResult($results, 0, $warnings, 0);
  }

  /**
   * Filters out submodules from a list of candidate modules.
   *
   * A submodule is detected when its path is contained within another module's path.
   * For example, 'modules/contrib/metatag/metatag_views' is a submodule of
   * 'modules/contrib/metatag'.
   *
   * @param array $candidates
   *   Array of candidate modules keyed by name, with 'path', 'info', and 'type'.
   * @param array $all_module_paths
   *   Array of ALL module paths (active and inactive) keyed by module name.
   *
   * @return array
   *   Filtered array containing only parent modules (not submodules).
   */
  protected function filterSubmodules(array $candidates, array $all_module_paths): array {
    // For each candidate, check if its path is contained within ANY module's path.
    $parent_modules = [];
    foreach ($candidates as $name => $data) {
      $module_path = $data['path'];
      $is_submodule = FALSE;

      foreach ($all_module_paths as $other_name => $other_path) {
        // Skip comparing with itself.
        if ($name === $other_name) {
          continue;
        }

        // Check if this module's path starts with another module's path.
        // Add trailing slash to avoid false positives (e.g., 'metatag' vs 'metatag_views').
        if (str_starts_with($module_path . '/', $other_path . '/')) {
          // This module's path is inside another module's path - it's a submodule.
          $is_submodule = TRUE;
          break;
        }
      }

      if (!$is_submodule) {
        $parent_modules[$name] = $data;
      }
    }

    return $parent_modules;
  }

  /**
   * Checks if a module is a submodule of an active non-core module.
   */
  protected function isSubmoduleOfActiveModule(array $dependencies, array $enabled_names, array $all_modules): bool {
    foreach ($dependencies as $dependency) {
      if (str_starts_with($dependency, 'drupal:')) {
        continue;
      }

      $dep_name = $this->parseDependencyName($dependency);

      if (!in_array($dep_name, $enabled_names, TRUE) || !isset($all_modules[$dep_name])) {
        continue;
      }

      $dep_path = $all_modules[$dep_name]->getPath();
      if ($this->getExtensionType($dep_path, $all_modules[$dep_name]->info ?? []) !== 'core') {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Parses a dependency string to extract the module name.
   */
  protected function parseDependencyName(string $dependency): string {
    if (str_contains($dependency, ':')) {
      $dependency = explode(':', $dependency, 2)[1] ?? $dependency;
    }

    if (str_contains($dependency, ' ')) {
      $dependency = explode(' ', $dependency, 2)[0];
    }

    return trim($dependency);
  }

  /**
   * Analyzes active themes.
   */
  protected function analyzeActiveThemes(): array {
    $results = [];
    $themes = $this->themeHandler->listInfo();

    $theme_config = $this->configFactory->get('system.theme');
    $admin_theme = $theme_config->get('admin');
    $default_theme = $theme_config->get('default');

    // Build base theme map.
    $base_theme_map = [];
    foreach ($themes as $name => $theme) {
      $base = $theme->info['base theme'] ?? NULL;
      if ($base) {
        $base_theme_map[$base][] = $name;
      }
    }

    foreach ($themes as $name => $theme) {
      $info = $theme->info ?? [];
      $path = $theme->getPath();

      $is_base_for = $base_theme_map[$name] ?? [];
      $is_admin = ($name === $admin_theme);
      $is_default = ($name === $default_theme);

      $results[] = $this->createResultItem(
        'notice',
        'THEME_ACTIVE',
        (string) $this->t('Theme @name is active', ['@name' => $name]),
        [
          'name' => $name,
          'label' => $info['name'] ?? $name,
          'version' => $info['version'] ?? 'unknown',
          'package' => $info['package'] ?? 'Other',
          'type' => $this->getExtensionType($path, $info),
          'path' => $path,
          'base_theme' => $info['base theme'] ?? NULL,
          'is_base_for' => $is_base_for,
          'is_admin_theme' => $is_admin,
          'is_default_theme' => $is_default,
        ]
      );
    }

    return $this->createResult($results, 0, 0, count($themes));
  }

  /**
   * Analyzes themes that are active but not used.
   */
  protected function analyzeRedundantThemes(): array {
    $results = [];
    $warnings = 0;
    $themes = $this->themeHandler->listInfo();

    $theme_config = $this->configFactory->get('system.theme');
    $admin_theme = $theme_config->get('admin');
    $default_theme = $theme_config->get('default');

    // Build base theme map.
    $base_theme_map = [];
    foreach ($themes as $name => $theme) {
      $base = $theme->info['base theme'] ?? NULL;
      if ($base) {
        $base_theme_map[$base][] = $name;
      }
    }

    foreach ($themes as $name => $theme) {
      $info = $theme->info ?? [];
      $path = $theme->getPath();

      $is_base_for = $base_theme_map[$name] ?? [];
      $is_admin = ($name === $admin_theme);
      $is_default = ($name === $default_theme);

      // Warn about themes that are active but not used.
      if (empty($is_base_for) && !$is_admin && !$is_default) {
        $results[] = $this->createResultItem(
          'warning',
          'THEME_REDUNDANT',
          (string) $this->t('Theme @name is active but not used. Consider disabling.', ['@name' => $name]),
          [
            'name' => $name,
            'label' => $info['name'] ?? $name,
            'type' => $this->getExtensionType($path, $info),
            'path' => $path,
          ]
        );
        $warnings++;
      }
    }

    return $this->createResult($results, 0, $warnings, 0);
  }

  /**
   * Analyzes themes that exist in code but are not enabled.
   */
  protected function analyzeRemovableThemes(): array {
    $results = [];
    $notices = 0;

    $all_themes = $this->themeExtensionList->getList();
    $enabled_themes = $this->themeHandler->listInfo();

    foreach ($all_themes as $name => $extension) {
      if (isset($enabled_themes[$name])) {
        continue;
      }

      $info = $extension->info;
      $path = $extension->getPath();

      if (!empty($info['hidden'])) {
        continue;
      }

      $type = $this->getExtensionType($path, $info);
      if ($type === 'core') {
        continue;
      }

      $results[] = $this->createResultItem(
        'notice',
        'THEME_REMOVABLE',
        (string) $this->t('Theme @name exists but is not enabled. Consider removing.', ['@name' => $name]),
        [
          'name' => $name,
          'label' => $info['name'] ?? $name,
          'version' => $info['version'] ?? 'unknown',
          'package' => $info['package'] ?? 'Other',
          'type' => $type,
          'path' => $path,
          'base_theme' => $info['base theme'] ?? NULL,
          'is_admin_theme' => FALSE,
          'is_default_theme' => FALSE,
          'is_base_for' => [],
        ]
      );
      $notices++;
    }

    return $this->createResult($results, 0, 0, $notices);
  }

  /**
   * Analyzes module recommendations based on configuration.
   */
  protected function analyzeRecommendations(string $user_type, bool $shared_hosting, array $ignored_categories): array {
    $results = [];
    $errors = 0;
    $notices = 0;

    // Process grouped requirements.
    foreach (self::GROUPED_REQUIREMENTS as $group_id => $group) {
      if ($this->shouldSkipRecommendation($group, $user_type, $shared_hosting, $ignored_categories)) {
        continue;
      }

      $installed_module = $this->findInstalledModule($group['modules']);
      $is_installed = ($installed_module !== NULL);
      $is_required = ($group['priority'] === 'required');

      if ($is_installed) {
        // Installed modules use 'info' severity (doesn't count as issue).
        $results[] = $this->createResultItem(
          'info',
          'GROUPED_REQUIREMENT_MET',
          (string) $this->t('@label: @module is installed.', [
            '@label' => $group['label'],
            '@module' => $installed_module,
          ]),
          $this->buildGroupDetails($group_id, $group, $installed_module, TRUE)
        );
      }
      else {
        // Missing: required = error (affects score), recommended = notice (informational).
        $severity = $is_required ? 'error' : 'notice';
        $results[] = $this->createResultItem(
          $severity,
          'GROUPED_REQUIREMENT_MISSING',
          (string) $this->t('@label is required but not installed.', ['@label' => $group['label']]),
          $this->buildGroupDetails($group_id, $group, NULL, FALSE)
        );
        $is_required ? $errors++ : $notices++;
      }
    }

    // Process individual recommendations.
    foreach (self::MODULE_RECOMMENDATIONS as $module_name => $rec) {
      if ($this->shouldSkipRecommendation($rec, $user_type, $shared_hosting, $ignored_categories)) {
        continue;
      }

      $is_installed = $this->moduleHandler->moduleExists($module_name);
      $is_required = ($rec['priority'] === 'required');

      if ($is_installed) {
        // Installed modules use 'info' severity (doesn't count as issue).
        $results[] = $this->createResultItem(
          'info',
          'RECOMMENDATION_MET',
          (string) $this->t('@label is installed.', ['@label' => $rec['label']]),
          $this->buildModuleDetails($module_name, $rec, TRUE)
        );
      }
      else {
        // Missing: required = error (affects score), recommended = notice (informational).
        $severity = $is_required ? 'error' : 'notice';
        $message = $is_required
          ? (string) $this->t('@label is required but not installed.', ['@label' => $rec['label']])
          : (string) $this->t('@label is recommended.', ['@label' => $rec['label']]);

        $results[] = $this->createResultItem(
          $severity,
          'RECOMMENDATION_MISSING',
          $message,
          $this->buildModuleDetails($module_name, $rec, FALSE)
        );
        $is_required ? $errors++ : $notices++;
      }
    }

    return $this->createResult($results, $errors, 0, $notices);
  }

  /**
   * Builds a file with only missing recommendations for correct counter display.
   *
   * @param array $recommendations
   *   The full recommendations analysis results.
   *
   * @return array
   *   Filtered results with only missing modules and correct severity counts.
   */
  protected function buildMissingRecommendationsFile(array $recommendations): array {
    $results = [];
    $errors = 0;
    $notices = 0;

    foreach ($recommendations['results'] ?? [] as $item) {
      $is_installed = $item['details']['installed'] ?? TRUE;

      // Only include missing (not installed) recommendations.
      if ($is_installed) {
        continue;
      }

      $priority = $item['details']['priority'] ?? 'recommended';
      // Required = error (affects score), recommended = notice (informational).
      $severity = $priority === 'required' ? 'error' : 'notice';

      $results[] = [
        'severity' => $severity,
        'code' => $item['code'] ?? 'RECOMMENDATION_MISSING',
        'message' => $item['message'] ?? '',
        'details' => $item['details'] ?? [],
      ];

      if ($severity === 'error') {
        $errors++;
      }
      else {
        $notices++;
      }
    }

    return $this->createResult($results, $errors, 0, $notices);
  }

  /**
   * Analyzes active modules that are not recommended for production.
   *
   * @return array
   *   Analysis results with errors and warnings for problematic modules.
   */
  protected function analyzeNotRecommendedModules(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;

    foreach (self::NOT_RECOMMENDED_MODULES as $module_name => $config) {
      if (!$this->moduleHandler->moduleExists($module_name)) {
        continue;
      }

      $severity = $config['severity'] ?? 'warning';
      $category_labels = [
        'development' => $this->t('Development module'),
        'security_risk' => $this->t('Security risk'),
      ];

      $category_label = $category_labels[$config['category']] ?? $config['category'];

      $message = $severity === 'error'
        ? (string) $this->t('@label is active - @category. Must be disabled immediately.', [
            '@label' => $config['label'],
            '@category' => $category_label,
          ])
        : (string) $this->t('@label is active - @category. Should be disabled in production.', [
            '@label' => $config['label'],
            '@category' => $category_label,
          ]);

      $results[] = $this->createResultItem(
        $severity,
        'NOT_RECOMMENDED_MODULE',
        $message,
        [
          'module' => $module_name,
          'label' => $config['label'],
          'description' => $config['description'],
          'category' => $config['category'],
          'severity' => $severity,
        ]
      );

      if ($severity === 'error') {
        $errors++;
      }
      else {
        $warnings++;
      }
    }

    return $this->createResult($results, $errors, $warnings, 0);
  }

  /**
   * Analyzes active UI modules that are not needed in production.
   *
   * Detects modules with '_ui' or 'ui_' in their machine name, which are
   * typically administration interfaces not needed in production environments.
   *
   * @param bool $ignore_warnings
   *   If TRUE, report as notices instead of warnings (don't affect score).
   *
   * @return array
   *   Analysis results.
   */
  protected function analyzeUiModules(bool $ignore_warnings): array {
    $results = [];
    $warnings = 0;
    $notices = 0;

    // Patterns to detect UI modules.
    $ui_patterns = ['_ui', 'ui_'];

    // Modules to exclude from UI detection (false positives).
    $excluded_modules = [
      'jquery_ui',
      'jquery_ui_autocomplete',
      'jquery_ui_datepicker',
      'jquery_ui_draggable',
      'jquery_ui_droppable',
      'jquery_ui_menu',
      'jquery_ui_slider',
      'jquery_ui_touch_punch',
      'js_cookie',
    ];

    foreach ($this->moduleHandler->getModuleList() as $name => $extension) {
      // Skip excluded modules.
      if (in_array($name, $excluded_modules, TRUE)) {
        continue;
      }

      // Check if module name contains UI patterns.
      $is_ui_module = FALSE;
      foreach ($ui_patterns as $pattern) {
        if (str_contains($name, $pattern)) {
          $is_ui_module = TRUE;
          break;
        }
      }

      if (!$is_ui_module) {
        continue;
      }

      $info = $this->moduleExtensionList->getExtensionInfo($name);
      $severity = $ignore_warnings ? 'notice' : 'warning';

      $message = $ignore_warnings
        ? (string) $this->t('@label is an administration UI module.', [
            '@label' => $info['name'] ?? $name,
          ])
        : (string) $this->t('@label is an administration UI module - not needed in production.', [
            '@label' => $info['name'] ?? $name,
          ]);

      $results[] = $this->createResultItem(
        $severity,
        'UI_MODULE_ACTIVE',
        $message,
        [
          'module' => $name,
          'label' => $info['name'] ?? $name,
          'description' => $info['description'] ?? '',
          'package' => $info['package'] ?? 'Other',
        ]
      );

      if ($ignore_warnings) {
        $notices++;
      }
      else {
        $warnings++;
      }
    }

    return $this->createResult($results, 0, $warnings, $notices);
  }

  /**
   * Detects duplicate modules in the codebase.
   *
   * A duplicate module is one that exists in multiple locations in the codebase.
   * This can happen when the same module is in both contrib and custom directories,
   * or when different versions exist in different locations.
   *
   * @return array
   *   The analysis results with duplicate module warnings.
   */
  protected function analyzeDuplicateModules(): array {
    $results = [];
    $errors = 0;

    // Scan all potential module directories.
    $module_locations = [];
    $scan_paths = [
      DRUPAL_ROOT . '/modules',
      DRUPAL_ROOT . '/sites/all/modules',
    ];

    // Add site-specific paths.
    $site_path = \Drupal::getContainer()->getParameter('site.path');
    if ($site_path) {
      $scan_paths[] = DRUPAL_ROOT . '/' . $site_path . '/modules';
    }

    foreach ($scan_paths as $scan_path) {
      if (!is_dir($scan_path)) {
        continue;
      }
      $this->scanForModules($scan_path, $module_locations);
    }

    // Find duplicates (modules that exist in multiple locations).
    foreach ($module_locations as $module_name => $locations) {
      if (count($locations) > 1) {
        // Get versions for each location.
        $location_details = [];
        foreach ($locations as $path) {
          $info_file = $path . '/' . $module_name . '.info.yml';
          $version = 'unknown';
          if (file_exists($info_file)) {
            $info = \Symfony\Component\Yaml\Yaml::parseFile($info_file);
            $version = $info['version'] ?? 'unknown';
          }
          $location_details[] = [
            'path' => str_replace(DRUPAL_ROOT . '/', '', $path),
            'version' => $version,
          ];
        }

        $results[] = $this->createResultItem(
          'error',
          'DUPLICATE_MODULE',
          (string) $this->t('Module @name exists in @count locations. This can cause unpredictable behavior and should be resolved.', [
            '@name' => $module_name,
            '@count' => count($locations),
          ]),
          [
            'module' => $module_name,
            'locations' => $location_details,
            'count' => count($locations),
          ]
        );
        $errors++;
      }
    }

    return $this->createResult($results, $errors, 0, 0);
  }

  /**
   * Recursively scans a directory for module directories.
   *
   * @param string $dir
   *   The directory to scan.
   * @param array &$module_locations
   *   Reference to array mapping module names to their locations.
   * @param int $depth
   *   Current recursion depth.
   */
  protected function scanForModules(string $dir, array &$module_locations, int $depth = 0): void {
    // Limit recursion depth to prevent infinite loops.
    if ($depth > 4) {
      return;
    }

    if (!is_dir($dir) || !is_readable($dir)) {
      return;
    }

    $items = @scandir($dir);
    if ($items === FALSE) {
      return;
    }

    foreach ($items as $item) {
      if ($item === '.' || $item === '..') {
        continue;
      }

      $path = $dir . '/' . $item;
      if (!is_dir($path)) {
        continue;
      }

      // Check if this directory contains a .info.yml file (making it a module).
      $info_file = $path . '/' . $item . '.info.yml';
      if (file_exists($info_file)) {
        // Found a module.
        $module_locations[$item][] = $path;
      }

      // Recurse into subdirectories (for nested modules like contrib/custom).
      $this->scanForModules($path, $module_locations, $depth + 1);
    }
  }

  /**
   * Checks if a recommendation should be skipped based on config.
   */
  protected function shouldSkipRecommendation(array $rec, string $user_type, bool $shared_hosting, array $ignored_categories): bool {
    $category = $rec['category'] ?? 'other';

    if (!empty($ignored_categories[$category])) {
      return TRUE;
    }

    if (!in_array($user_type, $rec['user_types'], TRUE)) {
      return TRUE;
    }

    if ($shared_hosting && !empty($rec['requires_dedicated_server'])) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Finds the first installed module from a list.
   */
  protected function findInstalledModule(array $modules): ?string {
    foreach ($modules as $module) {
      if ($this->moduleHandler->moduleExists($module)) {
        return $module;
      }
    }
    return NULL;
  }

  /**
   * Builds details array for grouped requirements.
   */
  protected function buildGroupDetails(string $group_id, array $group, ?string $installed_module, bool $is_installed): array {
    return [
      'group_id' => $group_id,
      'label' => $group['label'],
      'description' => $group['description'],
      'modules' => $group['modules'],
      'installed_module' => $installed_module,
      'priority' => $group['priority'],
      'category' => $group['category'],
      'installed' => $is_installed,
      'requires_dedicated_server' => $group['requires_dedicated_server'],
    ];
  }

  /**
   * Builds details array for module recommendations.
   */
  protected function buildModuleDetails(string $module_name, array $rec, bool $is_installed): array {
    return [
      'module' => $module_name,
      'label' => $rec['label'],
      'description' => $rec['description'],
      'priority' => $rec['priority'],
      'category' => $rec['category'],
      'installed' => $is_installed,
      'requires_dedicated_server' => $rec['requires_dedicated_server'],
    ];
  }

  /**
   * Determines the extension type from its path.
   */
  protected function getExtensionType(string $path, array $info): string {
    if (str_starts_with($path, 'core/')) {
      return 'core';
    }

    if (str_contains($path, '/contrib/') || str_starts_with($path, 'modules/contrib/') || str_starts_with($path, 'themes/contrib/')) {
      return 'contrib';
    }

    if (str_contains($path, '/custom/') || str_starts_with($path, 'modules/custom/') || str_starts_with($path, 'themes/custom/')) {
      return 'custom';
    }

    if (!empty($info['project'])) {
      return 'contrib';
    }

    return 'custom';
  }

  /**
   * Gets module counts by type.
   */
  protected function getModuleCounts(array $active_results): array {
    return $this->countByType($active_results['results'] ?? []);
  }

  /**
   * Gets theme counts by type.
   */
  protected function getThemeCounts(array $active_results): array {
    $items = array_filter(
      $active_results['results'] ?? [],
      fn($item) => ($item['code'] ?? '') === 'THEME_ACTIVE'
    );
    return $this->countByType($items);
  }

  /**
   * Counts items by extension type.
   */
  protected function countByType(array $items): array {
    $counts = ['core' => 0, 'contrib' => 0, 'custom' => 0, 'total' => 0];

    foreach ($items as $item) {
      $type = $item['details']['type'] ?? 'custom';
      $counts[$type]++;
      $counts['total']++;
    }

    return $counts;
  }

  /**
   * Calculates scores for all factors.
   */
  protected function calculateScores(
    array $removable_modules,
    array $removable_themes,
    array $redundant_themes,
    array $recommendations,
    array $not_recommended,
    array $ui_modules,
    bool $ignore_ui_modules,
    array $duplicate_modules = [],
  ): array {
    // 1. Removable modules score.
    $removable_modules_count = count($removable_modules['results'] ?? []);
    $removable_modules_score = max(70, 100 - ($removable_modules_count * 3));

    // 2. Removable themes score.
    $removable_themes_count = count($removable_themes['results'] ?? []);
    $removable_themes_score = max(70, 100 - ($removable_themes_count * 5));

    // 3. Redundant themes score (active but not used).
    // Each redundant theme = -15 points (memory/maintenance overhead).
    $redundant_themes_count = count($redundant_themes['results'] ?? []);
    $redundant_themes_score = max(50, 100 - ($redundant_themes_count * 15));

    // 4. Recommended modules score.
    $required_missing = $recommendations['summary']['errors'] ?? 0;
    $rec_score = $required_missing === 0 ? 100 : max(0, 100 - ($required_missing * 20));

    // 5. Not recommended modules score.
    // Errors (critical dev modules like devel, kint) = -25 points each.
    // Warnings (less critical) = -15 points each.
    $not_rec_errors = $not_recommended['summary']['errors'] ?? 0;
    $not_rec_warnings = $not_recommended['summary']['warnings'] ?? 0;
    $not_rec_score = max(0, 100 - ($not_rec_errors * 25) - ($not_rec_warnings * 15));
    $not_rec_total = $not_rec_errors + $not_rec_warnings;

    // 6. UI modules score (only affects score if not ignored).
    $ui_count = count($ui_modules['results'] ?? []);
    // If ignored, UI modules don't affect the score (100%).
    // If not ignored, each UI module deducts 10 points.
    $ui_score = $ignore_ui_modules ? 100 : max(0, 100 - ($ui_count * 10));

    // 7. Duplicate modules score.
    $duplicate_count = $duplicate_modules['summary']['errors'] ?? 0;
    // Duplicate modules are critical - 30 points per duplicate.
    $duplicate_score = max(0, 100 - ($duplicate_count * 30));

    $factors = [
      'removable_modules' => [
        'score' => $removable_modules_score,
        'weight' => self::SCORE_WEIGHTS['removable_modules'],
        'label' => (string) $this->t('Removable Modules'),
        'description' => $removable_modules_count === 0
          ? (string) $this->t('No unused modules in codebase')
          : (string) $this->t('@count unused module(s) could be removed', ['@count' => $removable_modules_count]),
      ],
      'removable_themes' => [
        'score' => $removable_themes_score,
        'weight' => self::SCORE_WEIGHTS['removable_themes'],
        'label' => (string) $this->t('Removable Themes'),
        'description' => $removable_themes_count === 0
          ? (string) $this->t('No unused themes in codebase')
          : (string) $this->t('@count unused theme(s) could be removed', ['@count' => $removable_themes_count]),
      ],
      'redundant_themes' => [
        'score' => $redundant_themes_score,
        'weight' => self::SCORE_WEIGHTS['redundant_themes'],
        'label' => (string) $this->t('Redundant Themes'),
        'description' => $redundant_themes_count === 0
          ? (string) $this->t('All active themes are being used')
          : (string) $this->t('@count active theme(s) not in use - consider disabling', ['@count' => $redundant_themes_count]),
      ],
      'recommended_modules' => [
        'score' => $rec_score,
        'weight' => self::SCORE_WEIGHTS['recommended_modules'],
        'label' => (string) $this->t('Required Modules'),
        'description' => $required_missing === 0
          ? (string) $this->t('All required modules are installed')
          : (string) $this->t('@count required module(s) missing', ['@count' => $required_missing]),
      ],
      'not_recommended' => [
        'score' => $not_rec_score,
        'weight' => self::SCORE_WEIGHTS['not_recommended'],
        'label' => (string) $this->t('Production Readiness'),
        'description' => $not_rec_total === 0
          ? (string) $this->t('No development or risky modules detected')
          : (string) $this->t('@errors critical, @warnings risky modules active', [
              '@errors' => $not_rec_errors,
              '@warnings' => $not_rec_warnings,
            ]),
      ],
      'ui_modules' => [
        'score' => $ui_score,
        'weight' => self::SCORE_WEIGHTS['ui_modules'],
        'label' => (string) $this->t('UI Modules'),
        'description' => $ui_count === 0
          ? (string) $this->t('No administration UI modules detected')
          : ($ignore_ui_modules
              ? (string) $this->t('@count UI module(s) active (ignored)', ['@count' => $ui_count])
              : (string) $this->t('@count UI module(s) active - consider disabling', ['@count' => $ui_count])),
      ],
      'duplicate_modules' => [
        'score' => $duplicate_score,
        'weight' => self::SCORE_WEIGHTS['duplicate_modules'],
        'label' => (string) $this->t('Module Integrity'),
        'description' => $duplicate_count === 0
          ? (string) $this->t('No duplicate modules detected')
          : (string) $this->t('@count duplicate module(s) found - critical issue', ['@count' => $duplicate_count]),
      ],
    ];

    return [
      'factors' => $factors,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $config): array {
    return [
      'shared_hosting' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Shared hosting (limited server access)'),
        '#description' => $this->t('Check this if you are on shared hosting with limited server access. This will skip recommendations for advanced caching services (Memcache, Redis, Varnish) that require server-level installation.'),
        '#default_value' => $config['shared_hosting'] ?? FALSE,
      ],
      'ignore_ui_modules' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Ignore UI modules warnings'),
        '#description' => $this->t('By default, administration UI modules (views_ui, field_ui, webform_ui, etc.) generate warnings because they are not needed in production. Check this to ignore these warnings if your editors need these modules to manage content.'),
        '#default_value' => $config['ignore_ui_modules'] ?? FALSE,
      ],
      'ignore_cache' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Ignore cache recommendations'),
        '#description' => $this->t('Skip recommendations for caching modules (Page Cache, Dynamic Page Cache, Memcache, Redis, etc.).'),
        '#default_value' => $config['ignore_cache'] ?? FALSE,
      ],
      'ignore_performance' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Ignore performance recommendations'),
        '#description' => $this->t('Skip recommendations for performance modules (BigPipe, Lazy-load, AdvAgg, etc.).'),
        '#default_value' => $config['ignore_performance'] ?? FALSE,
      ],
      'ignore_security' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Ignore security recommendations'),
        '#description' => $this->t('Skip recommendations for security modules (SecKit, Password Policy, Flood Control, etc.).'),
        '#default_value' => $config['ignore_security'] ?? FALSE,
      ],
      'ignore_seo' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Ignore SEO recommendations'),
        '#description' => $this->t('Skip recommendations for SEO modules (Metatag, Pathauto, Simple Sitemap, etc.).'),
        '#default_value' => $config['ignore_seo'] ?? FALSE,
      ],
    ];
  }

  /**
   * Analyzes composer patches from composer.json.
   *
   * @return array
   *   The analysis results with patches information.
   */
  protected function analyzeComposerPatches(): array {
    $results = [];
    $notices = 0;

    // Find composer.json - typically in the project root (parent of DRUPAL_ROOT).
    $composer_paths = [
      DRUPAL_ROOT . '/../composer.json',
      DRUPAL_ROOT . '/composer.json',
    ];

    $composer_json = NULL;
    $composer_path = NULL;

    foreach ($composer_paths as $path) {
      $real_path = realpath($path);
      if ($real_path && file_exists($real_path)) {
        $content = file_get_contents($real_path);
        if ($content !== FALSE) {
          $composer_json = json_decode($content, TRUE);
          if (json_last_error() === JSON_ERROR_NONE) {
            $composer_path = $real_path;
            break;
          }
        }
      }
    }

    if ($composer_json === NULL) {
      $results[] = $this->createResultItem(
        'notice',
        'COMPOSER_NOT_FOUND',
        (string) $this->t('Could not find or parse composer.json file.'),
        ['searched_paths' => $composer_paths]
      );
      return $this->createResult($results, 0, 0, 1);
    }

    // Check for patches in different formats.
    // Format 1: cweagans/composer-patches - extra.patches.
    // Format 2: extra.patches-file pointing to external file.
    $patches = [];

    // Check for inline patches (extra.patches).
    if (!empty($composer_json['extra']['patches']) && is_array($composer_json['extra']['patches'])) {
      foreach ($composer_json['extra']['patches'] as $package => $package_patches) {
        if (!is_array($package_patches)) {
          continue;
        }
        foreach ($package_patches as $description => $patch_path) {
          $patches[] = [
            'package' => $package,
            'description' => $description,
            'path' => $patch_path,
            'source' => $this->determinePatchSource($patch_path),
          ];
        }
      }
    }

    // Check for patches-file reference.
    if (!empty($composer_json['extra']['patches-file'])) {
      $patches_file = $composer_json['extra']['patches-file'];
      $patches_file_path = dirname($composer_path) . '/' . $patches_file;

      if (file_exists($patches_file_path)) {
        $patches_content = file_get_contents($patches_file_path);
        if ($patches_content !== FALSE) {
          $patches_json = json_decode($patches_content, TRUE);
          if (json_last_error() === JSON_ERROR_NONE && !empty($patches_json['patches'])) {
            foreach ($patches_json['patches'] as $package => $package_patches) {
              if (!is_array($package_patches)) {
                continue;
              }
              foreach ($package_patches as $description => $patch_path) {
                $patches[] = [
                  'package' => $package,
                  'description' => $description,
                  'path' => $patch_path,
                  'source' => $this->determinePatchSource($patch_path),
                ];
              }
            }
          }
        }
      }
    }

    if (empty($patches)) {
      $results[] = $this->createResultItem(
        'notice',
        'NO_PATCHES',
        (string) $this->t('No composer patches detected.'),
        ['composer_path' => $composer_path]
      );
      return $this->createResult($results, 0, 0, 1);
    }

    // Create result items for each patch.
    foreach ($patches as $patch) {
      $results[] = $this->createResultItem(
        'notice',
        'COMPOSER_PATCH',
        (string) $this->t('Patch applied to @package: @description', [
          '@package' => $patch['package'],
          '@description' => $patch['description'],
        ]),
        [
          'package' => $patch['package'],
          'description' => $patch['description'],
          'path' => $patch['path'],
          'source' => $patch['source'],
          'source_label' => $patch['source'] === 'external' ?
            (string) $this->t('External URL') :
            (string) $this->t('Local file'),
        ]
      );
      $notices++;
    }

    $result = $this->createResult($results, 0, 0, $notices);
    $result['stats'] = [
      'total_patches' => count($patches),
      'external_patches' => count(array_filter($patches, fn($p) => $p['source'] === 'external')),
      'local_patches' => count(array_filter($patches, fn($p) => $p['source'] === 'local')),
      'packages_patched' => count(array_unique(array_column($patches, 'package'))),
    ];

    return $result;
  }

  /**
   * Determines if a patch path is external (URL) or local (file path).
   *
   * @param string $path
   *   The patch path.
   *
   * @return string
   *   Either 'external' or 'local'.
   */
  protected function determinePatchSource(string $path): string {
    if (str_starts_with($path, 'http://') || str_starts_with($path, 'https://')) {
      return 'external';
    }
    return 'local';
  }

  /**
   * Builds the composer patches content (table).
   *
   * @param array $files
   *   The analysis files.
   *
   * @return array
   *   Render array for the patches table.
   */
  protected function buildComposerPatchesContent(array $files): array {
    $data = $files['composer_patches'] ?? [];
    $results = $data['results'] ?? [];
    $stats = $data['stats'] ?? [];

    // Check if no patches found.
    $no_patches = count($results) === 1 &&
      isset($results[0]['code']) &&
      in_array($results[0]['code'], ['NO_PATCHES', 'COMPOSER_NOT_FOUND'], TRUE);

    if ($no_patches) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('No composer patches are applied to this project.'),
          'info'
        ),
      ];
    }

    // Build stats summary.
    $content = [];

    if (!empty($stats)) {
      $content['stats'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['audit-stats-summary']],
        'text' => [
          '#markup' => '<p><strong>' . $this->t('Summary:') . '</strong> ' .
            $this->t('@total patches applied to @packages package(s). @external external, @local local.', [
              '@total' => $stats['total_patches'] ?? 0,
              '@packages' => $stats['packages_patched'] ?? 0,
              '@external' => $stats['external_patches'] ?? 0,
              '@local' => $stats['local_patches'] ?? 0,
            ]) . '</p>',
        ],
      ];
    }

    // Build the table.
    $headers = [
      $this->ui->header((string) $this->t('Package')),
      $this->ui->header((string) $this->t('Description')),
      $this->ui->header((string) $this->t('Source')),
      $this->ui->header((string) $this->t('Path/URL')),
    ];

    $rows = [];
    foreach ($results as $item) {
      if (($item['code'] ?? '') !== 'COMPOSER_PATCH') {
        continue;
      }

      $details = $item['details'] ?? [];
      $is_external = ($details['source'] ?? 'local') === 'external';

      // For external URLs, make them clickable (truncated).
      $path = $details['path'] ?? '';
      if ($is_external) {
        $truncated_path = strlen($path) > 60 ? substr($path, 0, 57) . '...' : $path;
        $path_display = '<a href="' . htmlspecialchars($path) . '" target="_blank" title="' . htmlspecialchars($path) . '">' . htmlspecialchars($truncated_path) . '</a>';
      }
      else {
        $path_display = '<code>' . htmlspecialchars($path) . '</code>';
      }

      $source_display = $is_external ?
        '<span class="color-warning">' . $this->t('External URL') . '</span>' :
        '<span class="color-success">' . $this->t('Local file') . '</span>';

      $rows[] = $this->ui->row([
        $this->ui->cell('<code>' . htmlspecialchars($details['package'] ?? '') . '</code>'),
        $this->ui->cell(htmlspecialchars($details['description'] ?? '')),
        $this->ui->cell($source_display),
        $this->ui->cell($path_display),
      ]);
    }

    $content['table'] = $this->ui->table($headers, $rows);

    return $content;
  }

}
