<?php

declare(strict_types=1);

namespace Drupal\audit_performance\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Analyzes performance configuration and cache usage in custom code.
 *
 * Also detects production configuration issues (debug code, error display,
 * Twig debug) that impact both performance and security.
 */
#[AuditAnalyzer(
  id: 'performance',
  label: new TranslatableMarkup('Performance Audit'),
  description: new TranslatableMarkup('Analyzes cache configuration, CSS/JS aggregation, cache modules, production settings (error display, Twig debug), and detects debug code and cache issues in custom code.'),
  menu_title: new TranslatableMarkup('Performance'),
  output_directory: 'performance',
  weight: 4,
)]
class PerformanceAnalyzer extends AuditAnalyzerBase {

  /**
   * Score weights for different factors.
   *
   * Priority:
   * - Production settings (20%): Error display, Twig debug - critical for security/performance
   * - System config (25%): CSS/JS aggregation, page cache - critical for performance
   * - Cache modules (15%): Important but can be compensated
   * - Code issues (20%): Debug code + cache issues in custom code
   * - Performance patterns (20%): N+1 queries, saves in loops, S3/remote issues
   */
  protected const SCORE_WEIGHTS = [
    'production_settings' => 20,
    'system_config' => 25,
    'cache_modules' => 15,
    'code_issues' => 20,
    'performance_patterns' => 20,
  ];

  /**
   * Default recommended cache max-age in seconds (6 hours).
   */
  protected const DEFAULT_RECOMMENDED_CACHE_MAX_AGE = 21600;

  /**
   * Cache modules by user type.
   *
   * These are reminder checks - the actual recommendations are in audit_modules.
   */
  protected const CACHE_MODULES = [
    'anonymous' => [
      'page_cache' => [
        'label' => 'Internal Page Cache',
        'description' => 'Caches pages for anonymous users',
        'required' => TRUE,
      ],
      'dynamic_page_cache' => [
        'label' => 'Internal Dynamic Page Cache',
        'description' => 'Caches pages with placeholders for dynamic content',
        'required' => TRUE,
      ],
      'big_pipe' => [
        'label' => 'BigPipe',
        'description' => 'Streams page content for faster perceived performance',
        'required' => FALSE,
      ],
      'big_pipe_sessionless' => [
        'label' => 'BigPipe Sessionless',
        'description' => 'BigPipe for anonymous users without sessions',
        'required' => FALSE,
      ],
    ],
    'registered' => [
      'dynamic_page_cache' => [
        'label' => 'Internal Dynamic Page Cache',
        'description' => 'Caches pages with placeholders for dynamic content',
        'required' => TRUE,
      ],
      'big_pipe' => [
        'label' => 'BigPipe',
        'description' => 'Streams page content for faster perceived performance',
        'required' => TRUE,
      ],
    ],
    'mixed' => [
      'page_cache' => [
        'label' => 'Internal Page Cache',
        'description' => 'Caches pages for anonymous users',
        'required' => TRUE,
      ],
      'dynamic_page_cache' => [
        'label' => 'Internal Dynamic Page Cache',
        'description' => 'Caches pages with placeholders for dynamic content',
        'required' => TRUE,
      ],
      'big_pipe' => [
        'label' => 'BigPipe',
        'description' => 'Streams page content for faster perceived performance',
        'required' => TRUE,
      ],
      'big_pipe_sessionless' => [
        'label' => 'BigPipe Sessionless',
        'description' => 'BigPipe for anonymous users without sessions',
        'required' => FALSE,
      ],
    ],
  ];

  /**
   * Patterns that indicate cache problems in PHP code.
   *
   * Organized by category for better understanding of cache anti-patterns.
   */
  protected const CACHE_PROBLEM_PATTERNS = [
    // ==========================================================================
    // CRITICAL: Cache completely disabled
    // ==========================================================================
    'max_age_zero' => [
      'pattern' => '/[\'"]#cache[\'"]\s*=>\s*\[\s*[\'"]max-age[\'"]\s*=>\s*0\s*[,\]]/',
      'severity' => 'error',
      'code' => 'CACHE_MAX_AGE_ZERO',
      'message' => 'Cache max-age set to 0 disables caching completely',
      'recommendation' => 'Use lazy_builder for dynamic parts instead of disabling cache entirely',
    ],
    'cache_disable_assignment' => [
      'pattern' => '/\[\s*[\'"]#cache[\'"]\s*\]\s*\[\s*[\'"]max-age[\'"]\s*\]\s*=\s*0\s*;/',
      'severity' => 'error',
      'code' => 'CACHE_DISABLED_ASSIGNMENT',
      'message' => 'Cache disabled via direct assignment',
      'recommendation' => 'Avoid disabling cache; use cache contexts or lazy_builder instead',
    ],
    'kill_switch' => [
      'pattern' => '/page_cache_kill_switch[\'"]?\s*\)\s*->\s*trigger\s*\(/',
      'severity' => 'error',
      'code' => 'PAGE_CACHE_KILL_SWITCH',
      'message' => 'Page cache kill switch triggered - disables page caching for entire request',
      'recommendation' => 'Use cache contexts or placeholders instead of kill switch',
    ],
    'nocachepreview' => [
      'pattern' => '/[\'"]#cache[\'"]\s*=>\s*\[\s*[\'"]max-age[\'"]\s*=>\s*0.*?(?:preview|draft|unpublished)/i',
      'severity' => 'warning',
      'code' => 'CACHE_DISABLED_PREVIEW',
      'message' => 'Cache disabled for preview/draft - verify this is intentional',
      'recommendation' => 'Consider using cache contexts for preview mode instead',
    ],

    // ==========================================================================
    // WARNING: Generic cache tags (cause over-invalidation)
    // ==========================================================================
    'generic_node_list_tag' => [
      'pattern' => '/[\'"]tags[\'"]\s*=>\s*\[(?:[^\]]*,)?\s*[\'"]node_list[\'"]\s*(?:,|\])/',
      'severity' => 'warning',
      'code' => 'GENERIC_NODE_LIST_TAG',
      'message' => 'Using generic "node_list" tag invalidates cache for ALL node changes',
      'recommendation' => 'Use "node_list:{bundle}" (e.g., "node_list:article") for targeted invalidation',
    ],
    'generic_taxonomy_list_tag' => [
      'pattern' => '/[\'"]tags[\'"]\s*=>\s*\[(?:[^\]]*,)?\s*[\'"]taxonomy_term_list[\'"]\s*(?:,|\])/',
      'severity' => 'warning',
      'code' => 'GENERIC_TAXONOMY_LIST_TAG',
      'message' => 'Using generic "taxonomy_term_list" tag invalidates cache for ALL term changes',
      'recommendation' => 'Use "taxonomy_term_list:{vocabulary}" for targeted invalidation',
    ],
    'generic_user_list_tag' => [
      'pattern' => '/[\'"]tags[\'"]\s*=>\s*\[(?:[^\]]*,)?\s*[\'"]user_list[\'"]\s*(?:,|\])/',
      'severity' => 'warning',
      'code' => 'GENERIC_USER_LIST_TAG',
      'message' => 'Using generic "user_list" tag invalidates cache on ANY user change',
      'recommendation' => 'Consider more specific tags or cache contexts',
    ],
    'generic_config_tag' => [
      'pattern' => '/[\'"]tags[\'"]\s*=>\s*\[(?:[^\]]*,)?\s*[\'"]config[\'"]\s*(?:,|\])/',
      'severity' => 'warning',
      'code' => 'GENERIC_CONFIG_TAG',
      'message' => 'Using generic "config" tag invalidates on ANY config change',
      'recommendation' => 'Use specific config tag like "config:system.site"',
    ],
    'rendered_tag' => [
      'pattern' => '/[\'"]tags[\'"]\s*=>\s*\[(?:[^\]]*,)?\s*[\'"]rendered[\'"]\s*(?:,|\])/',
      'severity' => 'error',
      'code' => 'RENDERED_TAG_USED',
      'message' => 'Using "rendered" tag causes invalidation on almost every cache clear',
      'recommendation' => 'Use specific entity or config tags instead of "rendered"',
    ],

    // ==========================================================================
    // WARNING: Missing cache metadata
    // ==========================================================================
    'permanent_no_tags' => [
      'pattern' => '/[\'"]#cache[\'"]\s*=>\s*\[\s*[\'"]max-age[\'"]\s*=>\s*(?:Cache::PERMANENT|-1)\s*\]/',
      'severity' => 'warning',
      'code' => 'PERMANENT_CACHE_NO_TAGS',
      'message' => 'Permanent cache without tags - content may become stale',
      'recommendation' => 'Add appropriate cache tags for invalidation',
    ],
    'cache_no_contexts_user_data' => [
      'pattern' => '/(?:currentUser|current_user|getAccount)\(\).*?[\'"]#cache[\'"]\s*=>\s*\[[^\]]*\](?![^\]]*[\'"]contexts[\'"])/s',
      'severity' => 'warning',
      'code' => 'USER_DATA_NO_CONTEXT',
      'message' => 'User-specific data without "user" cache context',
      'recommendation' => 'Add cache context: "user" or "user.permissions"',
    ],

    // ==========================================================================
    // WARNING: Dynamic data without proper handling
    // ==========================================================================
    'time_function_in_render' => [
      'pattern' => '/(?:date|time|strtotime|DateTime|now)\s*\([^)]*\).*?[\'"]#markup[\'"]/',
      'severity' => 'warning',
      'code' => 'TIME_IN_RENDER',
      'message' => 'Time/date function near render array - may cause stale cached content',
      'recommendation' => 'Use #lazy_builder for dynamic time display',
    ],
    'request_in_render' => [
      'pattern' => '/(?:\\\\Drupal::request\(\)|->getRequest\(\)).*?(?:[\'"]#markup[\'"]|[\'"]#cache[\'"])/s',
      'severity' => 'warning',
      'code' => 'REQUEST_DATA_IN_RENDER',
      'message' => 'Request data used near render array without proper cache handling',
      'recommendation' => 'Add appropriate cache contexts (url, session, etc.)',
    ],
    'session_in_render' => [
      'pattern' => '/\$_SESSION|->getSession\(\).*?[\'"]#(?:markup|cache)[\'"]/',
      'severity' => 'warning',
      'code' => 'SESSION_DATA_IN_RENDER',
      'message' => 'Session data used in render without proper cache handling',
      'recommendation' => 'Use "session" cache context or #lazy_builder',
    ],

    // ==========================================================================
    // NOTICE: Patterns that need manual review
    // ==========================================================================
    'cache_set_permanent' => [
      'pattern' => '/->set\s*\([^)]+,\s*Cache::PERMANENT\s*[,\)]/',
      'severity' => 'notice',
      'code' => 'CACHE_PERMANENT',
      'message' => 'Using Cache::PERMANENT - ensure cache tags are properly set',
      'recommendation' => 'Verify that cache tags are provided for proper invalidation',
    ],
    'entity_query_no_tag' => [
      'pattern' => '/entityQuery\s*\(\s*[\'"](\w+)[\'"]\s*\)(?:(?!addTag|cache).)*?->execute\s*\(\s*\)/',
      'severity' => 'notice',
      'code' => 'ENTITY_QUERY_NO_CACHE_TAG',
      'message' => 'EntityQuery result may need cache tag for list invalidation',
      'recommendation' => 'Add "{entity_type}_list" or "{entity_type}_list:{bundle}" cache tag to render array',
    ],
    'markup_only' => [
      'pattern' => '/return\s*\[\s*[\'"]#markup[\'"]\s*=>\s*[^,\]]+\s*\]\s*;/',
      'severity' => 'notice',
      'code' => 'MARKUP_WITHOUT_CACHE',
      'message' => 'Render array with only #markup - missing cache metadata',
      'recommendation' => 'Add #cache with appropriate tags, contexts, and max-age',
    ],

    // ==========================================================================
    // WARNING: Improper cache bubbling
    // ==========================================================================
    'render_without_bubble' => [
      'pattern' => '/(?:node_view|entity_view|block_view)_alter.*?\$build\s*\[[\'"][^#]/',
      'severity' => 'notice',
      'code' => 'ALTER_WITHOUT_CACHE_BUBBLE',
      'message' => 'Alter hook modifying build array - verify cache metadata is bubbled',
      'recommendation' => 'Use CacheableMetadata::createFromRenderArray()->merge()->applyTo()',
    ],

    // ==========================================================================
    // WARNING: Known bad practices
    // ==========================================================================
    'drupal_static_reset_cache' => [
      'pattern' => '/drupal_static_reset\s*\(\s*\)/',
      'severity' => 'warning',
      'code' => 'DRUPAL_STATIC_RESET_ALL',
      'message' => 'Resetting all static caches can cause performance issues',
      'recommendation' => 'Reset only specific static caches by providing the function name',
    ],
    'cache_clear_all' => [
      'pattern' => '/\\\\Drupal::cache\(\)->deleteAll\(\)|cache_clear_all\s*\(/',
      'severity' => 'error',
      'code' => 'CACHE_CLEAR_ALL',
      'message' => 'Clearing all cache programmatically is very expensive',
      'recommendation' => 'Use Cache::invalidateTags() for targeted invalidation',
    ],
  ];

  /**
   * Patterns that indicate cache problems in Twig templates.
   */
  protected const TWIG_CACHE_PATTERNS = [
    'twig_cache_zero' => [
      'pattern' => '/\{\%\s*set\s+.*#cache.*max-age.*0/',
      'severity' => 'error',
      'code' => 'TWIG_CACHE_ZERO',
      'message' => 'Cache max-age 0 in Twig template',
      'recommendation' => 'Avoid disabling cache in templates; handle dynamic content in preprocess',
    ],
    'twig_raw_user_input' => [
      'pattern' => '/\{\{\s*(?:raw|content)\s*\}\}.*?(?:user|input|form)/i',
      'severity' => 'notice',
      'code' => 'TWIG_RAW_POSSIBLE_USER',
      'message' => 'Possible user input rendered - verify cache contexts',
      'recommendation' => 'Ensure user-specific content has proper cache contexts in preprocess',
    ],
  ];

  /**
   * Debug code patterns that should not be in production.
   *
   * These functions expose internal data, slow down execution,
   * and indicate code that was not properly reviewed before deployment.
   */
  protected const DEBUG_CODE_PATTERNS = [
    // ==========================================================================
    // CRITICAL: Devel module debug functions
    // ==========================================================================
    'dpm' => [
      'pattern' => '/\bdpm\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_DPM',
      'message' => 'dpm() debug function found - Devel module debug output',
      'recommendation' => 'Remove dpm() calls before deploying to production',
    ],
    'dsm' => [
      'pattern' => '/\bdsm\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_DSM',
      'message' => 'dsm() debug function found - Devel module debug output',
      'recommendation' => 'Remove dsm() calls before deploying to production',
    ],
    'dvm' => [
      'pattern' => '/\bdvm\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_DVM',
      'message' => 'dvm() debug function found - Devel module variable dump',
      'recommendation' => 'Remove dvm() calls before deploying to production',
    ],
    'dpr' => [
      'pattern' => '/\bdpr\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_DPR',
      'message' => 'dpr() debug function found - Devel module print_r output',
      'recommendation' => 'Remove dpr() calls before deploying to production',
    ],
    'dpq' => [
      'pattern' => '/\bdpq\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_DPQ',
      'message' => 'dpq() debug function found - Devel module query debug',
      'recommendation' => 'Remove dpq() calls before deploying to production',
    ],
    'dargs' => [
      'pattern' => '/\bdargs\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_DARGS',
      'message' => 'dargs() debug function found - Devel module arguments dump',
      'recommendation' => 'Remove dargs() calls before deploying to production',
    ],
    'ddebug_backtrace' => [
      'pattern' => '/\bddebug_backtrace\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_DDEBUG_BACKTRACE',
      'message' => 'ddebug_backtrace() found - Devel module backtrace output',
      'recommendation' => 'Remove ddebug_backtrace() calls before deploying to production',
    ],
    'kpr' => [
      'pattern' => '/\bkpr\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_KPR',
      'message' => 'kpr() debug function found - Devel Kint print_r output',
      'recommendation' => 'Remove kpr() calls before deploying to production',
    ],

    // ==========================================================================
    // CRITICAL: Kint debug functions
    // ==========================================================================
    'kint' => [
      'pattern' => '/\bkint\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_KINT',
      'message' => 'kint() debug function found - Kint debugger output',
      'recommendation' => 'Remove kint() calls before deploying to production',
    ],
    'ksm' => [
      'pattern' => '/\bksm\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_KSM',
      'message' => 'ksm() debug function found - Kint status message',
      'recommendation' => 'Remove ksm() calls before deploying to production',
    ],
    'd_function' => [
      'pattern' => '/\b[Kk]int::[dDs]\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_KINT_STATIC',
      'message' => 'Kint::d() or Kint::s() debug function found',
      'recommendation' => 'Remove Kint static calls before deploying to production',
    ],

    // ==========================================================================
    // CRITICAL: Symfony VarDumper / dd()
    // ==========================================================================
    'dd' => [
      'pattern' => '/\bdd\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_DD',
      'message' => 'dd() (dump and die) found - stops execution and dumps variables',
      'recommendation' => 'Remove dd() calls before deploying to production',
    ],
    'dump' => [
      'pattern' => '/\bdump\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_DUMP',
      'message' => 'dump() function found - Symfony VarDumper output',
      'recommendation' => 'Remove dump() calls before deploying to production',
    ],

    // ==========================================================================
    // WARNING: PHP native debug functions
    // ==========================================================================
    'var_dump' => [
      'pattern' => '/\bvar_dump\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_VAR_DUMP',
      'message' => 'var_dump() found - outputs raw variable data to browser',
      'recommendation' => 'Remove var_dump() calls; use proper logging instead',
    ],
    'print_r' => [
      'pattern' => '/\bprint_r\s*\([^,)]+\s*\)/',
      'severity' => 'error',
      'code' => 'DEBUG_PRINT_R',
      'message' => 'print_r() without return flag found - outputs to browser',
      'recommendation' => 'Remove print_r() calls or use with TRUE as second parameter for logging',
    ],
    'var_export_echo' => [
      'pattern' => '/(?:echo|print)\s+var_export\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_VAR_EXPORT_ECHO',
      'message' => 'var_export() with echo/print found - outputs to browser',
      'recommendation' => 'Remove debug output; use proper logging instead',
    ],
    'debug_print_backtrace' => [
      'pattern' => '/\bdebug_print_backtrace\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_PRINT_BACKTRACE',
      'message' => 'debug_print_backtrace() found - outputs call stack to browser',
      'recommendation' => 'Remove debug_print_backtrace() calls before deploying',
    ],
    'debug_backtrace_echo' => [
      'pattern' => '/(?:print_r|var_dump|var_export)\s*\(\s*debug_backtrace\s*\(/',
      'severity' => 'error',
      'code' => 'DEBUG_BACKTRACE_OUTPUT',
      'message' => 'debug_backtrace() output found - exposes call stack',
      'recommendation' => 'Remove backtrace debug output before deploying',
    ],

    // ==========================================================================
    // NOTICE: Patterns that might be intentional but should be reviewed
    // ==========================================================================
    'error_log_variable' => [
      'pattern' => '/error_log\s*\(\s*(?:print_r|var_export)\s*\(/',
      'severity' => 'notice',
      'code' => 'DEBUG_ERROR_LOG_DUMP',
      'message' => 'error_log() with variable dump found - verify this is intentional',
      'recommendation' => 'Consider using Drupal logger service instead of error_log()',
    ],
  ];

  /**
   * Debug patterns for Twig templates.
   */
  protected const TWIG_DEBUG_PATTERNS = [
    'twig_dump' => [
      'pattern' => '/\{\{\s*dump\s*\(/',
      'severity' => 'error',
      'code' => 'TWIG_DEBUG_DUMP',
      'message' => 'dump() in Twig template - debug output visible to users',
      'recommendation' => 'Remove {{ dump() }} from templates before deploying to production',
    ],
    'twig_kint' => [
      'pattern' => '/\{\{\s*kint\s*\(/',
      'severity' => 'error',
      'code' => 'TWIG_DEBUG_KINT',
      'message' => 'kint() in Twig template - debug output visible to users',
      'recommendation' => 'Remove {{ kint() }} from templates before deploying to production',
    ],
    'twig_devel_dump' => [
      'pattern' => '/\{\{\s*devel_dump\s*\(/',
      'severity' => 'error',
      'code' => 'TWIG_DEBUG_DEVEL',
      'message' => 'devel_dump() in Twig template - debug output visible to users',
      'recommendation' => 'Remove {{ devel_dump() }} from templates before deploying to production',
    ],
    'twig_print_var' => [
      'pattern' => '/\{\#\s*debug|\{\{\s*_context\s*\}\}/',
      'severity' => 'warning',
      'code' => 'TWIG_DEBUG_CONTEXT',
      'message' => 'Debug comment or _context dump in Twig template',
      'recommendation' => 'Remove debug comments and _context output before deploying',
    ],
  ];

  /**
   * Drupal performance anti-patterns that require context analysis.
   *
   * These patterns detect common performance issues in Drupal code that
   * cause N+1 queries, unnecessary database hits, S3/remote file problems,
   * and other performance bottlenecks.
   *
   * Note: Some patterns like N+1 queries and saves in loops require
   * multi-line context analysis and are handled by dedicated methods.
   */
  protected const DRUPAL_PERF_PATTERNS = [
    // ========================================================================
    // CRITICAL: N+1 Query Patterns (detected via context analysis)
    // ========================================================================
    'n1_load' => [
      'pattern' => '/->load\s*\(/',
      'requires_loop_context' => TRUE,
      'severity' => 'error',
      'code' => 'N1_ENTITY_LOAD',
      'message' => 'Entity load() inside a loop causes N+1 queries',
      'recommendation' => 'Use loadMultiple() before the loop to load all entities at once',
    ],
    'n1_referenced_entities' => [
      'pattern' => '/->referencedEntities\s*\(\s*\)/',
      'requires_loop_context' => TRUE,
      'severity' => 'error',
      'code' => 'N1_REFERENCED_ENTITIES',
      'message' => 'referencedEntities() inside a loop causes N+1 queries',
      'recommendation' => 'Pre-load referenced entities using EntityRepository or custom query',
    ],

    // ========================================================================
    // HIGH: Missing Range/Limit on Queries
    // ========================================================================
    'query_no_range' => [
      'pattern' => '/(?:entityQuery|getQuery)\s*\([^)]*\)/',
      'requires_no_range' => TRUE,
      'severity' => 'warning',
      'code' => 'QUERY_NO_RANGE',
      'message' => 'EntityQuery without range() may load unlimited results',
      'recommendation' => 'Add ->range(0, 50) or appropriate limit to prevent memory issues',
    ],
    'db_select_no_limit' => [
      'pattern' => '/(?:\\\\Drupal::database\(\)|->database\(\))->select\s*\(/',
      'requires_no_range' => TRUE,
      'severity' => 'warning',
      'code' => 'DB_SELECT_NO_LIMIT',
      'message' => 'Database select without range/limit may return unlimited rows',
      'recommendation' => 'Add ->range() or ->limit() to constrain result set',
    ],

    // ========================================================================
    // HIGH: Missing Access Check (Drupal 10 Requirement)
    // ========================================================================
    'missing_access_check' => [
      'pattern' => '/(?:entityQuery|getQuery)\s*\([^)]*\)/',
      'requires_no_access_check' => TRUE,
      'severity' => 'error',
      'code' => 'MISSING_ACCESS_CHECK',
      'message' => 'EntityQuery without accessCheck() - Drupal 10 requires explicit access check',
      'recommendation' => 'Add ->accessCheck(TRUE) or ->accessCheck(FALSE) with security justification',
    ],

    // ========================================================================
    // HIGH: Entity Save in Loop
    // ========================================================================
    'save_in_loop' => [
      'pattern' => '/->save\s*\(\s*\)/',
      'requires_loop_context' => TRUE,
      'severity' => 'warning',
      'code' => 'SAVE_IN_LOOP',
      'message' => 'Entity save() inside a loop - each save triggers hooks and cache invalidation',
      'recommendation' => 'Consider using batch operations or queue workers for bulk saves',
    ],

    // ========================================================================
    // S3/Remote File Operations (only reported when s3fs module is enabled)
    // ========================================================================
    'getimagesize_remote' => [
      'pattern' => '/getimagesize\s*\(\s*\$/',
      'severity' => 'warning',
      'code' => 'GETIMAGESIZE_REMOTE',
      'message' => 'getimagesize() with variable may cause slow S3/remote file access',
      'recommendation' => 'Use file_exists() first, cache dimensions, or use Drupal image styles',
      'requires_s3' => TRUE,
    ],
    'file_get_contents_variable' => [
      'pattern' => '/file_get_contents\s*\(\s*\$/',
      'severity' => 'notice',
      'code' => 'FILE_GET_CONTENTS_VARIABLE',
      'message' => 'file_get_contents() with variable URI may be slow for S3/remote files',
      'recommendation' => 'For remote files, use Guzzle HTTP client with proper timeouts and caching',
      'requires_s3' => TRUE,
    ],
    'realpath_usage' => [
      'pattern' => '/(?:->realpath\s*\(\s*\)|\\\\realpath\s*\(|\brealpath\s*\()/',
      'severity' => 'warning',
      'code' => 'REALPATH_USAGE',
      'message' => 'realpath() fails silently with S3 and other stream wrappers',
      'recommendation' => 'Use FileSystem::realpath() or avoid realpath with stream wrappers',
      'requires_s3' => TRUE,
    ],

    // ========================================================================
    // MEDIUM: Static Service Calls in Loops
    // ========================================================================
    'static_service_in_loop' => [
      'pattern' => '/\\\\Drupal::(service|entityTypeManager|currentUser|database|cache)\s*\(/',
      'requires_loop_context' => TRUE,
      'severity' => 'notice',
      'code' => 'STATIC_SERVICE_IN_LOOP',
      'message' => 'Static service call inside loop - minor overhead but bad practice',
      'recommendation' => 'Move \\Drupal:: calls outside loop or use dependency injection',
    ],

    // ========================================================================
    // HIGH: Queries in Preprocess/Render Hooks
    // ========================================================================
    'query_in_preprocess' => [
      'pattern' => '/function\s+\w+_preprocess_/',
      'requires_query_in_body' => TRUE,
      'severity' => 'warning',
      'code' => 'QUERY_IN_PREPROCESS',
      'message' => 'Database query in preprocess hook runs on every page render',
      'recommendation' => 'Move queries to controller/service layer and pass data via render arrays',
    ],
    'query_in_page_attachments' => [
      'pattern' => '/function\s+\w+_page_attachments/',
      'requires_query_in_body' => TRUE,
      'severity' => 'warning',
      'code' => 'QUERY_IN_PAGE_ATTACHMENTS',
      'message' => 'Database query in page_attachments runs on every page load',
      'recommendation' => 'Cache results or move logic to a dedicated service with proper caching',
    ],

    // ========================================================================
    // MEDIUM: Deprecated D7 Functions
    // ========================================================================
    'deprecated_node_load' => [
      'pattern' => '/\bnode_load\s*\(/',
      'severity' => 'warning',
      'code' => 'DEPRECATED_NODE_LOAD',
      'message' => 'node_load() is deprecated since Drupal 8',
      'recommendation' => 'Use \\Drupal::entityTypeManager()->getStorage("node")->load($nid)',
    ],
    'deprecated_user_load' => [
      'pattern' => '/\buser_load\s*\(/',
      'severity' => 'warning',
      'code' => 'DEPRECATED_USER_LOAD',
      'message' => 'user_load() is deprecated since Drupal 8',
      'recommendation' => 'Use \\Drupal::entityTypeManager()->getStorage("user")->load($uid)',
    ],
    'deprecated_entity_load' => [
      'pattern' => '/\bentity_load\s*\(/',
      'severity' => 'warning',
      'code' => 'DEPRECATED_ENTITY_LOAD',
      'message' => 'entity_load() is deprecated since Drupal 8',
      'recommendation' => 'Use \\Drupal::entityTypeManager()->getStorage($type)->load($id)',
    ],
    'deprecated_db_query' => [
      'pattern' => '/\bdb_query\s*\(/',
      'severity' => 'warning',
      'code' => 'DEPRECATED_DB_QUERY',
      'message' => 'db_query() is deprecated since Drupal 8',
      'recommendation' => 'Use \\Drupal::database()->query() or entityQuery for entities',
    ],
    'deprecated_db_select' => [
      'pattern' => '/\bdb_select\s*\(/',
      'severity' => 'warning',
      'code' => 'DEPRECATED_DB_SELECT',
      'message' => 'db_select() is deprecated since Drupal 8',
      'recommendation' => 'Use \\Drupal::database()->select() with dependency injection',
    ],

    // ========================================================================
    // HIGH: loadByProperties (Ignores Access Check)
    // ========================================================================
    'load_by_properties' => [
      'pattern' => '/->loadByProperties\s*\(/',
      'severity' => 'warning',
      'code' => 'LOAD_BY_PROPERTIES',
      'message' => 'loadByProperties() bypasses entity access control',
      'recommendation' => 'Use entityQuery with ->accessCheck(TRUE) for user-facing queries',
    ],

    // ========================================================================
    // File Operations (only reported when s3fs module is enabled)
    // ========================================================================
    'fopen_variable' => [
      'pattern' => '/\bfopen\s*\(\s*\$/',
      'severity' => 'notice',
      'code' => 'FOPEN_VARIABLE',
      'message' => 'fopen() with variable path - verify S3 compatibility',
      'recommendation' => 'Use Drupal file system service for stream wrapper compatibility',
      'requires_s3' => TRUE,
    ],
    'is_file_variable' => [
      'pattern' => '/\bis_file\s*\(\s*\$/',
      'severity' => 'notice',
      'code' => 'IS_FILE_VARIABLE',
      'message' => 'is_file() may be slow or fail with S3 stream wrappers',
      'recommendation' => 'Use file_exists() or FileSystem service for stream wrapper support',
      'requires_s3' => TRUE,
    ],
  ];

  /**
   * Views cache configuration issues in YAML files.
   */
  protected const VIEWS_CACHE_PATTERNS = [
    'views_no_cache' => [
      'pattern' => '/cache:\s*\n\s*type:\s*[\'"]?none[\'"]?/',
      'severity' => 'warning',
      'code' => 'VIEWS_NO_CACHE',
      'message' => 'Views display has caching disabled',
      'recommendation' => 'Enable tag-based or time-based caching for better performance',
    ],
  ];

  protected ConfigFactoryInterface $configFactory;
  protected ModuleHandlerInterface $moduleHandler;
  protected string $appRoot;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->configFactory = $container->get('config.factory');
    $instance->moduleHandler = $container->get('module_handler');
    $instance->appRoot = $container->getParameter('app.root');
    return $instance;
  }

  /**
   * Gets the user type from audit_modules configuration.
   */
  protected function getUserType(): string {
    return $this->configFactory->get('audit_modules.settings')->get('user_type') ?? 'mixed';
  }

  /**
   * Gets the recommended cache max-age from configuration.
   */
  protected function getRecommendedCacheMaxAge(): int {
    $value = $this->configFactory->get('audit_performance.settings')->get('recommended_cache_max_age');
    return $value ? (int) $value : self::DEFAULT_RECOMMENDED_CACHE_MAX_AGE;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $config): array {
    return [
      'recommended_cache_max_age' => [
        '#type' => 'number',
        '#title' => $this->t('Minimum page cache lifetime (seconds)'),
        '#description' => $this->t('Minimum recommended value for page cache duration. Pages cached for less time will trigger warnings. Guidelines: High-traffic/static sites: 24 hours (86400) | Medium sites: 6 hours (21600) | Dynamic sites: 1 hour (3600) | Real-time content: 15 min (900)'),
        '#default_value' => $config['recommended_cache_max_age'] ?? self::DEFAULT_RECOMMENDED_CACHE_MAX_AGE,
        '#min' => 60,
        '#max' => 604800,
        '#field_suffix' => $this->t('seconds'),
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    $user_type = $this->getUserType();

    $production_settings_results = $this->analyzeProductionSettings();
    $cache_modules_results = $this->analyzeCacheModules($user_type);
    $config_results = $this->analyzeSystemConfiguration();
    $code_results = $this->analyzeCodeIssues();
    $perf_patterns_results = $this->analyzePerformancePatterns();

    $scores = $this->calculateScores(
      $production_settings_results,
      $cache_modules_results,
      $config_results,
      $code_results,
      $perf_patterns_results
    );

    return [
      '_files' => [
        'production_settings' => $production_settings_results,
        'cache_modules' => $cache_modules_results,
        'configuration' => $config_results,
        'code_analysis' => $code_results,
        'performance_patterns' => $perf_patterns_results,
      ],
      'score' => $scores,
      'user_type' => $user_type,
    ];
  }

  /**
   * Analyzes production settings (error display, Twig debug, etc.).
   *
   * These settings should be disabled in production for security and performance.
   */
  protected function analyzeProductionSettings(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    // 1. Check error display level (system.logging.error_level).
    $logging_config = $this->configFactory->get('system.logging');
    $error_level = $logging_config->get('error_level') ?? 'hide';

    if ($error_level === 'hide') {
      $results[] = $this->createResultItem(
        'notice',
        'ERROR_DISPLAY_HIDDEN',
        (string) $this->t('Error messages are hidden from users.'),
        [
          'setting' => 'system.logging.error_level',
          'value' => $error_level,
          'is_production_safe' => TRUE,
        ]
      );
      $notices++;
    }
    else {
      $error_level_labels = [
        'all' => $this->t('All messages'),
        'some' => $this->t('Errors and warnings'),
        'verbose' => $this->t('All messages with backtrace'),
      ];
      $level_label = $error_level_labels[$error_level] ?? $error_level;

      $results[] = $this->createResultItem(
        'error',
        'ERROR_DISPLAY_VISIBLE',
        (string) $this->t('Error messages are visible to users (@level). This exposes sensitive information (file paths, database queries, stack traces).', [
          '@level' => $level_label,
        ]),
        [
          'setting' => 'system.logging.error_level',
          'value' => $error_level,
          'is_production_safe' => FALSE,
          'recommended' => 'hide',
        ]
      );
      $errors++;
    }

    // 2. Check Twig debug settings.
    // Twig debug can be enabled via services.yml or settings.php.
    $twig_debug = FALSE;
    $twig_auto_reload = FALSE;
    $twig_cache = TRUE;

    // Check via Twig environment if available.
    try {
      $twig = \Drupal::service('twig');
      if ($twig instanceof \Twig\Environment) {
        $twig_debug = $twig->isDebug();
        $twig_auto_reload = $twig->isAutoReload();
      }
    }
    catch (\Exception $e) {
      // Service not available, will check via config.
    }

    // Also check development.services.yml override indicator.
    // If Twig debug is enabled, it's usually through services.yml.
    if ($twig_debug) {
      $results[] = $this->createResultItem(
        'error',
        'TWIG_DEBUG_ENABLED',
        (string) $this->t('Twig debug mode is ENABLED. This exposes template file paths in HTML comments and severely impacts performance.'),
        [
          'setting' => 'twig.config.debug',
          'value' => TRUE,
          'is_production_safe' => FALSE,
          'recommended' => FALSE,
          'fix' => 'Set "debug: false" in services.yml or remove development.services.yml',
        ]
      );
      $errors++;
    }
    else {
      $results[] = $this->createResultItem(
        'notice',
        'TWIG_DEBUG_DISABLED',
        (string) $this->t('Twig debug mode is disabled.'),
        [
          'setting' => 'twig.config.debug',
          'value' => FALSE,
          'is_production_safe' => TRUE,
        ]
      );
      $notices++;
    }

    // 3. Check Twig auto_reload (causes performance issues).
    if ($twig_auto_reload) {
      $results[] = $this->createResultItem(
        'warning',
        'TWIG_AUTO_RELOAD_ENABLED',
        (string) $this->t('Twig auto_reload is enabled. Templates are recompiled on every request if modified, impacting performance.'),
        [
          'setting' => 'twig.config.auto_reload',
          'value' => TRUE,
          'is_production_safe' => FALSE,
          'recommended' => FALSE,
        ]
      );
      $warnings++;
    }
    else {
      $results[] = $this->createResultItem(
        'notice',
        'TWIG_AUTO_RELOAD_DISABLED',
        (string) $this->t('Twig auto_reload is disabled.'),
        [
          'setting' => 'twig.config.auto_reload',
          'value' => FALSE,
          'is_production_safe' => TRUE,
        ]
      );
      $notices++;
    }

    // 4. Check if Devel module is enabled (common dev module left in production).
    if ($this->moduleHandler->moduleExists('devel')) {
      $results[] = $this->createResultItem(
        'warning',
        'DEVEL_MODULE_ENABLED',
        (string) $this->t('Devel module is enabled. This development module should be disabled in production.'),
        [
          'module' => 'devel',
          'is_production_safe' => FALSE,
          'recommendation' => 'Disable Devel module with: drush pm:uninstall devel',
        ]
      );
      $warnings++;
    }

    // 5. Check if Kint module is enabled.
    if ($this->moduleHandler->moduleExists('kint')) {
      $results[] = $this->createResultItem(
        'warning',
        'KINT_MODULE_ENABLED',
        (string) $this->t('Kint module is enabled. This development module should be disabled in production.'),
        [
          'module' => 'kint',
          'is_production_safe' => FALSE,
          'recommendation' => 'Disable Kint module with: drush pm:uninstall kint',
        ]
      );
      $warnings++;
    }

    // 6. Check if Webprofiler is enabled (Devel submodule).
    if ($this->moduleHandler->moduleExists('webprofiler')) {
      $results[] = $this->createResultItem(
        'error',
        'WEBPROFILER_MODULE_ENABLED',
        (string) $this->t('Webprofiler module is enabled. This significantly impacts performance and exposes debugging toolbar.'),
        [
          'module' => 'webprofiler',
          'is_production_safe' => FALSE,
          'recommendation' => 'Disable Webprofiler module with: drush pm:uninstall webprofiler',
        ]
      );
      $errors++;
    }

    $result = $this->createResult($results, $errors, $warnings, $notices);
    $result['stats'] = [
      'error_level' => $error_level,
      'twig_debug' => $twig_debug,
      'twig_auto_reload' => $twig_auto_reload,
      'devel_enabled' => $this->moduleHandler->moduleExists('devel'),
    ];

    return $result;
  }

  /**
   * Analyzes cache modules status based on user type.
   */
  protected function analyzeCacheModules(string $user_type): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    $modules = self::CACHE_MODULES[$user_type] ?? self::CACHE_MODULES['mixed'];
    $installed_count = 0;
    $required_count = 0;
    $required_missing = 0;

    foreach ($modules as $module_name => $module_info) {
      $is_installed = $this->moduleHandler->moduleExists($module_name);
      $is_required = $module_info['required'];

      if ($is_required) {
        $required_count++;
      }

      if ($is_installed) {
        $installed_count++;
        $results[] = $this->createResultItem(
          'notice',
          'CACHE_MODULE_INSTALLED',
          (string) $this->t('@label is installed.', ['@label' => $module_info['label']]),
          [
            'module' => $module_name,
            'label' => $module_info['label'],
            'description' => $module_info['description'],
            'required' => $is_required,
            'installed' => TRUE,
          ]
        );
        $notices++;
      }
      else {
        $severity = $is_required ? 'error' : 'warning';
        $results[] = $this->createResultItem(
          $severity,
          'CACHE_MODULE_MISSING',
          (string) $this->t('@label is not installed.', ['@label' => $module_info['label']]),
          [
            'module' => $module_name,
            'label' => $module_info['label'],
            'description' => $module_info['description'],
            'required' => $is_required,
            'installed' => FALSE,
          ]
        );

        if ($is_required) {
          $errors++;
          $required_missing++;
        }
        else {
          $warnings++;
        }
      }
    }

    $result = $this->createResult($results, $errors, $warnings, $notices);
    $result['stats'] = [
      'user_type' => $user_type,
      'total_modules' => count($modules),
      'installed_count' => $installed_count,
      'required_count' => $required_count,
      'required_missing' => $required_missing,
    ];

    return $result;
  }

  /**
   * Analyzes system performance configuration.
   */
  protected function analyzeSystemConfiguration(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    $system_performance = $this->configFactory->get('system.performance');

    // Check CSS aggregation - this is critical for production performance.
    $css_preprocess = (bool) $system_performance->get('css.preprocess');
    if (!$css_preprocess) {
      $results[] = $this->createResultItem(
        'error',
        'CSS_AGGREGATION_DISABLED',
        (string) $this->t('CSS aggregation is disabled. This significantly increases page load time by requiring multiple HTTP requests.'),
        ['setting' => 'css.preprocess', 'value' => FALSE, 'recommended' => TRUE]
      );
      $errors++;
    }
    else {
      $results[] = $this->createResultItem(
        'notice',
        'CSS_AGGREGATION_ENABLED',
        (string) $this->t('CSS aggregation is enabled.'),
        ['setting' => 'css.preprocess', 'value' => TRUE]
      );
      $notices++;
    }

    // Check JS aggregation - this is critical for production performance.
    $js_preprocess = (bool) $system_performance->get('js.preprocess');
    if (!$js_preprocess) {
      $results[] = $this->createResultItem(
        'error',
        'JS_AGGREGATION_DISABLED',
        (string) $this->t('JavaScript aggregation is disabled. This significantly increases page load time by requiring multiple HTTP requests.'),
        ['setting' => 'js.preprocess', 'value' => FALSE, 'recommended' => TRUE]
      );
      $errors++;
    }
    else {
      $results[] = $this->createResultItem(
        'notice',
        'JS_AGGREGATION_ENABLED',
        (string) $this->t('JavaScript aggregation is enabled.'),
        ['setting' => 'js.preprocess', 'value' => TRUE]
      );
      $notices++;
    }

    // Check page cache max age against configured threshold.
    $cache_max_age = (int) $system_performance->get('cache.page.max_age');
    $recommended_max_age = $this->getRecommendedCacheMaxAge();

    if ($cache_max_age === 0) {
      $results[] = $this->createResultItem(
        'error',
        'PAGE_CACHE_DISABLED',
        (string) $this->t('Page cache is disabled (max-age = 0). This severely impacts performance - every page request hits the database.'),
        [
          'setting' => 'cache.page.max_age',
          'value' => 0,
          'recommended' => $recommended_max_age,
          'level' => 'disabled',
        ]
      );
      $errors++;
    }
    elseif ($cache_max_age < $recommended_max_age) {
      $results[] = $this->createResultItem(
        'error',
        'PAGE_CACHE_BELOW_RECOMMENDED',
        (string) $this->t('Page cache max-age (@current) is significantly below the recommended value (@recommended). This causes unnecessary cache rebuilds.', [
          '@current' => $this->formatDuration($cache_max_age),
          '@recommended' => $this->formatDuration($recommended_max_age),
        ]),
        [
          'setting' => 'cache.page.max_age',
          'value' => $cache_max_age,
          'recommended' => $recommended_max_age,
          'level' => 'below_recommended',
        ]
      );
      $errors++;
    }
    else {
      $results[] = $this->createResultItem(
        'notice',
        'PAGE_CACHE_OK',
        (string) $this->t('Page cache max-age (@current) meets or exceeds the recommended value (@recommended).', [
          '@current' => $this->formatDuration($cache_max_age),
          '@recommended' => $this->formatDuration($recommended_max_age),
        ]),
        [
          'setting' => 'cache.page.max_age',
          'value' => $cache_max_age,
          'recommended' => $recommended_max_age,
          'level' => 'ok',
        ]
      );
      $notices++;
    }

    // Check CSS/JS gzip compression setting.
    $gzip = (bool) $system_performance->get('css.gzip');
    if (!$gzip) {
      $results[] = $this->createResultItem(
        'notice',
        'GZIP_DISABLED',
        (string) $this->t('CSS/JS gzip compression is disabled. Consider enabling if server does not handle compression.'),
        ['setting' => 'css.gzip', 'value' => FALSE]
      );
      $notices++;
    }

    // Check Fast 404 configuration.
    // Fast 404 is configured in settings.php and prevents Drupal from
    // bootstrapping for common missing files (images, CSS, JS, etc.).
    $fast_404_enabled = $this->isFast404Enabled();
    if ($fast_404_enabled) {
      $results[] = $this->createResultItem(
        'notice',
        'FAST_404_ENABLED',
        (string) $this->t('Fast 404 is enabled. Missing static files are handled quickly without full Drupal bootstrap.'),
        [
          'setting' => 'fast_404',
          'enabled' => TRUE,
          'is_production_safe' => TRUE,
        ]
      );
      $notices++;
    }
    else {
      $results[] = $this->createResultItem(
        'warning',
        'FAST_404_DISABLED',
        (string) $this->t('Fast 404 is not enabled. Every 404 request for missing images, CSS, or JS files triggers a full Drupal bootstrap, impacting performance.'),
        [
          'setting' => 'fast_404',
          'enabled' => FALSE,
          'is_production_safe' => FALSE,
          'recommendation' => 'Add fast 404 configuration to settings.php. See default.settings.php for examples.',
        ]
      );
      $warnings++;
    }

    return $this->createResult($results, $errors, $warnings, $notices);
  }

  /**
   * Checks if Fast 404 is enabled.
   *
   * Fast 404 is typically configured in settings.php by including
   * $config['system.performance']['fast_404']['enabled'] = TRUE.
   *
   * @return bool
   *   TRUE if Fast 404 is enabled.
   */
  protected function isFast404Enabled(): bool {
    // Fast 404 configuration is in system.performance config.
    $fast_404_config = $this->configFactory->get('system.performance')->get('fast_404');

    // Check if fast_404 is explicitly enabled in config.
    if (is_array($fast_404_config) && !empty($fast_404_config['enabled'])) {
      return TRUE;
    }

    // Also check if it's set in settings.php via global $config.
    // This is the recommended way to enable fast 404.
    global $config;
    if (isset($config['system.performance']['fast_404']['enabled'])) {
      return (bool) $config['system.performance']['fast_404']['enabled'];
    }

    return FALSE;
  }

  /**
   * Checks if S3 or remote file storage modules are enabled.
   *
   * This checks for common S3/remote storage modules:
   * - s3fs: Amazon S3 File System
   * - flysystem: Flysystem (supports S3, Azure, etc.)
   * - flysystem_s3: Flysystem S3 adapter
   *
   * @return bool
   *   TRUE if any S3/remote storage module is enabled.
   */
  protected function isS3Enabled(): bool {
    $s3_modules = [
      's3fs',
      'flysystem',
      'flysystem_s3',
      's3_file_system',
      'amazons3',
    ];

    foreach ($s3_modules as $module) {
      if ($this->moduleHandler->moduleExists($module)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Formats a duration in seconds to a human-readable string.
   */
  protected function formatDuration(int $seconds): string {
    if ($seconds >= 86400) {
      $hours = $seconds / 3600;
      return $this->t('@hours hours', ['@hours' => round($hours)])->__toString();
    }
    if ($seconds >= 3600) {
      $hours = $seconds / 3600;
      return $this->t('@hours hour(s)', ['@hours' => round($hours, 1)])->__toString();
    }
    if ($seconds >= 60) {
      $minutes = $seconds / 60;
      return $this->t('@minutes minutes', ['@minutes' => round($minutes)])->__toString();
    }
    return $this->t('@seconds seconds', ['@seconds' => $seconds])->__toString();
  }

  /**
   * Analyzes custom code for cache issues and debug code.
   *
   * Scans PHP and Twig files for:
   * - Cache anti-patterns (max-age 0, generic tags, etc.)
   * - Debug functions (dpm, kint, var_dump, etc.)
   *
   * Note: The audit module directory is automatically excluded to avoid
   * false positives from pattern definitions in the analyzer code.
   */
  protected function analyzeCodeIssues(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    // Get directories and patterns from main audit module config.
    $audit_config = $this->configFactory->get('audit.settings');
    $directories = $this->parseDirectories($audit_config->get('scan_directories') ?? "web/modules/custom\nweb/themes/custom");
    $exclude_patterns = $this->parseExcludePatterns($audit_config->get('exclude_patterns') ?? '');

    // Always exclude the audit module directory to avoid false positives
    // from pattern definitions in the analyzer code itself.
    // Get the path dynamically as it may be in custom, contrib, or a profile.
    $audit_module_path = $this->moduleHandler->getModule('audit')->getPath();
    $exclude_patterns[] = $audit_module_path;

    // Use project root (parent of DRUPAL_ROOT) to allow scanning outside /web.
    $project_root = dirname($this->appRoot);

    $files_scanned = 0;
    $issues_found = [];

    foreach ($directories as $directory) {
      $full_path = $project_root . '/' . ltrim($directory, '/');

      if (!is_dir($full_path)) {
        continue;
      }

      $php_files = $this->findFiles($full_path, '*.php', $exclude_patterns);
      $twig_files = $this->findFiles($full_path, '*.twig', $exclude_patterns);

      // Scan PHP files for cache issues AND debug code.
      foreach ($php_files as $file) {
        $files_scanned++;
        $content = @file_get_contents($file);
        if ($content === FALSE) {
          continue;
        }

        $lines = explode("\n", $content);
        $relative_path = str_replace($project_root . '/', '', $file);

        // Check cache problem patterns.
        foreach (self::CACHE_PROBLEM_PATTERNS as $pattern_id => $pattern_info) {
          if (preg_match_all($pattern_info['pattern'], $content, $matches, PREG_OFFSET_CAPTURE)) {
            foreach ($matches[0] as $match) {
              $line_number = substr_count(substr($content, 0, $match[1]), "\n") + 1;
              $code_context = $this->extractCodeContext($lines, $line_number);

              $issues_found[] = [
                'file' => $relative_path,
                'line' => $line_number,
                'pattern_id' => $pattern_id,
                'severity' => $pattern_info['severity'],
                'code' => $pattern_info['code'],
                'message' => $pattern_info['message'],
                'recommendation' => $pattern_info['recommendation'] ?? '',
                'code_context' => $code_context,
                'category' => 'cache',
              ];
            }
          }
        }

        // Check debug code patterns.
        foreach (self::DEBUG_CODE_PATTERNS as $pattern_id => $pattern_info) {
          if (preg_match_all($pattern_info['pattern'], $content, $matches, PREG_OFFSET_CAPTURE)) {
            foreach ($matches[0] as $match) {
              $line_number = substr_count(substr($content, 0, $match[1]), "\n") + 1;
              $code_context = $this->extractCodeContext($lines, $line_number);

              $issues_found[] = [
                'file' => $relative_path,
                'line' => $line_number,
                'pattern_id' => $pattern_id,
                'severity' => $pattern_info['severity'],
                'code' => $pattern_info['code'],
                'message' => $pattern_info['message'],
                'recommendation' => $pattern_info['recommendation'] ?? '',
                'code_context' => $code_context,
                'category' => 'debug',
              ];
            }
          }
        }
      }

      // Scan Twig files for cache issues AND debug code.
      foreach ($twig_files as $file) {
        $files_scanned++;
        $content = @file_get_contents($file);
        if ($content === FALSE) {
          continue;
        }

        $lines = explode("\n", $content);
        $relative_path = str_replace($project_root . '/', '', $file);

        // Check Twig cache patterns.
        foreach (self::TWIG_CACHE_PATTERNS as $pattern_id => $pattern_info) {
          if (preg_match_all($pattern_info['pattern'], $content, $matches, PREG_OFFSET_CAPTURE)) {
            foreach ($matches[0] as $match) {
              $line_number = substr_count(substr($content, 0, $match[1]), "\n") + 1;
              $code_context = $this->extractCodeContext($lines, $line_number);

              $issues_found[] = [
                'file' => $relative_path,
                'line' => $line_number,
                'pattern_id' => $pattern_id,
                'severity' => $pattern_info['severity'],
                'code' => $pattern_info['code'],
                'message' => $pattern_info['message'],
                'recommendation' => $pattern_info['recommendation'] ?? '',
                'code_context' => $code_context,
                'category' => 'cache',
              ];
            }
          }
        }

        // Check Twig debug patterns.
        foreach (self::TWIG_DEBUG_PATTERNS as $pattern_id => $pattern_info) {
          if (preg_match_all($pattern_info['pattern'], $content, $matches, PREG_OFFSET_CAPTURE)) {
            foreach ($matches[0] as $match) {
              $line_number = substr_count(substr($content, 0, $match[1]), "\n") + 1;
              $code_context = $this->extractCodeContext($lines, $line_number);

              $issues_found[] = [
                'file' => $relative_path,
                'line' => $line_number,
                'pattern_id' => $pattern_id,
                'severity' => $pattern_info['severity'],
                'code' => $pattern_info['code'],
                'message' => $pattern_info['message'],
                'recommendation' => $pattern_info['recommendation'] ?? '',
                'code_context' => $code_context,
                'category' => 'debug',
              ];
            }
          }
        }
      }
    }

    // Convert issues to results.
    foreach ($issues_found as $issue) {
      $results[] = $this->createResultItem(
        $issue['severity'],
        $issue['code'],
        (string) $this->t('@message in @file:@line', [
          '@message' => $issue['message'],
          '@file' => $issue['file'],
          '@line' => $issue['line'],
        ]),
        [
          'file' => $issue['file'],
          'line' => $issue['line'],
          'pattern' => $issue['pattern_id'],
          'recommendation' => $issue['recommendation'],
          'code_context' => $issue['code_context'] ?? [],
        ]
      );

      match ($issue['severity']) {
        'error' => $errors++,
        'warning' => $warnings++,
        default => $notices++,
      };
    }

    // Count issues by category.
    $debug_issues = array_filter($issues_found, fn($i) => ($i['category'] ?? '') === 'debug');
    $cache_issues = array_filter($issues_found, fn($i) => ($i['category'] ?? '') === 'cache');

    // Add summary if no issues found.
    if (empty($issues_found)) {
      $results[] = $this->createResultItem(
        'notice',
        'NO_CODE_ISSUES',
        (string) $this->t('No debug code or cache issues detected in @count scanned files.', [
          '@count' => $files_scanned,
        ]),
        ['files_scanned' => $files_scanned]
      );
      $notices++;
    }

    $result = $this->createResult($results, $errors, $warnings, $notices);
    $result['stats'] = [
      'files_scanned' => $files_scanned,
      'issues_found' => count($issues_found),
      'debug_issues' => count($debug_issues),
      'cache_issues' => count($cache_issues),
      'directories' => $directories,
    ];

    return $result;
  }

  /**
   * Analyzes custom code for performance anti-patterns.
   *
   * Detects issues like:
   * - N+1 queries (load() in loops, referencedEntities() in loops)
   * - EntityQuery without range() or accessCheck()
   * - Entity save() in loops
   * - S3/remote file operations
   * - Queries in preprocess hooks
   * - Deprecated D7 functions
   * - loadByProperties without access control
   * - Views without caching
   */
  protected function analyzePerformancePatterns(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    // Check if S3 file system modules are enabled.
    // These patterns are only relevant when using remote file storage.
    $s3_enabled = $this->isS3Enabled();

    // Get directories and patterns from main audit module config.
    $audit_config = $this->configFactory->get('audit.settings');
    $directories = $this->parseDirectories($audit_config->get('scan_directories') ?? "web/modules/custom\nweb/themes/custom");
    $exclude_patterns = $this->parseExcludePatterns($audit_config->get('exclude_patterns') ?? '');

    // Always exclude the audit module directory.
    $audit_module_path = $this->moduleHandler->getModule('audit')->getPath();
    $exclude_patterns[] = $audit_module_path;

    $project_root = dirname($this->appRoot);

    $files_scanned = 0;
    $yml_files_scanned = 0;
    $issues_found = [];

    foreach ($directories as $directory) {
      $full_path = $project_root . '/' . ltrim($directory, '/');

      if (!is_dir($full_path)) {
        continue;
      }

      $php_files = $this->findFiles($full_path, '*.php', $exclude_patterns);
      $module_files = $this->findFiles($full_path, '*.module', $exclude_patterns);
      $yml_files = $this->findFiles($full_path, '*.yml', $exclude_patterns);

      // Combine PHP and module files.
      $code_files = array_unique(array_merge($php_files, $module_files));

      // Scan PHP/module files for performance patterns.
      foreach ($code_files as $file) {
        $files_scanned++;
        $content = @file_get_contents($file);
        if ($content === FALSE) {
          continue;
        }

        $lines = explode("\n", $content);
        $relative_path = str_replace($project_root . '/', '', $file);

        // Check each performance pattern.
        foreach (self::DRUPAL_PERF_PATTERNS as $pattern_id => $pattern_info) {
          // Skip S3-related patterns if S3 is not enabled.
          if (!empty($pattern_info['requires_s3']) && !$s3_enabled) {
            continue;
          }

          // Skip patterns that require special context if no context is needed.
          $requires_loop = !empty($pattern_info['requires_loop_context']);
          $requires_no_range = !empty($pattern_info['requires_no_range']);
          $requires_no_access_check = !empty($pattern_info['requires_no_access_check']);
          $requires_query_in_body = !empty($pattern_info['requires_query_in_body']);

          if (preg_match_all($pattern_info['pattern'], $content, $matches, PREG_OFFSET_CAPTURE)) {
            foreach ($matches[0] as $match) {
              $line_number = substr_count(substr($content, 0, $match[1]), "\n") + 1;
              $is_issue = FALSE;

              // Context-dependent checks.
              if ($requires_loop) {
                $is_issue = $this->isInsideLoop($lines, $line_number);
              }
              elseif ($requires_no_range) {
                $is_issue = $this->lacksPaginationInContext($lines, $line_number);
              }
              elseif ($requires_no_access_check) {
                $is_issue = $this->lacksAccessCheckInContext($lines, $line_number);
              }
              elseif ($requires_query_in_body) {
                $is_issue = $this->hasQueryInFunctionBody($lines, $line_number);
              }
              else {
                // Simple pattern match without context requirements.
                $is_issue = TRUE;
              }

              if ($is_issue) {
                $code_context = $this->extractCodeContext($lines, $line_number, 3);

                $issues_found[] = [
                  'file' => $relative_path,
                  'line' => $line_number,
                  'pattern_id' => $pattern_id,
                  'severity' => $pattern_info['severity'],
                  'code' => $pattern_info['code'],
                  'message' => $pattern_info['message'],
                  'recommendation' => $pattern_info['recommendation'] ?? '',
                  'code_context' => $code_context,
                  'category' => $this->getPerformanceCategory($pattern_info['code']),
                ];
              }
            }
          }
        }
      }

      // Scan YAML files for Views cache issues.
      foreach ($yml_files as $file) {
        // Only check views-related YAML files.
        if (!str_contains($file, 'views.view.')) {
          continue;
        }

        $yml_files_scanned++;
        $content = @file_get_contents($file);
        if ($content === FALSE) {
          continue;
        }

        $lines = explode("\n", $content);
        $relative_path = str_replace($project_root . '/', '', $file);

        foreach (self::VIEWS_CACHE_PATTERNS as $pattern_id => $pattern_info) {
          if (preg_match_all($pattern_info['pattern'], $content, $matches, PREG_OFFSET_CAPTURE)) {
            foreach ($matches[0] as $match) {
              $line_number = substr_count(substr($content, 0, $match[1]), "\n") + 1;
              $code_context = $this->extractCodeContext($lines, $line_number, 2);

              $issues_found[] = [
                'file' => $relative_path,
                'line' => $line_number,
                'pattern_id' => $pattern_id,
                'severity' => $pattern_info['severity'],
                'code' => $pattern_info['code'],
                'message' => $pattern_info['message'],
                'recommendation' => $pattern_info['recommendation'] ?? '',
                'code_context' => $code_context,
                'category' => 'views',
              ];
            }
          }
        }
      }
    }

    // Convert issues to results.
    foreach ($issues_found as $issue) {
      $results[] = $this->createResultItem(
        $issue['severity'],
        $issue['code'],
        (string) $this->t('@message in @file:@line', [
          '@message' => $issue['message'],
          '@file' => $issue['file'],
          '@line' => $issue['line'],
        ]),
        [
          'file' => $issue['file'],
          'line' => $issue['line'],
          'pattern' => $issue['pattern_id'],
          'recommendation' => $issue['recommendation'],
          'code_context' => $issue['code_context'] ?? [],
          'category' => $issue['category'],
        ]
      );

      match ($issue['severity']) {
        'error' => $errors++,
        'warning' => $warnings++,
        default => $notices++,
      };
    }

    // Group issues by category for stats.
    $categories = [];
    foreach ($issues_found as $issue) {
      $cat = $issue['category'] ?? 'other';
      $categories[$cat] = ($categories[$cat] ?? 0) + 1;
    }

    // Add summary if no issues found.
    if (empty($issues_found)) {
      $results[] = $this->createResultItem(
        'notice',
        'NO_PERF_ISSUES',
        (string) $this->t('No performance anti-patterns detected in @count scanned files.', [
          '@count' => $files_scanned,
        ]),
        ['files_scanned' => $files_scanned]
      );
      $notices++;
    }

    $result = $this->createResult($results, $errors, $warnings, $notices);
    $result['stats'] = [
      'files_scanned' => $files_scanned,
      'yml_files_scanned' => $yml_files_scanned,
      'issues_found' => count($issues_found),
      'categories' => $categories,
      'directories' => $directories,
    ];

    return $result;
  }

  /**
   * Checks if a line is inside a loop (foreach, for, while).
   *
   * @param array $lines
   *   All lines of the file.
   * @param int $target_line
   *   The line number to check (1-indexed).
   *
   * @return bool
   *   TRUE if the line appears to be inside a loop.
   */
  protected function isInsideLoop(array $lines, int $target_line): bool {
    // Look at the 15 lines before the target for loop constructs.
    $start_line = max(0, $target_line - 16);
    $context_lines = array_slice($lines, $start_line, $target_line - $start_line);
    $context = implode("\n", $context_lines);

    // Check for loop constructs.
    return (bool) preg_match('/(foreach|for\s*\(|while\s*\()/', $context);
  }

  /**
   * Checks if an entityQuery lacks pagination (range/limit).
   *
   * @param array $lines
   *   All lines of the file.
   * @param int $target_line
   *   The line number where entityQuery was found (1-indexed).
   *
   * @return bool
   *   TRUE if the query lacks pagination.
   */
  protected function lacksPaginationInContext(array $lines, int $target_line): bool {
    // Look at the 15 lines after the query for ->range() or ->execute().
    $end_line = min(count($lines), $target_line + 14);
    $following_lines = array_slice($lines, $target_line - 1, $end_line - $target_line + 1);
    $following = implode("\n", $following_lines);

    // If there's no execute(), it's probably just a definition or comment.
    if (!preg_match('/->execute\s*\(/', $following)) {
      return FALSE;
    }

    // Check if range() is present before execute().
    return !preg_match('/->range\s*\(/', $following);
  }

  /**
   * Checks if an entityQuery lacks accessCheck().
   *
   * @param array $lines
   *   All lines of the file.
   * @param int $target_line
   *   The line number where entityQuery was found (1-indexed).
   *
   * @return bool
   *   TRUE if the query lacks accessCheck().
   */
  protected function lacksAccessCheckInContext(array $lines, int $target_line): bool {
    // Look at the 15 lines after the query.
    $end_line = min(count($lines), $target_line + 14);
    $following_lines = array_slice($lines, $target_line - 1, $end_line - $target_line + 1);
    $following = implode("\n", $following_lines);

    // If there's no execute(), it's probably just a definition or comment.
    if (!preg_match('/->execute\s*\(/', $following)) {
      return FALSE;
    }

    // Check if accessCheck() is present before execute().
    return !preg_match('/->accessCheck\s*\(/', $following);
  }

  /**
   * Checks if a function body contains database queries.
   *
   * @param array $lines
   *   All lines of the file.
   * @param int $function_line
   *   The line number where the function definition starts (1-indexed).
   *
   * @return bool
   *   TRUE if the function body contains queries.
   */
  protected function hasQueryInFunctionBody(array $lines, int $function_line): bool {
    // Look at the next 50 lines for query patterns.
    $end_line = min(count($lines), $function_line + 49);
    $body_lines = array_slice($lines, $function_line - 1, $end_line - $function_line + 1);
    $body = implode("\n", $body_lines);

    // Check for query patterns.
    return (bool) preg_match('/(?:entityQuery|getQuery|->load\s*\(|db_select|::database\(\)|->select\s*\()/', $body);
  }

  /**
   * Gets the category for a performance issue code.
   *
   * @param string $code
   *   The issue code.
   *
   * @return string
   *   The category name.
   */
  protected function getPerformanceCategory(string $code): string {
    return match (TRUE) {
      str_starts_with($code, 'N1_') => 'n1_queries',
      str_starts_with($code, 'QUERY_') || $code === 'DB_SELECT_NO_LIMIT' => 'queries',
      str_starts_with($code, 'MISSING_ACCESS') => 'security',
      $code === 'SAVE_IN_LOOP' => 'n1_queries',
      str_contains($code, 'IMAGESIZE') || str_contains($code, 'FILE_') || str_contains($code, 'REALPATH') || str_contains($code, 'FOPEN') || str_contains($code, 'IS_FILE') => 's3_files',
      $code === 'STATIC_SERVICE_IN_LOOP' => 'performance',
      str_contains($code, 'PREPROCESS') || str_contains($code, 'PAGE_ATTACHMENTS') => 'hooks',
      str_starts_with($code, 'DEPRECATED_') => 'deprecated',
      $code === 'LOAD_BY_PROPERTIES' => 'security',
      $code === 'VIEWS_NO_CACHE' => 'views',
      default => 'other',
    };
  }

  /**
   * Parses directory configuration into array.
   */
  protected function parseDirectories(string $directories): array {
    $lines = array_filter(array_map('trim', explode("\n", $directories)));
    return array_values($lines);
  }

  /**
   * Parses exclude patterns configuration.
   */
  protected function parseExcludePatterns(string $patterns): array {
    return array_filter(array_map('trim', explode("\n", $patterns)));
  }

  /**
   * Finds files matching a pattern in a directory.
   */
  protected function findFiles(string $directory, string $pattern, array $exclude_patterns): array {
    $files = [];

    try {
      $iterator = new \RecursiveIteratorIterator(
        new \RecursiveDirectoryIterator($directory, \RecursiveDirectoryIterator::SKIP_DOTS)
      );
    }
    catch (\Exception $e) {
      return $files;
    }

    foreach ($iterator as $file) {
      if (!$file->isFile()) {
        continue;
      }

      $filename = $file->getFilename();
      $filepath = $file->getPathname();

      // Check if matches pattern.
      if (!fnmatch($pattern, $filename)) {
        continue;
      }

      // Check exclude patterns.
      $excluded = FALSE;
      foreach ($exclude_patterns as $exclude) {
        if (fnmatch($exclude, $filename) || str_contains($filepath, '/' . trim($exclude, '/') . '/')) {
          $excluded = TRUE;
          break;
        }
      }

      if (!$excluded) {
        $files[] = $filepath;
      }
    }

    return $files;
  }

  /**
   * Calculates scores for all factors.
   */
  protected function calculateScores(
    array $production_settings_results,
    array $cache_modules_results,
    array $config_results,
    array $code_results,
    array $perf_patterns_results,
  ): array {
    $factors = [];

    // 1. Production Settings score (error display, Twig debug, dev modules).
    $prod_errors = $production_settings_results['summary']['errors'] ?? 0;
    $prod_warnings = $production_settings_results['summary']['warnings'] ?? 0;
    // Each error reduces 25 points, warnings reduce 10 points.
    $prod_score = max(0, 100 - ($prod_errors * 25) - ($prod_warnings * 10));

    $prod_issues = $prod_errors + $prod_warnings;
    $prod_description = $prod_issues === 0
      ? (string) $this->t('All production settings are correctly configured')
      : (string) $this->t('@count production setting issue(s) found', ['@count' => $prod_issues]);

    $factors['production_settings'] = [
      'score' => $prod_score,
      'weight' => self::SCORE_WEIGHTS['production_settings'],
      'label' => (string) $this->t('Production Settings'),
      'description' => $prod_description,
    ];

    // 2. System Configuration score (CSS/JS aggregation, page cache).
    $config_errors = $config_results['summary']['errors'] ?? 0;
    $config_warnings = $config_results['summary']['warnings'] ?? 0;
    // Each error reduces 33 points (3 errors = 0 score), warnings reduce 15 points.
    $config_score = max(0, 100 - ($config_errors * 33) - ($config_warnings * 15));

    $config_description = $config_errors === 0 && $config_warnings === 0
      ? (string) $this->t('All system performance settings are properly configured')
      : (string) $this->t('@errors critical issue(s) in configuration', [
        '@errors' => $config_errors,
      ]);

    $factors['system_config'] = [
      'score' => $config_score,
      'weight' => self::SCORE_WEIGHTS['system_config'],
      'label' => (string) $this->t('System Configuration'),
      'description' => $config_description,
    ];

    // 3. Cache Modules score.
    $stats = $cache_modules_results['stats'] ?? [];
    $required_count = $stats['required_count'] ?? 1;
    $required_missing = $stats['required_missing'] ?? 0;
    $installed_count = $stats['installed_count'] ?? 0;
    $total_modules = $stats['total_modules'] ?? 1;

    // Score based on required modules (primary) and optional modules (bonus).
    $required_installed = $required_count - $required_missing;
    $base_score = $required_count > 0 ? (int) round(($required_installed / $required_count) * 80) : 80;
    $bonus_score = $total_modules > $required_count
      ? (int) round((($installed_count - $required_installed) / ($total_modules - $required_count)) * 20)
      : 20;
    $modules_score = min(100, $base_score + $bonus_score);

    $modules_description = $required_missing === 0
      ? (string) $this->t('All required cache modules installed (@count/@total)', [
        '@count' => $installed_count,
        '@total' => $total_modules,
      ])
      : (string) $this->t('@missing required cache module(s) missing', ['@missing' => $required_missing]);

    $factors['cache_modules'] = [
      'score' => $modules_score,
      'weight' => self::SCORE_WEIGHTS['cache_modules'],
      'label' => (string) $this->t('Cache Modules'),
      'description' => $modules_description,
    ];

    // 4. Code Issues score (debug code + cache issues).
    $code_errors = $code_results['summary']['errors'] ?? 0;
    $code_warnings = $code_results['summary']['warnings'] ?? 0;
    // Each error reduces 10 points, warnings reduce 3 points.
    $code_score = max(0, 100 - ($code_errors * 10) - ($code_warnings * 3));

    $files_scanned = $code_results['stats']['files_scanned'] ?? 0;
    $debug_issues = $code_results['stats']['debug_issues'] ?? 0;
    $cache_issues = $code_results['stats']['cache_issues'] ?? 0;
    $total_issues = $code_results['stats']['issues_found'] ?? 0;

    if ($total_issues === 0) {
      $code_description = (string) $this->t('No issues in @count files', ['@count' => $files_scanned]);
    }
    else {
      $parts = [];
      if ($debug_issues > 0) {
        $parts[] = (string) $this->t('@count debug', ['@count' => $debug_issues]);
      }
      if ($cache_issues > 0) {
        $parts[] = (string) $this->t('@count cache', ['@count' => $cache_issues]);
      }
      $code_description = (string) $this->t('@issues issue(s): @details', [
        '@issues' => $total_issues,
        '@details' => implode(', ', $parts),
      ]);
    }

    $factors['code_issues'] = [
      'score' => $code_score,
      'weight' => self::SCORE_WEIGHTS['code_issues'],
      'label' => (string) $this->t('Code Analysis'),
      'description' => $code_description,
    ];

    // 5. Performance Patterns score (N+1 queries, S3 issues, deprecated code).
    $perf_errors = $perf_patterns_results['summary']['errors'] ?? 0;
    $perf_warnings = $perf_patterns_results['summary']['warnings'] ?? 0;
    // Each error reduces 15 points, warnings reduce 5 points.
    $perf_score = max(0, 100 - ($perf_errors * 15) - ($perf_warnings * 5));

    $perf_files_scanned = $perf_patterns_results['stats']['files_scanned'] ?? 0;
    $perf_issues_found = $perf_patterns_results['stats']['issues_found'] ?? 0;
    $perf_categories = $perf_patterns_results['stats']['categories'] ?? [];

    if ($perf_issues_found === 0) {
      $perf_description = (string) $this->t('No anti-patterns in @count files', ['@count' => $perf_files_scanned]);
    }
    else {
      $cat_parts = [];
      foreach ($perf_categories as $cat => $count) {
        $cat_label = $this->getPerformanceCategoryLabel($cat);
        $cat_parts[] = (string) $this->t('@count @cat', ['@count' => $count, '@cat' => $cat_label]);
      }
      $perf_description = (string) $this->t('@issues issue(s): @details', [
        '@issues' => $perf_issues_found,
        '@details' => implode(', ', array_slice($cat_parts, 0, 3)),
      ]);
    }

    $factors['performance_patterns'] = [
      'score' => $perf_score,
      'weight' => self::SCORE_WEIGHTS['performance_patterns'],
      'label' => (string) $this->t('Performance Anti-Patterns'),
      'description' => $perf_description,
    ];

    return [
      'factors' => $factors,
    ];
  }

  /**
   * Gets a human-readable label for a performance category.
   *
   * @param string $category
   *   The category key.
   *
   * @return string
   *   The human-readable label.
   */
  protected function getPerformanceCategoryLabel(string $category): string {
    return match ($category) {
      'n1_queries' => (string) $this->t('N+1'),
      'queries' => (string) $this->t('queries'),
      'security' => (string) $this->t('security'),
      's3_files' => (string) $this->t('S3/files'),
      'performance' => (string) $this->t('perf'),
      'hooks' => (string) $this->t('hooks'),
      'deprecated' => (string) $this->t('deprecated'),
      'views' => (string) $this->t('views'),
      default => $category,
    };
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      // =======================================================================
      // Sections that affect score - use faceted lists.
      // =======================================================================
      'production_issues' => [
        'label' => $this->t('Production Settings Issues'),
        'description' => $this->t('Critical issues: error display visible, Twig debug enabled, development modules active. These expose sensitive information and degrade performance.'),
        'file_key' => 'production_settings',
        'affects_score' => TRUE,
        'score_factor_key' => 'production_settings',
        'weight' => self::SCORE_WEIGHTS['production_settings'],
      ],
      'configuration_issues' => [
        'label' => $this->t('System Configuration Issues'),
        'description' => $this->t('Performance-critical settings that are misconfigured: CSS/JS aggregation disabled, page cache too low or disabled, Fast 404 not configured.'),
        'file_key' => 'configuration',
        'affects_score' => TRUE,
        'score_factor_key' => 'system_config',
        'weight' => self::SCORE_WEIGHTS['system_config'],
      ],
      'cache_modules_issues' => [
        'label' => $this->t('Cache Modules Issues'),
        'description' => $this->t('Required or recommended cache modules that are not installed.'),
        'file_key' => 'cache_modules',
        'affects_score' => TRUE,
        'score_factor_key' => 'cache_modules',
        'weight' => self::SCORE_WEIGHTS['cache_modules'],
      ],
      'code_issues' => [
        'label' => $this->t('Code Analysis Issues'),
        'description' => $this->t('Debug functions and cache anti-patterns found in custom PHP and Twig code.'),
        'file_key' => 'code_analysis',
        'affects_score' => TRUE,
        'score_factor_key' => 'code_issues',
        'weight' => self::SCORE_WEIGHTS['code_issues'],
      ],
      'performance_patterns' => [
        'label' => $this->t('Performance Anti-Patterns'),
        'description' => $this->t('N+1 queries, entity saves in loops, S3/remote file issues, queries in hooks, deprecated D7 functions, and missing access checks.'),
        'file_key' => 'performance_patterns',
        'affects_score' => TRUE,
        'score_factor_key' => 'performance_patterns',
        'weight' => self::SCORE_WEIGHTS['performance_patterns'],
      ],
      // =======================================================================
      // Informational sections - use tables (no score impact).
      // =======================================================================
      'production_status' => [
        'label' => $this->t('Production Settings Status'),
        'description' => $this->t('Overview of all production-related settings and their current values.'),
        'file_key' => 'production_settings',
        'affects_score' => FALSE,
      ],
      'configuration_status' => [
        'label' => $this->t('System Configuration Status'),
        'description' => $this->t('Overview of all performance configuration settings.'),
        'file_key' => 'configuration',
        'affects_score' => FALSE,
      ],
      'cache_modules_status' => [
        'label' => $this->t('Cache Modules Status'),
        'description' => $this->t('Overview of cache modules and their installation status.'),
        'file_key' => 'cache_modules',
        'affects_score' => FALSE,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    $files = $data['_files'] ?? [];
    $user_type = $data['user_type'] ?? 'mixed';

    return match ($check_id) {
      // Faceted lists for scoring sections.
      'production_issues' => $this->buildProductionIssues($files),
      'configuration_issues' => $this->buildConfigurationIssues($files),
      'cache_modules_issues' => $this->buildCacheModulesIssues($files),
      'code_issues' => $this->buildCodeIssues($files),
      'performance_patterns' => $this->buildPerformancePatternIssues($files),
      // Tables for informational sections.
      'production_status' => $this->buildProductionStatus($files),
      'configuration_status' => $this->buildConfigurationStatus($files),
      'cache_modules_status' => $this->buildCacheModulesStatus($files, $user_type),
      default => [],
    };
  }

  /**
   * Builds faceted list for production settings issues.
   */
  protected function buildProductionIssues(array $files): array {
    $prod_data = $files['production_settings'] ?? [];
    $results = $prod_data['results'] ?? [];

    // Filter only issues (errors and warnings).
    $issue_results = array_filter($results, fn($r) => in_array($r['severity'] ?? '', ['error', 'warning'], TRUE));

    if (empty($issue_results)) {
      return ['message' => $this->ui->message(
        (string) $this->t('All production settings are correctly configured. Error display is hidden, Twig debug is disabled, and no development modules are active.'),
        'success'
      )];
    }

    $issues = [];
    foreach ($issue_results as $result) {
      $severity = $result['severity'] ?? 'warning';
      $code = $result['code'] ?? '';
      $details = $result['details'] ?? [];

      // Determine category based on code.
      $category = 'settings';
      if (str_contains($code, 'MODULE')) {
        $category = 'modules';
      }
      elseif (str_contains($code, 'TWIG')) {
        $category = 'twig';
      }
      elseif (str_contains($code, 'ERROR')) {
        $category = 'logging';
      }

      // Build description with recommendation.
      $description = $result['message'] ?? '';
      if (!empty($details['recommendation'])) {
        $description .= ' ' . $details['recommendation'];
      }
      elseif (!empty($details['fix'])) {
        $description .= ' ' . $details['fix'];
      }

      $issues[] = $this->ui->issue([
        'severity' => $severity,
        'code' => $code,
        'label' => $this->getIssueLabel($code, $details),
        'file' => $details['setting'] ?? $details['module'] ?? '',
        'description' => $description,
        'tags' => [$category, 'production'],
        'custom_data' => [
          'category' => $category,
          'setting' => $details['setting'] ?? '',
        ],
      ]);
    }

    $custom_filters = [
      'category' => [
        'label' => (string) $this->t('Category'),
        'attribute' => 'data-category',
      ],
    ];

    return [
      'list' => $this->ui->issueList(
        $issues,
        (string) $this->t('No issues match the selected filters.'),
        $custom_filters
      ),
    ];
  }

  /**
   * Builds table for production settings status (informational).
   */
  protected function buildProductionStatus(array $files): array {
    $prod_data = $files['production_settings'] ?? [];
    $results = $prod_data['results'] ?? [];

    $rows = [];

    // Error display setting.
    $error_result = $this->findResultByCode($results, ['ERROR_DISPLAY_HIDDEN', 'ERROR_DISPLAY_VISIBLE']);
    $error_hidden = ($error_result['code'] ?? '') === 'ERROR_DISPLAY_HIDDEN';
    $error_value = $error_result['details']['value'] ?? 'unknown';

    $error_display_labels = [
      'hide' => (string) $this->t('Hidden'),
      'some' => (string) $this->t('Errors and warnings'),
      'all' => (string) $this->t('All messages'),
      'verbose' => (string) $this->t('All with backtrace'),
    ];

    $rows[] = $this->ui->row([
      (string) $this->t('Error Display'),
      $this->ui->cell(
        $error_display_labels[$error_value] ?? $error_value,
        ['status' => $error_hidden ? 'ok' : 'error']
      ),
      $error_hidden ? (string) $this->t('Correctly hidden') : (string) $this->t('Visible to users'),
    ]);

    // Twig debug setting.
    $twig_result = $this->findResultByCode($results, ['TWIG_DEBUG_DISABLED', 'TWIG_DEBUG_ENABLED']);
    $twig_disabled = ($twig_result['code'] ?? '') === 'TWIG_DEBUG_DISABLED';

    $rows[] = $this->ui->row([
      (string) $this->t('Twig Debug'),
      $this->ui->cell(
        $twig_disabled ? (string) $this->t('Disabled') : (string) $this->t('Enabled'),
        ['status' => $twig_disabled ? 'ok' : 'error']
      ),
      $twig_disabled ? (string) $this->t('Production safe') : (string) $this->t('Exposes template paths'),
    ]);

    // Twig auto_reload setting.
    $reload_result = $this->findResultByCode($results, ['TWIG_AUTO_RELOAD_DISABLED', 'TWIG_AUTO_RELOAD_ENABLED']);
    $reload_disabled = ($reload_result['code'] ?? '') === 'TWIG_AUTO_RELOAD_DISABLED';

    $rows[] = $this->ui->row([
      (string) $this->t('Twig Auto-reload'),
      $this->ui->cell(
        $reload_disabled ? (string) $this->t('Disabled') : (string) $this->t('Enabled'),
        ['status' => $reload_disabled ? 'ok' : 'warning']
      ),
      $reload_disabled ? (string) $this->t('Production safe') : (string) $this->t('Impacts performance'),
    ]);

    // Development modules.
    $dev_modules = [
      'DEVEL_MODULE_ENABLED' => ['name' => 'Devel', 'severity' => 'warning'],
      'KINT_MODULE_ENABLED' => ['name' => 'Kint', 'severity' => 'warning'],
      'WEBPROFILER_MODULE_ENABLED' => ['name' => 'Webprofiler', 'severity' => 'error'],
    ];

    foreach ($dev_modules as $code => $info) {
      $module_result = $this->findResultByCode($results, [$code]);
      $is_enabled = !empty($module_result);

      $rows[] = $this->ui->row([
        (string) $this->t('@module Module', ['@module' => $info['name']]),
        $this->ui->cell(
          $is_enabled ? (string) $this->t('Enabled') : (string) $this->t('Not installed'),
          ['status' => $is_enabled ? $info['severity'] : 'ok']
        ),
        $is_enabled ? (string) $this->t('Should be disabled in production') : (string) $this->t('Not active'),
      ]);
    }

    $headers = [
      (string) $this->t('Setting'),
      (string) $this->t('Status'),
      (string) $this->t('Notes'),
    ];

    return ['table' => $this->ui->table($headers, $rows)];
  }

  /**
   * Gets a human-readable label for an issue code.
   */
  protected function getIssueLabel(string $code, array $details): string {
    return match ($code) {
      'ERROR_DISPLAY_VISIBLE' => (string) $this->t('Error Display Visible'),
      'TWIG_DEBUG_ENABLED' => (string) $this->t('Twig Debug Enabled'),
      'TWIG_AUTO_RELOAD_ENABLED' => (string) $this->t('Twig Auto-reload Enabled'),
      'DEVEL_MODULE_ENABLED' => (string) $this->t('Devel Module Active'),
      'KINT_MODULE_ENABLED' => (string) $this->t('Kint Module Active'),
      'WEBPROFILER_MODULE_ENABLED' => (string) $this->t('Webprofiler Module Active'),
      'CSS_AGGREGATION_DISABLED' => (string) $this->t('CSS Aggregation Disabled'),
      'JS_AGGREGATION_DISABLED' => (string) $this->t('JavaScript Aggregation Disabled'),
      'PAGE_CACHE_DISABLED' => (string) $this->t('Page Cache Disabled'),
      'PAGE_CACHE_BELOW_RECOMMENDED' => (string) $this->t('Page Cache Below Threshold'),
      'FAST_404_DISABLED' => (string) $this->t('Fast 404 Not Configured'),
      'CACHE_MODULE_MISSING' => $details['label'] ?? (string) $this->t('Cache Module Missing'),
      default => $details['label'] ?? $code,
    };
  }

  /**
   * Builds faceted list for cache modules issues.
   */
  protected function buildCacheModulesIssues(array $files): array {
    $cache_data = $files['cache_modules'] ?? [];
    $results = $cache_data['results'] ?? [];

    // Filter only missing modules (errors and warnings).
    $missing_results = array_filter($results, fn($r) => in_array($r['severity'] ?? '', ['error', 'warning'], TRUE));

    if (empty($missing_results)) {
      return ['message' => $this->ui->message(
        (string) $this->t('All required and recommended cache modules are installed.'),
        'success'
      )];
    }

    $issues = [];
    foreach ($missing_results as $result) {
      $severity = $result['severity'] ?? 'warning';
      $details = $result['details'] ?? [];
      $is_required = !empty($details['required']);

      $issues[] = $this->ui->issue([
        'severity' => $severity,
        'code' => $result['code'] ?? 'CACHE_MODULE_MISSING',
        'label' => $details['label'] ?? $details['module'] ?? '',
        'file' => $details['module'] ?? '',
        'description' => $details['description'] ?? $result['message'] ?? '',
        'tags' => [$is_required ? 'required' : 'recommended', 'cache'],
        'custom_data' => [
          'priority' => $is_required ? 'required' : 'recommended',
          'module' => $details['module'] ?? '',
        ],
      ]);
    }

    $custom_filters = [
      'priority' => [
        'label' => (string) $this->t('Priority'),
        'attribute' => 'data-priority',
      ],
    ];

    return [
      'list' => $this->ui->issueList(
        $issues,
        (string) $this->t('No issues match the selected filters.'),
        $custom_filters
      ),
    ];
  }

  /**
   * Builds table for cache modules status (informational).
   */
  protected function buildCacheModulesStatus(array $files, string $user_type): array {
    $cache_data = $files['cache_modules'] ?? [];
    $results = $cache_data['results'] ?? [];
    $stats = $cache_data['stats'] ?? [];

    $user_type_labels = [
      'anonymous' => (string) $this->t('Anonymous users'),
      'registered' => (string) $this->t('Registered users'),
      'mixed' => (string) $this->t('Mixed (anonymous + registered)'),
    ];

    $content = [];

    $content['info'] = $this->ui->message(
      (string) $this->t('User type configuration: @type (from Modules Analysis settings). Installed: @installed/@total modules.', [
        '@type' => $user_type_labels[$user_type] ?? $user_type,
        '@installed' => $stats['installed_count'] ?? 0,
        '@total' => $stats['total_modules'] ?? 0,
      ]),
      'info'
    );

    $rows = [];
    foreach ($results as $result) {
      $details = $result['details'] ?? [];
      $is_installed = !empty($details['installed']);
      $is_required = !empty($details['required']);

      $status_icon = $is_installed ? $this->ui->icon('check') : $this->ui->icon('cross');

      if ($is_installed) {
        $priority_badge = $this->ui->badge((string) $this->t('Installed'), 'success');
      }
      else {
        $priority_badge = $is_required
          ? $this->ui->badge((string) $this->t('Required'), 'error')
          : $this->ui->badge((string) $this->t('Recommended'), 'info');
      }

      $rows[] = $this->ui->row([
        $this->ui->cell($status_icon, ['align' => 'center']),
        $priority_badge,
        $this->ui->itemName((string) ($details['label'] ?? ''), (string) ($details['module'] ?? '')),
        $details['description'] ?? '',
      ]);
    }

    $headers = [
      $this->ui->header((string) $this->t('Status'), 'center'),
      (string) $this->t('Priority'),
      (string) $this->t('Module'),
      (string) $this->t('Description'),
    ];

    $content['table'] = $this->ui->table($headers, $rows);

    return $content;
  }

  /**
   * Builds faceted list for configuration issues.
   */
  protected function buildConfigurationIssues(array $files): array {
    $config_data = $files['configuration'] ?? [];
    $results = $config_data['results'] ?? [];

    // Filter only issues (errors and warnings).
    $issue_results = array_filter($results, fn($r) => in_array($r['severity'] ?? '', ['error', 'warning'], TRUE));

    if (empty($issue_results)) {
      return ['message' => $this->ui->message(
        (string) $this->t('All performance configuration settings are correctly configured.'),
        'success'
      )];
    }

    $issues = [];
    foreach ($issue_results as $result) {
      $severity = $result['severity'] ?? 'warning';
      $code = $result['code'] ?? '';
      $details = $result['details'] ?? [];

      // Determine category.
      $category = 'config';
      if (str_contains($code, 'CSS') || str_contains($code, 'JS')) {
        $category = 'aggregation';
      }
      elseif (str_contains($code, 'CACHE')) {
        $category = 'cache';
      }
      elseif (str_contains($code, '404')) {
        $category = 'fast404';
      }

      $issues[] = $this->ui->issue([
        'severity' => $severity,
        'code' => $code,
        'label' => $this->getIssueLabel($code, $details),
        'file' => $details['setting'] ?? '',
        'description' => $result['message'] ?? '',
        'tags' => [$category, 'performance'],
        'custom_data' => [
          'category' => $category,
          'setting' => $details['setting'] ?? '',
        ],
      ]);
    }

    $custom_filters = [
      'category' => [
        'label' => (string) $this->t('Category'),
        'attribute' => 'data-category',
      ],
    ];

    return [
      'list' => $this->ui->issueList(
        $issues,
        (string) $this->t('No issues match the selected filters.'),
        $custom_filters
      ),
    ];
  }

  /**
   * Builds table for configuration status (informational).
   */
  protected function buildConfigurationStatus(array $files): array {
    $config_data = $files['configuration'] ?? [];
    $results = $config_data['results'] ?? [];

    $rows = [];

    // CSS Aggregation.
    $css_result = $this->findResultByCode($results, ['CSS_AGGREGATION_ENABLED', 'CSS_AGGREGATION_DISABLED']);
    $css_enabled = ($css_result['code'] ?? '') === 'CSS_AGGREGATION_ENABLED';
    $rows[] = $this->ui->row([
      (string) $this->t('CSS Aggregation'),
      $this->ui->cell(
        $css_enabled ? (string) $this->t('Enabled') : (string) $this->t('Disabled'),
        ['status' => $css_enabled ? 'ok' : 'error']
      ),
      $css_enabled ? (string) $this->t('Reducing HTTP requests') : (string) $this->t('Multiple HTTP requests for CSS'),
    ]);

    // JS Aggregation.
    $js_result = $this->findResultByCode($results, ['JS_AGGREGATION_ENABLED', 'JS_AGGREGATION_DISABLED']);
    $js_enabled = ($js_result['code'] ?? '') === 'JS_AGGREGATION_ENABLED';
    $rows[] = $this->ui->row([
      (string) $this->t('JavaScript Aggregation'),
      $this->ui->cell(
        $js_enabled ? (string) $this->t('Enabled') : (string) $this->t('Disabled'),
        ['status' => $js_enabled ? 'ok' : 'error']
      ),
      $js_enabled ? (string) $this->t('Reducing HTTP requests') : (string) $this->t('Multiple HTTP requests for JS'),
    ]);

    // Page Cache.
    $cache_result = $this->findResultByCode($results, [
      'PAGE_CACHE_OK',
      'PAGE_CACHE_BELOW_RECOMMENDED',
      'PAGE_CACHE_DISABLED',
    ]);
    $cache_value = $cache_result['details']['value'] ?? 0;
    $cache_level = $cache_result['details']['level'] ?? 'unknown';
    $recommended = $cache_result['details']['recommended'] ?? $this->getRecommendedCacheMaxAge();

    $cache_status = $cache_level === 'ok' ? 'ok' : 'error';
    $cache_note = match ($cache_level) {
      'ok' => (string) $this->t('Meets threshold (@recommended)', ['@recommended' => $this->formatDuration($recommended)]),
      'below_recommended' => (string) $this->t('Below threshold (@recommended)', ['@recommended' => $this->formatDuration($recommended)]),
      default => (string) $this->t('Caching disabled'),
    };

    $rows[] = $this->ui->row([
      (string) $this->t('Page Cache Max-Age'),
      $this->ui->cell($this->formatDuration($cache_value), ['status' => $cache_status]),
      $cache_note,
    ]);

    // Fast 404.
    $fast404_result = $this->findResultByCode($results, ['FAST_404_ENABLED', 'FAST_404_DISABLED']);
    $fast404_enabled = ($fast404_result['code'] ?? '') === 'FAST_404_ENABLED';
    $rows[] = $this->ui->row([
      (string) $this->t('Fast 404'),
      $this->ui->cell(
        $fast404_enabled ? (string) $this->t('Enabled') : (string) $this->t('Disabled'),
        ['status' => $fast404_enabled ? 'ok' : 'warning']
      ),
      $fast404_enabled ? (string) $this->t('Quick handling of missing files') : (string) $this->t('Full bootstrap for 404s'),
    ]);

    $headers = [
      (string) $this->t('Setting'),
      (string) $this->t('Value'),
      (string) $this->t('Notes'),
    ];

    return [
      'table' => $this->ui->table($headers, $rows),
      'thresholds' => $this->ui->message(
        (string) $this->t('Configured recommended cache max-age: @recommended (configurable in Performance Analysis settings)', [
          '@recommended' => $this->formatDuration($recommended),
        ]),
        'info'
      ),
    ];
  }

  /**
   * Builds faceted list for code analysis issues.
   */
  protected function buildCodeIssues(array $files): array {
    $code_data = $files['code_analysis'] ?? [];
    $results = $code_data['results'] ?? [];
    $stats = $code_data['stats'] ?? [];

    // Filter out the "no issues" result.
    $issue_results = array_filter($results, fn($r) => ($r['code'] ?? '') !== 'NO_CODE_ISSUES');

    if (empty($issue_results)) {
      $dirs = implode(', ', $stats['directories'] ?? []);
      return ['message' => $this->ui->message(
        (string) $this->t('No debug code or cache issues found in @count scanned files. Directories: @dirs', [
          '@count' => $stats['files_scanned'] ?? 0,
          '@dirs' => $dirs,
        ]),
        'success'
      )];
    }

    $issues = [];
    foreach ($issue_results as $result) {
      $severity = $result['severity'] ?? 'notice';
      $code = $result['code'] ?? '';
      $details = $result['details'] ?? [];

      // Determine category based on code.
      $category = $this->getCodeCategory($code);

      // Determine file type.
      $file = $details['file'] ?? '';
      $file_type = str_ends_with($file, '.twig') ? 'twig' : 'php';

      // Build description with line number and recommendation.
      $line = $details['line'] ?? 0;
      $description = $result['message'] ?? '';
      if (!empty($details['recommendation'])) {
        $description .= ' ' . (string) $this->t('Recommendation: @rec', ['@rec' => $details['recommendation']]);
      }

      $issues[] = $this->ui->issue([
        'severity' => $severity,
        'code' => $code,
        'label' => $this->getCodeIssueLabel($code),
        'file' => $file,
        'line' => $line,
        'description' => $description,
        'tags' => [$category, $file_type],
        'code_snippet' => $this->buildCodeSnippet($details['code_context'] ?? []),
        'custom_data' => [
          'category' => $category,
          'file_type' => $file_type,
        ],
      ]);
    }

    $custom_filters = [
      'category' => [
        'label' => (string) $this->t('Category'),
        'attribute' => 'data-category',
      ],
      'file_type' => [
        'label' => (string) $this->t('File Type'),
        'attribute' => 'data-file_type',
      ],
    ];

    // Stats info.
    $dirs = implode(', ', $stats['directories'] ?? []);
    $content = [];
    $content['stats'] = $this->ui->message(
      (string) $this->t('@files files scanned | @debug debug issues | @cache cache issues | Directories: @dirs', [
        '@files' => $stats['files_scanned'] ?? 0,
        '@debug' => $stats['debug_issues'] ?? 0,
        '@cache' => $stats['cache_issues'] ?? 0,
        '@dirs' => $dirs,
      ]),
      'info'
    );

    $content['list'] = $this->ui->issueList(
      $issues,
      (string) $this->t('No issues match the selected filters.'),
      $custom_filters
    );

    return $content;
  }

  /**
   * Builds faceted list for performance pattern issues.
   */
  protected function buildPerformancePatternIssues(array $files): array {
    $perf_data = $files['performance_patterns'] ?? [];
    $results = $perf_data['results'] ?? [];
    $stats = $perf_data['stats'] ?? [];

    // Filter out the "no issues" result.
    $issue_results = array_filter($results, fn($r) => ($r['code'] ?? '') !== 'NO_PERF_ISSUES');

    if (empty($issue_results)) {
      $dirs = implode(', ', $stats['directories'] ?? []);
      return ['message' => $this->ui->message(
        (string) $this->t('No performance anti-patterns found in @count scanned files. Directories: @dirs', [
          '@count' => $stats['files_scanned'] ?? 0,
          '@dirs' => $dirs,
        ]),
        'success'
      )];
    }

    $issues = [];
    foreach ($issue_results as $result) {
      $severity = $result['severity'] ?? 'warning';
      $code = $result['code'] ?? '';
      $details = $result['details'] ?? [];

      // Get category from details.
      $category = $details['category'] ?? $this->getPerformanceCategory($code);

      // Determine file type.
      $file = $details['file'] ?? '';
      $file_type = 'php';
      if (str_ends_with($file, '.module')) {
        $file_type = 'module';
      }
      elseif (str_ends_with($file, '.yml')) {
        $file_type = 'yml';
      }

      // Build description with recommendation.
      $description = $result['message'] ?? '';
      if (!empty($details['recommendation'])) {
        $description .= ' ' . (string) $this->t('Recommendation: @rec', ['@rec' => $details['recommendation']]);
      }

      $issues[] = $this->ui->issue([
        'severity' => $severity,
        'code' => $code,
        'label' => $this->getPerfPatternLabel($code),
        'file' => $file,
        'line' => $details['line'] ?? 0,
        'description' => $description,
        'tags' => [$category, $file_type],
        'code_snippet' => $this->buildCodeSnippet($details['code_context'] ?? []),
        'custom_data' => [
          'category' => $category,
          'file_type' => $file_type,
        ],
      ]);
    }

    $custom_filters = [
      'category' => [
        'label' => (string) $this->t('Category'),
        'attribute' => 'data-category',
      ],
      'file_type' => [
        'label' => (string) $this->t('File Type'),
        'attribute' => 'data-file_type',
      ],
    ];

    // Stats info with category breakdown.
    $dirs = implode(', ', $stats['directories'] ?? []);
    $cat_parts = [];
    foreach ($stats['categories'] ?? [] as $cat => $count) {
      $cat_label = $this->getPerformanceCategoryLabel($cat);
      $cat_parts[] = (string) $this->t('@count @cat', ['@count' => $count, '@cat' => $cat_label]);
    }
    $cat_summary = !empty($cat_parts) ? ' | ' . implode(', ', $cat_parts) : '';

    $content = [];
    $content['stats'] = $this->ui->message(
      (string) $this->t('@files files scanned | @issues issues found@categories | Directories: @dirs', [
        '@files' => $stats['files_scanned'] ?? 0,
        '@issues' => $stats['issues_found'] ?? 0,
        '@categories' => $cat_summary,
        '@dirs' => $dirs,
      ]),
      'info'
    );

    $content['list'] = $this->ui->issueList(
      $issues,
      (string) $this->t('No issues match the selected filters.'),
      $custom_filters
    );

    return $content;
  }

  /**
   * Gets a human-readable label for a performance pattern issue code.
   */
  protected function getPerfPatternLabel(string $code): string {
    return match ($code) {
      // N+1 queries.
      'N1_ENTITY_LOAD' => (string) $this->t('N+1 Entity Load'),
      'N1_REFERENCED_ENTITIES' => (string) $this->t('N+1 Referenced Entities'),
      'SAVE_IN_LOOP' => (string) $this->t('Save in Loop'),
      // Query issues.
      'QUERY_NO_RANGE' => (string) $this->t('Query Without Range'),
      'DB_SELECT_NO_LIMIT' => (string) $this->t('DB Select No Limit'),
      'MISSING_ACCESS_CHECK' => (string) $this->t('Missing Access Check'),
      // S3/File issues.
      'GETIMAGESIZE_REMOTE' => (string) $this->t('getimagesize() Remote'),
      'FILE_GET_CONTENTS_VARIABLE' => (string) $this->t('file_get_contents() Variable'),
      'REALPATH_USAGE' => (string) $this->t('realpath() Usage'),
      'FOPEN_VARIABLE' => (string) $this->t('fopen() Variable'),
      'IS_FILE_VARIABLE' => (string) $this->t('is_file() Variable'),
      // Hook issues.
      'QUERY_IN_PREPROCESS' => (string) $this->t('Query in Preprocess'),
      'QUERY_IN_PAGE_ATTACHMENTS' => (string) $this->t('Query in Page Attachments'),
      // Static services.
      'STATIC_SERVICE_IN_LOOP' => (string) $this->t('Static Service in Loop'),
      // Deprecated D7.
      'DEPRECATED_NODE_LOAD' => (string) $this->t('node_load() Deprecated'),
      'DEPRECATED_USER_LOAD' => (string) $this->t('user_load() Deprecated'),
      'DEPRECATED_ENTITY_LOAD' => (string) $this->t('entity_load() Deprecated'),
      'DEPRECATED_DB_QUERY' => (string) $this->t('db_query() Deprecated'),
      'DEPRECATED_DB_SELECT' => (string) $this->t('db_select() Deprecated'),
      // Security.
      'LOAD_BY_PROPERTIES' => (string) $this->t('loadByProperties() No Access'),
      // Views.
      'VIEWS_NO_CACHE' => (string) $this->t('Views No Cache'),
      default => $code,
    };
  }

  /**
   * Gets the category for a code issue.
   */
  protected function getCodeCategory(string $code): string {
    // Debug categories.
    $debug_codes = [
      'DEBUG_DPM', 'DEBUG_DSM', 'DEBUG_DVM', 'DEBUG_DPR', 'DEBUG_DPQ',
      'DEBUG_DARGS', 'DEBUG_DDEBUG_BACKTRACE', 'DEBUG_KPR',
      'DEBUG_KINT', 'DEBUG_KSM', 'DEBUG_KINT_STATIC',
      'DEBUG_DD', 'DEBUG_DUMP',
      'DEBUG_VAR_DUMP', 'DEBUG_PRINT_R', 'DEBUG_VAR_EXPORT_ECHO',
      'DEBUG_PRINT_BACKTRACE', 'DEBUG_BACKTRACE_OUTPUT',
      'DEBUG_ERROR_LOG_DUMP',
      'TWIG_DEBUG_DUMP', 'TWIG_DEBUG_KINT', 'TWIG_DEBUG_DEVEL', 'TWIG_DEBUG_CONTEXT',
    ];

    if (in_array($code, $debug_codes, TRUE)) {
      return 'debug';
    }

    return 'cache';
  }

  /**
   * Gets a human-readable label for a code issue.
   */
  protected function getCodeIssueLabel(string $code): string {
    return match ($code) {
      // Debug codes.
      'DEBUG_DPM' => 'dpm()',
      'DEBUG_DSM' => 'dsm()',
      'DEBUG_DVM' => 'dvm()',
      'DEBUG_DPR' => 'dpr()',
      'DEBUG_DPQ' => 'dpq()',
      'DEBUG_DARGS' => 'dargs()',
      'DEBUG_DDEBUG_BACKTRACE' => 'ddebug_backtrace()',
      'DEBUG_KPR' => 'kpr()',
      'DEBUG_KINT' => 'kint()',
      'DEBUG_KSM' => 'ksm()',
      'DEBUG_KINT_STATIC' => 'Kint::d()',
      'DEBUG_DD' => 'dd()',
      'DEBUG_DUMP' => 'dump()',
      'DEBUG_VAR_DUMP' => 'var_dump()',
      'DEBUG_PRINT_R' => 'print_r()',
      'DEBUG_VAR_EXPORT_ECHO' => 'var_export() echo',
      'DEBUG_PRINT_BACKTRACE' => 'debug_print_backtrace()',
      'DEBUG_BACKTRACE_OUTPUT' => 'debug_backtrace() output',
      'DEBUG_ERROR_LOG_DUMP' => 'error_log() with dump',
      'TWIG_DEBUG_DUMP' => '{{ dump() }}',
      'TWIG_DEBUG_KINT' => '{{ kint() }}',
      'TWIG_DEBUG_DEVEL' => '{{ devel_dump() }}',
      'TWIG_DEBUG_CONTEXT' => '_context output',
      // Cache codes.
      'CACHE_MAX_AGE_ZERO' => 'Cache max-age 0',
      'CACHE_DISABLED_ASSIGNMENT' => 'Cache disabled',
      'PAGE_CACHE_KILL_SWITCH' => 'Kill switch triggered',
      'CACHE_DISABLED_PREVIEW' => 'Cache disabled (preview)',
      'CACHE_CLEAR_ALL' => 'Cache clear all',
      'TWIG_CACHE_ZERO' => 'Twig cache 0',
      'GENERIC_NODE_LIST_TAG' => 'Generic node_list tag',
      'GENERIC_TAXONOMY_LIST_TAG' => 'Generic taxonomy tag',
      'GENERIC_USER_LIST_TAG' => 'Generic user_list tag',
      'GENERIC_CONFIG_TAG' => 'Generic config tag',
      'RENDERED_TAG_USED' => 'Rendered tag used',
      'PERMANENT_CACHE_NO_TAGS' => 'Permanent cache no tags',
      'USER_DATA_NO_CONTEXT' => 'User data no context',
      'MARKUP_WITHOUT_CACHE' => 'Markup without cache',
      'TIME_IN_RENDER' => 'Time in render',
      'REQUEST_DATA_IN_RENDER' => 'Request data in render',
      'SESSION_DATA_IN_RENDER' => 'Session data in render',
      'CACHE_PERMANENT' => 'Cache::PERMANENT',
      'ENTITY_QUERY_NO_CACHE_TAG' => 'EntityQuery no cache tag',
      'ALTER_WITHOUT_CACHE_BUBBLE' => 'Alter without cache bubble',
      'DRUPAL_STATIC_RESET_ALL' => 'drupal_static_reset()',
      'TWIG_RAW_POSSIBLE_USER' => 'Twig raw user input',
      default => $code,
    };
  }

  /**
   * Builds a code snippet render array from code context.
   */
  protected function buildCodeSnippet(array $code_context): ?array {
    if (empty($code_context['lines'])) {
      return NULL;
    }

    return $this->ui->code(
      $code_context['lines'],
      [
        'highlight_line' => $code_context['highlight_line'] ?? NULL,
        'severity' => 'error',
      ]
    );
  }

  /**
   * Finds a result by one of the given codes.
   */
  protected function findResultByCode(array $results, array $codes): array {
    foreach ($results as $result) {
      if (in_array($result['code'] ?? '', $codes, TRUE)) {
        return $result;
      }
    }
    return [];
  }

  /**
   * Extracts code context around a specific line.
   *
   * @param array $lines
   *   Array of all lines in the file.
   * @param int $target_line
   *   The line number with the issue (1-indexed).
   * @param int $context_lines
   *   Number of lines to show before and after.
   *
   * @return array
   *   Array with 'lines' containing line data and 'highlight_line' with target.
   */
  protected function extractCodeContext(array $lines, int $target_line, int $context_lines = 2): array {
    $total_lines = count($lines);
    $target_index = $target_line - 1;

    // Calculate start and end indices.
    $start_index = max(0, $target_index - $context_lines);
    $end_index = min($total_lines - 1, $target_index + $context_lines);

    $context = [
      'lines' => [],
      'highlight_line' => $target_line,
    ];

    for ($i = $start_index; $i <= $end_index; $i++) {
      $line_number = $i + 1;
      $context['lines'][] = [
        'number' => $line_number,
        'content' => $lines[$i] ?? '',
        'is_highlight' => ($line_number === $target_line),
      ];
    }

    return $context;
  }

}
