<?php

declare(strict_types=1);

namespace Drupal\audit_phpcs\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Process\Process;

/**
 * Runs PHP CodeSniffer analysis against Drupal coding standards.
 *
 * This analyzer uses drupal/coder which provides two complementary standards:
 * - Drupal (~115 sniffs): Format, syntax, documentation, naming conventions
 * - DrupalPractice (~37 sniffs): Architectural patterns, proper API usage
 *
 * Both standards are always applied together as recommended by Drupal.org
 * for contributed modules and custom code.
 */
#[AuditAnalyzer(
  id: 'phpcs',
  label: new TranslatableMarkup('PHP CodeSniffer'),
  description: new TranslatableMarkup('Checks code against Drupal coding standards using drupal/coder (Drupal + DrupalPractice standards).'),
  menu_title: new TranslatableMarkup('CodeSniffer'),
  output_directory: 'phpcs',
  weight: 2,
)]
class PhpcsAnalyzer extends AuditAnalyzerBase {

  /**
   * Score weights for different factors.
   *
   * Using a single factor for coding standards compliance.
   * Errors and warnings are combined in a single section with faceted filters.
   */
  protected const SCORE_WEIGHTS = [
    'coding_standards' => 100,
  ];

  /**
   * Penalty multiplier applied to issue percentage.
   *
   * With multiplier 2.5:
   * - 4% issues → score 90
   * - 10% issues → score 75
   * - 20% issues → score 50
   * - 40%+ issues → score 0
   */
  protected const PENALTY_MULTIPLIER = 2.5;

  /**
   * Default file extensions to analyze (Drupal-specific).
   */
  protected const DEFAULT_EXTENSIONS = 'php,module,inc,install,test,profile,theme,info,txt,md,yml';

  /**
   * Default minimum severity level (1-10, 5 is balanced).
   */
  protected const DEFAULT_MIN_SEVERITY = 5;

  /**
   * Default timeout in seconds.
   */
  protected const DEFAULT_TIMEOUT = 300;

  /**
   * Default maximum violations to display.
   */
  protected const DEFAULT_MAX_VIOLATIONS = 2000;

  /**
   * Default parallel processes.
   */
  protected const DEFAULT_PARALLEL = 4;

  /**
   * Default report width.
   */
  protected const DEFAULT_REPORT_WIDTH = 120;

  /**
   * Common sniffs that can be excluded for legacy code.
   */
  protected const EXCLUDABLE_SNIFFS = [
    'Drupal.Commenting.DocComment' => 'DocBlock comment format validation',
    'Drupal.Commenting.FunctionComment' => 'Function/method comment requirements',
    'Drupal.Commenting.FileComment' => 'File-level comment requirements',
    'Drupal.Files.LineLength' => 'Line length limit (80 characters)',
    'DrupalPractice.CodeAnalysis.VariableAnalysis' => 'Unused variable detection',
    'DrupalPractice.Objects.GlobalDrupal' => 'Global \Drupal:: usage warnings',
  ];

  /**
   * The config factory service.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->configFactory = $container->get('config.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function checkRequirements(): array {
    $warnings = [];

    // Check for PHPCS binary.
    $phpcs_path = $this->findBinary('phpcs');
    if (!$phpcs_path) {
      $warnings[] = (string) $this->t('PHP CodeSniffer (phpcs) is not installed.');
      return $warnings;
    }

    // Check for PHPCBF binary (auto-fixer).
    $phpcbf_path = $this->findBinary('phpcbf');
    if (!$phpcbf_path) {
      $warnings[] = (string) $this->t('PHP Code Beautifier (phpcbf) is not installed. Auto-fix functionality will not be available.');
    }

    // Check for Drupal standards - both registered and available.
    $standards = $this->getInstalledStandards($phpcs_path);
    $drupal_registered = in_array('Drupal', $standards, TRUE);
    $drupal_practice_registered = in_array('DrupalPractice', $standards, TRUE);

    if (!$drupal_registered || !$drupal_practice_registered) {
      // Check if standards exist but are not registered.
      $coder_path = $this->findCoderStandardsPath();

      if ($coder_path) {
        // Check if the coder version is compatible with current PHPCS.
        $compatibility = $this->checkCoderCompatibility($coder_path);

        if (!$compatibility['compatible']) {
          $warnings[] = (string) $this->t('<strong>Incompatible drupal/coder version!</strong> Your drupal/coder (@coder_version) is incompatible with PHP_CodeSniffer @phpcs_version. Run: <code>composer require --dev drupal/coder:^8.3.0 dealerdirect/phpcodesniffer-composer-installer</code>', [
            '@coder_version' => $compatibility['coder_version'] ?? 'unknown',
            '@phpcs_version' => $compatibility['phpcs_version'] ?? 'unknown',
          ]);
        }
        else {
          // Standards exist but not registered - provide specific fix.
          $missing = [];
          if (!$drupal_registered) {
            $missing[] = 'Drupal';
          }
          if (!$drupal_practice_registered) {
            $missing[] = 'DrupalPractice';
          }
          $warnings[] = (string) $this->t('The @standards standard(s) exist but are not registered with PHPCS. Run: <code>composer require --dev dealerdirect/phpcodesniffer-composer-installer</code> to auto-register, or manually: <code>./vendor/bin/phpcs --config-set installed_paths @path</code>', [
            '@standards' => implode(' and ', $missing),
            '@path' => $coder_path,
          ]);
        }
      }
      else {
        // Standards not found at all.
        $warnings[] = (string) $this->t('Drupal coding standards are not installed. Run: <code>composer require --dev drupal/coder dealerdirect/phpcodesniffer-composer-installer</code>');
      }
    }

    return $warnings;
  }

  /**
   * Checks if the installed drupal/coder is compatible with current PHPCS.
   *
   * @param string $coder_path
   *   Path to coder_sniffer directory.
   *
   * @return array
   *   Array with 'compatible' boolean and version info.
   */
  protected function checkCoderCompatibility(string $coder_path): array {
    $result = [
      'compatible' => TRUE,
      'coder_version' => NULL,
      'phpcs_version' => NULL,
    ];

    // Get PHPCS version.
    $phpcs_path = $this->findBinary('phpcs');
    if ($phpcs_path) {
      try {
        $process = new Process([$phpcs_path, '--version']);
        $process->setTimeout(10);
        $process->run();
        $output = $process->getOutput();
        if (preg_match('/version\s+([0-9.]+)/i', $output, $matches)) {
          $result['phpcs_version'] = $matches[1];
        }
      }
      catch (\Exception $e) {
        // Ignore errors.
      }
    }

    // Try to detect coder version from composer.json or file dates.
    $coder_base = dirname($coder_path);
    $composer_file = $coder_base . '/composer.json';
    if (file_exists($composer_file)) {
      $composer_data = json_decode(file_get_contents($composer_file), TRUE);
      if (!empty($composer_data['version'])) {
        $result['coder_version'] = $composer_data['version'];
      }
    }

    // Check for known incompatibility: old coder uses class names like
    // Generic_Sniffs_Files_LineLengthSniff (PHPCS 2.x) vs namespaced
    // PHP_CodeSniffer\Standards\Generic\Sniffs\Files\LineLengthSniff (PHPCS 3.x+).
    $test_file = $coder_path . '/Drupal/Sniffs/Files/LineLengthSniff.php';
    if (file_exists($test_file)) {
      $content = file_get_contents($test_file);
      // Old versions extend "Generic_Sniffs_..." (PHPCS 2.x style).
      if (strpos($content, 'extends Generic_Sniffs_') !== FALSE) {
        $result['compatible'] = FALSE;
        if (empty($result['coder_version'])) {
          $result['coder_version'] = '8.2.x or older (PHPCS 2.x compatible only)';
        }
      }
    }

    return $result;
  }

  /**
   * Finds the path to Drupal coder standards if they exist on disk.
   *
   * @return string|null
   *   Path to coder_sniffer directory, or NULL if not found.
   */
  protected function findCoderStandardsPath(): ?string {
    $possible_paths = [
      dirname(DRUPAL_ROOT) . '/vendor/drupal/coder/coder_sniffer',
      DRUPAL_ROOT . '/vendor/drupal/coder/coder_sniffer',
      DRUPAL_ROOT . '/../vendor/drupal/coder/coder_sniffer',
    ];

    foreach ($possible_paths as $path) {
      // Check if Drupal standard exists in this path.
      if (is_dir($path . '/Drupal') && is_dir($path . '/DrupalPractice')) {
        return $path;
      }
    }

    return NULL;
  }

  /**
   * Gets list of installed PHPCS standards.
   *
   * @param string $phpcs_path
   *   Path to PHPCS binary.
   *
   * @return array
   *   Array of installed standard names.
   */
  protected function getInstalledStandards(string $phpcs_path): array {
    try {
      $process = new Process([$phpcs_path, '-i']);
      $process->setTimeout(30);
      $process->run();

      $output = $process->getOutput();
      // Parse: "The installed coding standards are Drupal, DrupalPractice, ..."
      if (preg_match('/installed coding standards are (.+)$/i', $output, $matches)) {
        return array_map('trim', explode(',', $matches[1]));
      }
    }
    catch (\Exception $e) {
      // Silently fail, standards check is informational.
    }

    return [];
  }

  /**
   * Gets detailed installation instructions.
   *
   * @return array
   *   Array with installation steps.
   */
  protected function getInstallationInstructions(): array {
    return [
      'title' => (string) $this->t('How to install PHP CodeSniffer for Drupal'),
      'steps' => [
        [
          'command' => 'composer require --dev drupal/coder',
          'description' => (string) $this->t('Install drupal/coder package. This includes PHPCS, Drupal and DrupalPractice standards, and all required dependencies.'),
        ],
        [
          'command' => './vendor/bin/phpcs -i',
          'description' => (string) $this->t('Verify installation. You should see "Drupal" and "DrupalPractice" in the list of installed standards.'),
        ],
      ],
      'notes' => [
        (string) $this->t('The dealerdirect/phpcodesniffer-composer-installer package (included as dependency) automatically registers the standards paths.'),
        (string) $this->t('For Drupal 11 with PHP 8.3+, drupal/coder 9.x (currently alpha) will be required when stable.'),
        (string) $this->t('Consider also installing mglaman/phpstan-drupal for complementary static analysis.'),
      ],
      'documentation' => 'https://www.drupal.org/docs/develop/development-tools/php-codesniffer',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $config): array {
    $form = [];

    // File extensions.
    $form['extensions'] = [
      '#type' => 'textfield',
      '#title' => $this->t('File extensions to analyze'),
      '#description' => $this->t('Comma-separated list of file extensions. <strong>Recommended:</strong> <code>@default</code><br>Includes all Drupal-specific extensions: PHP files, module files, install files, theme files, info.yml, and documentation.', [
        '@default' => self::DEFAULT_EXTENSIONS,
      ]),
      '#default_value' => $config['extensions'] ?? self::DEFAULT_EXTENSIONS,
      '#required' => TRUE,
    ];

    // Severity level.
    $form['min_severity'] = [
      '#type' => 'select',
      '#title' => $this->t('Minimum severity level'),
      '#description' => $this->t('Only report issues with this severity or higher. Lower values show more issues.<br><strong>Recommended:</strong> 5 for balanced analysis. Use 1 for strict (all issues), 8+ for critical only.'),
      '#options' => [
        1 => $this->t('1 - All issues (strictest)'),
        2 => $this->t('2 - Almost all'),
        3 => $this->t('3 - Minor issues and above'),
        4 => $this->t('4 - Moderate issues and above'),
        5 => $this->t('5 - Balanced (recommended)'),
        6 => $this->t('6 - Notable issues and above'),
        7 => $this->t('7 - Important issues only'),
        8 => $this->t('8 - Serious issues only'),
        9 => $this->t('9 - Critical issues only'),
        10 => $this->t('10 - Most critical only'),
      ],
      '#default_value' => $config['min_severity'] ?? self::DEFAULT_MIN_SEVERITY,
    ];

    // Ignore warnings.
    $form['ignore_warnings'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Ignore warnings (only show errors)'),
      '#description' => $this->t('When enabled, only coding standards errors will be reported. Warnings will be ignored.<br><strong>Recommended:</strong> Keep unchecked to see all issues. Enable only to focus on critical errors first in legacy codebases.'),
      '#default_value' => $config['ignore_warnings'] ?? FALSE,
    ];

    // Excluded sniffs - use numeric indices to avoid dots in config keys.
    $sniff_options = [];
    $sniff_list = array_keys(self::EXCLUDABLE_SNIFFS);
    foreach ($sniff_list as $index => $sniff) {
      $description = self::EXCLUDABLE_SNIFFS[$sniff];
      $sniff_options[$index] = $sniff . ' - ' . $description;
    }
    // Convert stored sniff names back to indices for default values.
    $stored_sniffs = $config['excluded_sniffs'] ?? [];
    $default_indices = [];
    foreach ($stored_sniffs as $sniff) {
      $index = array_search($sniff, $sniff_list, TRUE);
      if ($index !== FALSE) {
        $default_indices[$index] = $index;
      }
    }
    $form['excluded_sniffs'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Exclude specific rules'),
      '#description' => $this->t('Select rules to exclude from analysis. Useful for legacy code or specific project requirements.<br><strong>Recommended:</strong> Keep all enabled for new projects. For legacy migrations, temporarily exclude commenting rules.'),
      '#options' => $sniff_options,
      '#default_value' => $default_indices,
    ];

    // Performance settings fieldset.
    $form['performance'] = [
      '#type' => 'details',
      '#title' => $this->t('Performance settings'),
      '#open' => FALSE,
    ];

    $form['performance']['parallel'] = [
      '#type' => 'number',
      '#title' => $this->t('Parallel processes'),
      '#description' => $this->t('Number of parallel processes for analysis. <strong>Recommended:</strong> 4-10 depending on server CPU cores. Higher values speed up analysis on large codebases.'),
      '#default_value' => $config['parallel'] ?? self::DEFAULT_PARALLEL,
      '#min' => 1,
      '#max' => 20,
    ];

    $form['performance']['timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Execution timeout (seconds)'),
      '#description' => $this->t('Maximum time for PHPCS to complete. <strong>Recommended:</strong> 300 seconds (5 minutes). Increase for very large codebases.<br><em>Note: Also limited by PHP max_execution_time.</em>'),
      '#default_value' => $config['timeout'] ?? self::DEFAULT_TIMEOUT,
      '#min' => 60,
      '#max' => 3600,
    ];

    $form['performance']['use_cache'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable caching'),
      '#description' => $this->t('Cache analysis results between runs. Significantly speeds up repeated analysis when only few files change.'),
      '#default_value' => $config['use_cache'] ?? TRUE,
    ];

    // Display settings fieldset.
    $form['display'] = [
      '#type' => 'details',
      '#title' => $this->t('Display settings'),
      '#open' => FALSE,
    ];

    $form['display']['max_violations_display'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum violations to display'),
      '#description' => $this->t('Limits violations shown in results. <strong>Recommended:</strong> 2000. Set to 0 for unlimited (may slow down the interface with many violations).'),
      '#default_value' => $config['max_violations_display'] ?? self::DEFAULT_MAX_VIOLATIONS,
      '#min' => 0,
      '#max' => 10000,
    ];

    $form['display']['report_width'] = [
      '#type' => 'number',
      '#title' => $this->t('Report width (characters)'),
      '#description' => $this->t('Maximum width for code snippets and messages. <strong>Recommended:</strong> 120 characters.'),
      '#default_value' => $config['report_width'] ?? self::DEFAULT_REPORT_WIDTH,
      '#min' => 80,
      '#max' => 200,
    ];

    $form['display']['show_sniff_codes'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show sniff codes in results'),
      '#description' => $this->t('Display the sniff identifier (e.g., Drupal.Commenting.DocComment) with each violation. Useful for understanding which rule was violated and for exclusion configuration.'),
      '#default_value' => $config['show_sniff_codes'] ?? TRUE,
    ];

    $form['display']['group_by_file'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Group violations by file'),
      '#description' => $this->t('When enabled, violations are grouped by file. When disabled, they are shown in a flat list sorted by severity.'),
      '#default_value' => $config['group_by_file'] ?? TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function processConfigurationValue(string $key, mixed $value): mixed {
    // Convert numeric indices back to sniff names for excluded_sniffs.
    if ($key === 'excluded_sniffs' && is_array($value)) {
      $sniff_list = array_keys(self::EXCLUDABLE_SNIFFS);
      $sniff_names = [];
      foreach ($value as $index => $checked) {
        // Checkboxes return 0 for unchecked, index for checked.
        if ($checked !== 0 && isset($sniff_list[$index])) {
          $sniff_names[] = $sniff_list[$index];
        }
      }
      return $sniff_names;
    }

    return $value;
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    $requirements = $this->checkRequirements();

    // Check for critical requirements (PHPCS not installed).
    $phpcs_path = $this->findBinary('phpcs');
    if (!$phpcs_path) {
      return $this->createNotInstalledResult($requirements);
    }

    // Check for Drupal standards - either registered or available on disk.
    $standards = $this->getInstalledStandards($phpcs_path);
    $standards_registered = in_array('Drupal', $standards, TRUE) && in_array('DrupalPractice', $standards, TRUE);
    $coder_path = $this->findCoderStandardsPath();

    if (!$standards_registered && !$coder_path) {
      // Standards neither registered nor available on disk.
      return $this->createStandardsNotInstalledResult($standards, $requirements);
    }

    // If standards exist but not registered, check compatibility before using.
    if (!$standards_registered && $coder_path) {
      $compatibility = $this->checkCoderCompatibility($coder_path);
      if (!$compatibility['compatible']) {
        return $this->createIncompatibleCoderResult($compatibility, $requirements);
      }
    }

    // Load configuration.
    $config = $this->configFactory->get('audit_phpcs.settings');
    $audit_config = $this->configFactory->get('audit.settings');

    // Get configuration values with defaults.
    $extensions = $config->get('extensions') ?? self::DEFAULT_EXTENSIONS;
    $min_severity = (int) ($config->get('min_severity') ?? self::DEFAULT_MIN_SEVERITY);
    $ignore_warnings = (bool) $config->get('ignore_warnings');
    $timeout = (int) ($config->get('timeout') ?? self::DEFAULT_TIMEOUT);
    $max_violations = (int) ($config->get('max_violations_display') ?? self::DEFAULT_MAX_VIOLATIONS);
    $parallel = (int) ($config->get('parallel') ?? self::DEFAULT_PARALLEL);
    $use_cache = (bool) ($config->get('use_cache') ?? TRUE);
    $report_width = (int) ($config->get('report_width') ?? self::DEFAULT_REPORT_WIDTH);
    $show_sniff_codes = (bool) ($config->get('show_sniff_codes') ?? TRUE);
    $excluded_sniffs = array_filter($config->get('excluded_sniffs') ?? []);

    // Get paths from global audit configuration.
    $paths_to_analyze = $this->getScanDirectories($audit_config);
    $exclude_patterns = $this->getExcludePatterns($audit_config);

    if (empty($paths_to_analyze)) {
      return $this->createNoCodeResult();
    }

    $results = [];
    $total_errors = 0;
    $total_warnings = 0;
    $total_fixable = 0;
    $files_with_violations = [];
    $violations_by_sniff = [];

    foreach ($paths_to_analyze as $path) {
      $command = $this->buildCommand(
        $phpcs_path,
        $path,
        $extensions,
        $min_severity,
        $ignore_warnings,
        $exclude_patterns,
        $parallel,
        $use_cache,
        $report_width,
        $show_sniff_codes,
        $excluded_sniffs,
        $standards_registered ? NULL : $coder_path
      );

      try {
        $process = new Process($command);
        $process->setTimeout($timeout);
        $process->run();

        // PHPCS returns exit code 1 when violations found, 2 for errors.
        $output_json = $process->getOutput();
        if (empty($output_json)) {
          $output_json = $process->getErrorOutput();
        }

        $this->processPhpcsOutput(
          $output_json,
          $results,
          $total_errors,
          $total_warnings,
          $total_fixable,
          $files_with_violations,
          $violations_by_sniff,
          $ignore_warnings,
          $show_sniff_codes
        );
      }
      catch (\Exception $e) {
        // Try to extract JSON from exception message.
        $error_output = $e->getMessage();
        if (str_contains($error_output, '{')) {
          $json_start = strpos($error_output, '{');
          $json_output = substr($error_output, $json_start);
          $this->processPhpcsOutput(
            $json_output,
            $results,
            $total_errors,
            $total_warnings,
            $total_fixable,
            $files_with_violations,
            $violations_by_sniff,
            $ignore_warnings,
            $show_sniff_codes
          );
        }
      }
    }

    // Sort violations by sniff for analysis.
    arsort($violations_by_sniff);

    if (empty($results)) {
      $results[] = $this->createResultItem(
        'notice',
        'PHPCS_NO_VIOLATIONS',
        (string) $this->t('Excellent! No coding standards violations found.'),
        ['paths_analyzed' => $paths_to_analyze]
      );
    }

    // Build unified violations file (combining errors and warnings).
    $violations_file = $this->createResult(
      $results,
      $total_errors,
      $total_warnings,
      ($total_errors === 0 && $total_warnings === 0) ? 1 : 0
    );

    $stats = [
      'available' => TRUE,
      'total_errors' => $total_errors,
      'total_warnings' => $total_warnings,
      'total_fixable' => $total_fixable,
      'total_violations' => $total_errors + $total_warnings,
      'files_analyzed' => $this->countFilesInPaths($paths_to_analyze, $extensions),
      'files_with_violations' => count($files_with_violations),
      'paths_analyzed' => count($paths_to_analyze),
      'paths_list' => $paths_to_analyze,
      'violations_by_file' => $files_with_violations,
      'violations_by_sniff' => array_slice($violations_by_sniff, 0, 20, TRUE),
      'config' => [
        'extensions' => $extensions,
        'min_severity' => $min_severity,
        'ignore_warnings' => $ignore_warnings,
        'excluded_sniffs' => $excluded_sniffs,
        'parallel' => $parallel,
        'use_cache' => $use_cache,
        'max_violations_display' => $max_violations,
        'show_sniff_codes' => $show_sniff_codes,
      ],
      'standards' => ['Drupal', 'DrupalPractice'],
      'phpcbf_available' => $this->findBinary('phpcbf') !== NULL,
      // Add counts for each category.
      'errors_count' => $total_errors,
      'warnings_count' => $total_warnings,
    ];

    $scores = $this->calculateScores($stats);

    // Return structured output with _files for each section.
    return [
      '_files' => [
        // Scoring section: All violations combined.
        'violations' => $violations_file,
        // Informational section: Summary.
        'summary' => [
          'summary' => ['errors' => 0, 'warnings' => 0, 'notices' => 0],
          'results' => [],
          'data' => [
            'files_analyzed' => $stats['files_analyzed'],
            'files_with_violations' => count($files_with_violations),
            'paths_list' => $paths_to_analyze,
            'violations_by_sniff' => $stats['violations_by_sniff'],
            'config' => $stats['config'],
            'total_errors' => $total_errors,
            'total_warnings' => $total_warnings,
            'total_fixable' => $total_fixable,
            'phpcbf_available' => $stats['phpcbf_available'],
          ],
        ],
      ],
      'score' => $scores,
      'stats' => $stats,
    ];
  }

  /**
   * Builds the errors file (ERROR severity violations).
   *
   * @param array $all_results
   *   All result items from analysis.
   *
   * @return array
   *   Filtered results for errors.
   */
  protected function buildErrorsFile(array $all_results): array {
    $results = [];
    $errors = 0;

    foreach ($all_results as $item) {
      $type = $item['details']['type'] ?? 'ERROR';
      if ($type === 'ERROR') {
        $results[] = $item;
        $errors++;
      }
    }

    return $this->createResult($results, $errors, 0, 0);
  }

  /**
   * Builds the warnings file (WARNING severity violations).
   *
   * @param array $all_results
   *   All result items from analysis.
   *
   * @return array
   *   Filtered results for warnings.
   */
  protected function buildWarningsFile(array $all_results): array {
    $results = [];
    $warnings = 0;

    foreach ($all_results as $item) {
      $type = $item['details']['type'] ?? 'ERROR';
      if ($type === 'WARNING') {
        $results[] = $item;
        $warnings++;
      }
    }

    return $this->createResult($results, 0, $warnings, 0);
  }

  /**
   * Creates result when PHPCS is not installed.
   *
   * @param array $requirements
   *   Requirements warnings.
   *
   * @return array
   *   Result array.
   */
  protected function createNotInstalledResult(array $requirements): array {
    $instructions = $this->getInstallationInstructions();

    $results = [];
    $results[] = $this->createResultItem(
      'error',
      'PHPCS_NOT_INSTALLED',
      (string) $this->t('PHP CodeSniffer is not installed'),
      [
        'requirements' => $requirements,
        'installation' => $instructions,
      ]
    );

    return [
      '_files' => [
        'violations' => $this->createResult($results, 1, 0, 0),
        'summary' => [
          'summary' => ['errors' => 0, 'warnings' => 0, 'notices' => 0],
          'results' => [],
          'data' => [
            'installation_instructions' => $instructions,
            'requirements' => $requirements,
          ],
        ],
      ],
      'score' => $this->createEmptyScores((string) $this->t('PHPCS not installed')),
      'stats' => [
        'available' => FALSE,
        'installation_instructions' => $instructions,
      ],
    ];
  }

  /**
   * Creates result when Drupal standards are not installed.
   *
   * @param array $installed_standards
   *   List of currently installed standards.
   * @param array $requirements
   *   Requirements warnings.
   *
   * @return array
   *   Result array.
   */
  protected function createStandardsNotInstalledResult(array $installed_standards, array $requirements): array {
    $instructions = $this->getInstallationInstructions();

    $results = [];
    $results[] = $this->createResultItem(
      'error',
      'DRUPAL_STANDARDS_NOT_INSTALLED',
      (string) $this->t('Drupal coding standards are not properly installed'),
      [
        'requirements' => $requirements,
        'installed_standards' => $installed_standards,
        'required_standards' => ['Drupal', 'DrupalPractice'],
        'installation' => $instructions,
      ]
    );

    return [
      '_files' => [
        'violations' => $this->createResult($results, 1, 0, 0),
        'summary' => [
          'summary' => ['errors' => 0, 'warnings' => 0, 'notices' => 0],
          'results' => [],
          'data' => [
            'installed_standards' => $installed_standards,
            'installation_instructions' => $instructions,
            'requirements' => $requirements,
          ],
        ],
      ],
      'score' => $this->createEmptyScores((string) $this->t('Drupal standards not installed')),
      'stats' => [
        'available' => FALSE,
        'installed_standards' => $installed_standards,
        'installation_instructions' => $instructions,
      ],
    ];
  }

  /**
   * Creates result when drupal/coder is incompatible with PHPCS version.
   *
   * @param array $compatibility
   *   Compatibility check results.
   * @param array $requirements
   *   Requirements warnings.
   *
   * @return array
   *   Result array.
   */
  protected function createIncompatibleCoderResult(array $compatibility, array $requirements): array {
    $results = [];
    $results[] = $this->createResultItem(
      'error',
      'INCOMPATIBLE_CODER_VERSION',
      (string) $this->t('Incompatible drupal/coder version'),
      [
        'requirements' => $requirements,
        'coder_version' => $compatibility['coder_version'],
        'phpcs_version' => $compatibility['phpcs_version'],
        'solution' => (string) $this->t('Update drupal/coder to a version compatible with PHP_CodeSniffer @phpcs. Run: composer require --dev drupal/coder:^8.3.0 dealerdirect/phpcodesniffer-composer-installer', [
          '@phpcs' => $compatibility['phpcs_version'],
        ]),
      ]
    );

    return [
      '_files' => [
        'violations' => $this->createResult($results, 1, 0, 0),
        'summary' => [
          'summary' => ['errors' => 0, 'warnings' => 0, 'notices' => 0],
          'results' => [],
          'data' => [
            'coder_version' => $compatibility['coder_version'],
            'phpcs_version' => $compatibility['phpcs_version'],
            'upgrade_command' => 'composer require --dev drupal/coder:^8.3.0 dealerdirect/phpcodesniffer-composer-installer',
          ],
        ],
      ],
      'score' => $this->createEmptyScores((string) $this->t('Incompatible drupal/coder version')),
      'stats' => [
        'available' => FALSE,
        'incompatible_coder' => TRUE,
        'coder_version' => $compatibility['coder_version'],
        'phpcs_version' => $compatibility['phpcs_version'],
        'upgrade_command' => 'composer require --dev drupal/coder:^8.3.0 dealerdirect/phpcodesniffer-composer-installer',
      ],
    ];
  }

  /**
   * Creates result when no code is found to analyze.
   *
   * @return array
   *   Result array.
   */
  protected function createNoCodeResult(): array {
    $results = [];
    $results[] = $this->createResultItem(
      'notice',
      'NO_CUSTOM_CODE',
      (string) $this->t('No custom code found to analyze'),
      [
        'suggestion' => (string) $this->t('Check the "Directories to scan" setting in the main Audit configuration.'),
      ]
    );

    return [
      '_files' => [
        'violations' => $this->createResult($results, 0, 0, 1),
        'summary' => [
          'summary' => ['errors' => 0, 'warnings' => 0, 'notices' => 0],
          'results' => [],
          'data' => [],
        ],
      ],
      'score' => $this->createPerfectScores(),
      'stats' => ['available' => TRUE, 'no_code' => TRUE],
    ];
  }

  /**
   * Counts files in paths matching extensions.
   *
   * @param array $paths
   *   Paths to count files in.
   * @param string $extensions
   *   Comma-separated extensions.
   *
   * @return int
   *   Approximate file count.
   */
  protected function countFilesInPaths(array $paths, string $extensions): int {
    $count = 0;
    $ext_array = explode(',', $extensions);

    foreach ($paths as $path) {
      if (!is_dir($path)) {
        continue;
      }

      $iterator = new \RecursiveIteratorIterator(
        new \RecursiveDirectoryIterator($path, \RecursiveDirectoryIterator::SKIP_DOTS)
      );

      foreach ($iterator as $file) {
        if ($file->isFile()) {
          $ext = $file->getExtension();
          // Handle compound extensions like info.yml.
          $filename = $file->getFilename();
          foreach ($ext_array as $check_ext) {
            if (str_ends_with($filename, '.' . $check_ext)) {
              $count++;
              break;
            }
          }
        }
      }
    }

    return $count;
  }

  /**
   * Gets scan directories from global audit configuration.
   *
   * @param \Drupal\Core\Config\ImmutableConfig $audit_config
   *   The audit configuration.
   *
   * @return array
   *   Array of existing directory paths.
   */
  protected function getScanDirectories($audit_config): array {
    $scan_dirs_raw = $audit_config->get('scan_directories') ?? "web/modules/custom\nweb/themes/custom";
    $scan_dirs = array_filter(array_map('trim', explode("\n", $scan_dirs_raw)));

    $paths = [];
    foreach ($scan_dirs as $dir) {
      // Try relative to DRUPAL_ROOT first.
      $full_path = DRUPAL_ROOT . '/' . ltrim($dir, '/');
      if (is_dir($full_path)) {
        $paths[] = $full_path;
        continue;
      }

      // Try as absolute path.
      if (is_dir($dir)) {
        $paths[] = $dir;
        continue;
      }

      // Try relative to project root (one level up from DRUPAL_ROOT).
      $project_path = dirname(DRUPAL_ROOT) . '/' . ltrim($dir, '/');
      if (is_dir($project_path)) {
        $paths[] = $project_path;
      }
    }

    return $paths;
  }

  /**
   * Gets exclude patterns from global audit configuration.
   *
   * @param \Drupal\Core\Config\ImmutableConfig $audit_config
   *   The audit configuration.
   *
   * @return array
   *   Array of exclude patterns.
   */
  protected function getExcludePatterns($audit_config): array {
    $default_patterns = "*Test.php\n*TestBase.php\ntests/\nnode_modules/\nvendor/";
    $exclude_raw = $audit_config->get('exclude_patterns') ?? $default_patterns;
    return array_filter(array_map('trim', explode("\n", $exclude_raw)));
  }

  /**
   * Builds the PHPCS command with all options.
   *
   * @param string $phpcs_path
   *   Path to PHPCS binary.
   * @param string $path
   *   Path to analyze.
   * @param string $extensions
   *   File extensions.
   * @param int $min_severity
   *   Minimum severity level.
   * @param bool $ignore_warnings
   *   Whether to ignore warnings.
   * @param array $exclude_patterns
   *   Patterns to exclude.
   * @param int $parallel
   *   Number of parallel processes.
   * @param bool $use_cache
   *   Whether to use caching.
   * @param int $report_width
   *   Report width in characters.
   * @param bool $show_sniff_codes
   *   Whether to include sniff codes.
   * @param array $excluded_sniffs
   *   Sniffs to exclude.
   * @param string|null $coder_path
   *   Path to coder_sniffer directory if standards not registered, or NULL.
   *
   * @return array
   *   Command array.
   */
  protected function buildCommand(
    string $phpcs_path,
    string $path,
    string $extensions,
    int $min_severity,
    bool $ignore_warnings,
    array $exclude_patterns,
    int $parallel,
    bool $use_cache,
    int $report_width,
    bool $show_sniff_codes,
    array $excluded_sniffs,
    ?string $coder_path = NULL
  ): array {
    $command = [
      $phpcs_path,
    ];

    // If standards are not registered, specify the path to find them.
    if ($coder_path !== NULL) {
      $command[] = '--standard=' . $coder_path . '/Drupal,' . $coder_path . '/DrupalPractice';
    }
    else {
      $command[] = '--standard=Drupal,DrupalPractice';
    }

    $command[] = '--report=json';
    $command[] = '--extensions=' . $extensions;
    $command[] = '--report-width=' . $report_width;

    // Parallel processing.
    if ($parallel > 1) {
      $command[] = '--parallel=' . $parallel;
    }

    // Caching.
    if ($use_cache) {
      $cache_file = sys_get_temp_dir() . '/phpcs-audit-cache';
      $command[] = '--cache=' . $cache_file;
    }

    // Severity filter.
    if ($min_severity > 1) {
      $command[] = '--severity=' . $min_severity;
    }

    // Ignore warnings (errors only).
    if ($ignore_warnings) {
      $command[] = '-n';
    }

    // Include sniff codes in output.
    if ($show_sniff_codes) {
      $command[] = '-s';
    }

    // Exclude patterns.
    if (!empty($exclude_patterns)) {
      $command[] = '--ignore=' . implode(',', $exclude_patterns);
    }

    // Exclude specific sniffs.
    if (!empty($excluded_sniffs)) {
      $command[] = '--exclude=' . implode(',', $excluded_sniffs);
    }

    $command[] = $path;

    return $command;
  }

  /**
   * Processes PHPCS JSON output.
   *
   * @param string $json_output
   *   The JSON output from PHPCS.
   * @param array &$results
   *   Results array to append to.
   * @param int &$total_errors
   *   Total errors counter.
   * @param int &$total_warnings
   *   Total warnings counter.
   * @param int &$total_fixable
   *   Total fixable issues counter.
   * @param array &$files_with_violations
   *   Files with violations array.
   * @param array &$violations_by_sniff
   *   Violations grouped by sniff.
   * @param bool $ignore_warnings
   *   Whether warnings are being ignored.
   * @param bool $show_sniff_codes
   *   Whether to include sniff codes.
   */
  protected function processPhpcsOutput(
    string $json_output,
    array &$results,
    int &$total_errors,
    int &$total_warnings,
    int &$total_fixable,
    array &$files_with_violations,
    array &$violations_by_sniff,
    bool $ignore_warnings,
    bool $show_sniff_codes
  ): void {
    $phpcs_results = json_decode($json_output, TRUE);

    if (!$phpcs_results || !isset($phpcs_results['files'])) {
      return;
    }

    // Get totals from PHPCS summary if available.
    if (isset($phpcs_results['totals'])) {
      $total_fixable += $phpcs_results['totals']['fixable'] ?? 0;
    }

    foreach ($phpcs_results['files'] as $file => $file_data) {
      $relative_file = str_replace(DRUPAL_ROOT . '/', '', $file);
      // Also try removing parent directory path.
      $relative_file = str_replace(dirname(DRUPAL_ROOT) . '/', '', $relative_file);

      $file_errors = $file_data['errors'] ?? 0;
      $file_warnings = $ignore_warnings ? 0 : ($file_data['warnings'] ?? 0);

      if ($file_errors > 0 || $file_warnings > 0) {
        $files_with_violations[$relative_file] = [
          'errors' => $file_errors,
          'warnings' => $file_warnings,
          'fixable' => $file_data['fixable'] ?? 0,
        ];
      }

      $total_errors += $file_errors;
      if (!$ignore_warnings) {
        $total_warnings += $file_data['warnings'] ?? 0;
      }

      foreach ($file_data['messages'] ?? [] as $message) {
        $is_warning = ($message['type'] ?? 'ERROR') === 'WARNING';

        if ($is_warning && $ignore_warnings) {
          continue;
        }

        $severity = $is_warning ? 'warning' : 'error';
        $sniff_code = $message['source'] ?? 'Unknown';

        // Track by sniff.
        if (!isset($violations_by_sniff[$sniff_code])) {
          $violations_by_sniff[$sniff_code] = 0;
        }
        $violations_by_sniff[$sniff_code]++;

        // Extract code context for this violation.
        $line_number = $message['line'] ?? 0;
        $code_context = $this->extractCodeContext($file, $line_number);

        $results[] = $this->createResultItem(
          $severity,
          'PHPCS_VIOLATION',
          (string) $this->t('@type: @message', [
            '@type' => $message['type'] ?? 'VIOLATION',
            '@message' => $this->truncateMessage($message['message'] ?? '', 100),
          ]),
          [
            'file' => $relative_file,
            'file_absolute' => $file,
            'line' => $line_number,
            'column' => $message['column'] ?? 0,
            'type' => $message['type'] ?? 'ERROR',
            'message' => $message['message'] ?? '',
            'source' => $sniff_code,
            'severity' => $message['severity'] ?? 5,
            'fixable' => $message['fixable'] ?? FALSE,
            'code_context' => $code_context,
          ]
        );
      }
    }
  }

  /**
   * Extracts code context around a specific line.
   *
   * @param string $file_path
   *   Full path to the file.
   * @param int $line_number
   *   The line number (1-based).
   * @param int $context_lines
   *   Number of lines before and after to include.
   *
   * @return array|null
   *   Array with 'lines' and 'highlight_line', or NULL if file unreadable.
   */
  protected function extractCodeContext(string $file_path, int $line_number, int $context_lines = 2): ?array {
    if (!file_exists($file_path) || !is_readable($file_path) || $line_number < 1) {
      return NULL;
    }

    try {
      $file = new \SplFileObject($file_path);
      $file->setFlags(\SplFileObject::DROP_NEW_LINE);

      $start_line = max(1, $line_number - $context_lines);
      $end_line = $line_number + $context_lines;

      $lines = [];
      $current_line = 1;

      foreach ($file as $line_content) {
        $current_line = $file->key() + 1;

        if ($current_line >= $start_line && $current_line <= $end_line) {
          $lines[] = [
            'number' => $current_line,
            'content' => $line_content,
            'is_highlight' => ($current_line === $line_number),
          ];
        }

        if ($current_line > $end_line) {
          break;
        }
      }

      return [
        'lines' => $lines,
        'highlight_line' => $line_number,
        'start_line' => $start_line,
        'end_line' => min($end_line, $current_line),
      ];
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

  /**
   * Truncates a message for summary display.
   *
   * @param string $message
   *   The message to truncate.
   * @param int $length
   *   Maximum length.
   *
   * @return string
   *   Truncated message.
   */
  protected function truncateMessage(string $message, int $length): string {
    if (strlen($message) <= $length) {
      return $message;
    }
    return substr($message, 0, $length - 3) . '...';
  }

  /**
   * Gets a generic/grouped label for PHPCS violations based on sniff code.
   *
   * @param string $sniff
   *   The sniff code (e.g., "Drupal.Commenting.DocComment").
   * @param string $message
   *   The violation message.
   *
   * @return string
   *   Generic label for grouping in facets.
   */
  protected function getViolationLabel(string $sniff, string $message): string {
    $sniff_lower = strtolower($sniff);

    // Commenting/Documentation issues.
    if (str_contains($sniff_lower, 'commenting.doccomment')) {
      return (string) $this->t('DocBlock format issue');
    }
    if (str_contains($sniff_lower, 'commenting.functioncomment')) {
      return (string) $this->t('Function comment issue');
    }
    if (str_contains($sniff_lower, 'commenting.filecomment')) {
      return (string) $this->t('File comment issue');
    }
    if (str_contains($sniff_lower, 'commenting')) {
      return (string) $this->t('Comment/documentation issue');
    }

    // Naming conventions.
    if (str_contains($sniff_lower, 'namingconventions')) {
      return (string) $this->t('Naming convention violation');
    }

    // Formatting/whitespace.
    if (str_contains($sniff_lower, 'linelength')) {
      return (string) $this->t('Line too long');
    }
    if (str_contains($sniff_lower, 'whitespace') || str_contains($sniff_lower, 'formatting')) {
      return (string) $this->t('Formatting/whitespace issue');
    }

    // Array syntax.
    if (str_contains($sniff_lower, 'array')) {
      return (string) $this->t('Array syntax issue');
    }

    // Control structures.
    if (str_contains($sniff_lower, 'controlstructures')) {
      return (string) $this->t('Control structure issue');
    }

    // Functions.
    if (str_contains($sniff_lower, 'functions')) {
      return (string) $this->t('Function declaration issue');
    }

    // Classes.
    if (str_contains($sniff_lower, 'classes')) {
      return (string) $this->t('Class declaration issue');
    }

    // Files.
    if (str_contains($sniff_lower, 'files')) {
      return (string) $this->t('File structure issue');
    }

    // DrupalPractice specific.
    if (str_contains($sniff_lower, 'drupalpractice.objects.globaldrupal')) {
      return (string) $this->t('Global Drupal usage');
    }
    if (str_contains($sniff_lower, 'drupalpractice.codeanalysis')) {
      return (string) $this->t('Code analysis issue');
    }
    if (str_contains($sniff_lower, 'drupalpractice')) {
      return (string) $this->t('Drupal best practice issue');
    }

    // Strings.
    if (str_contains($sniff_lower, 'strings')) {
      return (string) $this->t('String handling issue');
    }

    // Operators.
    if (str_contains($sniff_lower, 'operators')) {
      return (string) $this->t('Operator usage issue');
    }

    // Scope.
    if (str_contains($sniff_lower, 'scope')) {
      return (string) $this->t('Scope issue');
    }

    // Generic fallback based on standard.
    if (str_contains($sniff_lower, 'drupal.')) {
      return (string) $this->t('Drupal coding standard violation');
    }

    return (string) $this->t('Coding standard violation');
  }

  /**
   * Gets fix suggestion for PHPCS violations.
   *
   * @param string $sniff
   *   The sniff code.
   * @param string $message
   *   The violation message.
   * @param bool $fixable
   *   Whether the issue is auto-fixable.
   *
   * @return string
   *   HTML fix suggestion.
   */
  protected function getViolationFix(string $sniff, string $message, bool $fixable): string {
    $sniff_lower = strtolower($sniff);

    // Auto-fixable issues.
    if ($fixable) {
      $fix = (string) $this->t('This issue can be automatically fixed. Run: <code>./vendor/bin/phpcbf path/to/file.php</code>');
      return $fix;
    }

    // Commenting issues.
    if (str_contains($sniff_lower, 'commenting.doccomment')) {
      return (string) $this->t('Review and fix the DocBlock format. Ensure proper tags (@param, @return, @throws) and formatting according to Drupal standards.');
    }
    if (str_contains($sniff_lower, 'commenting.functioncomment')) {
      return (string) $this->t('Add or fix the function/method documentation. Include @param for each parameter and @return for the return value.');
    }
    if (str_contains($sniff_lower, 'commenting.filecomment')) {
      return (string) $this->t('Add a file-level DocBlock at the top of the file with @file tag describing the file purpose.');
    }

    // Naming conventions.
    if (str_contains($sniff_lower, 'namingconventions')) {
      return (string) $this->t('Rename the element to follow Drupal naming conventions: lowerCamelCase for variables/functions, UpperCamelCase for classes.');
    }

    // Line length.
    if (str_contains($sniff_lower, 'linelength')) {
      return (string) $this->t('Break the line to be under 80 characters. Consider extracting complex expressions into variables.');
    }

    // Global Drupal usage.
    if (str_contains($sniff_lower, 'globaldrupal')) {
      return (string) $this->t('Use dependency injection instead of \\Drupal::service(). Inject the service through the constructor or create() method.');
    }

    // Array syntax.
    if (str_contains($sniff_lower, 'array')) {
      return (string) $this->t('Use short array syntax [] instead of array(). Ensure proper indentation and trailing commas.');
    }

    // Generic fix suggestion.
    return (string) $this->t('Review the Drupal coding standards documentation at <a href="https://www.drupal.org/docs/develop/standards" target="_blank">drupal.org/docs/develop/standards</a>.');
  }

  /**
   * Gets dynamic tags for PHPCS violations based on sniff code.
   *
   * @param string $sniff
   *   The sniff code.
   * @param string $severity
   *   The severity (error or warning).
   * @param bool $fixable
   *   Whether the issue is auto-fixable.
   *
   * @return array
   *   Array of tags.
   */
  protected function getViolationTags(string $sniff, string $severity, bool $fixable): array {
    $tags = ['phpcs', $severity];
    $sniff_lower = strtolower($sniff);

    if (str_contains($sniff_lower, 'commenting')) {
      $tags[] = 'documentation';
    }
    if (str_contains($sniff_lower, 'naming')) {
      $tags[] = 'naming';
    }
    if (str_contains($sniff_lower, 'whitespace') || str_contains($sniff_lower, 'formatting') || str_contains($sniff_lower, 'linelength')) {
      $tags[] = 'formatting';
    }
    if (str_contains($sniff_lower, 'drupalpractice')) {
      $tags[] = 'best-practice';
    }
    if (str_contains($sniff_lower, 'array')) {
      $tags[] = 'array';
    }
    if (str_contains($sniff_lower, 'controlstructures')) {
      $tags[] = 'control-flow';
    }
    if (str_contains($sniff_lower, 'functions')) {
      $tags[] = 'function';
    }
    if (str_contains($sniff_lower, 'classes')) {
      $tags[] = 'class';
    }
    if ($fixable) {
      $tags[] = 'auto-fixable';
    }

    return $tags;
  }

  /**
   * Creates empty scores when PHPCS is not available.
   *
   * @param string $reason
   *   Reason for empty scores.
   *
   * @return array
   *   Score data.
   */
  protected function createEmptyScores(string $reason = ''): array {
    $description = $reason ?: (string) $this->t('PHPCS not available');

    return [
      'overall' => 0,
      'factors' => [
        'coding_errors' => [
          'score' => 0,
          'weight' => self::SCORE_WEIGHTS['coding_errors'],
          'label' => (string) $this->t('Coding Errors'),
          'description' => $description,
        ],
        'coding_warnings' => [
          'score' => 0,
          'weight' => self::SCORE_WEIGHTS['coding_warnings'],
          'label' => (string) $this->t('Coding Warnings'),
          'description' => $description,
        ],
      ],
    ];
  }

  /**
   * Creates perfect scores when no code is found.
   *
   * @return array
   *   Score data.
   */
  protected function createPerfectScores(): array {
    return [
      'overall' => 100,
      'factors' => [
        'coding_errors' => [
          'score' => 100,
          'weight' => self::SCORE_WEIGHTS['coding_errors'],
          'label' => (string) $this->t('Coding Errors'),
          'description' => (string) $this->t('No custom code to analyze'),
        ],
        'coding_warnings' => [
          'score' => 100,
          'weight' => self::SCORE_WEIGHTS['coding_warnings'],
          'label' => (string) $this->t('Coding Warnings'),
          'description' => (string) $this->t('No custom code to analyze'),
        ],
      ],
    ];
  }

  /**
   * Calculates scores for all factors using volume-aware penalty scoring.
   *
   * Formula: score = 100 - (violation_percentage * PENALTY_MULTIPLIER)
   *
   * This considers the volume of files analyzed and applies a multiplier
   * to make issues more impactful than simple percentages.
   *
   * @param array $stats
   *   Statistics array.
   *
   * @return array
   *   Score data with overall and factors.
   */
  protected function calculateScores(array $stats): array {
    $factors = [];

    $total_errors = $stats['total_errors'];
    $total_warnings = $stats['total_warnings'];
    $total_fixable = $stats['total_fixable'] ?? 0;
    $errors_count = $stats['errors_count'] ?? $total_errors;
    $warnings_count = $stats['warnings_count'] ?? $total_warnings;
    $total_violations = $errors_count + $warnings_count;
    $files_analyzed = $stats['files_analyzed'] ?? 1;

    // Unified Coding Standards score: Based on total violations weighted by severity.
    // Errors count as 2x penalties compared to warnings.
    $weighted_violations = ($errors_count * 2) + $warnings_count;
    $violation_percentage = $files_analyzed > 0
      ? ($weighted_violations / $files_analyzed) * 100
      : 0;
    $score = (int) max(0, round(100 - ($violation_percentage * self::PENALTY_MULTIPLIER)));

    // Build description based on findings.
    if ($total_violations === 0) {
      $description = (string) $this->t('No violations in @files files — excellent code quality!', ['@files' => $files_analyzed]);
    }
    else {
      $description = (string) $this->t('@errors errors and @warnings warnings found in @files files', [
        '@errors' => $errors_count,
        '@warnings' => $warnings_count,
        '@files' => $files_analyzed,
      ]);
    }

    $factors['coding_standards'] = [
      'score' => $score,
      'weight' => self::SCORE_WEIGHTS['coding_standards'],
      'label' => (string) $this->t('Coding Standards'),
      'description' => $description,
      'badges' => [
        [
          'value' => (string) $errors_count . ' ' . $this->t('errors'),
          'type' => $errors_count === 0 ? 'success' : 'error',
          'label' => (string) $this->t('Critical violations'),
        ],
        [
          'value' => (string) $warnings_count . ' ' . $this->t('warnings'),
          'type' => $warnings_count === 0 ? 'success' : 'warning',
          'label' => (string) $this->t('Best practice issues'),
        ],
        [
          'value' => (string) $total_fixable . ' ' . $this->t('auto-fix'),
          'type' => $total_fixable > 0 ? 'info' : 'success',
          'label' => (string) $this->t('Auto-fixable with phpcbf'),
        ],
      ],
    ];

    // Calculate overall score (with single factor, it's the same as the factor score).
    $overall = $score;

    return [
      'overall' => $overall,
      'factors' => $factors,
    ];
  }

  /**
   * Finds a binary in common locations.
   *
   * @param string $name
   *   The binary name.
   *
   * @return string|null
   *   The path to the binary, or NULL if not found.
   */
  protected function findBinary(string $name): ?string {
    $paths = [
      DRUPAL_ROOT . '/../vendor/bin/' . $name,
      DRUPAL_ROOT . '/vendor/bin/' . $name,
      dirname(DRUPAL_ROOT) . '/vendor/bin/' . $name,
    ];

    foreach ($paths as $path) {
      if (file_exists($path) && is_executable($path)) {
        return $path;
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      // Scoring section: All coding standards violations.
      'coding_standards' => [
        'label' => $this->t('Coding Standards'),
        'description' => $this->t('Drupal and DrupalPractice coding standards violations. Use filters to separate errors from warnings.'),
        'file_key' => 'violations',
        'affects_score' => TRUE,
        'score_factor_key' => 'coding_standards',
        'weight' => self::SCORE_WEIGHTS['coding_standards'],
      ],
      // Informational section: Summary (table).
      'analysis_summary' => [
        'label' => $this->t('Analysis Summary'),
        'description' => $this->t('Summary of files analyzed, configuration, and most common violations.'),
        'file_key' => 'summary',
        'affects_score' => FALSE,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    $files = $data['_files'] ?? [];

    return match ($check_id) {
      // Scoring section: All violations (faceted list).
      'coding_standards' => $this->getCodingStandardsContent($files),
      // Informational section (table).
      'analysis_summary' => $this->buildAnalysisSummaryContent($data),
      default => [],
    };
  }

  /**
   * Gets the coding errors content (faceted list).
   *
   * @param array $files
   *   The _files array from analysis.
   *
   * @return array
   *   Render array content.
   */
  protected function getCodingErrorsContent(array $files): array {
    $results = $files['errors']['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('No coding standards errors found. Your code follows Drupal standards.'),
          'success'
        ),
      ];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $file = $details['file'] ?? '';
      $line = $details['line'] ?? 0;
      $message = $details['message'] ?? $item['message'] ?? '';
      $sniff = $details['source'] ?? '';
      $fixable = $details['fixable'] ?? FALSE;

      // Build description with the specific error message and fix suggestions.
      $description = '<p><strong>' . $this->t('Error:') . '</strong> ' . htmlspecialchars($message) . '</p>';
      $description .= '<p><strong>' . $this->t('How to fix:') . '</strong> ' . $this->getViolationFix($sniff, $message, $fixable) . '</p>';
      if (!empty($sniff)) {
        $description .= '<p class="audit-rule"><strong>' . $this->t('Rule:') . '</strong> <code>' . htmlspecialchars($sniff) . '</code></p>';
      }

      // Build label with auto-fixable indicator.
      $label = $this->getViolationLabel($sniff, $message);

      $issues[] = $this->ui->issue([
        'severity' => 'error',
        'code' => $fixable ? 'PHPCS_ERROR_FIXABLE' : 'PHPCS_ERROR',
        'file' => $file,
        'line' => $line ?: NULL,
        'label' => $label,
        'fixable' => $fixable,
        'description' => ['#markup' => $description],
        'code_snippet' => $this->ui->buildIssueCodeSnippet($details, 'error'),
        'tags' => $this->getViolationTags($sniff, 'error', $fixable),
      ]);
    }

    return ['list' => $this->ui->issueList($issues)];
  }

  /**
   * Gets the coding warnings content (faceted list).
   *
   * @param array $files
   *   The _files array from analysis.
   *
   * @return array
   *   Render array content.
   */
  protected function getCodingWarningsContent(array $files): array {
    $results = $files['warnings']['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('No coding standards warnings found. Your code follows best practices.'),
          'success'
        ),
      ];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $file = $details['file'] ?? '';
      $line = $details['line'] ?? 0;
      $message = $details['message'] ?? $item['message'] ?? '';
      $sniff = $details['source'] ?? '';
      $fixable = $details['fixable'] ?? FALSE;

      // Build description with the specific warning message and fix suggestions.
      $description = '<p><strong>' . $this->t('Warning:') . '</strong> ' . htmlspecialchars($message) . '</p>';
      $description .= '<p><strong>' . $this->t('How to fix:') . '</strong> ' . $this->getViolationFix($sniff, $message, $fixable) . '</p>';
      if (!empty($sniff)) {
        $description .= '<p class="audit-rule"><strong>' . $this->t('Rule:') . '</strong> <code>' . htmlspecialchars($sniff) . '</code></p>';
      }

      // Build label with auto-fixable indicator.
      $label = $this->getViolationLabel($sniff, $message);

      $issues[] = $this->ui->issue([
        'severity' => 'warning',
        'code' => $fixable ? 'PHPCS_WARNING_FIXABLE' : 'PHPCS_WARNING',
        'file' => $file,
        'line' => $line ?: NULL,
        'label' => $label,
        'fixable' => $fixable,
        'description' => ['#markup' => $description],
        'code_snippet' => $this->ui->buildIssueCodeSnippet($details, 'warning'),
        'tags' => $this->getViolationTags($sniff, 'warning', $fixable),
      ]);
    }

    return ['list' => $this->ui->issueList($issues)];
  }

  /**
   * Gets the unified coding standards content (faceted list with errors and warnings).
   *
   * @param array $files
   *   The _files array from analysis.
   *
   * @return array
   *   Render array content.
   */
  protected function getCodingStandardsContent(array $files): array {
    $results = $files['violations']['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('No coding standards violations found. Your code fully complies with Drupal standards.'),
          'success'
        ),
      ];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $file = $details['file'] ?? '';
      $line = $details['line'] ?? 0;
      $message = $details['message'] ?? $item['message'] ?? '';
      $sniff = $details['source'] ?? '';
      $fixable = $details['fixable'] ?? FALSE;
      $type = $details['type'] ?? 'ERROR';

      // Determine severity based on PHPCS type.
      $severity = ($type === 'ERROR') ? 'error' : 'warning';
      $severity_label = ($type === 'ERROR') ? $this->t('Error') : $this->t('Warning');

      // Build description with the specific message and fix suggestions.
      $description = '<p><strong>' . $severity_label . ':</strong> ' . htmlspecialchars($message) . '</p>';
      $description .= '<p><strong>' . $this->t('How to fix:') . '</strong> ' . $this->getViolationFix($sniff, $message, $fixable) . '</p>';
      if (!empty($sniff)) {
        $description .= '<p class="audit-rule"><strong>' . $this->t('Rule:') . '</strong> <code>' . htmlspecialchars($sniff) . '</code></p>';
      }

      // Build label with grouped category.
      $label = $this->getViolationLabel($sniff, $message);

      // Build code for filtering (includes type for faceted filtering).
      $code = 'PHPCS_' . $type;
      if ($fixable) {
        $code .= '_FIXABLE';
      }

      $issues[] = $this->ui->issue([
        'severity' => $severity,
        'code' => $code,
        'file' => $file,
        'line' => $line ?: NULL,
        'label' => $label,
        'fixable' => $fixable,
        'description' => ['#markup' => $description],
        'code_snippet' => $this->ui->buildIssueCodeSnippet($details, $severity),
        'tags' => $this->getViolationTags($sniff, $severity, $fixable),
      ]);
    }

    return ['list' => $this->ui->issueList($issues)];
  }

  /**
   * Builds content for the coding violations check.
   *
   * @param array $data
   *   Full analysis data including _files and stats.
   *
   * @return array
   *   Render array content.
   */
  protected function buildCodingViolationsContent(array $data): array {
    $files = $data['_files'] ?? [];
    $stats = $data['stats'] ?? [];
    $results = $files['violations']['results'] ?? [];

    // If there are results, show as faceted list.
    if (!empty($results)) {
      return $this->ui->buildIssueListFromResults(
        $results,
        (string) $this->t('No coding standards violations found.'),
        function (array $item, $ui): array {
          $details = $item['details'] ?? [];
          $file = $details['file'] ?? '';
          $line = $details['line'] ?? 0;
          $column = $details['column'] ?? 0;
          $message = $details['message'] ?? $item['message'] ?? '';
          $sniff = $details['sniff'] ?? '';
          $fixable = $details['fixable'] ?? FALSE;
          $severity_type = $details['type'] ?? 'warning';

          // Determine tags based on sniff code.
          $tags = ['phpcs'];
          if (str_contains($sniff, 'Commenting')) {
            $tags[] = 'documentation';
          }
          if (str_contains($sniff, 'Naming')) {
            $tags[] = 'naming';
          }
          if (str_contains($sniff, 'Formatting') || str_contains($sniff, 'WhiteSpace')) {
            $tags[] = 'formatting';
          }
          if (str_contains($sniff, 'DrupalPractice')) {
            $tags[] = 'best-practice';
          }
          if ($fixable) {
            $tags[] = 'auto-fixable';
          }

          // Build description.
          $description_parts = [];
          if (!empty($sniff)) {
            $description_parts[] = '<p class="audit-rule"><strong>Rule:</strong> <code>' . htmlspecialchars($sniff) . '</code></p>';
          }
          if ($fixable) {
            $description_parts[] = '<p class="audit-tip"><strong>Auto-fixable:</strong> Run <code>phpcbf</code> to fix automatically.</p>';
          }

          return [
            'severity' => $ui->normalizeSeverity($severity_type === 'ERROR' ? 'error' : 'warning'),
            'code' => $item['code'] ?? 'PHPCS_VIOLATION',
            'file' => $file . ($line ? ':' . $line : '') . ($column ? ':' . $column : ''),
            'label' => $this->truncateMessage($message, 120),
            'description' => !empty($description_parts) ? ['#markup' => implode('', $description_parts)] : NULL,
            'tags' => $tags,
          ];
        }
      );
    }

    // No issues - show success message.
    $files_analyzed = $stats['files_analyzed'] ?? 0;
    return [
      'message' => $this->ui->message(
        (string) $this->t('No coding standards violations found in @count files. Your code follows Drupal standards.', [
          '@count' => $files_analyzed,
        ]),
        'success'
      ),
    ];
  }

  /**
   * Builds content for the analysis summary check.
   *
   * @param array $data
   *   Full analysis data including _files and stats.
   *
   * @return array
   *   Render array content.
   */
  protected function buildAnalysisSummaryContent(array $data): array {
    $files = $data['_files'] ?? [];
    $stats = $data['stats'] ?? [];
    $summary_data = $files['summary']['data'] ?? [];

    // Merge with stats if summary_data is empty.
    $files_analyzed = $summary_data['files_analyzed'] ?? $stats['files_analyzed'] ?? 0;
    $files_with_violations = $summary_data['files_with_violations'] ?? $stats['files_with_violations'] ?? 0;
    $total_errors = $summary_data['total_errors'] ?? $stats['total_errors'] ?? 0;
    $total_warnings = $summary_data['total_warnings'] ?? $stats['total_warnings'] ?? 0;
    $total_fixable = $summary_data['total_fixable'] ?? $stats['total_fixable'] ?? 0;
    $violations_by_sniff = $summary_data['violations_by_sniff'] ?? $stats['violations_by_sniff'] ?? [];
    $paths_list = $summary_data['paths_list'] ?? $stats['paths_list'] ?? [];
    $phpcbf_available = $summary_data['phpcbf_available'] ?? $stats['phpcbf_available'] ?? FALSE;

    $content = [];

    // Statistics table.
    $stats_headers = [
      $this->ui->header((string) $this->t('Metric'), 'left', '50%'),
      $this->ui->header((string) $this->t('Value'), 'left'),
    ];

    $stats_rows = [];
    $stats_rows[] = $this->ui->row([
      $this->ui->cell((string) $this->t('Files Analyzed')),
      $this->ui->cell((string) $files_analyzed),
    ]);
    $stats_rows[] = $this->ui->row([
      $this->ui->cell((string) $this->t('Files with Violations')),
      $this->ui->cell((string) $files_with_violations),
    ], $files_with_violations > 0 ? 'warning' : NULL);
    $stats_rows[] = $this->ui->row([
      $this->ui->cell((string) $this->t('Errors')),
      $this->ui->cell((string) $total_errors),
    ], $total_errors > 0 ? 'error' : NULL);
    $stats_rows[] = $this->ui->row([
      $this->ui->cell((string) $this->t('Warnings')),
      $this->ui->cell((string) $total_warnings),
    ], $total_warnings > 0 ? 'warning' : NULL);

    if ($total_fixable > 0) {
      $stats_rows[] = $this->ui->row([
        $this->ui->cell((string) $this->t('Auto-fixable')),
        $this->ui->cell((string) $total_fixable),
      ]);
    }

    $content['statistics'] = $this->ui->table($stats_headers, $stats_rows);

    // Most common violations table.
    if (!empty($violations_by_sniff)) {
      $sniff_headers = [
        $this->ui->header((string) $this->t('Rule'), 'left', '70%'),
        $this->ui->header((string) $this->t('Count'), 'right'),
      ];

      $sniff_rows = [];
      foreach (array_slice($violations_by_sniff, 0, 10, TRUE) as $sniff => $count) {
        $sniff_rows[] = $this->ui->row([
          $this->ui->cell('<code>' . htmlspecialchars($sniff) . '</code>'),
          $this->ui->cell((string) $count),
        ]);
      }

      $content['top_violations_title'] = [
        '#type' => 'html_tag',
        '#tag' => 'h4',
        '#value' => (string) $this->t('Most Common Violations'),
        '#attributes' => ['class' => ['audit-section-title']],
      ];
      $content['top_violations'] = $this->ui->table($sniff_headers, $sniff_rows);
    }

    // Auto-fix instructions if phpcbf is available.
    if ($phpcbf_available && $total_fixable > 0) {
      $paths_str = !empty($paths_list) ? implode(' ', $paths_list) : 'web/modules/custom';

      $content['fix_instructions'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['audit-fix-section']],
        'message' => $this->ui->message(
          (string) $this->t('@count violations can be automatically fixed.', ['@count' => $total_fixable]),
          'info'
        ),
        'command' => [
          '#type' => 'html_tag',
          '#tag' => 'pre',
          '#value' => './vendor/bin/phpcbf --standard=Drupal,DrupalPractice ' . $paths_str,
          '#attributes' => ['class' => ['audit-code-block']],
        ],
      ];
    }

    return $content;
  }

  /**
   * {@inheritdoc}
   */
  public function buildDetailedResults(array $data): array {
    $stats = $data['stats'] ?? [];

    // Not installed - show installation instructions.
    if (!($stats['available'] ?? FALSE)) {
      // Check if it's an incompatibility issue.
      if (!empty($stats['incompatible_coder'])) {
        return ['incompatible' => $this->buildIncompatibleCoderSection($stats)];
      }
      return ['installation' => $this->buildInstallationSection($stats)];
    }

    // No code to analyze.
    if (!empty($stats['no_code'])) {
      return [
        'no_code' => $this->ui->message(
          (string) $this->t('No custom code found to analyze. Check the "Directories to scan" setting in the main Audit configuration.'),
          'info'
        ),
      ];
    }

    // Use parent implementation for standard sections with score circles.
    return parent::buildDetailedResults($data);
  }

  /**
   * Gets violations content (without section wrapper).
   *
   * @param array $data
   *   The analysis data.
   *
   * @return array
   *   Render array for the content.
   */
  protected function getViolationsContent(array $data): array {
    $content = [];
    $stats = $data['stats'] ?? [];

    // Not installed - show installation instructions.
    if (!($stats['available'] ?? FALSE)) {
      // Check if it's an incompatibility issue.
      if (!empty($stats['incompatible_coder'])) {
        return $this->buildIncompatibleCoderContent($stats);
      }
      else {
        return $this->buildInstallationContent($stats);
      }
    }

    // No code to analyze.
    if (!empty($stats['no_code'])) {
      $content['no_code'] = $this->ui->message(
        (string) $this->t('No custom code found to analyze. Check the "Directories to scan" setting in the main Audit configuration.'),
        'info'
      );
      return $content;
    }

    // Statistics section.
    $content['stats'] = $this->buildStatsSection($stats);

    // Top violations by sniff (helps identify systematic issues).
    if (!empty($stats['violations_by_sniff'])) {
      $content['by_sniff'] = $this->buildViolationsBySniffSection($stats['violations_by_sniff']);
    }

    // Violation details section.
    $violation_items = array_filter(
      $data['results'] ?? [],
      fn($item) => $item['code'] === 'PHPCS_VIOLATION'
    );

    if (!empty($violation_items)) {
      $max_display = $stats['config']['max_violations_display'] ?? self::DEFAULT_MAX_VIOLATIONS;
      $content['violations'] = $this->buildViolationDetailsSection($violation_items, $max_display, $stats['config']['show_sniff_codes'] ?? TRUE);
    }

    // How to fix section.
    if ($stats['total_errors'] > 0 || $stats['total_warnings'] > 0) {
      $content['how_to_fix'] = $this->buildHowToFixSection($stats);
    }

    return $content;
  }

  /**
   * Gets file compliance content (without section wrapper).
   *
   * @param array $data
   *   The analysis data.
   *
   * @return array
   *   Render array for the content.
   */
  protected function getFileComplianceContent(array $data): array {
    $stats = $data['stats'] ?? [];

    // Not available.
    if (!($stats['available'] ?? FALSE)) {
      return [
        'not_available' => $this->ui->message(
          (string) $this->t('File compliance information is not available. PHPCS must be properly installed and configured.'),
          'info'
        ),
      ];
    }

    // No code to analyze.
    if (!empty($stats['no_code'])) {
      return [
        'no_code' => $this->ui->message(
          (string) $this->t('No custom code found to analyze.'),
          'info'
        ),
      ];
    }

    // Show violations by file.
    if (!empty($stats['violations_by_file'])) {
      return $this->buildViolationsByFileContent($stats['violations_by_file'], $stats['phpcbf_available'] ?? FALSE);
    }

    // All files are clean.
    return [
      'all_clean' => $this->ui->message(
        (string) $this->t('All @count analyzed files follow coding standards!', [
          '@count' => $stats['files_analyzed'] ?? 0,
        ]),
        'success'
      ),
    ];
  }

  /**
   * Builds the installation instructions content (without section wrapper).
   *
   * @param array $stats
   *   Statistics data.
   *
   * @return array
   *   Render array.
   */
  protected function buildInstallationContent(array $stats): array {
    $instructions = $stats['installation_instructions'] ?? $this->getInstallationInstructions();

    $content = [];

    // Warning message.
    $content['warning'] = $this->ui->message(
      (string) $this->t('PHP CodeSniffer is not properly installed or configured.'),
      'error'
    );

    // Installation steps.
    $steps_content = [];
    foreach ($instructions['steps'] ?? [] as $index => $step) {
      $steps_content['step_' . $index] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['audit-install-step']],
        'number' => [
          '#markup' => '<strong>' . ($index + 1) . '. ' . ($step['description'] ?? '') . '</strong>',
        ],
        'command' => [
          '#type' => 'html_tag',
          '#tag' => 'pre',
          '#value' => $step['command'] ?? '',
          '#attributes' => ['class' => ['audit-code-block']],
        ],
      ];
    }

    $content['steps'] = $this->ui->section(
      (string) $this->t('Installation Steps'),
      $steps_content,
      ['open' => TRUE]
    );

    // Additional notes.
    if (!empty($instructions['notes'])) {
      $notes_list = [
        '#theme' => 'item_list',
        '#items' => $instructions['notes'],
      ];
      $content['notes'] = $this->ui->section(
        (string) $this->t('Additional Notes'),
        $notes_list,
        ['open' => FALSE]
      );
    }

    // Documentation link.
    if (!empty($instructions['documentation'])) {
      $content['docs'] = [
        '#markup' => '<p>' . $this->t('For more information, see the <a href="@url" target="_blank">official documentation</a>.', [
          '@url' => $instructions['documentation'],
        ]) . '</p>',
      ];
    }

    return $content;
  }

  /**
   * Builds content for incompatible drupal/coder version (without section wrapper).
   *
   * @param array $stats
   *   Statistics data with version info.
   *
   * @return array
   *   Render array.
   */
  protected function buildIncompatibleCoderContent(array $stats): array {
    $content = [];

    // Error message.
    $content['error'] = $this->ui->message(
      (string) $this->t('<strong>Incompatible drupal/coder version detected!</strong><br>Your installed <code>drupal/coder</code> (@coder_version) is not compatible with <code>PHP_CodeSniffer</code> @phpcs_version.', [
        '@coder_version' => $stats['coder_version'] ?? 'unknown',
        '@phpcs_version' => $stats['phpcs_version'] ?? 'unknown',
      ]),
      'error'
    );

    // Explanation.
    $content['explanation'] = [
      '#markup' => '<p>' . $this->t('This usually happens when an old version of drupal/coder (designed for PHP_CodeSniffer 2.x) is used with a newer PHP_CodeSniffer (3.x or 4.x). The class naming conventions changed between major versions.') . '</p>',
    ];

    // Solution.
    $solution_content = [];
    $solution_content['description'] = [
      '#markup' => '<p>' . $this->t('Run the following command to update to a compatible version:') . '</p>',
    ];
    $solution_content['command'] = [
      '#type' => 'html_tag',
      '#tag' => 'pre',
      '#value' => $stats['upgrade_command'] ?? 'composer require --dev drupal/coder:^8.3.0 dealerdirect/phpcodesniffer-composer-installer',
      '#attributes' => ['class' => ['audit-code-block']],
    ];
    $solution_content['note'] = $this->ui->message(
      (string) $this->t('The <code>dealerdirect/phpcodesniffer-composer-installer</code> package automatically registers the coding standards with PHPCS.'),
      'info'
    );

    $content['solution'] = $this->ui->section(
      (string) $this->t('How to Fix'),
      $solution_content,
      ['open' => TRUE]
    );

    // Version recommendations.
    $versions_content = [
      '#theme' => 'item_list',
      '#items' => [
        $this->t('<strong>Drupal 10 with PHP 8.1+:</strong> Use <code>drupal/coder:^8.3.0</code>'),
        $this->t('<strong>Drupal 11 with PHP 8.3+:</strong> Use <code>drupal/coder:^8.3.0</code> (or ^9.x when stable)'),
        $this->t('<strong>PHP_CodeSniffer 4.x:</strong> Requires <code>drupal/coder:^8.3.0</code> or newer'),
      ],
    ];
    $content['versions'] = $this->ui->section(
      (string) $this->t('Version Compatibility'),
      $versions_content,
      ['open' => FALSE]
    );

    return $content;
  }

  /**
   * Builds the summary section with key metrics.
   *
   * @param array $stats
   *   Statistics data.
   *
   * @return array
   *   Render array.
   */
  protected function buildSummarySection(array $stats): array {
    $total_violations = $stats['total_violations'] ?? ($stats['total_errors'] + $stats['total_warnings']);
    $fixable = $stats['total_fixable'] ?? 0;

    // Determine overall status.
    if ($total_violations === 0) {
      $status = 'success';
      $icon = '✓';
      $message = $this->t('Excellent! Your code follows Drupal coding standards.');
    }
    elseif ($stats['total_errors'] === 0) {
      $status = 'warning';
      $icon = '⚠';
      $message = $this->t('Good progress! Only warnings remain, no errors.');
    }
    else {
      $status = 'error';
      $icon = '✗';
      $message = $this->t('Issues found. Review errors and warnings below.');
    }

    $content = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-summary-box', 'audit-summary--' . $status]],
    ];

    $content['message'] = [
      '#markup' => '<div class="audit-summary-message"><span class="audit-summary-icon">' . $icon . '</span> ' . $message . '</div>',
    ];

    // Key metrics.
    $metrics = '<div class="audit-summary-metrics">';
    $metrics .= '<span class="audit-metric">' . $this->ui->badge((string) $stats['total_errors'], 'error') . ' ' . $this->t('Errors') . '</span>';
    $metrics .= '<span class="audit-metric">' . $this->ui->badge((string) $stats['total_warnings'], 'warning') . ' ' . $this->t('Warnings') . '</span>';
    $metrics .= '<span class="audit-metric">' . $this->ui->badge((string) $stats['files_with_violations'], 'neutral') . ' ' . $this->t('Files affected') . '</span>';

    if ($fixable > 0) {
      $metrics .= '<span class="audit-metric">' . $this->ui->badge((string) $fixable, 'info') . ' ' . $this->t('Auto-fixable') . '</span>';
    }
    $metrics .= '</div>';

    $content['metrics'] = ['#markup' => $metrics];

    return $content;
  }

  /**
   * Builds the configuration section.
   *
   * @param array $stats
   *   Statistics data.
   *
   * @return array
   *   Render array.
   */
  protected function buildConfigSection(array $stats): array {
    $config = $stats['config'] ?? [];

    $headers = [
      $this->ui->header((string) $this->t('Setting'), 'left', '40%'),
      $this->ui->header((string) $this->t('Value'), 'left'),
    ];

    $rows = [];

    // Standards used.
    $standards = $stats['standards'] ?? ['Drupal', 'DrupalPractice'];
    $rows[] = [
      $this->ui->itemName((string) $this->t('Coding Standards')),
      implode(' + ', array_map(fn($s) => $this->ui->badge($s, 'info'), $standards)),
    ];

    // Paths analyzed.
    $paths_list = implode(', ', array_map(fn($p) => '<code>' . basename($p) . '</code>', $stats['paths_list'] ?? []));
    $rows[] = [
      $this->ui->itemName((string) $this->t('Directories Analyzed')),
      $paths_list ?: '-',
    ];

    // Extensions.
    $rows[] = [
      $this->ui->itemName((string) $this->t('File Extensions')),
      $this->ui->cell('<code>' . htmlspecialchars($config['extensions'] ?? self::DEFAULT_EXTENSIONS, ENT_QUOTES, 'UTF-8') . '</code>'),
    ];

    // Severity.
    $rows[] = [
      $this->ui->itemName((string) $this->t('Minimum Severity')),
      (string) ($config['min_severity'] ?? self::DEFAULT_MIN_SEVERITY) . '/10',
    ];

    // Warnings ignored.
    $ignore_warnings = $config['ignore_warnings'] ?? FALSE;
    $rows[] = [
      $this->ui->itemName((string) $this->t('Warnings')),
      $this->ui->badge(
        $ignore_warnings ? (string) $this->t('Ignored') : (string) $this->t('Included'),
        $ignore_warnings ? 'warning' : 'success'
      ),
    ];

    // Excluded sniffs.
    $excluded = $config['excluded_sniffs'] ?? [];
    if (!empty($excluded)) {
      $rows[] = [
        $this->ui->itemName((string) $this->t('Excluded Rules')),
        count($excluded) . ' ' . $this->t('rules'),
      ];
    }

    $table = $this->ui->table($headers, $rows);

    return $this->ui->section(
      (string) $this->t('Analysis Configuration'),
      $table,
      ['open' => FALSE]
    );
  }

  /**
   * Builds the statistics section.
   *
   * @param array $stats
   *   Statistics data.
   *
   * @return array
   *   Render array.
   */
  protected function buildStatsSection(array $stats): array {
    $headers = [
      $this->ui->header((string) $this->t('Metric'), 'left', '50%'),
      $this->ui->header((string) $this->t('Value'), 'left'),
    ];

    $rows = [];

    // Files analyzed.
    $rows[] = [
      $this->ui->itemName((string) $this->t('Files Analyzed')),
      $this->ui->number($stats['files_analyzed'] ?? 0, []),
    ];

    // Total errors.
    $rows[] = $this->ui->row([
      $this->ui->itemName((string) $this->t('Total Errors')),
      $this->ui->cell(
        $this->ui->number($stats['total_errors'], ['error' => 1]) . ' ' .
        $this->ui->badge(
          $stats['total_errors'] === 0 ? (string) $this->t('OK') : (string) $this->t('Issues'),
          $stats['total_errors'] === 0 ? 'success' : 'error'
        )
      ),
    ], $stats['total_errors'] > 0 ? 'error' : NULL);

    // Total warnings.
    $rows[] = $this->ui->row([
      $this->ui->itemName((string) $this->t('Total Warnings')),
      $this->ui->cell(
        $this->ui->number($stats['total_warnings'], ['warning' => 1]) . ' ' .
        $this->ui->badge(
          $stats['total_warnings'] === 0 ? (string) $this->t('OK') : (string) $this->t('Warnings'),
          $stats['total_warnings'] === 0 ? 'success' : 'warning'
        )
      ),
    ], $stats['total_warnings'] > 0 ? 'warning' : NULL);

    // Fixable issues.
    $fixable = $stats['total_fixable'] ?? 0;
    if ($fixable > 0) {
      $rows[] = [
        $this->ui->itemName((string) $this->t('Auto-fixable Issues')),
        $this->ui->cell(
          $this->ui->number($fixable, []) . ' ' .
          $this->ui->badge((string) $this->t('Can fix with phpcbf'), 'info')
        ),
      ];
    }

    // Files with violations.
    $rows[] = [
      $this->ui->itemName((string) $this->t('Files with Violations')),
      (string) $stats['files_with_violations'] . ' / ' . ($stats['files_analyzed'] ?? '?'),
    ];

    $table = $this->ui->table($headers, $rows);

    return $this->ui->section(
      (string) $this->t('Analysis Results'),
      $table,
      ['open' => TRUE]
    );
  }

  /**
   * Builds violations by sniff section (helps identify systematic issues).
   *
   * @param array $violations_by_sniff
   *   Violations grouped by sniff code.
   *
   * @return array
   *   Render array.
   */
  protected function buildViolationsBySniffSection(array $violations_by_sniff): array {
    $headers = [
      $this->ui->header((string) $this->t('Rule'), 'left'),
      $this->ui->header((string) $this->t('Count'), 'center', '100px'),
      $this->ui->header((string) $this->t('Action'), 'left', '200px'),
    ];

    $rows = [];
    foreach ($violations_by_sniff as $sniff => $count) {
      // Determine if this sniff is excludable.
      $is_excludable = isset(self::EXCLUDABLE_SNIFFS[$sniff]);
      $action = $is_excludable
        ? $this->ui->badge((string) $this->t('Can exclude'), 'info')
        : $this->ui->badge((string) $this->t('Fix required'), 'warning');

      $rows[] = [
        $this->ui->cell('<code>' . htmlspecialchars($sniff, ENT_QUOTES, 'UTF-8') . '</code>'),
        $this->ui->cell($this->ui->number($count, ['warning' => 10, 'error' => 50]), ['align' => 'center']),
        $action,
      ];
    }

    $table = $this->ui->table($headers, $rows);

    $content = ['table' => $table];
    $content['note'] = $this->ui->message(
      (string) $this->t('Rules with many violations may indicate systematic issues. Consider addressing the most frequent ones first, or temporarily excluding them for legacy code.'),
      'info'
    );

    return $this->ui->section(
      (string) $this->t('Top Violations by Rule'),
      $content,
      ['open' => FALSE, 'count' => count($violations_by_sniff)]
    );
  }

  /**
   * Builds violations by file content (without section wrapper).
   *
   * @param array $violations_by_file
   *   Violations data by file.
   * @param bool $phpcbf_available
   *   Whether PHPCBF is available.
   *
   * @return array
   *   Render array.
   */
  protected function buildViolationsByFileContent(array $violations_by_file, bool $phpcbf_available): array {
    $headers = [
      $this->ui->header((string) $this->t('File'), 'left'),
      $this->ui->header((string) $this->t('Errors'), 'center', '80px'),
      $this->ui->header((string) $this->t('Warnings'), 'center', '80px'),
    ];

    if ($phpcbf_available) {
      $headers[] = $this->ui->header((string) $this->t('Fixable'), 'center', '80px');
    }

    $rows = [];
    foreach ($violations_by_file as $file => $counts) {
      $severity = $counts['errors'] > 0 ? 'error' : ($counts['warnings'] > 0 ? 'warning' : NULL);

      $row_data = [
        $this->ui->cell('<code>' . htmlspecialchars($file, ENT_QUOTES, 'UTF-8') . '</code>'),
        $this->ui->cell(
          $this->ui->number($counts['errors'], ['error' => 1]),
          ['align' => 'center']
        ),
        $this->ui->cell(
          $this->ui->number($counts['warnings'], ['warning' => 1]),
          ['align' => 'center']
        ),
      ];

      if ($phpcbf_available) {
        $fixable = $counts['fixable'] ?? 0;
        $row_data[] = $this->ui->cell(
          $fixable > 0 ? $this->ui->badge((string) $fixable, 'info') : '-',
          ['align' => 'center']
        );
      }

      $rows[] = $this->ui->row($row_data, $severity);
    }

    $table = $this->ui->table($headers, $rows);

    return ['table' => $table];
  }

  /**
   * Builds the violation details section with code snippets.
   *
   * @param array $violation_items
   *   Array of violation items.
   * @param int $max_display
   *   Maximum number to display.
   * @param bool $show_sniff_codes
   *   Whether to show sniff codes.
   *
   * @return array
   *   Render array.
   */
  protected function buildViolationDetailsSection(array $violation_items, int $max_display = 2000, bool $show_sniff_codes = TRUE): array {
    $content = [];
    $display_limit = $max_display > 0 ? $max_display : count($violation_items);
    $displayed_items = array_slice($violation_items, 0, $display_limit);

    // Group violations by file for better organization.
    $by_file = [];
    foreach ($displayed_items as $item) {
      $file = $item['details']['file'] ?? 'unknown';
      if (!isset($by_file[$file])) {
        $by_file[$file] = [];
      }
      $by_file[$file][] = $item;
    }

    // Sort files alphabetically by path for better grouping.
    ksort($by_file, SORT_STRING);

    foreach ($by_file as $file => $file_violations) {
      $file_content = [];

      foreach ($file_violations as $index => $item) {
        $details = $item['details'];
        $violation_id = 'violation-' . md5($file . '-' . $index);

        $file_content[$violation_id] = $this->buildSingleViolation($details, $show_sniff_codes);
      }

      // Create collapsible section for each file.
      $error_count = count(array_filter($file_violations, fn($v) => $v['details']['type'] === 'ERROR'));
      $warning_count = count($file_violations) - $error_count;
      $fixable_count = count(array_filter($file_violations, fn($v) => !empty($v['details']['fixable'])));

      // Build badges for file title.
      $file_badges = [];
      if ($error_count > 0) {
        $file_badges[] = $this->ui->badge((string) $error_count . ' E', 'error');
      }
      if ($warning_count > 0) {
        $file_badges[] = $this->ui->badge((string) $warning_count . ' W', 'warning');
      }
      if ($fixable_count > 0) {
        $file_badges[] = $this->ui->badge((string) $fixable_count . ' ✓', 'info');
      }

      // Use full path in title, with badges.
      $file_title = '<code class="audit-file-path-title">' . htmlspecialchars($file, ENT_QUOTES, 'UTF-8') . '</code> ' . implode(' ', $file_badges);

      $content['file_' . md5($file)] = $this->ui->section(
        $file_title,
        $file_content,
        [
          'open' => FALSE,
          'severity' => $error_count > 0 ? 'error' : 'warning',
        ]
      );
    }

    if (count($violation_items) > $display_limit) {
      $content['limit_note'] = $this->ui->message(
        (string) $this->t('Showing @limit of @total violations. Adjust "Maximum violations to display" in settings to see more.', [
          '@limit' => $display_limit,
          '@total' => count($violation_items),
        ]),
        'info'
      );
    }

    return $this->ui->section(
      (string) $this->t('Violation Details'),
      $content,
      ['open' => TRUE]
    );
  }

  /**
   * Builds a single violation display with code context.
   *
   * @param array $details
   *   Violation details.
   * @param bool $show_sniff_codes
   *   Whether to show sniff codes.
   *
   * @return array
   *   Render array for a single violation.
   */
  protected function buildSingleViolation(array $details, bool $show_sniff_codes = TRUE): array {
    $is_error = $details['type'] === 'ERROR';
    $is_fixable = $details['fixable'] ?? FALSE;
    $sniff_code = $details['source'] ?? '';
    $severity_level = $details['severity'] ?? 5;

    $build = [
      '#type' => 'container',
      '#attributes' => [
        'class' => [
          'audit-violation',
          'audit-violation--' . ($is_error ? 'error' : 'warning'),
        ],
      ],
    ];

    // Location bar - clean, separate line with key info.
    $location_html = '<div class="audit-violation-location-bar">';
    $location_html .= '<span class="audit-location-item audit-location-line">';
    $location_html .= '<strong>' . $this->t('Line') . '</strong> ' . $details['line'];
    $location_html .= '</span>';
    $location_html .= '<span class="audit-location-separator">•</span>';
    $location_html .= '<span class="audit-location-item audit-location-column">';
    $location_html .= '<strong>' . $this->t('Col') . '</strong> ' . $details['column'];
    $location_html .= '</span>';
    $location_html .= '<span class="audit-location-separator">•</span>';
    $location_html .= '<span class="audit-location-item audit-location-severity audit-severity-' . ($severity_level >= 7 ? 'high' : ($severity_level >= 4 ? 'medium' : 'low')) . '">';
    $location_html .= '<strong>' . $this->t('Severity') . '</strong> ' . $severity_level . '/10';
    $location_html .= '</span>';

    // Add badges on the right.
    $location_html .= '<span class="audit-violation-badges">';
    $location_html .= $this->ui->badge(
      $is_error ? (string) $this->t('ERROR') : (string) $this->t('WARNING'),
      $is_error ? 'error' : 'warning'
    );
    if ($is_fixable) {
      $location_html .= ' ' . $this->ui->badge((string) $this->t('Auto-fix'), 'info');
    }
    $location_html .= '</span>';
    $location_html .= '</div>';

    $build['location'] = ['#markup' => $location_html];

    // Message - the actual violation description.
    $build['message'] = [
      '#markup' => '<div class="audit-violation-message">' .
        htmlspecialchars($details['message'], ENT_QUOTES, 'UTF-8') .
        '</div>',
    ];

    // Sniff code with documentation link - styled as a footer.
    if ($show_sniff_codes && $sniff_code) {
      $build['sniff'] = $this->buildSniffInfo($sniff_code);
    }

    // Code context.
    if (!empty($details['code_context']['lines'])) {
      $build['code'] = $this->ui->code(
        $details['code_context']['lines'],
        [
          'highlight_line' => $details['code_context']['highlight_line'],
          'severity' => $is_error ? 'error' : 'warning',
          'language' => 'php',
        ]
      );
    }

    return $build;
  }

  /**
   * Builds sniff information with documentation link.
   *
   * @param string $sniff_code
   *   The sniff code (e.g., "Drupal.Commenting.DocComment").
   *
   * @return array
   *   Render array.
   */
  protected function buildSniffInfo(string $sniff_code): array {
    $parts = explode('.', $sniff_code);
    $standard = $parts[0] ?? '';

    // Build documentation URL.
    $doc_url = NULL;
    if ($standard === 'Drupal' || $standard === 'DrupalPractice') {
      // Link to Drupal coding standards.
      $doc_url = 'https://www.drupal.org/docs/develop/standards';
    }

    // Get human-readable description if available.
    $description = $this->getSniffDescription($sniff_code);

    // Build a clean footer with rule info.
    $info = '<div class="audit-sniff-footer">';
    $info .= '<span class="audit-sniff-label">' . $this->t('Rule:') . '</span> ';
    $info .= '<code class="audit-sniff-code">' . htmlspecialchars($sniff_code, ENT_QUOTES, 'UTF-8') . '</code>';

    if ($description) {
      $info .= '<span class="audit-sniff-separator">—</span>';
      $info .= '<span class="audit-sniff-description">' . htmlspecialchars($description, ENT_QUOTES, 'UTF-8') . '</span>';
    }

    if ($doc_url) {
      $info .= '<a href="' . htmlspecialchars($doc_url, ENT_QUOTES, 'UTF-8') . '" target="_blank" class="audit-sniff-docs-link">' .
        '<span class="audit-docs-icon">📖</span> ' . $this->t('Documentation') . '</a>';
    }

    $info .= '</div>';

    return ['#markup' => $info];
  }

  /**
   * Gets a human-readable description for common sniffs.
   *
   * @param string $sniff_code
   *   The sniff code.
   *
   * @return string|null
   *   Description or NULL if not available.
   */
  protected function getSniffDescription(string $sniff_code): ?string {
    $descriptions = [
      // Drupal standard - Commenting.
      'Drupal.Commenting.DocComment' => (string) $this->t('DocBlock comment format'),
      'Drupal.Commenting.DocComment.Missing' => (string) $this->t('Missing documentation comment'),
      'Drupal.Commenting.DocComment.MissingShort' => (string) $this->t('Missing short description in docblock'),
      'Drupal.Commenting.DocComment.ShortNotCapital' => (string) $this->t('Short description must start with capital'),
      'Drupal.Commenting.DocComment.ShortSingleLine' => (string) $this->t('Short description should be on single line'),
      'Drupal.Commenting.DocComment.SpacingAfter' => (string) $this->t('Spacing after docblock'),
      'Drupal.Commenting.DocComment.SpacingBefore' => (string) $this->t('Spacing before docblock'),
      'Drupal.Commenting.FunctionComment' => (string) $this->t('Function/method comment format'),
      'Drupal.Commenting.FunctionComment.Missing' => (string) $this->t('Missing function comment'),
      'Drupal.Commenting.FunctionComment.MissingParamComment' => (string) $this->t('Missing parameter description'),
      'Drupal.Commenting.FunctionComment.MissingParamType' => (string) $this->t('Missing parameter type'),
      'Drupal.Commenting.FunctionComment.MissingReturnComment' => (string) $this->t('Missing return description'),
      'Drupal.Commenting.FunctionComment.MissingReturnType' => (string) $this->t('Missing return type'),
      'Drupal.Commenting.FunctionComment.ParamMissingDefinition' => (string) $this->t('Parameter missing from docblock'),
      'Drupal.Commenting.FileComment' => (string) $this->t('File-level comment required'),
      'Drupal.Commenting.InlineComment' => (string) $this->t('Inline comment format'),
      'Drupal.Commenting.InlineComment.SpacingBefore' => (string) $this->t('Spacing before inline comment'),
      'Drupal.Commenting.InlineComment.InvalidEndChar' => (string) $this->t('Inline comment must end with period'),

      // Drupal standard - Files.
      'Drupal.Files.LineLength' => (string) $this->t('Line exceeds 80 characters'),
      'Drupal.Files.LineLength.TooLong' => (string) $this->t('Line is too long'),
      'Drupal.Files.EndFileNewline' => (string) $this->t('File must end with newline'),
      'Drupal.Files.TxtFileLineLength' => (string) $this->t('Text file line length'),

      // Drupal standard - Formatting.
      'Drupal.WhiteSpace.ScopeIndent' => (string) $this->t('Incorrect indentation'),
      'Drupal.WhiteSpace.EmptyLines' => (string) $this->t('Empty line issues'),
      'Drupal.WhiteSpace.Comma' => (string) $this->t('Comma spacing'),
      'Drupal.WhiteSpace.OperatorSpacing' => (string) $this->t('Operator spacing'),
      'Drupal.Formatting.SpaceInlineIf' => (string) $this->t('Ternary operator spacing'),
      'Drupal.Formatting.MultiLineAssignment' => (string) $this->t('Multi-line assignment format'),

      // Drupal standard - Arrays.
      'Drupal.Arrays.Array' => (string) $this->t('Array format (use short syntax)'),
      'Drupal.Arrays.Array.LongSyntax' => (string) $this->t('Use [] instead of array()'),

      // Drupal standard - Classes.
      'Drupal.Classes.ClassFileName' => (string) $this->t('Class file naming'),
      'Drupal.Classes.FullyQualifiedNamespace' => (string) $this->t('Use imports instead of FQN'),
      'Drupal.Classes.UnusedUseStatement' => (string) $this->t('Unused use statement'),
      'Drupal.Classes.UseLeadingBackslash' => (string) $this->t('Remove leading backslash from use'),

      // Drupal standard - Strings.
      'Drupal.Strings.UnnecessaryStringConcat' => (string) $this->t('Unnecessary string concatenation'),
      'Drupal.Semantics.FunctionT' => (string) $this->t('Translation function usage'),
      'Drupal.Semantics.FunctionT.NotLiteralString' => (string) $this->t('t() must use literal string'),

      // DrupalPractice standard.
      'DrupalPractice.CodeAnalysis.VariableAnalysis' => (string) $this->t('Unused or undefined variable'),
      'DrupalPractice.CodeAnalysis.VariableAnalysis.UnusedVariable' => (string) $this->t('Variable is defined but never used'),
      'DrupalPractice.CodeAnalysis.VariableAnalysis.UndefinedVariable' => (string) $this->t('Variable is used before being defined'),
      'DrupalPractice.Objects.GlobalDrupal' => (string) $this->t('Avoid using \\Drupal::, use dependency injection'),
      'DrupalPractice.Objects.GlobalFunction' => (string) $this->t('Avoid global functions, use services'),
      'DrupalPractice.Objects.GlobalClass' => (string) $this->t('Avoid global class instantiation'),
      'DrupalPractice.General.ClassName' => (string) $this->t('Class naming convention'),
      'DrupalPractice.General.MethodName' => (string) $this->t('Method naming convention'),
      'DrupalPractice.Commenting.AuthorTag' => (string) $this->t('@author tag not recommended'),
      'DrupalPractice.FunctionCalls.InsecureUnserialize' => (string) $this->t('Security: Use safe unserialization'),
    ];

    // Try exact match first.
    if (isset($descriptions[$sniff_code])) {
      return $descriptions[$sniff_code];
    }

    // Try parent sniff (without the specific check).
    $parts = explode('.', $sniff_code);
    if (count($parts) >= 3) {
      $parent_sniff = $parts[0] . '.' . $parts[1] . '.' . $parts[2];
      if (isset($descriptions[$parent_sniff])) {
        return $descriptions[$parent_sniff];
      }
    }

    return NULL;
  }

  /**
   * Builds the "How to Fix" section with actionable guidance.
   *
   * @param array $stats
   *   Statistics data.
   *
   * @return array
   *   Render array.
   */
  protected function buildHowToFixSection(array $stats): array {
    $content = [];

    // Auto-fix with PHPCBF.
    $fixable = $stats['total_fixable'] ?? 0;
    if ($fixable > 0 && ($stats['phpcbf_available'] ?? FALSE)) {
      $content['auto_fix'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['audit-fix-section']],
        'title' => ['#markup' => '<h4>' . $this->t('1. Auto-fix with PHPCBF (@count issues)', ['@count' => $fixable]) . '</h4>'],
        'description' => ['#markup' => '<p>' . $this->t('PHPCBF can automatically fix formatting issues like spacing, indentation, and array syntax:') . '</p>'],
        'command' => [
          '#type' => 'html_tag',
          '#tag' => 'pre',
          '#value' => './vendor/bin/phpcbf --standard=Drupal,DrupalPractice web/modules/custom/my_module',
          '#attributes' => ['class' => ['audit-code-block']],
        ],
        'note' => [
          '#markup' => '<p class="audit-note">' . $this->t('Replace <code>web/modules/custom/my_module</code> with the path you want to fix: a directory, a module, or a specific file.') . '</p>',
        ],
        'warning' => $this->ui->message(
          (string) $this->t('Always commit your code before running phpcbf, then review changes with git diff.'),
          'warning'
        ),
      ];
    }

    // Manual fixes.
    $step_number = $fixable > 0 && ($stats['phpcbf_available'] ?? FALSE) ? '2' : '1';
    $content['manual_fix'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-fix-section']],
      'title' => ['#markup' => '<h4>' . $this->t('@num. Manual Fixes Required', ['@num' => $step_number]) . '</h4>'],
      'description' => ['#markup' => '<p>' . $this->t('Some issues require manual correction:') . '</p>'],
      'list' => [
        '#theme' => 'item_list',
        '#items' => [
          $this->t('<strong>DocBlock comments:</strong> Add missing documentation for functions, classes, and files.'),
          $this->t('<strong>DrupalPractice issues:</strong> Use dependency injection instead of \\Drupal::service(), avoid deprecated functions.'),
          $this->t('<strong>Naming conventions:</strong> Follow Drupal naming standards for variables, functions, and classes.'),
        ],
      ],
    ];

    // Verify fixes.
    $verify_step = $fixable > 0 && ($stats['phpcbf_available'] ?? FALSE) ? '3' : '2';
    $content['verify'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-fix-section']],
      'title' => ['#markup' => '<h4>' . $this->t('@num. Verify Fixes', ['@num' => $verify_step]) . '</h4>'],
      'description' => ['#markup' => '<p>' . $this->t('After fixing issues, reload this page to run the analysis again and verify the issues have been resolved.') . '</p>'],
    ];

    // Resources.
    $content['resources'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-fix-section']],
      'title' => ['#markup' => '<h4>' . $this->t('Resources') . '</h4>'],
      'list' => [
        '#theme' => 'item_list',
        '#items' => [
          $this->t('<a href="@url" target="_blank">Drupal Coding Standards</a>', ['@url' => 'https://www.drupal.org/docs/develop/standards']),
          $this->t('<a href="@url" target="_blank">PHP CodeSniffer Documentation</a>', ['@url' => 'https://github.com/squizlabs/PHP_CodeSniffer/wiki']),
          $this->t('<a href="@url" target="_blank">drupal/coder on drupal.org</a>', ['@url' => 'https://www.drupal.org/project/coder']),
        ],
      ],
    ];

    return $this->ui->section(
      (string) $this->t('How to Fix These Issues'),
      $content,
      ['open' => TRUE, 'severity' => 'notice']
    );
  }

}
