<?php

declare(strict_types=1);

namespace Drupal\audit_phpstan\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Process\Process;

/**
 * Runs PHPStan static analysis.
 */
#[AuditAnalyzer(
  id: 'phpstan',
  label: new TranslatableMarkup('PHPStan Analysis'),
  description: new TranslatableMarkup('Runs PHPStan static analysis to detect potential bugs and type errors.'),
  menu_title: new TranslatableMarkup('PHPStan'),
  output_directory: 'phpstan',
  weight: 2,
)]
class PhpstanAnalyzer extends AuditAnalyzerBase {

  /**
   * Score weights for different factors.
   *
   * Weights sum to 100 and reflect relative importance:
   * - Type errors are the most critical (incorrect code behavior)
   * - Deprecations important for Drupal 11 readiness
   * - Undefined issues indicate missing code/typos
   *
   * Each factor has a corresponding visible section in getAuditChecks().
   */
  protected const SCORE_WEIGHTS = [
    'type_errors' => 45,
    'deprecations' => 35,
    'undefined_issues' => 20,
  ];

  /**
   * Penalty multiplier applied to issue percentage.
   *
   * With multiplier 2.5:
   * - 4% issues → score 90
   * - 10% issues → score 75
   * - 20% issues → score 50
   * - 40%+ issues → score 0
   */
  protected const PENALTY_MULTIPLIER = 2.5;

  /**
   * Default PHPStan level.
   */
  protected const DEFAULT_LEVEL = 5;

  /**
   * Default max errors to display.
   */
  protected const DEFAULT_MAX_ERRORS = 500;

  /**
   * PHPStan level descriptions.
   */
  protected const LEVEL_DESCRIPTIONS = [
    0 => 'Basic checks, unknown types, undefined variables',
    1 => 'Possibly undefined variables, dead code checks',
    2 => 'Unknown methods, validates @var annotations',
    3 => 'Return types, assigned types validation',
    4 => 'Type hints, dead catch analysis',
    5 => 'Argument types validation (recommended starting point)',
    6 => 'Reports missing type hints',
    7 => 'Union types checked',
    8 => 'Nullable types must be null-checked',
    9 => 'Full strictness, mixed type rules',
    10 => 'Bleeding edge (max experimental)',
  ];

  /**
   * Required packages for full Drupal support.
   */
  protected const REQUIRED_PACKAGES = [
    'phpstan/phpstan' => 'Core PHPStan tool',
    'phpstan/extension-installer' => 'Auto-discovers PHPStan extensions',
    'mglaman/phpstan-drupal' => 'Drupal-specific rules and type coverage',
    'phpstan/phpstan-deprecation-rules' => 'Detects deprecated code for Drupal 11 preparation',
  ];

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->configFactory = $container->get('config.factory');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->moduleHandler = $container->get('module_handler');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $config): array {
    $form = [];

    // Build level options with descriptions.
    $level_options = [];
    foreach (self::LEVEL_DESCRIPTIONS as $level => $description) {
      $label = $level;
      if ($level === 5) {
        $label .= ' (' . $this->t('recommended') . ')';
      }
      elseif ($level === 10) {
        $label .= ' (max)';
      }
      $level_options[$level] = $label . ' - ' . $description;
    }

    // Basic settings.
    $form['basic'] = [
      '#type' => 'details',
      '#title' => $this->t('Basic Settings'),
      '#open' => TRUE,
    ];

    $form['basic']['level'] = [
      '#type' => 'select',
      '#title' => $this->t('Analysis level'),
      '#description' => $this->t('PHPStan strictness level (0-10). <strong>Recommended: Level 5</strong> for most Drupal projects. Higher levels detect more issues but may report false positives. Start with level 5 and increase gradually as you fix errors.'),
      '#options' => $level_options,
      '#default_value' => $config['level'] ?? self::DEFAULT_LEVEL,
    ];

    $form['basic']['memory_limit'] = [
      '#type' => 'select',
      '#title' => $this->t('Memory limit'),
      '#description' => $this->t('Maximum memory PHPStan can use during analysis. Increase this if you get "out of memory" errors. <strong>Recommended: 512M</strong> for most projects, 1G for large codebases.'),
      '#options' => [
        '256M' => '256 MB',
        '512M' => '512 MB (' . $this->t('recommended') . ')',
        '1G' => '1 GB',
        '2G' => '2 GB',
      ],
      '#default_value' => $config['memory_limit'] ?? '512M',
    ];

    $form['basic']['max_errors_display'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum errors to display'),
      '#description' => $this->t('Limit the number of errors shown in the detailed results. Set a lower value for faster page loading if you have many errors.'),
      '#default_value' => $config['max_errors_display'] ?? self::DEFAULT_MAX_ERRORS,
      '#min' => 10,
      '#max' => 2000,
    ];

    // Ignore options - checkboxes to DISABLE recommended features.
    $form['ignore_options'] = [
      '#type' => 'details',
      '#title' => $this->t('Ignore Options'),
      '#description' => $this->t('By default, all recommended analysis features are enabled. Check the options below to <strong>disable</strong> specific features.'),
      '#open' => FALSE,
    ];

    $form['ignore_options']['skip_deprecations'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Skip deprecation checks'),
      '#description' => $this->t('Check this to <strong>ignore</strong> deprecated code detection. <em>Not recommended</em> - deprecation checks help prepare your code for Drupal 11.'),
      '#default_value' => $config['skip_deprecations'] ?? FALSE,
    ];

    $form['ignore_options']['skip_phpdoc_types'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Skip PHPDoc type inference'),
      '#description' => $this->t('Check this to <strong>ignore</strong> @var, @param, and @return annotations. <em>Not recommended</em> - Drupal APIs rely heavily on PHPDoc for type information.'),
      '#default_value' => $config['skip_phpdoc_types'] ?? FALSE,
    ];

    $form['ignore_options']['skip_entity_mapping'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Skip entity type mapping'),
      '#description' => $this->t('Check this to <strong>disable</strong> Drupal entity type mapping. <em>Not recommended</em> - entity mapping reduces false positives when working with entities.'),
      '#default_value' => $config['skip_entity_mapping'] ?? FALSE,
    ];

    // Entity mapping information (auto-detected, read-only display).
    $form['entity_mapping'] = [
      '#type' => 'details',
      '#title' => $this->t('Entity Type Mapping (Auto-detected)'),
      '#description' => $this->t('Entity type mappings are automatically detected based on installed modules. PHPStan will use these mappings to better understand your Drupal code.'),
      '#open' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name="audit_phpstan[ignore_options][skip_entity_mapping]"]' => ['checked' => FALSE],
        ],
      ],
    ];

    // Detect and display available entity types.
    $entity_type_info = $this->detectAvailableEntityTypes();
    $active_types = [];
    $inactive_types = [];

    foreach ($entity_type_info['details'] as $type_id => $info) {
      if ($info['active']) {
        $active_types[$type_id] = $info;
      }
      else {
        $inactive_types[$type_id] = $info;
      }
    }

    // Build active entity types table.
    $active_rows = [];
    foreach ($active_types as $type_id => $info) {
      $active_rows[] = [
        'data' => [
          ['data' => $type_id, 'class' => ['audit-entity-type-id']],
          ['data' => $info['label']],
          ['data' => $info['module']],
          ['data' => ['#markup' => '<span class="color-success">✓ ' . $this->t('Active') . '</span>']],
        ],
      ];
    }

    if (!empty($active_rows)) {
      $form['entity_mapping']['active_types'] = [
        '#type' => 'table',
        '#caption' => $this->t('Active Entity Type Mappings'),
        '#header' => [
          $this->t('Entity Type'),
          $this->t('Label'),
          $this->t('Module'),
          $this->t('Status'),
        ],
        '#rows' => $active_rows,
        '#empty' => $this->t('No active entity types detected.'),
        '#attributes' => ['class' => ['audit-entity-mapping-table']],
      ];
    }

    // Build inactive entity types info.
    if (!empty($inactive_types)) {
      $inactive_list = [];
      foreach ($inactive_types as $type_id => $info) {
        $inactive_list[] = $info['label'] . ' (' . $info['module'] . ')';
      }

      $form['entity_mapping']['inactive_info'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['audit-inactive-types']],
        'label' => [
          '#markup' => '<p><strong>' . $this->t('Not installed:') . '</strong> ' .
            implode(', ', $inactive_list) . '</p>',
        ],
        'note' => [
          '#markup' => '<p class="description">' .
            $this->t('These entity types will be automatically mapped when their modules are installed.') .
            '</p>',
        ],
      ];
    }

    // Custom entity mappings for entities not in the known list.
    $form['entity_mapping']['custom_entity_mapping'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom entity mappings'),
      '#description' => $this->t('Add mappings for custom entity types not listed above. Format: one mapping per line as <code>entity_type_id: Full\\Class\\Name</code>. Example:<br><code>my_custom_entity: Drupal\\my_module\\Entity\\MyCustomEntity</code>'),
      '#default_value' => $config['custom_entity_mapping'] ?? '',
      '#rows' => 3,
    ];

    // Advanced settings.
    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#open' => FALSE,
    ];

    $form['advanced']['report_unmatched_ignored_errors'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Report unmatched ignored errors'),
      '#description' => $this->t('When enabled, PHPStan will report if any ignored error patterns no longer match any errors. Useful to keep your ignored errors list clean.'),
      '#default_value' => $config['report_unmatched_ignored_errors'] ?? FALSE,
    ];

    $form['advanced']['ignore_errors'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Error patterns to ignore'),
      '#description' => $this->t('Patterns of errors to ignore (one per line). Use regular expressions. Example:<br><code>#Access to an undefined property Drupal\\\\Core\\\\Entity\\\\EntityInterface::\\$field_name#</code><br>This is useful for known false positives or errors you plan to fix later.'),
      '#default_value' => $config['ignore_errors'] ?? '',
      '#rows' => 4,
    ];

    $form['advanced']['excluded_paths'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Paths to exclude from analysis'),
      '#description' => $this->t('Paths to exclude from analysis (one per line, relative to Drupal root). Example:<br><code>web/modules/custom/my_module/tests</code><br><code>web/modules/custom/legacy_module</code>'),
      '#default_value' => $config['excluded_paths'] ?? '',
      '#rows' => 3,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function checkRequirements(): array {
    $warnings = [];

    $phpstan_path = $this->findBinary('phpstan');

    // Don't show simple warning if PHPStan is not installed.
    // The detailed results page will show comprehensive installation
    // instructions instead.
    if (!$phpstan_path) {
      return $warnings;
    }

    // Only check for extensions if PHPStan is installed.
    $extensions = $this->detectInstalledExtensions();
    $missing = $extensions['missing'] ?? [];

    // Warn about missing recommended packages.
    if (!empty($missing)) {
      $missing_list = implode(', ', array_keys($missing));
      $warnings[] = (string) $this->t('Missing recommended packages: @packages. See the PHPStan audit results for installation instructions.', [
        '@packages' => $missing_list,
      ]);
    }

    return $warnings;
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    $phpstan_path = $this->findBinary('phpstan');

    if (!$phpstan_path) {
      return $this->createNotInstalledResult();
    }

    // Get configuration.
    $audit_config = $this->configFactory->get('audit.settings');
    $phpstan_config = $this->configFactory->get('audit_phpstan.settings');

    // Get paths from global audit configuration.
    $paths_to_analyze = $this->getScanDirectories($audit_config);
    $exclude_patterns = $this->getExcludePatterns($audit_config);

    if (empty($paths_to_analyze)) {
      return $this->createNoCodeResult();
    }

    // Build full configuration array from form settings.
    $config = $this->buildPhpstanConfig($phpstan_config, $paths_to_analyze, $exclude_patterns);

    // Run PHPStan analysis.
    $analysis_result = $this->runPhpstan($phpstan_path, $config);

    // Build all result items.
    $all_results = $this->buildResultItems($analysis_result);

    // Build separate files for each issue type (like ViewsAnalyzer pattern).
    $type_errors_file = $this->buildTypeErrorsFile($all_results);
    $deprecation_errors_file = $this->buildDeprecationErrorsFile($all_results);
    $undefined_errors_file = $this->buildUndefinedErrorsFile($all_results);

    // Build statistics with categorized counts.
    $stats = $this->buildStats($analysis_result, $paths_to_analyze, $config['level'], $config['max_errors_display']);
    $stats['type_errors_count'] = $type_errors_file['summary']['errors'] ?? 0;
    $stats['deprecation_count'] = $deprecation_errors_file['summary']['warnings'] ?? 0;
    $stats['undefined_count'] = $undefined_errors_file['summary']['errors'] ?? 0;

    // Calculate scores using categorized data.
    $scores = $this->calculateScores($stats);

    // Build environment info for the informational section.
    $environment_data = [
      'extensions' => $stats['environment']['extensions'] ?? [],
      'drupal_rector' => $stats['environment']['drupal_rector'] ?? NULL,
      'config' => $stats['config'] ?? [],
      'paths_analyzed' => $stats['paths_analyzed'] ?? [],
      'files_analyzed' => $stats['files_analyzed'] ?? 0,
    ];

    // Return structured output with _files for each section (ViewsAnalyzer pattern).
    return [
      '_files' => [
        // All errors combined (for reference).
        'all_errors' => $this->createResult(
          $all_results,
          $stats['total_errors'],
          0,
          $stats['total_errors'] === 0 ? 1 : 0
        ),
        // Scoring section: Type errors.
        'type_errors' => $type_errors_file,
        // Scoring section: Deprecations.
        'deprecation_errors' => $deprecation_errors_file,
        // Scoring section: Undefined issues.
        'undefined_errors' => $undefined_errors_file,
        // Informational section: Environment.
        'environment' => [
          'summary' => ['errors' => 0, 'warnings' => 0, 'notices' => 0],
          'results' => [],
          'data' => $environment_data,
        ],
      ],
      'score' => $scores,
      'stats' => $stats,
    ];
  }

  /**
   * Builds the type errors file (type mismatches, wrong arguments).
   *
   * @param array $all_results
   *   All result items from analysis.
   *
   * @return array
   *   Filtered results for type errors.
   */
  protected function buildTypeErrorsFile(array $all_results): array {
    $results = [];
    $errors = 0;

    foreach ($all_results as $item) {
      $message = strtolower($item['details']['message'] ?? $item['message'] ?? '');
      $identifier = strtolower($item['details']['identifier'] ?? '');

      // Type errors: type mismatches, argument type issues, return type issues.
      $is_type_error = str_contains($message, 'type') ||
        str_contains($message, 'expects') ||
        str_contains($message, 'return') ||
        str_contains($identifier, 'argument') ||
        str_contains($identifier, 'return') ||
        str_contains($identifier, 'parameter') ||
        str_contains($identifier, 'missingType');

      // Exclude deprecations and undefined issues.
      $is_deprecation = str_contains($message, 'deprecat') || str_contains($identifier, 'deprecat');
      $is_undefined = str_contains($message, 'undefined') || str_contains($message, 'unknown');

      if ($is_type_error && !$is_deprecation && !$is_undefined) {
        $results[] = $item;
        $errors++;
      }
    }

    return $this->createResult($results, $errors, 0, 0);
  }

  /**
   * Builds the deprecation errors file.
   *
   * @param array $all_results
   *   All result items from analysis.
   *
   * @return array
   *   Filtered results for deprecation warnings.
   */
  protected function buildDeprecationErrorsFile(array $all_results): array {
    $results = [];
    $warnings = 0;

    foreach ($all_results as $item) {
      $message = strtolower($item['details']['message'] ?? $item['message'] ?? '');
      $identifier = strtolower($item['details']['identifier'] ?? '');

      // Deprecation issues.
      if (str_contains($message, 'deprecat') || str_contains($identifier, 'deprecat')) {
        // Change severity to warning for deprecations.
        $item['severity'] = 'warning';
        $results[] = $item;
        $warnings++;
      }
    }

    return $this->createResult($results, 0, $warnings, 0);
  }

  /**
   * Builds the undefined errors file (undefined methods, properties, variables).
   *
   * @param array $all_results
   *   All result items from analysis.
   *
   * @return array
   *   Filtered results for undefined issues.
   */
  protected function buildUndefinedErrorsFile(array $all_results): array {
    $results = [];
    $errors = 0;

    foreach ($all_results as $item) {
      $message = strtolower($item['details']['message'] ?? $item['message'] ?? '');
      $identifier = strtolower($item['details']['identifier'] ?? '');

      // Exclude deprecations first.
      $is_deprecation = str_contains($message, 'deprecat') || str_contains($identifier, 'deprecat');
      if ($is_deprecation) {
        continue;
      }

      // Undefined issues: undefined methods, properties, variables, classes.
      $is_undefined = str_contains($message, 'undefined') ||
        str_contains($message, 'unknown') ||
        str_contains($message, 'does not exist') ||
        str_contains($message, 'not found') ||
        str_contains($identifier, 'undefined') ||
        str_contains($identifier, 'class.notFound');

      if ($is_undefined) {
        $results[] = $item;
        $errors++;
      }
    }

    return $this->createResult($results, $errors, 0, 0);
  }

  /**
   * Builds PHPStan configuration from module settings.
   *
   * @param \Drupal\Core\Config\ImmutableConfig $phpstan_config
   *   The PHPStan module configuration.
   * @param array $paths
   *   Paths to analyze.
   * @param array $exclude_patterns
   *   Global exclude patterns from audit.settings.
   *
   * @return array
   *   Configuration array for PHPStan.
   */
  protected function buildPhpstanConfig($phpstan_config, array $paths, array $exclude_patterns = []): array {
    // Inverted logic: skip_* = TRUE means feature is DISABLED.
    // By default all skip_* are FALSE, so features are ENABLED.
    $skip_deprecations = $phpstan_config->get('skip_deprecations') ?? FALSE;
    $skip_phpdoc_types = $phpstan_config->get('skip_phpdoc_types') ?? FALSE;
    $skip_entity_mapping = $phpstan_config->get('skip_entity_mapping') ?? FALSE;

    $config = [
      'level' => $phpstan_config->get('level') ?? self::DEFAULT_LEVEL,
      'memory_limit' => $phpstan_config->get('memory_limit') ?? '512M',
      'max_errors_display' => $phpstan_config->get('max_errors_display') ?? self::DEFAULT_MAX_ERRORS,
      'paths' => $paths,
      // Features enabled by default, disabled if skip_* is TRUE.
      'check_deprecations' => !$skip_deprecations,
      'treat_phpdoc_types_as_certain' => !$skip_phpdoc_types,
      'report_unmatched_ignored_errors' => $phpstan_config->get('report_unmatched_ignored_errors') ?? FALSE,
      'ignore_errors' => [],
      'excluded_paths' => [],
      'exclude_patterns' => $exclude_patterns,
      'entity_mapping' => [],
    ];

    // Parse ignore errors.
    $ignore_errors_raw = $phpstan_config->get('ignore_errors') ?? '';
    if (!empty($ignore_errors_raw)) {
      $config['ignore_errors'] = array_filter(array_map('trim', explode("\n", $ignore_errors_raw)));
    }

    // Parse additional excluded paths from PHPStan-specific config.
    $excluded_paths_raw = $phpstan_config->get('excluded_paths') ?? '';
    if (!empty($excluded_paths_raw)) {
      $config['excluded_paths'] = array_filter(array_map('trim', explode("\n", $excluded_paths_raw)));
    }

    // Build entity mapping only if not skipped.
    if (!$skip_entity_mapping) {
      // Auto-detect entity types from installed modules.
      $entity_type_info = $this->detectAvailableEntityTypes();
      $config['entity_mapping'] = $entity_type_info['mappings'];

      // Parse custom entity mappings (for custom entity types not auto-detected).
      $custom_mappings_raw = $phpstan_config->get('custom_entity_mapping') ?? '';
      if (!empty($custom_mappings_raw)) {
        $lines = array_filter(array_map('trim', explode("\n", $custom_mappings_raw)));
        foreach ($lines as $line) {
          if (str_contains($line, ':')) {
            [$type, $class] = array_map('trim', explode(':', $line, 2));
            if (!empty($type) && !empty($class)) {
              $config['entity_mapping'][$type] = $class;
            }
          }
        }
      }
    }

    return $config;
  }

  /**
   * Creates result when PHPStan is not installed.
   *
   * @return array
   *   Result array.
   */
  protected function createNotInstalledResult(): array {
    $results = [];
    $results[] = $this->createResultItem(
      'error',
      'PHPSTAN_NOT_INSTALLED',
      (string) $this->t('PHPStan is not installed'),
      ['installation_instructions' => $this->getInstallationInstructions()]
    );

    return [
      '_files' => [
        'errors' => $this->createResult($results, 1, 0, 0),
        'environment' => [
          'summary' => ['errors' => 0, 'warnings' => 0, 'notices' => 0],
          'results' => [],
          'data' => [
            'extensions' => ['installed' => [], 'missing' => self::REQUIRED_PACKAGES],
            'installation_instructions' => $this->getInstallationInstructions(),
          ],
        ],
      ],
      'score' => $this->createEmptyScores((string) $this->t('PHPStan not installed')),
      'stats' => [
        'available' => FALSE,
        'installation_instructions' => $this->getInstallationInstructions(),
      ],
    ];
  }

  /**
   * Creates result when no code is found.
   *
   * @return array
   *   Result array.
   */
  protected function createNoCodeResult(): array {
    $results = [];
    $results[] = $this->createResultItem(
      'notice',
      'NO_CUSTOM_CODE',
      (string) $this->t('No custom modules or themes found to analyze'),
      []
    );

    return [
      '_files' => [
        'errors' => $this->createResult($results, 0, 0, 1),
        'environment' => [
          'summary' => ['errors' => 0, 'warnings' => 0, 'notices' => 0],
          'results' => [],
          'data' => [],
        ],
      ],
      'score' => $this->createPerfectScores(),
      'stats' => ['available' => TRUE, 'no_code' => TRUE],
    ];
  }

  /**
   * Runs PHPStan analysis.
   *
   * @param string $phpstan_path
   *   Path to PHPStan binary.
   * @param array $config
   *   Configuration array from buildPhpstanConfig().
   *
   * @return array
   *   Analysis result with errors grouped by file.
   */
  protected function runPhpstan(string $phpstan_path, array $config): array {
    $errors_by_file = [];
    $total_errors = 0;

    // Generate temporary NEON configuration file.
    $neon_content = $this->generateNeonConfig($config);
    $temp_neon_path = sys_get_temp_dir() . '/audit_phpstan_' . uniqid() . '.neon';

    // Log the generated NEON for debugging.
    \Drupal::logger('audit_phpstan')->debug('PHPStan NEON config generated: @config', ['@config' => $neon_content]);

    try {
      file_put_contents($temp_neon_path, $neon_content);

      $command = [
        $phpstan_path,
        'analyse',
        '--configuration=' . $temp_neon_path,
        '--error-format=json',
        '--no-progress',
        '--memory-limit=' . $config['memory_limit'],
      ];

      \Drupal::logger('audit_phpstan')->debug('PHPStan command: @cmd', ['@cmd' => implode(' ', $command)]);

      $process = new Process($command);
      $process->setTimeout(600);
      $process->run();

      // PHPStan returns exit code 1 when errors are found, but still outputs JSON.
      $output = $process->getOutput();
      $error_output = $process->getErrorOutput();

      \Drupal::logger('audit_phpstan')->debug('PHPStan exit code: @code', ['@code' => $process->getExitCode()]);
      \Drupal::logger('audit_phpstan')->debug('PHPStan stdout length: @len', ['@len' => strlen($output)]);
      \Drupal::logger('audit_phpstan')->debug('PHPStan stderr: @err', ['@err' => substr($error_output, 0, 500)]);

      if (empty($output)) {
        $output = $error_output;
      }

      $this->parsePhpstanOutput($output, $errors_by_file, $total_errors);
    }
    catch (\Exception $e) {
      \Drupal::logger('audit_phpstan')->error('PHPStan exception: @msg', ['@msg' => $e->getMessage()]);
      // Try to parse any JSON output from the error message.
      $error_output = $e->getMessage();
      if (str_contains($error_output, '{')) {
        $json_start = strpos($error_output, '{');
        $json_output = substr($error_output, $json_start);
        $this->parsePhpstanOutput($json_output, $errors_by_file, $total_errors);
      }
    }
    finally {
      // Clean up temporary file.
      if (file_exists($temp_neon_path)) {
        @unlink($temp_neon_path);
      }
    }

    return [
      'errors_by_file' => $errors_by_file,
      'total_errors' => $total_errors,
    ];
  }

  /**
   * Generates NEON configuration content from config array.
   *
   * @param array $config
   *   Configuration array.
   *
   * @return string
   *   NEON configuration content.
   */
  protected function generateNeonConfig(array $config): string {
    $neon = "parameters:\n";
    $neon .= "    level: " . $config['level'] . "\n";

    // Paths to analyze.
    $neon .= "    paths:\n";
    foreach ($config['paths'] as $path) {
      $neon .= "        - " . $path . "\n";
    }

    // Build exclude paths combining global patterns and PHPStan-specific paths.
    $exclude_items = [];

    // PHPStan-specific excluded paths (full paths) - only if they exist.
    if (!empty($config['excluded_paths'])) {
      foreach ($config['excluded_paths'] as $path) {
        $full_path = DRUPAL_ROOT . '/' . ltrim($path, '/');
        if (is_dir($full_path) || is_file($full_path)) {
          $exclude_items[] = $full_path;
        }
      }
    }

    // Global exclude patterns from audit.settings.
    // These are patterns like *Test.php, tests/, etc.
    if (!empty($config['exclude_patterns'])) {
      foreach ($config['exclude_patterns'] as $pattern) {
        // Directory patterns (ending with /) - only add if directory exists.
        if (str_ends_with($pattern, '/')) {
          foreach ($config['paths'] as $base_path) {
            $dir_path = $base_path . '/' . rtrim($pattern, '/');
            if (is_dir($dir_path)) {
              $exclude_items[] = $dir_path;
            }
          }
        }
        else {
          // File patterns (like *Test.php) - use glob pattern for all paths.
          foreach ($config['paths'] as $base_path) {
            $exclude_items[] = $base_path . '/**/' . $pattern;
          }
        }
      }
    }

    // Write excludePaths if we have any.
    if (!empty($exclude_items)) {
      $neon .= "    excludePaths:\n";
      foreach ($exclude_items as $path) {
        $neon .= "        - " . $path . "\n";
      }
    }

    // PHPDoc settings.
    if ($config['treat_phpdoc_types_as_certain']) {
      $neon .= "    treatPhpDocTypesAsCertain: true\n";
    }

    // Report unmatched ignored errors.
    $neon .= "    reportUnmatchedIgnoredErrors: " . ($config['report_unmatched_ignored_errors'] ? 'true' : 'false') . "\n";

    // Ignore errors.
    if (!empty($config['ignore_errors'])) {
      $neon .= "    ignoreErrors:\n";
      foreach ($config['ignore_errors'] as $pattern) {
        $neon .= "        - '" . addcslashes($pattern, "'") . "'\n";
      }
    }

    // Drupal extension configuration.
    // Note: phpstan-drupal 2.x expects entityMapping with 'class' subkey.
    // drupal_root is auto-discovered since phpstan-drupal 2.x, so we don't set it.
    if (!empty($config['entity_mapping'])) {
      $neon .= "    drupal:\n";
      $neon .= "        entityMapping:\n";
      foreach ($config['entity_mapping'] as $type => $class) {
        $neon .= "            " . $type . ":\n";
        $neon .= "                class: " . $class . "\n";
      }
    }

    return $neon;
  }

  /**
   * Parses PHPStan JSON output.
   *
   * @param string $output
   *   JSON output from PHPStan.
   * @param array &$errors_by_file
   *   Errors grouped by file (passed by reference).
   * @param int &$total_errors
   *   Total error count (passed by reference).
   */
  protected function parsePhpstanOutput(string $output, array &$errors_by_file, int &$total_errors): void {
    $phpstan_results = json_decode($output, TRUE);

    if (!$phpstan_results || !isset($phpstan_results['files'])) {
      return;
    }

    foreach ($phpstan_results['files'] as $file => $file_data) {
      $relative_file = str_replace(DRUPAL_ROOT . '/', '', $file);

      if (!isset($errors_by_file[$relative_file])) {
        $errors_by_file[$relative_file] = [];
      }

      foreach ($file_data['messages'] as $message) {
        $errors_by_file[$relative_file][] = [
          'line' => $message['line'] ?? 0,
          'message' => $message['message'] ?? '',
          'identifier' => $message['identifier'] ?? NULL,
          'tip' => $message['tip'] ?? NULL,
        ];
        $total_errors++;
      }
    }
  }

  /**
   * Builds statistics array.
   *
   * @param array $analysis_result
   *   Analysis result.
   * @param array $paths_analyzed
   *   Paths that were analyzed.
   * @param int $level
   *   Analysis level used.
   * @param int $max_display
   *   Max errors to display.
   *
   * @return array
   *   Statistics array.
   */
  protected function buildStats(array $analysis_result, array $paths_analyzed, int $level, int $max_display): array {
    $files_analyzed = $this->countFilesInPaths($paths_analyzed);

    // Detect environment info.
    $extensions = $this->detectInstalledExtensions();
    $drupal_check = $this->detectDrupalCheck();
    $drupal_rector = $this->detectDrupalRector();

    // Count deprecation-related errors.
    $deprecation_count = 0;
    foreach ($analysis_result['errors_by_file'] as $file_errors) {
      foreach ($file_errors as $error) {
        $message = strtolower($error['message'] ?? '');
        $identifier = strtolower($error['identifier'] ?? '');
        if (str_contains($message, 'deprecat') ||
            str_contains($identifier, 'deprecat')) {
          $deprecation_count++;
        }
      }
    }

    return [
      'available' => TRUE,
      'total_errors' => $analysis_result['total_errors'],
      'deprecation_count' => $deprecation_count,
      'files_with_errors' => count($analysis_result['errors_by_file']),
      'files_analyzed' => $files_analyzed,
      'paths_analyzed' => $paths_analyzed,
      'errors_by_file' => $analysis_result['errors_by_file'],
      'config' => [
        'level' => $level,
        'level_description' => self::LEVEL_DESCRIPTIONS[$level] ?? '',
        'max_errors_display' => $max_display,
      ],
      'environment' => [
        'extensions' => $extensions,
        'drupal_check' => $drupal_check,
        'drupal_rector' => $drupal_rector,
      ],
    ];
  }

  /**
   * Builds result items from analysis.
   *
   * @param array $analysis_result
   *   Analysis result.
   *
   * @return array
   *   Array of result items.
   */
  protected function buildResultItems(array $analysis_result): array {
    $results = [];

    if (empty($analysis_result['errors_by_file'])) {
      $results[] = $this->createResultItem(
        'notice',
        'PHPSTAN_NO_ERRORS',
        (string) $this->t('PHPStan analysis completed with no errors'),
        []
      );
      return $results;
    }

    foreach ($analysis_result['errors_by_file'] as $file => $errors) {
      foreach ($errors as $error) {
        $results[] = $this->createResultItem(
          'error',
          'PHPSTAN_ERROR',
          $this->truncateMessage($error['message'], 200),
          [
            'file' => $file,
            'line' => $error['line'],
            'message' => $error['message'],
            'identifier' => $error['identifier'],
            'tip' => $error['tip'],
            'code_context' => $this->extractCodeContext($file, $error['line']),
          ]
        );
      }
    }

    return $results;
  }

  /**
   * Extracts code context around a line.
   *
   * @param string $file
   *   Relative file path.
   * @param int $line
   *   Line number.
   * @param int $context_lines
   *   Number of lines before/after.
   *
   * @return array
   *   Code context with lines and highlight info.
   */
  protected function extractCodeContext(string $file, int $line, int $context_lines = 2): array {
    $full_path = DRUPAL_ROOT . '/' . $file;

    if (!file_exists($full_path) || !is_readable($full_path)) {
      return [];
    }

    try {
      $file_contents = file($full_path);
      if ($file_contents === FALSE) {
        return [];
      }

      $start_line = max(1, $line - $context_lines);
      $end_line = min(count($file_contents), $line + $context_lines);

      $lines = [];
      for ($i = $start_line; $i <= $end_line; $i++) {
        $lines[] = [
          'number' => $i,
          'content' => rtrim($file_contents[$i - 1] ?? ''),
          'is_highlight' => ($i === $line),
        ];
      }

      return [
        'lines' => $lines,
        'highlight_line' => $line,
      ];
    }
    catch (\Exception $e) {
      return [];
    }
  }

  /**
   * Truncates a message to a maximum length.
   *
   * @param string $message
   *   The message to truncate.
   * @param int $length
   *   Maximum length.
   *
   * @return string
   *   Truncated message.
   */
  protected function truncateMessage(string $message, int $length = 200): string {
    if (strlen($message) <= $length) {
      return $message;
    }
    return substr($message, 0, $length - 3) . '...';
  }

  /**
   * Gets fix suggestion for type errors.
   *
   * @param string $message
   *   The error message.
   * @param string $identifier
   *   The PHPStan rule identifier.
   *
   * @return string
   *   HTML fix suggestion.
   */
  protected function getTypeErrorFix(string $message, string $identifier): string {
    $message_lower = strtolower($message);

    if (str_contains($message_lower, 'expects') && str_contains($message_lower, 'given')) {
      return (string) $this->t('Check the argument type being passed. The function/method expects a different type than what is being provided. Add type casting or fix the value source.');
    }

    if (str_contains($message_lower, 'return type')) {
      return (string) $this->t('The method return type declaration does not match what is actually being returned. Either fix the return statement or update the return type hint.');
    }

    if (str_contains($message_lower, 'cannot be null')) {
      return (string) $this->t('A nullable value is being used where a non-null value is expected. Add a null check before using the value, or ensure the source never returns null.');
    }

    if (str_contains($message_lower, 'iterable')) {
      return (string) $this->t('The value is expected to be iterable (array/Traversable) but may not be. Ensure the value is an array or implement proper type checking.');
    }

    if (str_contains($identifier, 'missingType')) {
      return (string) $this->t('Add proper type hints to the function/method parameters and return types. This helps catch errors early and improves code documentation.');
    }

    return (string) $this->t('Review the types being used and ensure they match the expected signatures. Consider adding PHPDoc annotations or type hints for better type inference.');
  }

  /**
   * Gets tags for type errors based on message content.
   *
   * @param string $message
   *   The error message.
   * @param string $identifier
   *   The PHPStan rule identifier.
   *
   * @return array
   *   Array of tags.
   */
  protected function getTypeErrorTags(string $message, string $identifier): array {
    $tags = ['phpstan', 'type-error'];
    $message_lower = strtolower($message);

    if (str_contains($message_lower, 'return')) {
      $tags[] = 'return-type';
    }
    if (str_contains($message_lower, 'argument') || str_contains($message_lower, 'parameter')) {
      $tags[] = 'argument';
    }
    if (str_contains($message_lower, 'null')) {
      $tags[] = 'nullable';
    }
    if (str_contains($identifier, 'missingType')) {
      $tags[] = 'missing-type';
    }

    return $tags;
  }

  /**
   * Gets fix suggestion for deprecation warnings.
   *
   * @param string $message
   *   The deprecation message.
   * @param string $tip
   *   The PHPStan tip if available.
   *
   * @return string
   *   HTML fix suggestion.
   */
  protected function getDeprecationFix(string $message, string $tip): string {
    // If PHPStan provides a tip, it usually contains the replacement.
    if (!empty($tip)) {
      return (string) $this->t('Replace the deprecated code as suggested. The tip below contains the recommended replacement.');
    }

    $message_lower = strtolower($message);

    if (str_contains($message_lower, 'service')) {
      return (string) $this->t('This service is deprecated. Check the Drupal change records for the replacement service and update your dependency injection.');
    }

    if (str_contains($message_lower, 'function')) {
      return (string) $this->t('This function is deprecated. Look for a replacement in the Drupal API documentation or change records.');
    }

    if (str_contains($message_lower, 'method')) {
      return (string) $this->t('This method is deprecated. Check the class documentation for the replacement method.');
    }

    if (str_contains($message_lower, 'class')) {
      return (string) $this->t('This class is deprecated. Find and use the replacement class as documented in the Drupal change records.');
    }

    if (str_contains($message_lower, 'constant')) {
      return (string) $this->t('This constant is deprecated. Use the replacement constant as specified in the deprecation notice.');
    }

    return (string) $this->t('This code uses deprecated APIs. Search for "@deprecated" in the source code or check <a href="https://www.drupal.org/list-changes" target="_blank">Drupal change records</a> for the recommended replacement.');
  }

  /**
   * Gets tags for deprecation warnings based on message content.
   *
   * @param string $message
   *   The deprecation message.
   *
   * @return array
   *   Array of tags.
   */
  protected function getDeprecationTags(string $message): array {
    $tags = ['phpstan', 'deprecation', 'drupal11'];
    $message_lower = strtolower($message);

    if (str_contains($message_lower, 'service')) {
      $tags[] = 'service';
    }
    if (str_contains($message_lower, 'function')) {
      $tags[] = 'function';
    }
    if (str_contains($message_lower, 'method')) {
      $tags[] = 'method';
    }
    if (str_contains($message_lower, 'class')) {
      $tags[] = 'class';
    }
    if (str_contains($message_lower, 'hook')) {
      $tags[] = 'hook';
    }

    return $tags;
  }

  /**
   * Gets fix suggestion for undefined reference errors.
   *
   * @param string $message
   *   The error message.
   * @param string $identifier
   *   The PHPStan rule identifier.
   *
   * @return string
   *   HTML fix suggestion.
   */
  protected function getUndefinedFix(string $message, string $identifier): string {
    $message_lower = strtolower($message);

    if (str_contains($message_lower, 'undefined method')) {
      return (string) $this->t('The method does not exist on this class. Check for typos in the method name, ensure the correct class is being used, or add the missing method.');
    }

    if (str_contains($message_lower, 'undefined variable')) {
      return (string) $this->t('This variable is used before being defined. Initialize the variable before using it, or check for typos in the variable name.');
    }

    if (str_contains($message_lower, 'undefined property')) {
      return (string) $this->t('This property does not exist on the class. Define the property, check for typos, or verify you are using the correct class.');
    }

    if (str_contains($message_lower, 'class') && str_contains($message_lower, 'not found')) {
      return (string) $this->t('The class could not be found. Check for typos, ensure the correct namespace, or verify the class file exists and is autoloaded.');
    }

    if (str_contains($message_lower, 'function') && str_contains($message_lower, 'not found')) {
      return (string) $this->t('The function does not exist. Check for typos in the function name or ensure the module/library providing it is enabled.');
    }

    if (str_contains($message_lower, 'constant')) {
      return (string) $this->t('This constant is not defined. Check for typos or ensure the module defining it is enabled.');
    }

    return (string) $this->t('The referenced code element does not exist. Check for typos, verify the namespace, and ensure all dependencies are properly installed.');
  }

  /**
   * Gets a generic/grouped label for type errors.
   *
   * @param string $message
   *   The error message.
   * @param string $identifier
   *   The PHPStan rule identifier.
   *
   * @return string
   *   Generic label for grouping in facets.
   */
  protected function getTypeErrorLabel(string $message, string $identifier): string {
    $message_lower = strtolower($message);
    $identifier_lower = strtolower($identifier);

    // Argument/parameter type mismatch.
    if ((str_contains($message_lower, 'expects') && str_contains($message_lower, 'given')) ||
        str_contains($identifier_lower, 'argument')) {
      return (string) $this->t('Argument type mismatch');
    }

    // Return type issues.
    if (str_contains($message_lower, 'return type') ||
        str_contains($message_lower, 'should return') ||
        str_contains($identifier_lower, 'return')) {
      return (string) $this->t('Return type mismatch');
    }

    // Null/nullable issues.
    if (str_contains($message_lower, 'cannot be null') ||
        str_contains($message_lower, 'null given') ||
        str_contains($message_lower, 'might be null')) {
      return (string) $this->t('Nullable type issue');
    }

    // Missing type hints.
    if (str_contains($identifier_lower, 'missingtype')) {
      return (string) $this->t('Missing type hint');
    }

    // Property type issues.
    if (str_contains($message_lower, 'property') && str_contains($message_lower, 'type')) {
      return (string) $this->t('Property type mismatch');
    }

    // Array/iterable issues.
    if (str_contains($message_lower, 'array') || str_contains($message_lower, 'iterable')) {
      return (string) $this->t('Array/iterable type issue');
    }

    // Generic type error.
    return (string) $this->t('Type error');
  }

  /**
   * Gets a generic/grouped label for deprecation warnings.
   *
   * @param string $message
   *   The deprecation message.
   *
   * @return string
   *   Generic label for grouping in facets.
   */
  protected function getDeprecationLabel(string $message): string {
    $message_lower = strtolower($message);

    // Service deprecation.
    if (str_contains($message_lower, 'service')) {
      return (string) $this->t('Deprecated service');
    }

    // Hook deprecation.
    if (str_contains($message_lower, 'hook_')) {
      return (string) $this->t('Deprecated hook');
    }

    // Function deprecation.
    if (str_contains($message_lower, 'function') && !str_contains($message_lower, 'method')) {
      return (string) $this->t('Deprecated function');
    }

    // Method deprecation.
    if (str_contains($message_lower, 'method') || str_contains($message_lower, '::')) {
      return (string) $this->t('Deprecated method');
    }

    // Class deprecation.
    if (str_contains($message_lower, 'class') || str_contains($message_lower, 'interface')) {
      return (string) $this->t('Deprecated class/interface');
    }

    // Constant deprecation.
    if (str_contains($message_lower, 'constant')) {
      return (string) $this->t('Deprecated constant');
    }

    // Property deprecation.
    if (str_contains($message_lower, 'property')) {
      return (string) $this->t('Deprecated property');
    }

    // Generic deprecation.
    return (string) $this->t('Deprecated code');
  }

  /**
   * Gets a generic/grouped label for undefined reference errors.
   *
   * @param string $message
   *   The error message.
   *
   * @return string
   *   Generic label for grouping in facets.
   */
  protected function getUndefinedLabel(string $message): string {
    $message_lower = strtolower($message);

    // Undefined method.
    if (str_contains($message_lower, 'undefined method') ||
        str_contains($message_lower, 'call to undefined method')) {
      return (string) $this->t('Undefined method');
    }

    // Undefined variable.
    if (str_contains($message_lower, 'undefined variable') ||
        str_contains($message_lower, 'variable $')) {
      return (string) $this->t('Undefined variable');
    }

    // Undefined property.
    if (str_contains($message_lower, 'undefined property') ||
        str_contains($message_lower, 'access to an undefined property')) {
      return (string) $this->t('Undefined property');
    }

    // Class not found.
    if (str_contains($message_lower, 'class') && str_contains($message_lower, 'not found')) {
      return (string) $this->t('Class not found');
    }

    // Function not found.
    if (str_contains($message_lower, 'function') && str_contains($message_lower, 'not found')) {
      return (string) $this->t('Function not found');
    }

    // Constant not found.
    if (str_contains($message_lower, 'constant') && str_contains($message_lower, 'not found')) {
      return (string) $this->t('Constant not found');
    }

    // Unknown type/class.
    if (str_contains($message_lower, 'unknown')) {
      return (string) $this->t('Unknown reference');
    }

    // Generic undefined.
    return (string) $this->t('Undefined reference');
  }

  /**
   * Counts files in paths.
   *
   * @param array $paths
   *   Paths to count.
   *
   * @return int
   *   File count.
   */
  protected function countFilesInPaths(array $paths): int {
    $count = 0;
    $extensions = ['php', 'module', 'inc', 'install', 'theme', 'profile'];

    foreach ($paths as $path) {
      if (!is_dir($path)) {
        continue;
      }

      $iterator = new \RecursiveIteratorIterator(
        new \RecursiveDirectoryIterator($path, \RecursiveDirectoryIterator::SKIP_DOTS)
      );

      foreach ($iterator as $file) {
        if ($file->isFile() && in_array($file->getExtension(), $extensions)) {
          $count++;
        }
      }
    }

    return $count;
  }

  /**
   * Detects available entity types and returns their PHPStan mappings.
   *
   * This method automatically detects which content entity types are available
   * in the current Drupal installation and returns their class mappings for
   * PHPStan configuration.
   *
   * @return array
   *   Array with 'mappings' (entity_type => class) and 'details' (full info).
   */
  protected function detectAvailableEntityTypes(): array {
    $mappings = [];
    $details = [];

    // Define known entity types with their expected classes and provider modules.
    $known_entity_types = [
      'node' => [
        'class' => 'Drupal\\node\\Entity\\Node',
        'module' => 'node',
        'label' => 'Node (Content)',
        'description' => 'Standard content entities like articles, pages, etc.',
      ],
      'user' => [
        'class' => 'Drupal\\user\\Entity\\User',
        'module' => 'user',
        'label' => 'User',
        'description' => 'User accounts and profiles.',
      ],
      'taxonomy_term' => [
        'class' => 'Drupal\\taxonomy\\Entity\\Term',
        'module' => 'taxonomy',
        'label' => 'Taxonomy Term',
        'description' => 'Vocabulary terms for categorization.',
      ],
      'file' => [
        'class' => 'Drupal\\file\\Entity\\File',
        'module' => 'file',
        'label' => 'File',
        'description' => 'Uploaded files and documents.',
      ],
      'media' => [
        'class' => 'Drupal\\media\\Entity\\Media',
        'module' => 'media',
        'label' => 'Media',
        'description' => 'Media entities (images, videos, documents).',
      ],
      'block_content' => [
        'class' => 'Drupal\\block_content\\Entity\\BlockContent',
        'module' => 'block_content',
        'label' => 'Block Content',
        'description' => 'Custom block content entities.',
      ],
      'menu_link_content' => [
        'class' => 'Drupal\\menu_link_content\\Entity\\MenuLinkContent',
        'module' => 'menu_link_content',
        'label' => 'Menu Link',
        'description' => 'Custom menu link entities.',
      ],
      'comment' => [
        'class' => 'Drupal\\comment\\Entity\\Comment',
        'module' => 'comment',
        'label' => 'Comment',
        'description' => 'Comment entities.',
      ],
      'paragraph' => [
        'class' => 'Drupal\\paragraphs\\Entity\\Paragraph',
        'module' => 'paragraphs',
        'label' => 'Paragraph',
        'description' => 'Paragraphs module entities for structured content.',
      ],
      'webform_submission' => [
        'class' => 'Drupal\\webform\\Entity\\WebformSubmission',
        'module' => 'webform',
        'label' => 'Webform Submission',
        'description' => 'Webform submission entities.',
      ],
      'commerce_product' => [
        'class' => 'Drupal\\commerce_product\\Entity\\Product',
        'module' => 'commerce_product',
        'label' => 'Commerce Product',
        'description' => 'Commerce product entities.',
      ],
      'commerce_order' => [
        'class' => 'Drupal\\commerce_order\\Entity\\Order',
        'module' => 'commerce_order',
        'label' => 'Commerce Order',
        'description' => 'Commerce order entities.',
      ],
    ];

    foreach ($known_entity_types as $entity_type_id => $info) {
      $is_installed = $this->moduleHandler->moduleExists($info['module']);

      $details[$entity_type_id] = [
        'label' => $info['label'],
        'description' => $info['description'],
        'module' => $info['module'],
        'class' => $info['class'],
        'installed' => $is_installed,
      ];

      if ($is_installed) {
        // Verify the entity type actually exists in the system.
        try {
          $definition = $this->entityTypeManager->getDefinition($entity_type_id, FALSE);
          if ($definition) {
            // Use the actual class from the definition if available.
            $actual_class = $definition->getClass();
            $mappings[$entity_type_id] = $actual_class;
            $details[$entity_type_id]['class'] = $actual_class;
            $details[$entity_type_id]['active'] = TRUE;
          }
          else {
            $details[$entity_type_id]['active'] = FALSE;
          }
        }
        catch (\Exception $e) {
          $details[$entity_type_id]['active'] = FALSE;
        }
      }
      else {
        $details[$entity_type_id]['active'] = FALSE;
      }
    }

    return [
      'mappings' => $mappings,
      'details' => $details,
    ];
  }

  /**
   * Creates empty scores.
   *
   * @param string $reason
   *   Reason for empty scores.
   *
   * @return array
   *   Score data.
   */
  protected function createEmptyScores(string $reason = ''): array {
    $description = $reason ?: (string) $this->t('PHPStan not available');

    return [
      'overall' => 0,
      'factors' => [
        'type_errors' => [
          'score' => 0,
          'weight' => self::SCORE_WEIGHTS['type_errors'],
          'label' => (string) $this->t('Type Errors'),
          'description' => $description,
        ],
        'deprecations' => [
          'score' => 0,
          'weight' => self::SCORE_WEIGHTS['deprecations'],
          'label' => (string) $this->t('Deprecations'),
          'description' => $description,
        ],
        'undefined_issues' => [
          'score' => 0,
          'weight' => self::SCORE_WEIGHTS['undefined_issues'],
          'label' => (string) $this->t('Undefined References'),
          'description' => $description,
        ],
      ],
    ];
  }

  /**
   * Creates perfect scores.
   *
   * @return array
   *   Score data.
   */
  protected function createPerfectScores(): array {
    return [
      'overall' => 100,
      'factors' => [
        'type_errors' => [
          'score' => 100,
          'weight' => self::SCORE_WEIGHTS['type_errors'],
          'label' => (string) $this->t('Type Errors'),
          'description' => (string) $this->t('No custom code to analyze'),
        ],
        'deprecations' => [
          'score' => 100,
          'weight' => self::SCORE_WEIGHTS['deprecations'],
          'label' => (string) $this->t('Deprecations'),
          'description' => (string) $this->t('No custom code to analyze'),
        ],
        'undefined_issues' => [
          'score' => 100,
          'weight' => self::SCORE_WEIGHTS['undefined_issues'],
          'label' => (string) $this->t('Undefined References'),
          'description' => (string) $this->t('No custom code to analyze'),
        ],
      ],
    ];
  }

  /**
   * Calculates scores for all factors using volume-aware penalty scoring.
   *
   * Formula: score = 100 - (issue_percentage * PENALTY_MULTIPLIER)
   *
   * This considers the volume of files analyzed and applies a multiplier
   * to make issues more impactful than simple percentages.
   *
   * @param array $stats
   *   Statistics array.
   *
   * @return array
   *   Score data with overall and factors.
   */
  protected function calculateScores(array $stats): array {
    $factors = [];

    $total_errors = $stats['total_errors'];
    $type_errors_count = $stats['type_errors_count'] ?? 0;
    $deprecation_count = $stats['deprecation_count'] ?? 0;
    $undefined_count = $stats['undefined_count'] ?? 0;
    $files_analyzed = $stats['files_analyzed'] ?? 1;
    $files_with_errors = $stats['files_with_errors'];

    // Type Errors score: Based on count relative to files analyzed.
    // Volume-aware penalty scoring.
    $type_error_percentage = $files_analyzed > 0
      ? ($type_errors_count / $files_analyzed) * 100
      : 0;
    $type_error_score = (int) max(0, round(100 - ($type_error_percentage * self::PENALTY_MULTIPLIER)));

    $factors['type_errors'] = [
      'score' => $type_error_score,
      'weight' => self::SCORE_WEIGHTS['type_errors'],
      'label' => (string) $this->t('Type Errors'),
      'description' => $type_errors_count === 0
        ? (string) $this->t('No type errors in @files files', ['@files' => $files_analyzed])
        : (string) $this->t('@count type errors found — incorrect code behavior risks', [
          '@count' => $type_errors_count,
        ]),
      'badges' => [
        [
          'value' => (string) $type_errors_count . ' ' . $this->t('errors'),
          'type' => $type_errors_count === 0 ? 'success' : 'error',
          'label' => (string) $this->t('Type mismatches'),
        ],
      ],
    ];

    // Deprecations score: Critical for Drupal 11 readiness.
    $deprecation_percentage = $files_analyzed > 0
      ? ($deprecation_count / $files_analyzed) * 100
      : 0;
    $deprecation_score = (int) max(0, round(100 - ($deprecation_percentage * self::PENALTY_MULTIPLIER)));

    $factors['deprecations'] = [
      'score' => $deprecation_score,
      'weight' => self::SCORE_WEIGHTS['deprecations'],
      'label' => (string) $this->t('Deprecations'),
      'description' => $deprecation_count === 0
        ? (string) $this->t('No deprecated code — ready for Drupal 11')
        : (string) $this->t('@count deprecated usages — update before Drupal 11', [
          '@count' => $deprecation_count,
        ]),
      'badges' => [
        [
          'value' => (string) $deprecation_count . ' ' . $this->t('warnings'),
          'type' => $deprecation_count === 0 ? 'success' : 'warning',
          'label' => (string) $this->t('Drupal 11 blockers'),
        ],
      ],
    ];

    // Undefined Issues score: Missing code/typos.
    $undefined_percentage = $files_analyzed > 0
      ? ($undefined_count / $files_analyzed) * 100
      : 0;
    $undefined_score = (int) max(0, round(100 - ($undefined_percentage * self::PENALTY_MULTIPLIER)));

    $factors['undefined_issues'] = [
      'score' => $undefined_score,
      'weight' => self::SCORE_WEIGHTS['undefined_issues'],
      'label' => (string) $this->t('Undefined References'),
      'description' => $undefined_count === 0
        ? (string) $this->t('All references are properly defined')
        : (string) $this->t('@count undefined references — potential runtime errors', [
          '@count' => $undefined_count,
        ]),
      'badges' => [
        [
          'value' => (string) $undefined_count . ' ' . $this->t('errors'),
          'type' => $undefined_count === 0 ? 'success' : 'error',
          'label' => (string) $this->t('Missing definitions'),
        ],
      ],
    ];

    // Calculate weighted overall score.
    $total_weight = array_sum(self::SCORE_WEIGHTS);
    $weighted_sum = 0;
    foreach ($factors as $factor) {
      $weighted_sum += $factor['score'] * $factor['weight'];
    }
    $overall = (int) round($weighted_sum / $total_weight);

    return [
      'overall' => $overall,
      'factors' => $factors,
    ];
  }

  /**
   * Gets scan directories from global audit configuration.
   *
   * @param \Drupal\Core\Config\ImmutableConfig $audit_config
   *   The audit configuration.
   *
   * @return array
   *   Array of existing directory paths.
   */
  protected function getScanDirectories($audit_config): array {
    $scan_dirs_raw = $audit_config->get('scan_directories') ?? "web/modules/custom\nweb/themes/custom";
    $scan_dirs = array_filter(array_map('trim', explode("\n", $scan_dirs_raw)));

    $paths = [];
    foreach ($scan_dirs as $dir) {
      // Try relative to DRUPAL_ROOT first.
      $full_path = DRUPAL_ROOT . '/' . ltrim($dir, '/');
      if (is_dir($full_path)) {
        $paths[] = $full_path;
        continue;
      }

      // Try as absolute path.
      if (is_dir($dir)) {
        $paths[] = $dir;
        continue;
      }

      // Try relative to project root (one level up from DRUPAL_ROOT).
      $project_path = dirname(DRUPAL_ROOT) . '/' . ltrim($dir, '/');
      if (is_dir($project_path)) {
        $paths[] = $project_path;
      }
    }

    return $paths;
  }

  /**
   * Gets exclude patterns from global audit configuration.
   *
   * @param \Drupal\Core\Config\ImmutableConfig $audit_config
   *   The audit configuration.
   *
   * @return array
   *   Array of exclude patterns.
   */
  protected function getExcludePatterns($audit_config): array {
    $default_patterns = "*Test.php\n*TestBase.php\ntests/\nnode_modules/\nvendor/";
    $exclude_raw = $audit_config->get('exclude_patterns') ?? $default_patterns;
    return array_filter(array_map('trim', explode("\n", $exclude_raw)));
  }

  /**
   * Finds a binary in common locations.
   *
   * @param string $name
   *   The binary name.
   *
   * @return string|null
   *   The path to the binary, or NULL if not found.
   */
  protected function findBinary(string $name): ?string {
    $paths = [
      DRUPAL_ROOT . '/../vendor/bin/' . $name,
      DRUPAL_ROOT . '/vendor/bin/' . $name,
    ];

    foreach ($paths as $path) {
      if (file_exists($path)) {
        return $path;
      }
    }

    return NULL;
  }

  /**
   * Checks if PHPStan Drupal extension is installed.
   *
   * @return bool
   *   TRUE if installed.
   */
  protected function findDrupalExtension(): bool {
    $paths = [
      DRUPAL_ROOT . '/../vendor/mglaman/phpstan-drupal',
      DRUPAL_ROOT . '/vendor/mglaman/phpstan-drupal',
    ];

    foreach ($paths as $path) {
      if (is_dir($path)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Detects all installed PHPStan extensions and packages.
   *
   * @return array
   *   Array with 'installed' and 'missing' package information.
   */
  protected function detectInstalledExtensions(): array {
    $installed = [];
    $missing = [];

    $package_paths = [
      'phpstan/phpstan' => 'phpstan/phpstan',
      'phpstan/extension-installer' => 'phpstan/extension-installer',
      'mglaman/phpstan-drupal' => 'mglaman/phpstan-drupal',
      'phpstan/phpstan-deprecation-rules' => 'phpstan/phpstan-deprecation-rules',
    ];

    foreach ($package_paths as $package => $path) {
      $vendor_paths = [
        DRUPAL_ROOT . '/../vendor/' . $path,
        DRUPAL_ROOT . '/vendor/' . $path,
      ];

      $found = FALSE;
      foreach ($vendor_paths as $vendor_path) {
        if (is_dir($vendor_path)) {
          $found = TRUE;
          break;
        }
      }

      if ($found) {
        $installed[$package] = self::REQUIRED_PACKAGES[$package] ?? $package;
      }
      else {
        $missing[$package] = self::REQUIRED_PACKAGES[$package] ?? $package;
      }
    }

    return [
      'installed' => $installed,
      'missing' => $missing,
    ];
  }

  /**
   * Checks if drupal-check is installed as an alternative tool.
   *
   * @return array|null
   *   Tool info if installed, NULL otherwise.
   */
  protected function detectDrupalCheck(): ?array {
    $paths = [
      DRUPAL_ROOT . '/../vendor/bin/drupal-check',
      DRUPAL_ROOT . '/vendor/bin/drupal-check',
    ];

    foreach ($paths as $path) {
      if (file_exists($path)) {
        return [
          'path' => $path,
          'installed' => TRUE,
        ];
      }
    }

    return NULL;
  }

  /**
   * Checks if drupal-rector is installed for automatic fixes.
   *
   * @return array|null
   *   Tool info if installed, NULL otherwise.
   */
  protected function detectDrupalRector(): ?array {
    $paths = [
      DRUPAL_ROOT . '/../vendor/bin/rector',
      DRUPAL_ROOT . '/vendor/bin/rector',
    ];

    foreach ($paths as $path) {
      if (file_exists($path)) {
        // Check if drupal-rector is installed, not just rector.
        $drupal_rector_paths = [
          DRUPAL_ROOT . '/../vendor/palantirnet/drupal-rector',
          DRUPAL_ROOT . '/vendor/palantirnet/drupal-rector',
        ];

        foreach ($drupal_rector_paths as $dr_path) {
          if (is_dir($dr_path)) {
            return [
              'path' => $path,
              'installed' => TRUE,
              'is_drupal_rector' => TRUE,
            ];
          }
        }

        return [
          'path' => $path,
          'installed' => TRUE,
          'is_drupal_rector' => FALSE,
        ];
      }
    }

    return NULL;
  }

  /**
   * Gets installation instructions.
   *
   * @return array
   *   Installation instructions.
   */
  protected function getInstallationInstructions(): array {
    return [
      'full_install' => 'composer require --dev phpstan/phpstan phpstan/extension-installer mglaman/phpstan-drupal phpstan/phpstan-deprecation-rules',
      'phpstan' => 'composer require --dev phpstan/phpstan',
      'extension_installer' => 'composer require --dev phpstan/extension-installer',
      'drupal_extension' => 'composer require --dev mglaman/phpstan-drupal',
      'deprecation_rules' => 'composer require --dev phpstan/phpstan-deprecation-rules',
      'drupal_rector' => 'composer require --dev palantirnet/drupal-rector',
      'drupal_check' => 'composer require --dev mglaman/drupal-check',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      // Scoring section 1: Type errors (most critical).
      'type_errors' => [
        'label' => $this->t('Type Errors'),
        'description' => $this->t('Type mismatches, wrong argument types, and return type issues that indicate incorrect code behavior.'),
        'file_key' => 'type_errors',
        'affects_score' => TRUE,
        'score_factor_key' => 'type_errors',
        'weight' => self::SCORE_WEIGHTS['type_errors'],
      ],
      // Scoring section 2: Deprecations (Drupal 11 readiness).
      'deprecation_issues' => [
        'label' => $this->t('Deprecation Warnings'),
        'description' => $this->t('Deprecated code that must be updated before upgrading to Drupal 11.'),
        'file_key' => 'deprecation_errors',
        'affects_score' => TRUE,
        'score_factor_key' => 'deprecations',
        'weight' => self::SCORE_WEIGHTS['deprecations'],
      ],
      // Scoring section 3: Undefined issues.
      'undefined_issues' => [
        'label' => $this->t('Undefined References'),
        'description' => $this->t('Undefined methods, properties, variables, and classes that may cause runtime errors.'),
        'file_key' => 'undefined_errors',
        'affects_score' => TRUE,
        'score_factor_key' => 'undefined_issues',
        'weight' => self::SCORE_WEIGHTS['undefined_issues'],
      ],
      // Informational section: Environment status (table).
      'environment_status' => [
        'label' => $this->t('PHPStan Environment'),
        'description' => $this->t('Installed packages, configuration, and analysis summary.'),
        'file_key' => 'environment',
        'affects_score' => FALSE,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    $files = $data['_files'] ?? [];

    return match ($check_id) {
      // Scoring sections (faceted lists).
      'type_errors' => $this->getTypeErrorsContent($files),
      'deprecation_issues' => $this->getDeprecationIssuesContent($files),
      'undefined_issues' => $this->getUndefinedIssuesContent($files),
      // Informational section (table).
      'environment_status' => $this->buildEnvironmentStatusContent($data),
      default => [],
    };
  }

  /**
   * Gets the type errors content (faceted list).
   *
   * @param array $files
   *   The _files array from analysis.
   *
   * @return array
   *   Render array content.
   */
  protected function getTypeErrorsContent(array $files): array {
    $results = $files['type_errors']['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('No type errors found. Your code has proper type handling.'),
          'success'
        ),
      ];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $file = $details['file'] ?? '';
      $line = $details['line'] ?? 0;
      $message = $details['message'] ?? $item['message'] ?? '';
      $identifier = $details['identifier'] ?? '';
      $tip = $details['tip'] ?? '';

      // Build description with the specific error message and fix suggestions.
      $description = '<p><strong>' . $this->t('Error:') . '</strong> ' . htmlspecialchars($message) . '</p>';
      $description .= '<p><strong>' . $this->t('How to fix:') . '</strong> ' . $this->getTypeErrorFix($message, $identifier) . '</p>';
      if (!empty($tip)) {
        $description .= '<p class="audit-tip"><strong>' . $this->t('Tip:') . '</strong> ' . htmlspecialchars($tip) . '</p>';
      }
      if (!empty($identifier)) {
        $description .= '<p class="audit-rule"><strong>' . $this->t('Rule:') . '</strong> <code>' . htmlspecialchars($identifier) . '</code></p>';
      }

      $issues[] = $this->ui->issue([
        'severity' => 'error',
        'code' => 'TYPE_ERROR',
        'file' => $file,
        'line' => $line ?: NULL,
        'label' => $this->getTypeErrorLabel($message, $identifier),
        'description' => ['#markup' => $description],
        'code_snippet' => $this->ui->buildIssueCodeSnippet($details, 'error'),
        'tags' => $this->getTypeErrorTags($message, $identifier),
      ]);
    }

    return ['list' => $this->ui->issueList($issues)];
  }

  /**
   * Gets the deprecation issues content (faceted list).
   *
   * @param array $files
   *   The _files array from analysis.
   *
   * @return array
   *   Render array content.
   */
  protected function getDeprecationIssuesContent(array $files): array {
    $results = $files['deprecation_errors']['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('No deprecated code found. Your code is ready for Drupal 11.'),
          'success'
        ),
      ];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $file = $details['file'] ?? '';
      $line = $details['line'] ?? 0;
      $message = $details['message'] ?? $item['message'] ?? '';
      $identifier = $details['identifier'] ?? '';
      $tip = $details['tip'] ?? '';

      // Build description with the specific deprecation message and fix suggestions.
      $description = '<p><strong>' . $this->t('Warning:') . '</strong> ' . htmlspecialchars($message) . '</p>';
      $description .= '<p><strong>' . $this->t('How to fix:') . '</strong> ' . $this->getDeprecationFix($message, $tip) . '</p>';
      if (!empty($tip)) {
        $description .= '<p class="audit-tip"><strong>' . $this->t('PHPStan Tip:') . '</strong> ' . htmlspecialchars($tip) . '</p>';
      }
      if (!empty($identifier)) {
        $description .= '<p class="audit-rule"><strong>' . $this->t('Rule:') . '</strong> <code>' . htmlspecialchars($identifier) . '</code></p>';
      }
      $description .= '<p class="audit-info"><strong>' . $this->t('Note:') . '</strong> ' . $this->t('Consider using <a href="https://github.com/palantirnet/drupal-rector" target="_blank">Drupal Rector</a> to automatically fix many deprecation issues.') . '</p>';

      $issues[] = $this->ui->issue([
        'severity' => 'warning',
        'code' => 'DEPRECATION',
        'file' => $file,
        'line' => $line ?: NULL,
        'label' => $this->getDeprecationLabel($message),
        'description' => ['#markup' => $description],
        'code_snippet' => $this->ui->buildIssueCodeSnippet($details, 'warning'),
        'tags' => $this->getDeprecationTags($message),
      ]);
    }

    return ['list' => $this->ui->issueList($issues)];
  }

  /**
   * Gets the undefined issues content (faceted list).
   *
   * @param array $files
   *   The _files array from analysis.
   *
   * @return array
   *   Render array content.
   */
  protected function getUndefinedIssuesContent(array $files): array {
    $results = $files['undefined_errors']['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('No undefined references found. All methods, properties, and classes are properly defined.'),
          'success'
        ),
      ];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $file = $details['file'] ?? '';
      $line = $details['line'] ?? 0;
      $message = $details['message'] ?? $item['message'] ?? '';
      $identifier = $details['identifier'] ?? '';
      $tip = $details['tip'] ?? '';

      // Build description with the specific error message and fix suggestions.
      $description = '<p><strong>' . $this->t('Error:') . '</strong> ' . htmlspecialchars($message) . '</p>';
      $description .= '<p><strong>' . $this->t('How to fix:') . '</strong> ' . $this->getUndefinedFix($message, $identifier) . '</p>';
      if (!empty($tip)) {
        $description .= '<p class="audit-tip"><strong>' . $this->t('Tip:') . '</strong> ' . htmlspecialchars($tip) . '</p>';
      }
      if (!empty($identifier)) {
        $description .= '<p class="audit-rule"><strong>' . $this->t('Rule:') . '</strong> <code>' . htmlspecialchars($identifier) . '</code></p>';
      }

      $issues[] = $this->ui->issue([
        'severity' => 'error',
        'code' => 'UNDEFINED',
        'file' => $file,
        'line' => $line ?: NULL,
        'label' => $this->getUndefinedLabel($message),
        'description' => ['#markup' => $description],
        'code_snippet' => $this->ui->buildIssueCodeSnippet($details, 'error'),
        'tags' => ['phpstan', 'undefined'],
      ]);
    }

    return ['list' => $this->ui->issueList($issues)];
  }

  /**
   * Builds content for the static analysis errors check.
   *
   * @param array $data
   *   Full analysis data including _files and stats.
   *
   * @return array
   *   Render array content.
   */
  protected function buildStaticAnalysisErrorsContent(array $data): array {
    $files = $data['_files'] ?? [];
    $stats = $data['stats'] ?? [];
    $results = $files['errors']['results'] ?? [];

    // If there are results, show as faceted list.
    if (!empty($results)) {
      return $this->ui->buildIssueListFromResults(
        $results,
        (string) $this->t('No PHPStan errors found.'),
        function (array $item, $ui): array {
          $details = $item['details'] ?? [];
          $file = $details['file'] ?? '';
          $line = $details['line'] ?? 0;
          $message = $details['message'] ?? $item['message'] ?? '';
          $identifier = $details['identifier'] ?? '';
          $tip = $details['tip'] ?? '';

          // Determine tags based on message content.
          $tags = ['phpstan'];
          $message_lower = strtolower($message);
          if (str_contains($message_lower, 'deprecat')) {
            $tags[] = 'deprecation';
            $tags[] = 'drupal11';
          }
          if (str_contains($message_lower, 'undefined')) {
            $tags[] = 'undefined';
          }
          if (str_contains($message_lower, 'type')) {
            $tags[] = 'type-error';
          }

          // Build description with tip if available.
          $description_parts = [];
          if (!empty($tip)) {
            $description_parts[] = '<p class="audit-tip"><strong>Tip:</strong> ' . htmlspecialchars($tip) . '</p>';
          }
          if (!empty($identifier)) {
            $description_parts[] = '<p class="audit-rule"><strong>Rule:</strong> <code>' . htmlspecialchars($identifier) . '</code></p>';
          }

          return [
            'severity' => $ui->normalizeSeverity($item['severity'] ?? 'error'),
            'code' => $item['code'] ?? 'PHPSTAN_ERROR',
            'file' => $file . ($line ? ':' . $line : ''),
            'label' => $this->truncateMessage($message, 120),
            'description' => !empty($description_parts) ? ['#markup' => implode('', $description_parts)] : NULL,
            'tags' => $tags,
          ];
        }
      );
    }

    // No issues - show success message.
    $files_analyzed = $stats['files_analyzed'] ?? 0;
    return [
      'message' => $this->ui->message(
        (string) $this->t('No PHPStan errors found in @count files. Your code passes static analysis.', [
          '@count' => $files_analyzed,
        ]),
        'success'
      ),
    ];
  }

  /**
   * Builds content for the environment status check.
   *
   * @param array $data
   *   Full analysis data including _files and stats.
   *
   * @return array
   *   Render array content.
   */
  protected function buildEnvironmentStatusContent(array $data): array {
    $files = $data['_files'] ?? [];
    $stats = $data['stats'] ?? [];
    $environment_data = $files['environment']['data'] ?? [];
    $extensions = $environment_data['extensions'] ?? $stats['environment']['extensions'] ?? ['installed' => [], 'missing' => []];
    $config = $environment_data['config'] ?? $stats['config'] ?? [];
    $paths_analyzed = $environment_data['paths_analyzed'] ?? $stats['paths_analyzed'] ?? [];
    $files_analyzed = $environment_data['files_analyzed'] ?? $stats['files_analyzed'] ?? 0;

    $content = [];

    // Configuration info table.
    $config_headers = [
      $this->ui->header((string) $this->t('Setting'), 'left', '40%'),
      $this->ui->header((string) $this->t('Value'), 'left'),
    ];

    $config_rows = [];
    if (!empty($config['level'])) {
      $config_rows[] = $this->ui->row([
        $this->ui->cell((string) $this->t('Analysis Level')),
        $this->ui->cell('Level ' . $config['level']),
      ]);
    }
    $config_rows[] = $this->ui->row([
      $this->ui->cell((string) $this->t('Files Analyzed')),
      $this->ui->cell((string) $files_analyzed),
    ]);
    if (!empty($paths_analyzed)) {
      $paths_display = array_map(function ($path) {
        return str_replace(DRUPAL_ROOT . '/', '', $path);
      }, $paths_analyzed);
      $config_rows[] = $this->ui->row([
        $this->ui->cell((string) $this->t('Paths')),
        $this->ui->cell(implode(', ', $paths_display)),
      ]);
    }

    if (!empty($config_rows)) {
      $content['config'] = $this->ui->table($config_headers, $config_rows);
    }

    // Packages table.
    $pkg_headers = [
      $this->ui->header((string) $this->t('Package'), 'left', '50%'),
      $this->ui->header((string) $this->t('Status'), 'center'),
      $this->ui->header((string) $this->t('Description'), 'left'),
    ];

    $pkg_rows = [];
    foreach ($extensions['installed'] ?? [] as $package => $description) {
      $pkg_rows[] = $this->ui->row([
        $this->ui->cell('<code>' . $package . '</code>'),
        $this->ui->cell($this->ui->badge((string) $this->t('Installed'), 'success')),
        $this->ui->cell($description),
      ]);
    }
    foreach ($extensions['missing'] ?? [] as $package => $description) {
      $pkg_rows[] = $this->ui->row([
        $this->ui->cell('<code>' . $package . '</code>'),
        $this->ui->cell($this->ui->badge((string) $this->t('Missing'), 'warning')),
        $this->ui->cell($description),
      ], 'warning');
    }

    if (!empty($pkg_rows)) {
      $content['packages_title'] = [
        '#type' => 'html_tag',
        '#tag' => 'h4',
        '#value' => (string) $this->t('PHPStan Extensions'),
        '#attributes' => ['class' => ['audit-section-title']],
      ];
      $content['packages'] = $this->ui->table($pkg_headers, $pkg_rows);
    }

    // Missing packages warning.
    if (!empty($extensions['missing'])) {
      $missing_packages = array_keys($extensions['missing']);
      $install_command = 'composer require --dev ' . implode(' ', $missing_packages);
      $content['missing_warning'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['audit-fix-section']],
        'message' => $this->ui->message(
          (string) $this->t('Install missing packages for better analysis:'),
          'info'
        ),
        'command' => [
          '#type' => 'html_tag',
          '#tag' => 'pre',
          '#value' => $install_command,
          '#attributes' => ['class' => ['audit-code-block']],
        ],
      ];
    }

    return $content;
  }

  /**
   * Gets the PHPStan analysis content (without section wrapper).
   *
   * @param array $data
   *   The analysis data.
   *
   * @return array
   *   Render array content.
   */
  protected function getPhpstanAnalysisContent(array $data): array {
    $content = [];
    $stats = $data['stats'] ?? [];

    // Not installed - show installation instructions.
    if (!($stats['available'] ?? FALSE)) {
      $content['installation'] = $this->getInstallationContent($stats);
      return $content;
    }

    // No code to analyze.
    if (!empty($stats['no_code'])) {
      $content['no_code'] = $this->ui->message(
        (string) $this->t('No custom code found to analyze. Check the "Directories to scan" setting in the main Audit configuration.'),
        'info'
      );
      return $content;
    }

    // Environment section (packages, config file).
    $content['environment'] = $this->getEnvironmentContent($stats);

    // Statistics section.
    $content['stats'] = $this->getStatsContent($stats);

    // Drupal 11 readiness section (if deprecations found).
    if (($stats['deprecation_count'] ?? 0) > 0) {
      $content['drupal11'] = $this->getDrupal11ReadinessContent($stats);
    }

    // Error details section.
    $error_items = array_filter(
      $data['results'] ?? [],
      fn($item) => $item['code'] === 'PHPSTAN_ERROR'
    );

    if (!empty($error_items)) {
      $max_display = $stats['config']['max_errors_display'] ?? self::DEFAULT_MAX_ERRORS;
      $content['errors'] = $this->getErrorDetailsContent($error_items, $max_display);
    }

    // How to fix section.
    if ($stats['total_errors'] > 0) {
      $content['how_to_fix'] = $this->getHowToFixContent($stats);
    }

    return $content;
  }

  /**
   * {@inheritdoc}
   */
  public function buildDetailedResults(array $data): array {
    $stats = $data['stats'] ?? [];

    // Not installed - show installation instructions.
    if (!($stats['available'] ?? FALSE)) {
      return [
        'installation' => $this->buildInstallationSection($stats),
      ];
    }

    // No code to analyze.
    if (!empty($stats['no_code'])) {
      return [
        'no_code' => $this->ui->message(
          (string) $this->t('No custom code found to analyze. Check the "Directories to scan" setting in the main Audit configuration.'),
          'info'
        ),
      ];
    }

    // Use parent implementation for standard sections with score circles.
    return parent::buildDetailedResults($data);
  }

  /**
   * Gets the installation instructions content.
   *
   * @param array $stats
   *   Statistics data.
   *
   * @return array
   *   Render array content.
   */
  protected function getInstallationContent(array $stats): array {
    $instructions = $this->getInstallationInstructions();

    $content = [];

    $content['warning'] = $this->ui->message(
      (string) $this->t('PHPStan is not installed. Follow the steps below to install it.'),
      'error'
    );

    // Single command to install everything.
    $content['step1'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-fix-section']],
      'title' => ['#markup' => '<h4>' . $this->t('1. Install All Recommended Packages') . '</h4>'],
      'description' => ['#markup' => '<p>' . $this->t('Install PHPStan with all recommended Drupal extensions in one command:') . '</p>'],
      'command' => [
        '#type' => 'html_tag',
        '#tag' => 'pre',
        '#value' => $instructions['full_install'],
        '#attributes' => ['class' => ['audit-code-block']],
      ],
    ];

    // Explain what each package does.
    $package_items = [];
    foreach (self::REQUIRED_PACKAGES as $package => $description) {
      $package_items[] = [
        '#markup' => '<code>' . $package . '</code> - ' . $description,
      ];
    }

    $content['packages'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-fix-section']],
      'title' => ['#markup' => '<h4>' . $this->t('Package Details') . '</h4>'],
      'list' => [
        '#theme' => 'item_list',
        '#items' => $package_items,
      ],
    ];

    // Reload page.
    $content['step2'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-fix-section']],
      'title' => ['#markup' => '<h4>' . $this->t('2. Reload This Page') . '</h4>'],
      'description' => ['#markup' => '<p>' . $this->t('After installation, reload this page to run the analysis. All configuration options (analysis level, entity mappings, ignored errors, etc.) can be configured from the <a href="@settings_url">Audit settings page</a>.', [
        '@settings_url' => '/admin/reports/audit/settings',
      ]) . '</p>'],
    ];

    return $content;
  }

  /**
   * Builds the installation instructions section.
   *
   * @param array $stats
   *   Statistics data.
   *
   * @return array
   *   Render array.
   *
   * @deprecated Use getInstallationContent() instead.
   */
  protected function buildInstallationSection(array $stats): array {
    return $this->ui->section(
      (string) $this->t('Installation Required'),
      $this->getInstallationContent($stats),
      ['open' => TRUE, 'severity' => 'error']
    );
  }

  /**
   * Gets the environment content showing installed packages and config.
   *
   * @param array $stats
   *   Statistics data.
   *
   * @return array
   *   Render array content.
   */
  protected function getEnvironmentContent(array $stats): array {
    $content = [];
    $environment = $stats['environment'] ?? [];
    $extensions = $environment['extensions'] ?? ['installed' => [], 'missing' => []];
    $drupal_rector = $environment['drupal_rector'] ?? NULL;

    // Installed packages table.
    $headers = [
      $this->ui->header((string) $this->t('Package'), 'left', '50%'),
      $this->ui->header((string) $this->t('Status'), 'left'),
      $this->ui->header((string) $this->t('Description'), 'left'),
    ];

    $rows = [];

    // Show installed packages.
    foreach ($extensions['installed'] as $package => $description) {
      $rows[] = $this->ui->row([
        '<code>' . $package . '</code>',
        $this->ui->badge((string) $this->t('Installed'), 'success'),
        $description,
      ]);
    }

    // Show missing packages.
    foreach ($extensions['missing'] as $package => $description) {
      $rows[] = $this->ui->row([
        '<code>' . $package . '</code>',
        $this->ui->badge((string) $this->t('Missing'), 'warning'),
        $description,
      ], 'warning');
    }

    $content['packages'] = $this->ui->table($headers, $rows);

    // Show missing package warning and install command.
    if (!empty($extensions['missing'])) {
      $missing_packages = array_keys($extensions['missing']);
      $install_command = 'composer require --dev ' . implode(' ', $missing_packages);

      $content['missing_warning'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['audit-fix-section', 'audit-margin-top']],
        'message' => $this->ui->message(
          (string) $this->t('Some recommended packages are missing. Install them for better Drupal analysis:'),
          'warning'
        ),
        'command' => [
          '#type' => 'html_tag',
          '#tag' => 'pre',
          '#value' => $install_command,
          '#attributes' => ['class' => ['audit-code-block']],
        ],
      ];
    }

    // Additional tools.
    if ($drupal_rector && $drupal_rector['installed']) {
      $content['tools_title'] = [
        '#markup' => '<h4 class="audit-subsection-title">' . $this->t('Additional Tools Detected') . '</h4>',
      ];

      $tool_rows = [];
      if ($drupal_rector['is_drupal_rector'] ?? FALSE) {
        $tool_rows[] = [
          '<code>drupal-rector</code>',
          $this->ui->badge((string) $this->t('Installed'), 'success'),
          (string) $this->t('Can automatically fix many deprecations'),
        ];
      }

      if (!empty($tool_rows)) {
        $content['tools_table'] = $this->ui->table(
          [
            $this->ui->header((string) $this->t('Tool'), 'left'),
            $this->ui->header((string) $this->t('Status'), 'left'),
            $this->ui->header((string) $this->t('Description'), 'left'),
          ],
          $tool_rows
        );
      }
    }

    return $content;
  }

  /**
   * Builds the environment section showing installed packages and config.
   *
   * @param array $stats
   *   Statistics data.
   *
   * @return array
   *   Render array.
   *
   * @deprecated Use getEnvironmentContent() instead.
   */
  protected function buildEnvironmentSection(array $stats): array {
    return $this->ui->section(
      (string) $this->t('PHPStan Environment'),
      $this->getEnvironmentContent($stats),
      ['open' => FALSE]
    );
  }

  /**
   * Gets the Drupal 11 readiness content.
   *
   * @param array $stats
   *   Statistics data.
   *
   * @return array
   *   Render array content.
   */
  protected function getDrupal11ReadinessContent(array $stats): array {
    $content = [];
    $deprecation_count = $stats['deprecation_count'] ?? 0;
    $total_errors = $stats['total_errors'];
    $drupal_rector = $stats['environment']['drupal_rector'] ?? NULL;

    // Calculate readiness percentage.
    $non_deprecation_errors = $total_errors - $deprecation_count;
    $readiness_percent = $total_errors > 0
      ? (int) round((($total_errors - $deprecation_count) / $total_errors) * 100)
      : 100;

    $content['summary'] = $this->ui->message(
      (string) $this->t('Found @count deprecation warnings that need to be addressed before upgrading to Drupal 11.', [
        '@count' => $deprecation_count,
      ]),
      'warning'
    );

    // Readiness info.
    $rows = [];

    $rows[] = [
      $this->ui->itemName((string) $this->t('Deprecation Warnings')),
      $this->ui->cell(
        $this->ui->number($deprecation_count, ['warning' => 1]) . ' ' .
        $this->ui->badge((string) $this->t('Drupal 11 blockers'), 'warning')
      ),
    ];

    $rows[] = [
      $this->ui->itemName((string) $this->t('Other Static Analysis Errors')),
      (string) $non_deprecation_errors,
    ];

    $content['stats'] = $this->ui->table(
      [
        $this->ui->header((string) $this->t('Metric'), 'left', '50%'),
        $this->ui->header((string) $this->t('Value'), 'left'),
      ],
      $rows
    );

    // Drupal-rector suggestion.
    $content['fix_title'] = [
      '#markup' => '<h4 class="audit-subsection-title">' . $this->t('Automatic Fixes with Drupal Rector') . '</h4>',
    ];

    if ($drupal_rector && ($drupal_rector['is_drupal_rector'] ?? FALSE)) {
      $content['rector_available'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['audit-fix-section']],
        'message' => $this->ui->message(
          (string) $this->t('Drupal Rector is installed. You can automatically fix many deprecations:'),
          'success'
        ),
        'command' => [
          '#type' => 'html_tag',
          '#tag' => 'pre',
          '#value' => "./vendor/bin/rector process web/modules/custom/my_module\n\n# Preview changes without applying:\n./vendor/bin/rector process web/modules/custom/my_module --dry-run",
          '#attributes' => ['class' => ['audit-code-block']],
        ],
        'note' => [
          '#markup' => '<p class="audit-note">' . $this->t('Replace <code>web/modules/custom/my_module</code> with the path you want to fix.') . '</p>',
        ],
      ];
    }
    else {
      $content['rector_install'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['audit-fix-section']],
        'message' => $this->ui->message(
          (string) $this->t('Install Drupal Rector to automatically fix many deprecation issues:'),
          'info'
        ),
        'command' => [
          '#type' => 'html_tag',
          '#tag' => 'pre',
          '#value' => "composer require --dev palantirnet/drupal-rector\n\n# Then run:\n./vendor/bin/rector process web/modules/custom/my_module --dry-run",
          '#attributes' => ['class' => ['audit-code-block']],
        ],
      ];
    }

    return $content;
  }

  /**
   * Builds the Drupal 11 readiness section.
   *
   * @param array $stats
   *   Statistics data.
   *
   * @return array
   *   Render array.
   *
   * @deprecated Use getDrupal11ReadinessContent() instead.
   */
  protected function buildDrupal11ReadinessSection(array $stats): array {
    return $this->ui->section(
      (string) $this->t('Drupal 11 Readiness'),
      $this->getDrupal11ReadinessContent($stats),
      ['open' => TRUE, 'severity' => 'warning']
    );
  }

  /**
   * Gets the statistics content.
   *
   * @param array $stats
   *   Statistics data.
   *
   * @return array
   *   Render array content.
   */
  protected function getStatsContent(array $stats): array {
    $headers = [
      $this->ui->header((string) $this->t('Metric'), 'left', '50%'),
      $this->ui->header((string) $this->t('Value'), 'left'),
    ];

    $rows = [];

    // Total errors.
    $rows[] = $this->ui->row([
      $this->ui->itemName((string) $this->t('Total Errors')),
      $this->ui->cell(
        $this->ui->number($stats['total_errors'], ['error' => 1]) . ' ' .
        $this->ui->badge(
          $stats['total_errors'] === 0 ? (string) $this->t('OK') : (string) $this->t('Issues'),
          $stats['total_errors'] === 0 ? 'success' : 'error'
        )
      ),
    ], $stats['total_errors'] > 0 ? 'error' : NULL);

    // Files with errors.
    $rows[] = $this->ui->row([
      $this->ui->itemName((string) $this->t('Files with Errors')),
      (string) $stats['files_with_errors'] . ' / ' . $stats['files_analyzed'],
    ], $stats['files_with_errors'] > 0 ? 'warning' : NULL);

    // Analysis level.
    $rows[] = [
      $this->ui->itemName((string) $this->t('Analysis Level')),
      (string) $this->t('Level @level', ['@level' => $stats['config']['level'] ?? self::DEFAULT_LEVEL]),
    ];

    // Paths analyzed.
    $paths_display = array_map(function ($path) {
      return str_replace(DRUPAL_ROOT . '/', '', $path);
    }, $stats['paths_analyzed'] ?? []);

    $rows[] = [
      $this->ui->itemName((string) $this->t('Paths Analyzed')),
      implode(', ', $paths_display),
    ];

    return $this->ui->table($headers, $rows);
  }

  /**
   * Builds the statistics section.
   *
   * @param array $stats
   *   Statistics data.
   *
   * @return array
   *   Render array.
   *
   * @deprecated Use getStatsContent() instead.
   */
  protected function buildStatsSection(array $stats): array {
    return $this->ui->section(
      (string) $this->t('Analysis Results'),
      $this->getStatsContent($stats),
      ['open' => TRUE]
    );
  }

  /**
   * Gets the error details content.
   *
   * @param array $error_items
   *   Array of error items.
   * @param int $max_display
   *   Maximum errors to display.
   *
   * @return array
   *   Render array content.
   */
  protected function getErrorDetailsContent(array $error_items, int $max_display = 500): array {
    $content = [];
    $display_limit = $max_display > 0 ? $max_display : count($error_items);
    $displayed_items = array_slice($error_items, 0, $display_limit);

    // Group by file.
    $by_file = [];
    foreach ($displayed_items as $item) {
      $file = $item['details']['file'] ?? 'unknown';
      if (!isset($by_file[$file])) {
        $by_file[$file] = [];
      }
      $by_file[$file][] = $item;
    }

    // Sort files alphabetically.
    ksort($by_file, SORT_STRING);

    foreach ($by_file as $file => $file_errors) {
      $file_content = [];

      foreach ($file_errors as $index => $item) {
        $details = $item['details'];
        $error_id = 'error-' . md5($file . '-' . $index);

        $file_content[$error_id] = $this->buildSingleError($details);
      }

      // Build badges for file title.
      $error_count = count($file_errors);
      $file_badge = $this->ui->badge((string) $error_count . ' E', 'error');

      $file_title = '<code class="audit-file-path-title">' . htmlspecialchars($file, ENT_QUOTES, 'UTF-8') . '</code> ' . $file_badge;

      $content['file_' . md5($file)] = $this->ui->section(
        $file_title,
        $file_content,
        [
          'open' => FALSE,
          'severity' => 'error',
        ]
      );
    }

    if (count($error_items) > $display_limit) {
      $content['limit_note'] = $this->ui->message(
        (string) $this->t('Showing @limit of @total errors. Adjust "Maximum errors to display" in settings to see more.', [
          '@limit' => $display_limit,
          '@total' => count($error_items),
        ]),
        'info'
      );
    }

    return $content;
  }

  /**
   * Builds the error details section.
   *
   * @param array $error_items
   *   Array of error items.
   * @param int $max_display
   *   Maximum errors to display.
   *
   * @return array
   *   Render array.
   *
   * @deprecated Use getErrorDetailsContent() instead.
   */
  protected function buildErrorDetailsSection(array $error_items, int $max_display = 500): array {
    return $this->ui->section(
      (string) $this->t('Error Details'),
      $this->getErrorDetailsContent($error_items, $max_display),
      ['open' => TRUE]
    );
  }

  /**
   * Builds a single error display.
   *
   * @param array $details
   *   Error details.
   *
   * @return array
   *   Render array.
   */
  protected function buildSingleError(array $details): array {
    $build = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['audit-violation', 'audit-violation--error'],
      ],
    ];

    // Location bar.
    $location_html = '<div class="audit-violation-location-bar">';
    $location_html .= '<span class="audit-location-item audit-location-line">';
    $location_html .= '<strong>' . $this->t('Line') . '</strong> ' . $details['line'];
    $location_html .= '</span>';
    $location_html .= '<span class="audit-violation-badges">';
    $location_html .= $this->ui->badge((string) $this->t('ERROR'), 'error');
    $location_html .= '</span>';
    $location_html .= '</div>';

    $build['location'] = ['#markup' => $location_html];

    // Message.
    $build['message'] = [
      '#markup' => '<div class="audit-violation-message">' .
        htmlspecialchars($details['message'], ENT_QUOTES, 'UTF-8') .
        '</div>',
    ];

    // Tip if available.
    if (!empty($details['tip'])) {
      $build['tip'] = [
        '#markup' => '<div class="audit-sniff-footer">' .
          '<span class="audit-sniff-label">' . $this->t('Tip:') . '</span> ' .
          '<span class="audit-sniff-description">' . htmlspecialchars($details['tip'], ENT_QUOTES, 'UTF-8') . '</span>' .
          '</div>',
      ];
    }

    // Identifier if available.
    if (!empty($details['identifier'])) {
      $build['identifier'] = [
        '#markup' => '<div class="audit-sniff-footer">' .
          '<span class="audit-sniff-label">' . $this->t('Rule:') . '</span> ' .
          '<code class="audit-sniff-code">' . htmlspecialchars($details['identifier'], ENT_QUOTES, 'UTF-8') . '</code>' .
          '</div>',
      ];
    }

    // Code context.
    if (!empty($details['code_context']['lines'])) {
      $build['code'] = $this->ui->code(
        $details['code_context']['lines'],
        [
          'highlight_line' => $details['code_context']['highlight_line'],
          'severity' => 'error',
          'language' => 'php',
        ]
      );
    }

    return $build;
  }

  /**
   * Gets the how to fix content.
   *
   * @param array $stats
   *   Statistics data.
   *
   * @return array
   *   Render array content.
   */
  protected function getHowToFixContent(array $stats): array {
    $content = [];
    $level = $stats['config']['level'] ?? self::DEFAULT_LEVEL;
    $total_errors = $stats['total_errors'];

    // Understanding errors.
    $content['understand'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-fix-section']],
      'title' => ['#markup' => '<h4>' . $this->t('1. Understand the Errors') . '</h4>'],
      'description' => ['#markup' => '<p>' . $this->t('PHPStan detects potential bugs through static analysis. Common issues include:') . '</p>'],
      'list' => [
        '#theme' => 'item_list',
        '#items' => [
          $this->t('<strong>Type mismatches:</strong> Passing wrong types to functions or returning unexpected types.'),
          $this->t('<strong>Undefined methods/properties:</strong> Calling methods or accessing properties that may not exist.'),
          $this->t('<strong>Dead code:</strong> Unreachable code or unused variables.'),
          $this->t('<strong>Null pointer issues:</strong> Accessing properties or methods on potentially null values.'),
          $this->t('<strong>Deprecations:</strong> Usage of deprecated APIs that will be removed in future Drupal versions.'),
        ],
      ],
    ];

    // Fix the issues.
    $content['fix'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-fix-section']],
      'title' => ['#markup' => '<h4>' . $this->t('2. Fix the Issues') . '</h4>'],
      'description' => ['#markup' => '<p>' . $this->t('The detected issues must be fixed manually in your code. In some cases, <a href="@url" target="_blank">Drupal Rector</a> can help automate some of the fixes.', [
        '@url' => 'https://github.com/palantirnet/drupal-rector',
      ]) . '</p>'],
    ];

    // Level progression.
    if ($level < 9 && $total_errors < 50) {
      $content['level_up'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['audit-fix-section']],
        'title' => ['#markup' => '<h4>' . $this->t('3. Increase Strictness Level') . '</h4>'],
        'description' => ['#markup' => '<p>' . $this->t('Once you\'ve fixed most errors at level @current, consider increasing to level @next for stricter analysis:', [
          '@current' => $level,
          '@next' => $level + 1,
        ]) . '</p>'],
        'list' => [
          '#theme' => 'item_list',
          '#items' => [
            $this->t('Level @level: @desc', ['@level' => $level + 1, '@desc' => self::LEVEL_DESCRIPTIONS[$level + 1] ?? '']),
          ],
        ],
      ];
    }

    // Verify fixes.
    $content['verify'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-fix-section']],
      'title' => ['#markup' => '<h4>' . $this->t('@num. Verify Fixes', ['@num' => ($level < 9 && $total_errors < 50) ? '4' : '3']) . '</h4>'],
      'description' => ['#markup' => '<p>' . $this->t('After fixing issues, reload this page to run the analysis again and verify the issues have been resolved.') . '</p>'],
    ];

    // Resources.
    $content['resources'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-fix-section']],
      'title' => ['#markup' => '<h4>' . $this->t('Resources') . '</h4>'],
      'list' => [
        '#theme' => 'item_list',
        '#items' => [
          $this->t('<a href="@url" target="_blank">PHPStan Documentation</a>', ['@url' => 'https://phpstan.org/user-guide/getting-started']),
          $this->t('<a href="@url" target="_blank">PHPStan Rule Levels Explained</a>', ['@url' => 'https://phpstan.org/user-guide/rule-levels']),
          $this->t('<a href="@url" target="_blank">PHPStan Drupal Extension</a>', ['@url' => 'https://github.com/mglaman/phpstan-drupal']),
          $this->t('<a href="@url" target="_blank">Drupal Rector</a>', ['@url' => 'https://github.com/palantirnet/drupal-rector']),
        ],
      ],
    ];

    return $content;
  }

  /**
   * Builds the how to fix section.
   *
   * @param array $stats
   *   Statistics data.
   *
   * @return array
   *   Render array.
   *
   * @deprecated Use getHowToFixContent() instead.
   */
  protected function buildHowToFixSection(array $stats): array {
    return $this->ui->section(
      (string) $this->t('How to Fix These Issues'),
      $this->getHowToFixContent($stats),
      ['open' => TRUE, 'severity' => 'notice']
    );
  }

}
