<?php

declare(strict_types=1);

namespace Drupal\audit_search_api\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Analyzes Search API indexes and servers configuration.
 */
#[AuditAnalyzer(
  id: 'search_api',
  label: new TranslatableMarkup('Search API Audit'),
  description: new TranslatableMarkup('Audits Search API indexes and servers, showing their status and configuration.'),
  menu_title: new TranslatableMarkup('Search API'),
  output_directory: 'search_api',
  weight: 3,
)]
class SearchApiAnalyzer extends AuditAnalyzerBase {

  /**
   * Points deducted per error.
   */
  protected const POINTS_PER_ERROR = 25;

  /**
   * Points deducted per warning.
   */
  protected const POINTS_PER_WARNING = 10;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->moduleHandler = $container->get('module_handler');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    $servers_analysis = $this->analyzeServers();
    $indexes_analysis = $this->analyzeIndexes();

    $total_errors = ($indexes_analysis['summary']['errors'] ?? 0) + ($servers_analysis['summary']['errors'] ?? 0);
    $total_warnings = ($indexes_analysis['summary']['warnings'] ?? 0) + ($servers_analysis['summary']['warnings'] ?? 0);

    $scores = $this->calculateScores($total_errors, $total_warnings);

    return [
      '_files' => [
        'servers' => $servers_analysis,
        'indexes' => $indexes_analysis,
      ],
      'summary' => [
        'errors' => $total_errors,
        'warnings' => $total_warnings,
        'notices' => 0,
      ],
      'totals' => [
        'servers' => $servers_analysis['totals'] ?? [],
        'indexes' => $indexes_analysis['totals'] ?? [],
      ],
      'score' => $scores,
    ];
  }

  /**
   * Calculates scores based on errors and warnings.
   *
   * @param int $errors
   *   Number of errors.
   * @param int $warnings
   *   Number of warnings.
   *
   * @return array
   *   Score data.
   */
  protected function calculateScores(int $errors, int $warnings): array {
    $base_score = 100;

    // Deduct points for errors and warnings.
    $deductions = ($errors * self::POINTS_PER_ERROR) + ($warnings * self::POINTS_PER_WARNING);
    $overall_score = max(0, $base_score - $deductions);

    return [
      'overall' => $overall_score,
      'factors' => [
        'connectivity' => [
          'score' => $errors > 0 ? max(0, 100 - ($errors * self::POINTS_PER_ERROR)) : 100,
          'label' => (string) $this->t('Server Connectivity'),
          'errors' => $errors,
        ],
        'configuration' => [
          'score' => $warnings > 0 ? max(0, 100 - ($warnings * self::POINTS_PER_WARNING)) : 100,
          'label' => (string) $this->t('Configuration'),
          'warnings' => $warnings,
        ],
      ],
    ];
  }

  /**
   * Analyzes Search API servers.
   *
   * Only reports:
   * - Error: Server enabled but not available (cannot connect).
   * - Warning: Server disabled.
   * - No notice for enabled and available servers.
   *
   * @return array
   *   Analysis results for servers.
   */
  protected function analyzeServers(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $totals = ['total' => 0, 'enabled' => 0, 'disabled' => 0, 'unavailable' => 0];

    try {
      /** @var \Drupal\search_api\ServerInterface[] $servers */
      $servers = $this->entityTypeManager
        ->getStorage('search_api_server')
        ->loadMultiple();

      foreach ($servers as $server) {
        $totals['total']++;
        $is_enabled = $server->status();
        $backend_id = $server->getBackendId();
        $backend_label = NULL;

        try {
          $backend = $server->getBackend();
          $backend_label = $backend->label();
        }
        catch (\Exception $e) {
          $backend_label = $backend_id;
        }

        // Count indexes using this server.
        $indexes_using = $this->getIndexesForServer($server->id());

        $severity = 'info';
        $code = 'SERVER_OK';
        $message = '';
        $is_available = TRUE;

        if ($is_enabled) {
          $totals['enabled']++;

          // Check server availability.
          try {
            $is_available = $server->isAvailable();
          }
          catch (\Exception $e) {
            $is_available = FALSE;
          }

          if (!$is_available) {
            // Error: Cannot connect to server.
            $totals['unavailable']++;
            $severity = 'error';
            $code = 'SERVER_UNAVAILABLE';
            $message = (string) $this->t('Server "@name" is enabled but not available.', ['@name' => $server->label()]);
            $errors++;
          }
          // Enabled and available: no issue to report.
        }
        else {
          // Warning: Server disabled.
          $totals['disabled']++;
          $severity = 'warning';
          $code = 'SERVER_DISABLED';
          $message = (string) $this->t('Server "@name" is disabled.', ['@name' => $server->label()]);
          $warnings++;
        }

        $results[] = $this->createResultItem(
          $severity,
          $code,
          $message,
          [
            'id' => $server->id(),
            'label' => $server->label(),
            'enabled' => $is_enabled,
            'available' => $is_available,
            'backend_id' => $backend_id,
            'backend_label' => $backend_label,
            'indexes_count' => count($indexes_using),
            'indexes' => $indexes_using,
          ]
        );
      }
    }
    catch (\Exception $e) {
      $results[] = $this->createResultItem(
        'error',
        'SERVER_LOAD_ERROR',
        (string) $this->t('Error loading Search API servers: @error', ['@error' => $e->getMessage()]),
        ['error' => $e->getMessage()]
      );
      $errors++;
    }

    $output = $this->createResult($results, $errors, $warnings, 0);
    $output['totals'] = $totals;

    return $output;
  }

  /**
   * Analyzes Search API indexes.
   *
   * Only reports:
   * - Error: Index enabled but has no server configured.
   * - Warning: Index disabled.
   * - No notice for enabled indexes with server.
   *
   * @return array
   *   Analysis results for indexes.
   */
  protected function analyzeIndexes(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $totals = ['total' => 0, 'enabled' => 0, 'disabled' => 0, 'no_server' => 0];

    try {
      /** @var \Drupal\search_api\IndexInterface[] $indexes */
      $indexes = $this->entityTypeManager
        ->getStorage('search_api_index')
        ->loadMultiple();

      foreach ($indexes as $index) {
        $totals['total']++;
        $is_enabled = $index->status();
        $server_id = $index->getServerId();
        $server_name = NULL;

        if ($server_id) {
          try {
            $server = $this->entityTypeManager
              ->getStorage('search_api_server')
              ->load($server_id);
            $server_name = $server ? $server->label() : NULL;
          }
          catch (\Exception $e) {
            $server_name = NULL;
          }
        }

        $tracker = $index->getTrackerInstance();
        $indexed_count = 0;
        $total_count = 0;

        if ($tracker && $is_enabled) {
          try {
            $indexed_count = $tracker->getIndexedItemsCount();
            $total_count = $tracker->getTotalItemsCount();
          }
          catch (\Exception $e) {
            // Tracker may fail if server is unavailable.
          }
        }

        $severity = 'info';
        $code = 'INDEX_OK';
        $message = '';

        if ($is_enabled) {
          $totals['enabled']++;

          // Check if index has no server configured.
          if (!$server_id) {
            $totals['no_server']++;
            $severity = 'error';
            $code = 'INDEX_NO_SERVER';
            $message = (string) $this->t('Index "@name" is enabled but has no server configured.', ['@name' => $index->label()]);
            $errors++;
          }
          // Enabled with server: no issue to report.
        }
        else {
          // Warning: Index disabled.
          $totals['disabled']++;
          $severity = 'warning';
          $code = 'INDEX_DISABLED';
          $message = (string) $this->t('Index "@name" is disabled.', ['@name' => $index->label()]);
          $warnings++;
        }

        $results[] = $this->createResultItem(
          $severity,
          $code,
          $message,
          [
            'id' => $index->id(),
            'label' => $index->label(),
            'enabled' => $is_enabled,
            'server_id' => $server_id,
            'server_name' => $server_name,
            'datasources' => array_keys($index->getDatasources()),
            'indexed_items' => $indexed_count,
            'total_items' => $total_count,
            'read_only' => $index->isReadOnly(),
          ]
        );
      }
    }
    catch (\Exception $e) {
      $results[] = $this->createResultItem(
        'error',
        'INDEX_LOAD_ERROR',
        (string) $this->t('Error loading Search API indexes: @error', ['@error' => $e->getMessage()]),
        ['error' => $e->getMessage()]
      );
      $errors++;
    }

    $output = $this->createResult($results, $errors, $warnings, 0);
    $output['totals'] = $totals;

    return $output;
  }

  /**
   * Gets indexes using a specific server.
   *
   * @param string $server_id
   *   The server ID.
   *
   * @return array
   *   Array of index info.
   */
  protected function getIndexesForServer(string $server_id): array {
    $indexes_using = [];

    try {
      /** @var \Drupal\search_api\IndexInterface[] $indexes */
      $indexes = $this->entityTypeManager
        ->getStorage('search_api_index')
        ->loadMultiple();

      foreach ($indexes as $index) {
        if ($index->getServerId() === $server_id) {
          $indexes_using[] = [
            'id' => $index->id(),
            'label' => $index->label(),
            'enabled' => $index->status(),
          ];
        }
      }
    }
    catch (\Exception $e) {
      // Ignore index loading errors.
    }

    return $indexes_using;
  }

  /**
   * {@inheritdoc}
   */
  public function checkRequirements(): array {
    $warnings = [];

    if (!$this->moduleHandler->moduleExists('search_api')) {
      $warnings[] = (string) $this->t('Search API module is not installed.');
    }

    return $warnings;
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      'servers' => [
        'label' => $this->t('Search API Servers'),
        'description' => $this->t('Checks Search API server connectivity and status.'),
        'file_key' => 'servers',
        'affects_score' => TRUE,
        'score_factor_key' => 'connectivity',
      ],
      'indexes' => [
        'label' => $this->t('Search API Indexes'),
        'description' => $this->t('Checks Search API index configuration and status.'),
        'file_key' => 'indexes',
        'affects_score' => TRUE,
        'score_factor_key' => 'configuration',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    $files = $data['_files'] ?? [];
    $totals = $data['totals'] ?? [];

    return match ($check_id) {
      'servers' => $this->getServersContent($files, $totals),
      'indexes' => $this->getIndexesContent($files, $totals),
      default => [],
    };
  }

  /**
   * Gets the servers content (without section wrapper).
   *
   * @param array $files
   *   The analysis files data.
   * @param array $totals
   *   The totals data.
   *
   * @return array
   *   Render array for the content.
   */
  protected function getServersContent(array $files, array $totals): array {
    $results = $files['servers']['results'] ?? [];
    $servers_totals = $totals['servers'] ?? [];
    $rows = [];

    // Count issues for title.
    $errors = 0;
    $warnings = 0;

    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $is_enabled = !empty($details['enabled']);
      $is_available = $details['available'] ?? TRUE;
      $severity = $item['severity'] ?? 'info';

      if ($severity === 'error') {
        $errors++;
      }
      elseif ($severity === 'warning') {
        $warnings++;
      }

      // Determine status icon and text.
      if (!$is_enabled) {
        $status_icon = $this->ui->icon('cross');
        $status_text = (string) $this->t('Disabled');
        $row_severity = 'warning';
      }
      elseif (!$is_available) {
        $status_icon = $this->ui->icon('cross');
        $status_text = (string) $this->t('Unavailable');
        $row_severity = 'error';
      }
      else {
        $status_icon = $this->ui->icon('check');
        $status_text = (string) $this->t('Available');
        $row_severity = NULL;
      }

      $indexes_count = $details['indexes_count'] ?? 0;
      $indexes_markup = $indexes_count > 0
        ? $this->ui->badge((string) $indexes_count, 'info')
        : $this->ui->badge('0', 'neutral');

      $rows[] = $this->ui->row([
        $this->ui->cell($status_icon, ['align' => 'center']),
        $this->ui->cell($this->ui->itemName((string) ($details['label'] ?? ''), (string) ($details['id'] ?? ''))),
        $this->ui->cell((string) ($details['backend_label'] ?? $details['backend_id'] ?? '-')),
        $this->ui->cell($indexes_markup, ['align' => 'center']),
        $this->ui->cell($status_text),
      ], $row_severity);
    }

    $headers = [
      $this->ui->header('', 'center', '40px'),
      $this->ui->header((string) $this->t('Server')),
      $this->ui->header((string) $this->t('Backend')),
      $this->ui->header((string) $this->t('Indexes'), 'center'),
      $this->ui->header((string) $this->t('Status')),
    ];

    return $this->ui->table($headers, $rows, [
      'empty' => (string) $this->t('No Search API servers found.'),
    ]);
  }

  /**
   * Gets the indexes content (without section wrapper).
   *
   * @param array $files
   *   The analysis files data.
   * @param array $totals
   *   The totals data.
   *
   * @return array
   *   Render array for the content.
   */
  protected function getIndexesContent(array $files, array $totals): array {
    $results = $files['indexes']['results'] ?? [];
    $indexes_totals = $totals['indexes'] ?? [];
    $rows = [];

    // Count issues for title.
    $errors = 0;
    $warnings = 0;

    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $is_enabled = !empty($details['enabled']);
      $has_server = !empty($details['server_id']);
      $severity = $item['severity'] ?? 'info';

      if ($severity === 'error') {
        $errors++;
      }
      elseif ($severity === 'warning') {
        $warnings++;
      }

      // Determine status icon and text.
      if (!$is_enabled) {
        $status_icon = $this->ui->icon('cross');
        $status_text = (string) $this->t('Disabled');
        $row_severity = 'warning';
      }
      elseif (!$has_server) {
        $status_icon = $this->ui->icon('cross');
        $status_text = (string) $this->t('No server');
        $row_severity = 'error';
      }
      else {
        $status_icon = $this->ui->icon('check');
        $status_text = (string) $this->t('Active');
        $row_severity = NULL;
      }

      $server_markup = $details['server_name']
        ? $this->ui->itemName($details['server_name'], $details['server_id'])
        : $this->ui->badge((string) $this->t('None'), 'neutral');

      $indexed = $details['indexed_items'] ?? 0;
      $total = $details['total_items'] ?? 0;
      if ($is_enabled && $total > 0) {
        $percentage = round(($indexed / $total) * 100, 1);
        $indexing_markup = $indexed . ' / ' . $total . ' (' . $percentage . '%)';
      }
      else {
        $indexing_markup = '-';
      }

      $datasources = $details['datasources'] ?? [];
      $datasources_markup = $datasources ? implode(', ', $datasources) : '-';

      $rows[] = $this->ui->row([
        $this->ui->cell($status_icon, ['align' => 'center']),
        $this->ui->cell($this->ui->itemName((string) ($details['label'] ?? ''), (string) ($details['id'] ?? ''))),
        $this->ui->cell($server_markup),
        $this->ui->cell($datasources_markup),
        $this->ui->cell($indexing_markup),
        $this->ui->cell($status_text),
      ], $row_severity);
    }

    $headers = [
      $this->ui->header('', 'center', '40px'),
      $this->ui->header((string) $this->t('Index')),
      $this->ui->header((string) $this->t('Server')),
      $this->ui->header((string) $this->t('Datasources')),
      $this->ui->header((string) $this->t('Indexed')),
      $this->ui->header((string) $this->t('Status')),
    ];

    return $this->ui->table($headers, $rows, [
      'empty' => (string) $this->t('No Search API indexes found.'),
    ]);
  }

}
