<?php

declare(strict_types=1);

namespace Drupal\audit_security\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\user\PermissionHandlerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Performs security audits on the Drupal site.
 */
#[AuditAnalyzer(
  id: 'security',
  label: new TranslatableMarkup('Security Audit'),
  description: new TranslatableMarkup('Detects security configuration errors and potential vulnerabilities.'),
  menu_title: new TranslatableMarkup('Security'),
  output_directory: 'security',
  weight: 1,
)]
class SecurityAnalyzer extends AuditAnalyzerBase {

  /**
   * Score weights for different security factors.
   */
  protected const SCORE_WEIGHTS = [
    'permissions' => 20,
    'error_reporting' => 12,
    'input_formats' => 15,
    'file_system' => 15,
    'trusted_hosts' => 12,
    'admin_user' => 8,
    'views_access' => 9,
    'upload_extensions' => 9,
  ];

  /**
   * Dangerous file extensions that should not be uploadable.
   */
  protected const DANGEROUS_EXTENSIONS = [
    'php', 'phtml', 'php3', 'php4', 'php5', 'php7', 'phps', 'phar',
    'exe', 'com', 'bat', 'cmd', 'vbs', 'vbe', 'js', 'jse',
    'ws', 'wsf', 'wsc', 'wsh', 'ps1', 'ps2', 'psc1', 'psc2',
    'msc', 'msi', 'msp', 'mst', 'hta', 'cpl', 'reg', 'inf',
    'scr', 'pif', 'application', 'gadget', 'jar', 'py', 'pl',
    'cgi', 'sh', 'bash', 'asp', 'aspx', 'cer', 'csr', 'jsp',
    'drv', 'sys', 'dll', 'so', 'dylib',
  ];

  /**
   * Unsafe HTML tags for text formats.
   */
  protected const UNSAFE_TAGS = [
    'script', 'iframe', 'object', 'embed', 'link', 'style',
    'applet', 'meta', 'base', 'form', 'input', 'button',
  ];

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The database connection.
   */
  protected Connection $database;

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity field manager.
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The permission handler.
   */
  protected PermissionHandlerInterface $permissionHandler;

  /**
   * The file system service.
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The stream wrapper manager.
   */
  protected StreamWrapperManagerInterface $streamWrapperManager;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->configFactory = $container->get('config.factory');
    $instance->database = $container->get('database');
    $instance->moduleHandler = $container->get('module_handler');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->permissionHandler = $container->get('user.permissions');
    $instance->fileSystem = $container->get('file_system');
    $instance->streamWrapperManager = $container->get('stream_wrapper_manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    // Run all security checks.
    $check_results = [
      'permissions' => $this->checkAdminPermissions(),
      'error_reporting' => $this->checkErrorReporting(),
      'input_formats' => $this->checkInputFormats(),
      'file_system' => $this->checkFileSystem(),
      'trusted_hosts' => $this->checkTrustedHosts(),
      'admin_user' => $this->checkAdminUser(),
      'views_access' => $this->checkViewsAccess(),
      'upload_extensions' => $this->checkUploadExtensions(),
    ];

    // Calculate scores.
    $scores = $this->calculateScores($check_results);

    // Build _files structure from check_results.
    $files = [];
    foreach ($check_results as $check_id => $check_data) {
      $files[$check_id] = $this->createResult(
        $check_data['results'] ?? [],
        $check_data['errors'] ?? 0,
        $check_data['warnings'] ?? 0,
        $check_data['notices'] ?? 0
      );
    }

    return [
      '_files' => $files,
      'score' => $scores,
      'check_results' => $check_results,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      'permissions' => [
        'label' => $this->t('Administrative Permissions'),
        'description' => $this->t('Checks for dangerous permissions granted to untrusted roles.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'permissions',
        'score_factor_key' => 'permissions',
        'weight' => self::SCORE_WEIGHTS['permissions'],
      ],
      'error_reporting' => [
        'label' => $this->t('Error Reporting'),
        'description' => $this->t('Checks if error messages are hidden from users.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'error_reporting',
        'score_factor_key' => 'error_reporting',
        'weight' => self::SCORE_WEIGHTS['error_reporting'],
      ],
      'input_formats' => [
        'label' => $this->t('Text Formats'),
        'description' => $this->t('Checks for unsafe HTML tags in text formats accessible by untrusted users.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'input_formats',
        'score_factor_key' => 'input_formats',
        'weight' => self::SCORE_WEIGHTS['input_formats'],
      ],
      'file_system' => [
        'label' => $this->t('File System Security'),
        'description' => $this->t('Checks private file path, .htaccess, and vendor directory configuration.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'file_system',
        'score_factor_key' => 'file_system',
        'weight' => self::SCORE_WEIGHTS['file_system'],
      ],
      'trusted_hosts' => [
        'label' => $this->t('Trusted Hosts'),
        'description' => $this->t('Checks if trusted host patterns are configured to prevent HTTP Host header attacks.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'trusted_hosts',
        'score_factor_key' => 'trusted_hosts',
        'weight' => self::SCORE_WEIGHTS['trusted_hosts'],
      ],
      'admin_user' => [
        'label' => $this->t('Admin Account'),
        'description' => $this->t('Checks the security status of the administrative account (UID 1).'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'admin_user',
        'score_factor_key' => 'admin_user',
        'weight' => self::SCORE_WEIGHTS['admin_user'],
      ],
      'views_access' => [
        'label' => $this->t('Views Access Control'),
        'description' => $this->t('Checks for View displays without access control configured.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'views_access',
        'score_factor_key' => 'views_access',
        'weight' => self::SCORE_WEIGHTS['views_access'],
      ],
      'upload_extensions' => [
        'label' => $this->t('Upload Security'),
        'description' => $this->t('Checks for dangerous file extensions allowed in upload fields.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'upload_extensions',
        'score_factor_key' => 'upload_extensions',
        'weight' => self::SCORE_WEIGHTS['upload_extensions'],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    $check_results = $data['check_results'] ?? [];
    $check_data = $check_results[$check_id] ?? [];

    return match ($check_id) {
      'permissions' => $this->buildPermissionsContent($check_data),
      'error_reporting' => $this->buildErrorReportingContent($check_data),
      'input_formats' => $this->buildInputFormatsContent($check_data),
      'file_system' => $this->buildFileSystemContent($check_data),
      'trusted_hosts' => $this->buildTrustedHostsContent($check_data),
      'admin_user' => $this->buildAdminUserContent($check_data),
      'views_access' => $this->buildViewsAccessContent($check_data),
      'upload_extensions' => $this->buildUploadExtensionsContent($check_data),
      default => [],
    };
  }

  /**
   * Builds content for permissions check.
   */
  protected function buildPermissionsContent(array $check_data): array {
    $issues = $check_data['issues'] ?? [];

    if (empty($issues)) {
      return ['message' => $this->ui->message(
        (string) $this->t('No dangerous permissions granted to untrusted roles.'),
        'success'
      )];
    }

    $headers = [
      (string) $this->t('Role'),
      (string) $this->t('Permission'),
    ];

    $rows = [];
    foreach ($issues as $issue) {
      $rows[] = [
        $issue['role'] ?? '',
        $issue['permission_label'] ?? $issue['permission'] ?? '',
      ];
    }

    return [
      'info' => $this->ui->message(
        (string) $this->t('Review permissions at /admin/people/permissions and remove restricted permissions from untrusted roles.'),
        'warning'
      ),
      'table' => $this->ui->table($headers, $rows),
    ];
  }

  /**
   * Builds content for error reporting check.
   */
  protected function buildErrorReportingContent(array $check_data): array {
    $passed = $check_data['passed'] ?? FALSE;
    $current_level = $check_data['current_level'] ?? 'unknown';

    if ($passed) {
      return ['message' => $this->ui->message(
        (string) $this->t('Error messages are logged, not displayed to users.'),
        'success'
      )];
    }

    return [
      'message' => $this->ui->message(
        (string) $this->t('Error messages are displayed to users (current: @level). Set to "None" at /admin/config/development/logging.', [
          '@level' => $current_level,
        ]),
        'warning'
      ),
    ];
  }

  /**
   * Builds content for input formats check.
   */
  protected function buildInputFormatsContent(array $check_data): array {
    $unsafe_formats = $check_data['unsafe_formats'] ?? [];

    if (empty($unsafe_formats)) {
      return ['message' => $this->ui->message(
        (string) $this->t('All text formats are properly filtered.'),
        'success'
      )];
    }

    $headers = [
      (string) $this->t('Format'),
      (string) $this->t('Issues'),
    ];

    $rows = [];
    foreach ($unsafe_formats as $format) {
      $rows[] = [
        $format['format_name'] ?? $format['format_id'],
        implode(', ', $format['issues'] ?? []),
      ];
    }

    return [
      'info' => $this->ui->message(
        (string) $this->t('Review text formats at /admin/config/content/formats and enable HTML filtering for untrusted users.'),
        'warning'
      ),
      'table' => $this->ui->table($headers, $rows),
    ];
  }

  /**
   * Builds content for file system check.
   */
  protected function buildFileSystemContent(array $check_data): array {
    $issues = $check_data['issues'] ?? [];

    if (empty($issues)) {
      return ['message' => $this->ui->message(
        (string) $this->t('File system is properly configured.'),
        'success'
      )];
    }

    $headers = [
      (string) $this->t('Issue'),
      (string) $this->t('Details'),
    ];

    $rows = [];
    foreach ($issues as $issue) {
      $rows[] = [
        $issue['message'] ?? '',
        $issue['path'] ?? '-',
      ];
    }

    return [
      'info' => $this->ui->message(
        (string) $this->t('Configure the private file system path outside the webroot and ensure .htaccess files are present.'),
        'warning'
      ),
      'table' => $this->ui->table($headers, $rows),
    ];
  }

  /**
   * Builds content for trusted hosts check.
   */
  protected function buildTrustedHostsContent(array $check_data): array {
    $passed = $check_data['passed'] ?? FALSE;
    $patterns = $check_data['patterns'] ?? [];

    if ($passed) {
      $content = ['message' => $this->ui->message(
        (string) $this->t('Trusted host patterns are configured.'),
        'success'
      )];

      if (!empty($patterns)) {
        $content['patterns'] = [
          '#theme' => 'item_list',
          '#title' => $this->t('Configured patterns'),
          '#items' => array_map(fn($p) => ['#markup' => '<code>' . htmlspecialchars($p, ENT_QUOTES, 'UTF-8') . '</code>'], $patterns),
        ];
      }

      return $content;
    }

    return [
      'message' => $this->ui->message(
        (string) $this->t('Trusted host patterns are not configured. Add $settings[\'trusted_host_patterns\'] to settings.php.'),
        'warning'
      ),
    ];
  }

  /**
   * Builds content for admin user check.
   */
  protected function buildAdminUserContent(array $check_data): array {
    $passed = $check_data['passed'] ?? FALSE;
    $super_user_disabled = $check_data['super_user_disabled'] ?? FALSE;

    if ($super_user_disabled) {
      return ['message' => $this->ui->message(
        (string) $this->t('Super user access is disabled in settings.php.'),
        'success'
      )];
    }

    if ($passed) {
      return ['message' => $this->ui->message(
        (string) $this->t('Admin account (UID 1) is properly secured.'),
        'success'
      )];
    }

    $files = $check_data['results'] ?? [];

    return $this->ui->buildIssueListFromResults(
      $files,
      (string) $this->t('Admin account is properly secured.'),
      function (array $item, $ui): array {
        return [
          'severity' => $item['severity'] ?? 'notice',
          'code' => $item['code'] ?? 'ADMIN_USER',
          'label' => $item['message'] ?? '',
          'tags' => ['security'],
        ];
      }
    );
  }

  /**
   * Builds content for views access check.
   */
  protected function buildViewsAccessContent(array $check_data): array {
    $skipped = $check_data['skipped'] ?? FALSE;
    $unprotected = $check_data['unprotected_views'] ?? [];

    if ($skipped) {
      return ['message' => $this->ui->message(
        (string) $this->t('Views module is not installed.'),
        'info'
      )];
    }

    if (empty($unprotected)) {
      return ['message' => $this->ui->message(
        (string) $this->t('All View displays have access control configured.'),
        'success'
      )];
    }

    $headers = [
      (string) $this->t('View'),
      (string) $this->t('Display'),
    ];

    $rows = [];
    foreach ($unprotected as $view) {
      $rows[] = [
        $view['view_label'] ?? $view['view_id'],
        $view['display_title'] ?? $view['display_id'],
      ];
    }

    return [
      'info' => $this->ui->message(
        (string) $this->t('Configure access control for each View display at /admin/structure/views.'),
        'warning'
      ),
      'table' => $this->ui->table($headers, $rows),
    ];
  }

  /**
   * Builds content for upload extensions check.
   */
  protected function buildUploadExtensionsContent(array $check_data): array {
    $dangerous_fields = $check_data['dangerous_fields'] ?? [];

    if (empty($dangerous_fields)) {
      return ['message' => $this->ui->message(
        (string) $this->t('No dangerous file extensions are allowed in upload fields.'),
        'success'
      )];
    }

    $headers = [
      (string) $this->t('Field'),
      (string) $this->t('Entity/Bundle'),
      (string) $this->t('Dangerous Extensions'),
    ];

    $rows = [];
    foreach ($dangerous_fields as $field) {
      $rows[] = [
        $field['field_label'] ?? $field['field_name'],
        $field['entity_type'] . '/' . $field['bundle'],
        implode(', ', $field['dangerous_extensions'] ?? []),
      ];
    }

    return [
      'info' => $this->ui->message(
        (string) $this->t('Review file field settings and remove dangerous extensions like php, exe, js from allowed uploads.'),
        'warning'
      ),
      'table' => $this->ui->table($headers, $rows),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $config): array {
    $form = [];

    // Load all roles.
    $roles = [];
    $admin_roles = [];

    try {
      $role_storage = $this->entityTypeManager->getStorage('user_role');
      $all_roles = $role_storage->loadMultiple();

      foreach ($all_roles as $role_id => $role) {
        /** @var \Drupal\user\RoleInterface $role */
        // Skip anonymous and authenticated base roles.
        if (in_array($role_id, [AccountInterface::ANONYMOUS_ROLE, AccountInterface::AUTHENTICATED_ROLE])) {
          continue;
        }
        $roles[$role_id] = $role->label();

        // Track roles that are marked as admin in Drupal.
        if ($role->isAdmin()) {
          $admin_roles[] = $role_id;
        }
      }
    }
    catch (\Exception $e) {
      // Role storage not available.
    }

    // Determine default values: use saved config or fall back to admin roles.
    $default_trusted = $config['trusted_roles'] ?? [];
    if (empty($default_trusted)) {
      $default_trusted = $admin_roles;
    }

    $form['trusted_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Trusted roles'),
      '#description' => $this->t('Select roles that should be considered trusted (administrators). These roles will not be checked for dangerous permissions. Roles marked as "Administrator role" in Drupal are pre-selected by default.'),
      '#options' => $roles,
      '#default_value' => $default_trusted,
    ];

    return $form;
  }

  /**
   * Checks for dangerous permissions granted to untrusted roles.
   *
   * @return array
   *   Check results.
   */
  protected function checkAdminPermissions(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $issues = [];

    // Get untrusted roles (all except admin).
    $untrusted_roles = $this->getUntrustedRoles();

    // Get all permissions with 'restrict access' flag.
    $all_permissions = $this->permissionHandler->getPermissions();
    $restricted_permissions = [];
    foreach ($all_permissions as $perm_name => $perm_info) {
      if (!empty($perm_info['restrict access'])) {
        $restricted_permissions[$perm_name] = $perm_info['title'] ?? $perm_name;
      }
    }

    // Check each untrusted role.
    try {
      $role_storage = $this->entityTypeManager->getStorage('user_role');
      foreach ($untrusted_roles as $role_id) {
        /** @var \Drupal\user\RoleInterface $role */
        $role = $role_storage->load($role_id);
        if (!$role) {
          continue;
        }

        $role_permissions = $role->getPermissions();
        $dangerous_perms = array_intersect($role_permissions, array_keys($restricted_permissions));

        foreach ($dangerous_perms as $perm) {
          $issues[] = [
            'role' => $role->label(),
            'role_id' => $role_id,
            'permission' => $perm,
            'permission_label' => (string) ($restricted_permissions[$perm] ?? $perm),
          ];
        }
      }
    }
    catch (\Exception $e) {
      // Role storage not available.
    }

    if (!empty($issues)) {
      $results[] = $this->createResultItem(
        'warning',
        'ADMIN_PERMISSIONS',
        (string) $this->t('@count restricted permissions granted to untrusted roles', ['@count' => count($issues)]),
        ['issues' => $issues]
      );
      $warnings++;
    }

    return [
      'results' => $results,
      'errors' => $errors,
      'warnings' => $warnings,
      'notices' => 0,
      'passed' => empty($issues),
      'issues' => $issues,
    ];
  }

  /**
   * Checks error reporting configuration.
   *
   * @return array
   *   Check results.
   */
  protected function checkErrorReporting(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;

    $error_level = $this->configFactory->get('system.logging')->get('error_level');

    // 'hide' is the secure setting.
    if ($error_level !== 'hide') {
      $results[] = $this->createResultItem(
        'warning',
        'ERROR_REPORTING',
        (string) $this->t('Error messages are displayed to users. This can reveal sensitive information about your site.'),
        [
          'current_level' => $error_level,
          'recommended' => 'hide',
        ]
      );
      $warnings++;
    }

    return [
      'results' => $results,
      'errors' => $errors,
      'warnings' => $warnings,
      'notices' => 0,
      'passed' => $error_level === 'hide',
      'current_level' => $error_level,
    ];
  }

  /**
   * Checks input formats for unsafe configurations.
   *
   * @return array
   *   Check results.
   */
  protected function checkInputFormats(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $unsafe_formats = [];

    // Get untrusted roles.
    $untrusted_roles = $this->getUntrustedRoles();

    try {
      $format_storage = $this->entityTypeManager->getStorage('filter_format');
      $formats = $format_storage->loadMultiple();

      foreach ($formats as $format_id => $format) {
        /** @var \Drupal\filter\FilterFormatInterface $format */
        $format_roles = $format->get('roles') ?: [];

        // Check if format is accessible by untrusted roles.
        $accessible_by_untrusted = !empty(array_intersect($format_roles, $untrusted_roles))
          || in_array(AccountInterface::ANONYMOUS_ROLE, $format_roles)
          || in_array(AccountInterface::AUTHENTICATED_ROLE, $format_roles);

        if (!$accessible_by_untrusted) {
          continue;
        }

        $filters = $format->get('filters') ?: [];
        $has_html_filter = FALSE;
        $has_html_escape = FALSE;
        $allowed_tags = [];

        foreach ($filters as $filter_id => $filter_config) {
          if (empty($filter_config['status'])) {
            continue;
          }

          if ($filter_id === 'filter_html') {
            $has_html_filter = TRUE;
            // Parse allowed tags.
            $allowed_html = $filter_config['settings']['allowed_html'] ?? '';
            if (preg_match_all('/<([a-z0-9]+)/i', $allowed_html, $matches)) {
              $allowed_tags = array_map('strtolower', $matches[1]);
            }
          }

          if ($filter_id === 'filter_html_escape') {
            $has_html_escape = TRUE;
          }
        }

        // Check for issues.
        $format_issues = [];

        if (!$has_html_filter && !$has_html_escape) {
          $format_issues[] = (string) $this->t('No HTML filtering enabled');
        }

        if ($has_html_filter && !empty($allowed_tags)) {
          $unsafe_allowed = array_intersect($allowed_tags, self::UNSAFE_TAGS);
          if (!empty($unsafe_allowed)) {
            $format_issues[] = (string) $this->t('Unsafe tags allowed: @tags', ['@tags' => implode(', ', $unsafe_allowed)]);
          }
        }

        if (!empty($format_issues)) {
          $unsafe_formats[] = [
            'format_id' => $format_id,
            'format_name' => $format->label(),
            'issues' => $format_issues,
          ];
        }
      }
    }
    catch (\Exception $e) {
      // Filter format storage not available.
    }

    if (!empty($unsafe_formats)) {
      $results[] = $this->createResultItem(
        'warning',
        'INPUT_FORMATS',
        (string) $this->t('@count text formats have potential security issues', ['@count' => count($unsafe_formats)]),
        ['formats' => $unsafe_formats]
      );
      $warnings++;
    }

    return [
      'results' => $results,
      'errors' => $errors,
      'warnings' => $warnings,
      'notices' => 0,
      'passed' => empty($unsafe_formats),
      'unsafe_formats' => $unsafe_formats,
    ];
  }

  /**
   * Checks file system security.
   *
   * @return array
   *   Check results.
   */
  protected function checkFileSystem(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $issues = [];

    // Check private files location.
    $private_path = Settings::get('file_private_path');
    if (empty($private_path)) {
      $issues[] = [
        'type' => 'private_files',
        'message' => (string) $this->t('Private file system path is not configured'),
        'severity' => 'warning',
      ];
      $warnings++;
    }
    elseif ($this->isPathInWebroot($private_path)) {
      $issues[] = [
        'type' => 'private_files',
        'message' => (string) $this->t('Private files directory is inside the webroot'),
        'severity' => 'error',
        'path' => $private_path,
      ];
      $errors++;
    }

    // Check .htaccess in public files.
    $public_path = $this->getPublicFilesPath();
    if ($public_path) {
      $htaccess_path = $public_path . '/.htaccess';
      if (!file_exists($htaccess_path)) {
        $issues[] = [
          'type' => 'htaccess',
          'message' => (string) $this->t('.htaccess file missing in public files directory'),
          'severity' => 'warning',
          'path' => $public_path,
        ];
        $warnings++;
      }
      elseif (is_writable($htaccess_path)) {
        $issues[] = [
          'type' => 'htaccess',
          'message' => (string) $this->t('.htaccess file is writable in public files directory'),
          'severity' => 'warning',
          'path' => $htaccess_path,
        ];
        $warnings++;
      }
    }

    // Check vendor directory location.
    $vendor_in_webroot = file_exists(DRUPAL_ROOT . '/vendor/autoload.php');
    if ($vendor_in_webroot) {
      $issues[] = [
        'type' => 'vendor',
        'message' => (string) $this->t('Vendor directory is inside the webroot'),
        'severity' => 'warning',
      ];
      $warnings++;
    }

    if (!empty($issues)) {
      $results[] = $this->createResultItem(
        $errors > 0 ? 'error' : 'warning',
        'FILE_SYSTEM',
        (string) $this->t('@count file system security issues detected', ['@count' => count($issues)]),
        ['issues' => $issues]
      );
    }

    return [
      'results' => $results,
      'errors' => $errors,
      'warnings' => $warnings,
      'notices' => 0,
      'passed' => empty($issues),
      'issues' => $issues,
    ];
  }

  /**
   * Checks trusted hosts configuration.
   *
   * @return array
   *   Check results.
   */
  protected function checkTrustedHosts(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;

    $trusted_hosts = Settings::get('trusted_host_patterns');
    $is_configured = !empty($trusted_hosts);

    if (!$is_configured) {
      $results[] = $this->createResultItem(
        'warning',
        'TRUSTED_HOSTS',
        (string) $this->t('Trusted host patterns are not configured. This can expose the site to HTTP Host header attacks.'),
        ['configured' => FALSE]
      );
      $warnings++;
    }

    return [
      'results' => $results,
      'errors' => $errors,
      'warnings' => $warnings,
      'notices' => 0,
      'passed' => $is_configured,
      'patterns' => $trusted_hosts ?: [],
    ];
  }

  /**
   * Checks admin user (UID 1) status.
   *
   * @return array
   *   Check results.
   */
  protected function checkAdminUser(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    // Check if super user access is disabled.
    $super_user_disabled = Settings::get('security.enable_super_user') === FALSE;

    if ($super_user_disabled) {
      return [
        'results' => $results,
        'errors' => 0,
        'warnings' => 0,
        'notices' => 0,
        'passed' => TRUE,
        'super_user_disabled' => TRUE,
      ];
    }

    try {
      $user_storage = $this->entityTypeManager->getStorage('user');
      /** @var \Drupal\user\UserInterface $admin */
      $admin = $user_storage->load(1);

      if ($admin) {
        $is_blocked = $admin->isBlocked();
        $username = $admin->getAccountName();

        // Check if username is something obvious.
        $obvious_usernames = ['admin', 'administrator', 'root', 'drupal'];
        $has_obvious_username = in_array(strtolower($username), $obvious_usernames);

        if (!$is_blocked) {
          $results[] = $this->createResultItem(
            'notice',
            'ADMIN_USER_ACTIVE',
            (string) $this->t('The administrative account (UID 1) is active. Consider blocking it for security.'),
            ['username' => $username, 'blocked' => FALSE]
          );
          $notices++;
        }

        if ($has_obvious_username) {
          $results[] = $this->createResultItem(
            'warning',
            'ADMIN_USER_OBVIOUS',
            (string) $this->t('The administrative account has an obvious username "@name". Consider changing it.', ['@name' => $username]),
            ['username' => $username]
          );
          $warnings++;
        }
      }
    }
    catch (\Exception $e) {
      // User storage not available.
    }

    return [
      'results' => $results,
      'errors' => $errors,
      'warnings' => $warnings,
      'notices' => $notices,
      'passed' => empty($results),
      'super_user_disabled' => FALSE,
    ];
  }

  /**
   * Checks Views access control configuration.
   *
   * @return array
   *   Check results.
   */
  protected function checkViewsAccess(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $unprotected_views = [];

    if (!$this->moduleHandler->moduleExists('views')) {
      return [
        'results' => $results,
        'errors' => 0,
        'warnings' => 0,
        'notices' => 0,
        'passed' => TRUE,
        'skipped' => TRUE,
      ];
    }

    try {
      $view_storage = $this->entityTypeManager->getStorage('view');
      $views = $view_storage->loadMultiple();

      foreach ($views as $view_id => $view) {
        /** @var \Drupal\views\ViewEntityInterface $view */
        if (!$view->status()) {
          continue;
        }

        $displays = $view->get('display');
        foreach ($displays as $display_id => $display) {
          // Skip default display for now.
          if ($display_id === 'default') {
            continue;
          }

          $access_type = $display['display_options']['access']['type']
            ?? $displays['default']['display_options']['access']['type']
            ?? 'none';

          if ($access_type === 'none') {
            $unprotected_views[] = [
              'view_id' => $view_id,
              'view_label' => $view->label(),
              'display_id' => $display_id,
              'display_title' => $display['display_title'] ?? $display_id,
            ];
          }
        }
      }
    }
    catch (\Exception $e) {
      // Views storage not available.
    }

    if (!empty($unprotected_views)) {
      $results[] = $this->createResultItem(
        'warning',
        'VIEWS_ACCESS',
        (string) $this->t('@count View displays have no access control', ['@count' => count($unprotected_views)]),
        ['views' => $unprotected_views]
      );
      $warnings++;
    }

    return [
      'results' => $results,
      'errors' => $errors,
      'warnings' => $warnings,
      'notices' => 0,
      'passed' => empty($unprotected_views),
      'unprotected_views' => $unprotected_views,
    ];
  }

  /**
   * Checks for dangerous file upload extensions.
   *
   * @return array
   *   Check results.
   */
  protected function checkUploadExtensions(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $dangerous_fields = [];

    try {
      $field_storage = $this->entityTypeManager->getStorage('field_config');
      $fields = $field_storage->loadMultiple();

      foreach ($fields as $field) {
        /** @var \Drupal\field\FieldConfigInterface $field */
        $field_type = $field->getType();

        if (!in_array($field_type, ['file', 'image'])) {
          continue;
        }

        $settings = $field->getSettings();
        $extensions = $settings['file_extensions'] ?? '';
        $allowed = array_filter(array_map('trim', explode(' ', strtolower($extensions))));

        $dangerous_allowed = array_intersect($allowed, self::DANGEROUS_EXTENSIONS);

        if (!empty($dangerous_allowed)) {
          $dangerous_fields[] = [
            'field_name' => $field->getName(),
            'entity_type' => $field->getTargetEntityTypeId(),
            'bundle' => $field->getTargetBundle(),
            'field_label' => $field->label(),
            'dangerous_extensions' => array_values($dangerous_allowed),
          ];
        }
      }
    }
    catch (\Exception $e) {
      // Field config storage not available.
    }

    if (!empty($dangerous_fields)) {
      $results[] = $this->createResultItem(
        'warning',
        'UPLOAD_EXTENSIONS',
        (string) $this->t('@count fields allow dangerous file extensions', ['@count' => count($dangerous_fields)]),
        ['fields' => $dangerous_fields]
      );
      $warnings++;
    }

    return [
      'results' => $results,
      'errors' => $errors,
      'warnings' => $warnings,
      'notices' => 0,
      'passed' => empty($dangerous_fields),
      'dangerous_fields' => $dangerous_fields,
    ];
  }


  /**
   * Gets untrusted role IDs.
   *
   * @return array
   *   Array of role IDs that are not trusted.
   */
  protected function getUntrustedRoles(): array {
    // Get trusted roles from configuration.
    $config = $this->configFactory->get('audit_security.settings');
    $trusted_from_config = $config->get('trusted_roles') ?? [];

    // Filter out empty values (unchecked checkboxes return '0').
    $trusted = array_filter($trusted_from_config);

    try {
      $role_storage = $this->entityTypeManager->getStorage('user_role');
      $all_roles = $role_storage->loadMultiple();

      $untrusted = [];
      foreach ($all_roles as $role_id => $role) {
        /** @var \Drupal\user\RoleInterface $role */
        // A role is untrusted if it's not in the trusted list AND not marked
        // as admin role in Drupal.
        if (!in_array($role_id, $trusted) && !$role->isAdmin()) {
          $untrusted[] = $role_id;
        }
      }

      return $untrusted;
    }
    catch (\Exception $e) {
      return [AccountInterface::ANONYMOUS_ROLE, AccountInterface::AUTHENTICATED_ROLE];
    }
  }

  /**
   * Checks if a path is inside the webroot.
   *
   * @param string $path
   *   The path to check.
   *
   * @return bool
   *   TRUE if path is inside webroot.
   */
  protected function isPathInWebroot(string $path): bool {
    $real_path = $this->fileSystem->realpath($path);
    if (!$real_path) {
      // Path doesn't exist, check if it would be in webroot.
      if (str_starts_with($path, '/')) {
        return str_starts_with($path, DRUPAL_ROOT);
      }
      return TRUE;
    }

    return str_starts_with($real_path, DRUPAL_ROOT);
  }

  /**
   * Gets the public files path.
   *
   * @return string|null
   *   The public files path or NULL.
   */
  protected function getPublicFilesPath(): ?string {
    try {
      $wrapper = $this->streamWrapperManager->getViaScheme('public');
      if ($wrapper) {
        $path = $wrapper->realpath();
        // realpath() can return false if path doesn't exist.
        return $path !== FALSE ? $path : NULL;
      }
    }
    catch (\Exception $e) {
      // Stream wrapper not available.
    }

    return NULL;
  }

  /**
   * Calculates scores for all security factors.
   *
   * @param array $check_results
   *   Results from all checks.
   *
   * @return array
   *   Score data with overall and factors.
   */
  protected function calculateScores(array $check_results): array {
    $factors = [];

    // Permissions score.
    $perm_score = ($check_results['permissions']['passed'] ?? FALSE) ? 100 : 50;
    $factors['permissions'] = [
      'score' => $perm_score,
      'weight' => self::SCORE_WEIGHTS['permissions'],
      'label' => (string) $this->t('Permissions'),
      'description' => $perm_score === 100
        ? (string) $this->t('No dangerous permissions for untrusted roles')
        : (string) $this->t('Some restricted permissions granted to untrusted roles'),
    ];

    // Error reporting score.
    $error_score = ($check_results['error_reporting']['passed'] ?? FALSE) ? 100 : 0;
    $factors['error_reporting'] = [
      'score' => $error_score,
      'weight' => self::SCORE_WEIGHTS['error_reporting'],
      'label' => (string) $this->t('Error Reporting'),
      'description' => $error_score === 100
        ? (string) $this->t('Errors are logged, not displayed')
        : (string) $this->t('Error messages visible to users'),
    ];

    // Input formats score.
    $format_score = ($check_results['input_formats']['passed'] ?? FALSE) ? 100 : 40;
    $factors['input_formats'] = [
      'score' => $format_score,
      'weight' => self::SCORE_WEIGHTS['input_formats'],
      'label' => (string) $this->t('Text Formats'),
      'description' => $format_score === 100
        ? (string) $this->t('All text formats properly filtered')
        : (string) $this->t('Some text formats may allow unsafe content'),
    ];

    // File system score.
    $file_score = 100;
    if (!($check_results['file_system']['passed'] ?? TRUE)) {
      $file_score = ($check_results['file_system']['errors'] ?? 0) > 0 ? 0 : 50;
    }
    $factors['file_system'] = [
      'score' => $file_score,
      'weight' => self::SCORE_WEIGHTS['file_system'],
      'label' => (string) $this->t('File System'),
      'description' => $file_score === 100
        ? (string) $this->t('File system properly configured')
        : (string) $this->t('File system security issues detected'),
    ];

    // Trusted hosts score.
    $hosts_score = ($check_results['trusted_hosts']['passed'] ?? FALSE) ? 100 : 0;
    $factors['trusted_hosts'] = [
      'score' => $hosts_score,
      'weight' => self::SCORE_WEIGHTS['trusted_hosts'],
      'label' => (string) $this->t('Trusted Hosts'),
      'description' => $hosts_score === 100
        ? (string) $this->t('Trusted host patterns configured')
        : (string) $this->t('Trusted host patterns not configured'),
    ];

    // Admin user score.
    $admin_score = ($check_results['admin_user']['passed'] ?? FALSE) ? 100 : 70;
    $factors['admin_user'] = [
      'score' => $admin_score,
      'weight' => self::SCORE_WEIGHTS['admin_user'],
      'label' => (string) $this->t('Admin Account'),
      'description' => $admin_score === 100
        ? (string) $this->t('Admin account properly secured')
        : (string) $this->t('Admin account security can be improved'),
    ];

    // Views access score.
    $views_score = ($check_results['views_access']['passed'] ?? TRUE) ? 100 : 50;
    $factors['views_access'] = [
      'score' => $views_score,
      'weight' => self::SCORE_WEIGHTS['views_access'],
      'label' => (string) $this->t('Views Access'),
      'description' => $views_score === 100
        ? (string) $this->t('All Views have access control')
        : (string) $this->t('Some Views lack access control'),
    ];

    // Upload extensions score.
    $upload_score = ($check_results['upload_extensions']['passed'] ?? TRUE) ? 100 : 30;
    $factors['upload_extensions'] = [
      'score' => $upload_score,
      'weight' => self::SCORE_WEIGHTS['upload_extensions'],
      'label' => (string) $this->t('Upload Security'),
      'description' => $upload_score === 100
        ? (string) $this->t('No dangerous upload extensions allowed')
        : (string) $this->t('Some fields allow dangerous file types'),
    ];

    return [
      'factors' => $factors,
    ];
  }

}
