<?php

declare(strict_types=1);

namespace Drupal\audit_seo\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Analyzes SEO technical configuration.
 */
#[AuditAnalyzer(
  id: 'seo',
  label: new TranslatableMarkup('SEO Audit'),
  description: new TranslatableMarkup('Analyzes SEO modules, URL patterns, metatag configuration, and media optimization.'),
  menu_title: new TranslatableMarkup('SEO'),
  output_directory: 'seo',
  weight: 3,
)]
class SeoAnalyzer extends AuditAnalyzerBase {

  /**
   * Score weights for different factors.
   *
   * Priority: Essential Modules (40%) > URL Structure (30%) > Robots.txt (15%) > Media SEO (15%)
   * Note: Recommended modules are notice-only and do NOT affect scoring.
   */
  protected const SCORE_WEIGHTS = [
    'essential_modules' => 40,
    'url_structure' => 30,
    'robots_txt' => 15,
    'media_seo' => 15,
  ];

  /**
   * Essential SEO modules (error if missing).
   */
  protected const ESSENTIAL_MODULES = [
    'metatag' => [
      'label' => 'Metatag',
      'description' => 'Provides meta tags for SEO optimization',
    ],
    'pathauto' => [
      'label' => 'Pathauto',
      'description' => 'Automatically generates URL aliases',
    ],
    'simple_sitemap' => [
      'label' => 'Simple XML Sitemap',
      'description' => 'Generates XML sitemaps for search engines',
    ],
    'redirect' => [
      'label' => 'Redirect',
      'description' => 'Manages URL redirections (301/302) to preserve SEO',
    ],
  ];

  /**
   * Recommended SEO modules (notice if missing).
   */
  protected const RECOMMENDED_MODULES = [
    'schema_metatag' => [
      'label' => 'Schema.org Metatag',
      'description' => 'Adds structured data for rich snippets',
    ],
    'metatag_open_graph' => [
      'label' => 'Metatag Open Graph',
      'description' => 'Adds Open Graph tags for social sharing',
    ],
    'metatag_twitter_cards' => [
      'label' => 'Metatag Twitter Cards',
      'description' => 'Adds Twitter Card tags for Twitter sharing',
    ],
    'easy_breadcrumb' => [
      'label' => 'Easy Breadcrumb',
      'description' => 'Automatic breadcrumb navigation for better UX',
    ],
  ];

  /**
   * Entity types to check for SEO configuration.
   */
  protected const SEO_ENTITY_TYPES = [
    'node' => 'Content',
    'taxonomy_term' => 'Taxonomy',
    'media' => 'Media',
    'user' => 'User',
  ];

  protected ModuleHandlerInterface $moduleHandler;
  protected EntityFieldManagerInterface $entityFieldManager;
  protected EntityTypeBundleInfoInterface $entityTypeBundleInfo;
  protected EntityTypeManagerInterface $entityTypeManager;
  protected ConfigFactoryInterface $configFactory;
  protected LanguageManagerInterface $languageManager;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->moduleHandler = $container->get('module_handler');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->entityTypeBundleInfo = $container->get('entity_type.bundle.info');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->configFactory = $container->get('config.factory');
    $instance->languageManager = $container->get('language_manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    // Analyze each category.
    $url_structure = $this->analyzeUrlStructure();
    $media_seo = $this->analyzeMediaSeo();
    $robots_txt = $this->analyzeRobotsTxt();

    // Build essential modules issues file (for faceted list).
    $essential_modules_issues = $this->buildEssentialModulesIssuesFile();

    // Build URL structure issues file (errors and warnings only).
    $url_structure_issues = $this->buildUrlStructureIssuesFile($url_structure);

    // Build media SEO issues file (warnings only - alt disabled).
    $media_seo_issues = $this->buildMediaSeoIssuesFile($media_seo);

    // Build robots.txt issues file (errors and warnings only).
    $robots_txt_issues = $this->buildRobotsTxtIssuesFile($robots_txt);

    // Calculate scores.
    $scores = $this->calculateScores($url_structure, $media_seo, $robots_txt);

    return [
      '_files' => [
        // Issue files for faceted lists (scoring).
        'essential_modules_issues' => $essential_modules_issues,
        'url_structure_issues' => $url_structure_issues,
        'media_seo_issues' => $media_seo_issues,
        'robots_txt_issues' => $robots_txt_issues,
        // Status files for tables (informational).
        'modules_status' => $this->createResult([], 0, 0, 0),
        'url_structure_status' => $url_structure,
        'media_seo_status' => $media_seo,
        'robots_txt_status' => $robots_txt,
      ],
      'score' => $scores,
    ];
  }

  /**
   * Builds the essential modules issues file.
   *
   * Contains only missing essential modules as errors.
   */
  protected function buildEssentialModulesIssuesFile(): array {
    $results = [];
    $errors = 0;

    foreach (self::ESSENTIAL_MODULES as $module => $info) {
      if (!$this->moduleHandler->moduleExists($module)) {
        $results[] = $this->createResultItem(
          'error',
          'ESSENTIAL_MODULE_MISSING',
          (string) $this->t('@label module is not installed', ['@label' => $info['label']]),
          [
            'module' => $module,
            'label' => $info['label'],
            'description' => $info['description'],
          ]
        );
        $errors++;
      }
    }

    return $this->createResult($results, $errors, 0, 0);
  }

  /**
   * Builds the URL structure issues file.
   *
   * Contains only errors and warnings (missing patterns, missing languages).
   */
  protected function buildUrlStructureIssuesFile(array $url_structure): array {
    $results = [];
    $errors = 0;
    $warnings = 0;

    // Filter only errors and warnings from the full analysis.
    foreach ($url_structure['results'] ?? [] as $item) {
      $severity = $item['severity'] ?? 'notice';
      if ($severity === 'error') {
        $results[] = $item;
        $errors++;
      }
      elseif ($severity === 'warning') {
        $results[] = $item;
        $warnings++;
      }
    }

    return $this->createResult($results, $errors, $warnings, 0);
  }

  /**
   * Builds the media SEO issues file.
   *
   * Contains only warnings (image fields with alt disabled).
   */
  protected function buildMediaSeoIssuesFile(array $media_seo): array {
    $results = [];
    $warnings = 0;

    // Filter only warnings (alt disabled).
    foreach ($media_seo['results'] ?? [] as $item) {
      $severity = $item['severity'] ?? 'notice';
      if ($severity === 'warning') {
        $results[] = $item;
        $warnings++;
      }
    }

    return $this->createResult($results, 0, $warnings, 0);
  }

  /**
   * Analyzes URL structure (Pathauto patterns).
   */
  protected function analyzeUrlStructure(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    $pathauto_installed = $this->moduleHandler->moduleExists('pathauto');

    if (!$pathauto_installed) {
      $results[] = $this->createResultItem(
        'error',
        'PATHAUTO_NOT_INSTALLED',
        (string) $this->t('Pathauto module is not installed. URLs will not be SEO-friendly.'),
        ['module' => 'pathauto']
      );
      $errors++;

      return $this->createResult($results, $errors, $warnings, $notices);
    }

    $results[] = $this->createResultItem(
      'notice',
      'PATHAUTO_INSTALLED',
      (string) $this->t('Pathauto module is installed.'),
      ['module' => 'pathauto']
    );
    $notices++;

    // Get all active languages.
    $languages = $this->languageManager->getLanguages();
    $langcodes = array_keys($languages);
    $is_multilingual = count($langcodes) > 1;

    // Get all patterns with detailed info.
    $patterns_data = $this->getPathautoPatterns();

    // Analyze each bundle+language combination.
    $bundle_analysis = [];
    $missing_combinations = [];
    $configured_combinations = [];

    foreach (self::SEO_ENTITY_TYPES as $entity_type_id => $entity_type_label) {
      if (!$this->entityTypeManager->hasDefinition($entity_type_id)) {
        continue;
      }

      $bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);

      foreach ($bundles as $bundle_id => $bundle_info) {
        $bundle_label = $bundle_info['label'] ?? $bundle_id;

        // Get pattern coverage for this bundle.
        $coverage = $this->getBundlePatternCoverage($entity_type_id, $bundle_id, $patterns_data, $langcodes);

        $bundle_key = "{$entity_type_id}:{$bundle_id}";
        $bundle_analysis[$bundle_key] = [
          'entity_type' => $entity_type_id,
          'entity_type_label' => $entity_type_label,
          'bundle' => $bundle_id,
          'bundle_label' => $bundle_label,
          'has_pattern' => $coverage['has_pattern'],
          'applies_to_all_bundles' => $coverage['applies_to_all_bundles'],
          'pattern_id' => $coverage['pattern_id'],
          'pattern_string' => $coverage['pattern_string'],
          'languages_covered' => $coverage['languages_covered'],
          'languages_missing' => $coverage['languages_missing'],
          'all_languages_covered' => $coverage['all_languages_covered'],
        ];

        if (!$coverage['has_pattern']) {
          // No pattern at all for this bundle.
          $results[] = $this->createResultItem(
            'warning',
            'PATHAUTO_PATTERN_MISSING',
            (string) $this->t('No URL pattern for @type: @bundle', [
              '@type' => $entity_type_label,
              '@bundle' => $bundle_label,
            ]),
            $bundle_analysis[$bundle_key]
          );
          $warnings++;
          $missing_combinations[] = $bundle_analysis[$bundle_key];
        }
        elseif ($is_multilingual && !$coverage['all_languages_covered']) {
          // Has pattern but not for all languages.
          foreach ($coverage['languages_missing'] as $langcode) {
            $lang_name = $languages[$langcode]->getName() ?? $langcode;
            $results[] = $this->createResultItem(
              'error',
              'PATHAUTO_LANGUAGE_MISSING',
              (string) $this->t('No URL pattern for @type: @bundle in language @lang', [
                '@type' => $entity_type_label,
                '@bundle' => $bundle_label,
                '@lang' => $lang_name,
              ]),
              [
                'entity_type' => $entity_type_id,
                'entity_type_label' => $entity_type_label,
                'bundle' => $bundle_id,
                'bundle_label' => $bundle_label,
                'language' => $langcode,
                'language_name' => $lang_name,
              ]
            );
            $errors++;
          }
          $missing_combinations[] = $bundle_analysis[$bundle_key];
        }
        else {
          // Fully configured.
          $results[] = $this->createResultItem(
            'notice',
            'PATHAUTO_PATTERN_EXISTS',
            (string) $this->t('URL pattern configured for @type: @bundle', [
              '@type' => $entity_type_label,
              '@bundle' => $bundle_label,
            ]),
            $bundle_analysis[$bundle_key]
          );
          $notices++;
          $configured_combinations[] = $bundle_analysis[$bundle_key];
        }
      }
    }

    $result = $this->createResult($results, $errors, $warnings, $notices);
    $result['stats'] = [
      'pathauto_installed' => $pathauto_installed,
      'is_multilingual' => $is_multilingual,
      'languages' => $langcodes,
      'language_names' => array_map(fn($l) => $l->getName(), $languages),
      'bundles_fully_configured' => count($configured_combinations),
      'bundles_with_issues' => count($missing_combinations),
      'total_bundles' => count($bundle_analysis),
    ];
    $result['patterns'] = $patterns_data;
    $result['bundle_analysis'] = $bundle_analysis;

    return $result;
  }

  /**
   * Gets all Pathauto patterns from config with detailed info.
   */
  protected function getPathautoPatterns(): array {
    $patterns = [];
    $config_names = $this->configFactory->listAll('pathauto.pattern.');

    foreach ($config_names as $config_name) {
      $config = $this->configFactory->get($config_name);
      $selection_criteria = $config->get('selection_criteria') ?? [];

      // Parse selection criteria to extract bundles and languages.
      $bundles = [];
      $langcodes = [];
      $has_bundle_condition = FALSE;
      $has_language_condition = FALSE;

      foreach ($selection_criteria as $criteria) {
        $criteria_id = $criteria['id'] ?? '';

        // Check for bundle condition (e.g., entity_bundle:node).
        if (str_starts_with($criteria_id, 'entity_bundle:')) {
          $has_bundle_condition = TRUE;
          if (!empty($criteria['bundles'])) {
            $bundles = array_merge($bundles, array_keys($criteria['bundles']));
          }
        }

        // Check for language condition.
        if ($criteria_id === 'language') {
          $has_language_condition = TRUE;
          if (!empty($criteria['langcodes'])) {
            $langcodes = array_merge($langcodes, array_keys(array_filter($criteria['langcodes'])));
          }
        }
      }

      $patterns[] = [
        'id' => $config->get('id'),
        'label' => $config->get('label'),
        'type' => $config->get('type'),
        'pattern' => $config->get('pattern'),
        'weight' => $config->get('weight') ?? 0,
        'selection_criteria' => $selection_criteria,
        'bundles' => array_unique($bundles),
        'has_bundle_condition' => $has_bundle_condition,
        'applies_to_all_bundles' => !$has_bundle_condition,
        'langcodes' => array_unique($langcodes),
        'has_language_condition' => $has_language_condition,
        'applies_to_all_languages' => !$has_language_condition || empty($langcodes),
      ];
    }

    // Sort by weight (lower weight = higher priority).
    usort($patterns, fn($a, $b) => $a['weight'] <=> $b['weight']);

    return $patterns;
  }

  /**
   * Gets pattern coverage for a specific bundle.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $bundle
   *   The bundle ID.
   * @param array $patterns
   *   All Pathauto patterns.
   * @param array $all_langcodes
   *   All active language codes.
   *
   * @return array
   *   Coverage information including which languages are covered.
   */
  protected function getBundlePatternCoverage(string $entity_type, string $bundle, array $patterns, array $all_langcodes): array {
    $pathauto_type_map = [
      'node' => 'canonical_entities:node',
      'taxonomy_term' => 'canonical_entities:taxonomy_term',
      'media' => 'canonical_entities:media',
      'user' => 'canonical_entities:user',
    ];

    $pathauto_type = $pathauto_type_map[$entity_type] ?? "canonical_entities:{$entity_type}";

    $coverage = [
      'has_pattern' => FALSE,
      'applies_to_all_bundles' => FALSE,
      'pattern_id' => NULL,
      'pattern_string' => NULL,
      'languages_covered' => [],
      'languages_missing' => $all_langcodes,
      'all_languages_covered' => FALSE,
    ];

    // Find patterns that apply to this bundle.
    $matching_patterns = [];

    foreach ($patterns as $pattern) {
      if ($pattern['type'] !== $pathauto_type) {
        continue;
      }

      // Check if pattern applies to this bundle.
      $applies_to_bundle = FALSE;

      if ($pattern['applies_to_all_bundles']) {
        // No bundle restriction - applies to all.
        $applies_to_bundle = TRUE;
      }
      elseif (in_array($bundle, $pattern['bundles'], TRUE)) {
        // Bundle explicitly listed.
        $applies_to_bundle = TRUE;
      }

      if ($applies_to_bundle) {
        $matching_patterns[] = $pattern;
      }
    }

    if (empty($matching_patterns)) {
      return $coverage;
    }

    // We have at least one pattern.
    $coverage['has_pattern'] = TRUE;

    // Use the first matching pattern (highest priority by weight).
    $primary_pattern = $matching_patterns[0];
    $coverage['pattern_id'] = $primary_pattern['id'];
    $coverage['pattern_string'] = $primary_pattern['pattern'];
    $coverage['applies_to_all_bundles'] = $primary_pattern['applies_to_all_bundles'];

    // Calculate language coverage from all matching patterns.
    $languages_covered = [];

    foreach ($matching_patterns as $pattern) {
      if ($pattern['applies_to_all_languages']) {
        // Pattern applies to all languages.
        $languages_covered = $all_langcodes;
        break;
      }
      else {
        // Pattern applies to specific languages.
        $languages_covered = array_merge($languages_covered, $pattern['langcodes']);
      }
    }

    $languages_covered = array_unique($languages_covered);
    $languages_missing = array_diff($all_langcodes, $languages_covered);

    $coverage['languages_covered'] = array_values($languages_covered);
    $coverage['languages_missing'] = array_values($languages_missing);
    $coverage['all_languages_covered'] = empty($languages_missing);

    return $coverage;
  }

  /**
   * Analyzes Media SEO (image alt/title configuration).
   *
   * Lists all image fields and checks if alt and title fields are enabled.
   */
  protected function analyzeMediaSeo(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    // Analyze image fields for alt/title configuration.
    $image_fields_data = $this->analyzeImageFieldsConfig();

    if (empty($image_fields_data['fields'])) {
      $results[] = $this->createResultItem(
        'notice',
        'NO_IMAGE_FIELDS',
        (string) $this->t('No image fields found in the system.'),
        []
      );
      $notices++;
    }
    else {
      foreach ($image_fields_data['fields'] as $field_data) {
        $alt_enabled = $field_data['alt_enabled'];
        $title_enabled = $field_data['title_enabled'];

        // Determine overall status for this field.
        if ($alt_enabled && $title_enabled) {
          $results[] = $this->createResultItem(
            'notice',
            'IMAGE_FIELD_OK',
            (string) $this->t('Image field @field (@type: @bundle) has alt and title enabled.', [
              '@field' => $field_data['field_label'],
              '@type' => $field_data['entity_type_label'],
              '@bundle' => $field_data['bundle_label'],
            ]),
            $field_data
          );
          $notices++;
        }
        elseif (!$alt_enabled) {
          // Alt disabled is a warning (important for accessibility/SEO).
          $results[] = $this->createResultItem(
            'warning',
            'IMAGE_ALT_DISABLED',
            (string) $this->t('Alt text disabled on @field (@type: @bundle)', [
              '@field' => $field_data['field_label'],
              '@type' => $field_data['entity_type_label'],
              '@bundle' => $field_data['bundle_label'],
            ]),
            $field_data
          );
          $warnings++;
        }
        else {
          // Only title disabled is a notice (less critical).
          $results[] = $this->createResultItem(
            'notice',
            'IMAGE_TITLE_DISABLED',
            (string) $this->t('Title field disabled on @field (@type: @bundle)', [
              '@field' => $field_data['field_label'],
              '@type' => $field_data['entity_type_label'],
              '@bundle' => $field_data['bundle_label'],
            ]),
            $field_data
          );
          $notices++;
        }
      }
    }

    $result = $this->createResult($results, $errors, $warnings, $notices);
    $result['stats'] = [
      'total_image_fields' => $image_fields_data['total'],
      'fields_with_alt_enabled' => $image_fields_data['alt_enabled_count'],
      'fields_without_alt_enabled' => $image_fields_data['total'] - $image_fields_data['alt_enabled_count'],
      'fields_with_title_enabled' => $image_fields_data['title_enabled_count'],
      'fields_without_title_enabled' => $image_fields_data['total'] - $image_fields_data['title_enabled_count'],
    ];
    $result['fields'] = $image_fields_data['fields'];

    return $result;
  }

  /**
   * Analyzes image field configurations.
   *
   * Checks if alt and title fields are enabled (user can fill them).
   */
  protected function analyzeImageFieldsConfig(): array {
    $fields = [];
    $alt_enabled_count = 0;
    $title_enabled_count = 0;

    foreach (self::SEO_ENTITY_TYPES as $entity_type_id => $entity_type_label) {
      if (!$this->entityTypeManager->hasDefinition($entity_type_id)) {
        continue;
      }

      $bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);

      foreach ($bundles as $bundle_id => $bundle_info) {
        try {
          $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle_id);

          foreach ($field_definitions as $field_name => $field_definition) {
            if ($field_definition->getType() !== 'image') {
              continue;
            }

            $settings = $field_definition->getSettings();
            // alt_field: whether the alt field is enabled (default is TRUE in Drupal).
            // title_field: whether the title field is enabled (default is FALSE).
            $alt_enabled = $settings['alt_field'] ?? TRUE;
            $title_enabled = !empty($settings['title_field']);

            if ($alt_enabled) {
              $alt_enabled_count++;
            }
            if ($title_enabled) {
              $title_enabled_count++;
            }

            $fields[] = [
              'entity_type' => $entity_type_id,
              'entity_type_label' => $entity_type_label,
              'bundle' => $bundle_id,
              'bundle_label' => $bundle_info['label'] ?? $bundle_id,
              'field_name' => $field_name,
              'field_label' => (string) $field_definition->getLabel(),
              'alt_enabled' => $alt_enabled,
              'title_enabled' => $title_enabled,
            ];
          }
        }
        catch (\Exception $e) {
          // Skip bundles that don't support fields.
        }
      }
    }

    return [
      'fields' => $fields,
      'total' => count($fields),
      'alt_enabled_count' => $alt_enabled_count,
      'title_enabled_count' => $title_enabled_count,
    ];
  }

  /**
   * Calculates scores for all factors.
   *
   * Factors:
   * - Essential Modules (40%): % of essential modules installed
   * - URL Structure (30%): % of bundles with Pathauto patterns (+ language coverage)
   * - Robots.txt (15%): Robots.txt file configuration quality
   * - Media SEO (15%): % of image fields with alt/title enabled
   *
   * Note: Recommended modules are notice-only and do NOT affect scoring.
   */
  protected function calculateScores(
    array $url_structure,
    array $media_seo,
    array $robots_txt,
  ): array {
    $factors = [];

    // Essential Modules score (50% weight).
    // Check all 4 essential modules: metatag, pathauto, simple_sitemap, redirect.
    $essential_installed = 0;
    $essential_total = count(self::ESSENTIAL_MODULES);
    $missing_modules = [];

    foreach (self::ESSENTIAL_MODULES as $module => $info) {
      if ($this->moduleHandler->moduleExists($module)) {
        $essential_installed++;
      }
      else {
        $missing_modules[] = $info['label'];
      }
    }

    $essential_score = $essential_total > 0
      ? (int) round(($essential_installed / $essential_total) * 100)
      : 100;

    if (empty($missing_modules)) {
      $essential_description = (string) $this->t('All @total essential modules installed', [
        '@total' => $essential_total,
      ]);
    }
    else {
      $essential_description = (string) $this->t('@installed/@total installed. Missing: @missing', [
        '@installed' => $essential_installed,
        '@total' => $essential_total,
        '@missing' => implode(', ', $missing_modules),
      ]);
    }

    $factors['essential_modules'] = [
      'score' => $essential_score,
      'weight' => self::SCORE_WEIGHTS['essential_modules'],
      'label' => (string) $this->t('Essential Modules'),
      'description' => $essential_description,
    ];

    // URL Structure score (35% weight).
    $url_stats = $url_structure['stats'] ?? [];
    $pathauto_installed = $url_stats['pathauto_installed'] ?? FALSE;

    if (!$pathauto_installed) {
      // Pathauto not installed - score is 0 for this factor.
      $url_score = 0;
      $url_description = (string) $this->t('Pathauto not installed');
    }
    else {
      $total = $url_stats['total_bundles'] ?? 1;
      $fully_configured = $url_stats['bundles_fully_configured'] ?? 0;
      $url_score = $total > 0 ? (int) round(($fully_configured / $total) * 100) : 100;

      $is_multilingual = $url_stats['is_multilingual'] ?? FALSE;
      if ($is_multilingual) {
        $url_description = (string) $this->t('@configured/@total bundles configured (all languages)', [
          '@configured' => $fully_configured,
          '@total' => $total,
        ]);
      }
      else {
        $url_description = (string) $this->t('@configured/@total bundles have URL patterns', [
          '@configured' => $fully_configured,
          '@total' => $total,
        ]);
      }
    }

    $factors['url_structure'] = [
      'score' => $url_score,
      'weight' => self::SCORE_WEIGHTS['url_structure'],
      'label' => (string) $this->t('URL Structure'),
      'description' => $url_description,
    ];

    // Media SEO score (15% weight).
    $media_stats = $media_seo['stats'] ?? [];
    $total_image_fields = $media_stats['total_image_fields'] ?? 0;

    if ($total_image_fields === 0) {
      $media_score = 100;
      $media_description = (string) $this->t('No image fields found');
    }
    else {
      $alt_enabled = $media_stats['fields_with_alt_enabled'] ?? 0;
      $title_enabled = $media_stats['fields_with_title_enabled'] ?? 0;

      // 70% weight for alt, 30% for title.
      $alt_percentage = ($alt_enabled / $total_image_fields) * 70;
      $title_percentage = ($title_enabled / $total_image_fields) * 30;
      $media_score = (int) round($alt_percentage + $title_percentage);

      $media_description = (string) $this->t('@alt/@total alt enabled, @title/@total title enabled', [
        '@alt' => $alt_enabled,
        '@title' => $title_enabled,
        '@total' => $total_image_fields,
      ]);
    }

    $factors['media_seo'] = [
      'score' => $media_score,
      'weight' => self::SCORE_WEIGHTS['media_seo'],
      'label' => (string) $this->t('Media SEO'),
      'description' => $media_description,
    ];

    // Robots.txt score (15% weight).
    $robots_stats = $robots_txt['stats'] ?? [];
    $robots_errors = $robots_txt['summary']['errors'] ?? 0;
    $robots_warnings = $robots_txt['summary']['warnings'] ?? 0;

    // Calculate score based on issues found.
    // File not found = -50 points (error).
    // Missing sitemap reference = -20 points (warning).
    // Blocking important paths = -15 points each (warning).
    if (!($robots_stats['file_exists'] ?? FALSE)) {
      $robots_score = 50;
      $robots_description = (string) $this->t('robots.txt file not found');
    }
    elseif ($robots_errors > 0 || $robots_warnings > 0) {
      // Each error = -25 points, each warning = -15 points.
      $robots_score = max(0, 100 - ($robots_errors * 25) - ($robots_warnings * 15));
      $robots_description = (string) $this->t('@errors error(s), @warnings warning(s) found', [
        '@errors' => $robots_errors,
        '@warnings' => $robots_warnings,
      ]);
    }
    else {
      $robots_score = 100;
      $robots_description = (string) $this->t('robots.txt properly configured');
    }

    $factors['robots_txt'] = [
      'score' => $robots_score,
      'weight' => self::SCORE_WEIGHTS['robots_txt'],
      'label' => (string) $this->t('Robots.txt'),
      'description' => $robots_description,
    ];

    return [
      'factors' => $factors,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      // Scoring sections - faceted lists.
      'essential_modules_issues' => [
        'label' => $this->t('Essential SEO Modules'),
        'description' => $this->t('Missing essential SEO modules that impact site indexing and URL management.'),
        'file_key' => 'essential_modules_issues',
        'affects_score' => TRUE,
        'score_factor_key' => 'essential_modules',
        'weight' => self::SCORE_WEIGHTS['essential_modules'],
      ],
      'url_structure_issues' => [
        'label' => $this->t('URL Pattern Issues'),
        'description' => $this->t('Entity bundles without proper Pathauto URL patterns or missing language coverage.'),
        'file_key' => 'url_structure_issues',
        'affects_score' => TRUE,
        'score_factor_key' => 'url_structure',
        'weight' => self::SCORE_WEIGHTS['url_structure'],
      ],
      'media_seo_issues' => [
        'label' => $this->t('Image Field Issues'),
        'description' => $this->t('Image fields missing alt text configuration, affecting accessibility and SEO.'),
        'file_key' => 'media_seo_issues',
        'affects_score' => TRUE,
        'score_factor_key' => 'media_seo',
        'weight' => self::SCORE_WEIGHTS['media_seo'],
      ],
      'robots_txt_issues' => [
        'label' => $this->t('Robots.txt Issues'),
        'description' => $this->t('Issues with the robots.txt file that may affect search engine crawling.'),
        'file_key' => 'robots_txt_issues',
        'affects_score' => TRUE,
        'score_factor_key' => 'robots_txt',
        'weight' => self::SCORE_WEIGHTS['robots_txt'],
      ],
      // Informational sections - tables.
      'modules_status' => [
        'label' => $this->t('SEO Modules Status'),
        'description' => $this->t('Overview of all essential and recommended SEO modules installation status.'),
        'file_key' => 'modules_status',
        'affects_score' => FALSE,
      ],
      'url_structure_status' => [
        'label' => $this->t('URL Patterns Overview'),
        'description' => $this->t('Complete list of entity bundles and their Pathauto URL pattern configuration.'),
        'file_key' => 'url_structure_status',
        'affects_score' => FALSE,
      ],
      'media_seo_status' => [
        'label' => $this->t('Image Fields Overview'),
        'description' => $this->t('Complete list of image fields and their alt/title configuration.'),
        'file_key' => 'media_seo_status',
        'affects_score' => FALSE,
      ],
      'robots_txt_status' => [
        'label' => $this->t('Robots.txt Analysis'),
        'description' => $this->t('Detailed analysis of robots.txt file content and configuration.'),
        'file_key' => 'robots_txt_status',
        'affects_score' => FALSE,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    $files = $data['_files'] ?? [];

    return match ($check_id) {
      // Issue sections (faceted lists).
      'essential_modules_issues' => $this->getEssentialModulesIssuesContent($files),
      'url_structure_issues' => $this->getUrlStructureIssuesContent($files),
      'media_seo_issues' => $this->getMediaSeoIssuesContent($files),
      'robots_txt_issues' => $this->getRobotsTxtIssuesContent($files),
      // Status sections (tables).
      'modules_status' => $this->getModulesStatusContent($files),
      'url_structure_status' => $this->getUrlStructureStatusContent($files),
      'media_seo_status' => $this->getMediaSeoStatusContent($files),
      'robots_txt_status' => $this->getRobotsTxtStatusContent($files),
      default => [],
    };
  }

  /**
   * Gets the essential modules issues content (faceted list).
   */
  protected function getEssentialModulesIssuesContent(array $files): array {
    $results = $files['essential_modules_issues']['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('All essential SEO modules are installed.'),
          'success'
        ),
      ];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $module_name = $details['label'] ?? '';
      $module_machine = $details['module'] ?? '';

      $issues[] = $this->ui->issue([
        'severity' => $item['severity'] ?? 'error',
        'code' => $item['code'] ?? 'ESSENTIAL_MODULE_MISSING',
        // file: shown in summary (visible) - module name.
        'file' => $module_name,
        // label: shown when expanded - descriptive title.
        'label' => (string) $this->t('Essential Module Missing'),
        // description: shown when expanded - explanation.
        'description' => (string) $this->t('@description This module is essential for SEO. Install it with: <code>composer require drupal/@machine</code>', [
          '@description' => $details['description'] ?? '',
          '@machine' => $module_machine,
        ]),
        'tags' => ['module'],
      ]);
    }

    return [
      'list' => $this->ui->issueList($issues),
    ];
  }

  /**
   * Gets the URL structure issues content (faceted list).
   */
  protected function getUrlStructureIssuesContent(array $files): array {
    $results = $files['url_structure_issues']['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('All entity bundles have properly configured URL patterns.'),
          'success'
        ),
      ];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $code = $item['code'] ?? '';

      // Build tags based on issue type.
      $tags = [];
      if (!empty($details['entity_type'])) {
        $tags[] = $details['entity_type'];
      }
      if ($code === 'PATHAUTO_LANGUAGE_MISSING') {
        $tags[] = 'language';
      }
      elseif ($code === 'PATHAUTO_PATTERN_MISSING') {
        $tags[] = 'missing_pattern';
      }

      // Build file (visible in summary) from entity type and bundle.
      $entity_type_label = $details['entity_type_label'] ?? $details['entity_type'] ?? '';
      $bundle_label = $details['bundle_label'] ?? $details['bundle'] ?? '';
      $file = $entity_type_label . ': ' . $bundle_label;

      // Build label and description based on issue type.
      if ($code === 'PATHAUTO_LANGUAGE_MISSING') {
        $language_name = $details['language_name'] ?? $details['language'] ?? '';
        $label = (string) $this->t('URL Pattern Missing for Language');
        $description = (string) $this->t('This bundle does not have a URL pattern configured for the @lang language. URLs in this language will not be SEO-friendly and may use internal node IDs instead of meaningful paths.', [
          '@lang' => $language_name,
        ]);
      }
      else {
        $label = (string) $this->t('URL Pattern Missing');
        $description = (string) $this->t('This bundle does not have a Pathauto URL pattern configured. Content of this type will not have SEO-friendly URLs, potentially hurting search engine rankings.');
      }

      $issues[] = $this->ui->issue([
        'severity' => $item['severity'] ?? 'warning',
        'code' => $code,
        // file: shown in summary (visible) - entity:bundle.
        'file' => $file,
        // label: shown when expanded - descriptive title.
        'label' => $label,
        // description: shown when expanded - explanation.
        'description' => $description,
        'tags' => $tags,
      ]);
    }

    return [
      'list' => $this->ui->issueList($issues),
    ];
  }

  /**
   * Gets the media SEO issues content (faceted list).
   */
  protected function getMediaSeoIssuesContent(array $files): array {
    $results = $files['media_seo_issues']['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('All image fields have alt text properly configured.'),
          'success'
        ),
      ];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];

      // file (visible in summary): field name with context.
      $field_label = $details['field_label'] ?? $details['field_name'] ?? '';
      $entity_type_label = $details['entity_type_label'] ?? '';
      $bundle_label = $details['bundle_label'] ?? '';
      $file = $field_label . ' (' . $entity_type_label . ': ' . $bundle_label . ')';

      $issues[] = $this->ui->issue([
        'severity' => $item['severity'] ?? 'warning',
        'code' => $item['code'] ?? 'IMAGE_ALT_DISABLED',
        // file: shown in summary (visible) - field name with context.
        'file' => $file,
        // label: shown when expanded - descriptive title.
        'label' => (string) $this->t('Alt Text Disabled'),
        // description: shown when expanded - explanation.
        'description' => (string) $this->t('This image field does not have alt text enabled. Alt text is essential for accessibility (screen readers) and SEO. Search engines use alt text to understand image content. Enable it in the field settings.'),
        'tags' => [$details['entity_type'] ?? 'unknown'],
      ]);
    }

    return [
      'list' => $this->ui->issueList($issues),
    ];
  }

  /**
   * Gets the modules status content (table).
   */
  protected function getModulesStatusContent(array $files): array {
    // Essential modules table.
    $essential_headers = [
      (string) $this->t('Module'),
      (string) $this->t('Machine Name'),
      (string) $this->t('Description'),
      (string) $this->t('Status'),
    ];

    $essential_rows = [];
    foreach (self::ESSENTIAL_MODULES as $module => $info) {
      $installed = $this->moduleHandler->moduleExists($module);
      $essential_rows[] = $this->ui->row([
        $info['label'],
        $this->ui->cell('<code>' . $module . '</code>'),
        $info['description'],
        $this->ui->cell(
          $installed ? (string) $this->t('Installed') : (string) $this->t('MISSING'),
          ['status' => $installed ? 'ok' : 'error']
        ),
      ], $installed ? NULL : 'error');
    }

    // Recommended modules table.
    $recommended_rows = [];
    foreach (self::RECOMMENDED_MODULES as $module => $info) {
      $installed = $this->moduleHandler->moduleExists($module);
      $recommended_rows[] = $this->ui->row([
        $info['label'],
        $this->ui->cell('<code>' . $module . '</code>'),
        $info['description'],
        $this->ui->cell(
          $installed ? (string) $this->t('Installed') : (string) $this->t('Recommended'),
          ['status' => $installed ? 'ok' : 'notice']
        ),
      ]);
    }

    return [
      'essential_title' => ['#markup' => '<h4>' . (string) $this->t('Essential Modules') . '</h4>'],
      'essential_table' => $this->ui->table($essential_headers, $essential_rows),
      'recommended_title' => ['#markup' => '<h4>' . (string) $this->t('Recommended Modules') . '</h4>'],
      'recommended_table' => $this->ui->table($essential_headers, $recommended_rows),
    ];
  }

  /**
   * Gets the URL structure status content (table).
   */
  protected function getUrlStructureStatusContent(array $files): array {
    $url_data = $files['url_structure_status'] ?? [];
    $stats = $url_data['stats'] ?? [];
    $bundle_analysis = $url_data['bundle_analysis'] ?? [];

    $pathauto_installed = $stats['pathauto_installed'] ?? FALSE;
    $is_multilingual = $stats['is_multilingual'] ?? FALSE;
    $language_names = $stats['language_names'] ?? [];

    if (!$pathauto_installed) {
      return [
        'error' => $this->ui->message(
          (string) $this->t('Pathauto module is not installed. Install it to generate SEO-friendly URLs automatically.'),
          'error'
        ),
      ];
    }

    // Summary.
    $summary_text = (string) $this->t('@configured of @total bundles fully configured.', [
      '@configured' => $stats['bundles_fully_configured'] ?? 0,
      '@total' => $stats['total_bundles'] ?? 0,
    ]);

    if ($is_multilingual) {
      $summary_text .= ' ' . (string) $this->t('Site has @count languages: @langs.', [
        '@count' => count($language_names),
        '@langs' => implode(', ', $language_names),
      ]);
    }

    // Build headers.
    $headers = [
      (string) $this->t('Entity Type'),
      (string) $this->t('Bundle'),
      (string) $this->t('Machine Name'),
      (string) $this->t('Pattern'),
    ];

    if ($is_multilingual) {
      $headers[] = (string) $this->t('Languages');
    }

    $headers[] = (string) $this->t('Status');

    // Build rows.
    $rows = [];
    foreach ($bundle_analysis as $data) {
      $has_pattern = $data['has_pattern'] ?? FALSE;
      $all_languages = $data['all_languages_covered'] ?? FALSE;

      // Determine status and row severity.
      if (!$has_pattern) {
        $status = (string) $this->t('No pattern');
        $status_cell = $this->ui->cell($status, ['status' => 'warning']);
        $row_severity = 'warning';
      }
      elseif ($is_multilingual && !$all_languages) {
        $missing_langs = $data['languages_missing'] ?? [];
        $missing_names = array_map(
          fn($code) => $language_names[$code] ?? $code,
          $missing_langs
        );
        $status = (string) $this->t('Missing: @langs', ['@langs' => implode(', ', $missing_names)]);
        $status_cell = $this->ui->cell($status, ['status' => 'error']);
        $row_severity = 'error';
      }
      else {
        $status = (string) $this->t('OK');
        $status_cell = $this->ui->cell($status, ['status' => 'ok']);
        $row_severity = NULL;
      }

      // Pattern info.
      $pattern_info = '-';
      if ($has_pattern) {
        $pattern_info = $data['pattern_string'] ?? '';
        if ($data['applies_to_all_bundles'] ?? FALSE) {
          $pattern_info .= ' <small>(' . (string) $this->t('all bundles') . ')</small>';
        }
      }

      // Languages covered.
      $languages_cell = '-';
      if ($has_pattern) {
        if ($is_multilingual) {
          $covered_langs = $data['languages_covered'] ?? [];
          if (count($covered_langs) === count($language_names)) {
            $languages_cell = (string) $this->t('All');
          }
          else {
            $covered_names = array_map(
              fn($code) => $language_names[$code] ?? $code,
              $covered_langs
            );
            $languages_cell = implode(', ', $covered_names);
          }
        }
        else {
          $languages_cell = (string) $this->t('All');
        }
      }

      $cells = [
        $data['entity_type_label'] ?? '',
        $data['bundle_label'] ?? '',
        $this->ui->cell('<code>' . ($data['bundle'] ?? '') . '</code>'),
        $this->ui->cell($pattern_info),
      ];

      // Add languages column only for multilingual sites.
      if ($is_multilingual) {
        $cells[] = $languages_cell;
      }

      $cells[] = $status_cell;

      $rows[] = $this->ui->row($cells, $row_severity);
    }

    return [
      'summary' => $this->ui->message($summary_text, 'info'),
      'table' => $this->ui->table($headers, $rows),
    ];
  }

  /**
   * Gets the media SEO status content (table).
   */
  protected function getMediaSeoStatusContent(array $files): array {
    $media_data = $files['media_seo_status'] ?? [];
    $stats = $media_data['stats'] ?? [];
    $fields = $media_data['fields'] ?? [];

    $total = $stats['total_image_fields'] ?? 0;

    if ($total === 0) {
      return [
        'no_fields' => $this->ui->message(
          (string) $this->t('No image fields found in the system.'),
          'info'
        ),
      ];
    }

    // Summary message.
    $summary_text = (string) $this->t('@total image fields found. @alt have alt enabled, @title have title enabled.', [
      '@total' => $total,
      '@alt' => $stats['fields_with_alt_enabled'] ?? 0,
      '@title' => $stats['fields_with_title_enabled'] ?? 0,
    ]);

    // Build headers.
    $headers = [
      (string) $this->t('Entity Type'),
      (string) $this->t('Bundle'),
      (string) $this->t('Field'),
      (string) $this->t('Machine Name'),
      (string) $this->t('Alt Text'),
      (string) $this->t('Title'),
    ];

    // Build rows.
    $rows = [];
    foreach ($fields as $field_data) {
      $alt_enabled = $field_data['alt_enabled'] ?? FALSE;
      $title_enabled = $field_data['title_enabled'] ?? FALSE;

      // Row severity: warning if alt disabled, none otherwise.
      $row_severity = !$alt_enabled ? 'warning' : NULL;

      $rows[] = $this->ui->row([
        $field_data['entity_type_label'] ?? '',
        $field_data['bundle_label'] ?? '',
        $field_data['field_label'] ?? '',
        $this->ui->cell('<code>' . ($field_data['field_name'] ?? '') . '</code>'),
        $this->ui->cell(
          $alt_enabled ? (string) $this->t('Enabled') : (string) $this->t('Disabled'),
          ['status' => $alt_enabled ? 'ok' : 'warning']
        ),
        $this->ui->cell(
          $title_enabled ? (string) $this->t('Enabled') : (string) $this->t('Disabled'),
          ['status' => $title_enabled ? 'ok' : 'warning']
        ),
      ], $row_severity);
    }

    return [
      'summary' => $this->ui->message($summary_text, 'info'),
      'table' => $this->ui->table($headers, $rows),
    ];
  }

  /**
   * Analyzes the robots.txt file.
   *
   * Checks for:
   * - File existence
   * - Sitemap reference
   * - Blocking important paths
   * - Common misconfigurations
   *
   * @return array
   *   The analysis results.
   */
  protected function analyzeRobotsTxt(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    // Try to find robots.txt in common locations.
    $robots_paths = [
      DRUPAL_ROOT . '/robots.txt',
      DRUPAL_ROOT . '/../robots.txt',
    ];

    $robots_content = NULL;
    $robots_path = NULL;

    foreach ($robots_paths as $path) {
      $real_path = realpath($path);
      if ($real_path && file_exists($real_path) && is_readable($real_path)) {
        $content = file_get_contents($real_path);
        if ($content !== FALSE) {
          $robots_content = $content;
          $robots_path = $real_path;
          break;
        }
      }
    }

    // Check if file exists.
    if ($robots_content === NULL) {
      $results[] = $this->createResultItem(
        'error',
        'ROBOTS_NOT_FOUND',
        (string) $this->t('robots.txt file not found. Search engines may have unrestricted access to all paths.'),
        ['searched_paths' => $robots_paths]
      );
      $errors++;

      $result = $this->createResult($results, $errors, $warnings, $notices);
      $result['stats'] = [
        'file_exists' => FALSE,
        'file_path' => NULL,
        'file_size' => 0,
      ];
      return $result;
    }

    // Parse the robots.txt content.
    $lines = explode("\n", $robots_content);
    $has_sitemap = FALSE;
    $has_user_agent = FALSE;
    $has_disallow = FALSE;
    $sitemaps = [];
    $disallowed_paths = [];
    $blocked_important_paths = [];

    // Important paths that should NOT be blocked.
    $important_paths = [
      '/' => 'Root path - blocks entire site',
      '/node/' => 'Content nodes',
      '/taxonomy/' => 'Taxonomy pages',
      '/user/' => 'User pages (may be intentional)',
    ];

    // Paths that SHOULD be blocked in production.
    $should_block_paths = [
      '/admin/' => 'Admin pages',
      '/node/add/' => 'Content creation',
      '/user/login' => 'Login page',
      '/user/register' => 'Registration page',
      '/search/' => 'Search pages (optional)',
    ];

    $blocked_should_block = [];

    foreach ($lines as $line_number => $line) {
      $trimmed_line = trim($line);

      // Skip empty lines and comments.
      if (empty($trimmed_line) || str_starts_with($trimmed_line, '#')) {
        continue;
      }

      // Parse directives.
      if (preg_match('/^User-agent:\s*(.+)$/i', $trimmed_line, $matches)) {
        $has_user_agent = TRUE;
      }
      elseif (preg_match('/^Sitemap:\s*(.+)$/i', $trimmed_line, $matches)) {
        $has_sitemap = TRUE;
        $sitemaps[] = trim($matches[1]);
      }
      elseif (preg_match('/^Disallow:\s*(.*)$/i', $trimmed_line, $matches)) {
        $has_disallow = TRUE;
        $path = trim($matches[1]);
        if (!empty($path)) {
          $disallowed_paths[] = $path;

          // Check if blocking important paths.
          foreach ($important_paths as $important => $description) {
            if ($path === $important || ($important !== '/' && str_starts_with($important, $path))) {
              $blocked_important_paths[$path] = [
                'description' => $description,
                'line_number' => $line_number,
                'line_content' => $line,
              ];
            }
          }

          // Check if blocking paths that should be blocked.
          foreach ($should_block_paths as $should_block => $description) {
            if (str_starts_with($should_block, $path) || $path === $should_block) {
              $blocked_should_block[$should_block] = TRUE;
            }
          }
        }
      }
    }

    // Check for sitemap reference.
    if (!$has_sitemap) {
      $results[] = $this->createResultItem(
        'warning',
        'ROBOTS_NO_SITEMAP',
        (string) $this->t('No Sitemap directive found in robots.txt. Add a Sitemap line to help search engines discover your content.'),
        ['recommendation' => 'Sitemap: https://example.com/sitemap.xml']
      );
      $warnings++;
    }
    else {
      $results[] = $this->createResultItem(
        'notice',
        'ROBOTS_HAS_SITEMAP',
        (string) $this->t('Sitemap reference found: @sitemaps', [
          '@sitemaps' => implode(', ', $sitemaps),
        ]),
        ['sitemaps' => $sitemaps]
      );
      $notices++;
    }

    // Check for User-agent directive.
    if (!$has_user_agent) {
      $results[] = $this->createResultItem(
        'warning',
        'ROBOTS_NO_USER_AGENT',
        (string) $this->t('No User-agent directive found. Robots.txt may not be effective.'),
        ['recommendation' => 'Add "User-agent: *" as the first directive']
      );
      $warnings++;
    }

    // Check for blocked important paths.
    foreach ($blocked_important_paths as $path => $info) {
      // Root path blocking is an error.
      $severity = ($path === '/') ? 'error' : 'warning';

      // Build code snippet with context (2 lines before and after).
      $line_number = $info['line_number'];
      $code_snippet = $this->buildCodeSnippet($lines, $line_number, 2);

      $results[] = $this->createResultItem(
        $severity,
        'ROBOTS_BLOCKING_IMPORTANT',
        (string) $this->t('Blocking important path: @path - @description', [
          '@path' => $path,
          '@description' => $info['description'],
        ]),
        [
          'path' => $path,
          'description' => $info['description'],
          'line_number' => $line_number + 1,
          'line_content' => $info['line_content'],
          'code_snippet' => $code_snippet,
        ]
      );
      if ($severity === 'error') {
        $errors++;
      }
      else {
        $warnings++;
      }
    }

    // Check if admin paths are blocked (good practice).
    if (empty($blocked_should_block)) {
      $results[] = $this->createResultItem(
        'notice',
        'ROBOTS_ADMIN_NOT_BLOCKED',
        (string) $this->t('Admin and sensitive paths are not explicitly blocked. Consider adding Disallow rules for /admin/, /node/add/, etc.'),
        ['suggested_blocks' => array_keys($should_block_paths)]
      );
      $notices++;
    }

    // Add general info about the file.
    $results[] = $this->createResultItem(
      'notice',
      'ROBOTS_FILE_INFO',
      (string) $this->t('robots.txt found with @disallow disallow rules.', [
        '@disallow' => count($disallowed_paths),
      ]),
      [
        'path' => $robots_path,
        'disallowed_paths' => $disallowed_paths,
        'sitemaps' => $sitemaps,
      ]
    );
    $notices++;

    $result = $this->createResult($results, $errors, $warnings, $notices);
    $result['stats'] = [
      'file_exists' => TRUE,
      'file_path' => $robots_path,
      'file_size' => strlen($robots_content),
      'has_sitemap' => $has_sitemap,
      'has_user_agent' => $has_user_agent,
      'sitemaps_count' => count($sitemaps),
      'disallow_count' => count($disallowed_paths),
    ];
    $result['content'] = $robots_content;
    $result['sitemaps'] = $sitemaps;
    $result['disallowed_paths'] = $disallowed_paths;

    return $result;
  }

  /**
   * Builds the robots.txt issues file (errors and warnings only).
   *
   * @param array $robots_txt
   *   The full robots.txt analysis.
   *
   * @return array
   *   Filtered results containing only errors and warnings.
   */
  protected function buildRobotsTxtIssuesFile(array $robots_txt): array {
    $results = [];
    $errors = 0;
    $warnings = 0;

    foreach ($robots_txt['results'] ?? [] as $item) {
      $severity = $item['severity'] ?? 'notice';
      if ($severity === 'error') {
        $results[] = $item;
        $errors++;
      }
      elseif ($severity === 'warning') {
        $results[] = $item;
        $warnings++;
      }
    }

    return $this->createResult($results, $errors, $warnings, 0);
  }

  /**
   * Gets the robots.txt issues content (faceted list).
   *
   * @param array $files
   *   The analysis files.
   *
   * @return array
   *   Render array for the issues.
   */
  protected function getRobotsTxtIssuesContent(array $files): array {
    $results = $files['robots_txt_issues']['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('robots.txt is properly configured with no issues.'),
          'success'
        ),
      ];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $code = $item['code'] ?? '';
      $issue_code_snippet = NULL;

      // Determine label and description based on issue type.
      switch ($code) {
        case 'ROBOTS_NOT_FOUND':
          $file = 'robots.txt';
          $label = (string) $this->t('File Not Found');
          $description = (string) $this->t('The robots.txt file does not exist. Create one in your web root to control search engine crawling. Without it, all paths are accessible to crawlers.');
          $tags = ['missing', 'critical'];
          break;

        case 'ROBOTS_NO_SITEMAP':
          $file = 'Sitemap directive';
          $label = (string) $this->t('Missing Sitemap Reference');
          $description = (string) $this->t('Add a Sitemap directive to help search engines discover your content. Example: <code>Sitemap: https://yoursite.com/sitemap.xml</code>');
          $tags = ['sitemap', 'seo'];
          break;

        case 'ROBOTS_NO_USER_AGENT':
          $file = 'User-agent directive';
          $label = (string) $this->t('Missing User-agent');
          $description = (string) $this->t('The robots.txt should start with a User-agent directive. Add <code>User-agent: *</code> to apply rules to all crawlers.');
          $tags = ['syntax'];
          break;

        case 'ROBOTS_BLOCKING_IMPORTANT':
          $path = $details['path'] ?? '';
          $file = 'robots.txt:' . ($details['line_number'] ?? '');
          $label = (string) $this->t('Blocking Important Path: @path', ['@path' => $path]);
          $description = (string) $this->t('This disallow rule blocks @desc. If this is unintentional, remove or modify this rule. Blocking important content paths hurts SEO.', [
            '@desc' => $details['description'] ?? $path,
          ]);
          $tags = ['blocking', 'seo'];

          // Build code snippet render array if available.
          $code_context = $details['code_snippet'] ?? NULL;
          if ($code_context && !empty($code_context['lines'])) {
            $issue_code_snippet = $this->ui->code(
              $code_context['lines'],
              [
                'highlight_line' => $code_context['highlight_line'] ?? NULL,
                'severity' => $item['severity'] ?? 'warning',
              ]
            );
          }
          break;

        default:
          $file = 'robots.txt';
          $label = $item['message'] ?? '';
          $description = '';
          $tags = ['other'];
      }

      $issues[] = $this->ui->issue([
        'severity' => $item['severity'] ?? 'warning',
        'code' => $code,
        'file' => $file,
        'label' => $label,
        'description' => $description,
        'tags' => $tags,
        'code_snippet' => $issue_code_snippet,
      ]);
    }

    return [
      'list' => $this->ui->issueList($issues),
    ];
  }

  /**
   * Gets the robots.txt status content (informational table).
   *
   * @param array $files
   *   The analysis files.
   *
   * @return array
   *   Render array for the status table.
   */
  protected function getRobotsTxtStatusContent(array $files): array {
    $data = $files['robots_txt_status'] ?? [];
    $stats = $data['stats'] ?? [];

    if (!($stats['file_exists'] ?? FALSE)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('robots.txt file not found. Create one in your web root to control search engine access.'),
          'warning'
        ),
      ];
    }

    $content = [];

    // Summary info.
    $summary_text = (string) $this->t('File found at @path. @disallow disallow rules, @sitemaps sitemap reference(s).', [
      '@path' => $stats['file_path'] ?? 'unknown',
      '@disallow' => $stats['disallow_count'] ?? 0,
      '@sitemaps' => $stats['sitemaps_count'] ?? 0,
    ]);
    $content['summary'] = $this->ui->message($summary_text, 'info');

    // Sitemaps table.
    $sitemaps = $data['sitemaps'] ?? [];
    if (!empty($sitemaps)) {
      $content['sitemaps_title'] = ['#markup' => '<h4>' . (string) $this->t('Sitemap References') . '</h4>'];
      $sitemap_rows = [];
      foreach ($sitemaps as $sitemap) {
        $sitemap_link = '<a href="' . htmlspecialchars($sitemap) . '" target="_blank">' . htmlspecialchars($sitemap) . '</a>';
        $sitemap_rows[] = $this->ui->row([
          $this->ui->cell($sitemap_link),
        ]);
      }
      $content['sitemaps_table'] = $this->ui->table(
        [(string) $this->t('Sitemap URL')],
        $sitemap_rows
      );
    }

    // Disallow rules table.
    $disallowed = $data['disallowed_paths'] ?? [];
    if (!empty($disallowed)) {
      $content['disallow_title'] = ['#markup' => '<h4>' . (string) $this->t('Disallow Rules') . '</h4>'];
      $disallow_rows = [];
      foreach ($disallowed as $path) {
        $disallow_rows[] = $this->ui->row([
          $this->ui->cell('<code>' . htmlspecialchars($path) . '</code>'),
        ]);
      }
      $content['disallow_table'] = $this->ui->table(
        [(string) $this->t('Blocked Path')],
        $disallow_rows
      );
    }

    // Raw content preview.
    $raw_content = $data['content'] ?? '';
    if (!empty($raw_content)) {
      $content['raw_title'] = ['#markup' => '<h4>' . (string) $this->t('File Content') . '</h4>'];
      $content['raw_content'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['audit-code-block']],
        'code' => [
          '#markup' => '<pre><code>' . htmlspecialchars($raw_content) . '</code></pre>',
        ],
      ];
    }

    return $content;
  }

  /**
   * Builds a code snippet with context lines.
   *
   * @param array $lines
   *   All lines of the file.
   * @param int $target_line
   *   The target line number (0-indexed).
   * @param int $context
   *   Number of lines to show before and after.
   *
   * @return array
   *   Array with 'lines' containing line data and 'highlight_line' for the target.
   */
  protected function buildCodeSnippet(array $lines, int $target_line, int $context = 2): array {
    $total_lines = count($lines);
    $start = max(0, $target_line - $context);
    $end = min($total_lines - 1, $target_line + $context);

    $snippet_lines = [];
    for ($i = $start; $i <= $end; $i++) {
      $snippet_lines[] = [
        'number' => $i + 1,
        'content' => $lines[$i],
        'is_highlight' => ($i === $target_line),
      ];
    }

    return [
      'lines' => $snippet_lines,
      'highlight_line' => $target_line + 1,
    ];
  }

}
