<?php

declare(strict_types=1);

namespace Drupal\audit_status\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Analyzes Drupal system status.
 */
#[AuditAnalyzer(
  id: 'status',
  label: new TranslatableMarkup('Status Audit'),
  description: new TranslatableMarkup('Analyzes PHP version, Drupal version, database version, and system status.'),
  menu_title: new TranslatableMarkup('Status'),
  output_directory: 'status',
  weight: 4,
)]
class StatusAnalyzer extends AuditAnalyzerBase {

  /**
   * PHP compatibility matrix: Drupal version => [supported PHP versions].
   * @see https://www.drupal.org/docs/system-requirements/php-requirements
   */
  protected const PHP_COMPATIBILITY = [
    '10.3' => ['8.1', '8.2', '8.3'],
    '10.4' => ['8.2', '8.3', '8.4'],
    '10.5' => ['8.2', '8.3', '8.4'],
    '10.6' => ['8.2', '8.3', '8.4'],
    '11.0' => ['8.3'],
    '11.1' => ['8.3', '8.4'],
    '11.2' => ['8.3', '8.4'],
    '11.3' => ['8.3', '8.4', '8.5'],
    '12.0' => ['8.5'],
  ];

  /**
   * MySQL/MariaDB minimum versions by Drupal major version.
   * @see https://www.drupal.org/docs/system-requirements/database-server-requirements
   */
  protected const MYSQL_REQUIREMENTS = [
    '10' => ['mysql' => '5.7.8', 'mariadb' => '10.3.7'],
    '11' => ['mysql' => '8.0.0', 'mariadb' => '10.6.0'],
    '12' => ['mysql' => '8.0.0', 'mariadb' => '10.11.0'],
  ];

  /**
   * PostgreSQL minimum versions by Drupal major version.
   * @see https://www.drupal.org/docs/system-requirements/database-server-requirements
   */
  protected const PGSQL_REQUIREMENTS = [
    '10' => '12.0',
    '11' => '16.0',
    '12' => '18.0',
  ];

  /**
   * SQLite minimum versions by Drupal major version.
   * @see https://www.drupal.org/docs/system-requirements/database-server-requirements
   */
  protected const SQLITE_REQUIREMENTS = [
    '10' => '3.26',
    '11' => '3.45',
    '12' => '3.45',
  ];

  /**
   * Score weights for different factors.
   */
  protected const SCORE_WEIGHTS = [
    'php_version' => 25,
    'database_version' => 25,
    'drupal_requirements' => 30,
    'cron_status' => 10,
    'update_status' => 10,
  ];

  protected Connection $database;
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->database = $container->get('database');
    $instance->moduleHandler = $container->get('module_handler');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    $drupal_version = \Drupal::VERSION;
    $drupal_major = $this->getDrupalMajorVersion($drupal_version);
    $drupal_minor = $this->getDrupalMinorVersion($drupal_version);

    // Analyze each area separately.
    $php_analysis = $this->analyzePhpVersion($drupal_minor);
    $db_analysis = $this->analyzeDatabaseVersion($drupal_major);
    $req_analysis = $this->analyzeSystemRequirements();

    // Calculate scores.
    $scores = $this->calculateScores($php_analysis, $db_analysis, $req_analysis);

    // Get system info.
    $system_info = $this->getSystemInfo($drupal_version, $drupal_major, $drupal_minor);

    // Structure output using _files pattern.
    return [
      '_files' => [
        'system_info' => $this->createResult([], 0, 0, 0),
        'php_version' => $this->createResult(
          $php_analysis['results'],
          $php_analysis['errors'],
          $php_analysis['warnings'],
          $php_analysis['notices']
        ),
        'database_version' => $this->createResult(
          $db_analysis['results'],
          $db_analysis['errors'],
          $db_analysis['warnings'],
          $db_analysis['notices']
        ),
        'requirements' => $this->createResult(
          $req_analysis['results'],
          $req_analysis['errors'],
          $req_analysis['warnings'],
          $req_analysis['notices']
        ),
      ],
      'score' => $scores,
      'system_info' => $system_info,
      'php_analysis' => $php_analysis,
      'db_analysis' => $db_analysis,
      'req_analysis' => $req_analysis,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      'system_info' => [
        'label' => $this->t('System Information'),
        'description' => $this->t('Overview of your system configuration including Drupal, PHP, database, and web server versions.'),
        'file_types' => [],
        'affects_score' => FALSE,
        'file_key' => 'system_info',
      ],
      'php_version' => [
        'label' => $this->t('PHP Version'),
        'description' => $this->t('Checks PHP version compatibility with your Drupal installation.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'php_version',
        'score_factor_key' => 'php_version',
        'weight' => self::SCORE_WEIGHTS['php_version'],
      ],
      'database_version' => [
        'label' => $this->t('Database Version'),
        'description' => $this->t('Checks database version compatibility with your Drupal installation.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'database_version',
        'score_factor_key' => 'database_version',
        'weight' => self::SCORE_WEIGHTS['database_version'],
      ],
      'requirements' => [
        'label' => $this->t('System Requirements'),
        'description' => $this->t('Analyzes Drupal system requirements and status reports.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'requirements',
        'score_factor_key' => 'drupal_requirements',
        'weight' => self::SCORE_WEIGHTS['drupal_requirements'],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    return match ($check_id) {
      'system_info' => $this->buildSystemInfoContent($data),
      'php_version' => $this->buildPhpVersionContent($data),
      'database_version' => $this->buildDatabaseVersionContent($data),
      'requirements' => $this->buildRequirementsContent($data),
      default => [],
    };
  }

  /**
   * Builds content for the system info check.
   */
  protected function buildSystemInfoContent(array $data): array {
    $system_info = $data['system_info'] ?? [];
    if (empty($system_info)) {
      return [];
    }

    $php_version = $system_info['php']['version'] ?? '-';
    $supported = $system_info['php']['supported_for_drupal'] ?? [];
    $php_major_minor = $system_info['php']['major_minor'] ?? '';
    $is_supported = in_array($php_major_minor, $supported, TRUE);

    $headers = [
      $this->ui->header((string) $this->t('Property'), 'left', '30%'),
      $this->ui->header((string) $this->t('Value'), 'left'),
    ];

    $rows = [];

    // Drupal version.
    $rows[] = [
      $this->ui->itemName((string) $this->t('Drupal Version')),
      $system_info['drupal']['version'] ?? '-',
    ];

    // PHP version.
    $rows[] = $this->ui->row([
      $this->ui->itemName((string) $this->t('PHP Version')),
      $this->ui->cell(
        $php_version . ' ' . $this->ui->badge(
          $is_supported ? (string) $this->t('Compatible') : (string) $this->t('Incompatible'),
          $is_supported ? 'success' : 'error'
        )
      ),
    ], $is_supported ? NULL : 'error');

    // Database info.
    if (!empty($system_info['database']['driver'])) {
      $rows[] = [
        $this->ui->itemName((string) $this->t('Database')),
        $system_info['database']['driver'] . ' ' . ($system_info['database']['version'] ?? ''),
      ];
    }

    // Server info.
    if (!empty($system_info['server']['software'])) {
      $rows[] = [
        $this->ui->itemName((string) $this->t('Web Server')),
        $system_info['server']['software'],
      ];
    }

    $rows[] = [
      $this->ui->itemName((string) $this->t('Operating System')),
      $system_info['server']['os'] ?? PHP_OS,
    ];

    return ['table' => $this->ui->table($headers, $rows)];
  }

  /**
   * Builds content for the PHP version check.
   */
  protected function buildPhpVersionContent(array $data): array {
    $system_info = $data['system_info'] ?? [];
    $files = $data['_files'] ?? [];

    // Show any issues as faceted list.
    $results = $files['php_version']['results'] ?? [];
    if (!empty($results)) {
      return $this->ui->buildIssueListFromResults(
        $results,
        (string) $this->t('PHP version is compatible.'),
        function (array $item, $ui): array {
          return [
            'severity' => $item['severity'] ?? 'warning',
            'code' => $item['code'] ?? 'PHP_ISSUE',
            'label' => $item['message'] ?? '',
            'tags' => ['config'],
          ];
        }
      );
    }

    // No issues - show success message.
    return [
      'message' => $this->ui->message(
        (string) $this->t('PHP @version is compatible and recommended for your Drupal installation.', [
          '@version' => $system_info['php']['version'] ?? PHP_VERSION,
        ]),
        'success'
      ),
    ];
  }

  /**
   * Builds content for the database version check.
   */
  protected function buildDatabaseVersionContent(array $data): array {
    $system_info = $data['system_info'] ?? [];
    $files = $data['_files'] ?? [];

    $results = $files['database_version']['results'] ?? [];

    return $this->ui->buildIssueListFromResults(
      $results,
      (string) $this->t('Database @driver @version is compatible with your Drupal installation.', [
        '@driver' => $system_info['database']['driver'] ?? 'Unknown',
        '@version' => $system_info['database']['version'] ?? '',
      ]),
      function (array $item, $ui): array {
        return [
          'severity' => $item['severity'] ?? 'error',
          'code' => $item['code'] ?? 'DB_ISSUE',
          'label' => $item['message'] ?? '',
          'tags' => ['config'],
        ];
      }
    );
  }

  /**
   * Builds content for the requirements check.
   */
  protected function buildRequirementsContent(array $data): array {
    $files = $data['_files'] ?? [];
    $results = $files['requirements']['results'] ?? [];

    return $this->ui->buildIssueListFromResults(
      $results,
      (string) $this->t('All system requirements are met.'),
      function (array $item, $ui): array {
        $details = $item['details'] ?? [];
        $description = $details['description'] ?? '';

        return [
          'severity' => $item['severity'] ?? 'warning',
          'code' => $item['code'] ?? 'REQ_ISSUE',
          'label' => $item['message'] ?? '',
          'description' => $description ? ['#markup' => '<p>' . $description . '</p>'] : NULL,
          'tags' => ['config'],
        ];
      }
    );
  }

  /**
   * Analyzes PHP version compatibility.
   *
   * @param string $drupal_minor
   *   The Drupal minor version (e.g., '10.3').
   *
   * @return array
   *   Analysis results with errors, warnings, notices counts.
   */
  protected function analyzePhpVersion(string $drupal_minor): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    $php_version = PHP_VERSION;
    $php_major_minor = $this->getMajorMinorVersion($php_version);

    $supported_versions = self::PHP_COMPATIBILITY[$drupal_minor] ?? [];
    $is_compatible = in_array($php_major_minor, $supported_versions, TRUE);

    if (empty($supported_versions)) {
      $results[] = $this->createResultItem(
        'warning',
        'PHP_UNKNOWN_DRUPAL',
        (string) $this->t('Unknown Drupal version @version - cannot verify PHP compatibility', ['@version' => $drupal_minor]),
        [
          'php_version' => $php_version,
          'drupal_version' => $drupal_minor,
        ]
      );
      $warnings++;
    }
    elseif (!$is_compatible) {
      $results[] = $this->createResultItem(
        'error',
        'PHP_INCOMPATIBLE',
        (string) $this->t('PHP @php is not compatible with Drupal @drupal. Supported: @supported', [
          '@php' => $php_version,
          '@drupal' => $drupal_minor,
          '@supported' => implode(', ', $supported_versions),
        ]),
        [
          'php_version' => $php_version,
          'php_major_minor' => $php_major_minor,
          'drupal_version' => $drupal_minor,
          'supported_versions' => $supported_versions,
        ]
      );
      $errors++;
    }
    else {
      $latest_supported = end($supported_versions);
      if ($php_major_minor !== $latest_supported) {
        $results[] = $this->createResultItem(
          'warning',
          'PHP_UPGRADE_AVAILABLE',
          (string) $this->t('PHP @php is compatible but @latest is recommended for Drupal @drupal', [
            '@php' => $php_version,
            '@latest' => $latest_supported,
            '@drupal' => $drupal_minor,
          ]),
          [
            'php_version' => $php_version,
            'recommended' => $latest_supported,
            'supported_versions' => $supported_versions,
          ]
        );
        $warnings++;
      }
    }

    return [
      'results' => $results,
      'errors' => $errors,
      'warnings' => $warnings,
      'notices' => $notices,
      'is_compatible' => $is_compatible,
      'is_latest' => $is_compatible && $php_major_minor === ($supported_versions[array_key_last($supported_versions)] ?? ''),
    ];
  }

  /**
   * Analyzes database version compatibility.
   *
   * @param string $drupal_major
   *   The Drupal major version (e.g., '10').
   *
   * @return array
   *   Analysis results with errors, warnings, notices counts.
   */
  protected function analyzeDatabaseVersion(string $drupal_major): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;
    $is_compatible = TRUE;

    try {
      $db_version = $this->database->version();
      $db_driver = $this->database->driver();

      $compatibility = $this->checkDatabaseCompatibility($db_driver, $db_version, $drupal_major);

      if (!$compatibility['compatible']) {
        $results[] = $this->createResultItem(
          'error',
          'DATABASE_INCOMPATIBLE',
          (string) $this->t('@driver @version is not compatible with Drupal @drupal. Minimum required: @minimum', [
            '@driver' => $compatibility['name'],
            '@version' => $db_version,
            '@drupal' => $drupal_major,
            '@minimum' => $compatibility['minimum'],
          ]),
          [
            'driver' => $db_driver,
            'version' => $db_version,
            'minimum_required' => $compatibility['minimum'],
          ]
        );
        $errors++;
        $is_compatible = FALSE;
      }
    }
    catch (\Exception $e) {
      $results[] = $this->createResultItem(
        'error',
        'DATABASE_ERROR',
        (string) $this->t('Unable to determine database version: @error', ['@error' => $e->getMessage()]),
        ['error' => $e->getMessage()]
      );
      $errors++;
      $is_compatible = FALSE;
    }

    return [
      'results' => $results,
      'errors' => $errors,
      'warnings' => $warnings,
      'notices' => $notices,
      'is_compatible' => $is_compatible,
    ];
  }

  /**
   * Checks database compatibility with Drupal version.
   *
   * @param string $driver
   *   The database driver.
   * @param string $version
   *   The database version.
   * @param string $drupal_major
   *   The Drupal major version.
   *
   * @return array
   *   Compatibility info with 'compatible', 'name', 'minimum' keys.
   */
  protected function checkDatabaseCompatibility(string $driver, string $version, string $drupal_major): array {
    $is_mariadb = stripos($version, 'mariadb') !== FALSE || stripos($version, '-maria') !== FALSE;

    // Extract clean version number.
    $clean_version = preg_replace('/[^0-9.]/', '', $version);
    if (preg_match('/(\d+\.\d+(\.\d+)?)/', $clean_version, $matches)) {
      $clean_version = $matches[1];
    }

    if ($driver === 'mysql') {
      if ($is_mariadb) {
        $minimum = self::MYSQL_REQUIREMENTS[$drupal_major]['mariadb'] ?? '10.3.7';
        return [
          'compatible' => version_compare($clean_version, $minimum, '>='),
          'name' => 'MariaDB',
          'minimum' => $minimum,
        ];
      }
      else {
        $minimum = self::MYSQL_REQUIREMENTS[$drupal_major]['mysql'] ?? '5.7.8';
        return [
          'compatible' => version_compare($clean_version, $minimum, '>='),
          'name' => 'MySQL',
          'minimum' => $minimum,
        ];
      }
    }
    elseif ($driver === 'pgsql') {
      $minimum = self::PGSQL_REQUIREMENTS[$drupal_major] ?? '12.0';
      return [
        'compatible' => version_compare($clean_version, $minimum, '>='),
        'name' => 'PostgreSQL',
        'minimum' => $minimum,
      ];
    }
    elseif ($driver === 'sqlite') {
      $minimum = self::SQLITE_REQUIREMENTS[$drupal_major] ?? '3.26';
      return [
        'compatible' => version_compare($clean_version, $minimum, '>='),
        'name' => 'SQLite',
        'minimum' => $minimum,
      ];
    }

    return [
      'compatible' => TRUE,
      'name' => $driver,
      'minimum' => 'unknown',
    ];
  }

  /**
   * Analyzes system requirements from hook_requirements.
   *
   * @return array
   *   Analysis results.
   */
  protected function analyzeSystemRequirements(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    $this->moduleHandler->loadInclude('system', 'install');
    $requirements = $this->moduleHandler->invokeAll('requirements', ['runtime']);

    $cron_ok = TRUE;
    $updates_ok = TRUE;

    foreach ($requirements as $key => $requirement) {
      $severity = $requirement['severity'] ?? REQUIREMENT_OK;
      $title = (string) ($requirement['title'] ?? $key);
      $value = $this->renderRequirementValue($requirement['value'] ?? '');
      $description = $this->renderRequirementValue($requirement['description'] ?? '');

      // Track cron and update status.
      if (str_contains(strtolower($key), 'cron')) {
        $cron_ok = $severity <= REQUIREMENT_WARNING;
      }
      if (str_contains(strtolower($key), 'update')) {
        $updates_ok = $severity <= REQUIREMENT_WARNING;
      }

      if ($severity === REQUIREMENT_ERROR) {
        $results[] = $this->createResultItem(
          'error',
          'REQ_' . strtoupper($key),
          $title . ($value ? ': ' . $value : ''),
          ['description' => $description, 'key' => $key]
        );
        $errors++;
      }
      elseif ($severity === REQUIREMENT_WARNING) {
        $results[] = $this->createResultItem(
          'warning',
          'REQ_' . strtoupper($key),
          $title . ($value ? ': ' . $value : ''),
          ['description' => $description, 'key' => $key]
        );
        $warnings++;
      }
    }

    return [
      'results' => $results,
      'errors' => $errors,
      'warnings' => $warnings,
      'notices' => $notices,
      'cron_ok' => $cron_ok,
      'updates_ok' => $updates_ok,
      'total_requirements' => count($requirements),
    ];
  }

  /**
   * Calculates scores for all factors.
   *
   * @param array $php_analysis
   *   PHP analysis results.
   * @param array $db_analysis
   *   Database analysis results.
   * @param array $req_analysis
   *   Requirements analysis results.
   *
   * @return array
   *   Score data with overall and factors.
   */
  protected function calculateScores(array $php_analysis, array $db_analysis, array $req_analysis): array {
    $factors = [];

    // PHP Version score.
    $php_score = 0;
    if ($php_analysis['is_compatible']) {
      $php_score = $php_analysis['is_latest'] ? 100 : 70;
    }
    $factors['php_version'] = [
      'score' => $php_score,
      'weight' => self::SCORE_WEIGHTS['php_version'],
      'label' => (string) $this->t('PHP Version'),
      'description' => $php_score === 100
        ? (string) $this->t('Using recommended PHP version')
        : ($php_score > 0
          ? (string) $this->t('PHP compatible but upgrade available')
          : (string) $this->t('PHP version not compatible')),
    ];

    // Database Version score.
    $db_score = $db_analysis['is_compatible'] ? 100 : 0;
    $factors['database_version'] = [
      'score' => $db_score,
      'weight' => self::SCORE_WEIGHTS['database_version'],
      'label' => (string) $this->t('Database Version'),
      'description' => $db_score === 100
        ? (string) $this->t('Database version compatible')
        : (string) $this->t('Database version not compatible'),
    ];

    // Requirements score.
    $req_score = 100;
    if ($req_analysis['errors'] > 0) {
      $req_score = max(0, 100 - ($req_analysis['errors'] * 25));
    }
    elseif ($req_analysis['warnings'] > 0) {
      $req_score = max(50, 100 - ($req_analysis['warnings'] * 10));
    }
    $factors['drupal_requirements'] = [
      'score' => $req_score,
      'weight' => self::SCORE_WEIGHTS['drupal_requirements'],
      'label' => (string) $this->t('System Requirements'),
      'description' => $req_analysis['errors'] > 0
        ? (string) $this->t('@count critical issues', ['@count' => $req_analysis['errors']])
        : ($req_analysis['warnings'] > 0
          ? (string) $this->t('@count warnings', ['@count' => $req_analysis['warnings']])
          : (string) $this->t('All requirements met')),
    ];

    // Cron status score.
    $cron_score = $req_analysis['cron_ok'] ? 100 : 0;
    $factors['cron_status'] = [
      'score' => $cron_score,
      'weight' => self::SCORE_WEIGHTS['cron_status'],
      'label' => (string) $this->t('Cron Status'),
      'description' => $cron_score === 100
        ? (string) $this->t('Cron running correctly')
        : (string) $this->t('Cron issues detected'),
    ];

    // Update status score.
    $update_score = $req_analysis['updates_ok'] ? 100 : 50;
    $factors['update_status'] = [
      'score' => $update_score,
      'weight' => self::SCORE_WEIGHTS['update_status'],
      'label' => (string) $this->t('Update Status'),
      'description' => $update_score === 100
        ? (string) $this->t('All modules up to date')
        : (string) $this->t('Updates available'),
    ];

    return [
      'factors' => $factors,
    ];
  }

  /**
   * Gets system information summary.
   *
   * @param string $drupal_version
   *   Full Drupal version.
   * @param string $drupal_major
   *   Drupal major version.
   * @param string $drupal_minor
   *   Drupal minor version.
   *
   * @return array
   *   System information.
   */
  protected function getSystemInfo(string $drupal_version, string $drupal_major, string $drupal_minor): array {
    $db_info = [];
    try {
      $db_info = [
        'driver' => $this->database->driver(),
        'version' => $this->database->version(),
      ];
    }
    catch (\Exception $e) {
      $db_info = ['error' => $e->getMessage()];
    }

    return [
      'drupal' => [
        'version' => $drupal_version,
        'major' => $drupal_major,
        'minor' => $drupal_minor,
      ],
      'php' => [
        'version' => PHP_VERSION,
        'major_minor' => $this->getMajorMinorVersion(PHP_VERSION),
        'supported_for_drupal' => self::PHP_COMPATIBILITY[$drupal_minor] ?? [],
      ],
      'database' => $db_info,
      'server' => [
        'software' => $_SERVER['SERVER_SOFTWARE'] ?? 'unknown',
        'os' => PHP_OS,
      ],
    ];
  }

  /**
   * Gets the major version from a full version string.
   *
   * @param string $version
   *   The full version string.
   *
   * @return string
   *   The major version.
   */
  protected function getDrupalMajorVersion(string $version): string {
    $parts = explode('.', $version);
    return $parts[0] ?? '10';
  }

  /**
   * Gets the minor version (major.minor) from a full version string.
   *
   * @param string $version
   *   The full version string.
   *
   * @return string
   *   The major.minor version.
   */
  protected function getDrupalMinorVersion(string $version): string {
    $parts = explode('.', $version);
    return ($parts[0] ?? '10') . '.' . ($parts[1] ?? '0');
  }

  /**
   * Gets major.minor from a version string.
   *
   * @param string $version
   *   The full version string.
   *
   * @return string
   *   The major.minor version.
   */
  protected function getMajorMinorVersion(string $version): string {
    if (preg_match('/^(\d+\.\d+)/', $version, $matches)) {
      return $matches[1];
    }
    return $version;
  }

  /**
   * Renders a requirement value to string.
   *
   * @param mixed $value
   *   The value from hook_requirements (can be string, array, or TranslatableMarkup).
   *
   * @return string
   *   The rendered string value.
   */
  protected function renderRequirementValue(mixed $value): string {
    if (is_string($value)) {
      return $value;
    }

    if ($value instanceof TranslatableMarkup) {
      return (string) $value;
    }

    if (is_array($value)) {
      // Try to render as markup if it's a render array.
      try {
        return (string) \Drupal::service('renderer')->renderPlain($value);
      }
      catch (\Exception $e) {
        // Fallback: try to extract #markup.
        if (isset($value['#markup'])) {
          return (string) $value['#markup'];
        }
        return '';
      }
    }

    return (string) $value;
  }

}
