<?php

declare(strict_types=1);

namespace Drupal\audit_twig\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Analyzes Twig templates for performance issues and best practices.
 */
#[AuditAnalyzer(
  id: 'twig',
  label: new TranslatableMarkup('Twig Templates Audit'),
  description: new TranslatableMarkup('Analyzes Twig templates for cache bubbling issues, unused fields, anti-patterns, and theme suggestions.'),
  menu_title: new TranslatableMarkup('Twig'),
  output_directory: 'twig',
  weight: 2,
)]
class TwigAnalyzer extends AuditAnalyzerBase {

  /**
   * Score weights for different factors.
   *
   * Anti-patterns have the highest weight as they indicate code quality issues.
   * Cache bubbling is critical for proper cache invalidation.
   * Field optimization affects performance but is less critical.
   * External libraries affect privacy, security, and reliability.
   * Theme suggestions are informational only and not included in scoring.
   */
  protected const SCORE_WEIGHTS = [
    'anti_patterns' => 40,
    'cache_bubbling' => 30,
    'field_optimization' => 15,
    'external_libraries' => 15,
  ];

  /**
   * Common Drupal variables that are not fields.
   */
  protected const COMMON_DRUPAL_VARIABLES = [
    'attributes',
    'title_attributes',
    'content_attributes',
    'label',
    'url',
    'node',
    'user',
    'page',
    'is_front',
    'logged_in',
    'is_admin',
    'directory',
    'theme',
    'base_path',
    'front_page',
    'site_name',
    'site_slogan',
    'title',
    'title_prefix',
    'title_suffix',
    'view_mode',
    'bundle',
    'id',
    'elements',
    'db_is_active',
    'language',
  ];

  /**
   * Anti-patterns to detect with severity levels, fix instructions, and tags.
   *
   * Tags classify the type of issue:
   * - security: Potential security vulnerabilities (XSS, data exposure)
   * - performance: Code that impacts site performance
   * - cache: Cache bubbling or invalidation issues
   * - maintainability: Code that's hard to maintain or violates separation of concerns
   * - deprecation: Using deprecated functions that will be removed
   * - debug: Debug code that must not be in production
   * - best_practice: Recommended Drupal patterns and standards
   * - privacy: Third-party tracking or data concerns
   * - reliability: External dependencies that may fail
   */
  protected const ANTI_PATTERNS = [
    // ==========================================================================
    // ERROR: Critical issues that must be fixed
    // ==========================================================================
    'kint' => [
      'pattern' => '/\{\{\s*kint\s*\(/i',
      'message' => 'Debug function kint() found in template',
      'severity' => 'error',
      'code' => 'KINT_IN_TWIG',
      'fix' => '<p><strong>Problem:</strong> The <code>kint()</code> function outputs debug information and must never be in production code. It exposes internal data structures and can reveal sensitive information. Additionally, kint is extremely slow and will severely impact page load times.</p><p><strong>Solution:</strong> Remove the <code>{{ kint(...) }}</code> call entirely. If you need to debug, use it only locally and ensure it\'s removed before committing.</p>',
      'tags' => ['debug', 'security', 'performance'],
    ],
    'dump' => [
      'pattern' => '/\{\{\s*dump\s*\(/i',
      'message' => 'Debug function dump() found in template',
      'severity' => 'error',
      'code' => 'DUMP_IN_TWIG',
      'fix' => '<p><strong>Problem:</strong> The <code>dump()</code> function outputs debug information and must never be in production code. It can expose sensitive data and slow down page rendering.</p><p><strong>Solution:</strong> Remove the <code>{{ dump(...) }}</code> call entirely. For debugging, use it only locally with Twig debug mode enabled.</p>',
      'tags' => ['debug', 'security', 'performance'],
    ],
    'drupal_render' => [
      'pattern' => '/drupal_render\s*\(/i',
      'message' => 'Using drupal_render() breaks render system and cache bubbling',
      'severity' => 'error',
      'code' => 'DRUPAL_RENDER_IN_TWIG',
      'fix' => '<p><strong>Problem:</strong> Calling <code>drupal_render()</code> in a template bypasses Drupal\'s lazy rendering system. This breaks cache metadata bubbling and can cause stale content or cache issues.</p><p><strong>Solution:</strong> Let Drupal handle rendering automatically. Instead of <code>{{ drupal_render(content.field_name) }}</code>, simply use <code>{{ content.field_name }}</code>. Drupal will render it at the appropriate time with proper cache handling.</p>',
      'tags' => ['cache', 'performance'],
    ],
    'database_query' => [
      'pattern' => '/\\\\Drupal::database\s*\(\)|db_query\s*\(/i',
      'message' => 'Direct database query in template',
      'severity' => 'error',
      'code' => 'DB_QUERY_IN_TWIG',
      'fix' => '<p><strong>Problem:</strong> Database queries in templates violate the separation of concerns principle. Templates should only handle presentation, not data retrieval. This also makes caching difficult and can cause performance issues.</p><p><strong>Solution:</strong> Move the database query to a preprocess function or a service. Pass the results to the template as a variable. Example in your .theme file:<br><code>function mytheme_preprocess_node(&$variables) {<br>&nbsp;&nbsp;$variables[\'my_data\'] = \\Drupal::database()->query(...)->fetchAll();<br>}</code></p>',
      'tags' => ['security', 'performance', 'maintainability'],
    ],

    // Render array drilling - CRITICAL: breaks cache and can cause security issues.
    'render_array_drilling_markup' => [
      'pattern' => '/content\.[a-z_]+\s*\[\s*[0-9]+\s*\]\s*\[\s*[\'"]#/',
      'message' => 'Render array drilling detected (accessing [0][\'#markup\'] or similar)',
      'severity' => 'error',
      'code' => 'RENDER_ARRAY_DRILLING',
      'fix' => '<p><strong>Problem:</strong> Accessing render array internals like <code>content.field_name[0][\'#markup\']</code> bypasses field formatters, breaks cache metadata bubbling, and can expose unescaped content leading to XSS vulnerabilities.</p><p><strong>Solution:</strong> Render the field directly: <code>{{ content.field_name }}</code>. If you need just the value, use a preprocess function to extract it safely, or use the Twig Tweak module\'s <code>|field_value</code> filter.</p>',
      'tags' => ['security', 'cache'],
    ],
    'render_array_drilling_items' => [
      'pattern' => '/content\.[a-z_]+\s*\[\s*[\'"]#items[\'"]\s*\]/',
      'message' => 'Render array drilling detected (accessing [\'#items\'])',
      'severity' => 'error',
      'code' => 'RENDER_ARRAY_DRILLING_ITEMS',
      'fix' => '<p><strong>Problem:</strong> Accessing <code>[\'#items\']</code> from a render array breaks the render pipeline and loses cache metadata. The #items property is internal to the render system.</p><p><strong>Solution:</strong> Use a field formatter to display items as needed, or extract values in a preprocess function. For iteration, access the entity directly: <code>{% for item in node.field_name %}</code>.</p>',
      'tags' => ['cache', 'maintainability'],
    ],
    'render_array_drilling_url' => [
      'pattern' => '/content\.[a-z_]+\s*\[\s*[0-9]+\s*\]\s*\[\s*[\'"]#url[\'"]\s*\]/',
      'message' => 'Render array drilling detected (accessing [\'#url\'])',
      'severity' => 'error',
      'code' => 'RENDER_ARRAY_DRILLING_URL',
      'fix' => '<p><strong>Problem:</strong> Extracting URLs from render arrays like <code>content.field_link[0][\'#url\']</code> bypasses the render system and loses cache metadata.</p><p><strong>Solution:</strong> Extract the URL in a preprocess function:<br><code>$variables[\'my_url\'] = $node->field_link->first()->getUrl()->toString();</code><br>Then use <code>{{ my_url }}</code> in the template.</p>',
      'tags' => ['cache', 'maintainability'],
    ],

    // |t filter with variables - NEVER do this.
    't_filter_variable' => [
      'pattern' => '/\{\{\s*[a-z_][a-z0-9_]*\s*\|\s*t\s*[}\|]/',
      'message' => '|t filter used directly on a variable',
      'severity' => 'error',
      'code' => 'T_FILTER_ON_VARIABLE',
      'exclude' => ['"', "'"],
      'fix' => '<p><strong>Problem:</strong> Using <code>{{ variable|t }}</code> is incorrect. The <code>|t</code> filter is only for literal strings that can be extracted by translation tools. Variables cannot be translated this way.</p><p><strong>Solution:</strong> For literal strings, use: <code>{{ "Hello world"|t }}</code>. For dynamic content, either:<br>1. Translate in preprocess: <code>$variables[\'message\'] = t(\'Hello @name\', [\'@name\' => $name]);</code><br>2. Use {% trans %}: <code>{% trans %}Hello {{ name }}{% endtrans %}</code></p>',
      'tags' => ['best_practice', 'maintainability'],
    ],

    // ==========================================================================
    // WARNING: Important issues that should be fixed
    // ==========================================================================
    'drupal_entity' => [
      'pattern' => '/drupal_entity\s*\(/i',
      'message' => 'Entity loaded directly in template using drupal_entity()',
      'severity' => 'warning',
      'code' => 'DRUPAL_ENTITY_IN_TWIG',
      'fix' => '<p><strong>Problem:</strong> Loading entities in templates using <code>drupal_entity()</code> mixes data retrieval with presentation. This bypasses proper cache metadata bubbling (cache tags, contexts, max-age) and makes templates harder to maintain. Entity loads in templates are not cached and can cause performance issues.</p><p><strong>Solution:</strong> Load and render the entity in a preprocess function using the entity view builder, then pass it to the template:<br><code>$variables[\'related_node\'] = \\Drupal::entityTypeManager()->getViewBuilder(\'node\')->view($node, \'teaser\');</code><br>This ensures proper cache metadata bubbling.</p>',
      'tags' => ['performance', 'cache', 'maintainability'],
    ],
    'node_load' => [
      'pattern' => '/node_load\s*\(/i',
      'message' => 'Uses deprecated node_load() function',
      'severity' => 'warning',
      'code' => 'NODE_LOAD_DEPRECATED',
      'fix' => '<p><strong>Problem:</strong> The <code>node_load()</code> function is deprecated since Drupal 8.0.0 and will be removed in a future version.</p><p><strong>Solution:</strong> Use the entity type manager instead:<br><code>\\Drupal::entityTypeManager()->getStorage(\'node\')->load($nid)</code><br>Or simply: <code>Node::load($nid)</code> (with proper use statement).</p>',
      'tags' => ['deprecation'],
    ],
    'user_load' => [
      'pattern' => '/user_load\s*\(/i',
      'message' => 'Uses deprecated user_load() function',
      'severity' => 'warning',
      'code' => 'USER_LOAD_DEPRECATED',
      'fix' => '<p><strong>Problem:</strong> The <code>user_load()</code> function is deprecated since Drupal 8.0.0 and will be removed in a future version.</p><p><strong>Solution:</strong> Use the entity type manager instead:<br><code>\\Drupal::entityTypeManager()->getStorage(\'user\')->load($uid)</code><br>Or simply: <code>User::load($uid)</code> (with proper use statement).</p>',
      'tags' => ['deprecation'],
    ],
    'entity_load' => [
      'pattern' => '/entity_load\s*\(/i',
      'message' => 'Uses deprecated entity_load() function',
      'severity' => 'warning',
      'code' => 'ENTITY_LOAD_DEPRECATED',
      'fix' => '<p><strong>Problem:</strong> The <code>entity_load()</code> function is deprecated since Drupal 8.0.0 and will be removed in a future version.</p><p><strong>Solution:</strong> Use the entity type manager:<br><code>\\Drupal::entityTypeManager()->getStorage($entity_type)->load($id)</code></p>',
      'tags' => ['deprecation'],
    ],
    'render_filter' => [
      'pattern' => '/\|render\b/',
      'message' => 'Using |render filter - may cause performance issues on high traffic sites',
      'severity' => 'warning',
      'code' => 'RENDER_FILTER_INEFFICIENT',
      'fix' => '<p><strong>Problem:</strong> The <code>|render</code> filter forces immediate rendering, which can hurt performance on high traffic sites by preventing render caching optimizations.</p><p><strong>Solution:</strong> If you need just the value, access it directly via the entity: <code>{{ node.field_name.value }}</code>. If you need to render for cache metadata but access values, use: <code>{% set _cache = content.field_name %}{{ node.field_name.value }}</code>. Alternatively, use Twig Tweak\'s <code>|field_raw</code> filter.</p>',
      'tags' => ['performance', 'cache'],
    ],

    // |raw with field values - potential XSS vulnerability.
    'raw_field_value' => [
      'pattern' => '/(?:node|entity|paragraph|media|block|user)\.[a-z_]+\.value\s*\|\s*raw/',
      'message' => '|raw filter used with entity field value - potential XSS vulnerability',
      'severity' => 'warning',
      'code' => 'RAW_FILTER_FIELD_VALUE',
      'fix' => '<p><strong>Problem:</strong> Using <code>|raw</code> on a field value like <code>{{ node.body.value|raw }}</code> bypasses Twig\'s auto-escaping. If the field contains user-supplied content, this can lead to XSS (Cross-Site Scripting) attacks.</p><p><strong>Solution:</strong> Use the formatted field output which is already sanitized: <code>{{ content.body }}</code>. If you must use raw values, sanitize in a preprocess function using <code>\\Drupal\\Component\\Utility\\Xss::filter()</code> or <code>check_markup()</code>.</p>',
      'tags' => ['security'],
    ],
    'raw_content_field' => [
      'pattern' => '/content\.[a-z_]+\s*\|\s*raw/',
      'message' => '|raw filter used on content field - usually unnecessary',
      'severity' => 'warning',
      'code' => 'RAW_FILTER_CONTENT',
      'fix' => '<p><strong>Problem:</strong> Using <code>|raw</code> on content arrays like <code>{{ content.field_name|raw }}</code> is usually unnecessary because Drupal\'s render arrays handle escaping automatically. Adding |raw can introduce security vulnerabilities.</p><p><strong>Solution:</strong> Simply render without the raw filter: <code>{{ content.field_name }}</code>. Drupal will handle the escaping appropriately based on the field formatter.</p>',
      'tags' => ['security'],
    ],

    // Include without isolation - can leak variables and cause unexpected behavior.
    'include_without_isolation' => [
      'pattern' => '/\{\{\s*include\s*\(\s*[\'"][^\'"]+[\'"]\s*,\s*\{[^}]*\}\s*\)\s*\}\}/',
      'message' => 'Include without isolation - may cause variable leakage',
      'severity' => 'warning',
      'code' => 'INCLUDE_WITHOUT_ISOLATION',
      'exclude' => ['with_context', 'only'],
      'fix' => '<p><strong>Problem:</strong> Including templates without isolation passes all parent variables to the included template. This can cause unexpected behavior and makes templates harder to reuse.</p><p><strong>Solution:</strong> Add <code>with_context = false</code> or use <code>only</code>:<br><code>{{ include(\'template.html.twig\', {var: value}, with_context = false) }}</code><br>Or: <code>{% include \'template.html.twig\' with {var: value} only %}</code></p>',
      'tags' => ['maintainability', 'best_practice'],
    ],

    // Entity field access without cache bubbling.
    'entity_uri_without_bubble' => [
      'pattern' => '/file_url\s*\(\s*(?:node|entity|paragraph|media)\.[a-z_]+\.entity\.(?:uri|fileuri)/',
      'message' => 'Accessing entity file URI may lose cache metadata',
      'severity' => 'warning',
      'code' => 'ENTITY_URI_NO_BUBBLE',
      'fix' => '<p><strong>Problem:</strong> Accessing file URIs directly like <code>{{ file_url(node.field_image.entity.uri.value) }}</code> bypasses the render system. The image\'s cache tags won\'t bubble up, potentially showing stale images after updates. This also impacts performance since file operations are not cached.</p><p><strong>Solution:</strong> First force cache bubbling by rendering the content field, then access the URI:<br><code>{% set _cache = content.field_image %}<br>{{ file_url(node.field_image.entity.uri.value) }}</code><br>The <code>_cache</code> variable ensures cache metadata bubbles.</p>',
      'tags' => ['cache', 'performance'],
    ],

    // Inline CSS styles in templates.
    'inline_css' => [
      'pattern' => '/<style\b[^>]*>/i',
      'message' => 'Inline CSS styles found in template',
      'severity' => 'warning',
      'code' => 'INLINE_CSS_IN_TWIG',
      // Exclude email templates (need inline CSS) and Storybook stories.
      'file_exclude' => ['simplenews-', '.stories.twig'],
      'fix' => '<p><strong>Problem:</strong> Embedding CSS styles directly in Twig templates using <code>&lt;style&gt;</code> tags is a bad practice that harms maintainability:</p>'
        . '<ul>'
        . '<li>Styles cannot be aggregated or minified by Drupal\'s asset system</li>'
        . '<li>CSS becomes scattered across multiple template files</li>'
        . '<li>Harder to maintain consistent styling across the site</li>'
        . '<li>Cannot leverage browser caching for CSS</li>'
        . '</ul>'
        . '<p><strong>Solution:</strong> Move CSS to a dedicated stylesheet file and attach it via a library:</p>'
        . '<ol>'
        . '<li>Create a CSS file in your theme: <code>css/components/my-component.css</code></li>'
        . '<li>Define a library in <code>THEME.libraries.yml</code></li>'
        . '<li>Attach the library in the template: <code>{{ attach_library(\'THEME/my-component\') }}</code></li>'
        . '</ol>',
      'tags' => ['maintainability', 'best_practice'],
    ],

    // Inline JavaScript in templates.
    'inline_js' => [
      'pattern' => '/<script\b[^>]*>/i',
      'message' => 'Inline JavaScript found in template',
      'severity' => 'warning',
      'code' => 'INLINE_JS_IN_TWIG',
      'exclude' => ['application/ld+json', 'application/json'],
      // Exclude Storybook stories.
      'file_exclude' => ['.stories.twig'],
      'fix' => '<p><strong>Problem:</strong> Embedding JavaScript directly in Twig templates using <code>&lt;script&gt;</code> tags is a bad practice that harms maintainability:</p>'
        . '<ul>'
        . '<li>Scripts cannot be aggregated or minified by Drupal\'s asset system</li>'
        . '<li>JavaScript becomes scattered across multiple template files</li>'
        . '<li>Cannot leverage browser caching for JS</li>'
        . '<li>Harder to debug and maintain</li>'
        . '<li>May cause issues with Content Security Policy (CSP)</li>'
        . '</ul>'
        . '<p><strong>Solution:</strong> Move JavaScript to a dedicated file and attach it via a library:</p>'
        . '<ol>'
        . '<li>Create a JS file in your theme: <code>js/components/my-component.js</code></li>'
        . '<li>Define a library in <code>THEME.libraries.yml</code></li>'
        . '<li>Attach the library in the template: <code>{{ attach_library(\'THEME/my-component\') }}</code></li>'
        . '<li>Use <code>drupalSettings</code> to pass dynamic data from PHP to JavaScript</li>'
        . '</ol>',
      'tags' => ['maintainability', 'best_practice'],
    ],

    // ==========================================================================
    // NOTICE: Best practice suggestions
    // ==========================================================================
    'static_service' => [
      'pattern' => '/\\\\Drupal::service\s*\(/i',
      'message' => 'Static service call detected in template',
      'severity' => 'info',
      'code' => 'STATIC_SERVICE_CALL',
      'fix' => '<p><strong>Recommendation:</strong> While <code>\\Drupal::service()</code> works in templates, it\'s better practice to inject services in preprocess functions. This improves testability and follows Drupal\'s dependency injection patterns.</p><p><strong>Better approach:</strong> Call the service in your preprocess function and pass the result to the template as a variable.</p>',
      'tags' => ['best_practice', 'maintainability'],
    ],
    'drupal_static' => [
      'pattern' => '/\\\\Drupal::[a-zA-Z]+\s*\(/i',
      'message' => 'Static Drupal call detected in template',
      'severity' => 'info',
      'code' => 'DRUPAL_STATIC_CALL',
      'exclude' => ['\\Drupal::service', '\\Drupal::database'],
      'fix' => '<p><strong>Recommendation:</strong> Static calls like <code>\\Drupal::currentUser()</code> work but mix logic with presentation. Templates should focus on displaying data, not retrieving it.</p><p><strong>Better approach:</strong> Move the call to a preprocess function:<br><code>$variables[\'current_user\'] = \\Drupal::currentUser();</code><br>Then use <code>{{ current_user.displayName }}</code> in the template.</p>',
      'tags' => ['best_practice', 'maintainability'],
    ],

    // Attributes as strings instead of using Attributes object - SECURITY RISK.
    'attributes_as_string' => [
      'pattern' => '/<[a-z]+\s+class\s*=\s*["\'][^"\']*\{\{[^}]+\}\}/',
      'message' => 'Dynamic class in HTML attribute string - potential XSS vulnerability',
      'severity' => 'error',
      'code' => 'ATTRIBUTES_AS_STRING',
      'fix' => '<p><strong>Security Issue:</strong> Building attributes as strings like <code>&lt;div class="base {{ class }}"&gt;</code> is a <strong>security vulnerability</strong>. If the variable contains user input, it can lead to Cross-Site Scripting (XSS) attacks because Twig\'s auto-escaping does not protect against attribute injection.</p>'
        . '<p><strong>Example of attack:</strong> If <code>class</code> contains <code>foo" onclick="alert(1)</code>, the resulting HTML would be:<br><code>&lt;div class="base foo" onclick="alert(1)"&gt;</code></p>'
        . '<p><strong>Correct approach:</strong> Always use Drupal\'s Attributes object which properly escapes all values:</p>'
        . '<pre><code>{% set classes = [\'base\', my_class|clean_class] %}\n&lt;div{{ attributes.addClass(classes) }}&gt;</code></pre>'
        . '<p>The <code>|clean_class</code> filter sanitizes the value, and the Attributes object ensures proper HTML escaping. This also allows contributed modules to add their own attributes safely.</p>',
      'tags' => ['security', 'cache', 'best_practice'],
    ],

    // ==========================================================================
    // NOTICE: Documentation quality (detected programmatically, not by regex)
    // ==========================================================================
    'missing_documentation' => [
      'pattern' => NULL,
      'message' => 'Template file is missing documentation comment',
      'severity' => 'notice',
      'code' => 'MISSING_TEMPLATE_DOCUMENTATION',
      'fix' => '<p><strong>Recommendation:</strong> Every Twig template should start with a documentation comment that describes the template\'s purpose and available variables. This improves maintainability and helps other developers understand the template.</p>'
        . '<p><strong>Add a documentation block like this:</strong></p>'
        . '<pre><code>{#<br>/**<br> * @file<br> * Template for displaying [description].<br> *<br> * Available variables:<br> * - content: The render array with all fields.<br> * - attributes: HTML attributes for the container.<br> */<br>#}</code></pre>',
      'tags' => ['maintainability', 'best_practice'],
    ],
    'incomplete_documentation' => [
      'pattern' => NULL,
      'message' => 'Template has a comment but lacks proper documentation',
      'severity' => 'notice',
      'code' => 'INCOMPLETE_TEMPLATE_DOCUMENTATION',
      'fix' => '<p><strong>Recommendation:</strong> The template has a comment at the start, but it doesn\'t contain proper documentation. Good documentation should include the <code>@file</code> tag and describe available variables.</p>'
        . '<p><strong>Improve the documentation to include:</strong></p>'
        . '<ul><li><code>@file</code> - Identifies this as a file-level documentation</li>'
        . '<li>A brief description of what the template displays</li>'
        . '<li><code>Available variables:</code> section listing the variables passed to the template</li></ul>',
      'tags' => ['maintainability', 'best_practice'],
    ],
  ];

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Cached scan directories.
   */
  protected ?array $scanDirectories = NULL;

  /**
   * Cached exclude patterns.
   */
  protected ?array $excludePatterns = NULL;

  /**
   * Cached compiled regex patterns for anti-patterns.
   */
  protected ?array $compiledPatterns = NULL;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->configFactory = $container->get('config.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $config): array {
    return [
      'ignore_cache_analysis' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Ignore cache bubbling analysis'),
        '#description' => $this->t('Skip detection of {{ content }} usage and cache metadata bubbling issues.'),
        '#default_value' => $config['ignore_cache_analysis'] ?? FALSE,
      ],
      'ignore_anti_patterns' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Ignore anti-patterns analysis'),
        '#description' => $this->t('Skip detection of performance anti-patterns like drupal_entity(), database queries, etc.'),
        '#default_value' => $config['ignore_anti_patterns'] ?? FALSE,
      ],
      'ignore_field_sync' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Ignore field synchronization analysis'),
        '#description' => $this->t('Skip detection of fields configured in view modes but not used in templates, and vice versa.'),
        '#default_value' => $config['ignore_field_sync'] ?? FALSE,
      ],
      'ignore_theme_suggestions' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Ignore theme suggestions analysis'),
        '#description' => $this->t('Skip listing of theme hook suggestions registered from custom modules and themes.'),
        '#default_value' => $config['ignore_theme_suggestions'] ?? FALSE,
      ],
      'ignore_preprocess_analysis' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Ignore preprocess anti-patterns analysis'),
        '#description' => $this->t('Skip detection of drupal_render() and other anti-patterns in preprocess hooks.'),
        '#default_value' => $config['ignore_preprocess_analysis'] ?? FALSE,
      ],
      'ignore_external_libraries' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Ignore external libraries analysis'),
        '#description' => $this->t('Skip detection of external CSS/JS libraries in *.libraries.yml files (CDN, unpkg, etc.).'),
        '#default_value' => $config['ignore_external_libraries'] ?? FALSE,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      'cache_bubbling' => [
        'label' => $this->t('Cache Bubbling'),
        'description' => $this->t('Detects entity templates that access fields individually without rendering {{ content }}, which causes cache metadata to not bubble correctly and can lead to stale content.'),
        'file_types' => ['twig'],
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['cache_bubbling'],
        'file_key' => 'cache_bubbling',
        'score_factor_key' => 'cache_bubbling',
      ],
      'anti_patterns' => [
        'label' => $this->t('Twig Anti-Patterns'),
        'description' => $this->t('Detects problematic patterns in Twig templates: debug functions (kint, dump), drupal_render(), database queries, render array drilling, unsafe |raw filter usage, |t filter on variables, and deprecated entity load functions.'),
        'file_types' => ['twig'],
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['anti_patterns'],
        'file_key' => 'anti_patterns',
        'score_factor_key' => 'anti_patterns',
        'extra_file_keys' => ['preprocess_antipatterns'],
      ],
      'field_sync' => [
        'label' => $this->t('Field Rendering Optimization'),
        'description' => $this->t('Compares fields used in Twig templates against fields configured in entity view modes. Detects fields configured but not rendered, and fields rendered but not configured.'),
        'file_types' => ['twig', 'config'],
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['field_optimization'],
        'file_key' => 'field_sync',
        'score_factor_key' => 'field_optimization',
      ],
      'theme_suggestions' => [
        'label' => $this->t('Theme Suggestions'),
        'description' => $this->t('Lists custom theme suggestion hooks registered in modules and themes. Informational only, does not affect the audit score.'),
        'file_types' => ['php'],
        'affects_score' => FALSE,
        'weight' => 0,
        'file_key' => 'theme_suggestions',
        'score_factor_key' => NULL,
      ],
      'preprocess_antipatterns' => [
        'label' => $this->t('Preprocess Anti-Patterns'),
        'description' => $this->t('Detects anti-patterns in preprocess hooks: drupal_render(), renderer service render(), and debug functions (kint, dump) that should not be in production code.'),
        'file_types' => ['php'],
        // Note: This is combined with anti_patterns for scoring purposes.
        'affects_score' => FALSE,
        'weight' => 0,
        'file_key' => 'preprocess_antipatterns',
        'score_factor_key' => NULL,
      ],
      'external_libraries' => [
        'label' => $this->t('External Libraries'),
        'description' => $this->t('Detects CSS and JS files loaded from external CDNs or URLs in *.libraries.yml files. External resources can cause privacy, performance, reliability, and security concerns.'),
        'file_types' => ['yml'],
        'affects_score' => TRUE,
        'weight' => self::SCORE_WEIGHTS['external_libraries'],
        'file_key' => 'external_libraries',
        'score_factor_key' => 'external_libraries',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    $twig_config = $this->configFactory->get('audit_twig.settings');

    $files = [];
    // Only track stats needed for score calculations.
    $stats = [
      'templates_without_content' => 0,
      'anti_pattern_count' => 0,
      'missing_fields_count' => 0,
      'unknown_fields_count' => 0,
    ];

    // Analyze all templates first to gather data.
    $template_analysis = $this->analyzeAllTemplates();

    // Cache bubbling analysis.
    if (!$twig_config->get('ignore_cache_analysis')) {
      $cache_results = $this->analyzeCacheBubbling($template_analysis);
      $files['cache_bubbling'] = $cache_results;
      $stats['templates_without_content'] = $cache_results['stats']['without_content'] ?? 0;
    }

    // Anti-patterns analysis.
    if (!$twig_config->get('ignore_anti_patterns')) {
      $anti_pattern_results = $this->analyzeAntiPatterns($template_analysis);
      $files['anti_patterns'] = $anti_pattern_results;
      $stats['anti_pattern_count'] = count($anti_pattern_results['results'] ?? []);
    }

    // Field synchronization analysis.
    if (!$twig_config->get('ignore_field_sync')) {
      $field_sync_results = $this->analyzeFieldSync($template_analysis);
      $files['field_sync'] = $field_sync_results;
      $stats['missing_fields_count'] = $field_sync_results['stats']['missing'] ?? 0;
      $stats['unknown_fields_count'] = $field_sync_results['stats']['unknown'] ?? 0;
    }

    // Theme suggestions analysis.
    if (!$twig_config->get('ignore_theme_suggestions')) {
      $suggestions_results = $this->analyzeThemeSuggestions();
      $files['theme_suggestions'] = $suggestions_results;
    }

    // Preprocess anti-patterns analysis (drupal_render in preprocess hooks).
    if (!$twig_config->get('ignore_preprocess_analysis')) {
      $preprocess_results = $this->analyzePreprocessAntiPatterns();
      $files['preprocess_antipatterns'] = $preprocess_results;
      // Add preprocess errors to anti-pattern count for scoring.
      $stats['anti_pattern_count'] += count($preprocess_results['results'] ?? []);
    }

    // External libraries analysis.
    if (!$twig_config->get('ignore_external_libraries')) {
      $external_libs_results = $this->analyzeExternalLibraries();
      $files['external_libraries'] = $external_libs_results;
      $stats['external_libraries_count'] = count($external_libs_results['results'] ?? []);
    }

    $scores = $this->calculateScores($stats, $files);

    if (empty($files)) {
      $output = $this->createResult([], 0, 0, 0);
      $output['score'] = $scores;
      return $output;
    }

    return [
      '_files' => $files,
      'score' => $scores,
    ];
  }

  /**
   * Analyzes all Twig templates and extracts metadata.
   *
   * @return array
   *   Array of template analysis data.
   */
  protected function analyzeAllTemplates(): array {
    $templates = [];
    $paths_to_scan = $this->getScanDirectories();

    foreach ($paths_to_scan as $path) {
      $twig_files = $this->findTwigFiles($path);

      foreach ($twig_files as $file) {
        if ($this->shouldExclude($file)) {
          continue;
        }

        $content = file_get_contents($file);
        if ($content === FALSE) {
          continue;
        }

        $file_relative = str_replace(DRUPAL_ROOT . '/', '', $file);
        $filename = basename($file, '.html.twig');

        $template_info = [
          'file' => $file_relative,
          'filename' => $filename,
          'content' => $content,
          'entity_type' => NULL,
          'bundle' => NULL,
          'view_mode' => NULL,
          'uses_full_content' => FALSE,
          'uses_content_without' => FALSE,
          'uses_individual_fields' => FALSE,
          'fields_used' => [],
          'fields_in_without' => [],
          'anti_patterns' => [],
        ];

        // Parse template name for entity/bundle/viewmode.
        $this->parseTemplateName($filename, $template_info);

        // Analyze content usage.
        $this->analyzeContentUsage($content, $template_info);

        // Extract fields used.
        $this->extractFieldsUsed($content, $template_info);

        // Detect anti-patterns.
        $this->detectAntiPatterns($content, $template_info);

        $templates[$file_relative] = $template_info;
      }
    }

    return $templates;
  }

  /**
   * Parses template filename to extract entity type, bundle, and view mode.
   *
   * @param string $filename
   *   The template filename without extension.
   * @param array $info
   *   The template info array to populate.
   */
  protected function parseTemplateName(string $filename, array &$info): void {
    // Drupal template naming: entity-type--bundle--viewmode.html.twig
    // Examples: node--article--teaser, taxonomy-term--tags--full.
    $parts = explode('--', $filename);

    if (empty($parts)) {
      return;
    }

    $entity_type = $this->normalizeEntityType($parts[0]);
    if (!$this->isValidEntityType($entity_type)) {
      return;
    }

    $info['entity_type'] = $entity_type;

    if (count($parts) === 2) {
      // Could be bundle or view mode.
      if ($this->isViewMode($parts[1])) {
        $info['view_mode'] = $parts[1];
      }
      else {
        $info['bundle'] = $parts[1];
        $info['view_mode'] = 'default';
      }
    }
    elseif (count($parts) >= 3) {
      $info['bundle'] = $parts[1];
      $info['view_mode'] = $parts[2];
    }
  }

  /**
   * Normalizes entity type from template name.
   *
   * @param string $name
   *   The entity type name from template.
   *
   * @return string
   *   Normalized entity type.
   */
  protected function normalizeEntityType(string $name): string {
    $mappings = [
      'node' => 'node',
      'taxonomy-term' => 'taxonomy_term',
      'block' => 'block_content',
      'block-content' => 'block_content',
      'paragraph' => 'paragraph',
      'media' => 'media',
      'user' => 'user',
      'comment' => 'comment',
      'field' => 'field',
    ];

    return $mappings[$name] ?? str_replace('-', '_', $name);
  }

  /**
   * Checks if entity type is valid.
   *
   * @param string $entity_type
   *   The entity type to check.
   *
   * @return bool
   *   TRUE if valid.
   */
  protected function isValidEntityType(string $entity_type): bool {
    $valid_types = [
      'node',
      'taxonomy_term',
      'block_content',
      'paragraph',
      'media',
      'user',
      'comment',
    ];
    return in_array($entity_type, $valid_types, TRUE);
  }

  /**
   * Checks if a string is likely a view mode.
   *
   * @param string $name
   *   The name to check.
   *
   * @return bool
   *   TRUE if likely a view mode.
   */
  protected function isViewMode(string $name): bool {
    $common_view_modes = [
      'default',
      'full',
      'teaser',
      'search_result',
      'search_index',
      'rss',
      'token',
      'preview',
      'compact',
      'card',
      'featured',
      'mini',
      'embedded',
    ];
    return in_array($name, $common_view_modes, TRUE);
  }

  /**
   * Analyzes content variable usage in template.
   *
   * @param string $content
   *   The template content.
   * @param array $info
   *   The template info array to populate.
   */
  protected function analyzeContentUsage(string $content, array &$info): void {
    // First, check if {{ content }} is inside a Twig comment.
    // This is a common mistake - developers comment out the content render
    // but leave it in the template, breaking cache bubbling.
    $info['content_is_commented'] = FALSE;
    $info['content_commented_line'] = NULL;
    $info['content_in_documentation'] = FALSE;

    // Search line by line to find single-line commented content.
    // Patterns detected:
    // - {# {{ content }} #} - standard commented content
    // - {#{ content }#} - shorthand comment (# inside braces)
    $lines = explode("\n", $content);
    foreach ($lines as $line_num => $line) {
      // Pattern 1: {# ... {{ content }} ... #}
      if (preg_match('/\{#.*\{\{\s*content\s*(\|[^}]*)?\}\}.*#\}/', $line)) {
        $info['content_is_commented'] = TRUE;
        $info['content_commented_line'] = $line_num + 1; // 1-based line number
        break;
      }
      // Pattern 2: {#{ content }#} - shorthand where # is placed inside braces
      if (preg_match('/\{#\{\s*content\s*(\|[^}]*)?\}#\}/', $line)) {
        $info['content_is_commented'] = TRUE;
        $info['content_commented_line'] = $line_num + 1;
        break;
      }
    }

    // If not found on a single line, check for multiline comments.
    if (!$info['content_is_commented']) {
      // Check both patterns in multiline context.
      $multiline_patterns = [
        '/\{#.*?\{\{\s*content\s*(\|[^}]*)?\}\}.*?#\}/s',  // {# {{ content }} #}
        '/\{#\{\s*content\s*(\|[^}]*)?\}#\}/s',            // {#{ content }#}
      ];

      foreach ($multiline_patterns as $pattern) {
        if (preg_match($pattern, $content, $matches, PREG_OFFSET_CAPTURE)) {
          // Calculate the line number where the comment starts.
          $offset = $matches[0][1];
          $comment_start_line = substr_count(substr($content, 0, $offset), "\n") + 1;

          if ($comment_start_line === 1) {
            // Comment starts on line 1 - this is documentation.
            // The documentation mentions {{ content }}, so if the actual code
            // doesn't use it, we should flag it as missing.
            $info['content_in_documentation'] = TRUE;
          }
          else {
            // Comment starts after line 1 - this is commented-out code.
            $info['content_is_commented'] = TRUE;
            $info['content_commented_line'] = $comment_start_line;
          }
          break;
        }
      }
    }

    // Strip Twig comments before analyzing content usage.
    // This ensures we only detect actual rendered code, not commented code.
    $content_without_comments = preg_replace('/\{#.*?#\}/s', '', $content);

    // Check for {{ content }} - full render with cache bubbling.
    if (preg_match('/\{\{\s*content\s*\}\}/', $content_without_comments)) {
      $info['uses_full_content'] = TRUE;
    }

    // Check for {{ content|without(...) }} - also valid for bubbling.
    if (preg_match('/\{\{\s*content\s*\|\s*without\s*\(/i', $content_without_comments)) {
      $info['uses_content_without'] = TRUE;
      $info['uses_full_content'] = TRUE;

      // Extract fields in without() clause.
      if (preg_match_all('/without\s*\([^)]*[\'"]([a-z_][a-z0-9_]*)[\'"]/', $content_without_comments, $matches)) {
        $info['fields_in_without'] = array_unique($matches[1]);
      }
    }

    // Check for {{ content|cache_metadata }} - twig_tweak filter for cache bubbling.
    // This ensures cache metadata bubbles correctly without rendering the content.
    if (preg_match('/\{\{\s*content\s*\|\s*cache_metadata\s*\}\}/i', $content_without_comments)) {
      $info['uses_cache_metadata'] = TRUE;
      $info['uses_full_content'] = TRUE;
    }

    // Check for individual field access without full content.
    // Use original content here to detect all field accesses including commented ones.
    if (preg_match('/\{\{[^}]*content\.[a-z_][a-z0-9_]*/', $content_without_comments)) {
      $info['uses_individual_fields'] = TRUE;
    }
  }

  /**
   * Extracts fields used in template.
   *
   * @param string $content
   *   The template content.
   * @param array $info
   *   The template info array to populate.
   */
  protected function extractFieldsUsed(string $content, array &$info): void {
    // Strip Twig comments to avoid counting fields that are only in comments.
    $content_without_comments = preg_replace('/\{#.*?#\}/s', '', $content);

    $fields = [];

    // Pattern 1: {{ content.field_name }}.
    if (preg_match_all('/\{\{\s*content\.([a-z_][a-z0-9_]*)\s*[|}]/', $content_without_comments, $matches)) {
      $fields = array_merge($fields, $matches[1]);
    }

    // Pattern 2: content.field_name in conditions/loops.
    if (preg_match_all('/content\.([a-z_][a-z0-9_]*)/', $content_without_comments, $matches)) {
      $fields = array_merge($fields, $matches[1]);
    }

    // Pattern 3: {{ field_name }} (direct field variable).
    if (preg_match_all('/\{\{\s*([a-z_][a-z0-9_]*)\s*[|}]/', $content_without_comments, $matches)) {
      foreach ($matches[1] as $field) {
        if (str_starts_with($field, 'field_')) {
          $fields[] = $field;
        }
      }
    }

    // Filter out common Drupal variables.
    $fields = array_filter($fields, function ($field) {
      return !in_array($field, self::COMMON_DRUPAL_VARIABLES, TRUE);
    });

    $info['fields_used'] = array_unique($fields);
  }

  /**
   * Detects anti-patterns in template content.
   *
   * @param string $content
   *   The template content.
   * @param array $info
   *   The template info array to populate.
   */
  protected function detectAntiPatterns(string $content, array &$info): void {
    $lines = explode("\n", $content);
    $compiled_patterns = $this->getCompiledPatterns();
    $file = $info['file'] ?? '';

    foreach ($lines as $line_num => $line) {
      foreach ($compiled_patterns as $pattern_id => $pattern_config) {
        // Skip patterns that are NULL (detected programmatically, not by regex).
        if ($pattern_config['pattern'] === NULL) {
          continue;
        }

        // Check file exclusions first.
        if (!empty($pattern_config['file_exclude'])) {
          $skip_file = FALSE;
          foreach ($pattern_config['file_exclude'] as $file_pattern) {
            if (str_contains($file, $file_pattern)) {
              $skip_file = TRUE;
              break;
            }
          }
          if ($skip_file) {
            continue;
          }
        }

        if (preg_match($pattern_config['pattern'], $line)) {
          // Check content exclusions.
          if (!empty($pattern_config['exclude'])) {
            $skip = FALSE;
            foreach ($pattern_config['exclude'] as $exclude) {
              if (str_contains($line, $exclude)) {
                $skip = TRUE;
                break;
              }
            }
            if ($skip) {
              continue;
            }
          }

          // Extract code context around the issue.
          $code_context = $this->extractCodeContext($lines, $line_num + 1);

          $info['anti_patterns'][] = [
            'pattern' => $pattern_id,
            'line' => $line_num + 1,
            'code' => $pattern_config['code'],
            'message' => $pattern_config['message'],
            'severity' => $pattern_config['severity'],
            'code_context' => $code_context,
          ];
        }
      }
    }

    // Check for N+1 problem: Twig Tweak functions inside {% for %} loops.
    $this->detectNPlusOneInLoops($content, $lines, $info);

    // Check for missing documentation comment at the start of the file.
    $this->detectMissingDocumentation($content, $info);
  }

  /**
   * Detects missing documentation comment at the start of the file.
   *
   * @param string $content
   *   The full template content.
   * @param array $info
   *   The template info array to populate.
   */
  protected function detectMissingDocumentation(string $content, array &$info): void {
    // Skip Storybook story files - they don't need documentation.
    $file = $info['file'] ?? '';
    if (str_ends_with($file, '.stories.twig')) {
      return;
    }

    // Check if the file starts with a documentation comment.
    // Valid documentation starts with {# on the first line (possibly after whitespace).
    $trimmed = ltrim($content);

    // Check if it starts with a Twig comment that looks like documentation.
    // Documentation comments typically contain @file or describe variables.
    if (!str_starts_with($trimmed, '{#')) {
      // No comment at all at the start.
      $info['anti_patterns'][] = [
        'pattern' => 'missing_documentation',
        'line' => 1,
        'code' => 'MISSING_TEMPLATE_DOCUMENTATION',
        'message' => 'Template file is missing documentation comment',
        'severity' => 'notice',
        'code_context' => [],
      ];
      return;
    }

    // Has a comment, but check if it's actual documentation (not just a simple comment).
    // Documentation typically contains @file, "Variables:", or describes the template purpose.
    if (preg_match('/^\{#(.*?)#\}/s', $trimmed, $matches)) {
      $comment_content = $matches[1];

      // Check if it looks like proper documentation.
      $has_file_tag = str_contains($comment_content, '@file');
      $has_variables = preg_match('/variables\s*:/i', $comment_content);
      $has_description = strlen(trim($comment_content)) > 20; // At least some meaningful content

      if (!$has_file_tag && !$has_variables && !$has_description) {
        $info['anti_patterns'][] = [
          'pattern' => 'incomplete_documentation',
          'line' => 1,
          'code' => 'INCOMPLETE_TEMPLATE_DOCUMENTATION',
          'message' => 'Template has a comment but lacks proper documentation (missing @file or variable descriptions)',
          'severity' => 'notice',
          'code_context' => [],
        ];
      }
    }
  }

  /**
   * Detects N+1 anti-pattern: Twig Tweak functions inside for loops.
   *
   * @param string $content
   *   The full template content.
   * @param array $lines
   *   Array of lines.
   * @param array $info
   *   The template info array to populate.
   */
  protected function detectNPlusOneInLoops(string $content, array $lines, array &$info): void {
    // Functions that cause queries when called in loops.
    $dangerous_functions = [
      'drupal_entity',
      'drupal_view',
      'drupal_block',
      'drupal_field',
      'drupal_menu',
      'drupal_form',
      'drupal_region',
    ];

    $pattern = '/(' . implode('|', $dangerous_functions) . ')\s*\(/i';

    // Find all {% for %} ... {% endfor %} blocks.
    if (preg_match_all('/\{%\s*for\b.*?%\}(.*?)\{%\s*endfor\s*%\}/is', $content, $for_matches, PREG_OFFSET_CAPTURE)) {
      foreach ($for_matches[0] as $idx => $match) {
        $for_block = $match[0];
        $for_offset = $match[1];
        $for_body = $for_matches[1][$idx][0];

        // Check if any dangerous function is called inside the loop.
        if (preg_match_all($pattern, $for_body, $func_matches, PREG_OFFSET_CAPTURE)) {
          foreach ($func_matches[0] as $func_match) {
            $function_name = $func_matches[1][0][0];
            $func_offset_in_body = $func_match[1];

            // Calculate the actual line number.
            $content_before_for = substr($content, 0, $for_offset);
            $lines_before_for = substr_count($content_before_for, "\n");

            // Find the line within the for block.
            $body_before_func = substr($for_body, 0, $func_offset_in_body);
            $lines_in_body = substr_count($body_before_func, "\n");

            // +1 for 1-indexed line numbers.
            $line_number = $lines_before_for + $lines_in_body + 1;

            $code_context = $this->extractCodeContext($lines, $line_number);

            $info['anti_patterns'][] = [
              'pattern' => 'n_plus_one_loop',
              'line' => $line_number,
              'code' => 'N_PLUS_ONE_IN_LOOP',
              'message' => sprintf(
                '%s() called inside a for loop causes N+1 queries - use Views with caching or load entities in preprocess',
                $function_name
              ),
              'severity' => 'error',
              'code_context' => $code_context,
            ];
          }
        }
      }
    }
  }

  /**
   * Extracts code context around a specific line.
   *
   * @param array $lines
   *   Array of all lines in the file.
   * @param int $target_line
   *   The line number with the issue (1-indexed).
   * @param int $context_lines
   *   Number of lines to show before and after.
   *
   * @return array
   *   Array with 'lines' containing line data and 'highlight_line' with target.
   */
  protected function extractCodeContext(array $lines, int $target_line, int $context_lines = 2): array {
    $total_lines = count($lines);
    $target_index = $target_line - 1;

    $start_index = max(0, $target_index - $context_lines);
    $end_index = min($total_lines - 1, $target_index + $context_lines);

    $context = [
      'lines' => [],
      'highlight_line' => $target_line,
    ];

    for ($i = $start_index; $i <= $end_index; $i++) {
      $line_number = $i + 1;
      $context['lines'][] = [
        'number' => $line_number,
        'content' => $lines[$i] ?? '',
        'is_highlight' => ($line_number === $target_line),
      ];
    }

    return $context;
  }

  /**
   * Analyzes cache bubbling issues.
   *
   * @param array $templates
   *   The template analysis data.
   *
   * @return array
   *   Analysis results.
   */
  protected function analyzeCacheBubbling(array $templates): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $with_content = 0;
    $without_content = 0;
    $commented_content = 0;

    foreach ($templates as $file => $info) {
      // Only check entity templates.
      if (empty($info['entity_type'])) {
        continue;
      }

      // Check if {{ content }} is commented out - this is also an error.
      // Developers sometimes comment out content for debugging and forget to uncomment.
      if (!empty($info['content_is_commented']) && !$info['uses_full_content']) {
        $commented_content++;
        $errors++;

        $results[] = $this->createResultItem(
          'error',
          'COMMENTED_CONTENT_RENDER',
          (string) $this->t('Template @file has {{ content }} commented out - cache metadata will not bubble correctly', [
            '@file' => $info['filename'],
          ]),
          [
            'file' => $file,
            'line' => $info['content_commented_line'] ?? NULL,
            'entity_type' => $info['entity_type'],
            'bundle' => $info['bundle'],
            'view_mode' => $info['view_mode'],
            'fields_used' => $info['fields_used'] ?? [],
            'is_commented' => TRUE,
          ]
        );
        continue;
      }

      // Check if documentation mentions {{ content }} but code doesn't use it.
      // This happens when template has standard Drupal documentation explaining
      // how to use {{ content }}, but the developer didn't follow it.
      if (!empty($info['content_in_documentation']) && !$info['uses_full_content']) {
        $without_content++;
        $errors++;

        $results[] = $this->createResultItem(
          'error',
          'DOCUMENTED_CONTENT_NOT_USED',
          (string) $this->t('Template @file has documentation mentioning {{ content }} but does not use it - cache metadata will not bubble correctly', [
            '@file' => $info['filename'],
          ]),
          [
            'file' => $file,
            'entity_type' => $info['entity_type'],
            'bundle' => $info['bundle'],
            'view_mode' => $info['view_mode'],
            'fields_used' => $info['fields_used'] ?? [],
            'documented_but_unused' => TRUE,
          ]
        );
        continue;
      }

      if ($info['uses_full_content']) {
        $with_content++;
      }
      elseif ($info['uses_individual_fields']) {
        // Uses individual fields without {{ content }} - cache bubbling issue.
        // This is a critical error as it breaks Drupal's cache invalidation.
        $without_content++;
        $errors++;

        $results[] = $this->createResultItem(
          'error',
          'MISSING_CONTENT_RENDER',
          (string) $this->t('Template @file accesses fields individually without {{ content }} - cache metadata will not bubble correctly', [
            '@file' => $info['filename'],
          ]),
          [
            'file' => $file,
            'entity_type' => $info['entity_type'],
            'bundle' => $info['bundle'],
            'view_mode' => $info['view_mode'],
            'fields_used' => $info['fields_used'],
          ]
        );
      }
    }

    $result = $this->createResult($results, $errors, 0, 0);
    $result['stats'] = [
      'with_content' => $with_content,
      'without_content' => $without_content,
      'commented_content' => $commented_content,
    ];

    return $result;
  }

  /**
   * Analyzes anti-patterns in templates.
   *
   * @param array $templates
   *   The template analysis data.
   *
   * @return array
   *   Analysis results.
   */
  protected function analyzeAntiPatterns(array $templates): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    foreach ($templates as $file => $info) {
      foreach ($info['anti_patterns'] as $issue) {
        $severity = $issue['severity'];

        $results[] = $this->createResultItem(
          $severity,
          $issue['code'],
          $issue['message'],
          [
            'file' => $file,
            'line' => $issue['line'],
            'pattern' => $issue['pattern'],
            'code_context' => $issue['code_context'] ?? [],
          ]
        );

        if ($severity === 'error') {
          $errors++;
        }
        elseif ($severity === 'warning') {
          $warnings++;
        }
        elseif ($severity === 'notice') {
          $notices++;
        }
      }
    }

    return $this->createResult($results, $errors, $warnings, $notices);
  }

  /**
   * Analyzes field synchronization between templates and view mode config.
   *
   * @param array $templates
   *   The template analysis data.
   *
   * @return array
   *   Analysis results.
   */
  protected function analyzeFieldSync(array $templates): array {
    $results = [];
    $warnings = 0;
    $notices = 0;
    $missing_count = 0;
    $unknown_count = 0;
    $excludable_count = 0;
    $templates_analyzed = 0;
    $viewmodes_analyzed = [];

    foreach ($templates as $file => $info) {
      if (empty($info['entity_type']) || empty($info['bundle'])) {
        continue;
      }

      $view_mode = $info['view_mode'] ?? 'default';

      // Get fields configured in view mode.
      $configured_fields = $this->getViewModeFields(
        $info['entity_type'],
        $info['bundle'],
        $view_mode
      );

      if (empty($configured_fields)) {
        continue;
      }

      // Track statistics.
      $templates_analyzed++;
      $viewmode_key = "{$info['entity_type']}.{$info['bundle']}.{$view_mode}";
      $viewmodes_analyzed[$viewmode_key] = TRUE;

      // Check for fields in without() that are configured - these could be disabled.
      if ($info['uses_full_content'] && !empty($info['fields_in_without'])) {
        $excludable_fields = array_intersect($info['fields_in_without'], $configured_fields);
        // Exclude fields that are also explicitly rendered in the template.
        $excludable_fields = array_diff($excludable_fields, $info['fields_used']);
        foreach ($excludable_fields as $field) {
          $results[] = $this->createResultItem(
            'warning',
            'FIELD_EXCLUDABLE_IN_VIEWMODE',
            (string) $this->t('Field @field is excluded with without() but still configured in view mode - disable it in @entity/@bundle/@viewmode to save memory and processing', [
              '@field' => $field,
              '@entity' => $info['entity_type'],
              '@bundle' => $info['bundle'],
              '@viewmode' => $info['view_mode'] ?? 'default',
            ]),
            [
              'file' => $file,
              'field' => $field,
              'entity_type' => $info['entity_type'],
              'bundle' => $info['bundle'],
              'view_mode' => $info['view_mode'] ?? 'default',
              'type' => 'excludable',
            ]
          );
          $warnings++;
          $excludable_count++;
        }
        // Skip further checks for this template since it uses full content.
        continue;
      }

      // Skip if uses full content without exclusions (all fields rendered).
      if ($info['uses_full_content']) {
        continue;
      }

      $fields_used = $info['fields_used'];
      $fields_used = array_unique($fields_used);

      // Find missing fields (configured but not used in template).
      $missing_fields = array_diff($configured_fields, $fields_used);
      foreach ($missing_fields as $field) {
        $results[] = $this->createResultItem(
          'warning',
          'FIELD_CONFIGURED_NOT_USED',
          (string) $this->t('Field @field is configured in view mode but not rendered in template - disable it in @entity/@bundle/@viewmode to save memory and processing', [
            '@field' => $field,
            '@entity' => $info['entity_type'],
            '@bundle' => $info['bundle'],
            '@viewmode' => $info['view_mode'] ?? 'default',
          ]),
          [
            'file' => $file,
            'field' => $field,
            'entity_type' => $info['entity_type'],
            'bundle' => $info['bundle'],
            'view_mode' => $info['view_mode'] ?? 'default',
            'type' => 'missing',
          ]
        );
        $warnings++;
        $missing_count++;
      }

      // Find unknown fields (used in template but not configured).
      $unknown_fields = array_diff($fields_used, $configured_fields);
      // Filter to only field_* fields.
      $unknown_fields = array_filter($unknown_fields, fn($f) => str_starts_with($f, 'field_'));

      foreach ($unknown_fields as $field) {
        $results[] = $this->createResultItem(
          'notice',
          'FIELD_USED_NOT_CONFIGURED',
          (string) $this->t('Field @field is used in template @file but not visible in view mode configuration', [
            '@field' => $field,
            '@file' => $info['filename'],
          ]),
          [
            'file' => $file,
            'field' => $field,
            'entity_type' => $info['entity_type'],
            'bundle' => $info['bundle'],
            'view_mode' => $info['view_mode'] ?? 'default',
            'type' => 'unknown',
          ]
        );
        $notices++;
        $unknown_count++;
      }
    }

    $result = $this->createResult($results, 0, $warnings, $notices);
    $result['stats'] = [
      'missing' => $missing_count,
      'unknown' => $unknown_count,
      'excludable' => $excludable_count,
      'templates_analyzed' => $templates_analyzed,
      'viewmodes_analyzed' => count($viewmodes_analyzed),
    ];

    return $result;
  }

  /**
   * Gets visible fields for a view mode.
   *
   * @param string $entity_type
   *   The entity type.
   * @param string $bundle
   *   The bundle.
   * @param string $view_mode
   *   The view mode.
   *
   * @return array
   *   Array of visible field names.
   */
  protected function getViewModeFields(string $entity_type, string $bundle, string $view_mode): array {
    $config_name = "core.entity_view_display.{$entity_type}.{$bundle}.{$view_mode}";
    $config = $this->configFactory->get($config_name);

    if ($config->isNew()) {
      // Try default view mode.
      $config_name = "core.entity_view_display.{$entity_type}.{$bundle}.default";
      $config = $this->configFactory->get($config_name);
    }

    if ($config->isNew()) {
      return [];
    }

    $content = $config->get('content') ?? [];

    // Fields in 'content' are visible.
    $visible_fields = array_keys($content);

    // Filter to only field_* fields.
    return array_filter($visible_fields, fn($f) => str_starts_with($f, 'field_'));
  }

  /**
   * Analyzes theme suggestion hooks in .module and .theme files.
   *
   * @return array
   *   The analysis results.
   */
  protected function analyzeThemeSuggestions(): array {
    $results = [];
    $seen = [];
    $notices = 0;

    $paths_to_scan = $this->getScanDirectories();

    foreach ($paths_to_scan as $path) {
      $files = $this->findSuggestionFiles($path);

      foreach ($files as $file) {
        if ($this->shouldExclude($file)) {
          continue;
        }

        $content = file_get_contents($file);
        if ($content === FALSE) {
          continue;
        }

        // Find hook_theme_suggestions_HOOK() and _alter() implementations.
        if (preg_match_all(
          '/function\s+([a-z_]+)_theme_suggestions_([a-z_]+)\s*\([^)]*\)\s*\{/i',
          $content,
          $matches,
          PREG_SET_ORDER | PREG_OFFSET_CAPTURE
        )) {
          foreach ($matches as $match) {
            $function_name = $match[1][0] . '_theme_suggestions_' . $match[2][0];
            $offset = $match[0][1];
            $line = $this->getLineNumber($content, $offset);
            $file_relative = str_replace(DRUPAL_ROOT . '/', '', $file);

            $key = $file_relative . ':' . $line;
            if (isset($seen[$key])) {
              continue;
            }
            $seen[$key] = TRUE;

            $function_body = $this->extractFunctionBody($content, $offset);
            $analysis = $this->analyzeSuggestionLogic($function_body);

            $results[] = $this->createResultItem(
              'notice',
              'THEME_SUGGESTIONS_HOOK',
              (string) $this->t('Theme suggestions hook @func', ['@func' => $function_name]),
              [
                'function' => $function_name,
                'file' => $file_relative,
                'line' => $line,
                'suggestion_lines' => $analysis['suggestion_lines'],
              ]
            );
            $notices++;
          }
        }
      }
    }

    return $this->createResult($results, 0, 0, $notices);
  }

  /**
   * Analyzes preprocess hooks for anti-patterns like drupal_render().
   *
   * @return array
   *   The analysis results.
   */
  protected function analyzePreprocessAntiPatterns(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;

    $paths_to_scan = $this->getScanDirectories();

    foreach ($paths_to_scan as $path) {
      $files = $this->findSuggestionFiles($path);

      foreach ($files as $file) {
        if ($this->shouldExclude($file)) {
          continue;
        }

        $content = file_get_contents($file);
        if ($content === FALSE) {
          continue;
        }

        $file_relative = str_replace(DRUPAL_ROOT . '/', '', $file);
        $lines = explode("\n", $content);

        // Find all preprocess functions.
        if (preg_match_all(
          '/function\s+([a-z_]+)_preprocess_([a-z_]+)\s*\([^)]*\)\s*\{/i',
          $content,
          $matches,
          PREG_SET_ORDER | PREG_OFFSET_CAPTURE
        )) {
          foreach ($matches as $match) {
            $function_name = $match[1][0] . '_preprocess_' . $match[2][0];
            $offset = $match[0][1];
            $function_line = $this->getLineNumber($content, $offset);
            $function_body = $this->extractFunctionBody($content, $offset);

            // Check for drupal_render() in the function body.
            if (preg_match_all('/drupal_render\s*\(/i', $function_body, $render_matches, PREG_OFFSET_CAPTURE)) {
              foreach ($render_matches[0] as $render_match) {
                // Calculate the actual line number within the function.
                $render_offset = $render_match[1];
                $lines_before = substr_count(substr($function_body, 0, $render_offset), "\n");
                $line = $function_line + $lines_before;

                // Extract code context.
                $code_context = $this->extractCodeContext($lines, $line);

                $results[] = $this->createResultItem(
                  'error',
                  'DRUPAL_RENDER_IN_PREPROCESS',
                  (string) $this->t('drupal_render() used in preprocess function @func - breaks render system and cache bubbling', [
                    '@func' => $function_name,
                  ]),
                  [
                    'function' => $function_name,
                    'file' => $file_relative,
                    'line' => $line,
                    'code_context' => $code_context,
                  ]
                );
                $errors++;
              }
            }

            // Also check for \Drupal::service('renderer')->render().
            if (preg_match_all('/->render\s*\(/i', $function_body, $render_matches, PREG_OFFSET_CAPTURE)) {
              foreach ($render_matches[0] as $render_match) {
                $render_offset = $render_match[1];
                $lines_before = substr_count(substr($function_body, 0, $render_offset), "\n");
                $line = $function_line + $lines_before;

                $code_context = $this->extractCodeContext($lines, $line);

                $results[] = $this->createResultItem(
                  'error',
                  'RENDERER_IN_PREPROCESS',
                  (string) $this->t('Renderer service render() used in preprocess function @func - breaks render system and cache bubbling', [
                    '@func' => $function_name,
                  ]),
                  [
                    'function' => $function_name,
                    'file' => $file_relative,
                    'line' => $line,
                    'code_context' => $code_context,
                  ]
                );
                $errors++;
              }
            }

            // Check for kint() debug function.
            if (preg_match_all('/\bkint\s*\(/i', $function_body, $kint_matches, PREG_OFFSET_CAPTURE)) {
              foreach ($kint_matches[0] as $kint_match) {
                $kint_offset = $kint_match[1];
                $lines_before = substr_count(substr($function_body, 0, $kint_offset), "\n");
                $line = $function_line + $lines_before;

                $code_context = $this->extractCodeContext($lines, $line);

                $results[] = $this->createResultItem(
                  'error',
                  'KINT_IN_PREPROCESS',
                  (string) $this->t('Debug function kint() found in preprocess function @func - must be removed before production', [
                    '@func' => $function_name,
                  ]),
                  [
                    'function' => $function_name,
                    'file' => $file_relative,
                    'line' => $line,
                    'code_context' => $code_context,
                  ]
                );
                $errors++;
              }
            }

            // Check for dump() debug function.
            if (preg_match_all('/\bdump\s*\(/i', $function_body, $dump_matches, PREG_OFFSET_CAPTURE)) {
              foreach ($dump_matches[0] as $dump_match) {
                $dump_offset = $dump_match[1];
                $lines_before = substr_count(substr($function_body, 0, $dump_offset), "\n");
                $line = $function_line + $lines_before;

                $code_context = $this->extractCodeContext($lines, $line);

                $results[] = $this->createResultItem(
                  'error',
                  'DUMP_IN_PREPROCESS',
                  (string) $this->t('Debug function dump() found in preprocess function @func - must be removed before production', [
                    '@func' => $function_name,
                  ]),
                  [
                    'function' => $function_name,
                    'file' => $file_relative,
                    'line' => $line,
                    'code_context' => $code_context,
                  ]
                );
                $errors++;
              }
            }
          }
        }
      }
    }

    return $this->createResult($results, $errors, $warnings, 0);
  }

  /**
   * Analyzes *.libraries.yml files for external library references.
   *
   * Detects CSS and JS files loaded from external CDNs or URLs which can cause:
   * - Privacy concerns (third-party tracking)
   * - Performance issues (additional DNS lookups, connection overhead)
   * - Reliability issues (CDN downtime affects your site)
   * - Security concerns (if CDN is compromised)
   *
   * @return array
   *   The analysis results.
   */
  protected function analyzeExternalLibraries(): array {
    $results = [];
    $warnings = 0;

    $paths_to_scan = $this->getScanDirectories();

    // Common external URL patterns.
    $external_patterns = [
      'https://',
      'http://',
      '//',
    ];

    foreach ($paths_to_scan as $path) {
      $library_files = $this->findLibraryFiles($path);

      foreach ($library_files as $file) {
        if ($this->shouldExclude($file)) {
          continue;
        }

        $content = file_get_contents($file);
        if ($content === FALSE) {
          continue;
        }

        $file_relative = str_replace(DRUPAL_ROOT . '/', '', $file);
        $lines = explode("\n", $content);

        // Track which library we're currently in.
        $current_library = NULL;
        $external_assets = [];

        foreach ($lines as $line_num => $line) {
          // Detect library definition (no leading spaces, ends with colon).
          if (preg_match('/^([a-z0-9_-]+):\s*$/i', $line, $lib_match)) {
            // Save previous library's external assets.
            if ($current_library && !empty($external_assets)) {
              $this->addExternalLibraryResult(
                $results,
                $warnings,
                $file_relative,
                $current_library,
                $external_assets
              );
            }
            $current_library = $lib_match[1];
            $external_assets = [];
            continue;
          }

          // Check for external URLs in the line.
          // Skip metadata lines that don't affect frontend loading.
          $trimmed_line = trim($line);
          if (preg_match('/^(remote|url|homepage|version):\s*/i', $trimmed_line)) {
            continue;
          }
          // Skip if we're inside a license: block (indented under license:).
          if ($this->isInsideLicenseBlock($lines, $line_num)) {
            continue;
          }

          foreach ($external_patterns as $pattern) {
            if (str_contains($line, $pattern)) {
              // Extract the URL.
              if (preg_match('/["\']?(https?:\/\/[^\s"\']+|\/\/[^\s"\']+)["\']?/i', $line, $url_match)) {
                $url = $url_match[1];
                // Determine if it's CSS or JS based on context.
                $type = $this->detectAssetType($lines, $line_num);
                $external_assets[] = [
                  'url' => $url,
                  'type' => $type,
                  'line' => $line_num + 1,
                ];
              }
              break;
            }
          }
        }

        // Don't forget the last library.
        if ($current_library && !empty($external_assets)) {
          $this->addExternalLibraryResult(
            $results,
            $warnings,
            $file_relative,
            $current_library,
            $external_assets
          );
        }
      }
    }

    return $this->createResult($results, 0, $warnings, 0);
  }

  /**
   * Adds a result item for external library assets.
   *
   * @param array $results
   *   Results array to add to.
   * @param int $warnings
   *   Warnings counter.
   * @param string $file
   *   The file path.
   * @param string $library
   *   The library name.
   * @param array $assets
   *   Array of external assets.
   */
  protected function addExternalLibraryResult(array &$results, int &$warnings, string $file, string $library, array $assets): void {
    $css_assets = array_filter($assets, fn($a) => $a['type'] === 'css');
    $js_assets = array_filter($assets, fn($a) => $a['type'] === 'js');

    $asset_summary = [];
    if (!empty($css_assets)) {
      $asset_summary[] = count($css_assets) . ' CSS';
    }
    if (!empty($js_assets)) {
      $asset_summary[] = count($js_assets) . ' JS';
    }

    $results[] = $this->createResultItem(
      'warning',
      'EXTERNAL_LIBRARY',
      (string) $this->t('Library "@library" loads @count external assets (@summary) - consider hosting locally', [
        '@library' => $library,
        '@count' => count($assets),
        '@summary' => implode(', ', $asset_summary),
      ]),
      [
        'file' => $file,
        'library' => $library,
        'assets' => $assets,
        'css_count' => count($css_assets),
        'js_count' => count($js_assets),
        'urls' => array_column($assets, 'url'),
      ]
    );
    $warnings++;
  }

  /**
   * Detects if a line is within CSS or JS section of a libraries.yml file.
   *
   * @param array $lines
   *   All lines of the file.
   * @param int $current_line
   *   Current line index.
   *
   * @return string
   *   'css', 'js', or 'unknown'.
   */
  protected function detectAssetType(array $lines, int $current_line): string {
    // Look backwards for css: or js: section header.
    for ($i = $current_line; $i >= 0; $i--) {
      $line = trim($lines[$i]);
      if (preg_match('/^css:\s*$/i', $line)) {
        return 'css';
      }
      if (preg_match('/^js:\s*$/i', $line)) {
        return 'js';
      }
      // If we hit a library definition, stop looking.
      if (preg_match('/^[a-z0-9_-]+:\s*$/i', $line)) {
        break;
      }
    }
    return 'unknown';
  }

  /**
   * Checks if the current line is inside a license: block.
   *
   * In Drupal libraries.yml, the license block looks like:
   *   license:
   *     name: MIT
   *     url: https://example.com/license
   *     gpl-compatible: true
   *
   * These URLs are metadata and should not be flagged as external assets.
   *
   * @param array $lines
   *   All lines of the file.
   * @param int $current_line
   *   Current line index.
   *
   * @return bool
   *   TRUE if inside a license block.
   */
  protected function isInsideLicenseBlock(array $lines, int $current_line): bool {
    // Get current line's indentation level.
    $current_indent = strlen($lines[$current_line]) - strlen(ltrim($lines[$current_line]));

    // If it's not indented, it can't be inside a license block.
    if ($current_indent === 0) {
      return FALSE;
    }

    // Look backwards for the license: header or another section.
    for ($i = $current_line - 1; $i >= 0; $i--) {
      $line = $lines[$i];
      $trimmed = trim($line);

      // Skip empty lines.
      if ($trimmed === '') {
        continue;
      }

      $line_indent = strlen($line) - strlen(ltrim($line));

      // If we find license: at lower indent level, we're inside it.
      if (preg_match('/^license:\s*$/i', $trimmed) && $line_indent < $current_indent) {
        return TRUE;
      }

      // If we find any other section header at lower or equal indent, we're not in license.
      if ($line_indent < $current_indent && preg_match('/^[a-z0-9_-]+:\s*$/i', $trimmed)) {
        return FALSE;
      }

      // If we hit a library definition (no indent), stop looking.
      if ($line_indent === 0 && preg_match('/^[a-z0-9_-]+:\s*$/i', $trimmed)) {
        return FALSE;
      }
    }

    return FALSE;
  }

  /**
   * Finds all *.libraries.yml files in a directory.
   *
   * @param string $path
   *   The path to search.
   *
   * @return array
   *   Array of file paths.
   */
  protected function findLibraryFiles(string $path): array {
    $files = [];

    if (!is_dir($path)) {
      return $files;
    }

    $iterator = new \RecursiveIteratorIterator(
      new \RecursiveDirectoryIterator($path, \RecursiveDirectoryIterator::SKIP_DOTS),
      \RecursiveIteratorIterator::LEAVES_ONLY
    );

    foreach ($iterator as $file) {
      if ($file->isFile() && str_ends_with($file->getFilename(), '.libraries.yml')) {
        $files[] = $file->getPathname();
      }
    }

    return $files;
  }

  /**
   * Gets the configured scan directories with caching.
   *
   * @return array
   *   Array of absolute paths to scan.
   */
  protected function getScanDirectories(): array {
    if ($this->scanDirectories !== NULL) {
      return $this->scanDirectories;
    }

    $config = $this->configFactory->get('audit.settings');
    $scan_dirs_raw = $config->get('scan_directories') ?? '';

    $paths = [];
    $lines = array_filter(array_map('trim', explode("\n", $scan_dirs_raw)));

    foreach ($lines as $line) {
      if (!str_starts_with($line, '/')) {
        $path = DRUPAL_ROOT . '/' . $line;
      }
      else {
        $path = $line;
      }

      $path = str_replace('/web/web/', '/web/', $path);

      if (is_dir($path)) {
        $paths[] = $path;
      }
    }

    $this->scanDirectories = $paths;
    return $this->scanDirectories;
  }

  /**
   * Gets the configured exclude patterns with caching.
   *
   * @return array
   *   Array of patterns to exclude.
   */
  protected function getExcludePatterns(): array {
    if ($this->excludePatterns !== NULL) {
      return $this->excludePatterns;
    }

    $config = $this->configFactory->get('audit.settings');
    $exclude_raw = $config->get('exclude_patterns') ?? '';

    $this->excludePatterns = array_filter(array_map('trim', explode("\n", $exclude_raw)));
    return $this->excludePatterns;
  }

  /**
   * Gets pre-compiled regex patterns for anti-pattern detection.
   *
   * Compiles the patterns once and caches them for performance.
   *
   * @return array
   *   Array of compiled pattern configurations.
   */
  protected function getCompiledPatterns(): array {
    if ($this->compiledPatterns !== NULL) {
      return $this->compiledPatterns;
    }

    $this->compiledPatterns = self::ANTI_PATTERNS;
    return $this->compiledPatterns;
  }

  /**
   * Checks if a file path should be excluded.
   *
   * @param string $file_path
   *   The file path to check.
   *
   * @return bool
   *   TRUE if the file should be excluded.
   */
  protected function shouldExclude(string $file_path): bool {
    $exclude_patterns = $this->getExcludePatterns();

    foreach ($exclude_patterns as $pattern) {
      $regex = str_replace(['*', '/'], ['.*', '\/'], $pattern);
      if (preg_match('/' . $regex . '/', $file_path)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Finds all Twig files in a directory.
   *
   * @param string $path
   *   The directory path.
   *
   * @return array
   *   Array of file paths.
   */
  protected function findTwigFiles(string $path): array {
    $files = [];

    if (!is_dir($path)) {
      return $files;
    }

    $iterator = new \RecursiveIteratorIterator(
      new \RecursiveDirectoryIterator($path, \RecursiveDirectoryIterator::SKIP_DOTS)
    );

    foreach ($iterator as $file) {
      if ($file->isFile() && $file->getExtension() === 'twig') {
        $files[] = $file->getPathname();
      }
    }

    return $files;
  }

  /**
   * Finds .module and .theme files in a directory.
   *
   * @param string $path
   *   The directory path.
   *
   * @return array
   *   Array of file paths.
   */
  protected function findSuggestionFiles(string $path): array {
    $files = [];

    if (!is_dir($path)) {
      return $files;
    }

    $iterator = new \RecursiveIteratorIterator(
      new \RecursiveDirectoryIterator($path, \RecursiveDirectoryIterator::SKIP_DOTS)
    );

    foreach ($iterator as $file) {
      if (!$file->isFile()) {
        continue;
      }

      $extension = $file->getExtension();
      if (in_array($extension, ['module', 'theme', 'php'], TRUE)) {
        $files[] = $file->getPathname();
      }
    }

    return $files;
  }

  /**
   * Extracts function body from content starting at offset.
   *
   * @param string $content
   *   The file content.
   * @param int $offset
   *   The offset where the function starts.
   *
   * @return string
   *   The function body.
   */
  protected function extractFunctionBody(string $content, int $offset): string {
    $start = strpos($content, '{', $offset);
    if ($start === FALSE) {
      return '';
    }

    $depth = 1;
    $pos = $start + 1;
    $length = strlen($content);

    while ($pos < $length && $depth > 0) {
      $char = $content[$pos];
      if ($char === '{') {
        $depth++;
      }
      elseif ($char === '}') {
        $depth--;
      }
      $pos++;
    }

    return substr($content, $start, $pos - $start);
  }

  /**
   * Analyzes suggestion hook logic.
   *
   * @param string $function_body
   *   The function body.
   *
   * @return array
   *   Analysis results.
   */
  protected function analyzeSuggestionLogic(string $function_body): array {
    $analysis = [
      'adds_suggestions' => FALSE,
      'suggestion_lines' => [],
    ];

    if (preg_match_all('/\$suggestions\s*\[\s*\]\s*=\s*([^;]+);/', $function_body, $matches)) {
      $analysis['adds_suggestions'] = TRUE;
      foreach ($matches[1] as $expression) {
        $analysis['suggestion_lines'][] = trim($expression);
      }
    }

    if (preg_match_all('/array_push\s*\(\s*\$suggestions\s*,\s*([^)]+)\)/', $function_body, $matches)) {
      $analysis['adds_suggestions'] = TRUE;
      foreach ($matches[1] as $expression) {
        $analysis['suggestion_lines'][] = trim($expression);
      }
    }

    return $analysis;
  }

  /**
   * Gets the line number for a given offset in content.
   *
   * @param string $content
   *   The file content.
   * @param int $offset
   *   The character offset.
   *
   * @return int
   *   The line number.
   */
  protected function getLineNumber(string $content, int $offset): int {
    return substr_count(substr($content, 0, $offset), "\n") + 1;
  }

  /**
   * Calculates scores for all factors.
   *
   * @param array $stats
   *   Statistics array.
   * @param array $files
   *   Analysis files.
   *
   * @return array
   *   Score data with per-section scores and overall average.
   */
  protected function calculateScores(array $stats, array $files): array {
    $factors = [];

    // 1. Cache Bubbling score - critical errors, each one penalizes heavily.
    $without_content = $stats['templates_without_content'];
    // Each cache bubbling error deducts 20 points (critical issue).
    $cache_score = max(0, 100 - ($without_content * 20));

    $factors['cache_bubbling'] = [
      'score' => $cache_score,
      'weight' => self::SCORE_WEIGHTS['cache_bubbling'],
      'label' => (string) $this->t('Cache Bubbling'),
      'description' => $without_content === 0
        ? (string) $this->t('All entity templates use {{ content }} correctly')
        : (string) $this->t('@count @errors - templates with broken cache invalidation', [
          '@count' => $without_content,
          '@errors' => $without_content === 1 ? 'error' : 'errors',
        ]),
    ];

    // 2. Anti-patterns score.
    // Count from both twig and preprocess anti-patterns.
    $twig_antipatterns = $files['anti_patterns']['results'] ?? [];
    $preprocess_antipatterns = $files['preprocess_antipatterns']['results'] ?? [];
    $all_antipatterns = array_merge($twig_antipatterns, $preprocess_antipatterns);

    // Count by severity for weighted penalty.
    $ap_errors = count(array_filter($all_antipatterns, fn($r) => ($r['severity'] ?? '') === 'error'));
    $ap_warnings = count(array_filter($all_antipatterns, fn($r) => ($r['severity'] ?? '') === 'warning'));
    $ap_notices = count(array_filter($all_antipatterns, fn($r) => !in_array($r['severity'] ?? '', ['error', 'warning'], TRUE)));

    // Errors: -15 points, Warnings: -8 points, Notices: -3 points.
    $ap_score = max(0, 100 - ($ap_errors * 15) - ($ap_warnings * 8) - ($ap_notices * 3));
    $total_antipatterns = $ap_errors + $ap_warnings + $ap_notices;

    $factors['anti_patterns'] = [
      'score' => $ap_score,
      'weight' => self::SCORE_WEIGHTS['anti_patterns'],
      'label' => (string) $this->t('Anti-Patterns'),
      'description' => $total_antipatterns === 0
        ? (string) $this->t('No anti-patterns detected')
        : (string) $this->t('@errors errors, @warnings warnings, @notices notices', [
          '@errors' => $ap_errors,
          '@warnings' => $ap_warnings,
          '@notices' => $ap_notices,
        ]),
    ];

    // 3. Field Optimization score.
    // Count from field_sync results by type.
    $field_results = $files['field_sync']['results'] ?? [];
    $excludable_count = count(array_filter($field_results, fn($r) => ($r['details']['type'] ?? '') === 'excludable'));
    $missing_count = count(array_filter($field_results, fn($r) => ($r['details']['type'] ?? '') === 'missing'));
    $unknown_count = count(array_filter($field_results, fn($r) => ($r['details']['type'] ?? '') === 'unknown'));

    // Excludable and missing are warnings: -5 points each.
    // Unknown are notices: -2 points each.
    $field_score = max(0, 100 - (($excludable_count + $missing_count) * 5) - ($unknown_count * 2));
    $total_field_issues = $excludable_count + $missing_count + $unknown_count;

    $factors['field_optimization'] = [
      'score' => $field_score,
      'weight' => self::SCORE_WEIGHTS['field_optimization'],
      'label' => (string) $this->t('Field Rendering Optimization'),
      'description' => $total_field_issues === 0
        ? (string) $this->t('All fields are properly configured and rendered')
        : (string) $this->t('@warnings warnings, @notices notices', [
          '@warnings' => $excludable_count + $missing_count,
          '@notices' => $unknown_count,
        ]),
    ];

    // 4. External Libraries score.
    // Each external library is a warning that affects privacy/security/reliability.
    $external_libs_results = $files['external_libraries']['results'] ?? [];
    $external_libs_count = count($external_libs_results);

    // Count total external assets (CSS + JS files).
    $total_external_assets = 0;
    foreach ($external_libs_results as $result) {
      $total_external_assets += count($result['details']['assets'] ?? []);
    }

    // Each external library deducts 10 points, each additional asset deducts 5 points.
    $external_score = max(0, 100 - ($external_libs_count * 10) - (max(0, $total_external_assets - $external_libs_count) * 5));

    $factors['external_libraries'] = [
      'score' => $external_score,
      'weight' => self::SCORE_WEIGHTS['external_libraries'],
      'label' => (string) $this->t('External Libraries'),
      'description' => $external_libs_count === 0
        ? (string) $this->t('All CSS/JS assets are hosted locally')
        : (string) $this->t('@count @libraries with @assets external assets', [
          '@count' => $external_libs_count,
          '@libraries' => $external_libs_count === 1 ? 'library' : 'libraries',
          '@assets' => $total_external_assets,
        ]),
    ];

    return [
      'factors' => $factors,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    $files = $data['_files'] ?? [];

    return match ($check_id) {
      'cache_bubbling' => $this->getCacheBubblingContent($files),
      'anti_patterns' => $this->getAntiPatternsContent($files),
      'field_sync' => $this->getFieldOptimizationContent($files),
      'theme_suggestions' => $this->getThemeSuggestionsContent($files),
      'preprocess_antipatterns' => $this->getPreprocessAntiPatternsContent($files),
      'external_libraries' => $this->getExternalLibrariesContent($files),
      default => [],
    };
  }

  /**
   * Gets the theme suggestions content (without section wrapper).
   *
   * @param array $files
   *   The analysis files data.
   *
   * @return array
   *   Render array for the content.
   */
  protected function getThemeSuggestionsContent(array $files): array {
    $results = $files['theme_suggestions']['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('No theme suggestion hooks detected. All templates use standard Drupal theme suggestions.'),
          'success'
        ),
      ];
    }

    $rows = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];

      $function_cell = $this->ui->itemName(
        $details['function'] ?? '',
        ($details['file'] ?? '') . ':' . ($details['line'] ?? '')
      );

      $suggestions_html = '-';
      if (!empty($details['suggestion_lines'])) {
        $items = [];
        foreach ($details['suggestion_lines'] as $line) {
          $items[] = '<code>' . htmlspecialchars($line, ENT_QUOTES, 'UTF-8') . '</code>';
        }
        $suggestions_html = implode('<br>', $items);
      }

      $rows[] = $this->ui->row([
        $function_cell,
        $suggestions_html,
      ]);
    }

    $headers = [
      $this->ui->header((string) $this->t('Function')),
      $this->ui->header((string) $this->t('Suggestions')),
    ];

    return [
      'table' => $this->ui->table($headers, $rows),
    ];
  }

  /**
   * Gets the cache bubbling content (without section wrapper).
   *
   * @param array $files
   *   The analysis files data.
   *
   * @return array
   *   Render array for the content.
   */
  protected function getCacheBubblingContent(array $files): array {
    $success_message = (string) $this->t('All entity templates correctly use {{ content }} ensuring proper cache metadata bubbling.');

    // Filter only errors.
    $errors = array_filter(
      $files['cache_bubbling']['results'] ?? [],
      fn($r) => ($r['severity'] ?? '') === 'error'
    );

    // Fix messages for different error types.
    $fixes = [
      'missing' => '<p><strong>Problem:</strong> This template accesses fields individually (e.g., <code>{{ content.field_name }}</code>) without also rendering <code>{{ content }}</code>. This causes cache metadata to not bubble correctly, potentially showing stale content.</p>'
        . '<p><strong>Solution:</strong> Add <code>{{ content }}</code> somewhere in your template. If you don\'t want to render all fields, use:</p>'
        . '<ul><li><code>{{ content|without(\'field_name\', \'field_other\') }}</code> - Renders content except specified fields</li>'
        . '<li><code>{% set _cache = content %}{{ content.field_specific }}</code> - Captures cache metadata without rendering</li>'
        . '<li><code>{{ content|cache_metadata }}</code> - From Twig Tweak module, bubbles cache only</li></ul>',
      'commented' => '<p><strong>Problem:</strong> This template has <code>{{ content }}</code> but it is commented out with <code>{# ... #}</code>. Commented code is not executed, so cache metadata will not bubble correctly. This is a common mistake when debugging templates.</p>'
        . '<p><strong>Solution:</strong> Uncomment the <code>{{ content }}</code> line by removing the <code>{#</code> and <code>#}</code> around it. If you need to hide the visual output but still bubble cache metadata, use one of these alternatives:</p>'
        . '<ul><li><code>{{ content|without(\'field_name\', \'field_other\') }}</code> - Renders content except specified fields</li>'
        . '<li><code>{% set _cache = content %}</code> - Captures cache metadata without any visible output</li>'
        . '<li><code>{{ content|cache_metadata }}</code> - From Twig Tweak module, bubbles cache only</li></ul>',
      'documented' => '<p><strong>Problem:</strong> This template has standard Drupal documentation explaining how to use <code>{{ content }}</code>, but the actual template code does not use it. This means the cache metadata from the entity\'s render array will not bubble up correctly.</p>'
        . '<p><strong>Solution:</strong> Add <code>{{ content }}</code> to render all fields, or use one of these alternatives:</p>'
        . '<ul><li><code>{{ content|without(\'field_name\', \'field_other\') }}</code> - Renders content except specified fields</li>'
        . '<li><code>{% set _cache = content %}</code> - Captures cache metadata without any visible output</li>'
        . '<li><code>{{ content|cache_metadata }}</code> - From Twig Tweak module, bubbles cache only</li></ul>',
    ];

    return $this->ui->buildIssueListFromResults(
      $errors,
      $success_message,
      function (array $item, $ui) use ($fixes): array {
        $details = $item['details'] ?? [];
        $is_commented = !empty($details['is_commented']);
        $is_documented = !empty($details['documented_but_unused']);

        if ($is_commented) {
          $label = 'Template has {{ content }} commented out';
          $fix = $fixes['commented'];
        }
        elseif ($is_documented) {
          $label = 'Template documentation mentions {{ content }} but code does not use it';
          $fix = $fixes['documented'];
        }
        else {
          $label = 'Template accesses fields without {{ content }}';
          $fix = $fixes['missing'];
        }

        return [
          'severity' => 'error',
          'code' => $item['code'] ?? 'MISSING_CONTENT_RENDER',
          'label' => $label,
          'file' => $details['file'] ?? '',
          'line' => $details['line'] ?? NULL,
          'description' => ['#markup' => $fix],
          'code_snippet' => $ui->buildIssueCodeSnippet($details, 'error'),
          'tags' => ['cache', 'performance'],
        ];
      }
    );
  }

  /**
   * Gets the anti-patterns content (without section wrapper).
   *
   * @param array $files
   *   The analysis files data.
   *
   * @return array
   *   Render array for the content.
   */
  protected function getAntiPatternsContent(array $files): array {
    $all_antipatterns = array_merge(
      $files['anti_patterns']['results'] ?? [],
      $files['preprocess_antipatterns']['results'] ?? []
    );

    // Sort by severity for ordered display.
    $severity_order = ['error' => 0, 'warning' => 1, 'notice' => 2, 'info' => 3];
    usort($all_antipatterns, fn($a, $b) =>
      ($severity_order[$a['severity'] ?? 'info'] ?? 3) <=> ($severity_order[$b['severity'] ?? 'info'] ?? 3)
    );

    return $this->ui->buildIssueListFromResults(
      $all_antipatterns,
      (string) $this->t('No anti-patterns detected in Twig templates or preprocess hooks. Your code follows best practices.'),
      function (array $item, $ui): array {
        $details = $item['details'] ?? [];
        $code = $item['code'] ?? '';
        $sev = $item['severity'] ?? 'info';

        return [
          'severity' => $ui->normalizeSeverity($sev),
          'code' => $code,
          'label' => $item['message'] ?? '',
          'file' => $details['file'] ?? '',
          'line' => $details['line'] ?? NULL,
          'description' => ($fix = $this->getAntiPatternFix($code)) ? ['#markup' => $fix] : NULL,
          'code_snippet' => $ui->buildIssueCodeSnippet($details, $sev),
          'tags' => $this->getAntiPatternTags($code),
        ];
      }
    );
  }

  /**
   * Gets the preprocess anti-patterns content (without section wrapper).
   *
   * @param array $files
   *   The analysis files data.
   *
   * @return array
   *   Render array for the content.
   */
  protected function getPreprocessAntiPatternsContent(array $files): array {
    return $this->ui->buildIssueListFromResults(
      $files['preprocess_antipatterns']['results'] ?? [],
      (string) $this->t('No anti-patterns detected in preprocess hooks.'),
      function (array $item, $ui): array {
        $details = $item['details'] ?? [];
        $code = $item['code'] ?? 'PREPROCESS_ANTIPATTERN';
        $function = $details['function'] ?? '';
        $severity = $item['severity'] ?? 'warning';

        $fix = $this->getAntiPatternFix($code);
        if (!$fix) {
          $fix = '<p><strong>Problem:</strong> Anti-pattern detected in preprocess function <code>' . htmlspecialchars($function, ENT_QUOTES, 'UTF-8') . '</code>.</p>'
            . '<p><strong>Solution:</strong> Move this logic to a service or use proper dependency injection. Avoid debug functions and direct rendering in preprocess hooks.</p>';
        }

        $label = $item['message'] ?? 'Anti-pattern in preprocess';
        if ($function) {
          $label = $function . ': ' . $label;
        }

        return [
          'severity' => $ui->normalizeSeverity($severity),
          'code' => $code,
          'label' => $label,
          'file' => $details['file'] ?? '',
          'line' => $details['line'] ?? NULL,
          'description' => ['#markup' => $fix],
          'code_snippet' => $ui->buildIssueCodeSnippet($details, $severity),
          'tags' => $this->getAntiPatternTags($code),
        ];
      }
    );
  }

  /**
   * Gets the field optimization content (without section wrapper).
   *
   * @param array $files
   *   The analysis files data.
   *
   * @return array
   *   Render array for the content.
   */
  protected function getFieldOptimizationContent(array $files): array {
    $results = $files['field_sync']['results'] ?? [];

    // Sort results by type for ordered display: excludable -> missing -> unknown.
    $type_order = ['excludable' => 0, 'missing' => 1, 'unknown' => 2];
    usort($results, function ($a, $b) use ($type_order) {
      $type_a = $a['details']['type'] ?? 'unknown';
      $type_b = $b['details']['type'] ?? 'unknown';
      return ($type_order[$type_a] ?? 3) <=> ($type_order[$type_b] ?? 3);
    });

    // Fix messages per type.
    $fixes = [
      'excludable' => '<p><strong>Problem:</strong> This field is configured to display in the view mode but is not rendered in the template. The field data is being loaded unnecessarily, impacting performance.</p>'
        . '<p><strong>Solution:</strong> Go to Structure > Content types > [Bundle] > Manage display > [View Mode] and disable this field (set it to "Hidden"). This prevents Drupal from loading the field data when it won\'t be displayed.</p>',
      'missing' => '<p><strong>Problem:</strong> This field is enabled in the view mode but the template doesn\'t render <code>{{ content }}</code> or the specific field. The field may not appear on the page.</p>'
        . '<p><strong>Solution:</strong> Either:<br>1. Add <code>{{ content.field_name }}</code> to the template to render the field<br>2. Or disable the field in the view mode if it\'s not needed</p>',
      'unknown' => '<p><strong>Info:</strong> This field is referenced in the template (e.g., <code>{{ content.field_name }}</code>) but is not configured in the view mode, or the field doesn\'t exist.</p>'
        . '<p><strong>Recommendation:</strong> Either:<br>1. Enable the field in the view mode configuration<br>2. Remove the reference from the template if the field is no longer needed<br>3. Check if the field name is correct</p>',
    ];

    return $this->ui->buildIssueListFromResults(
      $results,
      (string) $this->t('All fields are properly configured in view modes and rendered in templates.'),
      function (array $item, $ui) use ($fixes): array {
        $details = $item['details'] ?? [];
        $type = $details['type'] ?? 'unknown';
        $field = $details['field'] ?? '';
        $entity_type = $details['entity_type'] ?? '';
        $bundle = $details['bundle'] ?? '';
        $view_mode = $details['view_mode'] ?? '';
        $context = "{$entity_type}/{$bundle}/{$view_mode}";
        $file = $details['file'] ?? '';

        // Map type to severity, code, and label.
        $config = match ($type) {
          'excludable' => [
            'severity' => 'warning',
            'code' => 'FIELD_EXCLUDABLE',
            'label_key' => 'Field "@field" is configured but not rendered',
            'tags' => ['performance'],
          ],
          'missing' => [
            'severity' => 'warning',
            'code' => 'FIELD_MISSING',
            'label_key' => 'Field "@field" is not rendered in template',
            'tags' => ['maintainability'],
          ],
          default => [
            'severity' => 'notice',
            'code' => 'FIELD_UNKNOWN',
            'label_key' => 'Field "@field" used in template but not configured',
            'tags' => ['maintainability'],
          ],
        };

        return [
          'severity' => $config['severity'],
          'code' => $config['code'],
          'label' => str_replace('@field', $field, $config['label_key']) . ' (' . $context . ')',
          'file' => $file,
          'line' => NULL,
          'description' => ['#markup' => $fixes[$type] ?? ''],
          'code_snippet' => NULL,
          'tags' => $config['tags'],
        ];
      }
    );
  }

  /**
   * Gets the external libraries content (without section wrapper).
   *
   * @param array $files
   *   The analysis files data.
   *
   * @return array
   *   Render array for the content.
   */
  protected function getExternalLibrariesContent(array $files): array {
    $fix_html = '<p><strong>Problem:</strong> External libraries loaded from CDNs can cause:</p>'
      . '<ul><li><strong>Privacy concerns:</strong> Third-party tracking and data collection</li>'
      . '<li><strong>Performance issues:</strong> Additional DNS lookups and potential latency</li>'
      . '<li><strong>Reliability problems:</strong> CDN downtime affects your site</li>'
      . '<li><strong>Security risks:</strong> Compromised CDNs can inject malicious code</li></ul>'
      . '<p><strong>Solution:</strong> Download external assets and host them locally. Use npm/yarn with a build process, or manually download files to your theme\'s assets folder and update the library definition.</p>';

    return $this->ui->buildIssueListFromResults(
      $files['external_libraries']['results'] ?? [],
      (string) $this->t('All CSS and JS assets are hosted locally. No external CDN dependencies detected.'),
      function (array $item, $ui) use ($fix_html): array {
        $details = $item['details'] ?? [];
        $library = $details['library'] ?? '';
        $file = $details['file'] ?? '';

        // Build list of external URLs for the label.
        $assets = $details['assets'] ?? [];
        $asset_count = count($assets);
        $urls_sample = [];
        foreach (array_slice($assets, 0, 2) as $asset) {
          $url = $asset['url'] ?? '';
          $parsed = parse_url($url);
          $domain = $parsed['host'] ?? $url;
          $urls_sample[] = $domain;
        }
        $urls_text = implode(', ', $urls_sample);
        if ($asset_count > 2) {
          $urls_text .= ' +' . ($asset_count - 2) . ' more';
        }

        $label = 'Library "' . $library . '" loads ' . $asset_count . ' external ' . ($asset_count === 1 ? 'asset' : 'assets');
        if ($urls_text) {
          $label .= ': ' . $urls_text;
        }

        return [
          'severity' => 'warning',
          'code' => 'EXTERNAL_LIBRARY',
          'label' => $label,
          'file' => $file,
          'line' => NULL,
          'description' => ['#markup' => $fix_html],
          'code_snippet' => NULL,
          'tags' => ['privacy', 'security', 'reliability', 'performance'],
        ];
      }
    );
  }

  /**
   * Builds the theme suggestions section.
   *
   * @param array $files
   *   The analysis files data.
   *
   * @return array
   *   Render array for the section.
   */
  protected function buildThemeSuggestionsSection(array $files): array {
    if (empty($files['theme_suggestions']['results'])) {
      return $this->ui->section(
        (string) $this->t('Theme Suggestion Hooks'),
        ['message' => $this->ui->message(
          (string) $this->t('No theme suggestion hooks detected. All templates use standard Drupal theme suggestions.'),
          'success'
        )],
        ['count' => 0, 'severity' => 'success']
      );
    }

    $suggestions_count = count($files['theme_suggestions']['results']);
    $rows = [];

    foreach ($files['theme_suggestions']['results'] as $item) {
      $details = $item['details'];

      $function_cell = $this->ui->itemName(
        $details['function'],
        $details['file'] . ':' . $details['line']
      );

      $suggestions_html = '-';
      if (!empty($details['suggestion_lines'])) {
        $items = [];
        foreach ($details['suggestion_lines'] as $line) {
          $items[] = '<code>' . htmlspecialchars($line, ENT_QUOTES, 'UTF-8') . '</code>';
        }
        $suggestions_html = implode('<br>', $items);
      }

      $rows[] = $this->ui->row([
        $function_cell,
        $suggestions_html,
      ]);
    }

    $headers = [
      $this->ui->header((string) $this->t('Function')),
      $this->ui->header((string) $this->t('Suggestions')),
    ];

    $content = [
      'table' => $this->ui->table($headers, $rows),
    ];

    return $this->ui->section(
      (string) $this->t('Theme Suggestion Hooks'),
      $content,
      ['count' => $suggestions_count, 'severity' => 'notice']
    );
  }

  /**
   * Builds the cache bubbling section.
   *
   * @param array $files
   *   The analysis files data.
   *
   * @return array
   *   Render array for the section.
   */
  protected function buildCacheBubblingSection(array $files): array {
    $success_message = (string) $this->t('All entity templates correctly use {{ content }} ensuring proper cache metadata bubbling.');

    if (empty($files['cache_bubbling']['results'])) {
      return $this->ui->section(
        (string) $this->t('Cache Bubbling Issues'),
        ['message' => $this->ui->message($success_message, 'success')],
        ['count' => 0, 'severity' => 'success']
      );
    }

    $errors = array_filter($files['cache_bubbling']['results'], fn($r) => $r['severity'] === 'error');

    if (empty($errors)) {
      return $this->ui->section(
        (string) $this->t('Cache Bubbling Issues'),
        ['message' => $this->ui->message($success_message, 'success')],
        ['count' => 0, 'severity' => 'success']
      );
    }

    $errors_count = count($errors);
    $content = [];

    // Add explanation of cache bubbling.
    $explanation = (string) $this->t('What is cache bubbling?') . ' '
      . (string) $this->t('In Drupal, when you render {{ content }} in a Twig template, cache metadata (tags, contexts, max-age) from all rendered fields automatically "bubbles up" to the parent entity. This ensures proper cache invalidation.')
      . ' ' . (string) $this->t('When you access fields individually without also rendering {{ content }}, the cache metadata may not propagate correctly, leading to stale content.');
    $content['explanation'] = $this->ui->message($explanation, 'error');

    // Build table of issues.
    $headers = [
      $this->ui->header((string) $this->t('Template')),
      $this->ui->header((string) $this->t('Entity/Bundle/ViewMode')),
      $this->ui->header((string) $this->t('Fields Accessed')),
    ];

    $rows = [];
    foreach ($errors as $item) {
      $details = $item['details'];

      $template_cell = '<code>' . htmlspecialchars($details['file'], ENT_QUOTES, 'UTF-8') . '</code>';

      $context = htmlspecialchars(
        $details['entity_type'] . '/' . ($details['bundle'] ?? '-') . '/' . ($details['view_mode'] ?? '-'),
        ENT_QUOTES,
        'UTF-8'
      );

      $fields_html = '-';
      if (!empty($details['fields_used'])) {
        $field_items = [];
        foreach ($details['fields_used'] as $field) {
          $field_items[] = '<code>{{ content.' . htmlspecialchars($field, ENT_QUOTES, 'UTF-8') . ' }}</code>';
        }
        $fields_html = implode('<br>', $field_items);
      }

      $rows[] = $this->ui->row([
        $template_cell,
        $context,
        $fields_html,
      ], 'error');
    }

    $content['table'] = $this->ui->table($headers, $rows);

    // Add solution.
    $solution = (string) $this->t('Solution: Add {{ content }} to your template to ensure cache metadata bubbles correctly. Use {{ content|without(\'field_name\') }} to exclude fields you render separately, or use {{ content|cache_metadata }} from twig_tweak module.');
    $content['solution'] = $this->ui->message($solution, 'info');

    return $this->ui->section(
      (string) $this->t('Cache Bubbling Issues'),
      $content,
      ['count' => $errors_count, 'severity' => 'error']
    );
  }

  /**
   * Builds the anti-patterns section.
   *
   * @param array $files
   *   The analysis files data.
   *
   * @return array
   *   Render array for the section.
   */
  protected function buildAntiPatternsSection(array $files): array {
    $all_antipatterns = array_merge(
      $files['anti_patterns']['results'] ?? [],
      $files['preprocess_antipatterns']['results'] ?? []
    );

    if (empty($all_antipatterns)) {
      return $this->ui->section(
        (string) $this->t('Anti-Patterns'),
        ['message' => $this->ui->message(
          (string) $this->t('No anti-patterns detected in Twig templates or preprocess hooks. Your code follows best practices.'),
          'success'
        )],
        ['count' => 0, 'severity' => 'success']
      );
    }

    // Count by severity.
    $error_count = count(array_filter($all_antipatterns, fn($r) => ($r['severity'] ?? '') === 'error'));
    $warning_count = count(array_filter($all_antipatterns, fn($r) => ($r['severity'] ?? '') === 'warning'));
    $total_count = count($all_antipatterns);

    // Determine overall severity.
    $severity = 'notice';
    if ($error_count > 0) {
      $severity = 'error';
    }
    elseif ($warning_count > 0) {
      $severity = 'warning';
    }

    // Group issues by file for better display.
    $issues_by_file = [];
    foreach ($all_antipatterns as $item) {
      $file = $item['details']['file'] ?? 'unknown';
      $issues_by_file[$file][] = $item;
    }

    $content = [];

    foreach ($issues_by_file as $file => $file_issues) {
      $file_key = 'file_' . md5($file);

      // Build table for this file.
      $headers = [
        $this->ui->header('', 'center', '80px'),
        $this->ui->header((string) $this->t('Line'), 'center', '60px'),
        $this->ui->header((string) $this->t('Code')),
        $this->ui->header((string) $this->t('Message')),
      ];

      $rows = [];
      foreach ($file_issues as $item) {
        $item_severity = $item['severity'] ?? 'notice';
        $details = $item['details'] ?? [];
        $line = $details['line'] ?? 0;
        $function = $details['function'] ?? '';

        $badge_variant = match ($item_severity) {
          'error' => 'error',
          'warning' => 'warning',
          default => 'info',
        };

        $badge = $this->ui->badge(ucfirst($item_severity), $badge_variant);
        $line_cell = $line > 0 ? (string) $line : '-';
        $code_cell = '<code>' . htmlspecialchars($item['code'] ?? '', ENT_QUOTES, 'UTF-8') . '</code>';

        $message = htmlspecialchars($item['message'] ?? '', ENT_QUOTES, 'UTF-8');
        if (!empty($function)) {
          $message = '<strong>' . (string) $this->t('Function:') . '</strong> <code>' . htmlspecialchars($function, ENT_QUOTES, 'UTF-8') . '()</code><br>' . $message;
        }

        $rows[] = $this->ui->row([
          $this->ui->cell($badge, ['align' => 'center']),
          $this->ui->cell($line_cell, ['align' => 'center']),
          $code_cell,
          $message,
        ], $item_severity);
      }

      $file_content = [
        'table' => $this->ui->table($headers, $rows),
      ];

      // Add code context for the first issue if available.
      $first_issue = $file_issues[0] ?? [];
      $code_context = $first_issue['details']['code_context'] ?? [];
      if (!empty($code_context['lines'])) {
        $file_content['code'] = $this->ui->code(
          $code_context['lines'],
          ['severity' => $first_issue['severity'] ?? 'error']
        );
      }

      $file_severity = 'notice';
      $file_errors = count(array_filter($file_issues, fn($r) => ($r['severity'] ?? '') === 'error'));
      $file_warnings = count(array_filter($file_issues, fn($r) => ($r['severity'] ?? '') === 'warning'));
      if ($file_errors > 0) {
        $file_severity = 'error';
      }
      elseif ($file_warnings > 0) {
        $file_severity = 'warning';
      }

      $content[$file_key] = $this->ui->section(
        $file,
        $file_content,
        ['count' => count($file_issues), 'severity' => $file_severity, 'open' => FALSE]
      );
    }

    return $this->ui->section(
      (string) $this->t('Anti-Patterns'),
      $content,
      ['count' => $total_count, 'severity' => $severity]
    );
  }

  /**
   * Builds the field optimization section.
   *
   * @param array $files
   *   The analysis files data.
   *
   * @return array
   *   Render array for the section.
   */
  protected function buildFieldOptimizationSection(array $files): array {
    if (empty($files['field_sync'])) {
      return $this->ui->section(
        (string) $this->t('Field Rendering Optimization'),
        ['message' => $this->ui->message(
          (string) $this->t('No entity templates with ViewMode configurations were found to analyze for field rendering optimization.'),
          'info'
        )],
        ['count' => 0, 'severity' => 'success']
      );
    }

    $field_sync_data = $files['field_sync'];
    $all_field_issues = $field_sync_data['results'] ?? [];
    $stats = $field_sync_data['stats'] ?? [];
    $templates_analyzed = $stats['templates_analyzed'] ?? 0;
    $viewmodes_analyzed = $stats['viewmodes_analyzed'] ?? 0;

    // Count by type.
    $excludable = array_filter($all_field_issues, fn($r) => ($r['details']['type'] ?? '') === 'excludable');
    $missing = array_filter($all_field_issues, fn($r) => ($r['details']['type'] ?? '') === 'missing');
    $unknown = array_filter($all_field_issues, fn($r) => ($r['details']['type'] ?? '') === 'unknown');

    $warning_count = count($excludable) + count($missing);
    $notice_count = count($unknown);
    $total_count = $warning_count + $notice_count;
    $has_issues = $total_count > 0;

    $content = [];

    // Summary message.
    $summary = (string) $this->t('Analysis summary: @templates templates analyzed across @viewmodes ViewMode configurations.', [
      '@templates' => $templates_analyzed,
      '@viewmodes' => $viewmodes_analyzed,
    ]);

    if (!$has_issues) {
      $content['success'] = $this->ui->message(
        (string) $this->t('Excellent! No field rendering issues detected. All fields configured in ViewModes are being properly rendered in their corresponding templates.') . ' ' . $summary,
        'success'
      );

      return $this->ui->section(
        (string) $this->t('Field Rendering Optimization'),
        $content,
        ['count' => 0, 'severity' => 'success']
      );
    }

    // Explanation message.
    $explanation = (string) $this->t('This section analyzes which fields are configured in each ViewMode and whether they are actually being rendered in the corresponding Twig template. Fields that are loaded but not rendered waste memory and processing time.') . ' ' . $summary;
    $content['explanation'] = $this->ui->message($explanation, 'warning');

    // Table headers for all subsections.
    $headers = [
      $this->ui->header((string) $this->t('Field')),
      $this->ui->header((string) $this->t('Template')),
      $this->ui->header((string) $this->t('ViewMode')),
    ];

    // Subsection: Fields excluded with without() - could be disabled.
    if (!empty($excludable)) {
      $excludable_rows = [];
      foreach ($excludable as $item) {
        $details = $item['details'];
        $excludable_rows[] = $this->ui->row([
          '<code>' . htmlspecialchars($details['field'], ENT_QUOTES, 'UTF-8') . '</code>',
          '<code>' . htmlspecialchars($details['file'], ENT_QUOTES, 'UTF-8') . '</code>',
          $details['entity_type'] . '/' . $details['bundle'] . '/' . ($details['view_mode'] ?? 'default'),
        ], 'warning');
      }

      $excludable_content = [
        'desc' => $this->ui->message(
          (string) $this->t('These fields are excluded using content|without() but are still configured as visible. Drupal loads them unnecessarily. Disable them in Structure > Content types > Manage display.'),
          'info'
        ),
        'table' => $this->ui->table($headers, $excludable_rows),
      ];

      $content['excludable'] = $this->ui->section(
        (string) $this->t('Fields excluded with without() - can be disabled'),
        $excludable_content,
        ['count' => count($excludable), 'severity' => 'warning', 'open' => FALSE]
      );
    }

    // Subsection: Fields configured but not rendered.
    if (!empty($missing)) {
      $missing_rows = [];
      foreach ($missing as $item) {
        $details = $item['details'];
        $missing_rows[] = $this->ui->row([
          '<code>' . htmlspecialchars($details['field'], ENT_QUOTES, 'UTF-8') . '</code>',
          '<code>' . htmlspecialchars($details['file'], ENT_QUOTES, 'UTF-8') . '</code>',
          $details['entity_type'] . '/' . $details['bundle'] . '/' . ($details['view_mode'] ?? 'default'),
        ], 'warning');
      }

      $missing_content = [
        'desc' => $this->ui->message(
          (string) $this->t('These fields are visible in the ViewMode but the template does not use {{ content }} and does not render them individually. Either render them or disable them in the ViewMode.'),
          'info'
        ),
        'table' => $this->ui->table($headers, $missing_rows),
      ];

      $content['missing'] = $this->ui->section(
        (string) $this->t('Fields configured but not rendered'),
        $missing_content,
        ['count' => count($missing), 'severity' => 'warning', 'open' => FALSE]
      );
    }

    // Subsection: Fields used but not configured (notices).
    if (!empty($unknown)) {
      $unknown_rows = [];
      foreach ($unknown as $item) {
        $details = $item['details'];
        $unknown_rows[] = $this->ui->row([
          '<code>' . htmlspecialchars($details['field'], ENT_QUOTES, 'UTF-8') . '</code>',
          '<code>' . htmlspecialchars($details['file'], ENT_QUOTES, 'UTF-8') . '</code>',
          $details['entity_type'] . '/' . $details['bundle'] . '/' . ($details['view_mode'] ?? 'default'),
        ]);
      }

      $unknown_content = [
        'desc' => $this->ui->message(
          (string) $this->t('These fields are accessed in the template but are not configured as visible in the ViewMode. This may be intentional if you are accessing field values directly.'),
          'info'
        ),
        'table' => $this->ui->table($headers, $unknown_rows),
      ];

      $content['unknown'] = $this->ui->section(
        (string) $this->t('Fields used in template but not in ViewMode'),
        $unknown_content,
        ['count' => count($unknown), 'severity' => 'notice', 'open' => FALSE]
      );
    }

    // Determine overall severity.
    $severity = 'notice';
    if ($warning_count > 0) {
      $severity = 'warning';
    }

    return $this->ui->section(
      (string) $this->t('Field Rendering Optimization'),
      $content,
      ['count' => $total_count, 'severity' => $severity]
    );
  }

  /**
   * Builds the external libraries section.
   *
   * @param array $files
   *   The analysis files data.
   *
   * @return array
   *   Render array for the section.
   */
  protected function buildExternalLibrariesSection(array $files): array {
    if (empty($files['external_libraries']['results'])) {
      return $this->ui->section(
        (string) $this->t('External Libraries'),
        ['message' => $this->ui->message(
          (string) $this->t('No external library references found. All CSS/JS assets are hosted locally.'),
          'success'
        )],
        ['count' => 0, 'severity' => 'success']
      );
    }

    $external_results = $files['external_libraries']['results'];
    $warning_count = count($external_results);

    $content = [];

    // Explanation of the issue.
    $explanation = (string) $this->t('Loading CSS/JS from external CDNs (unpkg, cdnjs, jsdelivr, etc.) can cause privacy concerns, performance issues, reliability issues, and security concerns. Recommendation: Download and host these libraries locally using Composer or manually.');
    $content['explanation'] = $this->ui->message($explanation, 'warning');

    // Group by file for better organization.
    $issues_by_file = [];
    foreach ($external_results as $item) {
      $file = $item['details']['file'] ?? 'unknown';
      $issues_by_file[$file][] = $item;
    }

    foreach ($issues_by_file as $file => $file_issues) {
      $file_key = 'file_' . md5($file);

      // Build table for this file.
      $headers = [
        $this->ui->header((string) $this->t('Library')),
        $this->ui->header((string) $this->t('Type'), 'center', '80px'),
        $this->ui->header((string) $this->t('External URLs')),
      ];

      $rows = [];
      foreach ($file_issues as $item) {
        $details = $item['details'] ?? [];
        $library = $details['library'] ?? 'unknown';
        $assets = $details['assets'] ?? [];

        $css_count = $details['css_count'] ?? 0;
        $js_count = $details['js_count'] ?? 0;
        $type_parts = [];
        if ($css_count > 0) {
          $type_parts[] = $this->ui->badge($css_count . ' CSS', 'info');
        }
        if ($js_count > 0) {
          $type_parts[] = $this->ui->badge($js_count . ' JS', 'info');
        }
        $type_cell = implode(' ', $type_parts);

        $urls_html = [];
        foreach ($assets as $asset) {
          $urls_html[] = '<code>' . htmlspecialchars($asset['url'] ?? '', ENT_QUOTES, 'UTF-8') . '</code>';
        }

        $rows[] = $this->ui->row([
          '<code>' . htmlspecialchars($library, ENT_QUOTES, 'UTF-8') . '</code>',
          $this->ui->cell($type_cell, ['align' => 'center']),
          implode('<br>', $urls_html),
        ], 'warning');
      }

      $file_content = [
        'table' => $this->ui->table($headers, $rows),
      ];

      $content[$file_key] = $this->ui->section(
        $file,
        $file_content,
        ['count' => count($file_issues), 'severity' => 'warning', 'open' => FALSE]
      );
    }

    return $this->ui->section(
      (string) $this->t('External Libraries'),
      $content,
      ['count' => $warning_count, 'severity' => 'warning']
    );
  }

  /**
   * Cache for anti-patterns indexed by code for O(1) lookup.
   *
   * @var array<string, array>|null
   */
  protected static ?array $antiPatternsByCode = NULL;

  /**
   * Gets anti-patterns indexed by code for fast lookup.
   *
   * @return array<string, array>
   *   Anti-patterns array keyed by code.
   */
  protected function getAntiPatternsByCode(): array {
    if (self::$antiPatternsByCode === NULL) {
      self::$antiPatternsByCode = [];
      foreach (self::ANTI_PATTERNS as $pattern) {
        if (isset($pattern['code'])) {
          self::$antiPatternsByCode[$pattern['code']] = $pattern;
        }
      }
    }
    return self::$antiPatternsByCode;
  }

  /**
   * Gets the fix description for an anti-pattern code.
   *
   * @param string $code
   *   The anti-pattern code (e.g., 'KINT_IN_TWIG').
   *
   * @return string|null
   *   The HTML fix description, or NULL if not found.
   */
  protected function getAntiPatternFix(string $code): ?string {
    return $this->getAntiPatternsByCode()[$code]['fix'] ?? NULL;
  }

  /**
   * Gets the classification tags for an anti-pattern code.
   *
   * @param string $code
   *   The anti-pattern code (e.g., 'KINT_IN_TWIG').
   *
   * @return array
   *   Array of tags, or empty array if not found.
   */
  protected function getAntiPatternTags(string $code): array {
    return $this->getAntiPatternsByCode()[$code]['tags'] ?? [];
  }

}
