<?php

declare(strict_types=1);

namespace Drupal\audit_updates\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ThemeExtensionList;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

// Define update status constants if not available.
// These are defined in core/modules/update/update.module.
if (!defined('UPDATE_NOT_SECURE')) {
  define('UPDATE_NOT_SECURE', 1);
}
if (!defined('UPDATE_REVOKED')) {
  define('UPDATE_REVOKED', 2);
}
if (!defined('UPDATE_NOT_SUPPORTED')) {
  define('UPDATE_NOT_SUPPORTED', 3);
}
if (!defined('UPDATE_NOT_CURRENT')) {
  define('UPDATE_NOT_CURRENT', 4);
}
if (!defined('UPDATE_CURRENT')) {
  define('UPDATE_CURRENT', 5);
}

/**
 * Analyzes pending module and theme updates.
 */
#[AuditAnalyzer(
  id: 'updates',
  label: new TranslatableMarkup('Updates Audit'),
  description: new TranslatableMarkup('Analyzes pending security and regular updates for modules and themes.'),
  menu_title: new TranslatableMarkup('Updates'),
  output_directory: 'updates',
  weight: 5,
)]
class UpdatesAnalyzer extends AuditAnalyzerBase {

  /**
   * Score weights for different factors.
   *
   * Security updates are weighted much higher than regular updates.
   * Update health affects the overall reliability of update checks.
   */
  protected const SCORE_WEIGHTS = [
    'security_updates' => 60,
    'pending_updates' => 25,
    'update_health' => 15,
  ];

  protected ModuleHandlerInterface $moduleHandler;
  protected ConfigFactoryInterface $configFactory;
  protected ModuleExtensionList $moduleExtensionList;
  protected ThemeExtensionList $themeExtensionList;
  protected Connection $database;
  protected KeyValueFactoryInterface $keyValueFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->moduleHandler = $container->get('module_handler');
    $instance->configFactory = $container->get('config.factory');
    $instance->moduleExtensionList = $container->get('extension.list.module');
    $instance->themeExtensionList = $container->get('extension.list.theme');
    $instance->database = $container->get('database');
    $instance->keyValueFactory = $container->get('keyvalue');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $config): array {
    return [
      'ignore_regular_updates' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Ignore regular updates in scoring'),
        '#description' => $this->t('Only penalize the score for security updates. Regular (non-security) updates will still be reported but won\'t affect the audit score. Useful for stable projects focused on security maintenance only.'),
        '#default_value' => $config['ignore_regular_updates'] ?? FALSE,
      ],
    ];
  }

  /**
   * Checks if regular updates should be ignored in scoring.
   */
  protected function ignoreRegularUpdates(): bool {
    return (bool) $this->configFactory->get('audit_updates.settings')->get('ignore_regular_updates');
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    $security_results = $this->analyzeSecurityUpdates();
    $regular_results = $this->analyzeRegularUpdates();
    $compatibility_results = $this->analyzeCompatibility();
    $update_health_results = $this->analyzeUpdateHealth();

    $scores = $this->calculateScores($security_results, $regular_results, $update_health_results);

    return [
      '_files' => [
        'security' => $security_results,
        'regular' => $regular_results,
        'compatibility' => $compatibility_results,
        'update_health' => $update_health_results,
      ],
      'score' => $scores,
    ];
  }

  /**
   * Analyzes security updates.
   *
   * @return array
   *   The analysis results for security updates.
   */
  protected function analyzeSecurityUpdates(): array {
    $results = [];
    $errors = 0;

    $project_data = $this->getProjectData();

    if ($project_data === NULL) {
      $results[] = $this->createResultItem(
        'error',
        'UPDATE_CHECK_FAILED',
        (string) $this->t('Could not retrieve update information. Ensure cron is running and check internet connectivity.'),
        ['update_check_failed' => TRUE]
      );
      return $this->createResult($results, 1, 0, 0);
    }

    foreach ($project_data as $project_name => $project) {
      $status = $project['status'] ?? NULL;

      if ($status === UPDATE_NOT_SECURE) {
        $results[] = $this->createResultItem(
          'error',
          'SECURITY_UPDATE',
          (string) $this->t('Security update available for @name: @current → @recommended', [
            '@name' => $project_name,
            '@current' => $project['existing_version'] ?? 'unknown',
            '@recommended' => $project['recommended'] ?? 'unknown',
          ]),
          [
            'name' => $project_name,
            'label' => $project['title'] ?? $project_name,
            'current_version' => $project['existing_version'] ?? 'unknown',
            'recommended_version' => $project['recommended'] ?? 'unknown',
            'project_type' => $project['project_type'] ?? 'module',
            'link' => $project['link'] ?? NULL,
            'release_link' => $project['releases'][$project['recommended']]['release_link'] ?? NULL,
          ]
        );
        $errors++;
      }
    }

    if ($errors === 0) {
      $results[] = $this->createResultItem(
        'notice',
        'NO_SECURITY_UPDATES',
        (string) $this->t('No security updates pending.'),
        []
      );
    }

    return $this->createResult($results, $errors, 0, $errors === 0 ? 1 : 0);
  }

  /**
   * Analyzes regular (non-security) updates.
   */
  protected function analyzeRegularUpdates(): array {
    $results = [];
    $warnings = 0;
    $notices = 0;

    $ignore_regular = $this->ignoreRegularUpdates();
    $project_data = $this->getProjectData();

    if ($project_data === NULL) {
      $results[] = $this->createResultItem(
        'error',
        'UPDATE_CHECK_FAILED',
        (string) $this->t('Could not retrieve update information.'),
        ['update_check_failed' => TRUE]
      );
      return $this->createResult($results, 1, 0, 0);
    }

    foreach ($project_data as $project_name => $project) {
      $status = $project['status'] ?? NULL;

      if ($status === UPDATE_NOT_CURRENT) {
        // Use notice instead of warning when regular updates are ignored.
        $severity = $ignore_regular ? 'notice' : 'warning';

        $results[] = $this->createResultItem(
          $severity,
          'UPDATE_AVAILABLE',
          (string) $this->t('Update available for @name: @current → @recommended', [
            '@name' => $project_name,
            '@current' => $project['existing_version'] ?? 'unknown',
            '@recommended' => $project['recommended'] ?? 'unknown',
          ]),
          [
            'name' => $project_name,
            'label' => $project['title'] ?? $project_name,
            'current_version' => $project['existing_version'] ?? 'unknown',
            'recommended_version' => $project['recommended'] ?? 'unknown',
            'project_type' => $project['project_type'] ?? 'module',
            'link' => $project['link'] ?? NULL,
            'release_link' => $project['releases'][$project['recommended']]['release_link'] ?? NULL,
          ]
        );

        if ($ignore_regular) {
          $notices++;
        }
        else {
          $warnings++;
        }
      }
    }

    if ($warnings === 0 && $notices === 0) {
      $results[] = $this->createResultItem(
        'notice',
        'ALL_UP_TO_DATE',
        (string) $this->t('All modules and themes are up to date.'),
        []
      );
      $notices++;
    }

    return $this->createResult($results, 0, $warnings, $notices);
  }

  /**
   * Gets project data from the update module.
   *
   * @return array|null
   *   The project data array, or NULL if unavailable.
   */
  protected function getProjectData(): ?array {
    static $project_data = NULL;

    if ($project_data !== NULL) {
      return $project_data;
    }

    // Load the update module file to ensure constants are defined.
    $this->moduleHandler->loadInclude('update', 'module');
    $this->moduleHandler->loadInclude('update', 'inc', 'update.compare');

    if (!function_exists('update_get_available')) {
      return NULL;
    }

    $available = update_get_available(TRUE);

    if (empty($available)) {
      return NULL;
    }

    if (!function_exists('update_calculate_project_data')) {
      return NULL;
    }

    $project_data = update_calculate_project_data($available);

    return $project_data ?: NULL;
  }

  /**
   * Analyzes module and theme compatibility with the next Drupal major version.
   *
   * @return array
   *   The analysis results for compatibility, separated by contrib and custom.
   */
  protected function analyzeCompatibility(): array {
    $results = [];
    $notices = 0;

    $current_version = $this->getCurrentDrupalMajorVersion();
    $next_version = $current_version + 1;

    // Get extensions by type.
    $contrib_modules = $this->getInstalledExtensions('module', 'contrib');
    $custom_modules = $this->getInstalledExtensions('module', 'custom');
    $contrib_themes = $this->getInstalledExtensions('theme', 'contrib');
    $custom_themes = $this->getInstalledExtensions('theme', 'custom');

    // Analyze contrib modules.
    $contrib_incompatible = [];
    foreach ($contrib_modules as $name => $info) {
      if (!$this->checkExtensionCompatibility($info, $next_version)) {
        $contrib_incompatible[] = $this->createCompatibilityItem($name, $info, $next_version, 'module', 'contrib');
        $notices++;
      }
    }

    // Analyze contrib themes.
    foreach ($contrib_themes as $name => $info) {
      if (!$this->checkExtensionCompatibility($info, $next_version)) {
        $contrib_incompatible[] = $this->createCompatibilityItem($name, $info, $next_version, 'theme', 'contrib');
        $notices++;
      }
    }

    // Analyze custom modules.
    $custom_incompatible = [];
    foreach ($custom_modules as $name => $info) {
      if (!$this->checkExtensionCompatibility($info, $next_version)) {
        $custom_incompatible[] = $this->createCompatibilityItem($name, $info, $next_version, 'module', 'custom');
        $notices++;
      }
    }

    // Analyze custom themes.
    foreach ($custom_themes as $name => $info) {
      if (!$this->checkExtensionCompatibility($info, $next_version)) {
        $custom_incompatible[] = $this->createCompatibilityItem($name, $info, $next_version, 'theme', 'custom');
        $notices++;
      }
    }

    // Build results with separation.
    if (!empty($contrib_incompatible)) {
      $results[] = $this->createResultItem(
        'notice',
        'CONTRIB_INCOMPATIBLE',
        (string) $this->t('@count contributed extension(s) not compatible with Drupal @version', [
          '@count' => count($contrib_incompatible),
          '@version' => $next_version,
        ]),
        [
          'items' => $contrib_incompatible,
          'next_drupal_version' => $next_version,
          'source' => 'contrib',
        ]
      );
    }

    if (!empty($custom_incompatible)) {
      $results[] = $this->createResultItem(
        'notice',
        'CUSTOM_INCOMPATIBLE',
        (string) $this->t('@count custom extension(s) not compatible with Drupal @version', [
          '@count' => count($custom_incompatible),
          '@version' => $next_version,
        ]),
        [
          'items' => $custom_incompatible,
          'next_drupal_version' => $next_version,
          'source' => 'custom',
        ]
      );
    }

    if (empty($contrib_incompatible) && empty($custom_incompatible)) {
      $results[] = $this->createResultItem(
        'notice',
        'ALL_COMPATIBLE',
        (string) $this->t('All installed modules and themes are compatible with Drupal @version.', [
          '@version' => $next_version,
        ]),
        ['next_drupal_version' => $next_version]
      );
    }

    return $this->createResult($results, 0, 0, $notices > 0 ? $notices : 1);
  }

  /**
   * Analyzes the health of the update system.
   *
   * Detects silent failures in Drupal's Update module caused by
   * desynchronization between the key-value store and the queue system.
   * This is a known issue documented in Drupal core issue #2920285.
   *
   * @return array
   *   The analysis results for update system health.
   *
   * @see https://www.drupal.org/project/drupal/issues/2920285
   */
  protected function analyzeUpdateHealth(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;

    // Get orphaned tasks from key-value store.
    $orphaned_tasks = $this->detectOrphanedTasks();

    // Get projects without release data.
    $projects_without_data = $this->detectProjectsWithoutReleaseData();

    // Check last update check timestamp.
    $last_check = \Drupal::state()->get('update.last_check', 0);
    $last_check_age = time() - $last_check;
    $one_week = 7 * 24 * 60 * 60;

    // Report orphaned tasks (these cause silent failures).
    if (!empty($orphaned_tasks)) {
      $results[] = $this->createResultItem(
        'error',
        'ORPHANED_FETCH_TASKS',
        (string) $this->t('@count project(s) have orphaned fetch tasks that prevent update checks from completing.', [
          '@count' => count($orphaned_tasks),
        ]),
        [
          'projects' => $orphaned_tasks,
          'issue_link' => 'https://www.drupal.org/project/drupal/issues/2920285',
          'fix_command' => "\\Drupal::keyValue('update_fetch_task')->deleteAll();",
        ]
      );
      $errors++;
    }

    // Report projects without release data.
    if (!empty($projects_without_data)) {
      $results[] = $this->createResultItem(
        'warning',
        'NO_RELEASE_DATA',
        (string) $this->t('@count project(s) have no release data available.', [
          '@count' => count($projects_without_data),
        ]),
        [
          'projects' => $projects_without_data,
        ]
      );
      $warnings++;
    }

    // Check if update data is stale.
    if ($last_check === 0) {
      $results[] = $this->createResultItem(
        'warning',
        'NEVER_CHECKED',
        (string) $this->t('Update status has never been checked. Run cron or manually check for updates.'),
        []
      );
      $warnings++;
    }
    elseif ($last_check_age > $one_week) {
      $results[] = $this->createResultItem(
        'warning',
        'STALE_DATA',
        (string) $this->t('Update data is @days days old. Consider running cron more frequently.', [
          '@days' => floor($last_check_age / 86400),
        ]),
        [
          'last_check' => $last_check,
          'last_check_formatted' => \Drupal::service('date.formatter')->format($last_check, 'short'),
        ]
      );
      $warnings++;
    }

    // If no issues found, report healthy status.
    if ($errors === 0 && $warnings === 0) {
      $results[] = $this->createResultItem(
        'notice',
        'UPDATE_SYSTEM_HEALTHY',
        (string) $this->t('Update system is functioning correctly.'),
        [
          'last_check' => $last_check,
          'last_check_formatted' => $last_check ? \Drupal::service('date.formatter')->format($last_check, 'short') : NULL,
        ]
      );
    }

    return $this->createResult($results, $errors, $warnings, $errors === 0 && $warnings === 0 ? 1 : 0);
  }

  /**
   * Detects orphaned tasks in the update_fetch_task key-value store.
   *
   * These are tasks that exist in the key-value store but have no
   * corresponding entry in the queue, causing silent failures.
   *
   * @return array
   *   Array of project names with orphaned tasks.
   */
  protected function detectOrphanedTasks(): array {
    $orphaned = [];

    try {
      // Get all entries from the update_fetch_task key-value store.
      $fetch_task_store = $this->keyValueFactory->get('update_fetch_task');
      $fetch_tasks = $fetch_task_store->getAll();

      if (empty($fetch_tasks)) {
        return [];
      }

      // Check if there are corresponding queue items.
      // The queue table may not exist or be empty, which is fine.
      $queue_count = 0;
      try {
        if ($this->database->schema()->tableExists('queue')) {
          $queue_count = (int) $this->database->select('queue', 'q')
            ->condition('name', 'update_fetch_tasks')
            ->countQuery()
            ->execute()
            ->fetchField();
        }
      }
      catch (\Exception $e) {
        // Queue table doesn't exist or is inaccessible.
      }

      // If we have fetch tasks but no queue items, all tasks are orphaned.
      // This is the classic desynchronization issue.
      if ($queue_count === 0 && !empty($fetch_tasks)) {
        foreach ($fetch_tasks as $project => $task) {
          $orphaned[] = [
            'name' => $project,
            'stuck_since' => $task['last_fetch'] ?? NULL,
          ];
        }
      }
    }
    catch (\Exception $e) {
      // Unable to check key-value store.
      \Drupal::logger('audit_updates')->warning('Failed to check update_fetch_task store: @message', [
        '@message' => $e->getMessage(),
      ]);
    }

    return $orphaned;
  }

  /**
   * Detects installed projects that have no release data available.
   *
   * @return array
   *   Array of project info without release data.
   */
  protected function detectProjectsWithoutReleaseData(): array {
    $missing = [];

    // Get all installed modules and themes.
    $installed_modules = $this->moduleExtensionList->getAllInstalledInfo();
    $installed_themes = $this->themeExtensionList->getAllInstalledInfo();
    $installed = array_merge($installed_modules, $installed_themes);

    // Get available update data.
    $this->moduleHandler->loadInclude('update', 'module');
    $this->moduleHandler->loadInclude('update', 'inc', 'update.compare');

    if (!function_exists('update_get_available')) {
      return [];
    }

    $available = update_get_available(FALSE);

    // Build list of projects that should have update data.
    $contrib_projects = [];
    foreach ($installed as $name => $info) {
      // Skip core extensions.
      if (!empty($info['package']) && $info['package'] === 'Core') {
        continue;
      }

      // Get the extension path.
      $path = '';
      try {
        if (isset($installed_modules[$name])) {
          $path = $this->moduleExtensionList->getPath($name);
        }
        elseif (isset($installed_themes[$name])) {
          $path = $this->themeExtensionList->getPath($name);
        }
      }
      catch (\Exception $e) {
        continue;
      }

      // Skip core and custom extensions.
      if (str_starts_with($path, 'core/')) {
        continue;
      }
      if (str_contains($path, '/custom/')) {
        continue;
      }

      // Only check contrib modules.
      if (str_contains($path, '/contrib/')) {
        $contrib_projects[$name] = $info;
      }
    }

    // Check which contrib projects don't have release data.
    foreach ($contrib_projects as $name => $info) {
      // The project name might differ from the module name.
      $project_name = $info['project'] ?? $name;

      if (!isset($available[$project_name]) && !isset($available[$name])) {
        $missing[] = [
          'name' => $name,
          'label' => $info['name'] ?? $name,
          'version' => $info['version'] ?? 'unknown',
          'path' => $path,
        ];
      }
    }

    return $missing;
  }

  /**
   * Creates a compatibility item for an incompatible extension.
   *
   * @param string $name
   *   The extension machine name.
   * @param array $info
   *   The extension info array.
   * @param int $next_version
   *   The next Drupal major version.
   * @param string $extension_type
   *   Either 'module' or 'theme'.
   * @param string $source
   *   Either 'contrib' or 'custom'.
   *
   * @return array
   *   The compatibility item.
   */
  protected function createCompatibilityItem(string $name, array $info, int $next_version, string $extension_type, string $source): array {
    $core_requirement = $info['core_version_requirement'] ?? $info['core'] ?? NULL;

    $item = [
      'name' => $name,
      'label' => $info['name'] ?? $name,
      'current_version' => $info['version'] ?? 'unknown',
      'core_version_requirement' => $core_requirement ?? (string) $this->t('Not specified'),
      'next_drupal_version' => $next_version,
      'extension_type' => $extension_type,
      'source' => $source,
    ];

    // Only add drupal.org link for contrib.
    if ($source === 'contrib') {
      $item['project_link'] = 'https://www.drupal.org/project/' . $name;
    }

    return $item;
  }

  /**
   * Checks if an extension is compatible with a Drupal version.
   *
   * @param array $info
   *   The extension info array.
   * @param int $version
   *   The Drupal major version to check.
   *
   * @return bool
   *   TRUE if compatible, FALSE otherwise.
   */
  protected function checkExtensionCompatibility(array $info, int $version): bool {
    $core_requirement = $info['core_version_requirement'] ?? NULL;

    // If no core_version_requirement, check legacy 'core' key.
    if ($core_requirement === NULL && isset($info['core'])) {
      $core_requirement = $info['core'];
    }

    // If no version requirement found, assume not compatible.
    if ($core_requirement === NULL) {
      return FALSE;
    }

    return $this->isCompatibleWithVersion($core_requirement, $version);
  }

  /**
   * Gets the current Drupal major version.
   *
   * @return int
   *   The major version number (e.g., 10, 11).
   */
  protected function getCurrentDrupalMajorVersion(): int {
    return (int) explode('.', \Drupal::VERSION)[0];
  }

  /**
   * Gets installed extensions filtered by type and source.
   *
   * @param string $extension_type
   *   Either 'module' or 'theme'.
   * @param string $source
   *   Either 'contrib' or 'custom'.
   *
   * @return array
   *   Array of extension info arrays keyed by extension name.
   */
  protected function getInstalledExtensions(string $extension_type, string $source): array {
    $extensions = [];

    if ($extension_type === 'module') {
      $installed = $this->moduleExtensionList->getAllInstalledInfo();
      $extension_list = $this->moduleExtensionList;
      $path_pattern = $source === 'contrib' ? 'modules/contrib/' : 'modules/custom/';
    }
    else {
      $installed = $this->themeExtensionList->getAllInstalledInfo();
      $extension_list = $this->themeExtensionList;
      $path_pattern = $source === 'contrib' ? 'themes/contrib/' : 'themes/custom/';
    }

    foreach ($installed as $name => $info) {
      // Skip core extensions.
      if (!empty($info['package']) && $info['package'] === 'Core') {
        continue;
      }

      // Get the extension path.
      $path = $extension_list->getPath($name);

      // Skip extensions in core directory.
      if (str_starts_with($path, 'core/')) {
        continue;
      }

      // Check if extension is in the expected directory.
      if (!str_contains($path, $path_pattern)) {
        continue;
      }

      $extensions[$name] = $info;
    }

    return $extensions;
  }

  /**
   * Checks if a core_version_requirement is compatible with a version.
   *
   * @param string $requirement
   *   The core_version_requirement string (e.g., "^9 || ^10 || ^11").
   * @param int $version
   *   The version number to check (e.g., 11).
   *
   * @return bool
   *   TRUE if compatible, FALSE otherwise.
   */
  protected function isCompatibleWithVersion(string $requirement, int $version): bool {
    // Normalize the requirement string.
    $requirement = strtolower(trim($requirement));

    // Handle different formats:
    // - "^10" - compatible with 10.x
    // - "^9 || ^10" - compatible with 9.x or 10.x
    // - ">=9" - compatible with 9 and above
    // - "~10.0" - compatible with 10.0.x and above in 10.x
    // - "10.x" - legacy format for Drupal 10
    // - "8.x-2.x || ^10" - mixed legacy and new format.
    $parts = preg_split('/\s*\|\|\s*/', $requirement);

    foreach ($parts as $part) {
      $part = trim($part);

      // Handle caret notation (^10, ^11).
      if (preg_match('/^\^(\d+)/', $part, $matches)) {
        if ((int) $matches[1] === $version) {
          return TRUE;
        }
      }

      // Handle >= notation (>=10).
      if (preg_match('/^>=\s*(\d+)/', $part, $matches)) {
        if ($version >= (int) $matches[1]) {
          return TRUE;
        }
      }

      // Handle exact version or tilde notation (~10).
      if (preg_match('/^~?(\d+)/', $part, $matches)) {
        if ((int) $matches[1] === $version) {
          return TRUE;
        }
      }

      // Handle legacy 8.x-X.x or 9.x format.
      if (preg_match('/^(\d+)\.x/', $part, $matches)) {
        if ((int) $matches[1] === $version) {
          return TRUE;
        }
      }
    }

    return FALSE;
  }

  /**
   * Calculates scores for all factors.
   */
  protected function calculateScores(array $security_results, array $regular_results, array $update_health_results): array {
    $factors = [];
    $ignore_regular = $this->ignoreRegularUpdates();

    // Check if update check failed.
    $update_check_failed = FALSE;
    foreach ($security_results['results'] ?? [] as $item) {
      if (!empty($item['details']['update_check_failed'])) {
        $update_check_failed = TRUE;
        break;
      }
    }

    // Security updates score.
    if ($update_check_failed) {
      $security_score = 0;
      $security_description = (string) $this->t('Cannot verify - update check failed');
    }
    else {
      $security_count = $security_results['summary']['errors'] ?? 0;
      $security_score = $security_count === 0 ? 100 : max(0, 100 - ($security_count * 25));
      $security_description = $security_count === 0
        ? (string) $this->t('No security updates pending')
        : (string) $this->t('@count security update(s) pending', ['@count' => $security_count]);
    }

    // Calculate total weight based on whether regular updates are ignored.
    $security_weight = $ignore_regular
      ? (self::SCORE_WEIGHTS['security_updates'] + self::SCORE_WEIGHTS['pending_updates'])
      : self::SCORE_WEIGHTS['security_updates'];

    $factors['security_updates'] = [
      'score' => $security_score,
      'weight' => $security_weight,
      'label' => (string) $this->t('Security Updates'),
      'description' => $security_description,
    ];

    // Regular updates score (only if not ignored).
    if (!$ignore_regular) {
      if ($update_check_failed) {
        $updates_score = 0;
        $updates_description = (string) $this->t('Cannot verify - update check failed');
      }
      else {
        $update_count = $regular_results['summary']['warnings'] ?? 0;
        $updates_score = $update_count === 0 ? 100 : max(50, 100 - ($update_count * 5));
        $updates_description = $update_count === 0
          ? (string) $this->t('All modules and themes up to date')
          : (string) $this->t('@count update(s) available', ['@count' => $update_count]);
      }

      $factors['pending_updates'] = [
        'score' => $updates_score,
        'weight' => self::SCORE_WEIGHTS['pending_updates'],
        'label' => (string) $this->t('Regular Updates'),
        'description' => $updates_description,
      ];
    }

    // Update health score - detects silent failures.
    $health_errors = $update_health_results['summary']['errors'] ?? 0;
    $health_warnings = $update_health_results['summary']['warnings'] ?? 0;

    if ($health_errors > 0) {
      // Orphaned tasks are critical - they cause silent failures.
      $health_score = 0;
      $health_description = (string) $this->t('Silent failures detected - update checks are not completing');
    }
    elseif ($health_warnings > 0) {
      // Warnings reduce the score but don't zero it out.
      $health_score = max(50, 100 - ($health_warnings * 25));
      $health_description = (string) $this->t('@count issue(s) detected', ['@count' => $health_warnings]);
    }
    else {
      $health_score = 100;
      $health_description = (string) $this->t('Update system is healthy');
    }

    $factors['update_health'] = [
      'score' => $health_score,
      'weight' => self::SCORE_WEIGHTS['update_health'],
      'label' => (string) $this->t('Update System Health'),
      'description' => $health_description,
    ];

    return [
      'factors' => $factors,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      'update_health' => [
        'label' => $this->t('Update System Health'),
        'description' => $this->t('Detects silent failures in the Drupal Update module caused by desynchronization between queue and key-value stores. See <a href="https://www.drupal.org/project/drupal/issues/2920285" target="_blank">issue #2920285</a>.'),
        'file_key' => 'update_health',
        'affects_score' => TRUE,
        'score_factor_key' => 'update_health',
      ],
      'security' => [
        'label' => $this->t('Security Updates'),
        'description' => $this->t('Checks for pending security updates that need immediate attention.'),
        'file_key' => 'security',
        'affects_score' => TRUE,
        'score_factor_key' => 'security_updates',
      ],
      'regular' => [
        'label' => $this->t('Regular Updates'),
        'description' => $this->t('Checks for available regular (non-security) updates to modules and themes.'),
        'file_key' => 'regular',
        'affects_score' => TRUE,
        'score_factor_key' => 'pending_updates',
      ],
      'compatibility' => [
        'label' => $this->t('Next Version Compatibility'),
        'description' => $this->t('Lists contributed modules that are not yet compatible with the next major Drupal version.'),
        'file_key' => 'compatibility',
        'affects_score' => FALSE,
        'score_factor_key' => NULL,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    $files = $data['_files'] ?? [];

    return match ($check_id) {
      'update_health' => $this->buildUpdateHealthContent($files),
      'security' => $this->buildSecurityContent($files),
      'regular' => $this->buildRegularContent($files),
      'compatibility' => $this->buildCompatibilityContent($files),
      default => [],
    };
  }

  /**
   * Builds the update health content.
   *
   * Displays information about silent failures in the Update module.
   */
  protected function buildUpdateHealthContent(array $files): array {
    $update_health = $files['update_health'] ?? [];
    $results = $update_health['results'] ?? [];

    $content = [];

    foreach ($results as $item) {
      $code = $item['code'] ?? '';
      $details = $item['details'] ?? [];

      switch ($code) {
        case 'ORPHANED_FETCH_TASKS':
          // This is a critical issue - show prominent error.
          $content['orphaned_error'] = $this->ui->message(
            (string) $this->t('Silent failures detected: @count project(s) have orphaned fetch tasks that prevent update checks from completing. This is a known Drupal core bug.', [
              '@count' => count($details['projects'] ?? []),
            ]),
            'error'
          );

          // Show the affected projects.
          if (!empty($details['projects'])) {
            $headers = [
              $this->ui->header((string) $this->t('Project')),
            ];

            $rows = [];
            foreach ($details['projects'] as $project) {
              $rows[] = $this->ui->row([
                $this->ui->cell($project['name'] ?? (string) $this->t('Unknown')),
              ], 'error');
            }

            $content['orphaned_table'] = $this->ui->table($headers, $rows, [
              'empty' => (string) $this->t('No orphaned tasks found.'),
            ]);
          }

          // Show fix instructions.
          $content['orphaned_fix'] = [
            '#type' => 'details',
            '#title' => $this->t('How to fix this issue'),
            '#open' => TRUE,
            'description' => [
              '#markup' => '<p>' . $this->t('Run this command via Drush to clear the orphaned tasks:') . '</p>' .
                '<pre><code>drush php:eval "\\Drupal::keyValue(\'update_fetch_task\')->deleteAll();"</code></pre>' .
                '<p>' . $this->t('After running this command, clear the cache and run cron to re-fetch update information.') . '</p>' .
                '<p>' . $this->t('For more information, see <a href="@link" target="_blank">Drupal issue #2920285</a>.', [
                  '@link' => $details['issue_link'] ?? 'https://www.drupal.org/project/drupal/issues/2920285',
                ]) . '</p>',
            ],
          ];
          break;

        case 'NO_RELEASE_DATA':
          $content['no_data_warning'] = $this->ui->message(
            (string) $this->t('@count project(s) have no release data available. These projects may not be reporting security updates correctly.', [
              '@count' => count($details['projects'] ?? []),
            ]),
            'warning'
          );

          if (!empty($details['projects'])) {
            $headers = [
              $this->ui->header((string) $this->t('Project')),
              $this->ui->header((string) $this->t('Version'), 'left', '120px'),
              $this->ui->header((string) $this->t('Path'), 'left', '250px'),
            ];

            $rows = [];
            foreach ($details['projects'] as $project) {
              $rows[] = $this->ui->row([
                $this->ui->cell($this->ui->itemName(
                  $project['label'] ?? $project['name'],
                  $project['name']
                )),
                $this->ui->cell($project['version'] ?? 'unknown', ['nowrap' => TRUE]),
                $this->ui->cell($project['path'] ?? '', ['nowrap' => TRUE]),
              ], 'warning');
            }

            $content['no_data_table'] = $this->ui->table($headers, $rows, [
              'empty' => (string) $this->t('No projects without data found.'),
            ]);
          }

          $content['no_data_help'] = [
            '#markup' => '<p class="audit-help">' . $this->t('This can happen when: (1) the project is installed via Composer without proper metadata, (2) the project is a dev version, or (3) network issues prevented fetching release data. Try running cron or clearing the update cache.') . '</p>',
          ];
          break;

        case 'NEVER_CHECKED':
          $content['never_checked'] = $this->ui->message(
            (string) $this->t('Update status has never been checked. Run cron or manually check for updates from the admin reports page.'),
            'warning'
          );
          break;

        case 'STALE_DATA':
          $content['stale_data'] = $this->ui->message(
            (string) $this->t('Update data is @days days old (last checked: @date). Consider running cron more frequently to ensure security updates are detected promptly.', [
              '@days' => floor((time() - ($details['last_check'] ?? 0)) / 86400),
              '@date' => $details['last_check_formatted'] ?? (string) $this->t('Unknown'),
            ]),
            'warning'
          );
          break;

        case 'UPDATE_SYSTEM_HEALTHY':
          $last_check_info = '';
          if (!empty($details['last_check_formatted'])) {
            $last_check_info = ' ' . (string) $this->t('Last checked: @date.', [
              '@date' => $details['last_check_formatted'],
            ]);
          }
          $content['healthy'] = $this->ui->message(
            (string) $this->t('Update system is functioning correctly. No silent failures detected.') . $last_check_info,
            'success'
          );
          break;
      }
    }

    if (empty($content)) {
      return $this->ui->message(
        (string) $this->t('Unable to determine update system health.'),
        'info'
      );
    }

    return $content;
  }

  /**
   * Builds the security updates content.
   */
  protected function buildSecurityContent(array $files): array {
    $security = $files['security'] ?? [];
    $results = $security['results'] ?? [];

    $security_items = [];
    $has_error = FALSE;

    foreach ($results as $item) {
      if (!empty($item['details']['update_check_failed'])) {
        $has_error = TRUE;
      }
      elseif (($item['code'] ?? '') === 'SECURITY_UPDATE') {
        $security_items[] = $item;
      }
    }

    if ($has_error) {
      return $this->ui->message(
        (string) $this->t('Could not retrieve update information. Ensure cron is running and check your internet connectivity.'),
        'error'
      );
    }

    if (count($security_items) === 0) {
      return $this->ui->message(
        (string) $this->t('No security updates pending.'),
        'success'
      );
    }

    return $this->buildUpdatesTable($security_items, TRUE);
  }

  /**
   * Builds the regular updates content.
   */
  protected function buildRegularContent(array $files): array {
    $regular = $files['regular'] ?? [];
    $results = $regular['results'] ?? [];
    $ignore_regular = $this->ignoreRegularUpdates();

    $update_items = [];
    foreach ($results as $item) {
      if (($item['code'] ?? '') === 'UPDATE_AVAILABLE') {
        $update_items[] = $item;
      }
    }

    $count = count($update_items);

    if ($count === 0) {
      return $this->ui->message(
        (string) $this->t('All modules and themes are up to date.'),
        'success'
      );
    }

    $content = [];

    if ($ignore_regular) {
      $content['info'] = $this->ui->message(
        (string) $this->t('Regular updates are configured to not affect the score. Only security updates are considered.'),
        'warning'
      );
    }

    $content['table'] = $this->buildUpdatesTable($update_items, FALSE, $ignore_regular);

    return $content;
  }

  /**
   * Builds the compatibility content with separate tables for contrib/custom.
   */
  protected function buildCompatibilityContent(array $files): array {
    $compatibility = $files['compatibility'] ?? [];
    $results = $compatibility['results'] ?? [];

    $contrib_items = [];
    $custom_items = [];
    $next_version = NULL;

    foreach ($results as $item) {
      $code = $item['code'] ?? '';
      $details = $item['details'] ?? [];

      if ($code === 'CONTRIB_INCOMPATIBLE') {
        $contrib_items = $details['items'] ?? [];
        $next_version = $details['next_drupal_version'] ?? NULL;
      }
      elseif ($code === 'CUSTOM_INCOMPATIBLE') {
        $custom_items = $details['items'] ?? [];
        $next_version = $details['next_drupal_version'] ?? NULL;
      }
      elseif ($code === 'ALL_COMPATIBLE') {
        $next_version = $details['next_drupal_version'] ?? NULL;
      }
    }

    // All compatible.
    if (empty($contrib_items) && empty($custom_items)) {
      $message = $next_version
        ? (string) $this->t('All installed modules and themes are compatible with Drupal @version.', ['@version' => $next_version])
        : (string) $this->t('All installed modules and themes are compatible with the next Drupal version.');
      return $this->ui->message($message, 'success');
    }

    $content = [];

    // Contrib section.
    if (!empty($contrib_items)) {
      $content['contrib_title'] = [
        '#markup' => '<h4>' . $this->t('Contributed Modules and Themes') . '</h4>',
      ];
      $content['contrib_info'] = $this->ui->message(
        (string) $this->t('The following @count contributed extension(s) are not yet compatible with Drupal @version. Check drupal.org for updates or compatibility patches.', [
          '@count' => count($contrib_items),
          '@version' => $next_version,
        ]),
        'info'
      );
      $content['contrib_table'] = $this->buildCompatibilityTable($contrib_items, (int) $next_version, TRUE);
    }

    // Custom section.
    if (!empty($custom_items)) {
      $content['custom_title'] = [
        '#markup' => '<h4>' . $this->t('Custom Modules and Themes') . '</h4>',
      ];
      $content['custom_info'] = $this->ui->message(
        (string) $this->t('The following @count custom extension(s) need to be updated for Drupal @version compatibility.', [
          '@count' => count($custom_items),
          '@version' => $next_version,
        ]),
        'info'
      );
      $content['custom_table'] = $this->buildCompatibilityTable($custom_items, (int) $next_version, FALSE);
    }

    return $content;
  }

  /**
   * Builds the compatibility table.
   *
   * @param array $items
   *   Array of incompatible extension items.
   * @param int $next_version
   *   The next Drupal major version.
   * @param bool $show_link
   *   Whether to show drupal.org links (for contrib).
   *
   * @return array
   *   Render array for the table.
   */
  protected function buildCompatibilityTable(array $items, int $next_version, bool $show_link = TRUE): array {
    $headers = [
      $this->ui->header((string) $this->t('Extension')),
      $this->ui->header((string) $this->t('Type'), 'left', '80px'),
      $this->ui->header((string) $this->t('Version'), 'left', '120px'),
      $this->ui->header((string) $this->t('Core Requirement'), 'left', '200px'),
    ];

    $rows = [];

    foreach ($items as $item) {
      // Build name with link for contrib.
      $name = (string) ($item['label'] ?? $item['name'] ?? $this->t('Unknown'));
      $machine_name = (string) ($item['name'] ?? '');
      $project_link = $item['project_link'] ?? NULL;

      if ($show_link && $project_link) {
        $name_cell = '<a href="' . $project_link . '" target="_blank">' . $this->ui->itemName($name, $machine_name) . '</a>';
      }
      else {
        $name_cell = $this->ui->itemName($name, $machine_name);
      }

      // Type badge.
      $extension_type = $item['extension_type'] ?? 'module';
      $type_label = $extension_type === 'theme'
        ? (string) $this->t('Theme')
        : (string) $this->t('Module');

      $current_version = (string) ($item['current_version'] ?? $this->t('Unknown'));
      $core_requirement = (string) ($item['core_version_requirement'] ?? $this->t('Not specified'));

      $rows[] = $this->ui->row([
        $this->ui->cell($name_cell),
        $this->ui->cell($this->ui->badge($type_label, 'neutral')),
        $this->ui->cell($current_version, ['nowrap' => TRUE]),
        $this->ui->cell($core_requirement, ['nowrap' => TRUE]),
      ]);
    }

    return $this->ui->table($headers, $rows, [
      'empty' => (string) $this->t('No incompatible extensions found.'),
    ]);
  }

  /**
   * Builds an updates table.
   */
  protected function buildUpdatesTable(array $items, bool $is_security, bool $is_notice = FALSE): array {
    $headers = [
      $this->ui->header((string) $this->t('Project')),
      $this->ui->header((string) $this->t('Type'), 'left', '100px'),
      $this->ui->header((string) $this->t('Current'), 'left', '120px'),
      $this->ui->header((string) $this->t('Recommended'), 'left', '150px'),
    ];

    $rows = [];

    foreach ($items as $item) {
      $details = $item['details'] ?? [];

      $project_type = $details['project_type'] ?? 'module';
      $type_label = $project_type === 'theme'
        ? (string) $this->t('Theme')
        : (string) $this->t('Module');

      // Build name with link using itemName helper.
      $name = (string) ($details['label'] ?? $details['name'] ?? $this->t('Unknown'));
      $machine_name = (string) ($details['name'] ?? '');
      if (!empty($details['link'])) {
        $name_cell = '<a href="' . $details['link'] . '" target="_blank">' . $this->ui->itemName($name, $machine_name) . '</a>';
      }
      else {
        $name_cell = $this->ui->itemName($name, $machine_name);
      }

      // Versions.
      $current = $details['current_version'] ?? (string) $this->t('Unknown');
      $recommended = $details['recommended_version'] ?? (string) $this->t('Unknown');
      if (!empty($details['release_link'])) {
        $recommended = '<a href="' . $details['release_link'] . '" target="_blank">' . $recommended . '</a>';
      }

      // Determine row severity.
      if ($is_security) {
        $row_severity = 'error';
      }
      elseif ($is_notice) {
        $row_severity = 'notice';
      }
      else {
        $row_severity = 'warning';
      }

      $rows[] = $this->ui->row([
        $this->ui->cell($name_cell),
        $this->ui->cell($this->ui->badge($type_label, 'neutral')),
        $this->ui->cell($current, ['nowrap' => TRUE]),
        $this->ui->cell($recommended, ['nowrap' => TRUE]),
      ], $row_severity);
    }

    return $this->ui->table($headers, $rows, [
      'empty' => (string) $this->t('No updates found.'),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function checkRequirements(): array {
    $warnings = [];

    // Check if update module is enabled (should always be true due to dependency).
    if (!$this->moduleHandler->moduleExists('update')) {
      $warnings[] = (string) $this->t('The Update module is required but not enabled.');
    }

    return $warnings;
  }

}
