<?php

declare(strict_types=1);

namespace Drupal\audit_views\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Analyzes Views configuration.
 */
#[AuditAnalyzer(
  id: 'views',
  label: new TranslatableMarkup('Views Audit'),
  description: new TranslatableMarkup('Analyzes Views displays, cache configuration, relationships, and Search API usage.'),
  menu_title: new TranslatableMarkup('Views'),
  output_directory: 'views',
  weight: 3,
)]
class ViewsAnalyzer extends AuditAnalyzerBase {

  /**
   * Default threshold for relationship warnings.
   */
  protected const DEFAULT_RELATIONSHIPS_THRESHOLD = 3;

  /**
   * Minimum recommended cache time in seconds (5 minutes).
   */
  protected const MIN_RECOMMENDED_CACHE_TIME = 300;

  /**
   * Admin view paths and patterns.
   */
  protected const ADMIN_PATH_PATTERNS = [
    '/admin/',
    '/admin',
  ];

  /**
   * Admin view base tables.
   */
  protected const ADMIN_BASE_TABLES = [
    'users_field_data',
    'node_field_revision',
    'watchdog',
    'file_managed',
    'taxonomy_term_field_data',
  ];

  /**
   * Views excluded from cache audit.
   *
   * These views intentionally have no cache configured by design.
   * For example, the 'watchdog' view from dblog module shows real-time
   * log entries and should not be cached.
   */
  protected const VIEWS_EXCLUDED_FROM_CACHE_AUDIT = [
    'watchdog',
  ];

  /**
   * Cache type labels.
   */
  protected const CACHE_TYPE_LABELS = [
    'none' => 'No cache',
    'tag' => 'Tag based',
    'time' => 'Time based',
    'search_api_none' => 'Search API (none)',
    'search_api_tag' => 'Search API (tag)',
    'search_api_time' => 'Search API (time)',
  ];

  /**
   * Display type labels.
   */
  protected const DISPLAY_TYPE_LABELS = [
    'default' => 'Master',
    'page' => 'Page',
    'block' => 'Block',
    'feed' => 'Feed',
    'attachment' => 'Attachment',
    'embed' => 'Embed',
    'entity_reference' => 'Entity Reference',
    'rest_export' => 'REST Export',
    'data_export' => 'Data Export',
    'eva' => 'EVA',
  ];

  /**
   * Style plugin labels (format).
   */
  protected const STYLE_PLUGIN_LABELS = [
    'default' => 'Unformatted list',
    'table' => 'Table',
    'grid' => 'Grid',
    'html_list' => 'HTML List',
    'list' => 'List',
    'rss' => 'RSS Feed',
    'opml' => 'OPML',
    'serializer' => 'Serializer',
    'data_export' => 'Data Export',
  ];

  /**
   * Base table to entity type mapping.
   */
  protected const BASE_TABLE_ENTITY_MAP = [
    'node_field_data' => 'node',
    'node' => 'node',
    'users_field_data' => 'user',
    'users' => 'user',
    'taxonomy_term_field_data' => 'taxonomy_term',
    'taxonomy_term' => 'taxonomy_term',
    'media_field_data' => 'media',
    'media' => 'media',
    'file_managed' => 'file',
    'comment_field_data' => 'comment',
    'comment' => 'comment',
    'block_content_field_data' => 'block_content',
    'block_content' => 'block_content',
    'paragraphs_item_field_data' => 'paragraph',
    'paragraphs_item' => 'paragraph',
    'commerce_product_field_data' => 'commerce_product',
    'commerce_product' => 'commerce_product',
    'commerce_product_variation_field_data' => 'commerce_product_variation',
    'commerce_order' => 'commerce_order',
    'webform_submission' => 'webform_submission',
  ];

  /**
   * Entity type labels.
   */
  protected const ENTITY_TYPE_LABELS = [
    'node' => 'Content',
    'user' => 'User',
    'taxonomy_term' => 'Taxonomy',
    'media' => 'Media',
    'file' => 'File',
    'comment' => 'Comment',
    'block_content' => 'Block',
    'paragraph' => 'Paragraph',
    'commerce_product' => 'Product',
    'commerce_product_variation' => 'Product Variation',
    'commerce_order' => 'Order',
    'webform_submission' => 'Webform Submission',
  ];

  /**
   * Bundle filter field names by entity type.
   */
  protected const BUNDLE_FILTER_FIELDS = [
    'node' => ['type', 'node_type'],
    'taxonomy_term' => ['vid', 'vocabulary'],
    'media' => ['bundle', 'media_type'],
    'comment' => ['comment_type'],
    'block_content' => ['type', 'block_content_type'],
    'paragraph' => ['type', 'paragraphs_type'],
    'commerce_product' => ['type', 'commerce_product_type'],
  ];

  protected EntityTypeManagerInterface $entityTypeManager;
  protected ConfigFactoryInterface $configFactory;
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->configFactory = $container->get('config.factory');
    $instance->moduleHandler = $container->get('module_handler');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $config): array {
    $form = [];

    $form['relationships_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Relationships warning threshold'),
      '#description' => $this->t('Number of relationships in a view display above which a performance warning is shown. Views with many relationships can be slow.'),
      '#default_value' => $config['relationships_threshold'] ?? self::DEFAULT_RELATIONSHIPS_THRESHOLD,
      '#min' => 1,
      '#max' => 20,
    ];

    if ($this->moduleHandler->moduleExists('search_api')) {
      $form['ignore_searchapi_mysql_warnings'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Ignore Search API MySQL rendering warnings'),
        '#description' => $this->t('When checked, Search API views rendering fields from MySQL will be reported as notices instead of warnings and will not affect the score.'),
        '#default_value' => $config['ignore_searchapi_mysql_warnings'] ?? FALSE,
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    $views_config = $this->configFactory->get('audit_views.settings');

    $relationships_threshold = (int) ($views_config->get('relationships_threshold') ?? self::DEFAULT_RELATIONSHIPS_THRESHOLD);
    $ignore_searchapi_warnings = (bool) ($views_config->get('ignore_searchapi_mysql_warnings') ?? FALSE);

    // Load views once and reuse for all analysis.
    $views_storage = $this->entityTypeManager->getStorage('view');
    $views = $views_storage->loadMultiple();

    $displays_results = $this->analyzeDisplays($views, $relationships_threshold);

    // Build issue files for faceted lists (only warnings).
    $cache_issues = $this->buildCacheIssuesFile($displays_results);
    $relationship_issues = $this->buildRelationshipIssuesFile($displays_results, $relationships_threshold);
    $cache_tag_issues = $this->buildCacheTagIssuesFile($displays_results);

    $files = [
      'displays' => $displays_results,
      'cache_issues' => $cache_issues,
      'relationship_issues' => $relationship_issues,
      'cache_tag_issues' => $cache_tag_issues,
    ];

    $searchapi_results = ['summary' => ['warnings' => 0, 'notices' => 0], 'results' => []];
    $searchapi_cache_results = ['summary' => ['warnings' => 0, 'notices' => 0], 'results' => []];
    $has_searchapi = $this->moduleHandler->moduleExists('search_api');
    if ($has_searchapi) {
      $searchapi_results = $this->analyzeSearchApi($views, $ignore_searchapi_warnings);
      $files['searchapi'] = $searchapi_results;

      // Analyze Search API views cache (excluding those with facets).
      $searchapi_cache_results = $this->analyzeSearchApiCache($views);
      $files['searchapi_cache'] = $searchapi_cache_results;

      // Build Search API cache issues file (only warnings - no cache).
      $searchapi_cache_issues = $this->buildSearchApiCacheIssuesFile($searchapi_cache_results);
      $files['searchapi_cache_issues'] = $searchapi_cache_issues;
    }

    $scores = $this->calculateScores($displays_results, $searchapi_results, $searchapi_cache_results, $relationships_threshold);

    return [
      '_files' => $files,
      'score' => $scores,
      'config' => [
        'relationships_threshold' => $relationships_threshold,
        'ignore_searchapi_mysql_warnings' => $ignore_searchapi_warnings,
      ],
      'has_searchapi' => $has_searchapi,
    ];
  }

  /**
   * Builds the cache issues file (warnings only).
   */
  protected function buildCacheIssuesFile(array $displays_results): array {
    $results = [];
    $warnings = 0;

    foreach ($displays_results['results'] ?? [] as $item) {
      $code = $item['code'] ?? '';
      $severity = $item['severity'] ?? 'info';

      // Only include cache-related warnings.
      if ($severity === 'warning' && in_array($code, ['NO_CACHE', 'LOW_CACHE_TIME'], TRUE)) {
        $results[] = $item;
        $warnings++;
      }
    }

    return $this->createResult($results, 0, $warnings, 0);
  }

  /**
   * Builds the relationship issues file (warnings only).
   */
  protected function buildRelationshipIssuesFile(array $displays_results, int $threshold): array {
    $results = [];
    $warnings = 0;

    foreach ($displays_results['results'] ?? [] as $item) {
      $code = $item['code'] ?? '';
      $severity = $item['severity'] ?? 'info';

      // Only include relationship warnings.
      if ($severity === 'warning' && $code === 'EXCESSIVE_RELATIONSHIPS') {
        $results[] = $item;
        $warnings++;
      }
    }

    return $this->createResult($results, 0, $warnings, 0);
  }

  /**
   * Builds the Search API cache issues file (warnings only - no cache).
   */
  protected function buildSearchApiCacheIssuesFile(array $searchapi_cache_results): array {
    $results = [];
    $warnings = 0;

    foreach ($searchapi_cache_results['results'] ?? [] as $item) {
      $severity = $item['severity'] ?? 'info';

      // Only include warnings (no cache configured).
      if ($severity === 'warning') {
        $results[] = $item;
        $warnings++;
      }
    }

    return $this->createResult($results, 0, $warnings, 0);
  }

  /**
   * Builds the cache tag issues file (generic tags).
   */
  protected function buildCacheTagIssuesFile(array $displays_results): array {
    $results = [];
    $warnings = 0;

    foreach ($displays_results['results'] ?? [] as $item) {
      $code = $item['code'] ?? '';

      // Only include generic cache tag warnings.
      if ($code === 'CACHE_TAG_GENERIC') {
        $results[] = $item;
        $warnings++;
      }
    }

    return $this->createResult($results, 0, $warnings, 0);
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    $checks = [
      // Scoring sections - faceted lists.
      'cache_issues' => [
        'label' => $this->t('Cache Issues'),
        'description' => $this->t('View displays with no cache or very low cache time configured.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'cache_issues',
        'score_factor_key' => 'cache_config',
        'weight' => 40,
      ],
      'relationship_issues' => [
        'label' => $this->t('Relationship Issues'),
        'description' => $this->t('View displays with excessive relationships that may cause performance problems.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'relationship_issues',
        'score_factor_key' => 'relationships',
        'weight' => 15,
      ],
      'cache_tag_issues' => [
        'label' => $this->t('Generic Cache Tags'),
        'description' => $this->t('Views using tag-based cache with generic tags (node_list, taxonomy_term_list) that invalidate too frequently.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'cache_tag_issues',
        'score_factor_key' => 'cache_tags',
        'weight' => 15,
      ],
      // Informational sections - tables.
      'displays_status' => [
        'label' => $this->t('View Displays Overview'),
        'description' => $this->t('Complete list of all view displays with their cache configuration and render modes.'),
        'file_types' => ['config'],
        'affects_score' => FALSE,
        'file_key' => 'displays',
      ],
    ];

    // Add Search API checks only if module exists.
    if ($this->moduleHandler->moduleExists('search_api')) {
      // Scoring sections for Search API.
      $checks['searchapi_issues'] = [
        'label' => $this->t('Search API Rendering Issues'),
        'description' => $this->t('Search API views rendering fields from database instead of indexed data.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'searchapi',
        'score_factor_key' => 'searchapi_rendering',
        'weight' => 20,
      ];
      $checks['searchapi_cache_issues'] = [
        'label' => $this->t('Search API Cache Issues'),
        'description' => $this->t('Search API views without facets that have no cache configured.'),
        'file_types' => ['config'],
        'affects_score' => TRUE,
        'file_key' => 'searchapi_cache_issues',
        'score_factor_key' => 'searchapi_cache',
        'weight' => 10,
      ];
      // Informational section for Search API.
      $checks['searchapi_cache_status'] = [
        'label' => $this->t('Search API Cache Overview'),
        'description' => $this->t('Complete list of Search API views without facets and their cache configuration.'),
        'file_types' => ['config'],
        'affects_score' => FALSE,
        'file_key' => 'searchapi_cache',
      ];
    }

    return $checks;
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    $files = $data['_files'] ?? [];
    $config = $data['config'] ?? [];

    return match ($check_id) {
      // Scoring sections (faceted lists).
      'cache_issues' => $this->getCacheIssuesContent($files),
      'relationship_issues' => $this->getRelationshipIssuesContent($files, $config),
      'cache_tag_issues' => $this->getCacheTagIssuesContent($files),
      'searchapi_issues' => $this->getSearchApiIssuesContent($files),
      'searchapi_cache_issues' => $this->getSearchApiCacheIssuesContent($files),
      // Informational sections (tables).
      'displays_status' => $this->buildDisplaysContent($files, $config),
      'searchapi_cache_status' => $this->buildSearchApiCacheContent($files),
      default => [],
    };
  }

  /**
   * Gets the cache issues content (faceted list).
   */
  protected function getCacheIssuesContent(array $files): array {
    $results = $files['cache_issues']['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('All view displays have proper cache configured.'),
          'success'
        ),
      ];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $code = $item['code'] ?? '';

      // file: visible in summary - View: Display.
      $view_label = $details['view_label'] ?? '';
      $display_title = $details['display_title'] ?: ($details['display_id'] ?? '');
      $file = $view_label . ': ' . $display_title;

      // label and description based on issue type.
      if ($code === 'NO_CACHE') {
        $label = (string) $this->t('No Cache Configured');
        $description = (string) $this->t('This view display has no cache configured. Every page load will execute the full query against the database, causing unnecessary load. Enable tag-based or time-based caching for better performance.');
      }
      else {
        $label = (string) $this->t('Low Cache Time');
        $description = (string) $this->t('This view display has a very low cache time configured. Consider increasing the cache time to at least 5 minutes for better performance.');
      }

      $issues[] = $this->ui->issue([
        'severity' => $item['severity'] ?? 'warning',
        'code' => $code,
        'file' => $file,
        'label' => $label,
        'description' => $description,
        'tags' => [$details['display_plugin'] ?? 'view', $details['is_admin'] ? 'admin' : 'frontend'],
      ]);
    }

    return [
      'list' => $this->ui->issueList($issues),
    ];
  }

  /**
   * Gets the relationship issues content (faceted list).
   */
  protected function getRelationshipIssuesContent(array $files, array $config): array {
    $results = $files['relationship_issues']['results'] ?? [];
    $threshold = $config['relationships_threshold'] ?? self::DEFAULT_RELATIONSHIPS_THRESHOLD;

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('All view displays have an optimal number of relationships (≤@threshold).', [
            '@threshold' => $threshold,
          ]),
          'success'
        ),
      ];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];

      // file: visible in summary - View: Display.
      $view_label = $details['view_label'] ?? '';
      $display_title = $details['display_title'] ?: ($details['display_id'] ?? '');
      $file = $view_label . ': ' . $display_title;

      $rel_count = $details['relationships_count'] ?? 0;

      $issues[] = $this->ui->issue([
        'severity' => $item['severity'] ?? 'warning',
        'code' => 'EXCESSIVE_RELATIONSHIPS',
        'file' => $file,
        'label' => (string) $this->t('Excessive Relationships (@count)', ['@count' => $rel_count]),
        'description' => (string) $this->t('This view display has @count relationships, which exceeds the recommended threshold of @threshold. Each relationship adds JOIN operations to the query, significantly impacting performance. Consider reducing relationships or using a different approach.', [
          '@count' => $rel_count,
          '@threshold' => $threshold,
        ]),
        'tags' => [$details['display_plugin'] ?? 'view', $details['is_admin'] ? 'admin' : 'frontend'],
      ]);
    }

    return [
      'list' => $this->ui->issueList($issues),
    ];
  }

  /**
   * Gets the cache tag issues content (faceted list).
   */
  protected function getCacheTagIssuesContent(array $files): array {
    $results = $files['cache_tag_issues']['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('All views with tag-based cache use specific cache tags or do not filter by bundle.'),
          'success'
        ),
      ];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];

      // file: visible in summary - View: Display.
      $view_label = $details['view_label'] ?? '';
      $display_title = $details['display_title'] ?: ($details['display_id'] ?? '');
      $file = $view_label . ': ' . $display_title;

      // Build suggested tags from entity bundles.
      $entity_type = $details['entity_type'] ?? 'entity';
      $bundles = $details['entity_bundles'] ?? [];
      $suggested_tags = [];
      foreach ($bundles as $bundle) {
        $clean_bundle = preg_replace('/\s*\(contextual\)$/', '', $bundle);
        $suggested_tags[] = $entity_type . '_list:' . $clean_bundle;
      }

      $description = (string) $this->t('This view uses tag-based cache with generic entity list tags (@entity_list). Any time a @entity is created, edited or deleted, this view cache is invalidated. If this view is a block used on multiple pages, all those pages will also be invalidated.', [
        '@entity_list' => $entity_type . '_list',
        '@entity' => $details['entity_label'] ?? $entity_type,
      ]);

      $description .= '<br><br><strong>' . (string) $this->t('Recommendations:') . '</strong><ul>';
      $description .= '<li>' . (string) $this->t('Configure bundle-specific cache tags: @tags', [
        '@tags' => implode(', ', $suggested_tags),
      ]) . '</li>';
      $description .= '<li>' . (string) $this->t('Install "Views Custom Cache Tags" or "Cache Control Override" module') . '</li>';
      $description .= '<li>' . (string) $this->t('Configure BigPipe placeholder for view blocks to prevent page cache invalidation') . '</li>';
      $description .= '</ul>';

      $issues[] = $this->ui->issue([
        'severity' => $item['severity'] ?? 'warning',
        'code' => 'CACHE_TAG_GENERIC',
        'file' => $file,
        'label' => (string) $this->t('Generic Cache Tags'),
        'description' => $description,
        'tags' => [$details['display_plugin'] ?? 'view', $entity_type],
      ]);
    }

    return [
      'list' => $this->ui->issueList($issues),
    ];
  }

  /**
   * Gets the Search API issues content (faceted list).
   */
  protected function getSearchApiIssuesContent(array $files): array {
    $results = $files['searchapi']['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('All Search API views are properly optimized to use indexed data.'),
          'success'
        ),
      ];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];

      // file: visible in summary - View: Display - Field.
      $view_label = $details['view_label'] ?? '';
      $display_id = $details['display_id'] ?? '';
      $field_id = $details['field_id'] ?? '';
      $file = $view_label . ': ' . $display_id . ' - ' . $field_id;

      // Determine label based on reason.
      $reason = $details['reason'] ?? '';
      if (str_contains($reason, 'link_to_item')) {
        $label = (string) $this->t('Link to Item Enabled');
        $description = (string) $this->t('This field has "Link to item" enabled, which requires loading the entity from the database to generate the URL. Disable this option and use a dedicated indexed link field instead.');
      }
      elseif (str_contains($reason, 'link_to_entity')) {
        $label = (string) $this->t('Link to Entity Enabled');
        $description = (string) $this->t('This field has "Link to entity" enabled, which requires loading the entity from the database. Disable this option and use indexed URL data instead.');
      }
      elseif (str_contains($reason, 'field_rendering')) {
        $label = (string) $this->t('Field Rendering from Database');
        $description = (string) $this->t('This field uses entity field rendering, which loads data from the database instead of the search index. Disable this option to use indexed values.');
      }
      else {
        $label = (string) $this->t('Non-Search API Plugin');
        $description = (string) $this->t('This field uses a standard Drupal Views plugin instead of a Search API plugin, causing entity loads from the database. Replace it with a Search API field handler.');
      }

      $issues[] = $this->ui->issue([
        'severity' => $item['severity'] ?? 'warning',
        'code' => 'SEARCHAPI_MYSQL_RENDER',
        'file' => $file,
        'label' => $label,
        'description' => $description,
        'tags' => ['search_api', $details['plugin_id'] ?? 'unknown'],
      ]);
    }

    return [
      'list' => $this->ui->issueList($issues),
    ];
  }

  /**
   * Gets the Search API cache issues content (faceted list).
   */
  protected function getSearchApiCacheIssuesContent(array $files): array {
    $results = $files['searchapi_cache_issues']['results'] ?? [];

    if (empty($results)) {
      return [
        'message' => $this->ui->message(
          (string) $this->t('All Search API views without facets have proper cache configured.'),
          'success'
        ),
      ];
    }

    $issues = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];

      // file: visible in summary - View: Display.
      $view_label = $details['view_label'] ?? '';
      $display_title = $details['display_title'] ?: ($details['display_id'] ?? '');
      $file = $view_label . ': ' . $display_title;

      $issues[] = $this->ui->issue([
        'severity' => $item['severity'] ?? 'warning',
        'code' => 'SEARCHAPI_NO_CACHE',
        'file' => $file,
        'label' => (string) $this->t('No Cache Configured'),
        'description' => (string) $this->t('This Search API view has no facets and no cache configured. Since facets are not used, caching can safely be enabled to improve performance. Enable time-based or tag-based caching.'),
        'tags' => ['search_api', $details['display_plugin'] ?? 'view'],
      ]);
    }

    return [
      'list' => $this->ui->issueList($issues),
    ];
  }

  /**
   * Builds content for the displays check.
   */
  protected function buildDisplaysContent(array $files, array $config): array {
    $relationships_threshold = $config['relationships_threshold'] ?? self::DEFAULT_RELATIONSHIPS_THRESHOLD;

    // Categorize displays in a single pass for better performance.
    $categories = [
      'enabled_admin' => [],
      'enabled_frontend' => [],
      'disabled_admin' => [],
      'disabled_frontend' => [],
    ];

    foreach ($files['displays']['results'] ?? [] as $item) {
      if (($item['code'] ?? '') !== 'VIEW_DISPLAY') {
        continue;
      }

      $details = $item['details'] ?? [];
      $enabled = $details['enabled'] ?? TRUE;
      $is_admin = $details['is_admin'] ?? FALSE;

      if ($enabled) {
        $key = $is_admin ? 'enabled_admin' : 'enabled_frontend';
      }
      else {
        $key = $is_admin ? 'disabled_admin' : 'disabled_frontend';
      }

      $categories[$key][] = $item;
    }

    // Combine for section totals.
    $enabled_displays = array_merge($categories['enabled_admin'], $categories['enabled_frontend']);
    $disabled_displays = array_merge($categories['disabled_admin'], $categories['disabled_frontend']);

    $content = [];

    // Enabled displays section.
    $enabled_stats = $this->calculateGroupStats($enabled_displays, $relationships_threshold);
    $enabled_content = [];

    if (!empty($enabled_displays)) {
      // Admin subgroup.
      $admin_stats = $this->calculateGroupStats($categories['enabled_admin'], $relationships_threshold);
      $admin_content = !empty($categories['enabled_admin'])
        ? $this->buildDisplaysTable($categories['enabled_admin'], $relationships_threshold)
        : $this->ui->message((string) $this->t('No administration views found.'), 'info');

      $enabled_content['admin'] = $this->ui->section(
        $this->buildGroupTitle((string) $this->t('Administration Views'), $admin_stats),
        $admin_content,
        [
          'open' => TRUE,
          'severity' => $this->getSeverityFromStats($admin_stats),
        ]
      );

      // Non-administration subgroup.
      $non_admin_stats = $this->calculateGroupStats($categories['enabled_frontend'], $relationships_threshold);
      $non_admin_content = !empty($categories['enabled_frontend'])
        ? $this->buildDisplaysTable($categories['enabled_frontend'], $relationships_threshold)
        : $this->ui->message((string) $this->t('No non-administration views found.'), 'info');

      $enabled_content['non_admin'] = $this->ui->section(
        $this->buildGroupTitle((string) $this->t('Non-administration Views'), $non_admin_stats),
        $non_admin_content,
        [
          'open' => TRUE,
          'severity' => $this->getSeverityFromStats($non_admin_stats),
        ]
      );
    }
    else {
      $enabled_content['message'] = $this->ui->message((string) $this->t('No enabled view displays found.'), 'info');
    }

    $content['enabled_displays'] = $this->ui->section(
      $this->buildGroupTitle((string) $this->t('Enabled Views'), $enabled_stats),
      $enabled_content,
      [
        'open' => TRUE,
        'severity' => $this->getSeverityFromStats($enabled_stats),
      ]
    );

    // Disabled displays section.
    $disabled_stats = $this->calculateGroupStats($disabled_displays, $relationships_threshold);
    $disabled_content = [];

    if (!empty($disabled_displays)) {
      // Admin subgroup.
      $admin_stats = $this->calculateGroupStats($categories['disabled_admin'], $relationships_threshold);
      $admin_content = !empty($categories['disabled_admin'])
        ? $this->buildDisplaysTable($categories['disabled_admin'], $relationships_threshold)
        : $this->ui->message((string) $this->t('No administration views found.'), 'info');

      $disabled_content['admin'] = $this->ui->section(
        $this->buildGroupTitle((string) $this->t('Administration Views'), $admin_stats),
        $admin_content,
        [
          'open' => TRUE,
          'severity' => $this->getSeverityFromStats($admin_stats),
        ]
      );

      // Non-administration subgroup.
      $non_admin_stats = $this->calculateGroupStats($categories['disabled_frontend'], $relationships_threshold);
      $non_admin_content = !empty($categories['disabled_frontend'])
        ? $this->buildDisplaysTable($categories['disabled_frontend'], $relationships_threshold)
        : $this->ui->message((string) $this->t('No non-administration views found.'), 'info');

      $disabled_content['non_admin'] = $this->ui->section(
        $this->buildGroupTitle((string) $this->t('Non-administration Views'), $non_admin_stats),
        $non_admin_content,
        [
          'open' => TRUE,
          'severity' => $this->getSeverityFromStats($non_admin_stats),
        ]
      );
    }
    else {
      $disabled_content['message'] = $this->ui->message((string) $this->t('No disabled view displays found.'), 'info');
    }

    $content['disabled_displays'] = $this->ui->section(
      $this->buildGroupTitle((string) $this->t('Disabled Views'), $disabled_stats),
      $disabled_content,
      [
        'open' => FALSE,
        'severity' => $this->getSeverityFromStats($disabled_stats),
      ]
    );

    return $content;
  }

  /**
   * Builds content for the Search API cache check.
   */
  protected function buildSearchApiCacheContent(array $files): array {
    $results = $files['searchapi_cache']['results'] ?? [];

    if (empty($results)) {
      return ['message' => $this->ui->message(
        (string) $this->t('No Search API views without facets found.'),
        'info'
      )];
    }

    return $this->buildSearchApiCacheSection($files['searchapi_cache']);
  }

  /**
   * Analyzes all view displays.
   *
   * @param array $views
   *   Array of view entities.
   * @param int $relationships_threshold
   *   The threshold for relationship warnings.
   *
   * @return array
   *   The analysis results.
   */
  protected function analyzeDisplays(array $views, int $relationships_threshold): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    foreach ($views as $view) {
      $view_id = $view->id();

      // Skip views excluded from cache audit entirely (e.g., watchdog).
      // These views intentionally have no cache by design.
      if (in_array($view_id, self::VIEWS_EXCLUDED_FROM_CACHE_AUDIT, TRUE)) {
        continue;
      }

      $view_label = $view->label() ?: $view_id;
      $base_table = $view->get('base_table');
      $displays = $view->get('display');

      // Get the default display for inherited settings.
      $default_display = $displays['default']['display_options'] ?? [];

      foreach ($displays as $display_id => $display) {
        $display_options = $display['display_options'] ?? [];
        $display_plugin = $display['display_plugin'] ?? 'unknown';

        // For master display, use its own options directly.
        // For other displays, merge with default display options for inherited values.
        if ($display_plugin === 'default') {
          $merged_options = $display_options;
        }
        else {
          $merged_options = array_merge($default_display, $display_options);
        }

        // Get display title.
        $display_title = $display_options['title'] ?? $default_display['title'] ?? '';

        // Determine if enabled (default is TRUE).
        $enabled = $merged_options['enabled'] ?? TRUE;

        // Get cache configuration.
        $cache_config = $merged_options['cache'] ?? [];
        $cache_type = $cache_config['type'] ?? 'none';

        // Get detailed cache info.
        $cache_details = $this->getCacheDetails($cache_config);

        // Get relationships count.
        $relationships = $merged_options['relationships'] ?? [];
        $relationships_count = count($relationships);

        // Determine if this is an admin view.
        $is_admin = $this->isAdminView($view_id, $base_table, $merged_options);

        // Get display type label.
        $display_type_label = self::DISPLAY_TYPE_LABELS[$display_plugin] ?? ucfirst($display_plugin);

        // Detect entity type and bundles.
        $entity_bundle_info = $this->detectEntityAndBundles($base_table, $merged_options);

        // Determine render mode (fields vs rendered entity).
        $render_mode_info = $this->detectRenderMode($merged_options);

        // Common details for all result items from this display.
        $common_details = [
          'view_id' => $view_id,
          'view_label' => $view_label,
          'display_id' => $display_id,
          'display_title' => $display_title,
          'display_plugin' => $display_plugin,
          'display_type_label' => $display_type_label,
          'enabled' => $enabled,
          'is_admin' => $is_admin,
          'cache_type' => $cache_type,
          'cache_label' => $cache_details['label'],
          'cache_details' => $cache_details['details'],
          'relationships_count' => $relationships_count,
          'relationships' => array_keys($relationships),
          'path' => $merged_options['path'] ?? '',
          'render_mode' => $render_mode_info['mode'],
          'render_mode_label' => $render_mode_info['label'],
          'is_fields_render' => $render_mode_info['is_fields'],
          'is_table_format' => $render_mode_info['is_table_format'],
          'row_plugin' => $render_mode_info['row_plugin'],
          'style_plugin' => $render_mode_info['style_plugin'],
          'fields_count' => $render_mode_info['fields_count'],
          'entity_type' => $entity_bundle_info['entity_type'],
          'entity_label' => $entity_bundle_info['entity_label'],
          'entity_bundles' => $entity_bundle_info['bundles'],
          'entity_bundle_label' => $entity_bundle_info['display_label'],
          'base_table' => $base_table,
          'cache_tag_optimizable' => $cache_type === 'tag' && !empty($entity_bundle_info['bundles']),
        ];

        // Store display info for table rendering (no issue).
        $results[] = $this->createResultItem(
          'info',
          'VIEW_DISPLAY',
          (string) $this->t('View @view: @display', [
            '@view' => $view_label,
            '@display' => $display_title ?: $display_id,
          ]),
          $common_details
        );

        // Generate specific issue items.
        if ($enabled) {
          // Warning: No cache configured.
          if ($cache_type === 'none') {
            $warnings++;
            $results[] = $this->createResultItem(
              'warning',
              'NO_CACHE',
              (string) $this->t('@view (@display): No cache configured. Enable tag-based or time-based cache to improve performance.', [
                '@view' => $view_label,
                '@display' => $display_title ?: $display_id,
              ]),
              $common_details
            );
          }

          // Warning: Time-based cache with very low cache time.
          if ($cache_type === 'time') {
            $cache_options = $cache_config['options'] ?? [];
            $results_lifespan = (int) ($cache_options['results_lifespan'] ?? 0);
            $output_lifespan = (int) ($cache_options['output_lifespan'] ?? 0);

            // Check if both cache times are below minimum (and not -1 which means permanent).
            $results_too_low = $results_lifespan > 0 && $results_lifespan < self::MIN_RECOMMENDED_CACHE_TIME;
            $output_too_low = $output_lifespan > 0 && $output_lifespan < self::MIN_RECOMMENDED_CACHE_TIME;

            if ($results_too_low || $output_too_low) {
              $warnings++;
              $low_parts = [];
              if ($results_too_low) {
                $low_parts[] = (string) $this->t('results: @time', ['@time' => $this->formatCacheTime($results_lifespan)]);
              }
              if ($output_too_low) {
                $low_parts[] = (string) $this->t('output: @time', ['@time' => $this->formatCacheTime($output_lifespan)]);
              }
              $results[] = $this->createResultItem(
                'warning',
                'LOW_CACHE_TIME',
                (string) $this->t('@view (@display): Cache time is very low (@times). Consider increasing to at least @recommended for better performance.', [
                  '@view' => $view_label,
                  '@display' => $display_title ?: $display_id,
                  '@times' => implode(', ', $low_parts),
                  '@recommended' => $this->formatCacheTime(self::MIN_RECOMMENDED_CACHE_TIME),
                ]),
                array_merge($common_details, [
                  'results_lifespan' => $results_lifespan,
                  'output_lifespan' => $output_lifespan,
                  'min_recommended' => self::MIN_RECOMMENDED_CACHE_TIME,
                ])
              );
            }
          }

          // Warning: Too many relationships.
          if ($relationships_count > $relationships_threshold) {
            $warnings++;
            $results[] = $this->createResultItem(
              'warning',
              'EXCESSIVE_RELATIONSHIPS',
              (string) $this->t('@view (@display): Too many relationships (@count). Reduce relationships or consider using a custom query.', [
                '@view' => $view_label,
                '@display' => $display_title ?: $display_id,
                '@count' => $relationships_count,
              ]),
              $common_details
            );
          }

          // Notice: Fields render mode on non-table formats.
          // Skip this notice for Search API views - using fields is the expected
          // and recommended approach for Search API views.
          $is_search_api_view = str_starts_with($base_table, 'search_api_index_');
          if ($render_mode_info['is_fields'] && !$render_mode_info['is_table_format'] && !$is_search_api_view) {
            $notices++;
            $results[] = $this->createResultItem(
              'notice',
              'FIELDS_NON_TABLE',
              (string) $this->t('@view (@display): Using fields on non-table format. Consider using rendered entity with a view mode.', [
                '@view' => $view_label,
                '@display' => $display_title ?: $display_id,
              ]),
              $common_details
            );
          }

          // Warning: Generic cache tags cause excessive invalidation.
          if ($cache_type === 'tag' && !empty($entity_bundle_info['bundles'])) {
            $warnings++;
            $entity_type = $entity_bundle_info['entity_type'] ?? 'entity';
            $bundles = $entity_bundle_info['bundles'];
            $suggested_tags = [];
            foreach ($bundles as $bundle) {
              $clean_bundle = preg_replace('/\s*\(contextual\)$/', '', $bundle);
              $suggested_tags[] = $entity_type . '_list:' . $clean_bundle;
            }
            $results[] = $this->createResultItem(
              'warning',
              'CACHE_TAG_GENERIC',
              (string) $this->t('@view (@display): This view uses tag-based cache but relies on generic entity list tags, which invalidate the cache whenever any @entity_label is modified. For better performance, use bundle-specific cache tags (@tags) so the cache only invalidates when content of that specific type changes. Install modules like "Views Custom Cache Tags" or "Cache Control Override" to configure custom tags.', [
                '@view' => $view_label,
                '@display' => $display_title ?: $display_id,
                '@entity_label' => strtolower($entity_bundle_info['entity_label'] ?? $entity_type),
                '@tags' => implode(', ', $suggested_tags),
              ]),
              $common_details
            );
          }
        }
        else {
          // Notice: Disabled display.
          $notices++;
          $results[] = $this->createResultItem(
            'notice',
            'DISABLED_DISPLAY',
            (string) $this->t('@view (@display): Display is disabled. Review if needed or delete to keep the view clean.', [
              '@view' => $view_label,
              '@display' => $display_title ?: $display_id,
            ]),
            $common_details
          );
        }
      }
    }

    return $this->createResult($results, $errors, $warnings, $notices);
  }

  /**
   * Detects the base entity type and filtered bundles for a view.
   *
   * @param string $base_table
   *   The view's base table.
   * @param array $display_options
   *   The merged display options.
   *
   * @return array
   *   Array with entity_type, entity_label, bundles, and display_label.
   */
  protected function detectEntityAndBundles(string $base_table, array $display_options): array {
    // Detect entity type from base table.
    $entity_type = self::BASE_TABLE_ENTITY_MAP[$base_table] ?? NULL;
    $entity_label = $entity_type ? (self::ENTITY_TYPE_LABELS[$entity_type] ?? ucfirst($entity_type)) : $base_table;

    $bundles = [];

    if ($entity_type) {
      // Get filter field names for this entity type.
      $bundle_fields = self::BUNDLE_FILTER_FIELDS[$entity_type] ?? [];

      // Check regular filters.
      $filters = $display_options['filters'] ?? [];
      foreach ($filters as $filter_id => $filter_config) {
        $field = $filter_config['field'] ?? '';
        $plugin_id = $filter_config['plugin_id'] ?? '';

        // Check if this filter is for a bundle field.
        if (in_array($field, $bundle_fields, TRUE) || str_contains($plugin_id, 'bundle')) {
          $values = $filter_config['value'] ?? [];
          if (is_array($values)) {
            // Values can be [bundle_id => bundle_id] or [bundle_id => TRUE].
            foreach ($values as $key => $value) {
              if ($value && !is_array($value)) {
                $bundles[] = is_string($key) ? $key : (string) $value;
              }
            }
          }
          elseif (is_string($values) && $values !== '') {
            $bundles[] = $values;
          }
        }
      }

      // Check contextual filters (arguments).
      $arguments = $display_options['arguments'] ?? [];
      foreach ($arguments as $arg_id => $arg_config) {
        $field = $arg_config['field'] ?? '';
        $plugin_id = $arg_config['plugin_id'] ?? '';

        // Check if this contextual filter is for a bundle field.
        if (in_array($field, $bundle_fields, TRUE) || str_contains($plugin_id, 'bundle') || str_contains($plugin_id, 'type')) {
          // Check for default value.
          $default_action = $arg_config['default_action'] ?? '';
          if ($default_action === 'default') {
            $default = $arg_config['default_argument_options']['argument'] ?? NULL;
            if ($default && is_string($default)) {
              $bundles[] = $default . ' (contextual)';
            }
          }
          // Check for validation with specific bundles.
          $validate_type = $arg_config['validate']['type'] ?? '';
          if (str_contains($validate_type, 'bundle') || str_contains($validate_type, 'node_type')) {
            $validate_bundles = $arg_config['validate_options']['bundles'] ?? [];
            foreach ($validate_bundles as $bundle => $enabled) {
              if ($enabled) {
                $bundles[] = $bundle;
              }
            }
          }
        }
      }
    }

    // Remove duplicates.
    $bundles = array_unique($bundles);

    // Build display label.
    if (empty($bundles)) {
      $display_label = $entity_label;
    }
    else {
      $display_label = $entity_label . ': ' . implode(', ', $bundles);
    }

    return [
      'entity_type' => $entity_type,
      'entity_label' => $entity_label,
      'bundles' => $bundles,
      'display_label' => $display_label,
      'base_table' => $base_table,
    ];
  }

  /**
   * Gets detailed cache configuration information.
   *
   * @param array $cache_config
   *   The cache configuration from the view display.
   *
   * @return array
   *   Array with type, label, and details.
   */
  protected function getCacheDetails(array $cache_config): array {
    $type = $cache_config['type'] ?? 'none';
    $options = $cache_config['options'] ?? [];

    $label = self::CACHE_TYPE_LABELS[$type] ?? ucfirst($type);
    $details = '';

    switch ($type) {
      case 'time':
        $results_lifespan = $options['results_lifespan'] ?? 0;
        $output_lifespan = $options['output_lifespan'] ?? 0;

        if ($results_lifespan === -1) {
          $results_text = 'never';
        }
        elseif ($results_lifespan > 0) {
          $results_text = $this->formatCacheTime($results_lifespan);
        }
        else {
          $results_text = '0';
        }

        if ($output_lifespan === -1) {
          $output_text = 'never';
        }
        elseif ($output_lifespan > 0) {
          $output_text = $this->formatCacheTime($output_lifespan);
        }
        else {
          $output_text = '0';
        }

        $details = (string) $this->t('Results: @results, Output: @output', [
          '@results' => $results_text,
          '@output' => $output_text,
        ]);
        $label = (string) $this->t('Time based (@details)', ['@details' => $details]);
        break;

      case 'tag':
        // Tag-based cache uses entity cache tags.
        $label = (string) $this->t('Tag based (entity list)');
        $details = (string) $this->t('Invalidated when entities change');
        break;

      case 'search_api_time':
        $results_lifespan = $options['results_lifespan'] ?? 0;
        if ($results_lifespan > 0) {
          $details = $this->formatCacheTime($results_lifespan);
          $label = (string) $this->t('Search API time (@time)', ['@time' => $details]);
        }
        break;

      case 'search_api_tag':
      case 'search_api_none':
        $label = self::CACHE_TYPE_LABELS[$type] ?? $type;
        break;

      case 'none':
        $label = (string) $this->t('No cache');
        $details = (string) $this->t('Performance risk');
        break;
    }

    return [
      'type' => $type,
      'label' => $label,
      'details' => $details,
      'options' => $options,
    ];
  }

  /**
   * Formats cache time in seconds to a human-readable format.
   *
   * @param int $seconds
   *   Time in seconds.
   *
   * @return string
   *   Human-readable time format.
   */
  protected function formatCacheTime(int $seconds): string {
    if ($seconds >= 86400) {
      $days = floor($seconds / 86400);
      return $days . 'd';
    }
    elseif ($seconds >= 3600) {
      $hours = floor($seconds / 3600);
      return $hours . 'h';
    }
    elseif ($seconds >= 60) {
      $minutes = floor($seconds / 60);
      return $minutes . 'm';
    }
    return $seconds . 's';
  }

  /**
   * Detects the render mode of a view display.
   *
   * @param array $display_options
   *   The display options.
   *
   * @return array
   *   Array with render mode information.
   */
  protected function detectRenderMode(array $display_options): array {
    $row_plugin = $display_options['row']['type'] ?? 'fields';
    $style_plugin = $display_options['style']['type'] ?? 'default';
    $fields = $display_options['fields'] ?? [];
    $fields_count = count($fields);

    // Table formats where fields are expected.
    $table_formats = ['table', 'grid', 'html_list', 'list'];
    $is_table_format = in_array($style_plugin, $table_formats, TRUE);

    // Determine if using fields or rendered entity.
    $is_fields = $row_plugin === 'fields';
    $is_entity = in_array($row_plugin, ['entity', 'node', 'entity:node', 'entity:taxonomy_term', 'entity:user', 'entity:media'], TRUE);

    // Check for specific entity row plugins.
    if (str_starts_with($row_plugin, 'entity:')) {
      $is_entity = TRUE;
      $is_fields = FALSE;
    }

    // Get format label.
    $format_label = self::STYLE_PLUGIN_LABELS[$style_plugin] ?? ucfirst($style_plugin);

    // Determine mode and row label.
    if ($is_fields) {
      $mode = 'fields';
      $row_label = (string) $this->t('Fields (@count)', ['@count' => $fields_count]);
    }
    elseif ($is_entity) {
      $mode = 'rendered_entity';
      $view_mode = $display_options['row']['options']['view_mode'] ?? 'default';
      $row_label = (string) $this->t('Rendered entity (@mode)', ['@mode' => $view_mode]);
    }
    else {
      $mode = 'other';
      $row_label = ucfirst($row_plugin);
    }

    // Combined label: Format - Row mode.
    $label = $format_label . ' - ' . $row_label;

    return [
      'mode' => $mode,
      'label' => $label,
      'format_label' => $format_label,
      'row_label' => $row_label,
      'is_fields' => $is_fields,
      'is_table_format' => $is_table_format,
      'row_plugin' => $row_plugin,
      'style_plugin' => $style_plugin,
      'fields_count' => $fields_count,
    ];
  }

  /**
   * Determines if a view is an admin view.
   *
   * @param string $view_id
   *   The view ID.
   * @param string $base_table
   *   The view's base table.
   * @param array $display_options
   *   The display options.
   *
   * @return bool
   *   TRUE if this is an admin view.
   */
  protected function isAdminView(string $view_id, string $base_table, array $display_options): bool {
    // Check by view ID patterns.
    $admin_view_ids = [
      'admin_',
      'content',
      'files',
      'media',
      'user_admin',
      'watchdog',
      'taxonomy_term',
      'block_content',
    ];

    foreach ($admin_view_ids as $pattern) {
      if (str_starts_with($view_id, $pattern)) {
        return TRUE;
      }
    }

    // Check by path.
    $path = $display_options['path'] ?? '';
    foreach (self::ADMIN_PATH_PATTERNS as $pattern) {
      if (str_starts_with($path, $pattern)) {
        return TRUE;
      }
    }

    // Check by base table.
    if (in_array($base_table, self::ADMIN_BASE_TABLES, TRUE)) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Analyzes Search API views for MySQL field rendering.
   *
   * Detects fields in Search API views that load data from the database
   * instead of using indexed values. This happens when:
   * - A field uses plugin_id 'search_api_field' with 'field_rendering: true'
   * - A field uses a non-Search API plugin (like 'field', 'entity_reference_label')
   * - A field has 'link_to_item: true' (Search API fields) which generates entity URLs
   * - A field has 'link_to_entity: true' (standard Views fields) which generates entity URLs
   *
   * @param array $views
   *   Array of view entities.
   * @param bool $ignore_warnings
   *   If TRUE, report issues as notices instead of warnings.
   *
   * @return array
   *   The analysis results.
   */
  protected function analyzeSearchApi(array $views, bool $ignore_warnings = FALSE): array {
    $results = [];
    $warnings = 0;
    $notices = 0;

    $severity = $ignore_warnings ? 'notice' : 'warning';

    foreach ($views as $view) {
      $view_id = $view->id();
      $view_label = $view->label() ?: $view_id;
      $base_table = $view->get('base_table');

      if (!str_starts_with($base_table, 'search_api_index_')) {
        continue;
      }

      $displays = $view->get('display');
      $default_display = $displays['default']['display_options'] ?? [];
      $default_fields = $default_display['fields'] ?? [];

      foreach ($displays as $display_id => $display) {
        if ($display['display_plugin'] === 'default') {
          continue;
        }

        $display_options = $display['display_options'] ?? [];
        // Merge fields properly: display-specific fields override default fields.
        $fields = array_merge($default_fields, $display_options['fields'] ?? []);

        foreach ($fields as $field_id => $field_config) {
          $plugin_id = $field_config['plugin_id'] ?? '';
          $field_rendering = $field_config['field_rendering'] ?? FALSE;
          $link_to_item = $field_config['link_to_item'] ?? FALSE;
          $link_to_entity = $field_config['link_to_entity'] ?? FALSE;
          // Also check in settings array for link_to_entity.
          $settings_link_to_entity = $field_config['settings']['link_to_entity'] ?? FALSE;

          // Collect all reasons for MySQL queries.
          $reasons = [];

          // Plugins that don't cause MySQL queries (safe to ignore).
          $safe_plugins = [
            'nothing',        // Global: Custom text (just HTML/Twig).
            'counter',        // Global: View result counter.
            'custom',         // Global: Custom text.
            'dropbutton',     // Dropbutton (uses other fields).
            'rendered_entity', // Already handled separately.
          ];

          // Case 1: search_api_field plugin with field_rendering enabled.
          if ($plugin_id === 'search_api_field' && $field_rendering === TRUE) {
            $reasons[] = 'field_rendering enabled';
          }
          // Case 2: Non-Search API plugin (entity field, standard field handlers).
          // Exclude safe plugins that don't cause MySQL queries.
          elseif (!str_starts_with($plugin_id, 'search_api') && $plugin_id !== '' && !in_array($plugin_id, $safe_plugins, TRUE)) {
            $reasons[] = 'non-Search API plugin';
          }

          // Case 3: link_to_item enabled (Search API field option to link to entity).
          if ($link_to_item === TRUE) {
            $reasons[] = 'link_to_item enabled';
          }

          // Case 4: link_to_entity enabled (standard Views field option).
          if ($link_to_entity === TRUE || $settings_link_to_entity === TRUE) {
            $reasons[] = 'link_to_entity enabled';
          }

          if (!empty($reasons)) {
            $reason = implode(', ', $reasons);
            $results[] = $this->createResultItem(
              $severity,
              'SEARCHAPI_MYSQL_RENDER',
              (string) $this->t('Search API view @view:@display renders field @field from database (@reason)', [
                '@view' => $view_label,
                '@display' => $display_id,
                '@field' => $field_id,
                '@reason' => $reason,
              ]),
              [
                'view_id' => $view_id,
                'view_label' => $view_label,
                'display_id' => $display_id,
                'field_id' => $field_id,
                'plugin_id' => $plugin_id,
                'field_rendering' => $field_rendering,
                'link_to_item' => $link_to_item,
                'link_to_entity' => $link_to_entity || $settings_link_to_entity,
                'reason' => $reason,
              ]
            );

            if ($ignore_warnings) {
              $notices++;
            }
            else {
              $warnings++;
            }
          }
        }
      }
    }

    return $this->createResult($results, 0, $warnings, $notices);
  }

  /**
   * Analyzes Search API views cache configuration excluding those with facets.
   *
   * This analysis identifies Search API views that don't have facets configured,
   * showing their cache settings to help optimize performance. Views with facets
   * are excluded because they typically require special cache handling.
   *
   * @param array $views
   *   Array of view entities.
   *
   * @return array
   *   The analysis results.
   */
  protected function analyzeSearchApiCache(array $views): array {
    $results = [];
    $warnings = 0;
    $notices = 0;

    // First, collect all view/display combinations that have facets configured.
    $displays_with_facets = $this->getDisplaysWithFacets();

    // Analyze each Search API view display.
    foreach ($views as $view) {
      $view_id = $view->id();
      $view_label = $view->label() ?: $view_id;
      $base_table = $view->get('base_table');

      // Only process Search API views.
      if (!str_starts_with($base_table, 'search_api_index_')) {
        continue;
      }

      // Extract index name from base table.
      $index_id = str_replace('search_api_index_', '', $base_table);

      $displays = $view->get('display');
      $default_display = $displays['default']['display_options'] ?? [];

      foreach ($displays as $display_id => $display) {
        // Skip master display.
        if ($display['display_plugin'] === 'default') {
          continue;
        }

        $display_options = $display['display_options'] ?? [];
        $display_plugin = $display['display_plugin'] ?? 'unknown';

        // Merge with default display options for inherited values.
        $merged_options = array_merge($default_display, $display_options);

        // Check if this display has facets.
        $facet_key = $view_id . '__' . $display_id;
        if (isset($displays_with_facets[$facet_key])) {
          // Skip displays with facets - they are excluded from this analysis.
          continue;
        }

        // Get display title.
        $display_title = $display_options['title'] ?? $default_display['title'] ?? '';

        // Determine if enabled (default is TRUE).
        $enabled = $merged_options['enabled'] ?? TRUE;

        // Skip disabled displays.
        if (!$enabled) {
          continue;
        }

        // Get cache configuration.
        $cache_config = $merged_options['cache'] ?? [];
        $cache_type = $cache_config['type'] ?? 'none';
        $cache_options = $cache_config['options'] ?? [];

        // Get detailed cache info.
        $cache_details = $this->getCacheDetails($cache_config);

        // Get display type label.
        $display_type_label = self::DISPLAY_TYPE_LABELS[$display_plugin] ?? ucfirst($display_plugin);

        // Get path if available.
        $path = $merged_options['path'] ?? '';

        // Calculate cache time for time-based cache.
        $cache_time_seconds = 0;
        $cache_time_label = '';
        if (in_array($cache_type, ['time', 'search_api_time'], TRUE)) {
          $results_lifespan = (int) ($cache_options['results_lifespan'] ?? 0);
          $output_lifespan = (int) ($cache_options['output_lifespan'] ?? 0);
          // Use the minimum of the two if both are set and positive.
          if ($results_lifespan > 0 && $output_lifespan > 0) {
            $cache_time_seconds = min($results_lifespan, $output_lifespan);
          }
          elseif ($results_lifespan > 0) {
            $cache_time_seconds = $results_lifespan;
          }
          elseif ($output_lifespan > 0) {
            $cache_time_seconds = $output_lifespan;
          }
          $cache_time_label = $cache_time_seconds > 0 ? $this->formatCacheTime($cache_time_seconds) : '';
        }

        // Determine severity based on cache configuration.
        $severity = 'info';
        if ($cache_type === 'none' || $cache_type === 'search_api_none') {
          $severity = 'warning';
          $warnings++;
        }
        elseif (in_array($cache_type, ['time', 'search_api_time'], TRUE) && $cache_time_seconds > 0 && $cache_time_seconds < self::MIN_RECOMMENDED_CACHE_TIME) {
          $severity = 'notice';
          $notices++;
        }

        $details = [
          'view_id' => $view_id,
          'view_label' => $view_label,
          'display_id' => $display_id,
          'display_title' => $display_title,
          'display_plugin' => $display_plugin,
          'display_type_label' => $display_type_label,
          'path' => $path,
          'index_id' => $index_id,
          'cache_type' => $cache_type,
          'cache_label' => $cache_details['label'],
          'cache_time_seconds' => $cache_time_seconds,
          'cache_time_label' => $cache_time_label,
          'has_facets' => FALSE,
        ];

        $results[] = $this->createResultItem(
          $severity,
          'SEARCHAPI_CACHE',
          (string) $this->t('Search API view @view:@display - Cache: @cache', [
            '@view' => $view_label,
            '@display' => $display_id,
            '@cache' => $cache_details['label'],
          ]),
          $details
        );
      }
    }

    return $this->createResult($results, 0, $warnings, $notices);
  }

  /**
   * Gets all view displays that have facets configured.
   *
   * Reads facet configuration to determine which view/display combinations
   * have facets attached to them.
   *
   * @return array
   *   Associative array where keys are "view_id__display_id" and values
   *   are arrays of facet IDs configured for that display.
   */
  protected function getDisplaysWithFacets(): array {
    $displays_with_facets = [];

    // Check if facets module is installed.
    if (!$this->moduleHandler->moduleExists('facets')) {
      return $displays_with_facets;
    }

    // Load all facet configurations.
    $facet_configs = $this->configFactory->listAll('facets.facet.');

    foreach ($facet_configs as $config_name) {
      $config = $this->configFactory->get($config_name);
      $facet_source_id = $config->get('facet_source_id');
      $facet_id = $config->get('id');
      $facet_name = $config->get('name');

      if (!$facet_source_id) {
        continue;
      }

      // Parse facet_source_id format: "search_api:views_page__view_id__display_id"
      // or "search_api:views_block__view_id__display_id".
      if (preg_match('/^search_api:views_(?:page|block|rest_export|data_export|embed|attachment)__(.+)__(.+)$/', $facet_source_id, $matches)) {
        $view_id = $matches[1];
        $display_id = $matches[2];
        $key = $view_id . '__' . $display_id;

        if (!isset($displays_with_facets[$key])) {
          $displays_with_facets[$key] = [];
        }
        $displays_with_facets[$key][] = [
          'id' => $facet_id,
          'name' => $facet_name,
        ];
      }
    }

    return $displays_with_facets;
  }

  /**
   * Score weights for each factor.
   */
  protected const SCORE_WEIGHTS = [
    'cache_config' => 40,
    'relationships' => 15,
    'cache_tags' => 15,
    'searchapi_rendering' => 20,
    'searchapi_cache' => 10,
  ];

  /**
   * Penalty multiplier applied to issue percentage.
   *
   * With multiplier 2.5:
   * - 4% issues → score 90
   * - 10% issues → score 75
   * - 20% issues → score 50
   * - 40%+ issues → score 0
   */
  protected const PENALTY_MULTIPLIER = 2.5;

  /**
   * Weight for frontend issues (more critical).
   */
  protected const FRONTEND_WEIGHT = 1.0;

  /**
   * Weight for admin issues (less critical).
   */
  protected const ADMIN_WEIGHT = 0.5;

  /**
   * Calculates scores for all factors using volume-aware penalty scoring.
   *
   * Formula: score = 100 - (issue_percentage * PENALTY_MULTIPLIER)
   *
   * This considers both the volume of views and applies a multiplier
   * to make issues more impactful than simple percentages.
   *
   * Frontend issues have higher weight (1.0) than admin issues (0.5).
   *
   * @param array $displays_results
   *   Displays analysis results.
   * @param array $searchapi_results
   *   Search API analysis results.
   * @param array $searchapi_cache_results
   *   Search API cache analysis results.
   * @param int $relationships_threshold
   *   The threshold for relationship warnings.
   *
   * @return array
   *   Score data with overall and factors.
   */
  protected function calculateScores(array $displays_results, array $searchapi_results, array $searchapi_cache_results, int $relationships_threshold): array {
    // Collect statistics in a single pass through the results.
    $stats = [
      'total_enabled' => 0,
      'total_frontend' => 0,
      'total_admin' => 0,
      // Views with tag-based cache (for cache_tags factor).
      'total_tag_cache' => 0,
      'total_tag_cache_frontend' => 0,
      'total_tag_cache_admin' => 0,
      // Cache issues.
      'cache_issues_frontend' => 0,
      'cache_issues_admin' => 0,
      // Relationship issues.
      'relationship_issues_frontend' => 0,
      'relationship_issues_admin' => 0,
      // Generic cache tag issues.
      'cache_tag_issues_frontend' => 0,
      'cache_tag_issues_admin' => 0,
    ];

    foreach ($displays_results['results'] ?? [] as $item) {
      if (($item['code'] ?? '') !== 'VIEW_DISPLAY') {
        continue;
      }

      $details = $item['details'] ?? [];
      $enabled = $details['enabled'] ?? TRUE;

      // Only count enabled displays for scoring.
      if (!$enabled) {
        continue;
      }

      $stats['total_enabled']++;
      $is_admin = $details['is_admin'] ?? FALSE;

      if ($is_admin) {
        $stats['total_admin']++;
      }
      else {
        $stats['total_frontend']++;
      }

      $cache_type = $details['cache_type'] ?? 'none';
      $has_cache_issue = FALSE;

      // Check for cache issues.
      if ($cache_type === 'none') {
        $has_cache_issue = TRUE;
      }
      elseif ($cache_type === 'time') {
        $cache_details = $details['cache_details'] ?? [];
        $cache_options = $cache_details['options'] ?? [];
        $results_lifespan = (int) ($cache_options['results_lifespan'] ?? 0);
        $output_lifespan = (int) ($cache_options['output_lifespan'] ?? 0);

        $results_too_low = $results_lifespan > 0 && $results_lifespan < self::MIN_RECOMMENDED_CACHE_TIME;
        $output_too_low = $output_lifespan > 0 && $output_lifespan < self::MIN_RECOMMENDED_CACHE_TIME;

        if ($results_too_low || $output_too_low) {
          $has_cache_issue = TRUE;
        }
      }

      if ($has_cache_issue) {
        if ($is_admin) {
          $stats['cache_issues_admin']++;
        }
        else {
          $stats['cache_issues_frontend']++;
        }
      }

      // Check for relationship issues.
      if (($details['relationships_count'] ?? 0) > $relationships_threshold) {
        if ($is_admin) {
          $stats['relationship_issues_admin']++;
        }
        else {
          $stats['relationship_issues_frontend']++;
        }
      }

      // Check for generic cache tag issues (tag-based cache with bundle filters).
      if ($cache_type === 'tag') {
        $stats['total_tag_cache']++;
        if ($is_admin) {
          $stats['total_tag_cache_admin']++;
        }
        else {
          $stats['total_tag_cache_frontend']++;
        }

        // Views with tag cache AND bundle filters have generic tag issue.
        if ($details['cache_tag_optimizable'] ?? FALSE) {
          if ($is_admin) {
            $stats['cache_tag_issues_admin']++;
          }
          else {
            $stats['cache_tag_issues_frontend']++;
          }
        }
      }
    }

    // Build factors with volume-aware penalty scoring.
    $factors = [];

    // Calculate weighted totals for percentage calculations.
    $weighted_total = ($stats['total_frontend'] * self::FRONTEND_WEIGHT)
      + ($stats['total_admin'] * self::ADMIN_WEIGHT);

    // Factor 1: Cache Configuration.
    $cache_issues_total = $stats['cache_issues_frontend'] + $stats['cache_issues_admin'];
    $cache_weighted_issues = ($stats['cache_issues_frontend'] * self::FRONTEND_WEIGHT)
      + ($stats['cache_issues_admin'] * self::ADMIN_WEIGHT);
    $cache_percentage = $weighted_total > 0 ? ($cache_weighted_issues / $weighted_total) * 100 : 0;
    $cache_score = (int) max(0, round(100 - ($cache_percentage * self::PENALTY_MULTIPLIER)));

    $cache_description = $cache_issues_total === 0
      ? (string) $this->t('All @count enabled views have proper cache configured', [
        '@count' => $stats['total_enabled'],
      ])
      : (string) $this->t('@issues of @total views with cache issues (@frontend frontend, @admin admin) — @percent% weighted', [
        '@issues' => $cache_issues_total,
        '@total' => $stats['total_enabled'],
        '@frontend' => $stats['cache_issues_frontend'],
        '@admin' => $stats['cache_issues_admin'],
        '@percent' => round($cache_percentage, 1),
      ]);

    $factors['cache_config'] = [
      'score' => $cache_score,
      'weight' => self::SCORE_WEIGHTS['cache_config'],
      'label' => (string) $this->t('Cache Configuration'),
      'description' => $cache_description,
    ];

    // Factor 2: Relationships.
    $rel_issues_total = $stats['relationship_issues_frontend'] + $stats['relationship_issues_admin'];
    $rel_weighted_issues = ($stats['relationship_issues_frontend'] * self::FRONTEND_WEIGHT)
      + ($stats['relationship_issues_admin'] * self::ADMIN_WEIGHT);
    $rel_percentage = $weighted_total > 0 ? ($rel_weighted_issues / $weighted_total) * 100 : 0;
    $rel_score = (int) max(0, round(100 - ($rel_percentage * self::PENALTY_MULTIPLIER)));

    $rel_description = $rel_issues_total === 0
      ? (string) $this->t('All views have optimal relationship count (≤@threshold)', [
        '@threshold' => $relationships_threshold,
      ])
      : (string) $this->t('@issues of @total views with excessive relationships (@frontend frontend, @admin admin) — @percent% weighted', [
        '@issues' => $rel_issues_total,
        '@total' => $stats['total_enabled'],
        '@frontend' => $stats['relationship_issues_frontend'],
        '@admin' => $stats['relationship_issues_admin'],
        '@percent' => round($rel_percentage, 1),
      ]);

    $factors['relationships'] = [
      'score' => $rel_score,
      'weight' => self::SCORE_WEIGHTS['relationships'],
      'label' => (string) $this->t('Relationships'),
      'description' => $rel_description,
    ];

    // Factor 3: Cache Tags.
    // Use tag-cache views as the base for percentage calculation.
    $tag_cache_weighted_total = ($stats['total_tag_cache_frontend'] * self::FRONTEND_WEIGHT)
      + ($stats['total_tag_cache_admin'] * self::ADMIN_WEIGHT);
    $cache_tag_issues_total = $stats['cache_tag_issues_frontend'] + $stats['cache_tag_issues_admin'];
    $cache_tags_weighted_issues = ($stats['cache_tag_issues_frontend'] * self::FRONTEND_WEIGHT)
      + ($stats['cache_tag_issues_admin'] * self::ADMIN_WEIGHT);
    $cache_tags_percentage = $tag_cache_weighted_total > 0
      ? ($cache_tags_weighted_issues / $tag_cache_weighted_total) * 100
      : 0;
    $cache_tags_score = (int) max(0, round(100 - ($cache_tags_percentage * self::PENALTY_MULTIPLIER)));

    $cache_tags_description = $cache_tag_issues_total === 0
      ? ($stats['total_tag_cache'] > 0
        ? (string) $this->t('All @count views with tag-based cache use specific cache tags', [
          '@count' => $stats['total_tag_cache'],
        ])
        : (string) $this->t('No views use tag-based cache'))
      : (string) $this->t('@issues of @total tag-cached views use generic entity list tags (@frontend frontend, @admin admin) — @percent% weighted', [
        '@issues' => $cache_tag_issues_total,
        '@total' => $stats['total_tag_cache'],
        '@frontend' => $stats['cache_tag_issues_frontend'],
        '@admin' => $stats['cache_tag_issues_admin'],
        '@percent' => round($cache_tags_percentage, 1),
      ]);

    $factors['cache_tags'] = [
      'score' => $cache_tags_score,
      'weight' => self::SCORE_WEIGHTS['cache_tags'],
      'label' => (string) $this->t('Cache Tags'),
      'description' => $cache_tags_description,
    ];

    // Factor 4 & 5: Search API (only if module is active).
    if ($this->moduleHandler->moduleExists('search_api')) {
      // Factor 4: Search API Rendering.
      // Use penalty per issue since we don't have a "total fields" count.
      $searchapi_mysql_warnings = $searchapi_results['summary']['warnings'] ?? 0;
      $searchapi_mysql_notices = $searchapi_results['summary']['notices'] ?? 0;
      $searchapi_rendering_total = $searchapi_mysql_warnings + $searchapi_mysql_notices;
      // Each warning = 10 points, each notice = 5 points.
      $searchapi_rendering_penalty = ($searchapi_mysql_warnings * 10) + ($searchapi_mysql_notices * 5);
      $searchapi_rendering_score = (int) max(0, 100 - $searchapi_rendering_penalty);

      $factors['searchapi_rendering'] = [
        'score' => $searchapi_rendering_score,
        'weight' => self::SCORE_WEIGHTS['searchapi_rendering'],
        'label' => (string) $this->t('Search API Rendering'),
        'description' => $searchapi_rendering_total === 0
          ? (string) $this->t('All Search API fields use indexed data')
          : (string) $this->t('@count fields loading from database (@warnings warnings, @notices notices)', [
            '@count' => $searchapi_rendering_total,
            '@warnings' => $searchapi_mysql_warnings,
            '@notices' => $searchapi_mysql_notices,
          ]),
      ];

      // Factor 5: Search API Cache.
      $searchapi_cache_total = count($searchapi_cache_results['results'] ?? []);
      $searchapi_cache_warnings = $searchapi_cache_results['summary']['warnings'] ?? 0;
      $searchapi_cache_percentage = $searchapi_cache_total > 0
        ? ($searchapi_cache_warnings / $searchapi_cache_total) * 100
        : 0;
      $searchapi_cache_score = (int) max(0, round(100 - ($searchapi_cache_percentage * self::PENALTY_MULTIPLIER)));

      $factors['searchapi_cache'] = [
        'score' => $searchapi_cache_score,
        'weight' => self::SCORE_WEIGHTS['searchapi_cache'],
        'label' => (string) $this->t('Search API Cache'),
        'description' => $searchapi_cache_warnings === 0
          ? (string) $this->t('All @count Search API views (without facets) have cache configured', [
            '@count' => $searchapi_cache_total,
          ])
          : (string) $this->t('@issues of @total Search API views without cache — @percent%', [
            '@issues' => $searchapi_cache_warnings,
            '@total' => $searchapi_cache_total,
            '@percent' => round($searchapi_cache_percentage, 1),
          ]),
      ];
    }

    return [
      'factors' => $factors,
    ];
  }

  /**
   * Builds the Search API Cache section for views without facets.
   *
   * @param array $searchapi_cache_data
   *   The searchapi_cache analysis data.
   *
   * @return array
   *   Render array for the section.
   */
  protected function buildSearchApiCacheSection(array $searchapi_cache_data): array {
    $results = $searchapi_cache_data['results'] ?? [];
    $summary = $searchapi_cache_data['summary'] ?? [];

    $section_content = [];

    // Add explanation message.
    $section_content['explanation'] = $this->ui->message(
      (string) $this->t('These Search API views do not have facets configured. Since they don\'t need to handle facet interactions, they can safely use caching to improve performance.'),
      'info'
    );

    // Build the table.
    $headers = [
      $this->ui->header('', 'left', '40px'),
      $this->ui->header((string) $this->t('View')),
      $this->ui->header((string) $this->t('Display')),
      $this->ui->header((string) $this->t('Type')),
      $this->ui->header((string) $this->t('Index')),
      $this->ui->header((string) $this->t('Cache')),
      $this->ui->header((string) $this->t('Cache Time'), 'center', '100px'),
      $this->ui->header((string) $this->t('Path')),
    ];

    $rows = [];
    foreach ($results as $item) {
      $details = $item['details'] ?? [];
      $type = $item['type'] ?? 'info';
      $cache_type = $details['cache_type'] ?? 'none';

      // Severity icon.
      if ($type === 'warning') {
        $severity_content = $this->ui->icon('warning');
      }
      elseif ($type === 'notice') {
        $severity_content = $this->ui->icon('info');
      }
      else {
        $severity_content = $this->ui->icon('check');
      }

      // View name.
      $view_content = $this->ui->itemName($details['view_label'], $details['view_id']);

      // Display name.
      $display_title = $details['display_title'] ?: $details['display_id'];
      $display_content = $this->ui->itemName($display_title, $details['display_id']);

      // Display type.
      $type_content = $details['display_type_label'] ?? $details['display_plugin'];

      // Index.
      $index_content = $details['index_id'] ?? '';

      // Cache badge.
      if ($cache_type === 'none' || $cache_type === 'search_api_none') {
        $cache_content = $this->ui->badge($details['cache_label'], 'error');
      }
      elseif (in_array($cache_type, ['time', 'search_api_time'], TRUE)) {
        $cache_time = $details['cache_time_seconds'] ?? 0;
        if ($cache_time > 0 && $cache_time < self::MIN_RECOMMENDED_CACHE_TIME) {
          $cache_content = $this->ui->badge($details['cache_label'], 'warning');
        }
        else {
          $cache_content = $this->ui->badge($details['cache_label'], 'success');
        }
      }
      else {
        $cache_content = $this->ui->badge($details['cache_label'], 'success');
      }

      // Cache time.
      $cache_time_content = $details['cache_time_label'] ?? '-';

      // Path.
      $path_content = $details['path'] ?? '';

      $rows[] = $this->ui->row([
        $this->ui->cell($severity_content, ['align' => 'center']),
        $this->ui->cell($view_content),
        $this->ui->cell($display_content),
        $this->ui->cell($type_content),
        $this->ui->cell($index_content),
        $this->ui->cell($cache_content),
        $this->ui->cell($cache_time_content, ['align' => 'center']),
        $this->ui->cell($path_content),
      ], $type === 'warning' ? 'warning' : ($type === 'notice' ? 'notice' : NULL));
    }

    $section_content['table'] = $this->ui->table($headers, $rows);

    // Determine section severity.
    $warnings = $summary['warnings'] ?? 0;
    $notices = $summary['notices'] ?? 0;
    $severity = NULL;
    if ($warnings > 0) {
      $severity = 'warning';
    }
    elseif ($notices > 0) {
      $severity = 'notice';
    }

    // Build section title with counts.
    $title_parts = [(string) $this->t('Displays: @count', ['@count' => count($results)])];
    if ($warnings > 0) {
      $title_parts[] = (string) $this->formatPlural($warnings, '1 warning', '@count warnings');
    }
    if ($notices > 0) {
      $title_parts[] = (string) $this->formatPlural($notices, '1 notice', '@count notices');
    }

    return $this->ui->section(
      (string) $this->t('Search API Cache (No Facets)') . ' (' . implode(', ', $title_parts) . ')',
      $section_content,
      [
        'open' => TRUE,
        'severity' => $severity,
      ]
    );
  }

  /**
   * Determines severity level from statistics.
   *
   * @param array $stats
   *   Statistics array from calculateGroupStats().
   *
   * @return string|null
   *   The severity level ('error', 'warning', 'notice') or NULL if none.
   */
  protected function getSeverityFromStats(array $stats): ?string {
    if ($stats['errors'] > 0) {
      return 'error';
    }
    if ($stats['warnings'] > 0) {
      return 'warning';
    }
    if ($stats['notices'] > 0) {
      return 'notice';
    }
    return NULL;
  }

  /**
   * Calculates statistics for a group of displays.
   *
   * @param array $displays
   *   Array of display items.
   * @param int $relationships_threshold
   *   The threshold for relationship warnings.
   *
   * @return array
   *   Statistics array with views_count, displays_count, errors, warnings, notices.
   */
  protected function calculateGroupStats(array $displays, int $relationships_threshold): array {
    $views_set = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    foreach ($displays as $item) {
      $details = $item['details'] ?? [];
      $views_set[$details['view_id'] ?? ''] = TRUE;

      $enabled = $details['enabled'] ?? TRUE;
      $cache_type = $details['cache_type'] ?? 'none';
      $relationships_count = $details['relationships_count'] ?? 0;
      $is_fields_render = $details['is_fields_render'] ?? FALSE;
      $is_table_format = $details['is_table_format'] ?? FALSE;
      $cache_tag_optimizable = $details['cache_tag_optimizable'] ?? FALSE;

      if ($enabled) {
        // No cache is a warning.
        if ($cache_type === 'none') {
          $warnings++;
        }
        // Too many relationships is a warning.
        if ($relationships_count > $relationships_threshold) {
          $warnings++;
        }
        // Fields render mode on non-table formats is a notice.
        if ($is_fields_render && !$is_table_format) {
          $notices++;
        }
        // Tag-based cache with generic tags is a warning.
        if ($cache_tag_optimizable) {
          $warnings++;
        }
      }
      else {
        // Disabled views are notices.
        $notices++;
      }
    }

    return [
      'views_count' => count($views_set),
      'displays_count' => count($displays),
      'errors' => $errors,
      'warnings' => $warnings,
      'notices' => $notices,
    ];
  }

  /**
   * Builds a title with statistics for a group.
   *
   * @param string $base_title
   *   The base title.
   * @param array $stats
   *   Statistics array from calculateGroupStats().
   *
   * @return string
   *   The formatted title.
   */
  protected function buildGroupTitle(string $base_title, array $stats): string {
    $parts = [];

    // Views and displays count.
    $parts[] = (string) $this->t('Views: @views', ['@views' => $stats['views_count']]);
    $parts[] = (string) $this->t('Displays: @displays', ['@displays' => $stats['displays_count']]);

    // Issues counts.
    $issues_parts = [];
    if ($stats['errors'] > 0) {
      $issues_parts[] = (string) $this->formatPlural($stats['errors'], '1 error', '@count errors');
    }
    if ($stats['warnings'] > 0) {
      $issues_parts[] = (string) $this->formatPlural($stats['warnings'], '1 warning', '@count warnings');
    }
    if ($stats['notices'] > 0) {
      $issues_parts[] = (string) $this->formatPlural($stats['notices'], '1 notice', '@count notices');
    }

    $title = $base_title . ' (' . implode(', ', $parts);
    if (!empty($issues_parts)) {
      $title .= ' - ' . implode(', ', $issues_parts);
    }
    $title .= ')';

    return $title;
  }

  /**
   * Builds a table for view displays.
   *
   * @param array $displays
   *   Array of display items.
   * @param int $relationships_threshold
   *   The threshold for relationship warnings.
   *
   * @return array
   *   Render array for the table.
   */
  protected function buildDisplaysTable(array $displays, int $relationships_threshold): array {
    $headers = [
      $this->ui->header('', 'left', '40px'),
      $this->ui->header((string) $this->t('View')),
      $this->ui->header((string) $this->t('Display')),
      $this->ui->header((string) $this->t('Display Type')),
      $this->ui->header((string) $this->t('Render Mode')),
      $this->ui->header((string) $this->t('Entity/Bundle')),
      $this->ui->header((string) $this->t('Cache')),
      $this->ui->header((string) $this->t('Relations'), 'center', '80px'),
      $this->ui->header((string) $this->t('Actions'), 'center', '100px'),
    ];

    $rows = [];

    foreach ($displays as $item) {
      $details = $item['details'];
      $enabled = $details['enabled'] ?? TRUE;
      $cache_type = $details['cache_type'] ?? 'none';
      $rel_count = $details['relationships_count'] ?? 0;
      $is_fields_render = $details['is_fields_render'] ?? FALSE;
      $is_table_format = $details['is_table_format'] ?? FALSE;
      $cache_tag_optimizable = $details['cache_tag_optimizable'] ?? FALSE;

      // Determine severity for the row.
      $severity = NULL;

      if ($enabled) {
        // No cache, too many relationships, or generic cache tags = warning.
        if ($cache_type === 'none' || $rel_count > $relationships_threshold || $cache_tag_optimizable) {
          $severity = 'warning';
        }
        // Fields on non-table format = notice.
        elseif ($is_fields_render && !$is_table_format) {
          $severity = 'notice';
        }
      }
      else {
        // Disabled views = notice.
        $severity = 'notice';
      }

      // Severity cell (first column) - using icon or badge.
      $severity_content = '';
      if ($severity === 'warning') {
        $severity_content = $this->ui->icon('warning');
      }
      elseif ($severity === 'notice') {
        $severity_content = $this->ui->icon('info');
      }
      else {
        $severity_content = $this->ui->icon('check');
      }

      // View name cell.
      $view_content = $this->ui->itemName($details['view_label'], $details['view_id']);

      // Display name cell.
      $display_title = $details['display_title'] ?: $details['display_id'];
      $display_content = $this->ui->itemName($display_title, $details['display_id']);

      // Display type.
      $type_content = $details['display_type_label'] ?? $details['display_plugin'];

      // Render mode cell.
      $render_label = $details['render_mode_label'] ?? 'Unknown';
      $render_content = $render_label;

      // Entity/Bundle cell.
      $entity_bundle_label = $details['entity_bundle_label'] ?? $details['base_table'] ?? '';
      $entity_content = $entity_bundle_label;

      // Cache type cell with badge for status.
      $cache_label = $details['cache_label'] ?? $cache_type;
      if ($cache_type === 'none') {
        $cache_content = $this->ui->badge($cache_label, 'error');
      }
      elseif ($cache_tag_optimizable) {
        $cache_content = $this->ui->badge($cache_label, 'warning');
      }
      else {
        $cache_content = $this->ui->badge($cache_label, 'success');
      }

      // Relationships cell with badge for status.
      if ($rel_count > $relationships_threshold) {
        $rel_content = $this->ui->badge((string) $rel_count, 'warning');
      }
      elseif ($rel_count > 0) {
        $rel_content = $this->ui->badge((string) $rel_count, 'info');
      }
      else {
        $rel_content = $this->ui->badge((string) $rel_count, 'neutral');
      }

      // Edit button cell.
      $view_id = $details['view_id'] ?? '';
      $display_id = $details['display_id'] ?? '';
      $edit_url = "/admin/structure/views/view/{$view_id}/edit/{$display_id}";
      $edit_content = '<a href="' . htmlspecialchars($edit_url, ENT_QUOTES, 'UTF-8') . '" class="button button--small" title="' . htmlspecialchars((string) $this->t('Edit view'), ENT_QUOTES, 'UTF-8') . '">';
      $edit_content .= '<span aria-hidden="true">✎</span>';
      $edit_content .= '</a>';

      $rows[] = $this->ui->row([
        $this->ui->cell($severity_content, ['align' => 'center']),
        $this->ui->cell($view_content),
        $this->ui->cell($display_content),
        $this->ui->cell($type_content),
        $this->ui->cell($render_content),
        $this->ui->cell($entity_content),
        $this->ui->cell($cache_content),
        $this->ui->cell($rel_content, ['align' => 'center']),
        $this->ui->cell($edit_content, ['align' => 'center']),
      ], $severity);
    }

    return $this->ui->table($headers, $rows);
  }

}
