<?php

declare(strict_types=1);

namespace Drupal\audit_watchdog\Plugin\AuditAnalyzer;

use Drupal\audit\Attribute\AuditAnalyzer;
use Drupal\audit\AuditAnalyzerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Analyzes watchdog (database logging) for errors, warnings, and issues.
 */
#[AuditAnalyzer(
  id: 'watchdog',
  label: new TranslatableMarkup('Watchdog Audit'),
  description: new TranslatableMarkup('Analyzes database logging for PHP errors, 404s, security issues, and overall log health.'),
  menu_title: new TranslatableMarkup('Watchdog'),
  output_directory: 'watchdog',
  weight: 8,
)]
class WatchdogAnalyzer extends AuditAnalyzerBase {

  /**
   * Score weights for different factors.
   */
  protected const SCORE_WEIGHTS = [
    'php_errors' => 30,
    'not_found' => 20,
    'security' => 30,
    'log_health' => 20,
  ];

  /**
   * Severity levels that indicate PHP errors.
   */
  protected const PHP_ERROR_TYPES = [
    'php',
  ];

  /**
   * Severity levels that indicate security issues.
   */
  protected const SECURITY_TYPES = [
    'security',
    'user',
  ];

  /**
   * Severity levels in watchdog.
   */
  protected const SEVERITY_LEVELS = [
    0 => 'Emergency',
    1 => 'Alert',
    2 => 'Critical',
    3 => 'Error',
    4 => 'Warning',
    5 => 'Notice',
    6 => 'Info',
    7 => 'Debug',
  ];

  protected Connection $database;
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->database = $container->get('database');
    $instance->moduleHandler = $container->get('module_handler');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function checkRequirements(): array {
    $warnings = [];

    if (!$this->moduleHandler->moduleExists('dblog')) {
      $warnings[] = (string) $this->t('Database Logging (dblog) module is not enabled. This analyzer requires dblog to analyze watchdog entries.');
    }

    return $warnings;
  }

  /**
   * {@inheritdoc}
   */
  public function analyze(): array {
    $results = [];
    $errors = 0;
    $warnings = 0;
    $notices = 0;

    // Get log statistics.
    $stats = $this->getLogStatistics();
    $php_errors = $this->getPhpErrors();
    $not_found_errors = $this->get404Errors();
    $security_issues = $this->getSecurityIssues();
    $log_age = $this->getLogAge();

    // Generate result items based on findings.

    // PHP Errors analysis.
    $php_error_count = $stats['php_errors'] ?? 0;
    $total_entries = $stats['total'] ?? 1;
    $php_error_percentage = $total_entries > 0 ? ($php_error_count / $total_entries) * 100 : 0;

    if ($php_error_percentage >= 10) {
      $errors++;
      $results[] = $this->createResultItem(
        'error',
        'HIGH_PHP_ERRORS',
        (string) $this->t('PHP errors make up @percent% of all log entries (@count errors). This indicates serious application issues.', [
          '@percent' => round($php_error_percentage, 1),
          '@count' => $php_error_count,
        ]),
        [
          'count' => $php_error_count,
          'percentage' => round($php_error_percentage, 1),
          'top_errors' => array_slice($php_errors, 0, 10),
        ]
      );
    }
    elseif ($php_error_count > 0) {
      $warnings++;
      $results[] = $this->createResultItem(
        'warning',
        'PHP_ERRORS_PRESENT',
        (string) $this->t('@count PHP errors found in logs (@percent% of entries).', [
          '@count' => $php_error_count,
          '@percent' => round($php_error_percentage, 1),
        ]),
        [
          'count' => $php_error_count,
          'percentage' => round($php_error_percentage, 1),
          'top_errors' => array_slice($php_errors, 0, 10),
        ]
      );
    }
    else {
      $notices++;
      $results[] = $this->createResultItem(
        'notice',
        'NO_PHP_ERRORS',
        (string) $this->t('No PHP errors found in logs.'),
        ['count' => 0]
      );
    }

    // 404 Errors analysis.
    $not_found_count = $stats['page_not_found'] ?? 0;
    $not_found_percentage = $total_entries > 0 ? ($not_found_count / $total_entries) * 100 : 0;

    if ($not_found_percentage >= 20) {
      $warnings++;
      $results[] = $this->createResultItem(
        'warning',
        'HIGH_404_ERRORS',
        (string) $this->t('404 errors make up @percent% of all log entries (@count errors). This may indicate broken links or crawler issues.', [
          '@percent' => round($not_found_percentage, 1),
          '@count' => $not_found_count,
        ]),
        [
          'count' => $not_found_count,
          'percentage' => round($not_found_percentage, 1),
          'top_urls' => array_slice($not_found_errors, 0, 20),
        ]
      );
    }
    elseif ($not_found_count > 100) {
      $notices++;
      $results[] = $this->createResultItem(
        'notice',
        '404_ERRORS_PRESENT',
        (string) $this->t('@count 404 errors found in logs.', [
          '@count' => $not_found_count,
        ]),
        [
          'count' => $not_found_count,
          'top_urls' => array_slice($not_found_errors, 0, 20),
        ]
      );
    }

    // Security issues analysis.
    $security_count = count($security_issues);
    if ($security_count > 0) {
      $failed_logins = 0;
      $access_denied = 0;
      foreach ($security_issues as $issue) {
        if (str_contains($issue['message'] ?? '', 'Login attempt failed')) {
          $failed_logins += $issue['count'] ?? 1;
        }
        if (str_contains($issue['message'] ?? '', 'Access denied')) {
          $access_denied += $issue['count'] ?? 1;
        }
      }

      if ($failed_logins > 50) {
        $warnings++;
        $results[] = $this->createResultItem(
          'warning',
          'HIGH_FAILED_LOGINS',
          (string) $this->t('@count failed login attempts detected. This may indicate a brute force attack.', [
            '@count' => $failed_logins,
          ]),
          [
            'count' => $failed_logins,
            'recommendation' => 'Consider implementing flood control or IP blocking.',
          ]
        );
      }

      if ($access_denied > 100) {
        $notices++;
        $results[] = $this->createResultItem(
          'notice',
          'ACCESS_DENIED_ENTRIES',
          (string) $this->t('@count access denied entries found.', [
            '@count' => $access_denied,
          ]),
          ['count' => $access_denied]
        );
      }
    }

    // Log age analysis.
    if (!empty($log_age['oldest']) && !empty($log_age['newest'])) {
      $oldest_timestamp = strtotime($log_age['oldest']);
      $newest_timestamp = strtotime($log_age['newest']);
      $days_covered = ($newest_timestamp - $oldest_timestamp) / 86400;

      $notices++;
      $results[] = $this->createResultItem(
        'notice',
        'LOG_AGE_INFO',
        (string) $this->t('Logs cover @days days, from @oldest to @newest.', [
          '@days' => round($days_covered, 1),
          '@oldest' => $log_age['oldest'],
          '@newest' => $log_age['newest'],
        ]),
        [
          'oldest' => $log_age['oldest'],
          'newest' => $log_age['newest'],
          'days_covered' => round($days_covered, 1),
        ]
      );
    }

    // Check if syslog is enabled.
    $has_syslog = $this->moduleHandler->moduleExists('syslog');
    if (!$has_syslog) {
      $notices++;
      $results[] = $this->createResultItem(
        'notice',
        'SYSLOG_NOT_ENABLED',
        (string) $this->t('Syslog module is not enabled. Consider enabling syslog for better log management and performance.'),
        ['recommendation' => 'Enable syslog module for production environments']
      );
    }

    // Calculate scores.
    $scores = $this->calculateScores($stats, $php_error_percentage, $not_found_percentage, $security_issues);

    $output = $this->createResult($results, $errors, $warnings, $notices);
    $output['_filename'] = 'analysis';
    $output['score'] = $scores;
    $output['statistics'] = $stats;
    $output['php_errors'] = $php_errors;
    $output['not_found_errors'] = $not_found_errors;
    $output['security_issues'] = $security_issues;
    $output['log_age'] = $log_age;

    return $output;
  }

  /**
   * Gets overall log statistics.
   */
  protected function getLogStatistics(): array {
    $stats = [
      'total' => 0,
      'by_type' => [],
      'by_severity' => [],
      'php_errors' => 0,
      'page_not_found' => 0,
    ];

    try {
      // Total count.
      $stats['total'] = (int) $this->database->select('watchdog', 'w')
        ->countQuery()
        ->execute()
        ->fetchField();

      // Count by type.
      $query = $this->database->select('watchdog', 'w');
      $query->addField('w', 'type');
      $query->addExpression('COUNT(*)', 'count');
      $query->groupBy('w.type');
      $query->orderBy('count', 'DESC');

      foreach ($query->execute() as $row) {
        $stats['by_type'][$row->type] = (int) $row->count;

        if (in_array($row->type, self::PHP_ERROR_TYPES, TRUE)) {
          $stats['php_errors'] += (int) $row->count;
        }
        if ($row->type === 'page not found') {
          $stats['page_not_found'] = (int) $row->count;
        }
      }

      // Count by severity.
      $query = $this->database->select('watchdog', 'w');
      $query->addField('w', 'severity');
      $query->addExpression('COUNT(*)', 'count');
      $query->groupBy('w.severity');

      foreach ($query->execute() as $row) {
        $severity_label = self::SEVERITY_LEVELS[$row->severity] ?? 'Unknown';
        $stats['by_severity'][$severity_label] = (int) $row->count;
      }
    }
    catch (\Exception $e) {
      // Table might not exist.
    }

    return $stats;
  }

  /**
   * Gets top PHP errors.
   */
  protected function getPhpErrors(): array {
    $errors = [];

    try {
      $query = $this->database->select('watchdog', 'w');
      $query->fields('w', ['message', 'variables', 'severity']);
      $query->addExpression('COUNT(*)', 'count');
      $query->condition('w.type', 'php');
      $query->groupBy('w.message');
      $query->groupBy('w.variables');
      $query->groupBy('w.severity');
      $query->orderBy('count', 'DESC');
      $query->range(0, 20);

      foreach ($query->execute() as $row) {
        $variables = @unserialize($row->variables);
        $message = $row->message;
        if (is_array($variables)) {
          $message = strtr($message, $variables);
        }

        $errors[] = [
          'message' => $this->truncateMessage($message, 200),
          'count' => (int) $row->count,
          'severity' => self::SEVERITY_LEVELS[$row->severity] ?? 'Unknown',
        ];
      }
    }
    catch (\Exception $e) {
      // Table might not exist.
    }

    return $errors;
  }

  /**
   * Gets top 404 errors.
   */
  protected function get404Errors(): array {
    $errors = [];

    try {
      $query = $this->database->select('watchdog', 'w');
      $query->fields('w', ['message', 'variables']);
      $query->addExpression('COUNT(*)', 'count');
      $query->condition('w.type', 'page not found');
      $query->groupBy('w.message');
      $query->groupBy('w.variables');
      $query->orderBy('count', 'DESC');
      $query->range(0, 50);

      foreach ($query->execute() as $row) {
        $variables = @unserialize($row->variables);
        $message = $row->message;
        if (is_array($variables)) {
          $message = strtr($message, $variables);
        }

        $errors[] = [
          'url' => $this->truncateMessage($message, 150),
          'count' => (int) $row->count,
        ];
      }
    }
    catch (\Exception $e) {
      // Table might not exist.
    }

    return $errors;
  }

  /**
   * Gets security-related issues.
   */
  protected function getSecurityIssues(): array {
    $issues = [];

    try {
      $query = $this->database->select('watchdog', 'w');
      $query->fields('w', ['type', 'message', 'variables']);
      $query->addExpression('COUNT(*)', 'count');
      $query->condition('w.type', self::SECURITY_TYPES, 'IN');
      $query->groupBy('w.type');
      $query->groupBy('w.message');
      $query->groupBy('w.variables');
      $query->orderBy('count', 'DESC');
      $query->range(0, 30);

      foreach ($query->execute() as $row) {
        $variables = @unserialize($row->variables);
        $message = $row->message;
        if (is_array($variables)) {
          $message = strtr($message, $variables);
        }

        $issues[] = [
          'type' => $row->type,
          'message' => $this->truncateMessage($message, 150),
          'count' => (int) $row->count,
        ];
      }
    }
    catch (\Exception $e) {
      // Table might not exist.
    }

    return $issues;
  }

  /**
   * Gets log age (oldest and newest entries).
   */
  protected function getLogAge(): array {
    $age = [
      'oldest' => NULL,
      'newest' => NULL,
    ];

    try {
      // Get oldest entry.
      $oldest = $this->database->select('watchdog', 'w')
        ->fields('w', ['timestamp'])
        ->orderBy('timestamp', 'ASC')
        ->range(0, 1)
        ->execute()
        ->fetchField();

      if ($oldest !== FALSE && $oldest !== NULL) {
        $age['oldest'] = date('Y-m-d H:i:s', (int) $oldest);
      }

      // Get newest entry.
      $newest = $this->database->select('watchdog', 'w')
        ->fields('w', ['timestamp'])
        ->orderBy('timestamp', 'DESC')
        ->range(0, 1)
        ->execute()
        ->fetchField();

      if ($newest !== FALSE && $newest !== NULL) {
        $age['newest'] = date('Y-m-d H:i:s', (int) $newest);
      }
    }
    catch (\Exception $e) {
      // Table might not exist.
    }

    return $age;
  }

  /**
   * Truncates a message to a maximum length.
   */
  protected function truncateMessage(string $message, int $max_length): string {
    $message = strip_tags($message);
    if (strlen($message) > $max_length) {
      return substr($message, 0, $max_length - 3) . '...';
    }
    return $message;
  }

  /**
   * Calculates scores for all factors.
   */
  protected function calculateScores(array $stats, float $php_error_percentage, float $not_found_percentage, array $security_issues): array {
    // PHP Errors score.
    $php_score = 100;
    if ($php_error_percentage >= 10) {
      $php_score = 0;
    }
    elseif ($php_error_percentage >= 5) {
      $php_score = 50;
    }
    elseif ($php_error_percentage >= 1) {
      $php_score = 75;
    }
    elseif (($stats['php_errors'] ?? 0) > 0) {
      $php_score = 90;
    }

    // 404 Errors score.
    $not_found_score = 100;
    if ($not_found_percentage >= 20) {
      $not_found_score = 50;
    }
    elseif ($not_found_percentage >= 10) {
      $not_found_score = 70;
    }
    elseif (($stats['page_not_found'] ?? 0) > 100) {
      $not_found_score = 85;
    }

    // Security score.
    $security_score = 100;
    $failed_logins = 0;
    foreach ($security_issues as $issue) {
      if (str_contains($issue['message'] ?? '', 'Login attempt failed')) {
        $failed_logins += $issue['count'] ?? 1;
      }
    }
    if ($failed_logins > 100) {
      $security_score = 50;
    }
    elseif ($failed_logins > 50) {
      $security_score = 70;
    }
    elseif ($failed_logins > 10) {
      $security_score = 85;
    }

    // Log health score (based on error/warning ratio).
    $log_health_score = 100;
    $critical_count = ($stats['by_severity']['Emergency'] ?? 0) +
                      ($stats['by_severity']['Alert'] ?? 0) +
                      ($stats['by_severity']['Critical'] ?? 0) +
                      ($stats['by_severity']['Error'] ?? 0);

    $total = $stats['total'] ?? 0;
    $critical_percentage = $total > 0 ? ($critical_count / $total) * 100 : 0;

    if ($critical_percentage >= 20) {
      $log_health_score = 40;
    }
    elseif ($critical_percentage >= 10) {
      $log_health_score = 60;
    }
    elseif ($critical_percentage >= 5) {
      $log_health_score = 80;
    }

    return [
      'factors' => [
        'php_errors' => [
          'score' => $php_score,
          'weight' => self::SCORE_WEIGHTS['php_errors'],
          'label' => (string) $this->t('PHP Errors'),
          'description' => $php_score === 100
            ? (string) $this->t('No PHP errors in logs')
            : (string) $this->t('@percent% of log entries are PHP errors', ['@percent' => round($php_error_percentage, 1)]),
        ],
        'not_found' => [
          'score' => $not_found_score,
          'weight' => self::SCORE_WEIGHTS['not_found'],
          'label' => (string) $this->t('404 Errors'),
          'description' => $not_found_score === 100
            ? (string) $this->t('Minimal 404 errors')
            : (string) $this->t('@percent% of log entries are 404 errors', ['@percent' => round($not_found_percentage, 1)]),
        ],
        'security' => [
          'score' => $security_score,
          'weight' => self::SCORE_WEIGHTS['security'],
          'label' => (string) $this->t('Security'),
          'description' => $security_score === 100
            ? (string) $this->t('No concerning security events')
            : (string) $this->t('@count failed login attempts detected', ['@count' => $failed_logins]),
        ],
        'log_health' => [
          'score' => $log_health_score,
          'weight' => self::SCORE_WEIGHTS['log_health'],
          'label' => (string) $this->t('Log Health'),
          'description' => $log_health_score === 100
            ? (string) $this->t('Healthy log distribution')
            : (string) $this->t('@percent% critical/error entries', ['@percent' => round($critical_percentage, 1)]),
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    return [
      'overview' => [
        'label' => $this->t('Log Overview'),
        'description' => $this->t('Overall statistics about log entries including total entries, PHP errors, 404 errors, and log period.'),
        'file_key' => 'analysis',
        'affects_score' => FALSE,
      ],
      'php_errors' => [
        'label' => $this->t('PHP Errors'),
        'description' => $this->t('Top PHP errors found in the watchdog logs with severity and occurrence count.'),
        'file_key' => 'analysis',
        'affects_score' => TRUE,
        'score_factor_key' => 'php_errors',
      ],
      'not_found' => [
        'label' => $this->t('404 Errors'),
        'description' => $this->t('Most frequently accessed non-existent URLs.'),
        'file_key' => 'analysis',
        'affects_score' => TRUE,
        'score_factor_key' => 'not_found',
      ],
      'severity' => [
        'label' => $this->t('Severity Breakdown'),
        'description' => $this->t('Distribution of log entries by severity level.'),
        'file_key' => 'analysis',
        'affects_score' => TRUE,
        'score_factor_key' => 'log_health',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    return match ($check_id) {
      'overview' => $this->getOverviewContent($data),
      'php_errors' => $this->getPhpErrorsContent($data),
      'not_found' => $this->getNotFoundContent($data),
      'severity' => $this->getSeverityContent($data),
      default => [],
    };
  }

  /**
   * Gets the overview content (without section wrapper).
   */
  protected function getOverviewContent(array $data): array {
    $stats = $data['statistics'] ?? [];
    $log_age = $data['log_age'] ?? [];

    $overview_headers = [
      $this->ui->header((string) $this->t('Metric')),
      $this->ui->header((string) $this->t('Value'), 'right'),
    ];

    $overview_rows = [
      $this->ui->row([
        $this->ui->cell((string) $this->t('Total log entries')),
        $this->ui->cell(number_format($stats['total'] ?? 0), ['align' => 'right']),
      ]),
      $this->ui->row([
        $this->ui->cell((string) $this->t('PHP errors')),
        $this->ui->cell(number_format($stats['php_errors'] ?? 0), ['align' => 'right']),
      ]),
      $this->ui->row([
        $this->ui->cell((string) $this->t('404 errors')),
        $this->ui->cell(number_format($stats['page_not_found'] ?? 0), ['align' => 'right']),
      ]),
    ];

    if (!empty($log_age['oldest']) && !empty($log_age['newest'])) {
      $overview_rows[] = $this->ui->row([
        $this->ui->cell((string) $this->t('Log period')),
        $this->ui->cell($log_age['oldest'] . ' - ' . $log_age['newest'], ['align' => 'right']),
      ]);
    }

    return $this->ui->table($overview_headers, $overview_rows);
  }

  /**
   * Gets the PHP errors content (without section wrapper).
   */
  protected function getPhpErrorsContent(array $data): array {
    $php_errors = $data['php_errors'] ?? [];

    if (empty($php_errors)) {
      return ['#markup' => '<p>' . $this->t('No PHP errors found in logs.') . '</p>'];
    }

    $headers = [
      $this->ui->header((string) $this->t('Error Message')),
      $this->ui->header((string) $this->t('Severity')),
      $this->ui->header((string) $this->t('Count'), 'right'),
    ];

    $rows = [];
    foreach ($php_errors as $error) {
      $severity_status = match ($error['severity'] ?? 'Unknown') {
        'Emergency', 'Alert', 'Critical', 'Error' => 'error',
        'Warning' => 'warning',
        default => NULL,
      };

      $rows[] = $this->ui->row([
        $this->ui->cell($error['message'] ?? ''),
        $this->ui->cell($this->ui->badge($error['severity'] ?? 'Unknown', $severity_status ?? 'neutral')),
        $this->ui->cell(number_format($error['count'] ?? 0), ['align' => 'right']),
      ], $severity_status);
    }

    return $this->ui->table($headers, $rows);
  }

  /**
   * Gets the 404 errors content (without section wrapper).
   */
  protected function getNotFoundContent(array $data): array {
    $not_found_errors = $data['not_found_errors'] ?? [];

    if (empty($not_found_errors)) {
      return ['#markup' => '<p>' . $this->t('No 404 errors found in logs.') . '</p>'];
    }

    $headers = [
      $this->ui->header((string) $this->t('URL')),
      $this->ui->header((string) $this->t('Count'), 'right'),
    ];

    $rows = [];
    foreach ($not_found_errors as $error) {
      $rows[] = $this->ui->row([
        $this->ui->cell($error['url'] ?? ''),
        $this->ui->cell(number_format($error['count'] ?? 0), ['align' => 'right']),
      ]);
    }

    return $this->ui->table($headers, $rows);
  }

  /**
   * Gets the severity breakdown content (without section wrapper).
   */
  protected function getSeverityContent(array $data): array {
    $stats = $data['statistics'] ?? [];

    if (empty($stats['by_severity'])) {
      return ['#markup' => '<p>' . $this->t('No severity data available.') . '</p>'];
    }

    $headers = [
      $this->ui->header((string) $this->t('Severity')),
      $this->ui->header((string) $this->t('Count'), 'right'),
      $this->ui->header((string) $this->t('Percentage'), 'right'),
    ];

    $rows = [];
    $total = $stats['total'] ?? 1;
    foreach ($stats['by_severity'] as $severity => $count) {
      $percentage = round(($count / $total) * 100, 1);
      $severity_status = match ($severity) {
        'Emergency', 'Alert', 'Critical', 'Error' => 'error',
        'Warning' => 'warning',
        'Notice' => 'notice',
        default => NULL,
      };

      $rows[] = $this->ui->row([
        $this->ui->cell($this->ui->badge($severity, $severity_status ?? 'neutral')),
        $this->ui->cell(number_format($count), ['align' => 'right']),
        $this->ui->cell($percentage . '%', ['align' => 'right']),
      ], $severity_status);
    }

    return $this->ui->table($headers, $rows);
  }

}
