<?php

declare(strict_types=1);

namespace Drupal\audit;

use Drupal\audit\Service\AuditComponentBuilder;
use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for audit_analyzer plugins.
 */
abstract class AuditAnalyzerBase extends PluginBase implements AuditAnalyzerInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The component builder service.
   */
  protected AuditComponentBuilder $ui;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->ui = $container->get('audit.component_builder');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function label(): string {
    return (string) $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function description(): string {
    return (string) ($this->pluginDefinition['description'] ?? '');
  }

  /**
   * {@inheritdoc}
   */
  public function getOutputDirectory(): string {
    return $this->pluginDefinition['output_directory'] ?? $this->getPluginId();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $config): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function checkRequirements(): array {
    return [];
  }

  /**
   * Creates a standardized result array.
   *
   * @param array $results
   *   Array of result items.
   * @param int $errors
   *   Number of errors.
   * @param int $warnings
   *   Number of warnings.
   * @param int $notices
   *   Number of notices.
   *
   * @return array
   *   Standardized result array with metadata, summary, and results.
   */
  protected function createResult(array $results, int $errors = 0, int $warnings = 0, int $notices = 0): array {
    return [
      'metadata' => [
        'analyzer' => $this->getPluginId(),
        'timestamp' => (new \DateTime())->format(\DateTime::ATOM),
      ],
      'summary' => [
        'errors' => $errors,
        'warnings' => $warnings,
        'notices' => $notices,
      ],
      'results' => $results,
    ];
  }

  /**
   * Creates a result item.
   *
   * @param string $severity
   *   The severity: 'error', 'warning', 'notice', or 'info'.
   * @param string $code
   *   The error code.
   * @param string $message
   *   The message.
   * @param array $details
   *   Additional details.
   *
   * @return array
   *   The result item.
   */
  protected function createResultItem(string $severity, string $code, string $message, array $details = []): array {
    return [
      'severity' => $severity,
      'code' => $code,
      'message' => $message,
      'details' => $details,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildDetailedResults(array $data): array {
    $checks = $this->getAuditChecks();

    // If no checks defined, return empty (analyzer doesn't use this pattern).
    if (empty($checks)) {
      return [];
    }

    $files = $data['_files'] ?? [];
    $score_factors = $data['score']['factors'] ?? [];
    $sections_data = [];

    foreach ($checks as $check_id => $check) {
      // Get file key (default to check_id).
      $file_key = $check['file_key'] ?? $check_id;
      $file_data = $files[$file_key] ?? [];

      // Calculate counters.
      $counters = $this->ui->calculateCounters($file_data);

      // Merge extra file keys if present.
      if (!empty($check['extra_file_keys'])) {
        foreach ($check['extra_file_keys'] as $extra_key) {
          $extra_counters = $this->ui->calculateCounters($files[$extra_key] ?? []);
          $counters['errors'] += $extra_counters['errors'];
          $counters['warnings'] += $extra_counters['warnings'];
          $counters['notices'] += $extra_counters['notices'];
          $counters['info'] += $extra_counters['info'];
          $counters['total'] += $extra_counters['total'];
        }
      }

      // Get score data if this check affects score.
      $score_factor_key = $check['score_factor_key'] ?? $check_id;
      $score_data = NULL;
      if ($check['affects_score'] ?? FALSE) {
        $score_data = $score_factor_key ? ($score_factors[$score_factor_key] ?? NULL) : NULL;
      }

      // Build content using buildCheckContent().
      $content = $this->buildCheckContent($check_id, $data);

      $sections_data[] = [
        'check_id' => $check_id,
        'check' => $check,
        'counters' => $counters,
        'score_data' => $score_data,
        'content' => $content,
      ];
    }

    // Sort sections: affects_score first, then by errors > warnings > notices.
    usort($sections_data, function ($a, $b) {
      // First: sections that affect score come first.
      $a_affects_score = $a['check']['affects_score'] ?? FALSE;
      $b_affects_score = $b['check']['affects_score'] ?? FALSE;
      if ($a_affects_score !== $b_affects_score) {
        return $b_affects_score <=> $a_affects_score;
      }
      // Then by errors.
      if ($a['counters']['errors'] !== $b['counters']['errors']) {
        return $b['counters']['errors'] - $a['counters']['errors'];
      }
      // Then by warnings.
      if ($a['counters']['warnings'] !== $b['counters']['warnings']) {
        return $b['counters']['warnings'] - $a['counters']['warnings'];
      }
      // Then by notices.
      if ($a['counters']['notices'] !== $b['counters']['notices']) {
        return $b['counters']['notices'] - $a['counters']['notices'];
      }
      return $b['counters']['total'] - $a['counters']['total'];
    });

    // Build the sections.
    $build = [];
    foreach ($sections_data as $section_data) {
      $build[$section_data['check_id']] = $this->buildCheckSection($section_data);
    }

    return $build;
  }

  /**
   * Builds a check section with score, counters, and content.
   *
   * @param array $section_data
   *   Section data with check, counters, score_data, and content.
   *
   * @return array
   *   Render array for the section.
   */
  protected function buildCheckSection(array $section_data): array {
    $check = $section_data['check'];
    $counters = $section_data['counters'];
    $score_data = $section_data['score_data'];
    $content = $section_data['content'];

    // Determine severity for section styling.
    $severity = 'success';
    if ($counters['errors'] > 0) {
      $severity = 'error';
    }
    elseif ($counters['warnings'] > 0) {
      $severity = 'warning';
    }
    elseif ($counters['notices'] > 0) {
      $severity = 'notice';
    }

    // Score circle (if affects_score) - shown BEFORE title.
    $score = NULL;
    if (($check['affects_score'] ?? FALSE) && $score_data !== NULL) {
      $score = $this->ui->score($score_data['score'] ?? 0, '', 'small');
    }

    // Counters badges - shown at FAR RIGHT of title line.
    $counters_render = [
      '#markup' => $this->ui->counterBadges($counters),
    ];

    // File types badges - shown INSIDE, below description.
    $file_types_render = NULL;
    if (!empty($check['file_types'])) {
      $file_types_html = '';
      foreach ($check['file_types'] as $type) {
        $file_types_html .= '<span class="audit-file-type audit-file-type--' . htmlspecialchars($type, ENT_QUOTES, 'UTF-8') . '">' . htmlspecialchars($type, ENT_QUOTES, 'UTF-8') . '</span> ';
      }
      $file_types_render = [
        '#markup' => $file_types_html,
      ];
    }

    return $this->ui->section(
      (string) ($check['label'] ?? ''),
      $content,
      [
        'severity' => $severity,
        'description' => (string) ($check['description'] ?? ''),
        'open' => FALSE,
        'score' => $score,
        'counters' => $counters_render,
        'file_types' => $file_types_render,
      ]
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildCheckContent(string $check_id, array $data): array {
    // Default implementation returns empty array.
    // Subclasses should override this to provide check-specific content.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function processConfigurationValue(string $key, mixed $value): mixed {
    // Default implementation: return value as-is.
    return $value;
  }

  /**
   * {@inheritdoc}
   */
  public function getAuditChecks(): array {
    // Default implementation returns empty array.
    // Subclasses should override this to declare their checks.
    return [];
  }

}
