<?php

declare(strict_types=1);

namespace Drupal\audit;

use Drupal\Component\Plugin\PluginInspectionInterface;

/**
 * Interface for audit_analyzer plugins.
 */
interface AuditAnalyzerInterface extends PluginInspectionInterface {

  /**
   * Returns the plugin label.
   *
   * @return string
   *   The plugin label.
   */
  public function label(): string;

  /**
   * Returns the plugin description.
   *
   * @return string
   *   The plugin description.
   */
  public function description(): string;

  /**
   * Returns the output directory for this analyzer.
   *
   * @return string
   *   The directory name (e.g., 'status', 'modules').
   */
  public function getOutputDirectory(): string;

  /**
   * Executes the analysis.
   *
   * @return array
   *   Array with 'summary' and 'results' keys.
   */
  public function analyze(): array;

  /**
   * Returns the configuration form elements for this analyzer.
   *
   * @param array $config
   *   The current configuration values.
   *
   * @return array
   *   Form elements array, or empty if no configuration needed.
   */
  public function buildConfigurationForm(array $config): array;

  /**
   * Validates the analyzer requirements.
   *
   * @return array
   *   Array of warning messages if requirements not met, empty otherwise.
   */
  public function checkRequirements(): array;

  /**
   * Builds the detailed results render array for this analyzer.
   *
   * This method is called by the controller to render analyzer-specific
   * content below the common header (score indicator and issues summary).
   *
   * @param array $data
   *   The analysis data loaded from the JSON file.
   *
   * @return array
   *   A render array with the analyzer-specific detailed results.
   */
  public function buildDetailedResults(array $data): array;

  /**
   * Processes configuration form values before saving.
   *
   * This method allows analyzers to transform form values before they are
   * saved to configuration. Useful for handling complex field types or
   * converting indices back to machine names.
   *
   * @param string $key
   *   The configuration key being saved.
   * @param mixed $value
   *   The form value.
   *
   * @return mixed
   *   The processed value to save.
   */
  public function processConfigurationValue(string $key, mixed $value): mixed;

  /**
   * Returns the audit checks performed by this analyzer.
   *
   * Each analyzer should define the specific checks it performs. This allows
   * other modules to discover what a given analyzer inspects, and enables
   * the detail page to show each check as a section with its score and results.
   *
   * @return array
   *   Array of check definitions, keyed by check machine name. Each check:
   *   - label: (string|\Drupal\Core\StringTranslation\TranslatableMarkup)
   *     Human-readable title for the check.
   *   - description: (string|\Drupal\Core\StringTranslation\TranslatableMarkup)
   *     Description of what this check analyzes.
   *   - file_types: (array) File types affected by this check. Possible values:
   *     'twig', 'php', 'js', 'yml', 'css', 'config' (for Drupal configuration),
   *     or empty array if it doesn't analyze files directly.
   *   - affects_score: (bool) Whether this check contributes to the total score.
   *     If FALSE, the check is informational only and won't appear in the
   *     score factors displayed in the header.
   *   - weight: (int) The weight/multiplier for score calculation (0-100).
   *     Only used when affects_score is TRUE. Higher weight means this check
   *     has more impact on the final score.
   *   - score_factor_key: (string|null) The key in score['factors'] for this
   *     check's score. If NULL, uses check_id. If FALSE, check has no score.
   *   - file_key: (string|null) The key in _files for this check's data.
   *     If NULL, uses check_id.
   *   - extra_file_keys: (array) Additional file keys whose counters should be
   *     merged into this check's counters.
   */
  public function getAuditChecks(): array;

  /**
   * Builds the content for a specific audit check.
   *
   * This method is called by the base class's buildDetailedResults() to get
   * the specific content for each check section. The base class handles
   * wrapping this content in a section with scores, counters, and badges.
   *
   * @param string $check_id
   *   The check machine name (key from getAuditChecks()).
   * @param array $data
   *   The full analysis data including '_files' and 'score'.
   *
   * @return array
   *   A render array with the check-specific content (tables, issue lists, etc).
   *   Return empty array if no content for this check.
   */
  public function buildCheckContent(string $check_id, array $data): array;

}
