<?php

declare(strict_types=1);

namespace Drupal\audit\Controller;

use Drupal\audit\AuditAnalyzerPluginManager;
use Drupal\audit\Service\AuditComponentBuilder;
use Drupal\audit\Service\AuditRunner;
use Drupal\audit\Service\AuditScoreStorage;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for displaying audit results.
 *
 * Regenerates analysis on each detail page request (no caching).
 * Uses component-based UI for consistent rendering.
 */
class AuditResultsController extends ControllerBase {

  /**
   * Constructs an AuditResultsController.
   *
   * @param \Drupal\audit\AuditAnalyzerPluginManager $pluginManager
   *   The plugin manager.
   * @param \Drupal\audit\Service\AuditRunner $runner
   *   The audit runner service.
   * @param \Drupal\audit\Service\AuditComponentBuilder $componentBuilder
   *   The component builder service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter.
   * @param \Drupal\audit\Service\AuditScoreStorage $scoreStorage
   *   The score storage service.
   */
  public function __construct(
    protected readonly AuditAnalyzerPluginManager $pluginManager,
    protected readonly AuditRunner $runner,
    protected readonly AuditComponentBuilder $componentBuilder,
    protected readonly DateFormatterInterface $dateFormatter,
    protected readonly AuditScoreStorage $scoreStorage,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('plugin.manager.audit_analyzer'),
      $container->get('audit.runner'),
      $container->get('audit.component_builder'),
      $container->get('date.formatter'),
      $container->get('audit.score_storage'),
    );
  }

  /**
   * Displays a list of all available audit analyzers with their status.
   *
   * @return array
   *   A render array.
   */
  public function list(): array {
    $build = [];

    $definitions = $this->pluginManager->getDefinitions();

    if (empty($definitions)) {
      $build['empty'] = [
        '#markup' => '<p>' . $this->t('No audit analyzers are available. Enable audit submodules to add analyzers.') . '</p>',
      ];
      return $build;
    }

    // Get all stored scores for the list view.
    $stored_scores = $this->runner->getAllStoredScores();

    $header = [
      'analyzer' => $this->t('Audit type'),
      'status' => $this->t('Status'),
      'scores' => $this->t('Scores'),
      'summary' => $this->t('Summary'),
      'last_run' => $this->t('Last run'),
      'operations' => '',
    ];

    $rows = [];

    foreach ($definitions as $id => $definition) {
      $score_data = $stored_scores[$id] ?? NULL;
      // Use menu_title for display, fallback to label if not set.
      $title = $definition['menu_title'] ?? $definition['label'] ?? $id;

      $rows[$id] = [
        'analyzer' => [
          'data' => [
            '#type' => 'inline_template',
            '#template' => '<strong>{{ label }}</strong>{% if description %}<br><small class="description">{{ description }}</small>{% endif %}',
            '#context' => [
              'label' => $title,
              'description' => $definition['description'] ?? '',
            ],
          ],
        ],
        'status' => ['data' => $this->buildStatusCell($score_data)],
        'scores' => ['data' => $this->buildScoresCell($score_data)],
        'summary' => ['data' => $this->buildSummaryText($score_data)],
        'last_run' => $this->buildLastRunText($score_data),
        'operations' => ['data' => $this->buildOperations($id)],
      ];
    }

    $build['actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-list-actions']],
      '#weight' => -10,
    ];

    // Get stored project score.
    $project_score_data = $this->scoreStorage->getProjectScore();
    if ($project_score_data !== NULL && isset($project_score_data['score'])) {
      $build['actions']['project_score'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['audit-project-score']],
        'label' => [
          '#markup' => '<span class="audit-project-score-label">' . $this->t('Project Score:') . '</span>',
        ],
        'score' => $this->componentBuilder->score($project_score_data['score'], '', 'medium'),
      ];
    }

    $build['actions']['run_all'] = [
      '#type' => 'link',
      '#title' => $this->t('Run all audits'),
      '#url' => Url::fromRoute('audit.run_all'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
      ],
    ];

    $build['analyzers'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No audit analyzers available.'),
      '#attributes' => ['class' => ['audit-analyzers-table']],
    ];

    $build['#attached']['library'][] = 'audit/admin';

    return $build;
  }

  /**
   * Starts the batch process to run all audits.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response.
   */
  public function runAll(): \Symfony\Component\HttpFoundation\RedirectResponse {
    $definitions = $this->pluginManager->getDefinitions();

    $operations = [];
    foreach ($definitions as $id => $definition) {
      $operations[] = [
        [static::class, 'runAnalyzerBatch'],
        [$id, (string) ($definition['menu_title'] ?? $definition['label'] ?? $id)],
      ];
    }

    $batch = [
      'title' => $this->t('Running all audits'),
      'operations' => $operations,
      'finished' => [static::class, 'batchFinished'],
      'progress_message' => $this->t('Processing @current of @total audits.'),
    ];

    batch_set($batch);

    return batch_process(Url::fromRoute('audit.reports'));
  }

  /**
   * Batch operation callback to run a single analyzer.
   *
   * @param string $analyzer_id
   *   The analyzer plugin ID.
   * @param string $label
   *   The analyzer label for display.
   * @param array $context
   *   The batch context.
   */
  public static function runAnalyzerBatch(string $analyzer_id, string $label, array &$context): void {
    $runner = \Drupal::service('audit.runner');
    $result = $runner->runAnalyzer($analyzer_id, TRUE);

    $context['results'][] = [
      'id' => $analyzer_id,
      'label' => $label,
      'success' => $result !== NULL,
    ];

    $context['message'] = t('Running audit: @label', ['@label' => $label]);
  }

  /**
   * Batch finished callback.
   *
   * @param bool $success
   *   Whether the batch completed successfully.
   * @param array $results
   *   The results from each operation.
   * @param array $operations
   *   Any unprocessed operations.
   */
  public static function batchFinished(bool $success, array $results, array $operations): void {
    $messenger = \Drupal::messenger();

    if ($success) {
      $successful = array_filter($results, fn($r) => $r['success']);
      $failed = array_filter($results, fn($r) => !$r['success']);

      $messenger->addStatus(t('Successfully completed @count audits.', [
        '@count' => count($successful),
      ]));

      if (!empty($failed)) {
        $failed_labels = array_map(fn($r) => $r['label'], $failed);
        $messenger->addWarning(t('Failed audits: @labels', [
          '@labels' => implode(', ', $failed_labels),
        ]));
      }
    }
    else {
      $messenger->addError(t('An error occurred while running the audits.'));
    }
  }

  /**
   * Builds summary text from stored score data.
   *
   * @param array|null $score_data
   *   The stored score data, or NULL if never run.
   *
   * @return array
   *   A render array for the summary text.
   */
  protected function buildSummaryText(?array $score_data): array {
    if ($score_data === NULL) {
      return [
        '#type' => 'inline_template',
        '#template' => '<em class="audit-no-data">{{ message }}</em>',
        '#context' => ['message' => $this->t('No report has been generated for this audit.')],
      ];
    }

    $errors = $score_data['errors'] ?? 0;
    $warnings = $score_data['warnings'] ?? 0;
    $notices = $score_data['notices'] ?? 0;

    // If no issues, show OK status.
    if ($errors === 0 && $warnings === 0 && $notices === 0) {
      return [
        '#type' => 'inline_template',
        '#template' => '<span class="audit-summary-badge audit-summary-badge--ok">{{ label }}</span>',
        '#context' => ['label' => $this->t('No issues')],
      ];
    }

    // Build stacked badges.
    $build = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-summary-stack']],
    ];

    if ($errors > 0) {
      $build['errors'] = [
        '#type' => 'inline_template',
        '#template' => '<span class="audit-summary-badge audit-summary-badge--error">{{ label }}</span>',
        '#context' => ['label' => $this->formatPlural($errors, '1 error', '@count errors')],
      ];
    }
    if ($warnings > 0) {
      $build['warnings'] = [
        '#type' => 'inline_template',
        '#template' => '<span class="audit-summary-badge audit-summary-badge--warning">{{ label }}</span>',
        '#context' => ['label' => $this->formatPlural($warnings, '1 warning', '@count warnings')],
      ];
    }
    if ($notices > 0) {
      $build['notices'] = [
        '#type' => 'inline_template',
        '#template' => '<span class="audit-summary-badge audit-summary-badge--notice">{{ label }}</span>',
        '#context' => ['label' => $this->formatPlural($notices, '1 notice', '@count notices')],
      ];
    }

    return $build;
  }

  /**
   * Builds the last run text.
   *
   * @param array|null $score_data
   *   The stored score data, or NULL if never run.
   *
   * @return string
   *   The formatted last run date or '-'.
   */
  protected function buildLastRunText(?array $score_data): string {
    if ($score_data === NULL || empty($score_data['timestamp'])) {
      return '-';
    }

    return $this->dateFormatter->format($score_data['timestamp'], 'custom', 'd/m/Y - H:i');
  }

  /**
   * Builds operation links for an analyzer.
   *
   * @param string $analyzer_id
   *   The analyzer plugin ID.
   *
   * @return array
   *   A render array for the operations.
   */
  protected function buildOperations(string $analyzer_id): array {
    return [
      '#type' => 'operations',
      '#links' => [
        'view' => [
          'title' => $this->t('Run & View'),
          'url' => Url::fromRoute('audit.reports.detail', ['analyzer_id' => $analyzer_id]),
        ],
      ],
    ];
  }

  /**
   * Builds the status cell with total score for an analyzer.
   *
   * @param array|null $score_data
   *   The stored score data, or NULL if never run.
   *
   * @return array
   *   A render array for the status cell.
   */
  protected function buildStatusCell(?array $score_data): array {
    if ($score_data === NULL) {
      return [
        '#markup' => '<span class="audit-no-scores">-</span>',
      ];
    }

    $total_score = $score_data['total_score'] ?? NULL;
    if ($total_score === NULL) {
      return [
        '#markup' => '<span class="audit-no-scores">-</span>',
      ];
    }

    return $this->componentBuilder->score($total_score, '', 'small');
  }

  /**
   * Builds the scores cell with mini score circles for all factors.
   *
   * @param array|null $score_data
   *   The stored score data, or NULL if never run.
   *
   * @return array
   *   A render array for the scores cell.
   */
  protected function buildScoresCell(?array $score_data): array {
    if ($score_data === NULL) {
      return [
        '#markup' => '<span class="audit-no-scores">-</span>',
      ];
    }

    $factors = $score_data['factors'] ?? [];
    if (empty($factors)) {
      return [
        '#markup' => '<span class="audit-no-scores">-</span>',
      ];
    }

    $build = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-scores-list']],
    ];

    foreach ($factors as $factor_id => $factor) {
      $score = $factor['score'] ?? 0;
      $label = $factor['label'] ?? $factor_id;

      $build[$factor_id] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['audit-score-item'],
          'title' => $label . ': ' . ($factor['description'] ?? ''),
        ],
        'circle' => $this->componentBuilder->score($score, '', 'mini'),
      ];
    }

    return $build;
  }

  /**
   * Displays detailed results for a specific analyzer.
   *
   * Regenerates the analysis on each request (no caching).
   *
   * @param string $analyzer_id
   *   The analyzer plugin ID.
   *
   * @return array
   *   A render array.
   */
  public function detail(string $analyzer_id): array {
    $build = [];

    $definitions = $this->pluginManager->getDefinitions();
    if (!isset($definitions[$analyzer_id])) {
      $build['error'] = [
        '#markup' => '<p>' . $this->t('Analyzer not found: @id', ['@id' => $analyzer_id]) . '</p>',
      ];
      return $build;
    }

    $definition = $definitions[$analyzer_id];

    // Run the analyzer fresh on each request.
    $results = $this->runner->runAnalyzer($analyzer_id, TRUE);

    // Build the header section.
    $build['header'] = $this->buildDetailHeader($definition, $results);

    if ($results === NULL) {
      $build['error'] = $this->componentBuilder->message(
        (string) $this->t('The audit could not be executed. Check the logs for more information.'),
        'error'
      );
    }
    else {
      // Check for requirements warnings.
      if (!empty($results['metadata']['requirements_warnings'])) {
        $build['requirements_warning'] = [
          '#theme' => 'status_messages',
          '#message_list' => [
            'warning' => $results['metadata']['requirements_warnings'],
          ],
        ];
      }

      // Build analyzer-specific detailed results.
      $build['detailed_results'] = $this->buildAnalyzerDetailedResults($analyzer_id, $results);
    }

    $build['#attached']['library'][] = 'audit/admin';

    // Disable caching for this page.
    $build['#cache'] = [
      'max-age' => 0,
    ];

    return $build;
  }

  /**
   * Builds the detail page header with description and timestamp.
   *
   * @param array $definition
   *   The analyzer plugin definition.
   * @param array|null $results
   *   The analysis results, or NULL if failed.
   *
   * @return array
   *   A render array for the header.
   */
  protected function buildDetailHeader(array $definition, ?array $results): array {
    $header = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-detail-header']],
      '#weight' => -10,
    ];

    if ($results !== NULL && !empty($results['metadata']['timestamp'])) {
      $timestamp = strtotime($results['metadata']['timestamp']);
      $formatted_date = $timestamp ? $this->dateFormatter->format($timestamp, 'custom', 'd/m/Y - H:i:s') : $results['metadata']['timestamp'];

      $header['last_run'] = [
        '#markup' => '<p class="audit-last-run"><strong>' . $this->t('Generated:') . '</strong> ' . $formatted_date . '</p>',
      ];
    }

    if (!empty($definition['description'])) {
      $header['description'] = [
        '#markup' => '<p class="audit-description">' . $definition['description'] . '</p>',
      ];
    }

    return $header;
  }

  /**
   * Builds the analyzer-specific detailed results.
   *
   * @param string $analyzer_id
   *   The analyzer plugin ID.
   * @param array $data
   *   The analysis data.
   *
   * @return array
   *   A render array for the detailed results.
   */
  protected function buildAnalyzerDetailedResults(string $analyzer_id, array $data): array {
    try {
      $plugin = $this->pluginManager->createInstance($analyzer_id);
      return $plugin->buildDetailedResults($data);
    }
    catch (\Exception $e) {
      return [];
    }
  }

  /**
   * Returns the title for the detail page.
   *
   * @param string $analyzer_id
   *   The analyzer plugin ID.
   *
   * @return string
   *   The page title.
   */
  public function detailTitle(string $analyzer_id): string {
    $definitions = $this->pluginManager->getDefinitions();
    if (isset($definitions[$analyzer_id])) {
      return (string) ($definitions[$analyzer_id]['label'] ?? $analyzer_id);
    }
    return $analyzer_id;
  }

}
