<?php

declare(strict_types=1);

namespace Drupal\audit\Form;

use Drupal\audit\AuditAnalyzerPluginManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Audit module settings.
 */
class AuditSettingsForm extends ConfigFormBase {

  /**
   * The config name.
   */
  protected const SETTINGS = 'audit.settings';

  /**
   * Constructs an AuditSettingsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\audit\AuditAnalyzerPluginManager $pluginManager
   *   The plugin manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    protected readonly AuditAnalyzerPluginManager $pluginManager,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('plugin.manager.audit_analyzer'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'audit_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    $names = [static::SETTINGS];

    foreach ($this->pluginManager->getDefinitions() as $definition) {
      $provider = $definition['provider'] ?? NULL;
      if ($provider && $provider !== 'audit') {
        $names[] = $provider . '.settings';
      }
    }

    return $names;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config(static::SETTINGS);

    // External dashboard section (first, most prominent).
    $form['external_dashboard'] = [
      '#type' => 'details',
      '#title' => $this->t('External Dashboard'),
      '#open' => (bool) $config->get('external_dashboard.enabled'),
      '#weight' => -100,
    ];

    $form['external_dashboard']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Send data to external dashboard'),
      '#default_value' => $config->get('external_dashboard.enabled'),
    ];

    $form['external_dashboard']['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#default_value' => $config->get('external_dashboard.api_key'),
      '#states' => [
        'visible' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
        'required' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // General settings section (collapsed by default).
    $form['general'] = [
      '#type' => 'details',
      '#title' => $this->t('General Settings'),
      '#open' => FALSE,
      '#weight' => -90,
    ];

    $form['general']['user_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Primary user type'),
      '#description' => $this->t('Select the primary type of users visiting your site. This helps tailor recommendations for caching, performance, and security.'),
      '#options' => [
        'anonymous' => $this->t('Mostly anonymous users (e.g., news sites, blogs)'),
        'registered' => $this->t('Mostly registered users (e.g., intranets, membership sites)'),
        'mixed' => $this->t('Mixed anonymous and registered (e.g., forums, e-commerce)'),
      ],
      '#default_value' => $config->get('user_type') ?? 'mixed',
    ];

    $form['general']['scan_directories'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Directories to scan'),
      '#description' => $this->t('Enter one directory path per line, relative to the Drupal root. These directories will be scanned by analyzers that perform code analysis (Performance, Twig, PHPStan, PHPCS, Rector, etc.).'),
      '#default_value' => $config->get('scan_directories') ?? "web/modules/custom\nweb/themes/custom",
      '#rows' => 4,
    ];

    $form['general']['exclude_patterns'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Exclude patterns'),
      '#description' => $this->t('Enter one pattern per line. Files or directories matching these patterns will be excluded from scanning (e.g., *Test.php, tests/, node_modules/).'),
      '#default_value' => $config->get('exclude_patterns') ?? "*Test.php\n*TestBase.php\ntests/\nnode_modules/",
      '#rows' => 4,
    ];

    // Score multipliers details (collapsible).
    $form['general']['score_multipliers'] = [
      '#type' => 'details',
      '#title' => $this->t('Project Score Multipliers'),
      '#description' => $this->t('These multipliers determine how each audit contributes to the <strong>overall Project Score</strong> displayed on the audit overview page. The Project Score is calculated as a weighted average: audits with higher multipliers have more influence on the final score. For example, if "Updates" has a multiplier of 5 (Critical) and "Twig" has a multiplier of 2 (Low), security updates will affect the Project Score 2.5 times more than Twig issues. Set a multiplier to 0 to completely exclude that audit from the Project Score calculation.'),
      '#open' => FALSE,
    ];

    $definitions = $this->pluginManager->getDefinitions();
    $multiplier_options = [
      0 => $this->t('0 - Excluded'),
      1 => $this->t('1 - Low'),
      2 => $this->t('2 - Below Average'),
      3 => $this->t('3 - Average'),
      4 => $this->t('4 - Above Average'),
      5 => $this->t('5 - Critical'),
    ];

    foreach ($definitions as $id => $definition) {
      $default_weight = $definition['weight'] ?? 3;
      $form['general']['score_multipliers']['multiplier_' . $id] = [
        '#type' => 'select',
        '#title' => $definition['label'],
        '#options' => $multiplier_options,
        '#default_value' => $config->get('multipliers.' . $id) ?? $default_weight,
      ];
    }

    // Submodules configuration container.
    $form['submodules'] = [
      '#type' => 'details',
      '#title' => $this->t('Audit Submodules Settings'),
      '#description' => $this->t('Settings for each audit submodule. Adjust parameters and enable or disable specific checks.'),
      '#open' => TRUE,
      '#weight' => -80,
      '#tree' => TRUE,
    ];

    foreach ($definitions as $id => $definition) {
      $provider = $definition['provider'] ?? 'audit';
      $analyzer = $this->pluginManager->createInstance($id);
      $analyzer_config = $this->getAnalyzerConfig($provider);

      $config_form = $analyzer->buildConfigurationForm($analyzer_config);

      // Skip submodules without configuration options.
      if (empty($config_form)) {
        continue;
      }

      $form['submodules'][$id] = [
        '#type' => 'details',
        '#title' => $definition['label'],
        '#open' => FALSE,
      ];

      $warnings = $analyzer->checkRequirements();
      if (!empty($warnings)) {
        $form['submodules'][$id]['warnings'] = [
          '#theme' => 'status_messages',
          '#message_list' => ['warning' => $warnings],
        ];
      }

      foreach ($config_form as $key => $element) {
        $form['submodules'][$id][$provider . '_' . $key] = $element;
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    if ($form_state->getValue('enabled') && empty($form_state->getValue('api_key'))) {
      $form_state->setErrorByName('api_key', $this->t('API Key is required when external dashboard is enabled.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config(static::SETTINGS);

    $config
      ->set('external_dashboard.enabled', (bool) $form_state->getValue('enabled'))
      ->set('external_dashboard.api_key', $form_state->getValue('api_key'))
      ->set('user_type', $form_state->getValue('user_type'))
      ->set('scan_directories', $form_state->getValue('scan_directories'))
      ->set('exclude_patterns', $form_state->getValue('exclude_patterns'));

    // Save score multipliers.
    $definitions = $this->pluginManager->getDefinitions();
    $general_values = $form_state->getValue('general') ?? [];
    $multipliers_values = $general_values['score_multipliers'] ?? [];

    foreach ($definitions as $id => $definition) {
      $multiplier_key = 'multiplier_' . $id;
      if (isset($multipliers_values[$multiplier_key])) {
        $config->set('multipliers.' . $id, (int) $multipliers_values[$multiplier_key]);
      }
    }

    $config->save();

    $definitions = $this->pluginManager->getDefinitions();
    $saved_providers = [];
    $submodules_values = $form_state->getValue('submodules') ?? [];

    foreach ($definitions as $id => $definition) {
      $provider = $definition['provider'] ?? 'audit';

      if (isset($saved_providers[$provider]) || $provider === 'audit') {
        continue;
      }

      $analyzer = $this->pluginManager->createInstance($id);
      $config_form = $analyzer->buildConfigurationForm([]);

      if (!empty($config_form)) {
        $provider_config = $this->configFactory->getEditable($provider . '.settings');
        $analyzer_values = $submodules_values[$id] ?? [];

        foreach ($config_form as $key => $element) {
          $form_key = $provider . '_' . $key;
          $value = $analyzer_values[$form_key] ?? NULL;
          if ($value !== NULL) {
            // Allow analyzer to process value before saving.
            $value = $analyzer->processConfigurationValue($key, $value);
            $provider_config->set($key, $value);
          }
        }

        $provider_config->save();
        $saved_providers[$provider] = TRUE;
      }
    }

    parent::submitForm($form, $form_state);
  }

  /**
   * Gets the configuration array for an analyzer's provider.
   *
   * @param string $provider
   *   The provider module name.
   *
   * @return array
   *   The configuration array.
   */
  protected function getAnalyzerConfig(string $provider): array {
    if ($provider === 'audit') {
      return [];
    }

    $config = $this->config($provider . '.settings');
    return $config->get() ?? [];
  }

}
