<?php

declare(strict_types=1);

namespace Drupal\audit\Plugin\Derivative;

use Drupal\audit\AuditAnalyzerPluginManager;
use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Derives secondary local task plugins for audit analyzers.
 *
 * Creates a secondary tab for each available audit analyzer plugin.
 * These tabs appear under the "Overview" primary tab.
 */
class AuditAnalyzerLocalTasks extends DeriverBase implements ContainerDeriverInterface {

  use StringTranslationTrait;

  /**
   * Constructs an AuditAnalyzerLocalTasks instance.
   *
   * @param \Drupal\audit\AuditAnalyzerPluginManager $pluginManager
   *   The audit analyzer plugin manager.
   */
  public function __construct(
    protected readonly AuditAnalyzerPluginManager $pluginManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id): static {
    return new static(
      $container->get('plugin.manager.audit_analyzer'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition): array {
    $this->derivatives = [];

    $definitions = $this->pluginManager->getDefinitions();
    $weight = 0;

    foreach ($definitions as $id => $definition) {
      // Use menu_title if available, otherwise fall back to label.
      $title = $definition['menu_title'] ?? $definition['label'] ?? $id;

      $this->derivatives[$id] = [
        'title' => $title,
        'route_name' => 'audit.reports.detail',
        'route_parameters' => ['analyzer_id' => $id],
        'parent_id' => 'audit.reports',
        'weight' => $weight++,
      ] + $base_plugin_definition;
    }

    return $this->derivatives;
  }

}
