<?php

declare(strict_types=1);

namespace Drupal\audit\Service;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Service for building audit UI components.
 *
 * Provides methods that return render arrays using audit_* theme hooks.
 * All visual presentation is handled by Twig templates in /templates.
 */
class AuditComponentBuilder {

  use StringTranslationTrait;

  /**
   * Cache for file contents to avoid reading the same file multiple times.
   *
   * @var array<string, array|false>
   */
  protected array $fileCache = [];

  /**
   * Constructs a new AuditComponentBuilder.
   *
   * @param \Drupal\Core\StringTranslation\TranslationInterface $stringTranslation
   *   The string translation service.
   */
  public function __construct(TranslationInterface $stringTranslation) {
    $this->stringTranslation = $stringTranslation;
  }

  /**
   * Creates a collapsible section.
   *
   * @param string $title
   *   Section title.
   * @param array $content
   *   Content render array.
   * @param array $options
   *   Options:
   *   - open: (bool) Whether expanded by default.
   *   - severity: (string) error, warning, notice, success (for border color).
   *   - description: (string) Description shown when expanded.
   *   - score: (array) Render array for score circle (before title).
   *   - counters: (array) Render array for counters badges (far right).
   *   - file_types: (array) Render array for file type badges (inside, below description).
   *
   * @return array
   *   Render array.
   */
  public function section(string $title, array $content, array $options = []): array {
    return [
      '#theme' => 'audit_section',
      '#title' => $title,
      '#content' => $content,
      '#open' => $options['open'] ?? FALSE,
      '#severity' => $options['severity'] ?? NULL,
      '#description' => $options['description'] ?? NULL,
      '#score' => $options['score'] ?? NULL,
      '#counters' => $options['counters'] ?? NULL,
      '#file_types' => $options['file_types'] ?? NULL,
    ];
  }

  /**
   * Creates a data table.
   *
   * @param array $headers
   *   Array of headers. Each can be:
   *   - string: Simple header label.
   *   - array: ['label' => ..., 'align' => ..., 'width' => ...].
   * @param array $rows
   *   Array of rows. Each row can be:
   *   - array of cells (simple).
   *   - ['cells' => [...], 'severity' => ...].
   *   Each cell can be:
   *   - string: Simple content.
   *   - ['content' => ..., 'align' => ..., 'status' => ...].
   * @param array $options
   *   Options:
   *   - empty: (string) Message when no rows.
   *   - caption: (string) Table caption.
   *
   * @return array
   *   Render array.
   */
  public function table(array $headers, array $rows, array $options = []): array {
    return [
      '#theme' => 'audit_table',
      '#headers' => $headers,
      '#rows' => $rows,
      '#empty' => $options['empty'] ?? (string) $this->t('No data available.'),
      '#caption' => $options['caption'] ?? NULL,
    ];
  }

  /**
   * Creates a code block with line numbers.
   *
   * @param array $lines
   *   Array of lines. Each line:
   *   - ['number' => ..., 'content' => ..., 'is_highlight' => ...].
   * @param array $options
   *   Options:
   *   - highlight_line: (int) Line number to highlight.
   *   - severity: (string) error, warning, notice.
   *   - language: (string) Code language.
   *
   * @return array
   *   Render array.
   */
  public function code(array $lines, array $options = []): array {
    return [
      '#theme' => 'audit_code',
      '#lines' => $lines,
      '#highlight_line' => $options['highlight_line'] ?? NULL,
      '#severity' => $options['severity'] ?? 'error',
      '#language' => $options['language'] ?? NULL,
    ];
  }

  /**
   * Creates a score circle.
   *
   * @param int|float $score
   *   Score value (0-100, can be negative).
   * @param string $label
   *   Optional label below circle.
   * @param string $size
   *   Size: large, medium, small, mini.
   *
   * @return array
   *   Render array.
   */
  public function score(int|float $score, string $label = '', string $size = 'large'): array {
    return [
      '#theme' => 'audit_score',
      '#score' => (int) round($score),
      '#label' => $label,
      '#size' => $size,
    ];
  }

  /**
   * Creates a factor card.
   *
   * @param int|float $score
   *   Score value.
   * @param string $label
   *   Factor label.
   * @param string $description
   *   Optional description.
   * @param array $badges
   *   Optional array of badges. Each badge: ['value' => X, 'type' => Y].
   *   Types: error, warning, info, success, neutral.
   *
   * @return array
   *   Render array.
   */
  public function factor(int|float $score, string $label, string $description = '', array $badges = []): array {
    return [
      '#theme' => 'audit_factor',
      '#score' => (int) round($score),
      '#label' => $label,
      '#description' => $description,
      '#badges' => $badges,
    ];
  }

  /**
   * Creates an item with icon and name.
   *
   * @param string $label
   *   Human-readable label.
   * @param array $options
   *   Options:
   *   - icon: (string) check, cross, warning, info, dash.
   *   - machine_name: (string) Machine name.
   *   - description: (string) Description.
   *   - severity: (string) error, warning, notice, success.
   *
   * @return array
   *   Render array.
   */
  public function item(string $label, array $options = []): array {
    return [
      '#theme' => 'audit_item',
      '#label' => $label,
      '#icon' => $options['icon'] ?? NULL,
      '#machine_name' => $options['machine_name'] ?? NULL,
      '#description' => $options['description'] ?? NULL,
      '#severity' => $options['severity'] ?? NULL,
    ];
  }

  /**
   * Creates a status message.
   *
   * @param string $message
   *   Message text.
   * @param string $type
   *   Type: info, success, warning, error.
   *
   * @return array
   *   Render array.
   */
  public function message(string $message, string $type = 'info'): array {
    return [
      '#theme' => 'audit_message',
      '#message' => $message,
      '#type' => $type,
    ];
  }

  /**
   * Creates a score section with multiple factors.
   *
   * @param array $factors
   *   Array of factors with 'score', 'label', 'description', 'badges'.
   *   Each badge in badges array: ['value' => X, 'type' => Y, 'label' => Z].
   *
   * @return array
   *   Render array.
   */
  public function scoreSection(array $factors): array {
    if (empty($factors)) {
      return [];
    }

    $build = [
      '#type' => 'container',
      '#attributes' => ['class' => ['audit-factors']],
    ];

    foreach ($factors as $id => $factor) {
      $build[$id] = $this->factor(
        $factor['score'] ?? 0,
        $factor['label'] ?? '',
        $factor['description'] ?? '',
        $factor['badges'] ?? []
      );
    }

    return $build;
  }

  /**
   * Creates an issue item (collapsible result with code snippet).
   *
   * @param array $options
   *   Options:
   *   - severity: (string) error, warning, notice, or null for info.
   *   - code: (string) Issue code identifier.
   *   - label: (string) Human-readable issue label/message.
   *   - file: (string) File path where issue was found.
   *   - line: (int) Line number in the file.
   *   - description: (string|\Drupal\Component\Render\MarkupInterface)
   *     Detailed description and how to fix.
   *   - code_snippet: (array) Render array for code block.
   *   - tags: (array) Classification tags (security, performance, cache, etc.).
   *   - custom_data: (array) Custom data attributes for filtering (key => value).
   *     Keys should use hyphens, e.g. ['entity-type' => 'node', 'bundle' => 'article'].
   *
   * @return array
   *   Render array.
   */
  public function issue(array $options): array {
    return [
      '#theme' => 'audit_issue',
      '#severity' => $options['severity'] ?? NULL,
      '#code' => $options['code'] ?? '',
      '#label' => $options['label'] ?? '',
      '#file' => $options['file'] ?? '',
      '#line' => $options['line'] ?? NULL,
      '#description' => $options['description'] ?? NULL,
      '#code_snippet' => $options['code_snippet'] ?? NULL,
      '#tags' => $options['tags'] ?? [],
      '#fixable' => $options['fixable'] ?? FALSE,
      '#custom_data' => $options['custom_data'] ?? [],
    ];
  }

  /**
   * Creates a list of issues with optional custom filters.
   *
   * @param array $issues
   *   Array of issue render arrays (from issue() method).
   * @param string|null $empty
   *   Message to display when no issues.
   * @param array $custom_filters
   *   Optional custom filter configuration for this list.
   *   Each filter is keyed by filter_key with an array containing:
   *   - label: (string) Human-readable filter group label.
   *   - attribute: (string) Data attribute name (e.g., 'data-entity-type').
   *   - styled: (bool) Whether to apply styling (default: true with white bg).
   *   - styles: (array) Optional custom styles for values.
   *     Each style is keyed by value with ['label' => ..., 'bg' => ..., 'text' => ...].
   *
   * @return array
   *   Render array.
   */
  public function issueList(array $issues, ?string $empty = NULL, array $custom_filters = []): array {
    $build = [
      '#theme' => 'audit_issue_list',
      '#issues' => $issues,
      '#empty' => $empty ?? (string) $this->t('No issues found.'),
    ];

    if (!empty($custom_filters)) {
      $build['#custom_filters'] = json_encode($custom_filters, JSON_HEX_APOS | JSON_HEX_QUOT);
    }

    return $build;
  }

  // =========================================================================
  // Helper methods for building table cells.
  // =========================================================================

  /**
   * Creates a header definition.
   *
   * @param string $label
   *   Header label.
   * @param string|null $align
   *   Alignment: left, center, right.
   * @param string|null $width
   *   Width (e.g., '100px', '20%').
   *
   * @return array
   *   Header definition.
   */
  public function header(string $label, ?string $align = NULL, ?string $width = NULL): array {
    $h = ['#label' => $label];
    if ($align) {
      $h['#align'] = $align;
    }
    if ($width) {
      $h['#width'] = $width;
    }
    return $h;
  }

  /**
   * Creates a row definition.
   *
   * @param array $cells
   *   Array of cells.
   * @param string|null $severity
   *   Row severity: error, warning, notice.
   *
   * @return array
   *   Row definition.
   */
  public function row(array $cells, ?string $severity = NULL): array {
    $r = ['#cells' => $cells];
    if ($severity) {
      $r['#severity'] = $severity;
    }
    return $r;
  }

  /**
   * Creates a cell definition.
   *
   * @param string $content
   *   Cell content (can contain HTML).
   * @param array $options
   *   Options:
   *   - align: (string) left, center, right.
   *   - status: (string) ok, warning, error.
   *   - nowrap: (bool) Prevent wrapping.
   *
   * @return array
   *   Cell definition.
   */
  public function cell(string $content, array $options = []): array {
    $cell = ['#content' => $content];
    foreach ($options as $key => $value) {
      $cell['#' . $key] = $value;
    }
    return $cell;
  }

  // =========================================================================
  // Helper methods for inline HTML (for use in table cells).
  // =========================================================================

  /**
   * Creates a badge HTML string.
   *
   * @param string $label
   *   Badge label.
   * @param string $variant
   *   Variant: success, warning, error, info, neutral.
   *
   * @return string
   *   HTML string.
   */
  public function badge(string $label, string $variant = 'neutral'): string {
    return '<span class="audit-badge audit-badge--' . htmlspecialchars($variant, ENT_QUOTES, 'UTF-8') . '">'
      . htmlspecialchars($label, ENT_QUOTES, 'UTF-8') . '</span>';
  }

  /**
   * Creates an icon HTML string.
   *
   * @param string $type
   *   Icon type: check, cross, warning, info, dash.
   *
   * @return string
   *   HTML string.
   */
  public function icon(string $type): string {
    $icons = [
      'check' => ['symbol' => '✓', 'label' => $this->t('Yes')],
      'cross' => ['symbol' => '✗', 'label' => $this->t('No')],
      'warning' => ['symbol' => '⚠', 'label' => $this->t('Warning')],
      'info' => ['symbol' => 'ℹ', 'label' => $this->t('Info')],
      'dash' => ['symbol' => '—', 'label' => $this->t('N/A')],
    ];
    $icon = $icons[$type] ?? $icons['dash'];
    return '<span class="audit-icon audit-icon--' . htmlspecialchars($type, ENT_QUOTES, 'UTF-8')
      . '" aria-label="' . $icon['label'] . '">' . $icon['symbol'] . '</span>';
  }

  /**
   * Creates an item name HTML string with optional machine name.
   *
   * @param string $label
   *   Human-readable label.
   * @param string|null $machine_name
   *   Optional machine name.
   *
   * @return string
   *   HTML string.
   */
  public function itemName(string $label, ?string $machine_name = NULL): string {
    $html = '<strong>' . htmlspecialchars($label, ENT_QUOTES, 'UTF-8') . '</strong>';
    if ($machine_name !== NULL && $machine_name !== '') {
      $html .= '<br><code class="audit-machine">' . htmlspecialchars($machine_name, ENT_QUOTES, 'UTF-8') . '</code>';
    }
    return $html;
  }

  /**
   * Formats a number with status color class.
   *
   * @param int|float $value
   *   The value.
   * @param array $thresholds
   *   Thresholds: ['warning' => X, 'error' => Y].
   *
   * @return string
   *   HTML string.
   */
  public function number(int|float $value, array $thresholds = []): string {
    $class = 'audit-num';
    if (!empty($thresholds['error']) && $value >= $thresholds['error']) {
      $class .= ' audit-num--error';
    }
    elseif (!empty($thresholds['warning']) && $value >= $thresholds['warning']) {
      $class .= ' audit-num--warning';
    }
    return '<span class="' . $class . '">' . number_format($value) . '</span>';
  }

  // =========================================================================
  // Code snippet utilities.
  // =========================================================================

  /**
   * Gets file contents with caching to avoid reading the same file twice.
   *
   * @param string $file
   *   The file path.
   *
   * @return array|false
   *   Array of lines, or FALSE if file cannot be read.
   */
  protected function getFileContents(string $file): array|false {
    if (!isset($this->fileCache[$file])) {
      if (!file_exists($file)) {
        $this->fileCache[$file] = FALSE;
      }
      else {
        $contents = @file($file);
        $this->fileCache[$file] = $contents !== FALSE ? $contents : FALSE;
      }
    }
    return $this->fileCache[$file];
  }

  /**
   * Clears the file cache.
   *
   * Call this after processing a batch of files to free memory.
   */
  public function clearFileCache(): void {
    $this->fileCache = [];
  }

  /**
   * Builds a code snippet from a file with context lines around a specific line.
   *
   * @param string $file
   *   The file path.
   * @param int $line
   *   The line number to highlight.
   * @param string $severity
   *   The severity (error, warning, notice).
   * @param int $context
   *   Number of lines of context before and after.
   *
   * @return array
   *   Render array using audit_code theme.
   */
  public function codeSnippet(string $file, int $line, string $severity = 'error', int $context = 3): array {
    $file_contents = $this->getFileContents($file);
    if ($file_contents === FALSE) {
      return [];
    }

    $lines = [];
    $start = max(0, $line - $context - 1);
    $end = min(count($file_contents), $line + $context);

    for ($i = $start; $i < $end; $i++) {
      $line_number = $i + 1;
      $lines[] = [
        'number' => $line_number,
        'content' => rtrim($file_contents[$i] ?? ''),
        'is_highlight' => $line_number === $line,
      ];
    }

    if (empty($lines)) {
      return [];
    }

    return $this->code($lines, [
      'highlight_line' => $line,
      'severity' => $severity,
      'language' => $this->getFileLanguage($file),
    ]);
  }

  /**
   * Builds a code snippet showing the start of a file.
   *
   * Used for file-level issues where there's no specific line number.
   *
   * @param string $file
   *   The file path.
   * @param string $severity
   *   The severity (error, warning, notice).
   * @param int $num_lines
   *   Number of lines to show from the start.
   *
   * @return array
   *   Render array using audit_code theme.
   */
  public function codeSnippetFromStart(string $file, string $severity = 'error', int $num_lines = 15): array {
    $file_contents = $this->getFileContents($file);
    if ($file_contents === FALSE) {
      return [];
    }

    $lines = [];
    $end = min(count($file_contents), $num_lines);

    for ($i = 0; $i < $end; $i++) {
      $lines[] = [
        'number' => $i + 1,
        'content' => rtrim($file_contents[$i] ?? ''),
        'is_highlight' => FALSE,
      ];
    }

    if (empty($lines)) {
      return [];
    }

    return $this->code($lines, [
      'severity' => $severity,
      'language' => $this->getFileLanguage($file),
    ]);
  }

  /**
   * Gets the language identifier for a file based on its extension.
   *
   * @param string $file
   *   The file path.
   *
   * @return string
   *   The language identifier (e.g., 'twig', 'php').
   */
  public function getFileLanguage(string $file): string {
    $extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));

    return match ($extension) {
      'twig' => 'twig',
      'php', 'module', 'inc', 'install', 'theme' => 'php',
      'yml', 'yaml' => 'yaml',
      'js' => 'javascript',
      'css' => 'css',
      'json' => 'json',
      'html', 'htm' => 'html',
      default => 'text',
    };
  }

  // =========================================================================
  // Counter badges.
  // =========================================================================

  /**
   * Builds counter badges for errors, warnings, notices.
   *
   * @param array $counters
   *   Array with 'errors', 'warnings', 'notices', and optionally 'total' counts.
   *
   * @return string
   *   HTML string with counter badges.
   */
  public function counterBadges(array $counters): string {
    $errors = $counters['errors'] ?? 0;
    $warnings = $counters['warnings'] ?? 0;
    $notices = $counters['notices'] ?? 0;
    $total = $counters['total'] ?? ($errors + $warnings + $notices);

    if ($total === 0) {
      return '<span class="audit-summary-badge audit-summary-badge--ok">' . $this->t('OK') . '</span>';
    }

    if ($errors === 0 && $warnings === 0 && $notices === 0) {
      return '<span class="audit-summary-badge audit-summary-badge--ok">' . $this->t('@count analyzed', ['@count' => $total]) . '</span>';
    }

    $badges = [];
    if ($errors > 0) {
      $badges[] = '<span class="audit-summary-badge audit-summary-badge--error">' . $this->formatPlural($errors, '1 error', '@count errors') . '</span>';
    }
    if ($warnings > 0) {
      $badges[] = '<span class="audit-summary-badge audit-summary-badge--warning">' . $this->formatPlural($warnings, '1 warning', '@count warnings') . '</span>';
    }
    if ($notices > 0) {
      $badges[] = '<span class="audit-summary-badge audit-summary-badge--notice">' . $this->formatPlural($notices, '1 notice', '@count notices') . '</span>';
    }

    return implode(' ', $badges);
  }

  /**
   * Formats a count using singular/plural.
   *
   * @param int $count
   *   The count.
   * @param string $singular
   *   Singular form.
   * @param string $plural
   *   Plural form.
   *
   * @return string
   *   Formatted string.
   */
  protected function formatPlural(int $count, string $singular, string $plural): string {
    return $count === 1
      ? str_replace('1', (string) $count, $singular)
      : str_replace('@count', (string) $count, $plural);
  }

  /**
   * Calculates counters (errors, warnings, notices, info, total) from results.
   *
   * @param array $file_data
   *   Data array containing 'results' with severity-typed items.
   *
   * @return array
   *   Array with 'errors', 'warnings', 'notices', 'info', and 'total' counts.
   */
  public function calculateCounters(array $file_data): array {
    $counters = [
      'errors' => 0,
      'warnings' => 0,
      'notices' => 0,
      'info' => 0,
      'total' => 0,
    ];

    $results = $file_data['results'] ?? [];
    foreach ($results as $result) {
      $severity = $result['severity'] ?? 'info';
      $counters['total']++;

      switch ($severity) {
        case 'error':
          $counters['errors']++;
          break;

        case 'warning':
          $counters['warnings']++;
          break;

        case 'notice':
          $counters['notices']++;
          break;

        default:
          $counters['info']++;
          break;
      }
    }

    return $counters;
  }

  // =========================================================================
  // Severity and issue list utilities.
  // =========================================================================

  /**
   * Normalizes severity for display.
   *
   * Converts 'info' to NULL (no severity indicator) for issue display.
   *
   * @param string $severity
   *   The severity string (error, warning, notice, info).
   *
   * @return string|null
   *   The normalized severity, or NULL for info-level items.
   */
  public function normalizeSeverity(string $severity): ?string {
    return $severity === 'info' ? NULL : $severity;
  }

  /**
   * Builds an issue list from results with a mapper callback.
   *
   * This centralizes the common pattern of building issue lists:
   * - Check if results empty → return success message
   * - Map each result to an issue via callback
   * - Return wrapped issue list
   *
   * @param array $results
   *   Array of result items to convert to issues.
   * @param string $success_message
   *   Message to display when results are empty.
   * @param callable $issue_mapper
   *   Callback that receives (array $item, AuditComponentBuilder $ui) and
   *   returns an array of options for $this->issue().
   * @param array $custom_filters
   *   Optional custom filter configuration for the issue list.
   *   Each filter is keyed by filter_key with an array containing:
   *   - label: (string) Human-readable filter group label.
   *   - attribute: (string) Data attribute name (e.g., 'data-entity-type').
   *   - styled: (bool) Whether to apply styling (default: true with white bg).
   *   - styles: (array) Optional custom styles for values.
   *
   * @return array
   *   Render array with either a success message or an issue list.
   */
  public function buildIssueListFromResults(array $results, string $success_message, callable $issue_mapper, array $custom_filters = []): array {
    if (empty($results)) {
      return ['message' => $this->message($success_message, 'success')];
    }

    $issues = [];
    foreach ($results as $item) {
      $issue_options = $issue_mapper($item, $this);
      if (!empty($issue_options)) {
        $issues[] = $this->issue($issue_options);
      }
    }

    if (empty($issues)) {
      return ['message' => $this->message($success_message, 'success')];
    }

    return ['issues' => $this->issueList($issues, NULL, $custom_filters)];
  }

  /**
   * Builds a code snippet for an issue item if file and line are available.
   *
   * Helper method to reduce boilerplate in issue mappers.
   *
   * @param array $details
   *   The item details array with 'file' and 'line' keys.
   * @param string $severity
   *   The severity for highlighting.
   *
   * @return array|null
   *   Code snippet render array, or NULL if not applicable.
   */
  public function buildIssueCodeSnippet(array $details, string $severity = 'error'): ?array {
    $file = $details['file'] ?? '';
    $line = $details['line'] ?? NULL;

    if (!$file || !$line) {
      return NULL;
    }

    $snippet = $this->codeSnippet($file, (int) $line, $this->normalizeSeverity($severity) ?? 'notice');
    return !empty($snippet) ? $snippet : NULL;
  }

}
