<?php

declare(strict_types=1);

namespace Drupal\audit\Service;

use Drupal\audit\AuditAnalyzerPluginManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for running audit analyzers.
 *
 * Executes analyzers and stores scores in State API.
 * No longer writes JSON files to the private filesystem.
 */
class AuditRunner {

  /**
   * Constructs an AuditRunner.
   *
   * @param \Drupal\audit\AuditAnalyzerPluginManager $pluginManager
   *   The plugin manager.
   * @param \Drupal\audit\Service\AuditScoreStorage $scoreStorage
   *   The score storage service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    protected readonly AuditAnalyzerPluginManager $pluginManager,
    protected readonly AuditScoreStorage $scoreStorage,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly LoggerInterface $logger,
  ) {}

  /**
   * Runs a specific analyzer and returns results.
   *
   * @param string $analyzer_id
   *   The analyzer plugin ID.
   * @param bool $save_score
   *   Whether to save the score to State API.
   *
   * @return array|null
   *   The analysis results or NULL on failure.
   */
  public function runAnalyzer(string $analyzer_id, bool $save_score = TRUE): ?array {
    try {
      $analyzer = $this->pluginManager->createInstance($analyzer_id);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to create analyzer @id: @message', [
        '@id' => $analyzer_id,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }

    $warnings = $analyzer->checkRequirements();
    if (!empty($warnings)) {
      $this->logger->warning('Analyzer @id has unmet requirements: @warnings', [
        '@id' => $analyzer_id,
        '@warnings' => implode(', ', $warnings),
      ]);
      // Return partial result with requirements warnings.
      return [
        'metadata' => [
          'analyzer' => $analyzer_id,
          'timestamp' => (new \DateTime())->format(\DateTime::ATOM),
          'requirements_warnings' => $warnings,
        ],
        'summary' => [
          'errors' => 0,
          'warnings' => count($warnings),
          'notices' => 0,
        ],
        'score' => NULL,
        'results' => [],
      ];
    }

    try {
      $results = $analyzer->analyze();

      // Normalize the results structure.
      $normalized = $this->normalizeResults($analyzer_id, $results);

      // Save score to State API if requested.
      if ($save_score && !empty($normalized['score'])) {
        $this->scoreStorage->saveScore(
          $analyzer_id,
          $normalized['score'],
          $normalized['summary']
        );

        // Recalculate and save the project score.
        $this->scoreStorage->calculateAndSaveProjectScore();
      }

      $this->logger->info('Analyzer @id completed successfully.', ['@id' => $analyzer_id]);

      return $normalized;
    }
    catch (\Exception $e) {
      $this->logger->error('Analyzer @id failed: @message', [
        '@id' => $analyzer_id,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Normalizes analyzer results into a consistent structure.
   *
   * @param string $analyzer_id
   *   The analyzer plugin ID.
   * @param array $results
   *   The raw analyzer results.
   *
   * @return array
   *   Normalized results structure.
   */
  protected function normalizeResults(string $analyzer_id, array $results): array {
    $total_errors = 0;
    $total_warnings = 0;
    $total_notices = 0;
    $all_results = [];

    if (isset($results['_files'])) {
      // Multi-file analyzer.
      foreach ($results['_files'] as $file_data) {
        $total_errors += $file_data['summary']['errors'] ?? 0;
        $total_warnings += $file_data['summary']['warnings'] ?? 0;
        $total_notices += $file_data['summary']['notices'] ?? 0;

        if (!empty($file_data['results'])) {
          $all_results = array_merge($all_results, $file_data['results']);
        }
      }
    }
    else {
      // Single file analyzer.
      $total_errors = $results['summary']['errors'] ?? 0;
      $total_warnings = $results['summary']['warnings'] ?? 0;
      $total_notices = $results['summary']['notices'] ?? 0;
      $all_results = $results['results'] ?? [];
    }

    $normalized = [
      'metadata' => [
        'analyzer' => $analyzer_id,
        'timestamp' => (new \DateTime())->format(\DateTime::ATOM),
      ],
      'summary' => [
        'errors' => $total_errors,
        'warnings' => $total_warnings,
        'notices' => $total_notices,
      ],
      'score' => $results['score'] ?? NULL,
      'results' => $all_results,
    ];

    // Preserve _files for detailed results rendering.
    if (isset($results['_files'])) {
      $normalized['_files'] = $results['_files'];
    }

    // Include any additional data from the analyzer.
    $reserved_keys = ['_files', '_filename', 'summary', 'score', 'results', 'metadata'];
    foreach ($results as $key => $value) {
      if (!in_array($key, $reserved_keys, TRUE)) {
        $normalized[$key] = $value;
      }
    }

    return $normalized;
  }

  /**
   * Runs all available analyzers.
   *
   * @return array
   *   Array of results keyed by analyzer ID.
   */
  public function runAll(): array {
    $results = [];
    $definitions = $this->pluginManager->getDefinitions();

    foreach ($definitions as $id => $definition) {
      $results[$id] = $this->runAnalyzer($id);
    }

    return $results;
  }

  /**
   * Gets all available analyzer definitions.
   *
   * @return array
   *   Array of plugin definitions.
   */
  public function getAnalyzerDefinitions(): array {
    return $this->pluginManager->getDefinitions();
  }

  /**
   * Gets stored score for an analyzer.
   *
   * @param string $analyzer_id
   *   The analyzer plugin ID.
   *
   * @return array|null
   *   The stored score data or NULL.
   */
  public function getStoredScore(string $analyzer_id): ?array {
    return $this->scoreStorage->getScore($analyzer_id);
  }

  /**
   * Gets all stored scores.
   *
   * @return array
   *   Array of scores keyed by analyzer ID.
   */
  public function getAllStoredScores(): array {
    return $this->scoreStorage->getAllScores();
  }

}
