<?php

declare(strict_types=1);

namespace Drupal\audit\Service;

use Drupal\audit\AuditAnalyzerPluginManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\State\StateInterface;

/**
 * Service for storing audit scores in State API.
 *
 * Stores individual analyzer scores persistently for external API sync.
 * Each analyzer has its own state entry for independent updates.
 */
class AuditScoreStorage {

  /**
   * State key prefix for individual analyzer scores.
   */
  protected const STATE_PREFIX = 'audit.score.';

  /**
   * State key for the scores index.
   */
  protected const INDEX_KEY = 'audit.scores.index';

  /**
   * State key for the project score.
   */
  protected const PROJECT_SCORE_KEY = 'audit.project_score';

  /**
   * Constructs an AuditScoreStorage.
   *
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\audit\AuditAnalyzerPluginManager $pluginManager
   *   The audit analyzer plugin manager.
   */
  public function __construct(
    protected readonly StateInterface $state,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly AuditAnalyzerPluginManager $pluginManager,
  ) {}

  /**
   * Saves the score for an analyzer.
   *
   * @param string $analyzer_id
   *   The analyzer plugin ID.
   * @param array $score_data
   *   The score data with 'factors'.
   * @param array $summary
   *   The summary with 'errors', 'warnings', 'notices'.
   */
  public function saveScore(string $analyzer_id, array $score_data, array $summary): void {
    $factors = $score_data['factors'] ?? [];
    $total_score = $this->calculateTotalScore($factors);

    $data = [
      'timestamp' => time(),
      'total_score' => $total_score,
      'factors' => $factors,
      'errors' => $summary['errors'] ?? 0,
      'warnings' => $summary['warnings'] ?? 0,
      'notices' => $summary['notices'] ?? 0,
    ];

    $this->state->set(self::STATE_PREFIX . $analyzer_id, $data);

    // Update the index.
    $index = $this->state->get(self::INDEX_KEY, []);
    $index[$analyzer_id] = time();
    $this->state->set(self::INDEX_KEY, $index);
  }

  /**
   * Gets the score for an analyzer.
   *
   * @param string $analyzer_id
   *   The analyzer plugin ID.
   *
   * @return array|null
   *   The score data or NULL if not found.
   */
  public function getScore(string $analyzer_id): ?array {
    return $this->state->get(self::STATE_PREFIX . $analyzer_id);
  }

  /**
   * Gets all stored scores.
   *
   * @return array
   *   Array of scores keyed by analyzer ID.
   */
  public function getAllScores(): array {
    $index = $this->state->get(self::INDEX_KEY, []);

    if (empty($index)) {
      return [];
    }

    $keys = array_map(
      fn($id) => self::STATE_PREFIX . $id,
      array_keys($index)
    );

    $scores = $this->state->getMultiple($keys);

    $result = [];
    foreach ($scores as $key => $data) {
      $analyzer_id = str_replace(self::STATE_PREFIX, '', $key);
      $result[$analyzer_id] = $data;
    }

    return $result;
  }

  /**
   * Gets data formatted for external API.
   *
   * @return array
   *   Formatted data for API submission.
   */
  public function getDataForExternalApi(): array {
    $scores = $this->getAllScores();

    return [
      'site_uuid' => \Drupal::config('system.site')->get('uuid'),
      'collected_at' => time(),
      'analyzers' => $scores,
    ];
  }

  /**
   * Deletes the score for an analyzer.
   *
   * @param string $analyzer_id
   *   The analyzer plugin ID.
   */
  public function deleteScore(string $analyzer_id): void {
    $this->state->delete(self::STATE_PREFIX . $analyzer_id);

    $index = $this->state->get(self::INDEX_KEY, []);
    unset($index[$analyzer_id]);
    $this->state->set(self::INDEX_KEY, $index);
  }

  /**
   * Clears all stored scores.
   */
  public function clearAll(): void {
    $index = $this->state->get(self::INDEX_KEY, []);

    foreach (array_keys($index) as $analyzer_id) {
      $this->state->delete(self::STATE_PREFIX . $analyzer_id);
    }

    $this->state->delete(self::INDEX_KEY);
  }

  /**
   * Checks if a score exists for an analyzer.
   *
   * @param string $analyzer_id
   *   The analyzer plugin ID.
   *
   * @return bool
   *   TRUE if score exists.
   */
  public function hasScore(string $analyzer_id): bool {
    return $this->state->get(self::STATE_PREFIX . $analyzer_id) !== NULL;
  }

  /**
   * Gets the last update timestamp for an analyzer.
   *
   * @param string $analyzer_id
   *   The analyzer plugin ID.
   *
   * @return int|null
   *   The timestamp or NULL if not found.
   */
  public function getLastUpdate(string $analyzer_id): ?int {
    $data = $this->getScore($analyzer_id);
    return $data['timestamp'] ?? NULL;
  }

  /**
   * Calculates the total score from factors using weighted average.
   *
   * @param array $factors
   *   The factors array with 'score' and 'weight' for each factor.
   *
   * @return int
   *   The weighted average score (0-100).
   */
  protected function calculateTotalScore(array $factors): int {
    if (empty($factors)) {
      return 0;
    }

    $total_weight = 0;
    $weighted_sum = 0;

    foreach ($factors as $factor) {
      $score = $factor['score'] ?? 0;
      $weight = $factor['weight'] ?? 1;
      $weighted_sum += $score * $weight;
      $total_weight += $weight;
    }

    if ($total_weight === 0) {
      return 0;
    }

    return (int) round($weighted_sum / $total_weight);
  }

  /**
   * Gets the total score for an analyzer.
   *
   * @param string $analyzer_id
   *   The analyzer plugin ID.
   *
   * @return int|null
   *   The total score (0-100) or NULL if not found.
   */
  public function getTotalScore(string $analyzer_id): ?int {
    $data = $this->getScore($analyzer_id);
    return $data['total_score'] ?? NULL;
  }

  /**
   * Calculates and saves the project score.
   *
   * Uses all stored individual scores and configured multipliers
   * to calculate a weighted average project score.
   */
  public function calculateAndSaveProjectScore(): void {
    $scores = $this->getAllScores();

    if (empty($scores)) {
      $this->state->delete(self::PROJECT_SCORE_KEY);
      return;
    }

    $config = $this->configFactory->get('audit.settings');
    $definitions = $this->pluginManager->getDefinitions();

    $total_weight = 0;
    $weighted_sum = 0;

    foreach ($scores as $analyzer_id => $score_data) {
      $total_score = $score_data['total_score'] ?? NULL;

      if ($total_score === NULL) {
        continue;
      }

      // Get multiplier from config, fallback to definition weight.
      $multiplier = $config->get('multipliers.' . $analyzer_id);
      if ($multiplier === NULL) {
        $multiplier = $definitions[$analyzer_id]['weight'] ?? 3;
      }
      $multiplier = (int) $multiplier;

      // Skip analyzers with multiplier 0.
      if ($multiplier === 0) {
        continue;
      }

      $weighted_sum += $total_score * $multiplier;
      $total_weight += $multiplier;
    }

    if ($total_weight === 0) {
      $this->state->delete(self::PROJECT_SCORE_KEY);
      return;
    }

    $project_score = (int) round($weighted_sum / $total_weight);

    $this->state->set(self::PROJECT_SCORE_KEY, [
      'score' => $project_score,
      'timestamp' => time(),
      'analyzers_count' => count($scores),
    ]);
  }

  /**
   * Gets the stored project score.
   *
   * @return array|null
   *   The project score data with 'score', 'timestamp', and 'analyzers_count',
   *   or NULL if not calculated yet.
   */
  public function getProjectScore(): ?array {
    return $this->state->get(self::PROJECT_SCORE_KEY);
  }

  /**
   * Deletes the stored project score.
   */
  public function deleteProjectScore(): void {
    $this->state->delete(self::PROJECT_SCORE_KEY);
  }

}
