<?php

/**
 * Provides the Audit Export Clear tool plugin.
 *
 * PHP Version 8.1
 *
 * @file
 * Provides the Audit Export Clear tool plugin.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @version  GIT: 1.0.0
 * @link     https://www.drupal.org/project/audit_export
 */

declare(strict_types=1);

namespace Drupal\audit_export_tool\Plugin\tool\Tool;

use Drupal\audit_export_core\AuditExportPluginManager;
use Drupal\audit_export_core\Service\AuditExportAuditReport;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\TypedData\InputDefinitionInterface;
use Drupal\tool\TypedData\InputDefinitionRefinerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tool to clear audit report data.
 *
 * Removes stored report data for one or all audits. This is useful
 * to force a fresh run or to clean up old data.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://www.drupal.org/project/audit_export
 */
#[Tool(
    id: 'audit_export_clear',
    label: new TranslatableMarkup('Clear Audit Report Data'),
    description: new TranslatableMarkup(
        'Clears stored report data for one or all audits.
      Use this to force fresh data on next run.'
    ),
    operation: ToolOperation::Trigger,
    destructive: true,
    input_definitions: [
    'audit_id' => new InputDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Audit ID'),
        description: new TranslatableMarkup(
            'The audit to clear. Leave empty to clear ALL audits.'
        ),
        required: false,
    ),
    'confirm' => new InputDefinition(
        data_type: 'boolean',
        label: new TranslatableMarkup('Confirm'),
        description: new TranslatableMarkup('Must be TRUE to proceed.'),
        default_value: false,
    ),
    ],
    input_definition_refiners: [
    'audit_id' => [],
    ],
    output_definitions: [
    'cleared_count' => new ContextDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Cleared Count'),
        description: new TranslatableMarkup('Number of audits cleared.'),
    ),
    'cleared_audits' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Cleared Audits'),
        description: new TranslatableMarkup('List of audit IDs that were cleared.'),
    ),
    ],
)]
final class AuditExportClear extends ToolBase implements
    InputDefinitionRefinerInterface
{

    /**
     * The audit export plugin manager.
     *
     * @var \Drupal\audit_export_core\AuditExportPluginManager
     */
    protected AuditExportPluginManager $auditPluginManager;

    /**
     * The audit report service.
     *
     * @var \Drupal\audit_export_core\Service\AuditExportAuditReport
     */
    protected AuditExportAuditReport $auditReport;

    /**
     * {@inheritdoc}
     *
     * @param ContainerInterface $container         Service container.
     * @param array              $configuration     Plugin configuration.
     * @param string             $plugin_id         The plugin ID.
     * @param mixed              $plugin_definition Plugin definition.
     *
     * @return static
     *   The created instance.
     */
    public static function create(
        ContainerInterface $container,
        array $configuration,
        $plugin_id,
        $plugin_definition,
    ): static {
        $instance = parent::create(
            $container,
            $configuration,
            $plugin_id,
            $plugin_definition
        );
        $instance->auditPluginManager = $container->get(
            'plugin.manager.audit_export_audit'
        );
        $instance->auditReport = $container->get('audit_export_core.audit_report');
        return $instance;
    }

    /**
     * {@inheritdoc}
     *
     * @param string                   $name       The input name.
     * @param InputDefinitionInterface $definition The input definition.
     * @param array                    $values     The current values.
     *
     * @return InputDefinitionInterface
     *   The refined input definition.
     */
    public function refineInputDefinition(
        string $name,
        InputDefinitionInterface $definition,
        array $values,
    ): InputDefinitionInterface {
        if ($name === 'audit_id') {
            $definitions = $this->auditPluginManager->getDefinitions();
            $choices = array_merge([''], array_keys($definitions));
            $definition->addConstraint('Choice', ['choices' => $choices]);
        }
        return $definition;
    }

    /**
     * {@inheritdoc}
     *
     * @param array $values The input values.
     *
     * @return ExecutableResult
     *   The execution result.
     */
    protected function doExecute(array $values): ExecutableResult
    {
        $audit_id = !empty($values['audit_id'])
        ? (string) $values['audit_id']
        : null;
        $confirm = (bool) ($values['confirm'] ?? false);

        if (!$confirm) {
            return ExecutableResult::failure(
                $this->t('Confirmation required. Set confirm to TRUE to proceed.')
            );
        }

        try {
            $cleared_audits = [];

            if (empty($audit_id)) {
                // Clear all audits.
                $definitions = $this->auditPluginManager->getDefinitions();
                foreach (array_keys($definitions) as $id) {
                    $this->auditReport->clearReportData($id);
                    $cleared_audits[] = $id;
                }
            } else {
                // Validate audit exists.
                if (!$this->auditPluginManager->hasDefinition($audit_id)) {
                    return ExecutableResult::failure(
                        $this->t('Audit "@id" does not exist.', ['@id' => $audit_id])
                    );
                }

                $this->auditReport->clearReportData($audit_id);
                $cleared_audits[] = $audit_id;
            }

            return ExecutableResult::success(
                $this->t(
                    'Cleared data for @count audit(s).',
                    ['@count' => count($cleared_audits)]
                ),
                [
                'cleared_count' => count($cleared_audits),
                'cleared_audits' => $cleared_audits,
                ]
            );
        }
        catch (\Exception $e) {
            return ExecutableResult::failure(
                $this->t(
                    'Failed to clear audit data: @error',
                    ['@error' => $e->getMessage()]
                )
            );
        }
    }

    /**
     * {@inheritdoc}
     *
     * @param array            $values           The input values.
     * @param AccountInterface $account          The user account.
     * @param bool             $return_as_object Whether to return as object.
     *
     * @return bool|AccessResultInterface
     *   The access result.
     */
    protected function checkAccess(
        array $values,
        AccountInterface $account,
        bool $return_as_object = false,
    ): bool|AccessResultInterface {
        $access = AccessResult::allowedIfHasPermission(
            $account,
            'administer audit export'
        );
        return $return_as_object ? $access : $access->isAllowed();
    }

}
