<?php

/**
 * Provides the Audit Export CSV tool plugin.
 *
 * PHP Version 8.1
 *
 * @file
 * Provides the Audit Export CSV tool plugin.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @version  GIT: 1.0.0
 * @link     https://www.drupal.org/project/audit_export
 */

declare(strict_types=1);

namespace Drupal\audit_export_tool\Plugin\tool\Tool;

use Drupal\audit_export_core\AuditExportPluginManager;
use Drupal\audit_export_core\Service\AuditExportAuditReport;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\TypedData\InputDefinitionInterface;
use Drupal\tool\TypedData\InputDefinitionRefinerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tool to export audit report data as CSV.
 *
 * Generates a CSV file from the audit report and returns the file path
 * or CSV content directly.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://www.drupal.org/project/audit_export
 */
#[Tool(
    id: 'audit_export_csv',
    label: new TranslatableMarkup('Export Audit to CSV'),
    description: new TranslatableMarkup(
        'Exports an audit report to CSV format.
      Can save to filesystem or return CSV content directly.'
    ),
    operation: ToolOperation::Read,
    input_definitions: [
    'audit_id' => new InputDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Audit ID'),
        description: new TranslatableMarkup('The machine name of the audit.'),
        required: true,
    ),
    'save_to_file' => new InputDefinition(
        data_type: 'boolean',
        label: new TranslatableMarkup('Save to File'),
        description: new TranslatableMarkup(
            'Whether to save CSV to file. If FALSE, returns content directly.'
        ),
        default_value: true,
    ),
    'filename' => new InputDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Filename'),
        description: new TranslatableMarkup(
            'Custom filename (without extension). Defaults to audit_id_timestamp.'
        ),
        required: false,
    ),
    'include_headers' => new InputDefinition(
        data_type: 'boolean',
        label: new TranslatableMarkup('Include Headers'),
        description: new TranslatableMarkup('Whether to include headers row.'),
        default_value: true,
    ),
    'delimiter' => new InputDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Delimiter'),
        description: new TranslatableMarkup('CSV field delimiter character.'),
        default_value: ',',
        constraints: [
        'Choice' => [
          'choices' => [',', ';', '|'],
        ],
        ],
    ),
    ],
    input_definition_refiners: [
    'audit_id' => [],
    ],
    output_definitions: [
    'success' => new ContextDefinition(
        data_type: 'boolean',
        label: new TranslatableMarkup('Success'),
        description: new TranslatableMarkup('Whether the export was successful.'),
    ),
    'file_path' => new ContextDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('File Path'),
        description: new TranslatableMarkup('Path to saved CSV file.'),
    ),
    'file_url' => new ContextDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('File URL'),
        description: new TranslatableMarkup('URL to download the CSV file.'),
    ),
    'csv_content' => new ContextDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('CSV Content'),
        description: new TranslatableMarkup('The CSV content as string.'),
    ),
    'row_count' => new ContextDefinition(
        data_type: 'integer',
        label: new TranslatableMarkup('Row Count'),
        description: new TranslatableMarkup('Number of data rows exported.'),
    ),
    ],
)]
final class AuditExportCsv extends ToolBase implements
    InputDefinitionRefinerInterface
{

    /**
     * The audit export plugin manager.
     *
     * @var \Drupal\audit_export_core\AuditExportPluginManager
     */
    protected AuditExportPluginManager $auditPluginManager;

    /**
     * The audit report service.
     *
     * @var \Drupal\audit_export_core\Service\AuditExportAuditReport
     */
    protected AuditExportAuditReport $auditReport;

    /**
     * The file system service.
     *
     * @var \Drupal\Core\File\FileSystemInterface
     */
    protected FileSystemInterface $fileSystem;

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected ConfigFactoryInterface $configFactory;

    /**
     * The file URL generator.
     *
     * @var \Drupal\Core\File\FileUrlGeneratorInterface
     */
    protected FileUrlGeneratorInterface $fileUrlGenerator;

    /**
     * {@inheritdoc}
     *
     * @param ContainerInterface $container         Service container.
     * @param array              $configuration     Plugin configuration.
     * @param string             $plugin_id         The plugin ID.
     * @param mixed              $plugin_definition Plugin definition.
     *
     * @return static
     *   The created instance.
     */
    public static function create(
        ContainerInterface $container,
        array $configuration,
        $plugin_id,
        $plugin_definition,
    ): static {
        $instance = parent::create(
            $container,
            $configuration,
            $plugin_id,
            $plugin_definition
        );
        $instance->auditPluginManager = $container->get(
            'plugin.manager.audit_export_audit'
        );
        $instance->auditReport = $container->get('audit_export_core.audit_report');
        $instance->fileSystem = $container->get('file_system');
        $instance->configFactory = $container->get('config.factory');
        $instance->fileUrlGenerator = $container->get('file_url_generator');
        return $instance;
    }

    /**
     * {@inheritdoc}
     *
     * @param string                   $name       The input name.
     * @param InputDefinitionInterface $definition The input definition.
     * @param array                    $values     The current values.
     *
     * @return InputDefinitionInterface
     *   The refined input definition.
     */
    public function refineInputDefinition(
        string $name,
        InputDefinitionInterface $definition,
        array $values,
    ): InputDefinitionInterface {
        if ($name === 'audit_id') {
            $definitions = $this->auditPluginManager->getDefinitions();
            $choices = array_keys($definitions);
            $definition->addConstraint('Choice', ['choices' => $choices]);
        }
        return $definition;
    }

    /**
     * {@inheritdoc}
     *
     * @param array $values The input values.
     *
     * @return ExecutableResult
     *   The execution result.
     */
    protected function doExecute(array $values): ExecutableResult
    {
        $audit_id = $values['audit_id'];
        $save_to_file = (bool) ($values['save_to_file'] ?? true);
        $filename = !empty($values['filename'])
        ? (string) $values['filename']
        : null;
        $include_headers = (bool) ($values['include_headers'] ?? true);
        $delimiter = (string) ($values['delimiter'] ?? ',');

        try {
            // Validate audit exists.
            if (!$this->auditPluginManager->hasDefinition($audit_id)) {
                return ExecutableResult::failure(
                    $this->t('Audit "@id" does not exist.', ['@id' => $audit_id])
                );
            }

            // Get data.
            $all_data = $this->auditReport->getReportData($audit_id);

            if (empty($all_data) || !is_array($all_data)) {
                return ExecutableResult::failure(
                    $this->t(
                        'No report data available for audit "@id".',
                        ['@id' => $audit_id]
                    )
                );
            }

            // Get headers.
            $plugin = $this->auditPluginManager->createInstance($audit_id);
            $headers = $plugin->getHeaders();

            // Build CSV content.
            $csv_rows = [];

            if ($include_headers) {
                $csv_rows[] = $this->csvRow($headers, $delimiter);
            }

            foreach ($all_data as $row) {
                $csv_rows[] = $this->csvRow($row, $delimiter);
            }

            $csv_content = implode("\n", $csv_rows);

            if ($save_to_file) {
                // Generate filename.
                if (empty($filename)) {
                    $filename = $audit_id . '_' . date('Y-m-d_His');
                }
                $filename = preg_replace('/[^a-zA-Z0-9_-]/', '_', $filename);

                // Determine directory.
                $config = $this->configFactory->get('audit_export_core.settings');
                $filesystem = $config->get('audit_export_filesystem') ?: 'temporary';
                $fs_path = $config->get('audit_export_filesystem_path');
                $path = $fs_path ?: 'audit-export';

                $directory = $filesystem . '://' . $path;
                $this->fileSystem->prepareDirectory(
                    $directory,
                    FileSystemInterface::CREATE_DIRECTORY
                );

                $file_path = $directory . '/' . $filename . '.csv';
                $saved_path = $this->fileSystem->saveData(
                    $csv_content,
                    $file_path,
                    FileSystemInterface::EXISTS_REPLACE
                );

                if ($saved_path) {
                    $file_url = $this->fileUrlGenerator->generateAbsoluteString(
                        $saved_path
                    );

                    return ExecutableResult::success(
                        $this->t(
                            'Exported @count rows to CSV file.',
                            ['@count' => count($all_data)]
                        ),
                        [
                        'success' => true,
                        'file_path' => $saved_path,
                        'file_url' => $file_url,
                        'csv_content' => '',
                        'row_count' => count($all_data),
                        ]
                    );
                } else {
                    return ExecutableResult::failure(
                        $this->t('Failed to save CSV file.')
                    );
                }
            } else {
                // Return content directly.
                return ExecutableResult::success(
                    $this->t(
                        'Generated CSV with @count rows.',
                        ['@count' => count($all_data)]
                    ),
                    [
                    'success' => true,
                    'file_path' => '',
                    'file_url' => '',
                    'csv_content' => $csv_content,
                    'row_count' => count($all_data),
                    ]
                );
            }
        }
        catch (\Exception $e) {
            return ExecutableResult::failure(
                $this->t('CSV export failed: @error', ['@error' => $e->getMessage()])
            );
        }
    }

    /**
     * Converts an array row to a CSV line.
     *
     * @param array  $row       The row data.
     * @param string $delimiter The delimiter character.
     *
     * @return string
     *   The CSV formatted line.
     */
    protected function csvRow(array $row, string $delimiter): string
    {
        $escaped = array_map(
            function ($value) use ($delimiter) {
                $value = (string) $value;
                // Escape quotes and wrap if contains delimiter, quote, or newline.
                if (strpos($value, $delimiter) !== false
                    || strpos($value, '"') !== false
                    || strpos($value, "\n") !== false
                ) {
                    $value = '"' . str_replace('"', '""', $value) . '"';
                }
                return $value;
            },
            $row
        );

        return implode($delimiter, $escaped);
    }

    /**
     * {@inheritdoc}
     *
     * @param array            $values           The input values.
     * @param AccountInterface $account          The user account.
     * @param bool             $return_as_object Whether to return as object.
     *
     * @return bool|AccessResultInterface
     *   The access result.
     */
    protected function checkAccess(
        array $values,
        AccountInterface $account,
        bool $return_as_object = false,
    ): bool|AccessResultInterface {
        $access = AccessResult::allowedIfHasPermission(
            $account,
            'export audit reports'
        );
        return $return_as_object ? $access : $access->isAllowed();
    }

}
