<?php

/**
 * Provides the Audit Export Definition tool plugin.
 *
 * PHP Version 8.1
 *
 * @file
 * Provides the Audit Export Definition tool plugin.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @version  GIT: 1.0.0
 * @link     https://www.drupal.org/project/audit_export
 */

declare(strict_types=1);

namespace Drupal\audit_export_tool\Plugin\tool\Tool;

use Drupal\audit_export_core\AuditExportPluginManager;
use Drupal\audit_export_core\Service\AuditExportAuditReport;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\tool\Attribute\Tool;
use Drupal\tool\ExecutableResult;
use Drupal\tool\Tool\ToolBase;
use Drupal\tool\Tool\ToolOperation;
use Drupal\tool\TypedData\InputDefinition;
use Drupal\tool\TypedData\InputDefinitionInterface;
use Drupal\tool\TypedData\InputDefinitionRefinerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tool to get detailed definition of a specific audit plugin.
 *
 * Returns comprehensive information about an audit including its headers,
 * configuration, dependencies, and current status.
 *
 * @category Drupal
 * @package  AuditExportTool
 * @author   Drupal Contributors <info@drupal.org>
 * @license  GPL-2.0-or-later https://www.gnu.org/licenses/gpl-2.0.html
 * @link     https://www.drupal.org/project/audit_export
 */
#[Tool(
    id: 'audit_export_definition',
    label: new TranslatableMarkup('Get Audit Definition'),
    description: new TranslatableMarkup(
        'Retrieves the complete definition and schema of a specific audit plugin,
      including column headers, data type, and configuration.'
    ),
    operation: ToolOperation::Explain,
    input_definitions: [
    'audit_id' => new InputDefinition(
        data_type: 'string',
        label: new TranslatableMarkup('Audit ID'),
        description: new TranslatableMarkup('The machine name of the audit plugin.'),
        required: true,
    ),
    ],
    input_definition_refiners: [
    'audit_id' => [],
    ],
    output_definitions: [
    'definition' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Audit Definition'),
        description: new TranslatableMarkup('Complete audit definition.'),
    ),
    'headers' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Report Headers'),
        description: new TranslatableMarkup('Column headers for the audit report.'),
    ),
    'status' => new ContextDefinition(
        data_type: 'any',
        label: new TranslatableMarkup('Audit Status'),
        description: new TranslatableMarkup('Status with last run and row count.'),
    ),
    ],
)]
final class AuditExportDefinition extends ToolBase implements
    InputDefinitionRefinerInterface
{

    /**
     * The audit export plugin manager.
     *
     * @var \Drupal\audit_export_core\AuditExportPluginManager
     */
    protected AuditExportPluginManager $auditPluginManager;

    /**
     * The audit report service.
     *
     * @var \Drupal\audit_export_core\Service\AuditExportAuditReport
     */
    protected AuditExportAuditReport $auditReport;

    /**
     * {@inheritdoc}
     *
     * @param ContainerInterface $container         Service container.
     * @param array              $configuration     Plugin configuration.
     * @param string             $plugin_id         The plugin ID.
     * @param mixed              $plugin_definition Plugin definition.
     *
     * @return static
     *   The created instance.
     */
    public static function create(
        ContainerInterface $container,
        array $configuration,
        $plugin_id,
        $plugin_definition,
    ): static {
        $instance = parent::create(
            $container,
            $configuration,
            $plugin_id,
            $plugin_definition
        );
        $instance->auditPluginManager = $container->get(
            'plugin.manager.audit_export_audit'
        );
        $instance->auditReport = $container->get('audit_export_core.audit_report');
        return $instance;
    }

    /**
     * {@inheritdoc}
     *
     * @param string                   $name       The input name.
     * @param InputDefinitionInterface $definition The input definition.
     * @param array                    $values     The current values.
     *
     * @return InputDefinitionInterface
     *   The refined input definition.
     */
    public function refineInputDefinition(
        string $name,
        InputDefinitionInterface $definition,
        array $values,
    ): InputDefinitionInterface {
        if ($name === 'audit_id') {
            $definitions = $this->auditPluginManager->getDefinitions();
            $choices = array_keys($definitions);
            $definition->addConstraint('Choice', ['choices' => $choices]);
        }
        return $definition;
    }

    /**
     * {@inheritdoc}
     *
     * @param array $values The input values.
     *
     * @return ExecutableResult
     *   The execution result.
     */
    protected function doExecute(array $values): ExecutableResult
    {
        $audit_id = $values['audit_id'];

        try {
            // Check if audit exists.
            if (!$this->auditPluginManager->hasDefinition($audit_id)) {
                return ExecutableResult::failure(
                    $this->t('Audit "@id" does not exist.', ['@id' => $audit_id])
                );
            }

            // Get plugin definition.
            $plugin_definition = $this->auditPluginManager->getDefinition($audit_id);

            // Create plugin instance to get headers.
            $plugin = $this->auditPluginManager->createInstance($audit_id);
            $headers = $plugin->getHeaders();

            // Build definition output.
            $definition = [
            'id' => $audit_id,
            'label' => (string) $plugin_definition['label'],
            'description' => (string) ($plugin_definition['description'] ?? ''),
            'group' => $plugin_definition['group'] ?? 'general',
            'identifier' => $plugin_definition['identifier'] ?? null,
            'data_type' => $plugin_definition['data_type'] ?? 'flat',
            'provider' => $plugin_definition['provider'] ?? 'unknown',
            'dependencies' => $plugin_definition['dependencies'] ?? [],
            'class' => $plugin_definition['class'] ?? null,
            ];

            // Get current status.
            $last_date = $this->auditReport->getLastProcessedDate($audit_id);
            $report_data = $this->auditReport->getReportData($audit_id);

            $status = [
            'has_data' => !empty($report_data),
            'last_run' => $last_date ?: null,
            'row_count' => is_array($report_data) ? count($report_data) : 0,
            ];

            return ExecutableResult::success(
                $this->t(
                    'Retrieved definition for audit "@id".',
                    ['@id' => $audit_id]
                ),
                [
                'definition' => $definition,
                'headers' => $headers,
                'status' => $status,
                ]
            );
        }
        catch (\Exception $e) {
            return ExecutableResult::failure(
                $this->t(
                    'Failed to get audit definition: @error',
                    ['@error' => $e->getMessage()]
                )
            );
        }
    }

    /**
     * {@inheritdoc}
     *
     * @param array            $values           The input values.
     * @param AccountInterface $account          The user account.
     * @param bool             $return_as_object Whether to return as object.
     *
     * @return bool|AccessResultInterface
     *   The access result.
     */
    protected function checkAccess(
        array $values,
        AccountInterface $account,
        bool $return_as_object = false,
    ): bool|AccessResultInterface {
        $access = AccessResult::allowedIfHasPermission(
            $account,
            'view audit export reports'
        );
        return $return_as_object ? $access : $access->isAllowed();
    }

}
